/***************************************************************
 File:		RxHandler.c
 Module:	  	rx handler genRisc interface
 Purpose: 	holds all the implementation of the  FW - rx handler genRisc interface module
 Description:  This module contains the implementation of the  FW - rx handler genRisc interface
 			module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "ieee80211.h" //should be removed when the SHRAM files will be defined
#include "Pac_Api.h"
#include "RxHandler.h"
#include "RxHandler_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "PacketDescriptor.h"
#include "enet_pas.h"
#include "ShramRxH.h"
#include "HwMemoryMap.h"
#include "MacGenriscRxRegs.h"
#include "ShramSecurity.h"
#include "ShramRxDescriptors.h"
#include "ShramGenRiscMailboxes.h"
#include "HwQManager_API.h"
#include "MacHtExtensionsRegs.h"
#include "ShramPacketDescriptors.h"
#include "ShramRxDescriptorsPayload.h"
#include "mhi_umi.h"
#include "Utils_Api.h"
#include "EventsManager_api.h"
#include "Statistics_Api.h"
#include "ShramPhyStatDb.h"
#include "HostInterface_API.h"
#include "ShramGenriscInitVectorStartAddr.h"
#include "RxManager_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_GENRISCS_INTERFACE
#define LOG_LOCAL_FID 1



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void rxHandlerInterface_InitFwRxHandlerMailboxes(void);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
FwRxHandlerMailbox0FifoInParams_t	*pFwRxHandlerMailbox0FifoInParams;
FwRxHandlerMailbox0FifoOutParams_t	*pFwRxHandlerMailbox0FifoOutParams;
FwRxHandlerMailbox1FifoInParams_t	*pFwRxHandlerMailbox1FifoInParams;
FwRxHandlerMailbox1FifoOutParams_t	*pFwRxHandlerMailbox1FifoOutParams;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
isr_RxHandler  


Description:
------------
interrupt routine that handles errors scenarions from the RXH genRisc 



Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_RxHandler(void)
{
#if defined (ENET_INC_UMAC)
	RegMacGenriscRxGenriscUpperIrqClr_u irqClearRegister;
	RegMacGenriscRxGenriscUpperIrqStatus_u irqStatusRegister;
#elif defined (ENET_INC_LMAC)
	RegMacGenriscRxGenriscLowerIrqClr_u irqClearRegister;
	RegMacGenriscRxGenriscLowerIrqStatus_u irqStatusRegister;
#endif
	uint32 eventIndex;
#ifdef FW_RX_HANDLER_MAILBOX_0_ENABLED
	FwRxHandlerMailbox0FifoOutMessage_t *pBody0;
#endif //FW_RX_HANDLER_MAILBOX_0_ENABLED
#ifdef FW_RX_HANDLER_MAILBOX_1_ENABLED
	FwRxHandlerMailbox1FifoOutMessage_t *pBody1;
#endif //FW_RX_HANDLER_MAILBOX_1_ENABLED
#if defined FW_RX_HANDLER_MAILBOX_0_ENABLED || defined FW_RX_HANDLER_MAILBOX_1_ENABLED
	uint32 idx;
	K_MSG *pMsg;
#endif

	irqClearRegister.val = 0;
	irqStatusRegister.val = 0;


#if defined (ENET_INC_UMAC)
	RegAccess_Read(REG_MAC_GENRISC_RX_GENRISC_UPPER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscUpperIrqClr = (0x1 << eventIndex);

	RegAccess_Write(REG_MAC_GENRISC_RX_GENRISC_UPPER_IRQ_CLR, irqClearRegister.val);
#elif defined (ENET_INC_LMAC)
	RegAccess_Read(REG_MAC_GENRISC_RX_GENRISC_LOWER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscLowerIrqClr = (0x1 << eventIndex);

	RegAccess_Write(REG_MAC_GENRISC_RX_GENRISC_LOWER_IRQ_CLR, irqClearRegister.val);
#else
#error "either umac or lmac must be defined"
#endif

	switch (eventIndex)
	{
	case RX_HANDLER_2_MIPS_INTERRUPTS_PHY_STATUS_FIFO_OUT_NOT_EMPTY_START_BIT:
		EventManager_TriggerSwEvent(EVENT_ID_METRICS_FIFO_NOT_EMPTY);
		break;	
	case RX_HANDLER_2_MIPS_INTERRUPTS_ERROR_START_BIT:
		FATAL("RX Handler Error Start");
		break;
#ifdef FW_RX_HANDLER_MAILBOX_0_ENABLED
	case RX_HANDLER_2_MIPS_INTERRUPTS_MAILBOX_0_FIFO_OUT_NOT_EMPTY_START_BIT:
		idx = 0;
		while (idx < FW_GENRISC_RX_HANDLER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES)
		{
			pMsg = RxHandlerInterface_Mailbox0_Pop();
			if (pMsg)
			{
				pBody0 = (FwRxHandlerMailbox0FifoOutMessage_t *)pK_MSG_DATA(pMsg);

				//send message to designated task
				switch(pBody0->messageId)
				{
				default:
					DEBUG_FATAL("Unsupported Message ID");
				}
			}
			else
			{
				break;
			}
			idx++;
		}
		break;
#endif //FW_RX_HANDLER_MAILBOX_0_ENABLED
#ifdef FW_RX_HANDLER_MAILBOX_1_ENABLED
	case RX_HANDLER_2_MIPS_INTERRUPTS_MAILBOX_1_FIFO_OUT_NOT_EMPTY_START_BIT:
		idx = 0;
		while (idx < FW_GENRISC_RX_HANDLER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES)
		{
			pMsg = RxHandlerInterface_Mailbox1_Pop();
			if (pMsg)
			{
				pBody1 = (FwRxHandlerMailbox1FifoOutMessage_t *)pK_MSG_DATA(pMsg);

				//send message to designated task
				switch(pBody1->messageId)
				{
				default:
					DEBUG_FATAL("Unsupported Message ID");
				}
			}
			else
			{
				break;
			}
			idx++;
		}
		break;
#endif //FW_RX_HANDLER_MAILBOX_1_ENABLED
	default:
		DEBUG_FATAL("RX Handler Unknown Event");
	}
}


/**********************************************************************************
RxHandler_Halt  


Description:
------------
stop the RXH genRisc

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 RxHandler_Halt(void)
{	
	RegMacGenriscRxStopOp_u haltRegister;
	RegMacGenriscRxLastPcExecuted_u lastPcExecuted;
	
	haltRegister.val = 0;


	/* Stop the genrisc */
	haltRegister.bitFields.stopOp = RXH_GENRISC_HALT_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_RX_STOP_OP, haltRegister.val);
	/* Read last pc executed */
	RegAccess_Read(REG_MAC_GENRISC_RX_LAST_PC_EXECUTED, &lastPcExecuted.val);

	return lastPcExecuted.bitFields.lastExecuted;
}



/**********************************************************************************
RxHandler_SetUdpHeader

Description:
------------
fill the Udp header structure in shared ram

Input:
-----

dutSnifferUdpHeaderParams *snifferUdpHeaderParams_p

			
Output:
-------
	none

Returns:
--------
	void - 
	
**********************************************************************************/
void RxHandler_SetUdpHeader(UMI_ADD_STREAM_REQ_t* addStreamStructurePtr)
{
	//save udp header template to shared ram
	memcpy32(RxhSnifferParams.udpHeader, addStreamStructurePtr->udpHeader, UDP_HEADER_WORD_SIZE);

	//change padding to sniffer unique padding value
	((udpPacketHdr_t*)(RxhSnifferParams.udpHeader))->padding = RX_HANDLER_SNIFFER_UDP_HEADER_PADDING_BIG_ENDIAN;

	//save pre calculated checksum
	RxhSnifferParams.swPreCalcChecksum = SWITCH_ENDIANESS_TYPE_LONG(addStreamStructurePtr->swPreCalcChecksum);
}	


/**********************************************************************************
RxHandler_Init  


Description:
------------
fill the initizlization structre of the RXH genRisc and start the RXH genRisc

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void RxHandler_Init(void)
{	
	RegMacGenriscRxStartOp_u startRegister;
	RegMacHtExtensionsGenriscRxFreeListIdx_u hostRxDescriptorListRegister;
	RxHParams_t *pRxhInitializationStructure;
		
    uint32 timeStamp = 0;
	startRegister.val = 0;
	hostRxDescriptorListRegister.val = 0;

	//init mailbox
	rxHandlerInterface_InitFwRxHandlerMailboxes();

	pRxhInitializationStructure = (RxHParams_t *)CONVERT_TO_PHYSICAL_ADDR(&RxhInitializationStructure); 
	GenriscsInitVectorStartAddr.rxHandlerInitVectorPtr = (uint32)pRxhInitializationStructure;

	memset((void*)&RxhInitializationStructure, 0, sizeof(RxHParams_t));
	RxhInitializationStructure.securityDescriptorsBaseAddress = CONVERT_TO_PHYSICAL_ADDR(&rxSecurityDescriptors[0]);
	RxhInitializationStructure.rdPoolBaseAddress = CONVERT_TO_PHYSICAL_ADDR(PacketDescriptorsPtr);
	RxhInitializationStructure.fwRdPoolBaseAddress = CONVERT_TO_PHYSICAL_ADDR(PacketDescriptorsPtr);
    RxhInitializationStructure.managementRdPoolBaseAddress = CONVERT_TO_PHYSICAL_ADDR(PacketDescriptorsPtr);
	RxhInitializationStructure.freeRdListIndex = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	RxhInitializationStructure.freeFwRdListIndex = HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
    RxhInitializationStructure.freeManagementRdListIndex = HW_Q_MANAGER_RX_FREE_MANAGEMENT_RDS_LIST;
	RxhInitializationStructure.rxPpListIndex = HW_Q_MANAGER_RX_POST_PROCESSING_INPUT_LIST;
	RxhInitializationStructure.localCopyMaxPayloadSize= FW_RD_PAYLOAD_SIZE;
	RxhInitializationStructure.wlanInterface = 0; 				/* TBD add API to get wlan IF*/
	RxhInitializationStructure.loggerStream = 0;					/* TBD - Mechanism to chengae logger stream */
	RxhInitializationStructure.snifferQueueId = SID_FOR_LOGGER_RX_DATA << 0x3;	// TBD - use definition to shift the sid to proper location in queue id;					/* TBD - Mechanism to chengae logger stream */
#ifdef FW_RX_HANDLER_MAILBOX_0_ENABLED
	RxhInitializationStructure.mailbox0_FifoInStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailbox0FifoInParams);
	RxhInitializationStructure.mailbox0_FifoInNumOfEntries = FW_GENRISC_RX_HANDLER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES;
	RxhInitializationStructure.mailbox0_FifoOutStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailbox0FifoOutParams);
	RxhInitializationStructure.mailbox0_FifoOutNumOfEntries = FW_GENRISC_RX_HANDLER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES;
#else
	RxhInitializationStructure.mailbox0_FifoInStructStartAddress = RX_HANDLER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	RxhInitializationStructure.mailbox0_FifoInNumOfEntries = 0;
	RxhInitializationStructure.mailbox0_FifoOutStructStartAddress = RX_HANDLER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	RxhInitializationStructure.mailbox0_FifoOutNumOfEntries = 0;
#endif //FW_RX_HANDLER_MAILBOX_0_ENABLED
#ifdef FW_RX_HANDLER_MAILBOX_1_ENABLED
	RxhInitializationStructure.mailbox1_FifoInStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailbox1FifoInParams);
	RxhInitializationStructure.mailbox1_FifoInNumOfEntries = FW_GENRISC_RX_HANDLER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES;
	RxhInitializationStructure.mailbox1_FifoOutStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwRxHandlerMailbox1FifoOutParams);
	RxhInitializationStructure.mailbox1_FifoOutNumOfEntries = FW_GENRISC_RX_HANDLER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES;
#else
	RxhInitializationStructure.mailbox1_FifoInStructStartAddress = RX_HANDLER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	RxhInitializationStructure.mailbox1_FifoInNumOfEntries = 0;
	RxhInitializationStructure.mailbox1_FifoOutStructStartAddress = RX_HANDLER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	RxhInitializationStructure.mailbox1_FifoOutNumOfEntries = 0;
#endif //FW_RX_HANDLER_MAILBOX_1_ENABLED

    RxhInitializationStructure.categoryBitMapWord0 = ACTION_FRAME_CATEGORY_INIT_WORD0;
    RxhInitializationStructure.categoryBitMapWord1 = ACTION_FRAME_CATEGORY_INIT_WORD1;
    RxhInitializationStructure.categoryBitMapWord2 = ACTION_FRAME_CATEGORY_INIT_WORD2;
    RxhInitializationStructure.categoryBitMapWord3 = ACTION_FRAME_CATEGORY_INIT_WORD3;

    RxhInitializationStructure.SecKeysDatabase = CONVERT_TO_PHYSICAL_ADDR(&KeysDatabase[0]);
	RxhInitializationStructure.snifferStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(&RxhSnifferParams);
	RxhInitializationStructure.clientModeMcastFilterRingStartAddress = CONVERT_TO_PHYSICAL_ADDR(&clientModeMcastFilterRing[0]);
	RxhInitializationStructure.clientModeMcastFilterRingSize = CLIENT_MODE_MCAST_FILTER_RING_SIZE;
	hostRxDescriptorListRegister.bitFields.genriscRxFreeListIdx = HW_Q_MANAGER_RX_FREE_HOST_RDS_LIST;
	RegAccess_Write(REG_MAC_HT_EXTENSIONS_GENRISC_RX_FREE_LIST_IDX, hostRxDescriptorListRegister.val);
		
	RxhInitializationStructure.phyStatusFifoStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(&phyMetricFifo);
	RxhInitializationStructure.phyStatusFifoNumOfEntries = SIZE_OF_METRICS_FIFO;

	/* Start the genrisc */
	startRegister.bitFields.startOp = RXH_GENRISC_START_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_RX_START_OP, startRegister.val);

#ifndef ARC_SIM
	timeStamp = GET_TSF_TIMER_LOW();
	while(RXH_GENRISC_COMPLETED_CHI_MAGIC != RxhInitializationStructure.chiMagic)
	{
		ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= RXH_GENRISC_MAX_MICROSECONDS_FOR_INITIALIZATION); 
	}
#endif
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

static void rxHandlerInterface_InitFwRxHandlerMailboxes(void)
{
#ifdef FW_RX_HANDLER_MAILBOX_0_ENABLED
	memset(&FwRxHandlerMailbox0FifoInBuffer, 0x0, sizeof(FwRxHandlerMailbox0FifoInBuffer));
	memset(&FwRxHandlerMailbox0FifoOutBuffer, 0x0, sizeof(FwRxHandlerMailbox0FifoOutBuffer));
	pFwRxHandlerMailbox0FifoInParams = (FwRxHandlerMailbox0FifoInParams_t *)(&FwRxHandlerMailbox0FifoInBuffer);
	pFwRxHandlerMailbox0FifoOutParams = (FwRxHandlerMailbox0FifoOutParams_t *)(&FwRxHandlerMailbox0FifoOutBuffer);
#endif //FW_RX_HANDLER_MAILBOX_0_ENABLED

#ifdef FW_RX_HANDLER_MAILBOX_1_ENABLED
	memset(&FwRxHandlerMailbox1FifoInBuffer, 0x0, sizeof(FwRxHandlerMailbox1FifoInBuffer));
	memset(&FwRxHandlerMailbox1FifoOutBuffer, 0x0, sizeof(FwRxHandlerMailbox1FifoOutBuffer));
	pFwRxHandlerMailbox1FifoInParams = (FwRxHandlerMailbox1FifoInParams_t *)(&FwRxHandlerMailbox1FifoInBuffer);
	pFwRxHandlerMailbox1FifoOutParams = (FwRxHandlerMailbox1FifoOutParams_t *)(&FwRxHandlerMailbox1FifoOutBuffer);
#endif //FW_RX_HANDLER_MAILBOX_1_ENABLED
}

#ifdef FW_RX_HANDLER_MAILBOX_0_ENABLED
void RxHandlerInterface_Mailbox0_Push(FwRxHandlerMailbox0FifoInMessage_t *message)
{
	RegMacGenriscRxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwRxHandlerMailbox0FifoInParams->writeCounter - pFwRxHandlerMailbox0FifoInParams->readCounter) < FW_GENRISC_RX_HANDLER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES);

	memcpy32(&(pFwRxHandlerMailbox0FifoInParams->messageArray[(pFwRxHandlerMailbox0FifoInParams->writeCounter % FW_GENRISC_RX_HANDLER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES)]), 
				message, 
				(sizeof(FwRxHandlerMailbox0FifoInMessage_t) >> 0x2));

	pFwRxHandlerMailbox0FifoInParams->writeCounter++;

	/* set interrupt to genrisc */
	irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x0); //bit 0 for mailbox 0
	RegAccess_Write(REG_MAC_GENRISC_RX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *RxHandlerInterface_Mailbox0_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwRxHandlerMailbox0FifoOutMessage_t *pBody;

	if (pFwRxHandlerMailbox0FifoOutParams->readCounter != pFwRxHandlerMailbox0FifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwRxHandlerMailbox0FifoOutMessage_t));
		pBody = (FwRxHandlerMailbox0FifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwRxHandlerMailbox0FifoOutParams->messageArray[(pFwRxHandlerMailbox0FifoOutParams->readCounter % FW_GENRISC_RX_HANDLER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwRxHandlerMailbox0FifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwRxHandlerMailbox0FifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	
	return pMsg;
}
#endif //FW_RX_HANDLER_MAILBOX_0_ENABLED
		
#ifdef FW_RX_HANDLER_MAILBOX_1_ENABLED
void RxHandlerInterface_Mailbox1_Push(FwRxHandlerMailbox1FifoInMessage_t *message)
{
	RegMacGenriscRxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwRxHandlerMailbox1FifoInParams->writeCounter - pFwRxHandlerMailbox1FifoInParams->readCounter) < FW_GENRISC_RX_HANDLER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES);

	memcpy32(&(pFwRxHandlerMailbox1FifoInParams->messageArray[(pFwRxHandlerMailbox1FifoInParams->writeCounter % FW_GENRISC_RX_HANDLER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES)]), 
		message, 
		(sizeof(FwRxHandlerMailbox1FifoInMessage_t) >> 0x2));

	pFwRxHandlerMailbox1FifoInParams->writeCounter++;

	/* set interrupt to genrisc */
	irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x1); //bit 1 for mailbox 1
	RegAccess_Write(REG_MAC_GENRISC_RX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *RxHandlerInterface_Mailbox1_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwRxHandlerMailbox1FifoOutMessage_t *pBody;

	if (pFwRxHandlerMailbox1FifoOutParams->readCounter != pFwRxHandlerMailbox1FifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwRxHandlerMailbox1FifoOutMessage_t));
		pBody = (FwRxHandlerMailbox1FifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwRxHandlerMailbox1FifoOutParams->messageArray[(pFwRxHandlerMailbox1FifoOutParams->readCounter % FW_GENRISC_RX_HANDLER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwRxHandlerMailbox1FifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwRxHandlerMailbox1FifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}

	return pMsg;
}
#endif //FW_RX_HANDLER_MAILBOX_1_ENABLED















