/***********************************************************************************
 File:			RxManager.h
 Module:		Rx Manager
 Purpose: 		To handle all RDs/errors/indication from MAC Rx HW
 Description:	This file contains all definitions and the structures of the Rx manager 
 				which is responsible of handling all the RDs/indications/errors from 
 				MAC Rx HW (in wave500)
************************************************************************************/
#ifndef RX_MANAGER_H_
#define RX_MANAGER_H_
/*---------------------------------------------------------------------------------
/						includes					
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "mhi_ieee_address.h"
#include "CalendarWheel_Api.h"
#include "ConfigurationManager_api.h"
#include "HwGlobalDefinitions.h"
#include "replay_detection.h"


/*---------------------------------------------------------------------------------
/						defines				
/----------------------------------------------------------------------------------*/
#define MAX_NUM_OF_FRAGMENTATIONS_PER_VAP 	3
#if defined ENET_INC_ARCH_WAVE600
#define MAX_NUM_OF_FRAGMENTATIONS_PER_STA	(NUM_OF_TIDS)
#else
#define MAX_NUM_OF_FRAGMENTATIONS_PER_STA	(MAX_NUM_OF_FRAGMENTATIONS_PER_VAP) // In gen5, max num of frags per STA is blocked by the max num of frags per VAP.
#endif
#define RX_MANAGER_RECIPIENT_CALENDAR_WHEEL_SIZE 16
#define RX_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS 250
#define RX_MANAGER_CALENDAR_WHEEL_TIMER (OSAL_TIMERS_MS_TO_K_TICKS(RX_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS))
#define RX_MANAGER_FRAGMENTATION_TIMEOUT_IN_MILISECONDS 3000
#define RX_MANAGER_FRAGMENTATION_TIMEOUT_UNITS (RX_MANAGER_FRAGMENTATION_TIMEOUT_IN_MILISECONDS/RX_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS)
#define RX_MANAGER_CLASS_VIOLATION_TIMEOUT_IN_MILISECONDS 1000
#define RX_MANAGER_CLASS_VIOLATION_TIMEOUT_UNITS (RX_MANAGER_CLASS_VIOLATION_TIMEOUT_IN_MILISECONDS/RX_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS)

#if defined ENET_INC_ARCH_WAVE600
#define RX_MANAGER_NUM_OF_FRAGMENTATION_ENTRIES (HW_NUM_OF_STATIONS * MAX_NUM_OF_FRAGMENTATIONS_PER_STA) // (HW_NUM_OF_VAPS * MAX_NUM_OF_FRAGMENTATIONS_PER_VAP)
#else
#define RX_MANAGER_NUM_OF_FRAGMENTATION_ENTRIES (HW_NUM_OF_VAPS * MAX_NUM_OF_FRAGMENTATIONS_PER_VAP)
#endif

#if defined ENET_INC_ARCH_WAVE600
#define RX_MANAGER_INVALID_FRAGMENTATION_ENTRY 0xFFFF
#else
#define RX_MANAGER_INVALID_FRAGMENTATION_ENTRY 0xFF
#endif
#define RX_MANAGER_MULTICAST_REKEY_TIMEOUT_IN_MILISECONDS 1000
#define RX_MANAGER_MULTICAST_REKEY_TIMEOUT_UINTS (RX_MANAGER_MULTICAST_REKEY_TIMEOUT_IN_MILISECONDS/RX_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS)
#define RX_MANAGER_ROBUST_VALIDITY_OPTIONS 8
#define RX_MANAGER_ROBUST_TABLE_VALIDITY_MECHANISM_SHIFT 2
#define RX_MANAGER_ROBUST_TABLE_VALIDITY_PROTECTED_SHIFT 1
#define RX_MANAGER_TID_SET 1
#define RX_MANAGER_ROBUST 1
#define RX_MANAGER_NONE_ROBUST 0


#define RX_MANAGER_FW_CLASS_VIOLATION_CACHE_SIZE 16
#define RX_MANAGER_FW_CLASS_VIOLATION_CACHE_FULL ((0x1 << RX_MANAGER_FW_CLASS_VIOLATION_CACHE_SIZE) - 1)
#define RX_MANAGER_FW_CLASS_VIOLATION_CACHE_MASK (RX_MANAGER_FW_CLASS_VIOLATION_CACHE_FULL)
#define RX_MANAGER_FW_CLASS_VIOLATION_MASK_FREE_ENTRY 0x1


#define RX_MANAGER_SMPS_CONTROL_FIELD_MASK 	0x3
#define RX_MANAGER_SMPS_DISABLED		 	0x0
#define RX_MANAGER_SMPS_ENABLED		 		0x1
#define RX_MANAGER_SMPS_SM_MODE_DYNAMIC 	0x3

// RxDescriptor offset for the PN field
	#define PN_OFFSET 						(2 + CONVERT_WORDS_TO_BYTES(5))

/*---------------------------------------------------------------------------------
/						Macros					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data type definitions					
/----------------------------------------------------------------------------------*/
typedef void (*RxManagerListHandler)(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
typedef void (*RxManagerRdHandler)(Rd_t  *rxDescriptor);


typedef enum
{
	RX_MANAGER_ENTRY_TYPE_FRAGMENTATION,
	RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_HW,
	RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_FW,
	RX_MANAGER_ENTRY_TYPE_MULTICAST_REKEY,
	RX_MANAGER_NUM_OF_ENTRY_TYPES,
	RX_MANAGER_MAX_NUM_OF_ENTRY_TYPES        = 0xFF,
} RxManagerEntryType_e;

typedef enum
{
	RX_MANAGER_ACTION_CATEGORY_SPECTRUM_MANAGEMENT              = 0,
	RX_MANAGER_ACTION_CATEGORY_QOS                              = 1,	
	RX_MANAGER_ACTION_CATEGORY_DLS                              = 2,
	RX_MANAGER_ACTION_CATEGORY_BLOCK_ACK                        = 3,
	RX_MANAGER_ACTION_CATEGORY_PUBLIC                           = 4,
	RX_MANAGER_ACTION_CATEGORY_RADIO_MEASUREMENT                = 5,
	RX_MANAGER_ACTION_CATEGORY_FAST_BSS_TRANSITION              = 6,
	RX_MANAGER_ACTION_CATEGORY_HT                               = 7,
	RX_MANAGER_ACTION_CATEGORY_SA_QUERY                         = 8,
	RX_MANAGER_ACTION_CATEGORY_PROTECTED_DUAL_OF_PUBLIC_ACTION  = 9,
	RX_MANAGER_ACTION_CATEGORY_WNM                              = 10,
	RX_MANAGER_ACTION_CATEGORY_UNPROTECTED_WNM                  = 11,
	RX_MANAGER_ACTION_CATEGORY_TDLS                             = 12, /* This value should never be used */
	RX_MANAGER_ACTION_CATEGORY_MESH                             = 13,
	RX_MANAGER_ACTION_CATEGORY_MULTIHOP                         = 14,
	RX_MANAGER_ACTION_CATEGORY_SELF_PROTECTED                   = 15,
	RX_MANAGER_ACTION_CATEGORY_VHT                              = 16,
	RX_MANAGER_ACTION_CATEGORY_TWT                              = 17,
	RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC_PROTECTED        = 18,
	RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC                  = 19,
	RX_MANAGER_NUM_OF_ACTION_CATEGORY,	
	RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY                 = 0xFF,
	RX_MANAGER_MAX_NUM_OF_ACTION_CATEGORY                       = 0xFF,
} RxManagerActionCategory_e;

/*RX Manager STA states*/
typedef enum
{
    RX_MANAGER_DB_STA_CLOSED,
	RX_MANAGER_DB_STA_WAIT_FILTER,
    RX_MANAGER_DB_STA_OPEN,
    RX_MANAGER_DB_STA_NUM_STATES
} rxManagerDbStaState_e;

typedef struct _RxManagerTimeoutEntry_t
{	
	CalendarWheelEntry_t calendarWheelEntry;	
	
	uint8 type;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 reserved0;
	uint16 entryIndex;
#else	
	uint8 entryIndex;
	uint8 reserved0[2];
#endif
	union
	{
		//RxManagerFragmentationTimeoutEntry
		struct 
		{
			uint16 sequenceNumber;
#ifdef ENET_INC_ARCH_WAVE600
			uint16 stationId	:9;
			uint16 tid 			:4;
			uint16 reserved		:3;
#else
			uint8 stationId;
			uint8 tid;
#endif //ENET_INC_ARCH_WAVE600

			uint8 vapId;
#ifdef ENET_INC_ARCH_WAVE600
			uint8 reserved1;
			uint16 indexInStationArray;
#else
			uint8 indexInStationArray;
			uint8 reserved1[2];
#endif			
		}; 

		//RxManagerHwClassViolationTimeoutnEntry_t
		struct
		{	
			uint8 hwClassVapId;
		};


		//RxManagerFwClassViolationTimeoutnEntry_t
		struct
		{	
			IEEE_ADDR stationMacAddress;
			uint8 reserved2[2];/* This field is meant for padding so the size will be 2^n. This way division is avoided in calendar wheel for finding the index */
		};
		//RxManagerMulticastRekeyTimeoutnEntry_t
		struct
		{	
			uint8 prevKeyId;
			uint8 currKeyId;
			uint8 lastRxKeyId;
			bool timerActive;
		};		
	};
}RxManagerTimeoutEntry_t;

typedef struct _RxManagerTimeoutnEntriesDataBase_t
{	
	RxManagerTimeoutEntry_t rxManagerFragmentationEntries[RX_MANAGER_NUM_OF_FRAGMENTATION_ENTRIES];
#if defined ENET_INC_ARCH_WAVE600
	RxManagerTimeoutEntry_t rxManagerHwClassViolationEntries[NUM_OF_CONFIGURATION_MANAGER_BANDS][16];//[RX_CLASSIFIER_FRAME_CLASS_VIOLATION_CACHE_SIZE]
#else
	RxManagerTimeoutEntry_t rxManagerHwClassViolationEntries[16];
#endif
	RxManagerTimeoutEntry_t rxManagerFwClassViolationEntries[RX_MANAGER_FW_CLASS_VIOLATION_CACHE_SIZE];
	RxManagerTimeoutEntry_t rxManagerMulticastRekeyEntries[HW_NUM_OF_VAPS];
} RxManagerTimeoutnEntriesDataBase_t;

typedef struct RxManagerStationParams
{
#if defined ENET_INC_ARCH_WAVE600
	uint16 fragementationEntriesIndexes[MAX_NUM_OF_FRAGMENTATIONS_PER_STA];
#else
	uint8 fragementationEntriesIndexes[MAX_NUM_OF_FRAGMENTATIONS_PER_STA];
#endif	
	uint8 numOfActiveFragmentation;
	uint8 flags;
#ifndef ENET_INC_ARCH_WAVE600
    uint8 highest_tid_wmm;
    uint8 highest_tid_legacy;
#endif //!ENET_INC_ARCH_WAVE600
    rxManagerDbStaState_e rxManagerDbStaState;
} RxManagerStationParams_t;

typedef struct RxManagerVapParams
{
#if defined ENET_INC_ARCH_WAVE600
	uint32 numOfActiveFragmentation;  
#else
	uint8	numOfActiveFragmentation;  
#endif
	uint8	lastPN[NUM_OF_KEY_ID][PN_LENGTH];
} RxManagerVapParams_t;

typedef struct RxManagerGlobalParams
{
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
	uint8	clientModeMcastFilterRingIndex; 	/* Client Mode MCAST filter: points to the next element to be handled after handling a valid RD */
	uint8	clientModeMcastFilterErrorIndex;	/* Client Mode MCAST filter: points to the next element to be handled after handling an errornous RD  */	
#endif
	uint32	dmaForwardDoneCount;
#if defined ENET_INC_ARCH_WAVE600
	uint32 	numOfActiveTimeouts;
#else
	uint8 	numOfActiveTimeouts;
#endif
	uint8	dmaForwardClientId;
	uint8 	numOfActiveVaps;	// Number of VAPs which were added ( we don't separate per band since we don't do anything with it in gen6)
} RxManagerGlobalParams_t;

#endif /* RX_MANAGER_H_ */

