/***********************************************************************************
 File:			RxManager.c
 Module:		Rx Manager 
 Purpose: 		To handle all RDs/errors/indication from MAC Rx HW
 Description:   This file is the implementation of the Rx manager which is responsible 
 				of handling all the RDs/indications/errors from	MAC Rx HW (in wave500)
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RxManager_Api.h"
#include "RxManager.h"
#include "TxHandler_Api.h"
#include "RegAccess_Api.h"
#include "ieee80211.h"
#include "queue_utility.h"
#include "HwQManager_API.h"
#include "protocol.h"
#include "CalendarWheel_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "VapDatabase_Api.h"
#include "StaDatabase_Api.h"
#include "init_ifmsg.h"
#include "um_interface.h"
#include "ShramPacketDescriptors.h"
#include "ServicesHandler_Api.h"
#include "TsManager_API.h"
#include "ShramRxDescriptors.h"
#include "ShramRxDescriptorsPayload.h"
#include "ShramPsManager.h"
#include "stringLibApi.h"
#include "Utils_Api.h"
#include "RxMpduStructure.h"
#include "RxPp_Api.h"
#include "EventsManager_api.h"
#include "HwEventsAndErrors_Api.h"
#include "DmaManager_Api.h"
#include "HwMemoryMap.h"
#include "BSSmanager_API.h"
#include "Dut_Api.h"
#include "lm.h"
#include "lminfra.h"
#include "HostInterface_API.h"
#include "PsManager_Api.h"
#include "GidmManager_Api.h"
#include "GroupManager_API.h"
#ifndef ENET_INC_ARCH_WAVE600
#include "RxClassifier_Api.h"
#include "RxClassifier.h"
#endif
#ifdef ENET_INC_ARCH_WAVE600
#include "RxCoordinator_Api.h"
#include "ConfigurationManager_api.h"
#include "TwtManager_API.h"
#endif
#include "HostInterfaceAcc_Api.h"
#include "Pac_Api.h"
#include "PsSettingRegs.h"
#include "TxSelectorRegs.h"
#include "RegAccess_Api.h"
#include "TxSelector_Api.h"
#include "ieee_address.h"
#include "ShramRxH.h"
#include "VapDatabase_Api.h"
#include "RxHandler.h"
#include "Statistics_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_RX_MANAGER
#define LOG_LOCAL_FID 0

// Ps Settings Mode
#define PS_SETTING_MODE_SELECTIVE_REQUEST	(0) 
#define PS_SETTING_MODE_FULL_REQUEST 		(1) 

#define RX_MANAGER_INVALID_KEY_ID			(0xFF)

#define OMI_FRAME_MIN_LENGTH				(sizeof(OperatingModeNotificationPayload_t))
#define GIDM_FRAME_MIN_LENGTH				(sizeof(GroupIdManagementPayload_t))

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
//#define CLIENT_MODE_MCAST_TRACE 1
//#define REPLAY_DETECTION_TRACE 1
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct rxManDutDb
{
	UmiOperationMode_e	dutOperationMode;
	uint32 dutRxCount;
}rxManDutDb_t;
/*---------------------------------------------------------------------------------
/                           Public Variables                                
/----------------------------------------------------------------------------------*/

rxDescriptorQueue_t forwardRDsPendingList;
extern VapDbObject_t	VapDatabaseObj;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

static void rxManagerRxListNotEmpty(K_MSG *rxManagerMessage);
static void rxManagerFragmentationEvent(K_MSG *rxManagerMessage);
static void rxManagerTimer(K_MSG *rxManagerMessage);
static void rxManagerHwClassViolationEvent(K_MSG *rxManagerMessage);
static void rxManagerClassifierResourceTimeout(K_MSG *rxManagerMessage);
static void rxManagerClassifierConfigureNormalMode(K_MSG *rxManagerMessage);
static void rxManagerClassifierConfigureRssiMode(K_MSG *rxManagerMessage);
static void rxManagerConfigureNormalMode(K_MSG* rxManagerMessage);
static void rxManagerConfigureRssiMode(K_MSG* rxManagerMessage);
static void rxManagerDmaCompleted(K_MSG *rxManagerMessage);
static void rxManagerAddStation(K_MSG *rxManagerMessage);
static void rxManagerStaSetFilter(K_MSG *rxManagerMessage);
static void rxManagerStopTraffic(K_MSG *rxManagerMessage);
static void rxManagerRemoveStation(K_MSG *rxManagerMessage);
static void rxManagerAddVap(K_MSG *pMsg);
static void rxManagerRemoveVap(K_MSG *pMsg);
static void rxManagerLaSetAntsCfm(K_MSG *pMsg);
static void rxManagerPsSettingsFifoNotEmpty(K_MSG* rxManagerMessage);
static void rxManagerHandleBadPtrHostRd(Rd_t *headRxDescriptor, Rd_t *tailRxDescriptor);
static void rxManagerFwListHandler(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
#if defined (RX_LIST_DEBUG)
static void rxManagerDebugRdsListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
static void rxManagerDebugDropRdsListHandler (Rd_t *headRxDescriptor, Rd_t *tailRxDescriptor);
#endif
static void rxManagerDriverListHandler(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
static void rxManagerDataTemporaryForwardToFwListHandler(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
static void rxManagerErrorRdsListHandler(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
static void rxManagerReleaseFwdRdListHandler(Rd_t *headRxDescriptor, Rd_t *tailRxDescriptor);
static void rxManagerIsrAssert(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor);
static void rxManagerRdHandlerAssert(Rd_t  *rxDescriptor);
static void rxManagerMulticastDataRdHandler(Rd_t *rxDescriptor);
static void rxManagerUnsupportedRdHandler(Rd_t  *rxDescriptor);
static void rxManagerControlRdHandler(Rd_t  *rxDescriptor);
static void rxManagerFwManagementRdHandler(Rd_t  *rxDescriptor);
static void rxManagerFwUnassociatedManagementRdHandler(Rd_t  *rxDescriptor);
static void rxManagerUpdateReleaseList(Rd_t  **releaseListHead, Rd_t  **releaseListTail, Rd_t  **currentRxDescriptor, Rd_t  *endOfMsdulist);
static void rxManagerReleaseFragmentationTimeoutEntry (RxManagerTimeoutEntry_t *timeoutEntry);
static void rxManagerMulticastRekeyTimeout(RxManagerTimeoutEntry_t *timeoutEntry);
#ifndef ENET_INC_ARCH_WAVE600D2 
static void rxManagerCheckMulticastRekey(Rd_t *rxDescriptor);
#endif
static void rxManagerAddMulticastRekey(uint8 vapId);
static void rxManagerRemoveMulticastRekey(uint8 vapId);
static uint8 rxManagerConvertCategoryToInternalIndex (uint8 category);
static void rxManagerHandleValidActionFrame(Rd_t  *rxDescriptor, MANAGEMENT_BASIC_FRAME_HEADER *frame, uint8 category);
static void rxManagerHandleOneForwardRd(Rd_t *rxDescriptor , Rd_t *hostRxDescriptor);
static void rxManagerInitializeRxDescriptorsPools(void);
static void rxManagerInitializeTimeoutEntries(void);
static void rxManagerFindFreeCellInStationArray(StaId stationId, RxManagerTimeoutEntry_t *fragmentationEntry);
static void rxManagerNewFragmentationTimeoutEntry(RxManagerTimeoutEntry_t *fragmentationEntry, RxPpFifoFragLine_t *fragmentationEvent,StaId stationId, uint8 vapId);
static void rxManagerFindFragmentationEntryAndRemove(RxPpFifoFragLine_t *fragmentationEvent,StaId stationId);
static uint8 rxManagerIsMacAddressExist(IEEE_ADDR *macAddress);
static uint8 rxManagerFindFreeFwClassViolationEntry(void);
static void rxManagerStaManagerSendConfirm(StaId sid);
static bool rxManagerRemoveCcmpHeader(Rd_t  *rxDescriptor);
bool		rxManagerDoReplayDetectionStaMode(uint8 *lastPN, Rd_t *pRd);
static void rxManagerSaveNewPnStaMode(uint8 *lastPN, uint8 *currentPN);
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
static clientModeMcastFilterRingElement_t * ClientModeMcastFilterLocateRd (Rd_t *rd, uint8 *index);
bool ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket(Rd_t *rd);
void ClientModeMcastFilterClearTheRing(clientModeMcastFilterRingElement_t *pClientModeMcastFilterRingElement);
#endif

#if defined(ENET_INC_ARCH_WAVE600)
static uint8 RxManager_GetStaFrameClass(StaId stationId);
#endif
static void RxManager_UpdateDutOpeationMode(K_MSG* pMsg);
static void RxManager_SetDutRxCount(K_MSG* pMsg);
static void RxManager_GetDutRxCount(K_MSG* pMsg);
static void RxManager_IgnoreRd(Rd_t* rxDesc);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/


static const FunctionEntry_t afpTaskTable[TASK_RX_MANAGER_END - TASK_RX_MANAGER_START]=
{
	{rxManagerRxListNotEmpty,			        DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_RDS_LIST_NOT_EMPTY)},
	{rxManagerFragmentationEvent,		        DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_FRAGMENTATION_EVENT)},		
	{rxManagerHwClassViolationEvent,			DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CLASS_VIOLATION_EVENT)},
	{rxManagerTimer,		                    DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_TIMER)},	
	{rxManagerClassifierResourceTimeout,	    DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CLASSIFIER_RESOURCE_TIMEOUT)},//no msg
	{rxManagerClassifierConfigureRssiMode, 		DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CLASSIFIER_RSSI_MODE)},
	{rxManagerClassifierConfigureNormalMode,	DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CLASSIFIER_NORMAL_MODE)},	
	{rxManagerConfigureRssiMode,				DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CONFIG_RSSI_MODE)},			
	{rxManagerConfigureNormalMode,				DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_CONFIG_NORMAL_MODE)},			
	{rxManagerDmaCompleted,	                    DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_DMA_COMPLETED)},
	{rxManagerAddStation,	                	DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_STA_OPEN)},
	{rxManagerStaSetFilter,						DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_STA_SET_FILTER)},
	{rxManagerStopTraffic,						DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_STA_STOP_TRAFFIC)},
	{rxManagerRemoveStation,	                DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_REMOVE_STATION)},
	{rxManagerPsSettingsFifoNotEmpty,	        DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_PS_SETTINGS_FIFO_NOT_EMPTY)},
	{rxManagerAddVap,							DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_ADD_VAP)},
	{rxManagerRemoveVap,						DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_REMOVE_VAP)},	
	{rxManagerLaSetAntsCfm,						DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_LA_SET_ANTS_CFM)},
	{RxManager_UpdateDutOpeationMode, 			DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_UPDATE_DUT_OPERATION_MODE_IND)},
	{RxManager_SetDutRxCount,					DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_SET_DUT_RX_COUNT_REQ)},
	{RxManager_GetDutRxCount,					DOUBLE_CHECK_MSG_TYPE(RX_MANAGER_GET_DUT_RX_COUNT_REQ)},					
};

static const RxManagerListHandler RxManagerListsHandlersTable[HW_Q_MANAGER_NUM_OF_HIGH_PRIORITY_RX_READY_LISTS] =
{
	 rxManagerFwListHandler 	                     , /* HW_Q_MANAGER_RX_READY_LIST_FW */
	 rxManagerDriverListHandler                      , /* HW_Q_MANAGER_RX_READY_LIST_DRIVER */
	 rxManagerDataTemporaryForwardToFwListHandler    , /* HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW */
	 rxManagerErrorRdsListHandler                    , /* HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS */	
     rxManagerReleaseFwdRdListHandler                , /* HW_Q_MANAGER_RX_READY_LIST_RELEASE_FORWARD_RDS */
     rxManagerIsrAssert                              , /* HW_Q_MANAGER_RX_READY_LIST_BAD_BAD_PTR_RDS */
#if defined (RX_LIST_DEBUG)
 	 rxManagerDebugRdsListHandler                    , /* HW_Q_MANAGER_RX_READY_LIST_DEBUG */	
	 rxManagerDebugDropRdsListHandler				/* HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP */	
#endif 	 
};


static const RxManagerRdHandler RxManagerFwRdsHandlersTable[RD_TYPE_LAST] =
{
 	rxManagerRdHandlerAssert                         , /* RD_TYPE_UNICAST_QOS_DATA */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_NDP */
	rxManagerMulticastDataRdHandler                  , /* RX_DESCRIPTOR_TYPE_MULTICAST_DATA */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_UNICAST_NON_QOS_DATA */
	rxManagerControlRdHandler                        , /* RD_TYPE_CONTROL */
	rxManagerUnsupportedRdHandler                    , /* RD_TYPE_NOT_SUPPORTED */
	rxManagerFwManagementRdHandler                   , /* RD_TYPE_UNICAST_MGMT_TYPE_1 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_UNICAST_MGMT_TYPE_2 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_MULTICAST_MGMT */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_DROP */
	rxManagerFwUnassociatedManagementRdHandler       , /* RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_RESERVED_1 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2 = 12 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_RESERVED_3 */
	rxManagerRdHandlerAssert                         , /* RD_TYPE_RESERVED_4 */ 
	rxManagerRdHandlerAssert                         , /* RD_TYPE_LOGGER */
#if defined (ENET_INC_ARCH_WAVE600)
	rxManagerRdHandlerAssert						 , /* RD_TYPE_BIP */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_UNICAST_NON_QOS_DATA_PROTECTED */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_UNICAST_QOS_DATA_PROTECTED */
	rxManagerFwManagementRdHandler					 , /* RD_TYPE_UNICAST_MGMT_TYPE_1_PROTECTED */ 
	rxManagerRdHandlerAssert						 , /* RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED21 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED22 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED23 */ 
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED24 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED25 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED26 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED27 */ 
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED28 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED29 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_RESERVED30 */
	rxManagerRdHandlerAssert						 , /* RD_TYPE_WPA_PT */ 
#endif
	};	


static const uint8 RxManagerActionFrameClassTable[RX_MANAGER_NUM_OF_ACTION_CATEGORY] =
{
 	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_SPECTRUM_MANAGEMENT */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_QOS */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_DLS */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_BLOCK_ACK */
	FRAME_CLASS_1             , /* RX_MANAGER_ACTION_CATEGORY_PUBLIC */
	FRAME_CLASS_3        	  , /* RX_MANAGER_ACTION_CATEGORY_RADIO_MEASUREMENT */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_FAST_BSS_TRANSITION */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_HT */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_SA_QUERY */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_PROTECTED_DUAL_OF_PUBLIC_ACTION */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_WNM */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_UNPROTECTED_WNM */
	FRAME_CLASS_INVALID       , /* RX_MANAGER_ACTION_CATEGORY_TDLS - This value should never be used (TDLS)*/
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_MESH */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_MULTIHOP */ 
	FRAME_CLASS_1             , /* RX_MANAGER_ACTION_CATEGORY_SELF_PROTECTED */ 
	FRAME_CLASS_3	          , /* RX_MANAGER_ACTION_CATEGORY_VHT */
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_TWT */
	FRAME_CLASS_3	          , /* RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC_PROTECTED */ 
	FRAME_CLASS_3             , /* RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC */	
};	

static const uint8 RxManagerActionFrameRobustTable[RX_MANAGER_NUM_OF_ACTION_CATEGORY] =
{
 	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_SPECTRUM_MANAGEMENT */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_QOS */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_DLS */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_BLOCK_ACK */
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_PUBLIC */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_RADIO_MEASUREMENT */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_FAST_BSS_TRANSITION */
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_HT */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_SA_QUERY */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_PROTECTED_DUAL_OF_PUBLIC_ACTION */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_WNM */
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_UNPROTECTED_WNM */
	RX_MANAGER_NONE_ROBUST,  /* This value should never be used (TDLS)*/
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_MESH */
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_MULTIHOP */ 
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_SELF_PROTECTED */ 
	RX_MANAGER_NONE_ROBUST,	 /* RX_MANAGER_ACTION_CATEGORY_VHT */ 
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_TWT */ 
	RX_MANAGER_ROBUST,       /* RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC_PROTECTED */ 
	RX_MANAGER_NONE_ROBUST,  /* RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC */	
};	


static const uint8 RxManagerActionFrameRobustValidityTable[RX_MANAGER_ROBUST_VALIDITY_OPTIONS] = 
{
	TRUE,  /* robust management is inactive in station, frame not protected, category is not robust */
	TRUE,  /* robust management is inactive in station, frame not protected, category is robust */
	FALSE, /* robust management is inactive in station, frame is protected, category is not robust */
	FALSE, /* robust management is inactive in station, frame is protected, category is robust */
	TRUE,  /* robust management is active in station, frame not protected, category is not robust */
    FALSE, /* robust management is active in station, frame not protected, category is robust */
	FALSE, /* robust management is active in station, frame is protected, category is not robust */
	TRUE,  /* robust management is active in station, frame is protected, category is robust */
}; 



static uint16 RxManagerCalendarWheelSlots[RX_MANAGER_RECIPIENT_CALENDAR_WHEEL_SIZE];
static CalendarWheelParameters_t RxManagerCalendarWheel;
static RxManagerTimeoutnEntriesDataBase_t RxManagerTimeoutnEntriesDataBase;

static uint16 RxManagerBadForwardRdCnt;

static uint16 RxManagerFwClassViolationBitmap;
static uint32 RxManagerFwClassViolationCacheIsFull;
static HwQueueManagerRequestParams_t RxManagerHwQueueManagerRequestParams;


static RxManagerTimeoutEntry_t *RxManagerFreeFragmentationEntriesList;
static RxManagerStationParams_t RxManagerStationDatabase[HW_NUM_OF_STATIONS];
static RxManagerVapParams_t     RxManagerVapDatabase[HW_NUM_OF_VAPS];

static RxManagerGlobalParams_t  RxManagerGlobalParameters;

static Rd_t  *RxManagerForwardToHostHead = NULL;
static Rd_t  *RxManagerForwardToHostTail = NULL;
static Rd_t  *RxManagerHostForwardHead = NULL;
static Rd_t  *RxManagerHostForwardTail = NULL;

static uint32 RxManagerDoneListCount[HW_Q_MANAGER_NUM_OF_HIGH_PRIORITY_RX_READY_LISTS] = {0};

rxManDutDb_t rxManagerDutDb;

#if defined (RX_LIST_DEBUG)
uint32 debug_ucastRDs = 0;
uint32 debug_mcastRDs = 0;
uint32 debug_bcastRDs = 0;
uint32 debug_dropRDs = 0;
#endif

/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

rxManagerRxListNotEmpty 


Description:
------------
handle the event that one of the Rx lists became not empty 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void -
	
**********************************************************************************/
static void rxManagerRxListNotEmpty(K_MSG* rxManagerMessage)
{
	uint16 NotEmptyRxReadyList = 0;
	uint8 listIndex = 0;
	UNUSED_PARAM(rxManagerMessage);
	//ILOG0_V("rxManagerRxListNotEmpty");

	NotEmptyRxReadyList = HwQManager_GetNotEmptyRxReadyLists();
	
	//("rxManagerRxListNotEmpty : NotEmptyRxReadyList=%x",NotEmptyRxReadyList);
 
	while(NotEmptyRxReadyList)
	{
		/* Not using trailing zeros since there are only 4 lists and most of the time only the first 2
		   will have RDs */
		if(HW_Q_MANAGER_LIST_BIT_IS_SET_MASK & NotEmptyRxReadyList)
	    {
			//ILOG0_D("rxManagerRxListNotEmpty, List ID = %d", listIndex);
			ASSERT(listIndex < HW_Q_MANAGER_NUM_OF_HIGH_PRIORITY_RX_READY_LISTS);
			HwQManager_ClearRxListsInt(HW_Q_MANAGER_HIGH_PRIORITY_RX_READY_FIRST_LIST + listIndex);
			/* Add to the list index the first list in the Rx high priority ready list index */
			RxManagerHwQueueManagerRequestParams.dplIndex = listIndex + HW_Q_MANAGER_HIGH_PRIORITY_RX_READY_FIRST_LIST;
			HwQManager_FlushQ(&RxManagerHwQueueManagerRequestParams);
			ASSERT(RxManagerHwQueueManagerRequestParams.pHeadDesc != NULL_RD);
			ASSERT(((Rd_t  *)RxManagerHwQueueManagerRequestParams.pTailDesc)->nextRd == NEXT_RD_NULL);
			RxManagerDoneListCount[listIndex]++;
			RxManagerListsHandlersTable[listIndex]((Rd_t  *)RxManagerHwQueueManagerRequestParams.pHeadDesc,(Rd_t  *)RxManagerHwQueueManagerRequestParams.pTailDesc);			
		}
		NotEmptyRxReadyList >>= HW_Q_MANAGER_SHIFT_TO_NEXT_LIST;
		listIndex++;
	}

#if defined	(ENET_INC_ARCH_WAVE600)
	EventManager_TurnOnEvent(EVENT_ID_HIGH_PRI_RX_PD_READY);		
#else 
	EventManager_TurnOnEvent(EVENT_ID_RX_RD_LIST_NOT_EMPTY);		
#endif
	//("rxManagerRxListNotEmpty Un Masked Interrupt : NotEmptyRxReadyList=%x",NotEmptyRxReadyList);
}



/**********************************************************************************

rxManagerFragmentationEvent 


Description:
------------
handle the event that the FIFO of fragmentation events became not emoty 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerFragmentationEvent(K_MSG* rxManagerMessage)
{
	RxPpFifoFragLine_t fragmentationEvent;
	StaId stationId = 0;
	uint8 vapId = 0;
	RxManagerTimeoutEntry_t *fragmentationEntry = NULL;
	uint32 previousNumberOfActiveTimeouts = 0;	
#if defined (ENET_INC_ARCH_WAVE600)
	uint8 bandId;
#endif

	UNUSED_PARAM(rxManagerMessage);
	memset(&fragmentationEvent, 0, sizeof(fragmentationEvent));

	previousNumberOfActiveTimeouts = RxManagerGlobalParameters.numOfActiveTimeouts;

	/* Get first entry - the FIFO is not empty */
	RxPp_FragmentFifoRead(&fragmentationEvent);
	while(fragmentationEvent.valid)
	{
		SLOG0(0, 0, RxPpFifoFragLine_t, &fragmentationEvent);
		stationId = fragmentationEvent.stationId;
#if defined (ENET_INC_ARCH_WAVE600)
		bandId = ConfigurationManager_GetBandForStation(stationId);
#endif
		if(fragmentationEvent.fragStart)
		{
#if defined (ENET_INC_ARCH_WAVE600)
			GeneralStatistics.defragStart[bandId]++;
#else
			GeneralStatistics.defragStart++;
#endif //ENET_INC_ARCH_WAVE600

			/* Start fragmentation event */
			if ((RxManagerStationDatabase[stationId].rxManagerDbStaState == RX_MANAGER_DB_STA_OPEN) ||
				(RxManagerStationDatabase[stationId].rxManagerDbStaState == RX_MANAGER_DB_STA_WAIT_FILTER))
			{
				StaDb_GetVapId(stationId, &vapId);
				// In gen	6 we don't have limitation on the amount of fragmantation we can hold. We have an entry for all STAs * TIDs.
#ifndef ENET_INC_ARCH_WAVE600
				if(RxManagerVapDatabase[vapId].numOfActiveFragmentation < MAX_NUM_OF_FRAGMENTATIONS_PER_VAP)
				{
#endif					
					/* Take the first free entry from the free entries list */
					fragmentationEntry = RxManagerFreeFragmentationEntriesList;
					RxManagerFreeFragmentationEntriesList = RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries + RxManagerFreeFragmentationEntriesList->calendarWheelEntry.nextIndex;

                    /* Find free cell in the station array */
					rxManagerFindFreeCellInStationArray(stationId, fragmentationEntry);

					/* Add new entry - fill parameters update counters and start to count timeout */
					rxManagerNewFragmentationTimeoutEntry(fragmentationEntry, &fragmentationEvent, stationId, vapId);		
#ifndef ENET_INC_ARCH_WAVE600
				}
				else
				{
					/* No space for fragmentation in this VAP - reject the fragmentation */
					RxPp_UpdateWindowSn(fragmentationEvent.stationId, fragmentationEvent.tid, fragmentationEvent.sn + 1, RXPP_NON_BA_CLIENT_FORCE);
				}
#endif				
			}
			else
			{
				/* Station is not connected - flush fragments */
				RxPp_UpdateWindowSn(fragmentationEvent.stationId, fragmentationEvent.tid, fragmentationEvent.sn + 1, RXPP_NON_BA_CLIENT_FORCE);
			}
		}
		else
		{			
#if defined (ENET_INC_ARCH_WAVE600)
			GeneralStatistics.defragEnd[bandId]++;
#else
			GeneralStatistics.defragEnd++;
#endif //ENET_INC_ARCH_WAVE600
			
			/* End fragmentation event (succeeded or failed) - find entry if exist and remove */
			rxManagerFindFragmentationEntryAndRemove(&fragmentationEvent, stationId);
		}
		RxPp_FragmentFifoRead(&fragmentationEvent);
	}

	/* Unmasked the interrupt */
	EventManager_TurnOnEvent(EVENT_ID_FRAGMENT_FIFO_NOT_EMPTY);

  	if((0 == RxManagerGlobalParameters.numOfActiveTimeouts) && (0 != previousNumberOfActiveTimeouts))
	{	
		/* Cancel the timer since there are no entries and timer is active */
		OSAL_RESET_TIMER_EXPLICIT(RX_MANAGER_TIMER, TASK_RX_MANAGER);
	}
	else if((0 == previousNumberOfActiveTimeouts) && (0 != RxManagerGlobalParameters.numOfActiveTimeouts))
    {
		/* Setting the timer since it was not active and there are new entries */
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
    }
}


/**********************************************************************************

rxManagerTimer 


Description:
------------
handle the event that the Rx manager timer has occured - fragmentations timeouts and class 
violation cache timeouts should be checked


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerTimer(K_MSG* rxManagerMessage)
{	
	uint16 currentIndex = 0;
	RxManagerTimeoutEntry_t *currentTimeoutEntry = NULL;
	uint16 bitmapMask = 0;
	UNUSED_PARAM(rxManagerMessage);
	currentIndex = CalendarWheel_TimerEvent(&RxManagerCalendarWheel);
	while(CALENDAR_WHEEL_NULL_ENTRY_INDEX != currentIndex)
	{
		currentTimeoutEntry = ((RxManagerTimeoutEntry_t *)(&RxManagerTimeoutnEntriesDataBase)) + currentIndex;
		currentIndex = currentTimeoutEntry->calendarWheelEntry.nextIndex;
    	switch (currentTimeoutEntry->type)
		{
			/* Since there are only 4 types use switch case */
			case RX_MANAGER_ENTRY_TYPE_FRAGMENTATION:
#ifdef ENET_INC_ARCH_WAVE600		
				GeneralStatistics.defragTimeout[ConfigurationManager_GetBandForStation(currentTimeoutEntry->stationId)]++;
#else
				GeneralStatistics.defragTimeout++;
#endif //ENET_INC_ARCH_WAVE600

				RxPp_UpdateWindowSn(currentTimeoutEntry->stationId, 
					              	currentTimeoutEntry->tid, 
					              	(currentTimeoutEntry->sequenceNumber + 1), 
					              	RXPP_NON_BA_CLIENT_FORCE);
				rxManagerReleaseFragmentationTimeoutEntry(currentTimeoutEntry);
				break;
			case RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_HW:
#ifdef ENET_INC_ARCH_WAVE600
				RxCoordinator_ResetClass3ViolationValidEntry(currentTimeoutEntry->entryIndex, currentTimeoutEntry->hwClassVapId); 
#else
				RxClassifier_ResetClass3ViolationValidEntry(currentTimeoutEntry->entryIndex);
#endif
				RxManagerGlobalParameters.numOfActiveTimeouts --;	
				break;
			case RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_FW:
				/* Free the entry in the FW class violation cache - clear the occupied bit in the bitmap */
				bitmapMask = RX_MANAGER_FW_CLASS_VIOLATION_MASK_FREE_ENTRY << currentTimeoutEntry->entryIndex;
				RxManagerFwClassViolationBitmap &= ~bitmapMask;
				
				RxManagerGlobalParameters.numOfActiveTimeouts --;	
				break;
			case RX_MANAGER_ENTRY_TYPE_MULTICAST_REKEY:
				rxManagerMulticastRekeyTimeout(currentTimeoutEntry);				
				break;
				
			default:
				DEBUG_FATAL("RX Manager Timer Unknown Type");		
		}			
	}
	
    /* If there are entries start the timer again */
	if(0 != RxManagerGlobalParameters.numOfActiveTimeouts)
	{	
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
	}
}

/**********************************************************************************

rxManagerClassViolationEvenet 


Description:
------------
handle the event that the Rx classifier encountered a frame with class violation
from a new station

Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerHwClassViolationEvent(K_MSG* rxManagerMessage)
{

	RxManagerTimeoutEntry_t *hwClassViolationEntry = NULL;
	classViolationDetails_t classViolationEvent;
	UMI_FRAME_CLASS_ERROR frameClassErrorMessage;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 bandId;
#endif
	uint8 vapId =  rxManagerMessage->header.vapId;

    memset(&frameClassErrorMessage, 0, sizeof(UMI_FRAME_CLASS_ERROR));
    memset(&classViolationEvent, 0, sizeof(classViolationDetails_t));
	
	if(0 == RxManagerGlobalParameters.numOfActiveTimeouts)
	{	
		/* The timer is not set and there should be at least one new entry */
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
	}

#ifdef ENET_INC_ARCH_WAVE600
	RxCoordinator_ReadClassViolationCache(&classViolationEvent,vapId); 
	bandId = ConfigurationManager_GetBandForVap(vapId);
#else
	RxClassifier_ReadClassViolationCache(&classViolationEvent); 
#endif
	while(classViolationEvent.valid)
	{	
#ifdef ENET_INC_ARCH_WAVE600		
		GeneralStatistics.classViolationErrors[bandId]++;
#else
		GeneralStatistics.classViolationErrors++;
#endif //ENET_INC_ARCH_WAVE600

		if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
		{
			frameClassErrorMessage.frameClassErrorEntries[frameClassErrorMessage.u8numOfValidEntries].u8vapIndex = classViolationEvent.vapId & 0xF;	
		}
		else
		{
			frameClassErrorMessage.frameClassErrorEntries[frameClassErrorMessage.u8numOfValidEntries].u8vapIndex = classViolationEvent.vapId; 
		}
		
		vIEEE_ADDR_CopyAddr(&(frameClassErrorMessage.frameClassErrorEntries[frameClassErrorMessage.u8numOfValidEntries].sAddr), &classViolationEvent.macAddress);
		
		/* Each Rx manager class violation entry is injective to entry in the Rx classifer/Coordinator cache */
#ifdef ENET_INC_ARCH_WAVE600
		hwClassViolationEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[bandId][classViolationEvent.indexInCache];
#else
		hwClassViolationEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[classViolationEvent.indexInCache];
#endif
		hwClassViolationEntry->hwClassVapId = classViolationEvent.vapId;
	
        /* Start counting timeout for the cache cleaning */
		CalendarWheel_AddEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)hwClassViolationEntry, RX_MANAGER_CLASS_VIOLATION_TIMEOUT_UNITS);

		frameClassErrorMessage.u8numOfValidEntries ++;
	    if(frameClassErrorMessage.u8numOfValidEntries == MAX_NUMBER_FRAME_CLASS_ERROR_ENTRIES_IN_MESSAGE)
	    {
			
			vIF_SendMsg(&sUmiIndFreeQueue,
						UMI_MC_MAN_CLASS3_ERROR_IND,
						TASK_UM_IF_TASK,
						&frameClassErrorMessage,
						sizeof(UMI_FRAME_CLASS_ERROR), 
						vapId); 
			frameClassErrorMessage.u8numOfValidEntries = 0;
	    }
	
		RxManagerGlobalParameters.numOfActiveTimeouts ++;
#ifdef ENET_INC_ARCH_WAVE600
		RxCoordinator_ReadClassViolationCache(&classViolationEvent,vapId); 
#else
		RxClassifier_ReadClassViolationCache(&classViolationEvent); 
#endif
	}

	if(frameClassErrorMessage.u8numOfValidEntries)
	{
		/* The last message was not full so it wasnt sent - sent it now */
		vIF_SendMsg(&sUmiIndFreeQueue,
					UMI_MC_MAN_CLASS3_ERROR_IND,
					TASK_UM_IF_TASK,
					&frameClassErrorMessage,
					sizeof(UMI_FRAME_CLASS_ERROR), 
					vapId);
	}
	
	// Turn on the event
#ifdef ENET_INC_ARCH_WAVE600
	if (bandId == CONFIGURATION_MANAGER_BAND_0)
	{
		EventManager_TurnOnEvent(EVENT_ID_CLASS_VIOLATION_BAND0);
	}
	else
	{
		EventManager_TurnOnEvent(EVENT_ID_CLASS_VIOLATION_BAND1);
	}
#else
	EventManager_TurnOnEvent(EVENT_ID_CLASS_VIOLATION);
#endif

}


/**********************************************************************************

rxManagerClassifierResourceTimeout


Description:
------------
handle the event that the Rx classifer has waited for resource more than a 
defined time 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerClassifierResourceTimeout(K_MSG* rxManagerMessage)
{
	UNUSED_PARAM(rxManagerMessage);
#ifndef ENET_INC_ARCH_WAVE600
   //TBD6 add the code that releases RDs??
   DEBUG_FATAL("RX Manager Classifier Resource Timeout");

   EventManager_TurnOnEvent(EVENT_ID_RX_CLASSIFIER_RXH_AVAILABLE_PENDING_RTD);

#endif
}


/**********************************************************************************

rxManagerClassifierConfigureNormalMode 


Description:
------------
			Wraper for rxManagerClassifierConfigureNormalMode


Input: 
-----		K_MSG
	
		
Output:
-------
			None

Returns:
--------
			void  
	
**********************************************************************************/
static void rxManagerClassifierConfigureNormalMode(K_MSG* rxManagerMessage)
{
#ifdef ENET_INC_ARCH_WAVE600
	UNUSED_PARAM(rxManagerMessage);	
	ASSERT(0);// - need to add support to rssi in wave600
#else
	//configure Rx-Classifer in NORMAL_AFTER_RSSI mode
	RxClassifier_ConfigureNormalMode();

#ifdef BEEROCK_DEBUG
	ILOG0_V("[rxManagerClassifierConfigureNormalMode]--------------------end]");
#endif

	//send message to RX Manager
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CONFIG_NORMAL_MODE, TASK_RX_MANAGER, rxManagerMessage->header.vapId);
#endif
}

/**********************************************************************************

rxManagerClassifierConfigureRssiMode 


Description:
------------
			Wraper for rxManagerClassifierConfigureRssiMode


Input: 
-----		K_MSG
	
		
Output:
-------
			None

Returns:
--------
			void  
	
**********************************************************************************/
static void rxManagerClassifierConfigureRssiMode(K_MSG* rxManagerMessage)
{
	UNUSED_PARAM(rxManagerMessage);	
#ifdef ENET_INC_ARCH_WAVE600
	ASSERT(0);//need to add support to rssi in wave600
#else	
	//configure Rx-Classifer in RSSI mode
	RxClassifier_ConfigureRssiMode();

#ifdef BEEROCK_DEBUG
	ILOG0_V("[rxManagerClassifierConfigureRssiMode]----------------------end]");
#endif

	//send message to RX Manager
	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_CONFIG_RSSI_MODE, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);		
#endif
}

/**********************************************************************************
rxManagerConfigureNormalMode 


Description:
------------
			configure RX Manager to send notification to Host regarding frame class violation as i normal mode


Input: 
-----	
			
	
		
Output:
-------
			

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerConfigureNormalMode(K_MSG* rxManagerMessage)
{
	K_MSG *pMsg = NULL;
	ProcessRequestParams_t *processRequestParams = NULL;

	//request process manager to finish process & resume all
	pMsg = OSAL_GET_MESSAGE(sizeof(ProcessRequestParams_t));

	processRequestParams = (ProcessRequestParams_t*)pK_MSG_DATA(pMsg);

	processRequestParams->processId 					= PROCESS_ID_CHANNEL_SWITCH;
	processRequestParams->endProcessMsg					= CHANNEL_SWITCH_MANAGER_PROCESS_FINISH_CFM;
	processRequestParams->requesterParams				= NULL;
	processRequestParams->postProcessServiceBitmap 		= TRUE << SERVICE_ID_RESUME_ALL;
	processRequestParams->returnTask					= TASK_CHANNEL_SWITCH_MANAGER;
	processRequestParams->updateParamsBeforeFinalizing 	= TRUE;
	processRequestParams->serviceData       			= 0;
	processRequestParams->vapId							= rxManagerMessage->header.vapId; 	
	processRequestParams->dualBandProcess					= FALSE;
	processRequestParams->processMsgHandledByCdbProcessMan	= FALSE;
#ifdef BEEROCK_DEBUG
	ILOG0_V("[rxManagerConfigureNormalMode]------------------------------end]");
#endif

	OSAL_SEND_MESSAGE(PROCESS_MANAGER_PROCESS_EXCUTION_FINISHED, TASK_PROCESS_MANAGER, pMsg, processRequestParams->vapId);
}

/**********************************************************************************
rxManagerConfigureRssiMode 


Description:
------------
			configure RX Manager NOT to send notification to Host regarding frame class violation


Input: 
-----	
			
	
		
Output:
-------
			

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerConfigureRssiMode(K_MSG* rxManagerMessage)
{
	OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_SET_RSSI_CFM, TASK_CHANNEL_SWITCH_MANAGER, rxManagerMessage->header.vapId);

#ifdef BEEROCK_DEBUG
	ILOG0_V("[rxManagerConfigureRssiMode]--------------------------------end]");
#endif
}

/**********************************************************************************

rxManagerDmaCompleted 


Description:
------------
handle the event that the DMA from SHRAM to DDR has finished 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
release the Rx descriptor of the FW	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerDmaCompleted(K_MSG* rxManagerMessage)
{
	TX_INTERRUPT_SAVE_AREA;	
	Rd_t  *rxDescriptor;
	uint32 dmaManagerTransferCounter = 0;
	uint8 clientId = RxManagerGlobalParameters.dmaForwardClientId;
	Rd_t  *releaseListHead = NULL, *releaseListTail = NULL;
	Rd_t  *forwardHead = NULL, *forwardTail = NULL;
	UNUSED_PARAM(rxManagerMessage);
	
	/*The payload has been transferred from SHRAM to DDR*/
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	// Get current DMA counter (if more will be added, we will handle it in the next MSG)
	dmaManagerTransferCounter = DmaManager_GetTransferCount(clientId);
	while (dmaManagerTransferCounter != RxManagerGlobalParameters.dmaForwardDoneCount)
	{
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		/*Increment DMA forward count*/
		RxManagerGlobalParameters.dmaForwardDoneCount++;
		/*Need to free FW RDs*/
		ASSERT(RxManagerForwardToHostHead != NULL);
		rxDescriptor = RxManagerForwardToHostHead;
		if (RxManagerForwardToHostHead == RxManagerForwardToHostTail)
		{
			RxManagerForwardToHostHead = NULL;
			RxManagerForwardToHostTail = NULL;
		}
		else
		{
			RxManagerForwardToHostHead = (Rd_t *)GET_NEXT_RD(rxDescriptor);
		}
		/*FW RDs are linked*/
		if (releaseListHead == NULL)
		{
			releaseListHead = rxDescriptor;
		}
		releaseListTail = rxDescriptor;
		/*Need to forward Host RDs*/
		ASSERT(RxManagerHostForwardHead != NULL);
		rxDescriptor = RxManagerHostForwardHead;
		if (RxManagerHostForwardHead == RxManagerHostForwardTail)
		{
			RxManagerHostForwardHead = NULL;
			RxManagerHostForwardTail = NULL;
		}
		else
		{
			RxManagerHostForwardHead = (Rd_t *)GET_NEXT_RD(rxDescriptor);
		}
		/*Host RDs are linked*/
		if (forwardHead == NULL)
		{
			forwardHead = rxDescriptor;
		}
		forwardTail = rxDescriptor;
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		// Get updated done counter from DMA Manager
		dmaManagerTransferCounter = DmaManager_GetTransferCount(clientId);
	}
	// Allow DMA Manager to send MSG again (it was disabled since the MSG was sent)
	EventManager_TurnOnEvent(EVENT_ID_DMA_FORWARD);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	DEBUG_ASSERT(releaseListHead != NULL);
	DEBUG_ASSERT(forwardHead != NULL);
	/*Now release FW list*/
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = releaseListHead;
	RxManagerHwQueueManagerRequestParams.pTailDesc = releaseListTail;
	HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
	/*Make sure forward tail points to NULL RD*/
	ASSERT(forwardTail!= NULL);
	forwardTail->nextRd = NEXT_RD_NULL;
	/*Send the forward RDs to Driver*/
	rxManagerDriverListHandler(forwardHead, forwardTail);
}


/**********************************************************************************
RxManager_GetHighestTidForPsType

Description:
------------
    find the highest tid works in WMM and in Legacy PS type. 
Input:
-----
    station index , psManagerStatusFifoEntry
Output:
-------
    None
Returns:
--------
    none
**********************************************************************************/

#ifndef ENET_INC_ARCH_WAVE600
static void RxManager_GetHighestTidForPsType(StaId staIndex, uint8 uapsd)
{
    uint8 highestTid_wmm = NUM_OF_TID;
    uint8 highestTid_legacy = NUM_OF_TID;
    uint8 tidIndex = 0;
    uint8 acIndex;
    
    /*  Qos Info filed:     |      7        |       65              |   4       | 3   2   1   0  |
                            |reserved       |   max SP length       | reserved  |BE  BK  VI  VO  | */      

    while (tidIndex < NUM_OF_TID)
    {
        acIndex = au8Convert8021dToWmeUserPriority[tidIndex];
        if (uapsd & (1 << ((ACCESS_CATEGORY_NUM -1) - acIndex)))
        {
            highestTid_wmm = tidIndex;
        }   
        else
        {
            highestTid_legacy = tidIndex;
        }
        tidIndex++;
    }
    RxManagerStationDatabase[staIndex].highest_tid_legacy = highestTid_legacy;
    RxManagerStationDatabase[staIndex].highest_tid_wmm = highestTid_wmm;
}
#endif //!ENET_INC_ARCH_WAVE600

 
/**********************************************************************************

rxManagerRemoveStation 


Description:
------------
handle the event that a station should be removed 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerAddStation(K_MSG *rxManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(rxManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	StaId sid = pAddSta->u16SID;
#ifndef ENET_INC_ARCH_WAVE600
	uint8 uapsd = pAddSta->u8UAPSD_Queues;
#endif //!ENET_INC_ARCH_WAVE600

	uint8 isQosStation = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_WME);
	uint8 isXFilterOpen = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);

	if (isXFilterOpen == TRUE)
	{
    	RxManagerStationDatabase[sid].rxManagerDbStaState = RX_MANAGER_DB_STA_OPEN;
	}
	else
	{
    	RxManagerStationDatabase[sid].rxManagerDbStaState = RX_MANAGER_DB_STA_WAIT_FILTER;
	}
	/*Store flags for later use*/
	RxManagerStationDatabase[sid].flags = pAddSta->u8Flags;

#ifndef ENET_INC_ARCH_WAVE600
    RxManager_GetHighestTidForPsType(sid, uapsd);
#endif //!ENET_INC_ARCH_WAVE600

    /*reset RXPP*/
	RxPp_ResetStation(sid, isQosStation, isXFilterOpen);
	
	/*Send Confirmation to STA Manager*/
	rxManagerStaManagerSendConfirm(sid);
}

/**********************************************************************************

rxManagerStaSetFilter 


Description:
------------
handle the event that the filter is opened


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerStaSetFilter(K_MSG *rxManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(rxManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_802_1X_FILTER *pSetFilter = (UMI_802_1X_FILTER *)pK_MSG_DATA(pMsg);
	StaId sid = pSetFilter->u16SID;

	RxManagerStationDatabase[sid].rxManagerDbStaState = RX_MANAGER_DB_STA_OPEN;
    RxPp_SetForwardStatus(sid, TID_BITMAP_ALL_TIDS, RXPP_FORWARD_ALL_TIDS_TO_HOSTIF);
	/*Send Confirmation to STA Manager*/
	rxManagerStaManagerSendConfirm(sid);
}

/**********************************************************************************

rxManagerStopTraffic 


Description:
------------
handle the event that traffic to a STA should be stopped


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerStopTraffic(K_MSG *rxManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(rxManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(pMsg);
	StaId sid = pStopTraffic->u16SID;
	
    RxPp_SetForwardStatus(sid, TID_BITMAP_ALL_TIDS, RXPP_FORWARD_ALL_TIDS_TO_UMAC);
	/*Send Confirmation to STA Manager*/
	rxManagerStaManagerSendConfirm(sid);
}

/**********************************************************************************

rxManagerRemoveStation 


Description:
------------
handle the event that a station should be removed 


Input: 
-----	
rxManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerRemoveStation(K_MSG* rxManagerMessage)
{
   BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(rxManagerMessage);
   K_MSG *pMsg = staManagerReq->psMsg;
   UMI_STA_REMOVE *pRemoveSta = (UMI_STA_REMOVE *)pK_MSG_DATA(pMsg);
   StaId stationId = pRemoveSta->u16SID;
   uint8 vapId = 0;
   uint16 tidBitmap = 0;
   RxManagerTimeoutEntry_t *fragmentationEntry = NULL;
   uint16 entryIndex = RX_MANAGER_INVALID_FRAGMENTATION_ENTRY;
   uint32 j = 0;

   StaDb_GetVapId(stationId, &vapId);

   /* First check if there are fragmentations - in most cases there will be no fragmentation so
      there is no point to check the array */   
   if(RxManagerStationDatabase[stationId].numOfActiveFragmentation)
   {
	   for(j = 0; j < MAX_NUM_OF_FRAGMENTATIONS_PER_STA; j++)
	   {
			entryIndex = RxManagerStationDatabase[stationId].fragementationEntriesIndexes[j];
			if(RX_MANAGER_INVALID_FRAGMENTATION_ENTRY != entryIndex)
			{
				fragmentationEntry = RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries + entryIndex;
				/* Stop counting timeout for this fragmentation */
				CalendarWheel_RemoveEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t*)fragmentationEntry);
				
				/* The station/TID queue can be flushed also by the TS manager. This doesnt cause a race
				   since that when removing the station the queue should be flushed and flushing an empty
				   queue wont cause to anything */ 
				   
				tidBitmap |= RX_MANAGER_TID_SET << fragmentationEntry->tid;
				rxManagerReleaseFragmentationTimeoutEntry(fragmentationEntry);
			}	
	   }

	   RxPp_FlushMultiTid(stationId, tidBitmap, RXPP_DISCARD, RXPP_NON_BA_CLIENT_FORCE);
   
	   /* Cancel the timer since it was set - there were entries in the station */
	   if(0 == RxManagerGlobalParameters.numOfActiveTimeouts)
       {	
	       	OSAL_RESET_TIMER_EXPLICIT(RX_MANAGER_TIMER, TASK_RX_MANAGER);
	   }
   }

   RxManagerStationDatabase[stationId].rxManagerDbStaState = RX_MANAGER_DB_STA_CLOSED;
   /*Send Confirmation to STA Manager*/
   rxManagerStaManagerSendConfirm(stationId);
}


#ifndef ENET_INC_ARCH_WAVE600
/**********************************************************************************
rxManagerPsSettingsSendNdpPd

Description:
------------
    prepares NDP PD and push it to the relevant Q. 
Input:
-----
    psManagerStatusFifoEntry
Output:
-------
    None
Returns:
--------
    none
**********************************************************************************/
static void rxManagerPsSettingsSendNdpPd(PsManagerStatusFifoEntry_t psManagerStatusFifoEntry, 	StaId stationId, uint8 HighestTidIndex)
{

    TxPd_t* pDesc;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

    uint8 vapId;
    uint32 currentTsf;

    StaDb_GetVapId(stationId, &vapId);
                
    /*Get PD from Management from PS Settings Pool*/
    pDesc = ResourceManager_GetDescriptor(DESC_POOL_PS_SETTINGS_NDP);
	ASSERT((TxPd_t *)NULL_PD != pDesc);        

    pDesc->nextPdPointer = NEXT_PD_NULL;
    pDesc->aMsduTailPointer = 0;
    pDesc->pdCounter = 0;
    pDesc->mcUnicast = 0;
    pDesc->pdType = PD_TYPE_NDP;
    pDesc->retransmissionIndication = 0;
    pDesc->aggregationIndication = 0;
    pDesc->dataLength = 0;
    pDesc->ethType = 0;
    pDesc->packetPointer = 0;
    pDesc->txQTid = HighestTidIndex;         
    pDesc->txQStaId = stationId;       
    pDesc->txQVapId = vapId ;       
    pDesc->txQGroupId = HW_TX_Q_TYPE_STA_TID;        
#if defined (ENET_INC_ARCH_WAVE600)
	currentTsf = Pac_TimGetTsfLowPerBand(ConfigurationManager_GetBandForVap(pDesc->txQVapId));
#else
	currentTsf = Pac_TimGetTsfLow();
#endif
    pDesc->ttlCount = HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(currentTsf);     
    pDesc->status = 0;      

 	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.pHeadDesc = pDesc;
	hwQueueManagerRequestParams.secondaryAddr = HighestTidIndex;
	hwQueueManagerRequestParams.primaryAddr = stationId;
    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
     
    HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
    
    }


/**********************************************************************************

rxManagerPsSettingsUpdatetxSelectorBitmap 

Description:
------------
	Update TxSelector Bitmap 
Input: 
-----	
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerPsSettingsUpdatetxSelectorBitmap(StaId stationId, uint8 tid)
{
    TxSelectorActionParams_t txSelectorActionParams;
        
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID; 
    txSelectorActionParams.stationOrVapNum = stationId;
    //Assuming psRequestMode is configured as PS_SETTING_MODE_SELECTIVE_REQUEST
    txSelectorActionParams.allTids = TX_SELECTOR_SINGLE_TID_UPDATE;
    txSelectorActionParams.tidOrAc = tid;
    txSelectorActionParams.action = TX_SELECTOR_PS_MODE_POWER_SAVE;

   TxSelector_SetPowerSaveRequest(&txSelectorActionParams);
}

#endif //!ENET_INC_ARCH_WAVE600


/**********************************************************************************

rxManagerPsSettingsFifoNotEmpty 

Description:
------------
	Read and process Ps Setting Fifo 
Input: 
-----	
	rxManagerMessage - Empty Message (No DAta Message)
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerPsSettingsFifoNotEmpty(K_MSG* rxManagerMessage)
{	
#ifndef ENET_INC_ARCH_WAVE600
	PsManagerFifoParams_t psManagerFifoParams;
	PsManagerStatusFifoEntry_t psManagerStatusFifoEntry;
	uint8 fifoIndex;
    StaId stationId;
    uint8 highestTid;

	PsManager_GetFifoParameters(&psManagerFifoParams);

	for(fifoIndex = 0; fifoIndex < psManagerFifoParams.numOfValidEntries; fifoIndex++)
	{
		psManagerStatusFifoEntry = PsManagerStatusFifo[psManagerFifoParams.currentIndexInFifo];

		SLOG0(0, 0, PsManagerStatusFifoEntry_t, &psManagerStatusFifoEntry);
        stationId = psManagerStatusFifoEntry.addr2Index;
        highestTid = psManagerStatusFifoEntry.psduPsType ? RxManagerStationDatabase[stationId].highest_tid_wmm : RxManagerStationDatabase[stationId].highest_tid_legacy;

        ILOG2_D("rxManagerPsSettingsFifoNotEmpty %x", psManagerStatusFifoEntry);

		if(psManagerStatusFifoEntry.noFreeNdpPd)
		{
            rxManagerPsSettingsSendNdpPd(psManagerStatusFifoEntry,stationId, highestTid);
            rxManagerPsSettingsUpdatetxSelectorBitmap(stationId, highestTid);
             // Setting PS Request must reset current SP length to maximum
            if (psManagerStatusFifoEntry.psduPsType)
            {
                StaDB_SetSpLengthToMaximum(stationId);
            }
		}
		else if(psManagerStatusFifoEntry.setPsStatus || psManagerStatusFifoEntry.clearPsStatus)
		{
			ILOG2_DD("rxManagerPsSettingsFifoNotEmpty  Station Changed Ps Mode Set Ps Status %x Clear Ps Status %x", psManagerStatusFifoEntry.setPsStatus,psManagerStatusFifoEntry.clearPsStatus);
//			ILOG0_DD("rxManagerPsSettingsFifoNotEmpty  Station Changed Ps Mode Set Ps Status %x Clear Ps Status %x", psManagerStatusFifoEntry.setPsStatus,psManagerStatusFifoEntry.clearPsStatus);
		}
				
		psManagerFifoParams.currentIndexInFifo++;
		psManagerFifoParams.currentIndexInFifo &= PS_MANAGER_STATUS_FIFO_SIZE_MASK;
	}  

	PsManager_SetFifoParameters(&psManagerFifoParams);

	EventManager_TurnOnEvent(EVENT_ID_PS_SETTING_FIFO_NOT_EMPTY);	
#else
	// We should only be in this function if we are NOT in wave600. In wave600 we should have been in PAC MANAGER in LM.
	UNUSED_PARAM(rxManagerMessage);
	ASSERT(0);
#endif
}


/**********************************************************************************

rxManagerFwListHandler 


Description:
------------
This function handles the list of RDs meant for the FW

Input: 
-----
headRxDescriptor - the first RD in the list
tailRxDescriptor - not used

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerFwListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
	Rd_t  *nextRxDescriptor = NULL;
	Rd_t  *currentRxDescriptor = NULL;
	UNUSED_PARAM(tailRxDescriptor);
	
	currentRxDescriptor = headRxDescriptor;
	while((Rd_t  *)NULL_RD != currentRxDescriptor)
	{
		nextRxDescriptor = (Rd_t  *)GET_NEXT_RD(currentRxDescriptor);
		/* Each RD is sent to the proper module wich is responsible of releasing it */
		RxManagerFwRdsHandlersTable[currentRxDescriptor->rdType](currentRxDescriptor);
		currentRxDescriptor = nextRxDescriptor;
	}
}

#if defined (RX_LIST_DEBUG)
static void rxManagerDebugRdsListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
	Rd_t  *nextRxDescriptor = NULL;
	Rd_t  *currentRxDescriptor = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	currentRxDescriptor = headRxDescriptor;
	while((Rd_t  *)NULL_RD != currentRxDescriptor)
	{
		debug_ucastRDs++;
		nextRxDescriptor = (Rd_t  *)GET_NEXT_RD(currentRxDescriptor);
		SLOG0(0, 0, Rd_t, currentRxDescriptor);
		currentRxDescriptor = nextRxDescriptor;
	}	
	/* Push to HOST I/F */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT;
	hwQueueManagerRequestParams.pHeadDesc = headRxDescriptor;
	hwQueueManagerRequestParams.pTailDesc = tailRxDescriptor;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);	
}

static void rxManagerDebugDropRdsListHandler (Rd_t *headRxDescriptor, Rd_t *tailRxDescriptor)
{
	Rd_t *nextRxDescriptor = NULL;
	Rd_t *currentRxDescriptor = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	currentRxDescriptor = headRxDescriptor;
	while((Rd_t *)NULL_RD != currentRxDescriptor)
	{
		debug_dropRDs++;
		nextRxDescriptor = (Rd_t *)GET_NEXT_RD(currentRxDescriptor);
		SLOG0(0, 0, Rd_t, currentRxDescriptor);
		currentRxDescriptor = nextRxDescriptor;
	}
	/* Push to HOST I/F */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	hwQueueManagerRequestParams.pHeadDesc = headRxDescriptor;
	hwQueueManagerRequestParams.pTailDesc = tailRxDescriptor;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);	
}
#endif

/**********************************************************************************

rxManagerDriverListHandler 


Description:
------------
This function handles the list of RDs meant for the driver

Input: 
-----
headRxDescriptor - the first RD in the list
tailRxDescriptor - the last RD in the list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerDriverListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerDriverListHandler");
#endif

	//SLOG0(0, 0, Rd_t, headRxDescriptor);
    hostInterfaceRings_PutOnRdsList(headRxDescriptor,tailRxDescriptor,&driverRdsDoneList);
	EventManager_TriggerSwEvent(EVENT_ID_DESCRIPTORS_PENDING_FOR_HOST_IF);
}

/**********************************************************************************

rxManagerDataTemporaryForwardToFwListHandler 


Description:
------------
This function handles the list of data RDs that temporary were forward to FW

Input: 
-----
headRxDescriptor - the first RD in the list
tailRxDescriptor - not used

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerDataTemporaryForwardToFwListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
	Rd_t  *nextRxDescriptor = NULL;
	Rd_t  *currentRxDescriptor = NULL;
	Rd_t  *amsduTailRxDescriptor = NULL;
	Rd_t  *releaseListHead = NULL;
	Rd_t  *releaseListTail = NULL;
	uint8 isStationConnected = FALSE;	
	StaId stationId = 0;
	UNUSED_PARAM(tailRxDescriptor);	   
	currentRxDescriptor = headRxDescriptor;
#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerDataTemporaryForwardToFwListHandler");
#endif
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_READY_LIST_DRIVER;
	
	while((Rd_t  *)NULL_RD != currentRxDescriptor)
	{

#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
		/* If this is a MCAST packet free the element in the ring */
		if (currentRxDescriptor->rdType == RD_TYPE_MULTICAST_DATA)
		{
			ClientModeMcastFilterLocateRd(currentRxDescriptor, &RxManagerGlobalParameters.clientModeMcastFilterErrorIndex);
		}	
#endif
		stationId = currentRxDescriptor->rxQueueStaId;
		isStationConnected = StaDB_IsStationConnected(stationId);
		if(isStationConnected)
		{
			/* When station is during connecting process EAPOLs are acepted even if they are in A-MSDU
			   This means that A-MSDUs are not discarded as whole but all their RDs are checked one by one */
			if(currentRxDescriptor->ethType == ETH_TYPE_EAPOL)	
			{
				if ((currentRxDescriptor->sop) && (currentRxDescriptor->eop))
				{
					/* The next of the current is saved before passing it forward - preventing race scenario in which
					   is released before the Rx manager continues - the next will be lost */					
					nextRxDescriptor = (Rd_t  *)GET_NEXT_RD(currentRxDescriptor);
					RxManagerHwQueueManagerRequestParams.pHeadDesc = currentRxDescriptor; 
	   		    	
	   		    	/* EAPOLs are sent one by one to the hostAPD (not as a list) since there are not supposed to be
				  	   many EAPOLs on the list */
					HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
				}
				else
				{
					/*EAPOL may arrive fragmented - at this point all the fragments should be available*/
					/*Current must be first fragment*/
					ASSERT(currentRxDescriptor->sop);
					/*Set first fragment to head*/
					RxManagerHwQueueManagerRequestParams.pHeadDesc = currentRxDescriptor; 
					/*Move to next*/
					nextRxDescriptor = (Rd_t *)GET_NEXT_RD(currentRxDescriptor);
					/*We reached the end of the list without finding the last fragment*/
					ASSERT((Rd_t *)NULL_RD != nextRxDescriptor);
					/*Have we reached the last fragment*/
					while (nextRxDescriptor->eop == 0)
					{
						/*Move to next*/
						nextRxDescriptor = (Rd_t *)GET_NEXT_RD(nextRxDescriptor);
						/*We reached the end of the list without finding the last fragment*/
						ASSERT((Rd_t *)NULL_RD != nextRxDescriptor);
					}
					/*nextRxDescriptor is the last fragment*/
					/*Set last fragment to tail*/
					RxManagerHwQueueManagerRequestParams.pTailDesc = nextRxDescriptor; 
					/*Move to next now before passing forward - preventing race scenario in which
					   is released before the Rx manager continues - the next will be lost */					
					nextRxDescriptor = (Rd_t *)GET_NEXT_RD(nextRxDescriptor);
	   		    	/* EAPOLs are sent one by one to the hostAPD (not as a list) since there are not supposed to be
				  	   many EAPOLs on the list */
					HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
				}
				
				/* The tail is also used as previous pointer - instead of creating the list from scratch only the
				   EAPOLs are removed. In case there will be only few EAPOLs as expected this will save writes to 
				   the SHRAM*/
				if (releaseListTail != NULL)
				{
					releaseListTail->nextRd = SET_NEXT_RD(nextRxDescriptor);
				}
				currentRxDescriptor = nextRxDescriptor;
			}
			else
			{			
				/* This is a different case than the ones below since A-MSDU should not be discarded as a whole
				   but instead to be procssed one by one */
				rxManagerUpdateReleaseList(&releaseListHead, &releaseListTail, &currentRxDescriptor, currentRxDescriptor);
			}
			
		}
		else
		{
			/* Station is not connected - remove A-MSDU */
			if(currentRxDescriptor->amsduAgg)
			{
				/* In case of A-MSDU and the station is in removal process discard all the A-MSDU as a whole */
				amsduTailRxDescriptor = (Rd_t  *)CONVERT_OFFSET_TO_RD(currentRxDescriptor->tailRd);
				rxManagerUpdateReleaseList(&releaseListHead, &releaseListTail, &currentRxDescriptor, amsduTailRxDescriptor);
			}
			else
			{
				rxManagerUpdateReleaseList(&releaseListHead, &releaseListTail, &currentRxDescriptor, currentRxDescriptor);
			}	
		}
	}
	if(NULL != releaseListHead)
	{
		/* retrun the RDs to host free RDs list (empty list shouldnt be instereted) - the list
		   always goes to the liberator since it counts the free RDs */
		RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
		RxManagerHwQueueManagerRequestParams.pHeadDesc = releaseListHead; 
		RxManagerHwQueueManagerRequestParams.pTailDesc= releaseListTail;
		HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
	}
}

/***********************************************************************
* ClientModeMcastFilterLocateRd
* 
* Description:
* ------------
* Look for the element in the ring that has the given RD 
* 
* Input:
* ------
* rd - RD to look for in the ring
* index - the location where the previous RD was found at (optimize the search)
* 
* Output:
* -------
* None 
* 
* Returns:
* --------
* clientModeMcastFilterRingElement_t * - pointer to the element in ring that holds the searched RD 
* 
************************************************************************/    
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
static clientModeMcastFilterRingElement_t * ClientModeMcastFilterLocateRd (Rd_t *rd, uint8 *index)
{
#ifdef CLIENT_MODE_MCAST_TRACE
	uint32 								*temp = NULL;
#endif

	clientModeMcastFilterRingElement_t *pRing = NULL;
	uint8								elementCounter = 0;

	/* Get the Out pointer = base + index */
	pRing = &clientModeMcastFilterRing[*index];

#ifdef REPLAY_DETECTION_TRACE
	ILOG0_D("[ClientModeMcastFilterLocateRd] URID. current index = %d",*index);
#endif

#ifdef CLIENT_MODE_MCAST_TRACE
	temp = (uint32*)pRing;
	ILOG0_DDDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [1] clientModeMcastFilterRing[current] = 0x%x 0x%x 0x%x 0x%x", temp[0], temp[1], temp[2], temp[3]);	
	ILOG0_DDDDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [2] own = %d valid = %d pRing->sequenceNumber = %d rd->sequenceNumber = %d pRing = 0x%x", pRing->own, pRing->valid, pRing->sequenceNumber, rd->sequenceNumber, pRing);
	ILOG0_DDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [3] pRing->rdPointer = 0x%x rd offset = 0x%x rd = 0x%x", pRing->rdPointer, (uint32)(CONVERT_RD_TO_OFFSET(rd)), (uint32)(rd));
#endif

	while (pRing->rdPointer != (uint32)(CONVERT_RD_TO_OFFSET(rd))) 
	{
		/* Make sure we don't get stuck here */
		elementCounter++;
		ASSERT(elementCounter < CLIENT_MODE_MCAST_FILTER_RING_SIZE);

#ifdef CLIENT_MODE_MCAST_TRACE
		ILOG0_D("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [5] elementCounter = %d", elementCounter);
		ILOG0_D("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [5] index = %d", *index);
#endif

		/* Advance the pointer */
		(*index) ++;
		/* wraparound (make sure we won't step off the ring size) */
		(*index) &= (CLIENT_MODE_MCAST_FILTER_RING_SIZE - 1);
		/* check next frame in the ring */
		pRing = &clientModeMcastFilterRing[*index];
		
	}

	/* Own bit must be set */
	ASSERT(pRing->own == 1);

#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_DDDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterLocateRd [6] Own = %d Valid = %d elementCounter = 0x%x *index = %d", pRing->own, pRing->valid, elementCounter, *index);
#endif

	/* Clear the Own and Valid bits (Valid bit is already '0' if we came from rxManagerErrorRdsListHandler) */
	pRing->own = 0;
	pRing->valid = 0;

	/* Advance the index and wraparound */
	(*index) ++;
	(*index) &= (CLIENT_MODE_MCAST_FILTER_RING_SIZE - 1);
#ifdef REPLAY_DETECTION_TRACE
	ILOG0_D("[ClientModeMcastFilterLocateRd] URID. new index = %d",*index);
#endif
	return(pRing);
}
#endif

/**********************************************************************************

rxManagerErrorRdsListHandler 


Description:
------------
This function handles the list of RDs with errors

Input: 
-----
headRxDescriptor - the first RD in the list
tailRxDescriptor - the last RD in the list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerErrorRdsListHandler (Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
	Rd_t  *currentRxDescriptor = NULL;
	UMI_TKIP_MIC_FAILURE tkipMicFailureMessage;
	uint8 entryIndexInMessage = 0;
	uint8 vapId = 0;
	StaId ts_stationIndex = INVALID_STA_INDEX;
	uint8 ts_tid = NUM_TID;
#if defined (ENET_INC_ARCH_WAVE600)
	uint32 previousStaId;
#endif


	memset(&tkipMicFailureMessage, 0, sizeof(UMI_TKIP_MIC_FAILURE));

	ILOG0_V("rxManagerErrorRdsListHandler");
	
	currentRxDescriptor = headRxDescriptor;

#ifdef REPLAY_DETECTION_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerErrorRdsListHandler");
#endif

	// Klocwork: while loop is implemented for not to exit from this loop But klocwork will give error as "INFINITE_LOOP.LOCAL: Infinite loop" ignore it. 
	while((Rd_t  *)NULL_RD != currentRxDescriptor)
	{
		SLOG0(0, 0, Rd_t, currentRxDescriptor);
#if !defined (ENET_INC_ARCH_WAVE600)	
		/* In case of TKIP MIC failure without replay detection error and ICV error an indication will be send to hostAPD */
		if((RD_TKIP_MIC_FAILURE & currentRxDescriptor->securityStatus) && 
		   (!currentRxDescriptor->statusRxppError) && 
		   (!(RD_TKIP_ICV_FAILURE & currentRxDescriptor->securityStatus)))
#else
		// check if we are in TKIP - unicast packet->STADB, multicast packet->vapDB (since we are STA mode)
		if(((currentRxDescriptor->micFailure) && (!currentRxDescriptor->statusRxppError))  && 
		(((currentRxDescriptor->multicast == FALSE) &&(StaDB_GetDataEncryptionType(currentRxDescriptor->rxQueueStaId) == UMI_RSN_CIPHER_SUITE_TKIP)) ||
		((currentRxDescriptor->multicast == TRUE) && (VapDB_GetDataEncryptionType(currentRxDescriptor->rxQueueVapId) == UMI_RSN_CIPHER_SUITE_TKIP))))

#endif
		{
#if defined (ENET_INC_ARCH_WAVE600)
			// Check if the new RD belongs to a STA from a different band than all the previous ones.
			// We want all the STAs in the same MSG to belong to the same band.
			if (entryIndexInMessage > 0)
			{
				ASSERT(entryIndexInMessage < MAX_NUMBER_TKIP_MIC_FAILURE_ENTRIES_IN_MESSAGE)
				previousStaId = tkipMicFailureMessage.micFailureEntry[entryIndexInMessage - 1].stationId;
				if (ConfigurationManager_GetBandForStation(currentRxDescriptor->rxQueueStaId) != ConfigurationManager_GetBandForStation(previousStaId))
				{
					// Send the MSG with the previous STAs since the new one belong to the other band. Use the VAP of the last STA in the array as indication to the band.
					StaDb_GetVapId(previousStaId, &vapId);					
					vIF_SendMsg(&sUmiIndFreeQueue,
							    UMI_MC_MAN_TKIP_MIC_FAILURE_INDICATION,
							    TASK_UM_IF_TASK,
							    &tkipMicFailureMessage,
							    sizeof(UMI_TKIP_MIC_FAILURE),
							    vapId); 

					entryIndexInMessage = 0;
					tkipMicFailureMessage.u8numOfValidEntries = 0;	

				}
			}
#endif
			tkipMicFailureMessage.u8numOfValidEntries ++;
			tkipMicFailureMessage.micFailureEntry[entryIndexInMessage].stationId = currentRxDescriptor->rxQueueStaId;
			tkipMicFailureMessage.micFailureEntry[entryIndexInMessage].isGroupKey = currentRxDescriptor->multicast;
			entryIndexInMessage ++;
			// Check if we already filled all the entries in the array
			if(entryIndexInMessage == MAX_NUMBER_TKIP_MIC_FAILURE_ENTRIES_IN_MESSAGE)
			{
				StaDb_GetVapId(currentRxDescriptor->rxQueueStaId, &vapId);					
				vIF_SendMsg(&sUmiIndFreeQueue,
						    UMI_MC_MAN_TKIP_MIC_FAILURE_INDICATION,
						    TASK_UM_IF_TASK,
						    &tkipMicFailureMessage,
						    sizeof(UMI_TKIP_MIC_FAILURE),
						    vapId); 

				entryIndexInMessage = 0;
				tkipMicFailureMessage.u8numOfValidEntries = 0;	
			}
			/*Update statistics*/
			if (currentRxDescriptor->multicast)
			{
#if defined (ENET_INC_ARCH_WAVE600)
				pBaaCounters->groupMicFailurePackets[ConfigurationManager_GetBandForVap(vapId)]++;
#else
				pBaaCounters->groupMicFailurePackets++;
#endif //ENET_INC_ARCH_WAVE600
			}
			else
			{	
#if defined (ENET_INC_ARCH_WAVE600)
				pBaaCounters->fwPairWiseMicFailurePackets[ConfigurationManager_GetBandForVap(vapId)]++;
#else
				pBaaCounters->fwPairWiseMicFailurePackets++;
#endif //ENET_INC_ARCH_WAVE600				
			}
		}
#if !defined (ENET_INC_ARCH_WAVE600)
		/* In case we had a RXH error without security error, a message is sent to TS manager to close recipient TID*/
		else if ((currentRxDescriptor->rdType == RD_TYPE_UNICAST_QOS_DATA) &&
				 (currentRxDescriptor->statusRxhError) &&
				 ((currentRxDescriptor->securityStatus & (RD_TKIP_MIC_FAILURE | RD_TKIP_ICV_FAILURE)) == 0) &&
				 ((currentRxDescriptor->rxQueueStaId != ts_stationIndex) || (currentRxDescriptor->rxQueueTid != ts_tid)))
#else
		else if (((currentRxDescriptor->rdType == RD_TYPE_UNICAST_QOS_DATA) &&
			 	((currentRxDescriptor->statusRxhError) || (currentRxDescriptor->statusDeaggregatorError))) &&
			 	(currentRxDescriptor->securityStatus == 0) &&
				((currentRxDescriptor->rxQueueStaId != ts_stationIndex) || (currentRxDescriptor->rxQueueTid != ts_tid)))

#endif

				{

					/* Store SID and TID*/
					ts_stationIndex = currentRxDescriptor->rxQueueStaId;
					ts_tid = currentRxDescriptor->rxQueueTid;

					RxPp_FlushTid(currentRxDescriptor->rxQueueStaId, currentRxDescriptor->rxQueueTid, RXPP_DONT_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);	
				}
		if (currentRxDescriptor->statusRxhError)
		{
#ifdef ENET_INC_ARCH_WAVE600
			GeneralStatistics.rxhErrors[ConfigurationManager_GetBandForVap(currentRxDescriptor->rxQueueVapId)]++;
#else
			GeneralStatistics.rxhErrors++;
#endif //ENET_INC_ARCH_WAVE600
		}
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
		/* If this is a MCAST packet free the element in the ring */
		if (currentRxDescriptor->rdType == RD_TYPE_MULTICAST_DATA)
		{
#ifdef CLIENT_MODE_MCAST_TRACE
			ILOG0_D("[CLIENT_MODE_MCAST] rxManagerErrorRdsListHandler. ErrorIndex = %d",RxManagerGlobalParameters.clientModeMcastFilterErrorIndex);
#endif
			if ((currentRxDescriptor->statusRxhError == 0) || ((currentRxDescriptor->statusRxhError == 1) && (currentRxDescriptor->rxhErrorCause == RXH_ERROR_CAUSE_VALUE_MPDU_HEADER_CONVERSION_FAILED)))
			{
				ClientModeMcastFilterLocateRd(currentRxDescriptor, &RxManagerGlobalParameters.clientModeMcastFilterErrorIndex);
			}
		}
#endif // CLIENT_MODE_MULTICAST_FILTER_SUPPORT
		
		/*If this is a data RD dropped by RXH, close reordering*/
		if(currentRxDescriptor->amsduAgg)
		{
			/* In case it is an A-MSDU there is no point checking the other RDs in the A-MSDU */
			currentRxDescriptor = (Rd_t  *)CONVERT_OFFSET_TO_RD(currentRxDescriptor->tailRd);
		}		
		
		currentRxDescriptor = (Rd_t  *)GET_NEXT_RD(currentRxDescriptor);
	}

	if(entryIndexInMessage)
	{
		/* The last message was not full so it wasnt sent - sent it now */	
		StaDb_GetVapId(tkipMicFailureMessage.micFailureEntry[entryIndexInMessage - 1].stationId, &vapId);					
		vIF_SendMsg(&sUmiIndFreeQueue,
					UMI_MC_MAN_TKIP_MIC_FAILURE_INDICATION,
					TASK_UM_IF_TASK,
					&tkipMicFailureMessage,
					sizeof(UMI_TKIP_MIC_FAILURE),
					vapId);
	}

	/* retrun the RDs to the liberator which will put them in the proper free RD pool according to 
	   their RD source field */
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = headRxDescriptor; 
	RxManagerHwQueueManagerRequestParams.pTailDesc= tailRxDescriptor;
	HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
}

/**********************************************************************************

rxManagerManagementListHandler 


Description:
------------
This function handles the list of management RDs

Input: 
-----
headRxDescriptor - the first RD in the list
tailRxDescriptor - the last RD in the list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void rxManagerReleaseFwdRdListHandler (Rd_t *headRxDescriptor, Rd_t *tailRxDescriptor)
{
    Rd_t* rxDescriptor = (Rd_t*)NULL_RD;
	Rd_t* hostRxDescriptor = NULL;
	Rd_t* nextHostRxDescriptor = NULL;
	uint32 dramPtr = 0;
	uint32 system_XbarAddr = 0;
    
#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerReleaseFwdRdListHandler");
#endif
	system_XbarAddr = System_GetXbarAddress();
	hostRxDescriptor = headRxDescriptor;

	while (((Rd_t *)NULL_RD != hostRxDescriptor) && (((Rd_t*)NULL_RD) != forwardRDsPendingList.pHead))
	{
		nextHostRxDescriptor = (Rd_t *)GET_NEXT_RD(hostRxDescriptor);
		dramPtr = CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(hostRxDescriptor->dramPointer);
		if((dramPtr & MAC_XBAR_WLAN_IP_MASK) != system_XbarAddr)
		{
			/*get RD from Pending List*/
			rxDescriptor = hostInterfaceRingsGetRdFromList(&forwardRDsPendingList);

			// verify Max payload size - SDL requirement
			if (rxDescriptor->dataLength > DDR_MAX_MANAGEMENT_PAYLOAD_SIZE)
			{
				/* Frame is not valid for some reason - discard Rd, retrun the RD to the liberator which will 
				put it in the proper free RD pool according to its RD source field */
				RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
				RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
				HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
			}
			else
			{
				/* rxDescriptor->securityStatus holds the category */
				rxManagerHandleOneForwardRd(rxDescriptor, hostRxDescriptor);
				hostRxDescriptor = nextHostRxDescriptor;
			}
		}
		else
		{
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_READY_LIST_BAD_PTR_RDS;
			RxManagerHwQueueManagerRequestParams.pHeadDesc = hostRxDescriptor;
			HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
			RxManagerBadForwardRdCnt++;

			if(RxManagerBadForwardRdCnt == NUM_OF_FORWARD_RX_DESCRIPTORS)
			{
				FATAL("FORWARD BAD Pointer List has crossed the max boundary");
			}
			hostRxDescriptor = nextHostRxDescriptor;
		}
	}

    if (hostRxDescriptor != (Rd_t*)NULL_RD) // There are RDs left in host descriptors list
	{   
        /* retrun the RDs directly to the forward list (and not to Liberator)*/
        RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_FREE_FORWARD_RDS_LIST;
        RxManagerHwQueueManagerRequestParams.pHeadDesc = hostRxDescriptor; 
        RxManagerHwQueueManagerRequestParams.pTailDesc= tailRxDescriptor;
        HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
    }

}

/**********************************************************************************

rxManagerIsrAssert 


Description:
------------
This interrupt is not enabled in fw and should not have triggered.

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerIsrAssert(Rd_t  *headRxDescriptor, Rd_t  *tailRxDescriptor)
{
	UNUSED_PARAM(headRxDescriptor);
	UNUSED_PARAM(tailRxDescriptor);
	DEBUG_FATAL("RX Manager Isr Handler");
}

/**********************************************************************************

rxManagerRdHandlerAssert 


Description:
------------
This function handles RD that was received in the FW list but should no be there

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerRdHandlerAssert(Rd_t  *rxDescriptor)
{
	UNUSED_PARAM(rxDescriptor);	
	DEBUG_FATAL("RX Manager RD Handler");
}

/**********************************************************************************

rxManagerUnsupportedRdHandler 


Description:
------------
This function handles RD of unsupported frames
Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerUnsupportedRdHandler(Rd_t  *rxDescriptor)
{
	/* Currently there are no operations needed for unsupported frames - the RD is released */
#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerUnsupportedRdHandler. >>>>>> Why does RX Manager think this is MCAST");
#endif
	/* retrun the RD to the liberator which will put it in the proper free RD pool according to 
	  its RD source field */
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
	HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
}

    
/***********************************************************************
* ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket
* 
* Description:
* ------------
*  Check if the current RD is a MCAST PACKET that was localy originated by checking the SA
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
bool ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket(Rd_t *rd)
{
	bool 								isRdLocalyOriginatedMcastPacket = FALSE;
	clientModeMcastFilterRingElement_t *pRing = NULL;
	IEEE_ADDR							*ourMacAddress;
	IEEE_ADDR							*mcastPacketSourceMacAddress;

	/* Look for the element that holds the current RD */
#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_D("[ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket] rxManagerErrorRdsListHandler. FilterIndex = %d",RxManagerGlobalParameters.clientModeMcastFilterRingIndex);
#endif
	pRing = ClientModeMcastFilterLocateRd(rd, &RxManagerGlobalParameters.clientModeMcastFilterRingIndex);

	/* Get our MAC address */
	ourMacAddress = &(VapDatabaseObj.vapDbSwEntries[rd->rxQueueVapId].macAddress);

	/* Get the SA from the element on the ring */
	mcastPacketSourceMacAddress = &(pRing->SA);

	/* Compare our address with the address of the corresponding element on the ring */
	isRdLocalyOriginatedMcastPacket = boIEEE_ADDR_IsEqual(ourMacAddress, mcastPacketSourceMacAddress);

#ifdef CLIENT_MODE_MCAST_TRACE			
	ILOG0_DDDDDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket [2] mcastPacketSourceMacAddress = 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x ", mcastPacketSourceMacAddress->au8Addr[5], mcastPacketSourceMacAddress->au8Addr[4], mcastPacketSourceMacAddress->au8Addr[3], mcastPacketSourceMacAddress->au8Addr[2], mcastPacketSourceMacAddress->au8Addr[1], mcastPacketSourceMacAddress->au8Addr[0]);
	ILOG0_DDDDDD("[CLIENT_MODE_MCAST] ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket [2] ourMacAddress = 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x ", ourMacAddress->au8Addr[5], ourMacAddress->au8Addr[4], ourMacAddress->au8Addr[3], ourMacAddress->au8Addr[2], ourMacAddress->au8Addr[1], ourMacAddress->au8Addr[0]);	
#endif

	return(isRdLocalyOriginatedMcastPacket);
}
#endif


/**********************************************************************************

rxManagerMulticastDataRdHandler 


Description:
------------
This function handles multicast data

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerMulticastDataRdHandler(Rd_t *rxDescriptor)
{
	StaId stationId = 0;
	uint8   vapId = 0;
	
#ifndef ENET_INC_ARCH_WAVE600D2
	uint8*  pLastPn = NULL;
	uint8	keyId = 0;
#endif	
	bool 	pass = TRUE;

	stationId = rxDescriptor->rxQueueStaId;
	vapId = rxDescriptor->rxQueueVapId; 
	
#ifndef ENET_INC_HW_FPGA
	//if we are in AP mode we should not recived multi cast data
	// In Palladium, this assert can occur in WDS due to timing issue, since WDS is not yet updated in DB
	if((VapDb_GetVapMode(vapId) == VAP_MODE_AP) && (StaDB_GetNegotiatedWds(stationId) == FALSE))
	{
		if (StaDB_IsStationConnected(stationId))
		{
			FATAL("we got Multicast Data RD while we are in AP mode from connected STA ");
		}
		else
		{
			FATAL("we got Multicast Data RD while we are in AP mode from not connected STA ");
		}		
	}
#endif
	/* Each RD is sent to the proper module wich is responsible of releasing it */
#if defined (RX_LIST_DEBUG)
	SLOG0(0, 0, Rd_t, rxDescriptor);
	if(rxDescriptor->broadcast == 1)
	{
		debug_bcastRDs++;
	}
	else
	{
		debug_mcastRDs++;
	}
#endif	
	/*Update statistics*/
	pBaaCounters->rxGroupFrame[rxDescriptor->rxQueueVapId]++;

	RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
	ASSERT(stationId < HW_NUM_OF_STATIONS);
	//in WAVE600D2 HW supports rekey check for STA mode therefore no need to execute by FW 	
#ifndef ENET_INC_ARCH_WAVE600D2 
	rxManagerCheckMulticastRekey(rxDescriptor);   
#endif 
#ifdef CLIENT_MODE_MCAST_TRACE	
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerMulticastDataRdHandler");
#endif
	/*If STA is 4 address mode drop multicast data*/
	if (StaDB_GetNegotiatedWds(stationId) == FALSE)
	{
		//in WAVE600D2 HW supports replay detection for STA mode therefore no need to execute by FW 
#ifndef ENET_INC_ARCH_WAVE600D2 
		if ((rxDescriptor->protected == TRUE) && (rxDescriptor->securityWepType == FALSE))
		{
			keyId = rxDescriptor->keyId;
			// keyId for multicast packets is 1 or 2	
			ASSERT(keyId == 1 || keyId == 2);
			// reduce keyId by 1 to fit the lastPN array - keyId 2 assigned to 1, keyId 1 assigned to 0.
			keyId--;
			// choose the last PN according to Vap Index and Key Index		
			pLastPn = RxManagerVapDatabase[vapId].lastPN[keyId];
			pass = rxManagerDoReplayDetectionStaMode(pLastPn, rxDescriptor);
		}
#endif
		// pass variable holds TRUE value if replay detection passed and FALSE if failed
		if (pass == TRUE)
		{
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
			// if we get here it means we passed replay detection
			if (ClientModeMcastFilterIsRdLocalyOriginatedMcastPacket(rxDescriptor))
			{
#ifdef CLIENT_MODE_MCAST_TRACE
				ILOG0_D("[CLIENT_MODE_MCAST] rxManagerMulticastDataRdHandler dropping localy origniated packet SN = %d", rxDescriptor->sequenceNumber);
#endif
				RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
			}			
			else
			{
				RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT;
			}
#else
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT;
#endif // CLIENT_MODE_MULTICAST_FILTER_SUPPORT
			HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
		}
		else
		{
#ifdef REPLAY_DETECTION_TRACE
			ILOG0_V("[rxManagerMulticastDataRdHandler] URID. you got here due to replay detection fail");
#endif
			/*Update statistics*/
#if defined (ENET_INC_ARCH_WAVE600)
			pBaaCounters->fwMulticastReplayedPackets[ConfigurationManager_GetBandForStation(stationId)]++;
#else
			pBaaCounters->fwMulticastReplayedPackets++;
#endif //ENET_INC_ARCH_WAVE600

			// if we get here it means we faild replay detection - send to error list	
			rxManagerErrorRdsListHandler(rxDescriptor, rxDescriptor);
		}
	}
	else
	{
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
#ifdef CLIENT_MODE_MCAST_TRACE
		ILOG0_D("[CLIENT_MODE_MCAST] rxManagerMulticastDataRdHandler WDS is active and it is not forwarding mcast packets - free the element from the ring SN = %d", rxDescriptor->sequenceNumber);
#endif
		ClientModeMcastFilterLocateRd(rxDescriptor, &RxManagerGlobalParameters.clientModeMcastFilterErrorIndex);
#endif // CLIENT_MODE_MULTICAST_FILTER_SUPPORT
		RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
		HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
	}	
	
}

/**********************************************************************************

rxManagerControlRdHandler 


Description:
------------
This function handles contril RDs

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerControlRdHandler(Rd_t  *rxDescriptor)
{ 
	K_MSG* psMsg = NULL;
	TsManagerPacketReceivedMessage_t* tsManagerMessageParameters = NULL;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    bool dont_drop_rd = FALSE;
	StaId stationId = rxDescriptor->rxQueueStaId;

    // Rd should be dropped when not suiting the rules
	if (stationId < HW_NUM_OF_STATIONS && BAR_SUBTYPE == rxDescriptor->frameSubtype)
	{
        dont_drop_rd = TRUE;
	}

#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerControlRdHandler. >>>>>> Why does RX Manager think this is MCAST");
#endif
#if defined (ENET_INC_ARCH_WAVE600)
	pBaaCounters->fwctrlFramesRecieved[ConfigurationManager_GetBandForStation(stationId)]++;
#else
	pBaaCounters->fwctrlFramesRecieved++;
#endif //ENET_INC_ARCH_WAVE600

    ILOG1_D("-------------------- We recived  %d  in CONTROL------------------",rxDescriptor->frameSubtype);

    if (dont_drop_rd)
    {
    	/* Only BARs should arrive to FW */ 
    	psMsg = OSAL_GET_MESSAGE( sizeof(TsManagerPacketReceivedMessage_t));
        tsManagerMessageParameters = ((TsManagerPacketReceivedMessage_t *)psMsg->abData);
    	tsManagerMessageParameters->rxDescriptor = rxDescriptor;
    	
    	OSAL_SEND_MESSAGE(TS_MANAGER_BAR_PACKET_RECEIVED, TASK_TS_MANAGER, psMsg, VAP_ID_DO_NOT_CARE);        
    }
    else
    {
        
        /* ACK_SUBTYPE 
or BA_SUBTYPE 
           retrun the RD to the liberator which will put it in the proper free RD pool according to 
           its RD source field 
        */
        hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
        hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
        hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST; 
        hwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
        HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
    }
}

/**********************************************************************************

rxManagerFwManagementRdHandler 


Description:
------------
This function handles management frames meant for FW from associated stations
(currently only action and no ACK action frames supposed to arrive to FW)

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerFwManagementRdHandler(Rd_t  *rxDescriptor)
{ 
	RxMpdu_t *rxMpdu = NULL;
	MANAGEMENT_BASIC_FRAME_HEADER *frame = NULL;
	FM_PAYLOAD_ACTION_COMMON *payload = NULL;
	RxManagerTimeoutEntry_t *classViolationEntry = NULL;
	StaId stationId = 0;
	uint8 category = 0;
	uint8 categoryIndex = 0;
	uint8 robustValidityIndex = 0;
	uint8 stationFrameClassAllowed = 0;
    uint8 isStationEncryptManagement = 0;
	uint8 isMacAddressExist = FALSE;
	uint8 classViolationEntryIndex = 0;
	uint8 vapId = 0;
	UMI_FRAME_CLASS_ERROR frameClassErrorMessage;
	FrameControl_t* pFrameControl = NULL;
	uint16 previousNumberOfActiveTimeouts = 0;

	vapId = rxDescriptor->rxQueueVapId;
	stationId = rxDescriptor->rxQueueStaId;
#if defined (ENET_INC_ARCH_WAVE600)
	pBaaCounters->fwMngmntframesRecieved[ConfigurationManager_GetBandForVap(vapId)]++;
#else
	pBaaCounters->fwMngmntframesRecieved++;
#endif //ENET_INC_ARCH_WAVE600

    memset(&frameClassErrorMessage, 0, sizeof(UMI_FRAME_CLASS_ERROR));

#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerFwManagementRdHandler. >>>>>> Why does RX Manager think this is MCAST");
#endif


	/* Check Byte offset is aligned to 4 Byte (The valid values are 0 or 4)*/
	DEBUG_ASSERT((rxDescriptor->dramByteOffset & FOUR_BYTE_ALIGNED_MASK) == 0);

	rxMpdu = (RxMpdu_t *)(CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR((CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset)));
	frame = (MANAGEMENT_BASIC_FRAME_HEADER *)rxMpdu->frame;
	pFrameControl = (FrameControl_t *)&frame->u16FrameControl;
	payload = (FM_PAYLOAD_ACTION_COMMON *)frame_getPayloadPointerFromExistingManagementFrame(frame);
	if (pFrameControl->protectedFrame == TRUE)
	{
		/*If frame is encrypted skip CCMP header*/
		payload = (FM_PAYLOAD_ACTION_COMMON *)((uint8 *)payload + sizeof(CCMP_HEADER));
	}

	// verify minimum length of action frame & Max payload size - SDL requirement
	if ((rxDescriptor->dataLength >= ACTION_FRAME_MIN_LENGTH) && ((rxDescriptor->dataLength < DDR_MAX_MANAGEMENT_PAYLOAD_SIZE)))
	{
		category = payload->u8CategoryCode;
		
		/* convert category to internal index to be used in the Rx manager arrays */
		categoryIndex = rxManagerConvertCategoryToInternalIndex(category);
	}
	else // invalid frame - ignore by marking invalid category
	{
		ILOG0_D("rxManagerFwManagementRdHandler: ignore - length is too short/long - %d", rxDescriptor->dataLength);
		categoryIndex = RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY;
	}

	
	previousNumberOfActiveTimeouts = RxManagerGlobalParameters.numOfActiveTimeouts;

	if(categoryIndex != RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY)
	{
    	/* Valid category - check frame class violation */
#if defined(ENET_INC_ARCH_WAVE600)
		stationFrameClassAllowed = RxManager_GetStaFrameClass(stationId);
#else
		stationFrameClassAllowed = RxClassifier_GetStaFrameClass(stationId);
#endif 
		if(RxManagerActionFrameClassTable[categoryIndex] <= stationFrameClassAllowed)
		{
			/* Valid frame class */
			isStationEncryptManagement = (MTLK_BFIELD_GET(RxManagerStationDatabase[stationId].flags,STA_ADD_FLAGS_MFP)) & 
										 (RxManagerStationDatabase[stationId].rxManagerDbStaState == RX_MANAGER_DB_STA_OPEN);
			
			/* Calculate index for the robust validity table */
			pFrameControl = (FrameControl_t*)&(frame->u16FrameControl);

			robustValidityIndex = isStationEncryptManagement << RX_MANAGER_ROBUST_TABLE_VALIDITY_MECHANISM_SHIFT;
			robustValidityIndex += (pFrameControl->protectedFrame << RX_MANAGER_ROBUST_TABLE_VALIDITY_PROTECTED_SHIFT);
			robustValidityIndex += RxManagerActionFrameRobustTable[categoryIndex];
			if(RxManagerActionFrameRobustValidityTable[robustValidityIndex])
			{
				/* Frame pass robust validity check - send to the proper handler */
				rxManagerHandleValidActionFrame(rxDescriptor, frame, category);
				return;
			}
		}
		else
		{
			/* Not valid class frame - check if need to send class violation indication to host */
            isMacAddressExist = rxManagerIsMacAddressExist(&frame->sIEEE_ADDRaddr2);
			if(!isMacAddressExist)
			{
				/* In case the MAC address exist do nothing */
                if(RxManagerFwClassViolationBitmap != RX_MANAGER_FW_CLASS_VIOLATION_CACHE_FULL)
                {
					/* Find free entry */
					classViolationEntryIndex = rxManagerFindFreeFwClassViolationEntry();

					/* Strat counting timeout */
					classViolationEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerFwClassViolationEntries[classViolationEntryIndex];
					vIEEE_ADDR_CopyAddr(&classViolationEntry->stationMacAddress, &frame->sIEEE_ADDRaddr2);
					CalendarWheel_AddEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)classViolationEntry, RX_MANAGER_CLASS_VIOLATION_TIMEOUT_UNITS);
					RxManagerGlobalParameters.numOfActiveTimeouts ++;
					
                    /* Send class violation indication to host */
					frameClassErrorMessage.u8numOfValidEntries = 1;

					if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
					{
						frameClassErrorMessage.frameClassErrorEntries[0].u8vapIndex = vapId & 0xF; 
					}
					else
					{
						frameClassErrorMessage.frameClassErrorEntries[0].u8vapIndex = vapId; 
					}

					vIEEE_ADDR_CopyAddr(&(frameClassErrorMessage.frameClassErrorEntries[0].sAddr), &frame->sIEEE_ADDRaddr2);
					vIF_SendMsg(&sUmiIndFreeQueue,
								UMI_MC_MAN_CLASS3_ERROR_IND,
								TASK_UM_IF_TASK,
								&frameClassErrorMessage,
								sizeof(UMI_FRAME_CLASS_ERROR),
		            	    	vapId); /* This field is redundant in this message */
                }
				else
				{
					/* FW class violation cache is full - reject new MAC address */
					RxManagerFwClassViolationCacheIsFull ++;
				}				
			}
		}
	}

	if((0 == previousNumberOfActiveTimeouts) && (0 != RxManagerGlobalParameters.numOfActiveTimeouts))
    {
		/* Setting the timer since it was not active and there are new entries */
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
    }

	/* Frame is not valid for some reason - discard Rd, retrun the RD to the liberator which will 
	   put it in the proper free RD pool according to its RD source field */
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
	HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
}


/**********************************************************************************

rxManagerFwUnassociatedManagementRdHandler 


Description:
------------
This function handles management frames meant for FW from unassociated stations
(currently only class 1 action and no ACK action frames supposed to arrive to FW)

Input: 
-----
rxDescriptor - the RD that should be handled

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerFwUnassociatedManagementRdHandler(Rd_t  *rxDescriptor)
{ 
	RxMpdu_t *rxMpdu = NULL;
	MANAGEMENT_BASIC_FRAME_HEADER *frame = NULL;
	FrameControl_t *pFrameControl = NULL;
	FM_PAYLOAD_ACTION_COMMON *payload = NULL;
	RxManagerTimeoutEntry_t *classViolationEntry = NULL;
	uint8 category = 0;
	uint8 categoryIndex = 0;
	uint8 isMacAddressExist = FALSE;
	uint8 classViolationEntryIndex = 0;
	uint8 vapId = 0;
	UMI_FRAME_CLASS_ERROR frameClassErrorMessage;
	uint16 previousNumberOfActiveTimeouts = 0;

	vapId = rxDescriptor->rxQueueVapId;
#if defined (ENET_INC_ARCH_WAVE600)
	pBaaCounters->fwMngmntframesRecieved[ConfigurationManager_GetBandForVap(vapId)]++;
#else
	pBaaCounters->fwMngmntframesRecieved++;
#endif //ENET_INC_ARCH_WAVE600
	
    memset(&frameClassErrorMessage, 0, sizeof(UMI_FRAME_CLASS_ERROR));	

	/* Check Byte offset is aligned to 4 Byte (The valid values are 0 or 4)*/
	DEBUG_ASSERT((rxDescriptor->dramByteOffset & FOUR_BYTE_ALIGNED_MASK) == 0);

#ifdef CLIENT_MODE_MCAST_TRACE
	ILOG0_V("[CLIENT_MODE_MCAST] rxManagerFwUnassociatedManagementRdHandler. >>>>>> Why does RX Manager think this is MCAST");
#endif

	//SDL requirement: Rx Length check
	if ((rxDescriptor->dataLength < ACTION_FRAME_MIN_LENGTH) || (rxDescriptor->dataLength > DDR_MAX_MANAGEMENT_PAYLOAD_SIZE))
	{
		categoryIndex = RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY;
	}
	
	rxMpdu = (RxMpdu_t *)(CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR((CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset)));
	frame = (MANAGEMENT_BASIC_FRAME_HEADER *)rxMpdu->frame;
	pFrameControl = (FrameControl_t*)&frame->u16FrameControl;
	payload = (FM_PAYLOAD_ACTION_COMMON *)frame_getPayloadPointerFromExistingManagementFrame(frame);

	// verify minimum length of action frame
	if (rxDescriptor->dataLength >= ACTION_FRAME_MIN_LENGTH)
	{
		category = payload->u8CategoryCode;
		
		/* convert category to internal index to be used in the Rx manager arrays */
		categoryIndex = rxManagerConvertCategoryToInternalIndex(category);
	}
	else // invalid frame - ignore by marking invalid category
	{
		ILOG0_D("rxManagerFwManagementRdHandler: ignore - length is too short - %d", rxDescriptor->dataLength);
		categoryIndex = RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY;
	}

	previousNumberOfActiveTimeouts = RxManagerGlobalParameters.numOfActiveTimeouts;

	if(categoryIndex != RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY)
	{
    	/* Valid category - check frame class violation */
		if(FRAME_CLASS_1 == RxManagerActionFrameClassTable[categoryIndex])
		{
			/* Only class 1 is allowed for unassociated station */
			if(!pFrameControl->protectedFrame)
			{
				//TBD what to do in case it is encrypted - statistics or message to driver?
		    	/* Frame class 1 are not supposed to be encrypted */
				rxManagerHandleValidActionFrame(rxDescriptor, frame, category);
			    return;	
			}
		}
		else
		{
			/* Not valid class frame - check if need to send class violation indication to host */
            isMacAddressExist = rxManagerIsMacAddressExist(&frame->sIEEE_ADDRaddr2);
			if(!isMacAddressExist)
			{
				/* In case the MAC address exist do nothing */
                if(RxManagerFwClassViolationBitmap != RX_MANAGER_FW_CLASS_VIOLATION_CACHE_FULL)
                {
					/* Find free entry */
					classViolationEntryIndex = rxManagerFindFreeFwClassViolationEntry();

					/* Strat counting timeout */
					classViolationEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerFwClassViolationEntries[classViolationEntryIndex];
					vIEEE_ADDR_CopyAddr(&classViolationEntry->stationMacAddress, &frame->sIEEE_ADDRaddr2);
					CalendarWheel_AddEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)classViolationEntry, RX_MANAGER_CLASS_VIOLATION_TIMEOUT_UNITS);
					RxManagerGlobalParameters.numOfActiveTimeouts ++;
					
                    /* Send class violation indication to host */
					frameClassErrorMessage.u8numOfValidEntries = 1;


					if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
					{
						frameClassErrorMessage.frameClassErrorEntries[0].u8vapIndex = vapId & 0xF; 
					}
					else
					{
						frameClassErrorMessage.frameClassErrorEntries[0].u8vapIndex = vapId; 
					}
					
					vIEEE_ADDR_CopyAddr(&(frameClassErrorMessage.frameClassErrorEntries[0].sAddr), &frame->sIEEE_ADDRaddr2);
					vIF_SendMsg(&sUmiIndFreeQueue,
								UMI_MC_MAN_CLASS3_ERROR_IND,
								TASK_UM_IF_TASK,
								&frameClassErrorMessage,
								sizeof(UMI_FRAME_CLASS_ERROR),
		            	    	vapId); /* This field is redundant in this message */
                }
				else
				{
					/* FW class violation cache is full - reject new MAC address */
					RxManagerFwClassViolationCacheIsFull ++;
				}				
			}
		}
	}


	if((0 == previousNumberOfActiveTimeouts) && (0 != RxManagerGlobalParameters.numOfActiveTimeouts))
    {
		/* Setting the timer since it was not active and there are new entries */
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
    }

	/* Frame is not valid for some reason - discard Rd, retrun the RD to the liberator which will 
	   put it in the proper free RD pool according to its RD source field */
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
	HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
}


/**********************************************************************************

rxManagerUpdateReleaseList 


Description:
------------
This fucntion updates the release list and update the current RD 

Input: 
-----
releaseListHead - the first RD of the release list
releaseListTail - the tail RD of the release list
currentRxDescriptor - the current RD that is being handled
endOfMsduList - the last RD in the A-MSDU (==current when the list is one RD)



		
Output:
-------
	current is moved to the next RD after the end of the A-MSDU list

Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerUpdateReleaseList(Rd_t  **releaseListHead, Rd_t  **releaseListTail, Rd_t  **currentRxDescriptor, Rd_t  *endOfMsduList)
{
	if(NULL == *releaseListHead)
	{
		/* The list is empty */
		*releaseListHead = *currentRxDescriptor;
	}
	
	*releaseListTail = endOfMsduList; 
	*currentRxDescriptor = (Rd_t  *)GET_NEXT_RD(endOfMsduList);
}


/**********************************************************************************

rxManagerReleaseFragmentationTimeoutEntry 


Description:
------------
This fucntion rleases fragmentation timeout entry - update counters and return the entry
to the free list


Input: 
-----
timeoutEntry


		
Output:
-------


Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerReleaseFragmentationTimeoutEntry(RxManagerTimeoutEntry_t *timeoutEntry)
{
	StaId stationId = 0;

	stationId = timeoutEntry->stationId;
		
	/* Clear entry in the station array */
	RxManagerStationDatabase[stationId].fragementationEntriesIndexes[timeoutEntry->indexInStationArray] = RX_MANAGER_INVALID_FRAGMENTATION_ENTRY;
					
	/* Update fragmentation counters */
	RxManagerStationDatabase[stationId].numOfActiveFragmentation--;
	RxManagerVapDatabase[timeoutEntry->vapId].numOfActiveFragmentation--;
	RxManagerGlobalParameters.numOfActiveTimeouts--;

	/* Return the entry to the free list */
	timeoutEntry->calendarWheelEntry.nextIndex= RxManagerFreeFragmentationEntriesList - RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries;
	RxManagerFreeFragmentationEntriesList = timeoutEntry;
}

/**********************************************************************************

rxManagerMulticastRekeyTimeout 


Description:
------------
This function handles the Multicast Rekey Timeout event


Input: 
-----
timeoutEntry


		
Output:
-------


Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerMulticastRekeyTimeout(RxManagerTimeoutEntry_t *timeoutEntry)
{
	/*Need to invalidate previous key Id, if valid*/
	if (timeoutEntry->prevKeyId != RX_MANAGER_INVALID_KEY_ID)
	{
		ILOG0_DD("rxManagerMulticastRekeyTimeout, vap ID - %d, prev key ID - %d", timeoutEntry->entryIndex, timeoutEntry->prevKeyId);
		VapDb_InvalidateGroupKey(timeoutEntry->entryIndex, timeoutEntry->prevKeyId);
	}
	timeoutEntry->timerActive = FALSE;
	
	/*Decrement number of active timeout*/
	RxManagerGlobalParameters.numOfActiveTimeouts --;	
}


/**********************************************************************************

rxManagerCheckMulticastRekey 


Description:
------------
This function checks if multicast rekey happend - i.e. the AP started using the new key id


Input: 
-----
timeoutEntry


		
Output:
-------


Returns:
--------
	void - 
	
**********************************************************************************/
#ifndef ENET_INC_ARCH_WAVE600D2 
static void rxManagerCheckMulticastRekey(Rd_t *rxDescriptor)
{
	uint8 vapId = rxDescriptor->rxQueueVapId;
	
	if (VapDb_GetVapMode(vapId) == VAP_MODE_AP)
	{
		return;	/*Probably should be an assert*/
	}
	/*relevant only if protected and not wep*/
	if ((rxDescriptor->protected) && (!rxDescriptor->securityWepType))
	{
		RxManagerTimeoutEntry_t *multicastRekeyEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[vapId];

		if ((multicastRekeyEntry->lastRxKeyId != rxDescriptor->keyId) && (multicastRekeyEntry->currKeyId != RX_MANAGER_INVALID_KEY_ID))
		{
			ILOG0_DDD("rxManagerCheckMulticastRekey, vap ID - %d, last Rx key ID - %d, key Id - %d", multicastRekeyEntry->entryIndex, multicastRekeyEntry->lastRxKeyId, rxDescriptor->keyId);
			if (multicastRekeyEntry->lastRxKeyId != RX_MANAGER_INVALID_KEY_ID)
			{
				VapDb_InvalidateGroupKey(vapId, multicastRekeyEntry->lastRxKeyId);
				if (multicastRekeyEntry->timerActive == TRUE)
				{
					CalendarWheel_RemoveEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t*)multicastRekeyEntry);
					multicastRekeyEntry->timerActive = FALSE;
					RxManagerGlobalParameters.numOfActiveTimeouts --;		
				
					if(0 == RxManagerGlobalParameters.numOfActiveTimeouts)
					{	
						/* Cancel the timer since there are no entries and timer is active */
						OSAL_RESET_TIMER_EXPLICIT(RX_MANAGER_TIMER, TASK_RX_MANAGER);
					}
				}
			}
			multicastRekeyEntry->lastRxKeyId = rxDescriptor->keyId;
		}
	}
}

#endif
/**********************************************************************************

rxManagerAddMulticastRekey 


Description:
------------
Called when VAP is added to init multicast entries

Input: 
-----
vapId

		
Output:
-------


Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerAddMulticastRekey(uint8 vapId)
{
	RxManagerTimeoutEntry_t *multicastRekeyEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[vapId];
	
	/*Initialize Multicast Rekey entry*/
	multicastRekeyEntry->prevKeyId = RX_MANAGER_INVALID_KEY_ID;
	multicastRekeyEntry->currKeyId = RX_MANAGER_INVALID_KEY_ID;
	multicastRekeyEntry->lastRxKeyId = RX_MANAGER_INVALID_KEY_ID;
}


/**********************************************************************************

rxManagerRemoveMulticastRekey 


Description:
------------
Called when VAP is added to init multicast entries

Input: 
-----
vapId

		
Output:
-------


Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerRemoveMulticastRekey(uint8 vapId)
{
	RxManagerTimeoutEntry_t *multicastRekeyEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[vapId];

	if (multicastRekeyEntry->timerActive == TRUE)
	{
		CalendarWheel_RemoveEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t*)multicastRekeyEntry);
		multicastRekeyEntry->timerActive = FALSE;
		RxManagerGlobalParameters.numOfActiveTimeouts --;		
		
		if(0 == RxManagerGlobalParameters.numOfActiveTimeouts)
		{	
			/* Cancel the timer since there are no entries and timer is active */
			OSAL_RESET_TIMER_EXPLICIT(RX_MANAGER_TIMER, TASK_RX_MANAGER);
		}
	}
	/*Set current key to invalid*/
	multicastRekeyEntry->currKeyId = RX_MANAGER_INVALID_KEY_ID;
}


/**********************************************************************************

rxManagerConvertCategoryToInternalIndex 


Description:
------------
This fucntion convert category field in management action frames to an internal
index to be used in the Rx manager internal lookup tables


Input: 
-----
category - category value


		
Output:
-------


Returns:
--------
	internal index 
	
**********************************************************************************/
static uint8 rxManagerConvertCategoryToInternalIndex(uint8 category)
{
	/* In this function optimization was done to the categories 0 - 11 : as few cycles 
	   as possible for these categories since they are the common ones */
	if(category < TDLS)
	{
		return category;
	}
	if(category <= SELF_PROTECTED)
	{
		if(category != TDLS)
		{
			/* TDLS should appear only in data packets */
			return category;
		}
	}
	if(category == VHT_CATEGORY)
	{
		return RX_MANAGER_ACTION_CATEGORY_VHT;
	}
	if(category == VENDOR_SPECIFIC)
	{
		return RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC;
	}
	if (category == VENDOR_SPECIFIC_PROTECTED)
	{
		return RX_MANAGER_ACTION_CATEGORY_VENDOR_SPECIFIC_PROTECTED;
	}
    if (category == HE_TWT_CATEGORY)
    {
        return RX_MANAGER_ACTION_CATEGORY_TWT;
    }

	/* None valid category */
	return RX_MANAGER_ACTION_CATEGORY_INVALID_CATEGORY;
	
}


/**********************************************************************************

rxManagerHandleOmnFrame 


Description:
------------
This fucntion handles action frame of type Operating Mode Notification

iwpriv wlan1 sDoSimpleCLI 3 4 0 0

Input: 
-----

		
Output:
-------


Returns:
--------
	
	
**********************************************************************************/
void rxManagerHandleOmnFrame(OperatingModeNotificationPayload_t *pFramePayload, uint8 rxQueueStaId, uint8 vapId)
{
	K_MSG* groupManagerMessage = NULL;
	GroupManagerStationNssHasChangedMessage_t *groupManagerMessageParameters = NULL;

	uint8 newNss = 0;
	uint8 newBw = 0;

    if(0 == pFramePayload->operatingModeField.rxNssType)
    {
		K_MSG *kMsg_p = OSAL_GET_MESSAGE(sizeof(LaOperatingModeNotificationMsg_t));
		LaOperatingModeNotificationMsg_t *laOperatingModeNotificationMsg_p = (LaOperatingModeNotificationMsg_t *)pK_MSG_DATA(kMsg_p);	
		
		newNss = pFramePayload->operatingModeField.rxNss;
		newBw = pFramePayload->operatingModeField.channelWidth;

		/* Note: rxNssType is not sent to LA since it has no use for it */
		laOperatingModeNotificationMsg_p->channelWidth = newBw;
		laOperatingModeNotificationMsg_p->rxNss = newNss;
		laOperatingModeNotificationMsg_p->stationId = rxQueueStaId;

		//("[OMN][rxManagerHandleOmnFrame] channelWidth = %d rxNss = %d stationId = %d",laOperatingModeNotificationMsg->channelWidth, laOperatingModeNotificationMsg->rxNss, laOperatingModeNotificationMsg->stationId);

		/* Send message to Link Adaptation */
		OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_OPERATING_MODE_NOTIFICATION_REQ, TASK_LINK_ADAPTATION, kMsg_p, vapId);	

		groupManagerMessage = OSAL_GET_MESSAGE( sizeof(GroupManagerStationNssHasChangedMessage_t));
    	groupManagerMessageParameters = ((GroupManagerStationNssHasChangedMessage_t *)groupManagerMessage->abData);
		memset(groupManagerMessageParameters, 0, sizeof(GroupManagerStationNssHasChangedMessage_t));

		groupManagerMessageParameters->stationIndex = rxQueueStaId;
		groupManagerMessageParameters->newNss = newNss;
		OSAL_SEND_MESSAGE(GROUP_MANAGER_STATION_NSS_HAS_CHANGED, TASK_GROUP_MANAGER, groupManagerMessage, vapId);
    }
	else
	{
		// TODO: handle the case of type 1 -multi user BF vs single user use
	}						
}

/**********************************************************************************

rxManagerHandleValidActionFrame 


Description:
------------
This fucntion handles a valid action frame(and no ack action)

Input: 
-----
rxDescriptor - the Rx descriptor 
frame - the address of the payload (including the phy metrics)
category - the category of the action frame
		
Output:
-------


Returns:
--------
	
	
**********************************************************************************/
static void rxManagerHandleValidActionFrame(Rd_t  *rxDescriptor, MANAGEMENT_BASIC_FRAME_HEADER *frame, uint8 category)
{
	Rd_t  *hostRxDescriptor = NULL; 
	K_MSG* psMsg = NULL;
	TsManagerPacketReceivedMessage_t* tsManagerMessageParameters = NULL;
	FrameControl_t* pFrameControl;
	LinkAdaptationFixedAntennaSelection_t* linkAdaptationAntSel;
	K_MSG* pAntSelMsg;
	uint8 actionCode = 0x0;
	uint8 smpsControl = 0x0;
	uint32 dramPtr = 0;
	byte localPersistant = 0x0;
	StaId stationIndex = 0;

#ifdef ENET_INC_ARCH_WAVE600
    TwtManagerPacketReceivedMessage_t* twtManagerMessageParameters = NULL;
	uint8 bandId;
#endif //ENET_INC_ARCH_WAVE600
	GroupManagerStationNssHasChangedMessage_t *groupManagerMessageParameters = NULL;
	K_MSG* groupManagerMessage = NULL;
	bool ccmpHeaderCheck = FALSE;


	pFrameControl = (FrameControl_t *)&frame->u16FrameControl;

	stationIndex = rxDescriptor->rxQueueStaId;
	
#ifdef ENET_INC_ARCH_WAVE600
	bandId = ConfigurationManager_GetBandForVap(rxDescriptor->rxQueueVapId);
#endif //ENET_INC_ARCH_WAVE600

    switch (category)
	{
		case BLOCK_ACK_CATEGORY:		
			/*if Frame is protected need to append payload to MAC header*/
			if (pFrameControl->protectedFrame == TRUE)
			{
				ccmpHeaderCheck = rxManagerRemoveCcmpHeader(rxDescriptor);
				if (ccmpHeaderCheck == FALSE)
				{
					//If the framelength is short, do not proceed further, as the RD has been already moved to Liberator input list
					break;
				}
			}
			
			psMsg = OSAL_GET_MESSAGE( sizeof(TsManagerPacketReceivedMessage_t));
    		tsManagerMessageParameters = ((TsManagerPacketReceivedMessage_t*)psMsg->abData);
			tsManagerMessageParameters->rxDescriptor = rxDescriptor;
	
			OSAL_SEND_MESSAGE(TS_MANAGER_ACTION_PACKET_RECEIVED, TASK_TS_MANAGER, psMsg, rxDescriptor->rxQueueVapId);
			break;

		case HT_CATEGORY:
			/*if Frame is protected need to append payload to MAC header*/
			if (pFrameControl->protectedFrame == TRUE)
			{
				ccmpHeaderCheck = rxManagerRemoveCcmpHeader(rxDescriptor);
				if (ccmpHeaderCheck == FALSE)
				{
					//If the framelength is short, do not proceed further, as the RD has been already moved to Liberator input list
					break;
				}
			}
			
			actionCode = ((FM_PAYLOAD_ACTION_COMMON *)frame_getPayloadPointerFromExistingManagementFrame(frame))->u8ActionCode;
			
			if(actionCode == HT_CATEGORY_SM_POWER_SAVE)
			{
				
#ifdef ENET_INC_ARCH_WAVE600
				GeneralStatistics.rxSmps[bandId]++;
#else
				GeneralStatistics.rxSmps++;
#endif //ENET_INC_ARCH_WAVE600

				//send message to LA in order to update the max station's NSS accordingly
				pAntSelMsg = OSAL_GET_MESSAGE(sizeof(LinkAdaptationFixedAntennaSelection_t));
				
				linkAdaptationAntSel = (LinkAdaptationFixedAntennaSelection_t *)pK_MSG_DATA(pAntSelMsg);
				linkAdaptationAntSel->vapId		= rxDescriptor->rxQueueVapId;
				linkAdaptationAntSel->stationId	= stationIndex;
				linkAdaptationAntSel->mask		= 0x0; //Don't care in case of SMPS 
				linkAdaptationAntSel->retTask	= TASK_RX_MANAGER;
				linkAdaptationAntSel->retMsg 	= RX_MANAGER_LA_SET_ANTS_CFM;	
				//Action Frame HT Category with action code HT_CATEGORY_SM_POWER_SAVE should have length including u8DialogToken
				if (rxDescriptor->dataLength >= ACTION_FRAME_GENERAL_MIN_LENGTH)
				{
					smpsControl = (((FM_PAYLOAD_ACTION_FRAME_GENERAL *)frame_getPayloadPointerFromExistingManagementFrame(frame))->u8DialogToken)&RX_MANAGER_SMPS_CONTROL_FIELD_MASK ;
					if(smpsControl == RX_MANAGER_SMPS_ENABLED) //see standard: 7.3.1.22 SM Power Control field
					{	
						linkAdaptationAntSel->mode = SET_SMPS;	
						OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_ANTENNA_SELECTION_REQ, TASK_LINK_ADAPTATION, pAntSelMsg, rxDescriptor->rxQueueVapId); 
						
						groupManagerMessage = OSAL_GET_MESSAGE( sizeof(GroupManagerStationNssHasChangedMessage_t));
						groupManagerMessageParameters = ((GroupManagerStationNssHasChangedMessage_t *)groupManagerMessage->abData);
						memset(groupManagerMessageParameters, 0, sizeof(GroupManagerStationNssHasChangedMessage_t));

						groupManagerMessageParameters->stationIndex = stationIndex;
						groupManagerMessageParameters->newNss = SPATIAL_STREAM_1;
						OSAL_SEND_MESSAGE(GROUP_MANAGER_STATION_NSS_HAS_CHANGED, TASK_GROUP_MANAGER, groupManagerMessage ,linkAdaptationAntSel->vapId);
					}
					else
					{
						if((smpsControl == RX_MANAGER_SMPS_DISABLED )||(smpsControl == RX_MANAGER_SMPS_SM_MODE_DYNAMIC )) /* In case SM Power Save Enabled  is 0 or Sm mode is Dynamic */
						{
							linkAdaptationAntSel->mode = UNSET_SMPS;
							OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_ANTENNA_SELECTION_REQ, TASK_LINK_ADAPTATION, pAntSelMsg, rxDescriptor->rxQueueVapId); 
						}
						else
						{
							localPersistant = pAntSelMsg->header.bPersistentMsg;
							OSAL_FREE_MESSAGE(pAntSelMsg,localPersistant);
						}
					}
				}
				
			}
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
			RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor;
			HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams); 
			break;				

		case VENDOR_SPECIFIC:		
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
			RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor;
			HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);	
			break;
			
		case VHT_CATEGORY:
			/*if Frame is protected need to append payload to MAC header*/
			if (pFrameControl->protectedFrame == TRUE)
			{
				ccmpHeaderCheck = rxManagerRemoveCcmpHeader(rxDescriptor);
				if (ccmpHeaderCheck == FALSE)
				{
					//If the framelength is short, do not proceed further, as the RD has been already moved to Liberator input list
					break;
				}
			}
			
			actionCode = ((FM_PAYLOAD_ACTION_COMMON *)frame_getPayloadPointerFromExistingManagementFrame(frame))->u8ActionCode;
			
			if (actionCode == VHT_CATEGORY_OPERATING_MODE_NOTIFICATION)
			{
				if (rxDescriptor->dataLength >= OMI_FRAME_MIN_LENGTH)
				{
#ifdef ENET_INC_ARCH_WAVE600
					GeneralStatistics.rxOmn[bandId]++;
#else
					GeneralStatistics.rxOmn++;
#endif //ENET_INC_ARCH_WAVE600
					rxManagerHandleOmnFrame((OperatingModeNotificationPayload_t *)frame_getPayloadPointerFromExistingManagementFrame(frame), stationIndex, (uint8)rxDescriptor->rxQueueVapId);
				} //if length check fails, OMI frame length is invalid and it will be sent to liberator without processing.
			}
#ifndef ENET_INC_ARCH_WAVE600
			// In gen6 we don't support being beamformee (even not as STA). Need to check why we got here.
			// Above is the Comment from GroupManagerPhyDriver.c. PhyDrv_SetMuGroupUsp(). In Gen6 this will cause MAC FATAL and as part of SDL we dont want to Crash.
			// So this part of code is compiled out for WAVE600
			else if(actionCode == VHT_CATEGORY_GROUP_ID_MANAGEMENT)
			{
				if (rxDescriptor->dataLength >= GIDM_FRAME_MIN_LENGTH)
				{
					gidmManagerHandleRxActionFrame((GroupIdManagementPayload_t *)frame_getPayloadPointerFromExistingManagementFrame(frame), rxDescriptor->rxQueueVapId);
				}//if length check fails, frame length of GIDM is invalid and it will be sent to liberator without processing.
			}
#endif
			// Push RD to liberator
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
			RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDescriptor;
			HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);	
			break;
            
#ifdef ENET_INC_ARCH_WAVE600
        case HE_TWT_CATEGORY:
            /*if Frame is protected need to append payload to MAC header*/
			if (pFrameControl->protectedFrame == TRUE)
			{
				ccmpHeaderCheck = rxManagerRemoveCcmpHeader(rxDescriptor);
				if (ccmpHeaderCheck == FALSE)
				{
					//If the framelength is short, do not proceed further, as the RD has been already moved to Liberator input list
					break;
				}
			}
            // Send message to TWT module with the rxDescriptor 
            psMsg = OSAL_GET_MESSAGE(sizeof(TwtManagerPacketReceivedMessage_t));
            twtManagerMessageParameters = ((TwtManagerPacketReceivedMessage_t*)pK_MSG_DATA(psMsg));
            twtManagerMessageParameters->rxDescriptor = rxDescriptor;
            twtManagerMessageParameters->framePayload = (TwtFramePayload_t *)frame_getPayloadPointerFromExistingManagementFrame(frame);
            twtManagerMessageParameters->stationId = stationIndex;
            OSAL_SEND_MESSAGE(TWT_MANAGER_TWT_FRAME_RECIEVED, TASK_TWT_MANAGER, psMsg, rxDescriptor->rxQueueVapId);
            break;
#endif //ENET_INC_ARCH_WAVE600 

		default:
			/* Handle payload to host */
			/* get RD from the host pool */
			RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_FREE_FORWARD_RDS_LIST;
			HwQManager_PopPacket(&RxManagerHwQueueManagerRequestParams);
			hostRxDescriptor = (Rd_t  *)RxManagerHwQueueManagerRequestParams.pHeadDesc;
			if((Rd_t  *)NULL_RD != hostRxDescriptor)
			{
				dramPtr = CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(hostRxDescriptor->dramPointer);
				if((
dramPtr & MAC_XBAR_WLAN_IP_MASK) == System_GetXbarAddress())
				{
					rxManagerHandleBadPtrHostRd(rxDescriptor, hostRxDescriptor);
				}
				else
				{
					rxManagerHandleOneForwardRd(rxDescriptor, hostRxDescriptor);
				}
			}
			else
			{
				/* There are no free RDs in the host pool */
				/*Put RD to pending list*/
				hostInterfaceRings_PutOnRdsList(rxDescriptor,rxDescriptor,&forwardRDsPendingList);
			}
			break;
	}
	
}

/**********************************************************************************

rxManagerHandleBadPtrHostRd 


Description:
------------
This fucntion find the next valid RD from free list and prepare the host DMA, In
case no free RD add the RD in pending list.


Input: 
-----
rxDescriptor - the source Rx descriptor 
hostRxDescriptor - host Rx descriptor 


		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerHandleBadPtrHostRd(Rd_t *rxDescriptor, Rd_t *hostRxDescriptor)
{
	uint32 dramPtr = 0;
	uint32 system_XbarAddr = 0;

	system_XbarAddr = System_GetXbarAddress();
	
	do
	{
		dramPtr = CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(hostRxDescriptor->dramPointer);
		if((dramPtr & MAC_XBAR_WLAN_IP_MASK) != system_XbarAddr)
		{
			rxManagerHandleOneForwardRd(rxDescriptor, hostRxDescriptor);
			break;
		}
		// Push host rx descriptor to the bad pointer list.
		RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_READY_LIST_BAD_PTR_RDS;
		RxManagerHwQueueManagerRequestParams.pHeadDesc = hostRxDescriptor;
		HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
		RxManagerBadForwardRdCnt++;
		
		if(RxManagerBadForwardRdCnt == NUM_OF_FORWARD_RX_DESCRIPTORS)
		{
			FATAL("FORWARD Bad Pointer List has crossed the max boundary");
		}

		//Pop next available RD from the free forward list.
		RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_FREE_FORWARD_RDS_LIST;
		HwQManager_PopPacket(&RxManagerHwQueueManagerRequestParams);
		hostRxDescriptor = (Rd_t  *)RxManagerHwQueueManagerRequestParams.pHeadDesc;
		if(((Rd_t  *)NULL_RD == hostRxDescriptor))
		{
			hostInterfaceRings_PutOnRdsList(rxDescriptor,rxDescriptor,&forwardRDsPendingList);
		}
		
	}while(((Rd_t  *)NULL_RD != hostRxDescriptor));
}

/**********************************************************************************

rxManagerHandleOneForwardRd 


Description:
------------
This fucntion prepares the host Rx descriptor, prepares lists to be released after 
DMA operation is completed and calls DMA


Input: 
-----
rxDescriptor - the source Rx descriptor 
hostRxDescriptor - host Rx descriptor 

		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerHandleOneForwardRd(Rd_t *rxDescriptor , Rd_t *hostRxDescriptor)

{
	uint32 currentTsf;
	dmaDesc_t* pDmaDesc = NULL;

    /* Copy frame to DDR */
	/*Handle FW RD*/
	if (RxManagerForwardToHostHead == NULL)
	{
	    RxManagerForwardToHostHead = rxDescriptor;
	}
	else
	{
	    RxManagerForwardToHostTail->nextRd = SET_NEXT_RD(rxDescriptor);
	}
	RxManagerForwardToHostTail = rxDescriptor;
	/*Handle Foward RD*/
	if (RxManagerHostForwardHead == NULL)
	{
	    RxManagerHostForwardHead = hostRxDescriptor;
	}
	else
	{
	    RxManagerHostForwardTail->nextRd = SET_NEXT_RD(hostRxDescriptor);
	}
	RxManagerHostForwardTail = hostRxDescriptor;
	/*Initialize forward RD*/
	hostRxDescriptor->dramByteOffset = 0;
	/*Copy relevant data from Received RD to forward RD*/
	hostRxDescriptor->rxQueueVapId = rxDescriptor->rxQueueVapId;
	hostRxDescriptor->dataLength = rxDescriptor->dataLength;
	hostRxDescriptor->rxQueueStaId = rxDescriptor->rxQueueStaId;
	hostRxDescriptor->frameSubtype = rxDescriptor->frameSubtype;
	hostRxDescriptor->frameType = rxDescriptor->frameType;
	hostRxDescriptor->ethType = rxDescriptor->ethType;
	hostRxDescriptor->eop = rxDescriptor->eop;
	hostRxDescriptor->sop = rxDescriptor->sop;
	/* Set timestamp in RD */
#if defined (ENET_INC_ARCH_WAVE600)
	currentTsf = Pac_TimGetTsfLowPerBand(ConfigurationManager_GetBandForVap(rxDescriptor->rxQueueVapId));
#else
	currentTsf = Pac_TimGetTsfLow();
#endif
	hostRxDescriptor->ttlCount = HOST_INTERFACE_ACCELERATOR_CONVERT_USEC_TO_TTL_UNITS(currentTsf);
	pDmaDesc = DmaManager_AllocateDesc();
	pDmaDesc->destination = CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(hostRxDescriptor->dramPointer);
	pDmaDesc->direction = DMA_DIRECTION_SHRAM_TO_HOST;     
	pDmaDesc->source = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset);
	pDmaDesc->length = rxDescriptor->dataLength;
	pDmaDesc->clientId =  RxManagerGlobalParameters.dmaForwardClientId;
	DmaManager_NewDmaRequest(pDmaDesc);     

}

/**********************************************************************************

rxManagerInitializeRxDescriptorsPools 


Description:
------------
This fucntion initialize the FW and the host Rx descriptors pools
the payload in the host Rx descriptors is initialize by another entity


Input: 
-----
category - category value


		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerInitializeRxDescriptorsPools(void)
{
	uint32 i = 0;
	HwQueueManagerRequestParams_t DummyRd_RxManagerHwQueueManagerRequestParams;

	/* Initialize the RD pool of the FW */
	memset32(FwRxDescriptorsPool , 0 , CONVERT_BYTES_TO_WORDS(NUM_OF_FW_RX_DESCRIPTORS*sizeof(Rd_t)));
	memset(FwRxDescriptorsPayload , 0 , sizeof(FwRxDescriptorsPayload));
	for(i = 0; i < NUM_OF_FW_RX_DESCRIPTORS ; i++)
	{
		FwRxDescriptorsPool[i].nextRd = SET_NEXT_RD(&FwRxDescriptorsPool[i+1]);
		FwRxDescriptorsPool[i].dramPointer = CONVERT_BYTE_ADDRESS_TO_DRAM_POINTER(CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32) FwRxDescriptorsPayload[i]));
		FwRxDescriptorsPool[i].rdSource = RD_SOURCE_TO_LOCAL_MEMORY;
	}
	/* Initialize the last RD to point to null */

	FwRxDescriptorsPool[i-1].nextRd = NEXT_RD_NULL;

	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_FREE_FW_RDS_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = FwRxDescriptorsPool;
	RxManagerHwQueueManagerRequestParams.pTailDesc = &FwRxDescriptorsPool[i-1];
	HwQManager_PushPacketListToTail(&RxManagerHwQueueManagerRequestParams);
    
	/* Initialize the RD pool of the host */
	memset32(HostRxDescriptorsPool , 0 , CONVERT_BYTES_TO_WORDS(sizeof(HostRxDescriptorsPool)));

	
	/* The payload pointer is initialize by the host or by Rx in flow (depends on dataPathMode)*/

	for(i = 0; i < NUM_OF_HOST_RX_DESCRIPTORS; i++)
	{
		HostRxDescriptorsPool[i].nextRd = SET_NEXT_RD(&HostRxDescriptorsPool[i+1]);

		if (i < INDEX_OF_FORWARD_FIRST_RX_DESCRIPTOR)
		{			
			HostRxDescriptorsPool[i].rdSource = RD_SOURCE_MANAGEMENT;
		}
		else if (i < INDEX_OF_HOST_DATA_AND_LOGGER_FIRST_RX_DESCRIPTOR)
		{			
			HostRxDescriptorsPool[i].rdSource = RD_SOURCE_FORWARD;
		}
		else
		{		
			HostRxDescriptorsPool[i].rdSource = RD_SOURCE_TO_HOST_MEMORY;
		}
	}

	//assign last to NULL
	HostRxDescriptorsPool[i-1].nextRd = NEXT_RD_NULL;
	
	/* Put  Dummy RD on Dummy Rx list*/

	memset(DummyRxDescriptorsPool , 0 , NUM_OF_DUMMY_RX_DESCRIPTORS * sizeof(Rd_t ));

	DummyRxDescriptorsPool[0].nextRd = NEXT_RD_NULL;

	DummyRd_RxManagerHwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	DummyRd_RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_DUMMY_RD_LIST;
	DummyRd_RxManagerHwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	DummyRd_RxManagerHwQueueManagerRequestParams.pHeadDesc = &DummyRxDescriptorsPool[0];
	DummyRd_RxManagerHwQueueManagerRequestParams.pTailDesc = &DummyRxDescriptorsPool[0];
	
	HwQManager_PushPacketListToTail(&DummyRd_RxManagerHwQueueManagerRequestParams);
	
    forwardRDsPendingList.pHead = (Rd_t*)NULL_RD;
    forwardRDsPendingList.pTail = (Rd_t*)NULL_RD;

}

/**********************************************************************************

rxManagerInitializeTimeoutEntries 


Description:
------------
This fucntion initialize the timeout entries of the fragmentation and the class violation


Input: 
-----
category - category value


		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerInitializeTimeoutEntries(void)
{
	uint32 i = 0;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 bandId;
#endif

	/* The next three databases are alligned to words */
	memset(&RxManagerTimeoutnEntriesDataBase, 0, sizeof(RxManagerTimeoutnEntriesDataBase_t));

	RxManagerFreeFragmentationEntriesList = NULL;
	RxManagerFwClassViolationBitmap = 0;
	RxManagerFwClassViolationCacheIsFull = 0;

	/* Initialize free fragmentation entries list */
	RxManagerFreeFragmentationEntriesList = RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries;
	for(i = 0; i < RX_MANAGER_NUM_OF_FRAGMENTATION_ENTRIES ; i++)
	{
		/* The previous field is not needed for the free list */
		RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries[i].calendarWheelEntry.nextIndex = i + 1;
		RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries[i].entryIndex = i;
		RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries[i].type = RX_MANAGER_ENTRY_TYPE_FRAGMENTATION;
	}
	/* Initialize the last entry to point to null */
	RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries[RX_MANAGER_NUM_OF_FRAGMENTATION_ENTRIES - 1].calendarWheelEntry.nextIndex = RX_MANAGER_INVALID_FRAGMENTATION_ENTRY;

	/* Inialize Hw class violation entries */
#ifdef ENET_INC_ARCH_WAVE600
	for(bandId = CONFIGURATION_MANAGER_BAND_0; bandId < NUM_OF_CONFIGURATION_MANAGER_BANDS ; bandId++)
	{
		for(i = 0; i < 16 /* RX_CLASSIFIER_FRAME_CLASS_VIOLATION_CACHE_SIZE */ ; i++)
		{
			RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[bandId][i].entryIndex = i;
			RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[bandId][i].type = RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_HW;
		}
	}
#else
	for(i = 0; i < 16 /* RX_CLASSIFIER_FRAME_CLASS_VIOLATION_CACHE_SIZE */ ; i++)
	{
		RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[i].entryIndex = i;
		RxManagerTimeoutnEntriesDataBase.rxManagerHwClassViolationEntries[i].type = RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_HW;
	}

#endif
	/* Inialize Fw class violation entries */
	for(i = 0; i < RX_MANAGER_FW_CLASS_VIOLATION_CACHE_SIZE ; i++)
	{
		RxManagerTimeoutnEntriesDataBase.rxManagerFwClassViolationEntries[i].entryIndex = i;
		RxManagerTimeoutnEntriesDataBase.rxManagerFwClassViolationEntries[i].type = RX_MANAGER_ENTRY_TYPE_CLASS_VIOLATION_FW;
	}

	/* Inialize MulticastRekey entries */
	for(i = 0; i < HW_NUM_OF_VAPS ; i++)
	{
		RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[i].entryIndex = i;
		RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[i].type = RX_MANAGER_ENTRY_TYPE_MULTICAST_REKEY;
	}	
}

/**********************************************************************************

rxManagerFindFreeCellInStationArray 


Description:
------------
This fucntion finds a free cell in the fragmentation array in the internal station 
database and allocates it 


Input: 
-----
stationId - the station from where to allcate a cell
fragmentationEntry - the address of the fragmentation entry

		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerFindFreeCellInStationArray(StaId stationId, RxManagerTimeoutEntry_t *fragmentationEntry)
{
	uint32 i = 0;
	uint8 entryWasFound = FALSE;
	uint16 entryIndex = 0;
	
	while(!entryWasFound)
	{
		ASSERT(i < MAX_NUM_OF_FRAGMENTATIONS_PER_STA);

		entryIndex = RxManagerStationDatabase[stationId].fragementationEntriesIndexes[i];
		if(RX_MANAGER_INVALID_FRAGMENTATION_ENTRY == entryIndex)
		{
			RxManagerStationDatabase[stationId].fragementationEntriesIndexes[i] = fragmentationEntry->entryIndex;
			fragmentationEntry->indexInStationArray = i;
			entryWasFound = TRUE;
		}
		i++;
	}
}


/**********************************************************************************

rxManagerNewFragmentationTimeoutEntry 


Description:
------------
This fucntion adds a new fragmentation timeout entry: fill the entry with proper parameters, 
update counters and start to count timeout

Input: 
-----
fragmentationEntry - the address of the new fragmentation timeout entry
fragmentationEvent - a pointer to the fragmentation event from the Rx post processing  
stationId - the station to which this fragmentation belongs
vapId - the vap id to which the station belongs

		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerNewFragmentationTimeoutEntry(RxManagerTimeoutEntry_t *fragmentationEntry, RxPpFifoFragLine_t *fragmentationEvent, StaId stationId, uint8 vapId)
{
	/* Fill entry with relevant parameters */
	fragmentationEntry->sequenceNumber = fragmentationEvent->sn;
	fragmentationEntry->vapId = vapId;
	fragmentationEntry->stationId = stationId;
	fragmentationEntry->tid = fragmentationEvent->tid;

    /* Start counting timeout for the fragmentation */
	CalendarWheel_AddEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)fragmentationEntry, RX_MANAGER_FRAGMENTATION_TIMEOUT_UNITS);
		
	/* Update fragmentation counters */
	RxManagerStationDatabase[stationId].numOfActiveFragmentation++;
	RxManagerVapDatabase[vapId].numOfActiveFragmentation++;
	RxManagerGlobalParameters.numOfActiveTimeouts++;		
}

/**********************************************************************************

rxManagerSetPnStaMode 


Description:
------------
This function gets the initial PN from the Vap manager and set it in the Vap manager parameters database

Input: 
-----
vapId - Vap index, the Vap which the PN belongs to
*pnAddress - the initial PN number to set

		
Output:
-------
RxManagerVapDatabase[vapId].lastPN - set to pnAddress


Returns:
--------

	
**********************************************************************************/

void rxManagerSetPnStaMode(uint8 vapId, uint8 *pnAddress, uint8 keyId)
{
	ASSERT(keyId < NUM_OF_KEY_ID);
	MEMCPY(RxManagerVapDatabase[vapId].lastPN[keyId], pnAddress, PN_LENGTH);
}

/**********************************************************************************

ReplayDetection_DoReplayDetectionStaMode 


Description:
------------
This function perform a replay detection test for the upcoming frame. check the PN field and compare it to the last PN set.

Input: 
-----
*lastPN - the last PN set, this filed is located on VapParams database
*pRd - the new frame, contains the field of it's own PN.

		
Output:
-------
If fail - send statistics and mark the statusRxppError field of the pRd as TRUE.
If pass - goes to ReplayDetection_SaveNewPnStaMode.


Returns:
--------

	
**********************************************************************************/

bool rxManagerDoReplayDetectionStaMode(uint8* lastPN, Rd_t* pRd)
{
	int16 	res = 0;
	uint8*	currentPN = NULL;
	int8	index;
	uint8 	frameType = pRd->frameType;
	uint8 	vapIndex = pRd->rxQueueVapId;
	bool	pass;

	currentPN = (uint8*)(pRd) + PN_OFFSET;
	for (index = PN_LENGTH-1 ; index >= 0 ; index--)
	{	
		res = currentPN[index] - lastPN[index];
		if (res != 0)
		{
			break;
		}
	}
	if (res <= 0)
	{
#ifdef REPLAY_DETECTION_TRACE
		ILOG0_DD("[rxManagerDoReplayDetectionStaMode] URID. FAIL!!!! last PN is : %d, current PN is : %d",*lastPN,*currentPN);	
#endif
		if (frameType == FM_CONTROL_FTYPE_DATA)
		{
			pRxCounters->rxppVapCounts[vapIndex].replayData++;
		}
		else if(frameType == FM_CONTROL_FTYPE_MANAGEMENT)
		{			
						pRxCounters->rxppVapCounts[vapIndex].replayMngmnt++;
		}
		/*Mark RXPP error*/
		pRd->statusRxppError = 1;
		pass = FALSE;
	}
	else
	{
#if !defined (ENET_INC_ARCH_WAVE600)	
		if ((pRd->securityStatus & RD_TKIP_MIC_FAILURE) == 0)
#else
		if (pRd->micFailure == 0)
#endif
		{
			rxManagerSaveNewPnStaMode(lastPN, currentPN);
			pass = TRUE;
		}
		else
		{
			pass = FALSE;
		}
	}
	return pass;
}

/**********************************************************************************

ReplayDetection_SaveNewPnStaMode 


Description:
------------
This is a service function that only has one job - change the lastPN field to the current PN if needed.

Input: 
-----
*lastPN - the last PN set, this filed is located on VapParams database
*currentPN - the upcoming PN from the new frame arrived.

		
Output:
-------
sets lastPN to be currentN


Returns:
--------

	
**********************************************************************************/

void rxManagerSaveNewPnStaMode(uint8 *lastPN, uint8 *currentPN)
{
	MEMCPY(lastPN, currentPN, PN_LENGTH);
}


/**********************************************************************************

rxManagerFindFragmentationEntryAndRemove 


Description:
------------
This fucntion finds a fragmentation entry if exist and removes it - stop counting timeout, return to 
the free list and update counters

Input: 
-----
fragmentationEvent - a pointer to the fragmentation event from the Rx post processing  
stationId - the station in which to look for the entry
		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerFindFragmentationEntryAndRemove(RxPpFifoFragLine_t *fragmentationEvent, StaId stationId)
{
    uint16 entryIndex = 0;
	RxManagerTimeoutEntry_t *fragmentationEntry = NULL;
	uint8 i = 0;
	
	/* If an entry is not found nothing should be done - an entry was already removed or never allocated */	
	while(i < MAX_NUM_OF_FRAGMENTATIONS_PER_STA)
	{
		entryIndex = RxManagerStationDatabase[stationId].fragementationEntriesIndexes[i];				
		if(RX_MANAGER_INVALID_FRAGMENTATION_ENTRY != entryIndex)
		{
			fragmentationEntry = RxManagerTimeoutnEntriesDataBase.rxManagerFragmentationEntries + entryIndex;
			if(fragmentationEntry->tid == fragmentationEvent->tid)
			{
				/* The entry was found since each tid can have only one fragmentation simoltaneously */
	
				/* Stop counting timeout for this fragmentation */
				CalendarWheel_RemoveEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)fragmentationEntry);
	
				/* Relase entry - update counters and return to free list */
				rxManagerReleaseFragmentationTimeoutEntry(fragmentationEntry);
				
				/* Stop searching in the array if correlating entry was found */
				break;
			}
		}
		i++;
	}
}

/**********************************************************************************

rxManagerIsMacAddressExist 


Description:
------------
This fucntion returns if a MAC address exists in the FW class violation cache

Input: 
-----
macAddress - the MAC address to search
		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static uint8 rxManagerIsMacAddressExist(IEEE_ADDR *macAddress)
{
    uint32 tempBitmap = 0;
	uint32 index = 0;
	uint8 isEqual = FALSE;

	tempBitmap = RxManagerFwClassViolationBitmap;
		
	while(tempBitmap)
	{
		index = Utils_FindFirstSetAndClear(&tempBitmap);
		isEqual = boIEEE_ADDR_IsEqual(macAddress, &RxManagerTimeoutnEntriesDataBase.rxManagerFwClassViolationEntries[index].stationMacAddress);
		if(isEqual)
		{
			break;		
		}		
	}
	return isEqual;
}

/**********************************************************************************

rxManagerFindFreeFwClassViolationEntry 


Description:
------------
This fucntion returns a free entry in the FW class violation cache 
(it is assmued that there is at least one free entry)
it also sets the correlating bit in the bitmap to indicate that this entry is occupied

Input: 
-----

		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static uint8 rxManagerFindFreeFwClassViolationEntry(void)
{
    uint16 tempBitmap = 0;
	uint8 index = 0;

    /* '0' represents free entry */
	tempBitmap = ~RxManagerFwClassViolationBitmap;
	/* Mask irelevant bits */
	tempBitmap &= RX_MANAGER_FW_CLASS_VIOLATION_CACHE_MASK;

	index = Utils_FindFirstSet(tempBitmap);

	RxManagerFwClassViolationBitmap |= RX_MANAGER_FW_CLASS_VIOLATION_MASK_FREE_ENTRY << index;
		
	return index;
}

/**********************************************************************************

rxManagerStaManagerSendConfirm 


Description:
------------
Sends confirmation message back to STA Manager

Input: 
-----
Client ID
SID

		
Output:
-------


Returns:
--------

	
**********************************************************************************/
static void rxManagerStaManagerSendConfirm(StaId sid)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_RXM_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = sid;
	/*Send confirmation message*/
	ILOG2_DD("RX Manager, Send Confirmation, Client %d, SID %d", confirmMessage->clientId, sid);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

rxManagerRemoveCcmpHeader 


Description:
------------
For management frames with protected bit on, remove CCMP header by appending payload to MAC header

Input: 
-----
Client ID
SID

		
Output:
-------


Returns:
--------
Boolean (TRUE if successfull, FALSE if the framelength is short)

	
**********************************************************************************/
static bool rxManagerRemoveCcmpHeader(Rd_t  *rxDescriptor)
{
	RxMpdu_t *rxMpdu = NULL;
	MANAGEMENT_BASIC_FRAME_HEADER *frame = NULL;
    FM_PAYLOAD_ACTION_COMMON *pFramePayload;
	uint8 *dst = NULL;
	uint8 *src = NULL;
	uint16 dataToCopy = 0;
    uint32 frame_sizeOfExistingManagementFrameHeader_res;
	
	if (rxDescriptor->dataLength > (sizeof(MANAGEMENT_BASIC_FRAME_HEADER) + CCMP_HEADER_LENGTH + CCMP_MIC_LENGTH))
	{
		rxMpdu = (RxMpdu_t *)CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset);
	    frame = (MANAGEMENT_BASIC_FRAME_HEADER *)rxMpdu->frame;
    	pFramePayload = (FM_PAYLOAD_ACTION_COMMON *)frame_getPayloadPointerFromExistingManagementFrame(frame);
		dst = (uint8 *)pFramePayload;
		src = dst + sizeof(CCMP_HEADER);

        frame_sizeOfExistingManagementFrameHeader_res = frame_sizeOfExistingManagementFrameHeader(frame);

		ASSERT(rxDescriptor->dataLength > (frame_sizeOfExistingManagementFrameHeader_res + CCMP_HEADER_LENGTH + CCMP_MIC_LENGTH));
		dataToCopy = rxDescriptor->dataLength - frame_sizeOfExistingManagementFrameHeader_res - CCMP_HEADER_LENGTH - CCMP_MIC_LENGTH;        

		//original length include both ccmp header and mic length so we remove them
		rxDescriptor->dataLength = rxDescriptor->dataLength - (CCMP_HEADER_LENGTH + CCMP_MIC_LENGTH);
		MEMCPY(dst, src, dataToCopy);
		return TRUE;
	}
	else //length too short - invalid
	{
		RxManager_IgnoreRd(rxDescriptor);
		return FALSE;
	}
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

#if defined(ENET_INC_ARCH_WAVE600)
/**********************************************************************************

HwEventsRxManager_RdReady 


Description:
------------
handle the interupt from the HW queue manager - send message to the Rx manager

Input: 
-----	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void HwEventsRxManager_RdReady()
{
	/* Mask all not empty interrupts in HW event */
	EventManager_TurnOffEvent(EVENT_ID_HIGH_PRI_RX_PD_READY); 

	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_RDS_LIST_NOT_EMPTY, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);
}


#else //Wave500
/**********************************************************************************

isr_RxManager_RdReady 


Description:
------------
handle the interupt from the HW queue manager - send message to the Rx manager

Input: 
-----	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_RxManager_RdReady()
{
	/* Mask all not empty interrupts in HW event */
	EventManager_TurnOffEvent(EVENT_ID_RX_RD_LIST_NOT_EMPTY); 

	OSAL_SEND_NO_DATA_MESSAGE(RX_MANAGER_RDS_LIST_NOT_EMPTY, TASK_RX_MANAGER, VAP_ID_DO_NOT_CARE);
}

#endif //ENET_INC_ARCH_WAVE600

/**********************************************************************************

rxManagerAddVap 


Description:
------------
handle Add vap event - configure Classifier according to operation mode recieved in AddVap request

Input: 
-----	
K_MSG *pMsg	
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/

static void rxManagerAddVap(K_MSG *pMsg)
{
	uint32 vapIdx;
	UMI_ADD_VAP* addVapStructurePtr = NULL;
	
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	addVapStructurePtr = (UMI_ADD_VAP*) EXTRACT_VAP_MANAGER_MSG(pMsg);

	vapIdx = addVapStructurePtr->vapId;


	RxManagerGlobalParameters.numOfActiveVaps++;

	/*Initialize Multicast Rekey entry*/
	rxManagerAddMulticastRekey(vapIdx);

#if defined (ENET_INC_ARCH_WAVE600)
	if(addVapStructurePtr->operationMode == OPERATION_MODE_SNIFFER)
	{
		ILOG2_V("RX manager - setting sniffer mode configuration ");

		/* RX COORDINATOR - set sniffer mode configuration */
		RxCoordinator_SetSnifferModeConfiguration(vapIdx);
		ILOG2_V("RxCoordinator- set sniffer mode configuration - done! ");
	}
#else //!defined (ENET_INC_ARCH_WAVE600)
	if (RxManagerGlobalParameters.numOfActiveVaps == 1)
	{
		RxClassifier_HandleRxManagerAddVap(addVapStructurePtr);
	}
#endif	
	/*return message*/
	FILL_VAP_MANAGER_CONFIRM_MSG(pMsg, vapIdx, VAP_MANAGER_ADD_VAP, BSS_MANAGER_VAP_MANAGER_RX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, pMsg, vapIdx);
}


/**********************************************************************************

rxManagerRemoveVap 

Description:
------------
	handle Remove  vap event 

Input: 
-----	
K_MSG *pMsg	
		
Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerRemoveVap(K_MSG *pMsg)
{
	uint32 vapIdx;
	UMI_REMOVE_VAP* removeVapStructurePtr = NULL;
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	removeVapStructurePtr = (UMI_REMOVE_VAP*) EXTRACT_VAP_MANAGER_MSG(pMsg);

	vapIdx = removeVapStructurePtr->vapId;

	// Count number of active VAPs (needed in order to know when the first vap is added in order to configure the classifier)
	RxManagerGlobalParameters.numOfActiveVaps--;

	rxManagerRemoveMulticastRekey(vapIdx);

	/*return message*/
	FILL_VAP_MANAGER_CONFIRM_MSG(pMsg, vapIdx, VAP_MANAGER_REMOVE_VAP, BSS_MANAGER_VAP_MANAGER_RX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, pMsg, vapIdx);	
}


/**********************************************************************************

rxManagerLaSetAntsCfm 

Description:
------------
	handle LA confirmation on set antena req

Input: 
-----	
K_MSG *pMsg	
		
Returns:
--------
	void - 
	
**********************************************************************************/
static void rxManagerLaSetAntsCfm(K_MSG *pMsg)
{
	UNUSED_PARAM(pMsg);
}
 
 /**********************************************************************************
 
 RxManager_SetGroupKey 
 
 
 Description:
 ------------
 Handles Set Group Key
 
 Input: 
 -----
 
	 
		 
 Output:
 -------
	 
 
 Returns:
 --------
	 void - 
	 
 **********************************************************************************/
void RxManager_SetGroupKey(uint8 vapId, UMI_SET_KEY *psSetKey)
{
	RxManagerTimeoutEntry_t *multicastRekeyEntry = &RxManagerTimeoutnEntriesDataBase.rxManagerMulticastRekeyEntries[vapId];
	uint8 previousNumberOfActiveTimeouts = RxManagerGlobalParameters.numOfActiveTimeouts;

	if (VapDb_GetVapMode(vapId) == VAP_MODE_AP)
	{
		return;
	}
	/*Check if rekey*/
	if (multicastRekeyEntry->currKeyId != RX_MANAGER_INVALID_KEY_ID)
	{
		/*Check encryption type change*/		 
		if (VapDB_GetDataEncryptionType(vapId) != psSetKey->u16CipherSuite)
		{
		multicastRekeyEntry->lastRxKeyId = RX_MANAGER_INVALID_KEY_ID;
		multicastRekeyEntry->prevKeyId = RX_MANAGER_INVALID_KEY_ID;
		}
		else if (VapDb_GroupKeyValid(vapId, psSetKey->u16KeyIndex) == TRUE)
		{
			multicastRekeyEntry->lastRxKeyId = RX_MANAGER_INVALID_KEY_ID;
		}
	}
	if (multicastRekeyEntry->timerActive == TRUE)
	{
		CalendarWheel_RemoveEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t*)multicastRekeyEntry);
		multicastRekeyEntry->timerActive = FALSE;
		RxManagerGlobalParameters.numOfActiveTimeouts --;		 
	}
	/*Set prev to current*/
	multicastRekeyEntry->prevKeyId = multicastRekeyEntry->currKeyId;
	/*update current*/
	multicastRekeyEntry->currKeyId = psSetKey->u16KeyIndex;	 
	/* Start counting timeout for the rekey */
	CalendarWheel_AddEntry(&RxManagerCalendarWheel, (CalendarWheelEntry_t *)multicastRekeyEntry, RX_MANAGER_MULTICAST_REKEY_TIMEOUT_UINTS);
	multicastRekeyEntry->timerActive = TRUE;
	RxManagerGlobalParameters.numOfActiveTimeouts ++;		 
	if(0 == previousNumberOfActiveTimeouts)
	{
		/* Setting the timer since it was not active and there are new entries */
		OSAL_SET_TIMER_EXPLICIT(RX_MANAGER_TIMER, RX_MANAGER_CALENDAR_WHEEL_TIMER, TASK_RX_MANAGER);
	}
}


/**********************************************************************************

RxManager_ClearGroupKey 


Description:
------------
Handles Set Group Key

Input: 
-----

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void RxManager_ClearGroupKey(uint8 vapId)
{
	rxManagerRemoveMulticastRekey(vapId);
}
    
/***********************************************************************
* ClientModeMcastFilterClearTheRing
* 
* Description:
* ------------
* clear the ring
* 
* Input:
* ------
* pClientModeMcastFilterRingElement: pointer to the 
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
void ClientModeMcastFilterClearTheRing(clientModeMcastFilterRingElement_t *pClientModeMcastFilterRingElement)
{
	memset(pClientModeMcastFilterRingElement, 0, sizeof(clientModeMcastFilterRingElement_t)*CLIENT_MODE_MCAST_FILTER_RING_SIZE);
}
#endif
/**********************************************************************************

RxManager_GetStaFrameClass 

Description:
------------
	
Input: 
-----	

		
Returns:
--------

	
**********************************************************************************/
#if defined(ENET_INC_ARCH_WAVE600)
static uint8 RxManager_GetStaFrameClass(StaId stationId)
{
	if (StaDb_getStaState(stationId) != 	STA_STATE_FREE)
		return FRAME_CLASS_3;
	else
		return FRAME_CLASS_2;
}
#endif


static void RxManager_UpdateDutOpeationMode(K_MSG* pMsg)
{
	dutOperationModeUdateMsgParams_t* dutOpModeUpdateMsg = (dutOperationModeUdateMsgParams_t *)pK_MSG_DATA(pMsg);

	rxManagerDutDb.dutOperationMode = dutOpModeUpdateMsg->operationMode;
	/*Send confirmation to DUT*/
	OSAL_SEND_MESSAGE(dutOpModeUpdateMsg->retMsg,dutOpModeUpdateMsg->retTask,pMsg,pMsg->header.vapId);
	
}
UmiOperationMode_e RxManager_GetDutOpeationMode(void)
{
	return rxManagerDutDb.dutOperationMode;
}
void RxManager_IncDutRxCount()
{
	rxManagerDutDb.dutRxCount++;
}
static void RxManager_SetDutRxCount(K_MSG* pMsg)
{
	dutRxCountParams_t* dutRxCount = (dutRxCountParams_t*) pK_MSG_DATA(pMsg);
	rxManagerDutDb.dutRxCount = dutRxCount->rxCount;
	/*No need to return CFM*/
}
static void RxManager_GetDutRxCount(K_MSG* pMsg)
{
	dutRxCountParams_t* dutRxCount = (dutRxCountParams_t*) pK_MSG_DATA(pMsg);

	dutRxCount->rxCount = rxManagerDutDb.dutRxCount;

	OSAL_SEND_MESSAGE(dutRxCount->retMsg,dutRxCount->retTask,pMsg,pMsg->header.vapId);
}
/**********************************************************************************

RxManager_TaskEntry 


Description:
------------
the entry point of the Rx manager task

Input: 
-----
rxManagerMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void RxManager_TaskEntry(K_MSG *rxManagerMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(rxManagerMessage, afpTaskTable, TASK_RX_MANAGER_START, TASK_RX_MANAGER_END);
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

/**********************************************************************************

RxManager_PostInit 


Description:
------------
post Initialization of RX Manager

Input:
-----
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void RxManager_PostInit(void)
{
	dmaRegistration_t dmaReg;

	/*Register with DMA*/
	dmaReg.priority = DMA_PRIORITY_0;
	dmaReg.eventId = EVENT_ID_DMA_FORWARD;
	RxManagerGlobalParameters.dmaForwardDoneCount = 0;
	RxManagerGlobalParameters.dmaForwardClientId = DmaManager_Register(&dmaReg);
}


/**********************************************************************************

RxManager_Initialize 


Description:
------------
initialize internal structures and registers of the Rx manager and the RDs pools

Input:
-----
rxManagerConfigurationParameters - a pointer to a structure that holds the configuration parameters	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void RxManager_Initialize()
{
	uint32 staId 	= 0;
	uint32 fragNum 	= 0;

	memset(RxManagerStationDatabase, 				0, sizeof(RxManagerStationDatabase));
	memset(RxManagerVapDatabase, 					0, sizeof(RxManagerVapDatabase));
	memset(&RxManagerGlobalParameters, 				0, sizeof(RxManagerGlobalParameters));
	memset(&RxManagerHwQueueManagerRequestParams, 	0, sizeof(HwQueueManagerRequestParams_t));
	
	/* Initialize the indexes in the station database to invalid value (0 is a valid value)*/
    for(staId = 0; staId < HW_NUM_OF_STATIONS; staId++)
    {
		RxManagerStationDatabase[staId].rxManagerDbStaState = RX_MANAGER_DB_STA_CLOSED;
		/* the max fragementation a station can have is all the fragmentation of its VAP */
		for(fragNum = 0; fragNum < MAX_NUM_OF_FRAGMENTATIONS_PER_STA; fragNum++)
		{
			RxManagerStationDatabase[staId].fragementationEntriesIndexes[fragNum] = RX_MANAGER_INVALID_FRAGMENTATION_ENTRY;
		}
    }

	
	/* Initialize the timeout entries of the fragmentation and the class violation */
	rxManagerInitializeTimeoutEntries();	

	/* Initialize the global Hw queue manager request structure */
	RxManagerHwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	RxManagerHwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

	/* Initialize the calendar wheel of the fragmentations */
	CalendarWheel_Initialize(&RxManagerCalendarWheel, RX_MANAGER_RECIPIENT_CALENDAR_WHEEL_SIZE, RxManagerCalendarWheelSlots, (uint32)&RxManagerTimeoutnEntriesDataBase, sizeof(RxManagerTimeoutEntry_t));

	/* Initialize the RD pools of the FW and the host */
	rxManagerInitializeRxDescriptorsPools();

#if defined (CLIENT_MODE_MULTICAST_FILTER_SUPPORT)
	/* Client Mode MCAST filter: Init the Out pointers (indexes) for the MCAST and ERROR lists and clear the ring */
	RxManagerGlobalParameters.clientModeMcastFilterRingIndex = 0;
	RxManagerGlobalParameters.clientModeMcastFilterErrorIndex = 0;
	
	ClientModeMcastFilterClearTheRing(&clientModeMcastFilterRing[0]);
#endif
}

static void RxManager_IgnoreRd(Rd_t * rxDesc)
{
	ILOG0_D("Ignore Rd: length %d", rxDesc->dataLength);
	/* Frame is not valid for some reason - discard Rd, retrun the RD to the liberator which will 
	   put it in the proper free RD pool according to its RD source field */
	RxManagerHwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RxManagerHwQueueManagerRequestParams.pHeadDesc = rxDesc; 
	HwQManager_PushPacketToTail(&RxManagerHwQueueManagerRequestParams);
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
