/***********************************************************************************
 File:			RxPp_Api.h
 Module:			Rx Post processing
 Purpose: 		Owner for all access to the Rx Post processing driver
 Description:		This module is the API to the Rx Post Processor HW driver responsible
 				for the configuration of the RXPP HW
************************************************************************************/
#ifndef RXPP_API_H_
#define RXPP_API_H_
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RxPpRegs.h"
#include "RxDescriptor.h"
#include "ShramRxPpFifo.h"
#include "QueueUtility_Api.h"
#include "ieee80211.h"
#include "Utils_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/

#define RXPP_FRAG_FIFO_CLEAR		 	0x1

#if defined(ENET_INC_ARCH_WAVE600)
#define RX_PP_SKIP_REORDERING_RD_TYPES ((1 << RD_TYPE_NDP) 								| \
										(1 << RD_TYPE_CONTROL)							| \
										(1 << RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1)		| \
										(1 << RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2)		| \
										(1 << RD_TYPE_SNIFFER)							| \
										(1 << RD_TYPE_MULTICAST_DATA)					| \
										(1 << RD_TYPE_BIP)								| \
										(1 << RD_TYPE_MULTICAST_MGMT))
#else
#define RX_PP_SKIP_REORDERING_RD_TYPES ((1 << RD_TYPE_NDP) 								| \
										(1 << RD_TYPE_CONTROL) 							| \
										(1 << RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1) 		| \
										(1 << RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2)	 	| \
										(1 << RD_TYPE_SNIFFER) 							| \
										(1 << RD_TYPE_MULTICAST_DATA)					| \
										(1 << RD_TYPE_MULTICAST_MGMT))
#endif

#if defined(ENET_INC_ARCH_WAVE600)
#define RX_PP_SKIP_SECURITY_RD_TYPES   ((1 << RD_TYPE_SNIFFER)							| \
										(1 << RD_TYPE_MULTICAST_DATA)					| \
										(1 << RD_TYPE_MULTICAST_MGMT))
#endif //ENET_INC_ARCH_WAVE600


#if defined(ENET_INC_ARCH_WAVE600)
#define	RX_PP_SEND_TO_UMAC_ALL_DATA_RD_TYPES  (	(1 << RD_TYPE_MULTICAST_DATA)		 			| \
												(1 << RD_TYPE_UNICAST_QOS_DATA) 				| \
												(1 << RD_TYPE_UNICAST_NON_QOS_DATA_PROTECTED) 	| \
												(1 << RD_TYPE_UNICAST_QOS_DATA_PROTECTED) 		| \
												(1 << RD_TYPE_UNICAST_NON_QOS_DATA)) // Defines the send to UMAC bit per RD type: QOS DATA,DATA,Multicast DATA
#else
#define	RX_PP_SEND_TO_UMAC_ALL_DATA_RD_TYPES  (	(1 << RD_TYPE_MULTICAST_DATA)		 	| \
												(1 << RD_TYPE_UNICAST_QOS_DATA) 		| \
												(1 << RD_TYPE_UNICAST_NON_QOS_DATA)) // Defines the send to UMAC bit per RD type: QOS DATA,DATA,Multicast DATA
#endif


#define RX_PP_MANAGEMENT_TID			(8)

#define RXPP_RAM_OFFSET_FROM_BASE_ADDRESS 	  0x0

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/
#if defined(ENET_INC_ARCH_WAVE600)
typedef struct RxPpStaTidRam
{
	/* word 0 */
	uint32 sn 		  :12;
	uint32 reserved1  :4;
	uint32 windowSize :9;
	uint32 reserved2  :4;
	uint32 snValid    :1;
	uint32 pnValid    :1;
	uint32 fragEn     :1;
	
    /* word 1 */
    uint32 pn03; 
	
	/* word 2 */
    uint16 pn45;
	uint16 reserved4;
	
	/* word 3 */
	uint32 highestSn 		  :12;
	uint32 highestSnValid	  :1;
	uint32 reserved5		  :19;
} RxPpStaTidRam_t;
#if defined(ENET_INC_ARCH_WAVE600D2)
typedef struct RxPpReKeyBitmap_t
{
	/* word 0 */
	uint32 rekeyPn_0_31;
	
    /* word 1 */
	uint32 rekeyPn_32_47  	:16;
	uint32 key 				:2;
	uint32 rekeySet		  	:1;
	uint32 mgmtEn	    	:1;
	uint32 statusBit    	:1;
	uint32 oldKey		   	:2;
	uint32 reserved     	:9;
} RxPpReKeyBitmap_t;
#else //#if !defined(ENET_INC_ARCH_WAVE600)
typedef struct RxPpReKeyBitmap
{
	/* word 0 */
	uint32 rekeyPn_0_31;
	
    /* word 1 */
	uint32 rekeyPn_32_47  	:16;
	uint32 key 				:2;
	uint32 rekeySet		  	:1;
	uint32 mgmtEn	    	:1;
	uint32 statusBit    	:1;
	uint32 reserved     	:11;
} RxPpReKeyBitmap_t;
#endif
#else //#if !defined(ENET_INC_ARCH_WAVE600)
typedef struct RxPpStaTidRam
{
	/* word 0 */
	uint32 sn 		  :12;
	uint32 reserved1  :4;
	uint32 windowSize :7;
	uint32 reserved2  :1;
	uint32 snValid    :1;
	uint32 pnValid    :1;
	uint32 status     :1;
	uint32 inFragment :1;
	uint32 reserved3  :4;
	
    /* word 1 */
    uint32 pn03; 
	
	/* word 2 */
    uint16 pn45;
	uint16 reserved4;
	
	/* word 3 */	
	uint32 reserved5;
} RxPpStaTidRam_t;
#endif //#if defined(ENET_INC_ARCH_WAVE600)

typedef struct RxPpStaManagementRam
{
	RxPpStaTidRam_t managementTidParameters;
} RxPpStaManagementRam_t;

typedef struct RxPpStaUnicastRam
{
	RxPpStaTidRam_t tidsParameters[NUM_OF_TID];
} RxPpStaUnicastRam_t;

typedef struct RxPpUnicastRam
{
	RxPpStaUnicastRam_t UnicastStaArr[HW_NUM_OF_STATIONS];
} RxPpUnicastRam_t;

typedef struct RxPpManagementRam
{
	RxPpStaManagementRam_t ManagementStaArr[HW_NUM_OF_STATIONS];
} RxPpManagementRam_t;

typedef struct RxPpMMulticastRam
{
	RxPpStaManagementRam_t ManagementVapArr[HW_NUM_OF_VAPS];
} RxPpMMulticastRam_t;

#if defined(ENET_INC_ARCH_WAVE600)
typedef struct RxPpReKeyUnicastBitmap
{
	RxPpReKeyBitmap_t RxPpReKeyStaBitmap[HW_NUM_OF_STATIONS];
} RxPpReKeyUnicastBitmap_t;
#endif //#if defined(ENET_INC_ARCH_WAVE600)

#if defined(ENET_INC_ARCH_WAVE600D2)
typedef struct RxPpReKeyMulticastBitmap
{
	RxPpReKeyBitmap_t RxPpReKeyVapBitmap[HW_NUM_OF_VAPS];
} RxPpReKeyMulticastBitmap_t;
#endif //#if defined(ENET_INC_ARCH_WAVE600)

typedef struct RxPpRam
{
	RxPpUnicastRam_t    		rxppUnicastRam;
	RxPpManagementRam_t 		rxppManagementRam;
#if defined(ENET_INC_ARCH_WAVE600D2)
	RxPpMMulticastRam_t 		rxppMulticastRam;
#endif
#if defined(ENET_INC_ARCH_WAVE600)
	RxPpReKeyUnicastBitmap_t	rxppReKeyUnicastBitmap;
#endif //#if defined(ENET_INC_ARCH_WAVE600)
#if defined(ENET_INC_ARCH_WAVE600D2)
	RxPpReKeyMulticastBitmap_t	rxppReKeyMulticastBitmap;
#endif //#if defined(ENET_INC_ARCH_WAVE600)
} RxPpRam_t;

typedef enum RxPpQos
{
	RXPP_NON_QOS_STATION,
	RXPP_QOS_STATION,
} RxPpQos_e;

typedef enum RxPpFragFifoEvent_
{
	RXPP_FRAG_FIFO_END,
	RXPP_FRAG_FIFO_START
} RxPpFragFifoEvent_e;

// For No BA agreement clients flush and SN update are
// relevant only when this bit is set
typedef enum RxPpNonBaClientForce
{
	RXPP_NON_BA_CLIENT_DONT_FORCE = 0,
	RXPP_NON_BA_CLIENT_FORCE,
} RxPpNonBaClientForce_e;

// Determines if flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1) 
// or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
typedef enum RxPpDiscard
{
	RXPP_DONT_DISCARD = 0,
	RXPP_DISCARD,
} RxPpDiscard_e;

typedef enum RxPpUmacBit
{
	RXPP_UMAC,         // forward MPDU to reordering module
	RXPP_HOST_IF,	   // skip reordering
} RxPpUmacBit_e;

// Halt command enum
typedef enum HaltContinue
{
	RXPP_CONTINUE,
	RXPP_HALT,	
} RxPpHaltContinue_e;

typedef enum RxPpTidForwardStatus
{
	RXPP_FORWARD_ALL_TIDS_TO_HOSTIF = 0x0,
	RXPP_FORWARD_ALL_TIDS_TO_UMAC   = TID_BITMAP_ALL_TIDS
} RxPpTidForwardStatus_e;

#if defined(ENET_INC_ARCH_WAVE600)

typedef enum RxPpTidEnableFreg
{
	RXPP_DISABLE_ALL_TIDS_FREG = 0x0,
	RXPP_ENABLE_ALL_TIDS_FREG  = TID_BITMAP_ALL_TIDS
} RxPpTidEnableFreg_e;
#endif

// API struct for returning the status of the different state machines of the RXPP HW
typedef struct RxPpSmStatus
{
	uint8 RdAnalyzeSm;
	uint8 RdDigSm;
	uint8 RdClearSm;
	uint8 SwSnUpdateSm;
	uint8 SwWinUpdateSm;
	uint8 SwFlushSm;
	uint8 StatusUpdateSm;
}RxPpSmStatus_t;

/*REG_RX_PP_SHRAM_FIFO_STATUS 0x486C */
typedef struct RxPpShramFifoStatus
{
	uint16 ShramFifoLinesTaken;
	uint8  reserved0;
	uint8  ShramFifoFull;
	uint8  ShramFifoEmpty;
} RxPpShramFifoStatus_t;

typedef struct RxPpFifoFragLine
{
	StaId  stationId;
	uint8  tid;
	uint16 sn;
	uint8  fragStart;	// Indicates if Fragment start(1) or fragment End(0)
	uint8  valid;		// Indicates if the struct is valid or No Data read
	
} RxPpFifoFragLine_t;

/*---------------------------------------------------------------------------------
/						Public Function Declaration						
/----------------------------------------------------------------------------------*/
void RxPp_Init(void);
void RxPp_ResetStation(StaId staId, uint8 isQosStation, uint8 isXFilterOpen);
void RxPp_OpenTs(StaId staId, uint8 tid, uint16 winSize, uint16 seqNum);
void RxPp_CloseTs(StaId staId, uint8 tid);
void RxPp_UpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce);
void RxPp_FlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e force);
void RxPp_FlushMultiTid(StaId staId, uint16 tidBitmap,  RxPpDiscard_e discard, RxPpNonBaClientForce_e force);
void RxPp_SetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap);
void RxPp_Halt(void);
void RxPp_Resume(void);
void RxPp_Rekey(StaId staId);
void RxPp_ConfigFragmentFifo(void);
void RxPp_FragmentFifoRead(RxPpFifoFragLine_t* fifoFragLine);
bool RxPp_WindowChange(StaId staId, uint8 tid);

#if defined (ENET_INC_ARCH_WAVE600)
void RxPp_SetFragEn(StaId staId, TidBitmap_e tidBitmap, RxPpTidEnableFreg_e fragEnValBitmap);
void RxPp_SetPn(StaId staId, uint8 *pn, uint8 keyId, bool rekeyDisableFlag, bool staModeEnable);
void RxPp_SetNewPn(StaId staId, uint8 *pn, uint8 keyId, bool rekeyDisableFlag, bool staModeEnable);
#else //ENET_INC_ARCH_WAVE500
void RxPp_SetStationPn(StaId staId, uint8 *pn);
#endif 


#endif // RXPP_API_H_


