/***********************************************************************************
 File:		RxPostProcessing.c
 Module:		RxPostProcessing
 Purpose:		
 Description:	FW driver for Rx Post Processing module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RxPp_Api.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "RxDescriptor.h"
#include "RxPpRam.h"
#include "RxPpRegs.h"
#include "loggerAPI.h"
#include "OSAL_Interrupts.h"
#include "Pac_Api.h"
#include "ShramRxPpFifo.h"
#include "RxPp_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_RX_HW_ACCLERATORS
#define LOG_LOCAL_FID 1

#define RXPP_ZERO_SN			  0x0 // set SN to 0 when calling RxPp_ResetStation
#define RXPP_TID_BIT        	  0x1 // create the TID bitmap by shifting this bit by the offset specified as TID parameter
#define RXPP_WIN_SIZE_ONE   	  0x1 // Window size is set to 1 in RxPp_CloseTs() 

#define RXPP_SET_INSTRUCTION      0x1 // value used to set instruction type to the register. Instruction can be win size/sn/flash/status bit

#define RXPP_DROP_LISTS_NUM				8   // size of RxPpDropListConfigArr array

#define RXPP_MAX_TID_NUM				9

#define RX_PP_SW_INSTR_DONE 				(1)
#define RX_PP_ALL_SM_IDLE 					(0)

/* Halt Rx PP Definitions*/
#define RX_PP_SW_HALT   					(1)
#define RX_PP_SW_RESUME 					(0)

#define MAX_HALT_POLLING_DURATION			(200)
#define RX_PP_INSTRUCTION_PROCESS_MAX_TIME	(64)


typedef enum RxPpBackdoorUpdate
{
	RXPP_BACKDOOR_SN_VALID_BIT_ENABLE = 0x1,
	RXPP_BACKDOOR_PN_VALID_BIT_ENABLE = 0x2,
	RXPP_BACKDOOR_IN_FRAG_BIT_ENABLE = 0x8,
	RXPP_BACKDOOR_ALL_BITS_ENABLE = 0xF,
} RxPpBackdoorUpdate_e;

#ifdef ENET_INC_ARCH_WAVE600D2
#define DEBUG_RXPP
#endif


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void rxPpHalt(void);
static void rxPpResume(void);
static void rxPpUpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce);
static void rxPpUpdateWindowSize(StaId staId, uint8 tid, uint16 winSize);
static void rxPpFlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce);
static void rxPpFlushMultiTid(StaId staId, uint16 tidBitmap, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce);
static void rxPpSetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap);
static void rxPpSetFragEn(StaId staId, TidBitmap_e tidBitmap, RxPpTidEnableFreg_e fragEnValBitmap);
#ifndef ENET_INC_ARCH_WAVE600D2
static void rxPpSetStationPn(StaId staId, uint8 *pn);
#endif
static void rxPpWaitInstructionDone(void);
static void rxPpWaitRekeyInstructionDone(void);


/*---------------------------------------------------------------------------------
/						Static variables Declaration									
/----------------------------------------------------------------------------------*/
static RxPpRam_t *RxPpRam;
static uint8 IntialRxPn[UMI_RSN_SEQ_NUM_LEN] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};
static uint32 RxPpFragmentReadIndex;
#ifndef ENET_INC_ARCH_WAVE600D2
static uint16 RxPpLastSn[HW_NUM_OF_STATIONS][NUM_OF_TID];
#endif

#if defined (RX_LIST_DEBUG)
uint8 RxPpForwardRdTypeList[] =
{
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  		//	0- RD_TYPE_UNICAST_QOS_DATA
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST, 		//	1- RD_TYPE_NDP					
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  		//	2- RD_TYPE_MULTICAST_DATA		
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  		//	3- RD_TYPE_UNICAST_NON_QOS_DATA
	HW_Q_MANAGER_RX_READY_LIST_FW, 	        	//	4- RD_TYPE_CONTROL				
	HW_Q_MANAGER_RX_READY_LIST_FW, 	        	//	5- RD_TYPE_NOT_SUPPORTED			
	HW_Q_MANAGER_RX_READY_LIST_FW, 	        	//	6- RD_TYPE_UNICAST_MGMT_TYPE_1	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,      	//	7- RD_TYPE_UNICAST_MGMT_TYPE_2	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,      	//	8- RD_TYPE_MULTICAST_MGMT			
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,   	//	9- RD_TYPE_DROP					
	HW_Q_MANAGER_RX_READY_LIST_FW,    	    	//	10- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,   	//	11- RD_TYPE_LOGGER	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,	    	//	12- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,   	//	13- RD_TYPE_MAGIC_PACKET
	HW_Q_MANAGER_RX_READY_LIST_DEBUG,       	//	14- RD_TYPE_SNIFFER
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,    	//	15- RD_TYPE_RESERVED_0
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	16- RD_TYPE_BIP
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  		// 	17- RD_TYPE_UNICAST_NON_QOS_DATA_PROTECTED
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	 		//  	18- RD_TYPE_UNICAST_QOS_DATA_PROTECTED
	HW_Q_MANAGER_RX_READY_LIST_FW,	 			//  	19- RD_TYPE_UNICAST_MGMT_TYPE_1_PROTECTED
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,			// 	20- RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	21- RD_TYPE_RESERVED21
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED22
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED23
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		//	RD_TYPE_RESERVED24
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED25
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED26
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED27
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED28
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED29
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	RD_TYPE_RESERVED30
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,		// 	31- RD_TYPE_WPA_PT

};
#else
uint8 RxPpForwardRdTypeList[] =
{
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT, 	  //0- RD_TYPE_UNICAST_QOS_DATA
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //1- RD_TYPE_NDP					
	HW_Q_MANAGER_RX_READY_LIST_FW, 	  								  //2- RD_TYPE_MULTICAST_DATA	
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT, 	  //3- RD_TYPE_UNICAST_NON_QOS_DATA
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //4- RD_TYPE_CONTROL				
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //5- RD_TYPE_NOT_SUPPORTED			
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //6- RD_TYPE_UNICAST_MGMT_TYPE_1	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //7- RD_TYPE_UNICAST_MGMT_TYPE_2	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //8- RD_TYPE_MULTICAST_MGMT			
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //9- RD_TYPE_DROP					
	HW_Q_MANAGER_RX_READY_LIST_FW,    	                              //10- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //11- RD_TYPE_LOGGER	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,	                              //12- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //13- RD_TYPE_MAGIC_PACKET
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT,       //14- RD_TYPE_SNIFFER
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //15- RD_TYPE_RESERVED_0
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  //16- RD_TYPE_BIP
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT,		  //17- RD_TYPE_UNICAST_NON_QOS_DATA_PROTECTED
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT,		  //18- RD_TYPE_UNICAST_QOS_DATA_PROTECTED
	HW_Q_MANAGER_RX_READY_LIST_FW,							  		  //19- RD_TYPE_UNICAST_MGMT_TYPE_1_PROTECTED
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,								  //20- RD_TYPE_UNICAST_MGMT_TYPE_2_PROTECTED
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED21
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED22
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED23
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED24
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED25
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED26
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED27
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED28
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED29
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  // RD_TYPE_RESERVED30
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,							  //31- RD_TYPE_WPA_PT
};
#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

rxPpHalt

Description:
------------

	Halt The Rx PP State Machines

Returns:
--------
	void 
	
**********************************************************************************/\
static void rxPpHalt(void)
{

	RegRxPpSwHalt_u swHaltReg;


	RegRxPpSmStatusPart1_u smStatusReg;
	uint32 startTime = 0;
	uint32 currentPollingTime = 0;

	swHaltReg.val = 0;

	swHaltReg.bitFields.swHalt = RX_PP_SW_HALT;
	RegAccess_Write(REG_RX_PP_SW_HALT,swHaltReg.val);
	
	startTime = Pac_TimGetTsfLow();

	do
	{
		currentPollingTime = Pac_TimGetTsfLow();
		ASSERT((currentPollingTime - startTime) <= MAX_HALT_POLLING_DURATION);

		/* Poll Rx PP  until all its SM are idle*/
		RegAccess_Read(REG_RX_PP_SM_STATUS_PART1,&smStatusReg.val);
	}
	while(smStatusReg.val != RX_PP_ALL_SM_IDLE);

}

/**********************************************************************************

rxPpResume

Description:
------------
	Resume The Rx PP State Machines

Returns:
--------
	void 

**********************************************************************************/
static void rxPpResume(void)
{
	RegRxPpSwHalt_u swHaltReg;

	swHaltReg.val = 0;

	swHaltReg.bitFields.swHalt = RX_PP_SW_RESUME;
	RegAccess_Write(REG_RX_PP_SW_HALT,swHaltReg.val);
}

/**********************************************************************************

rxPpUpdateWindowSn

Description:
------------
	Sets the current sequence number of the BA Agreement

	The RXPP module will execute the instruction after completion of the current MPDU processing. 

Input:
-----
	uint16 staId - station index/VAP index in AP-MODE/STA-MODE accordingly
	uint8  tid - tid to set
	uint16 seqNum - sequence number
	RxPpNonBaClientForce_e nonBaForce - Determines if update sequence number is valid for non BA agreement clients
	
Returns:
--------
	void 
	
**********************************************************************************/
static void rxPpUpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce)
{
	RegRxPpSwInstCtl_u	swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;

	// Set update SN instruction bits values
	swInstCtlReg.bitFields.snUpdate = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.newSn = seqNum;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;
#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif
	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;
	
	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpUpdateWindowSize

Description:
------------
	Update The window size
	 	BA Agreement Exist -  window size is set according to Ba sise in agreement
	 	BA Agreement doesn't exist - Window size is set to 1
	 	Configure the Window Size of the specified Station TID to 1.	- 

	The RXPP module will execute the instruction after completion of the current MPDU processing.

	when winsize is set to 1 (i.e. no ba agreemnet) -> rxpp updates SN = SN - 1
	when winsize is set to another value (i.e. to an actual ba agreemnet) -> rxpp updates SN = SN + 1

	this is because the meaning of SN is different between legacy and ba agreemnet modes:
	in legacy SN meaning is the last received SN
	in an open baa SN meaning is the next SN that is expected to be received

Input:
-----
	StaId staId - station Index/VAP in AP-MODE/STA-MODE accordingly
	uint8 tid - TID to set
	uint8 winSize - BA Window size
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpUpdateWindowSize(StaId staId, uint8 tid, uint16 winSize)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;

	// Set win size instruction bits values
	swInstCtlReg.bitFields.winSizeUpdate = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.baWindowSize = winSize;
	// Set Instruction address
#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif
	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();	
}	



/**********************************************************************************

rxPpFlushTid

Description:
------------
	Flushes the RDs of the specified Traffic stream.
	And either discard RDs or forward to Host according to input

Input:
-----
	uint16 staID		  	  	  - station index/VAP index in AP-MODE/STA-MODE accordingly
	uint8 tid				      - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard	      - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force  - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpFlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	// Set flush instruction bits values 	
	swInstCtlReg.bitFields.flush = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.clearDest = discard;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;
	// Set Instruction address
#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif	
	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpFlushMultiTid

Description:
------------
	Flushes the RDs of the specified Traffic streams according to bitmap.
	And either discard RDs or forward to Host according to input

Input:
-----
	uint8         		             - staID	
	uint8 tidBitmap			 Bitmap of TIDs [0-8]. Valid only when mulTid = TRUE
	RxPpDiscard_e discard        - Determines if the flushed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
							 or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpFlushMultiTid(StaId staId, uint16 tidBitmap, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	// Set flush instruction bits values 	
	swInstCtlReg.bitFields.flush = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.clearDest = discard;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;
	swInstCtlReg.bitFields.mulTids = TRUE;
	// Set Instruction address
#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif	
	swInstAddrReg.bitFields.sta = staId;
	swInstAddrReg.bitFields.mulTidsEn = tidBitmap;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpSetForwardStatus

Description:
------------
	set the status bit of the specified Sta/TID - Usually called with all 1's TID Bitmap (TID_BITMAP_ALL_TIDS). 
	The status bit determines if RDs should be forwarded to UMAC or to HostIf. 
	Status bit is reset when station is disconnected but we still get RDs from the station that we would
	like them to be handled by UMAC.
	// NOTE: If set via this API it will override configuration via REG_RX_PP_RD_TYPE0_FORWARD_LIST


Input:
-----
	StaId staId - station Index
	TidBitmap_e tidBitmap - Bitmap of TIDs we want the instruction to modify 
	RxPpTidForwardStatus_e forwardStatusBitmap - Bitmap of forwatd status values: 
							0- RD are forwarded to HostIf  and from there to host(GSWIP/ Driver))					
							1- RDs are forwarded to UMAC

Returns:
--------
	void
	
**********************************************************************************/
void rxPpSetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;
	UNUSED_PARAM(tidBitmap);

    UNUSED_PARAM(tidBitmap);
	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	
	// set the status instruction bit field so that statusBitValue will be used
	swInstCtlReg.bitFields.statusBitUpdate = RXPP_SET_INSTRUCTION;	
	// Status bit determines if the RD will be forwarded to the UMAC or the Host IF. Status bit is configured for all TIDs of a station in one instruction
	// note that tidBitmap is don't care in case of status bit (in WAV600)
	// if set through here it will override REG_RX_PP_RD_TYPE0_FORWARD_LIST
	if (forwardStatusBitmap == RXPP_FORWARD_ALL_TIDS_TO_HOSTIF)
	{
		swInstCtlReg.bitFields.statusBitValue = FALSE;
	}
	else
	{
		swInstCtlReg.bitFields.statusBitValue = TRUE;
	}
#ifdef ENET_INC_ARCH_WAVE600D2
	// in case STA-MODE staId==vapId and indication is given via mcVapAccess = TRUE
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;	
#endif	
	swInstAddrReg.bitFields.sta = staId;
	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	rxPpWaitInstructionDone();
}	

/**********************************************************************************

rxPpSetFragEn


Description:
------------
	It will be use by the TS manager/Rx manager in case of changes in the RX dynamic fragmentation


Input:
-----
	StaId staId - station Index
	TidBitmap_e tidBitmap - TID we want the instruction to modify 
	RxPpTidEnableFreg fragEnValBitmap -true to enable the frag bit and false to disable it.

Returns:
--------
	void
	
**********************************************************************************/
void rxPpSetFragEn(StaId staId, TidBitmap_e tidBitmap, RxPpTidEnableFreg_e fragEnValBitmap)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;

	swInstCtlReg.bitFields.fragEnUpdate = RXPP_SET_INSTRUCTION;	
	swInstCtlReg.bitFields.mulTids = TRUE;
	
	// Set Instruction address
#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif	
	swInstAddrReg.bitFields.sta = staId;
	swInstAddrReg.bitFields.mulTidsEn = tidBitmap;
	swInstAddrReg.bitFields.mulTidsFragEnValue = fragEnValBitmap;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	rxPpWaitInstructionDone();
}

/**********************************************************************************
RxPp_SetPn

Description:
------------
	Call function to RxPp_SetNewPn or rxPpSetStationPn pending if PN should be reset upon init or upon rekey 
	and according to product version since in in products older than WAV600D2 setPN upon init supported only via backdoor to RxPP RAM 
	
Input:
-----
	StaId - staId
	pn -  the address of the new PN
	
Returns:
--------
	void	
**********************************************************************************/
void RxPp_SetPn(StaId staId, uint8 *pn, uint8 keyId, bool rekeyDisableFlag, bool staModeEnable)
{
	if (rekeyDisableFlag == TRUE) //not rekey but set PN because of init
	{
#ifdef ENET_INC_ARCH_WAVE600D2
		RxPp_SetNewPn(staId, pn, keyId, rekeyDisableFlag, staModeEnable);
#else // (!ENET_INC_ARCH_WAVE600D2)
		// in products older than WAV600D2 setPN upon init supported only via backdoor to RxPP RAM 
		rxPpSetStationPn(staId, pn);
#endif
	}
	else //set PN rekey
	{
		RxPp_SetNewPn(staId, pn, keyId, rekeyDisableFlag, staModeEnable);
	}
}

/**********************************************************************************

rxPpSetStationPn

Description:
------------
	Set the new PN for a specific station for all TIDs including management TID

Input:
-----
	StaId      		             - staId	
	uint8 tid 			         - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
#ifndef ENET_INC_ARCH_WAVE600D2

static void rxPpSetStationPn(StaId staId, uint8 *pn)
{
	uint32 tid = 0 ;
	uint32 pn03 = 0;
	uint32 pn45 = 0;
	uint32* pUnicastStaTid0Word2Params;
	uint32* pMgmtStaTidParamsWord2;


	pMgmtStaTidParamsWord2 = (uint32 *)&(RxPpRam->rxppManagementRam.ManagementStaArr[staId].managementTidParameters.pn45);

	DEBUG_ASSERT((((uint32)pMgmtStaTidParamsWord2) & 0x3) == 0);

	pn03 = ((pn[0]) | (pn[1] << 8) | (pn[2] << 16) | (pn[3] << 24)); 
	pn45 = ((pn[4]) | (pn[5] << 8));

	/* Set data PNs */
	for(tid = 0 ; tid < NUM_OF_TID;tid ++)
	{
		RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid].pn03 = pn03;
		pUnicastStaTid0Word2Params = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid].pn45);
		DEBUG_ASSERT((((uint32)pUnicastStaTid0Word2Params) & 0x3) == 0);
		*pUnicastStaTid0Word2Params = pn45;
	}	

	/* Set management PNs */
	RxPpRam->rxppManagementRam.ManagementStaArr[staId].managementTidParameters.pn03 = pn03;
	*pMgmtStaTidParamsWord2 = pn45;

}
#endif
/**********************************************************************************

rxPpWaitInstructionDone

Description:
------------
	1) 	1) Poll Instruction done bit to ensure  instruction has been processed
	2) If instruction process take too long ASSERT
	
Returns:
--------
	void
	
**********************************************************************************/
static void rxPpWaitInstructionDone(void)
{
	TX_INTERRUPT_SAVE_AREA;

	volatile RegRxPpSwInstAddr_u swInstrAddrReg;
	uint32 startTime = 0;
	uint32 currentPollingTime = 0;
	
	startTime = Pac_TimGetTsfLow();
	do
	{
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		RegAccess_Read(REG_RX_PP_SW_INST_ADDR, &swInstrAddrReg.val);	
		
		currentPollingTime = Pac_TimGetTsfLow();
		ASSERT(((currentPollingTime - startTime) <= RX_PP_INSTRUCTION_PROCESS_MAX_TIME) || (swInstrAddrReg.bitFields.swInstDone == RX_PP_SW_INSTR_DONE));

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}while(swInstrAddrReg.bitFields.swInstDone != RX_PP_SW_INSTR_DONE);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
RxPp_ResetStation 

Description:
------------
	Configure the window size and SN when a new station connects. 
	If the station is QOS station then the window size is configured to 1 else(Non QOS station)
	the window size is configured to 0.
Input:
-----
	StaId staId - station ID
	RxPpQos_e qos - Indicates if the new station supports QOS or not.
	isXFilterOpen -  indicates if the X filter is open
Output:
-------
	None
Returns:
--------
	void 

**********************************************************************************/
void RxPp_ResetStation(StaId staId, uint8 isQosStation, uint8 isXFilterOpen)
{
	// Set Win size
	RegRxPpSwInstCtl_u	swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;
	RegRxPpControlBackdoorUpdate_u controlBackdoorUpdateReg;
	volatile RegRxPpControlBackdoorUpdate_u poolReg;
	RxPpTidForwardStatus_e forwardStatus = RXPP_FORWARD_ALL_TIDS_TO_UMAC;
#ifdef ENET_INC_ARCH_WAVE600D2	
	uint8 keyId = 0;
	bool rekeyDisableFlag = TRUE; //init key
	bool staModeEnable = FALSE; //AP mode
#endif	
	uint8 tidIdx = 0;
	
	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	controlBackdoorUpdateReg.val = 0;
	poolReg.val = 0;

	rxPpHalt();

	/* Set all the PNs to invalid value so all packets will be dropped till the key is set */
#ifdef ENET_INC_ARCH_WAVE600D2
	RxPp_SetNewPn(staId, IntialRxPn, keyId, rekeyDisableFlag, staModeEnable);
#else // (!ENET_INC_ARCH_WAVE600D2)
	// in products older than WAV600D2 setPN upon init supported only via backdoor to RxPP RAM 
	rxPpSetStationPn(staId, IntialRxPn);
#endif	
	
	// reset INST_CTL register and write to it - 
	swInstCtlReg.val = 0;
	/* Reset the control register it will indicate no operation once the address register write 
		initiates a go operation on HW and won't influence the backdoor operation */
	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);

#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
	// swInstAddrReg.bitFields.mcVapAccess = TRUE;
#endif	
	swInstAddrReg.bitFields.sta = staId;

	// we reset all the valid bits to indicate the PN,SN,In frag are Not valid
	controlBackdoorUpdateReg.val = 0;
	// We set the bits in Control_BE bits in order to update the bits
	controlBackdoorUpdateReg.bitFields.controlBe = RXPP_BACKDOOR_ALL_BITS_ENABLE;
	 /* Configure the PN to be valid - it will get the value of 0xFFFFFFFF ,0xFFFF */ 
	controlBackdoorUpdateReg.bitFields.controlPnNumValid = TRUE;

	// Set the valid SN and Valid PN through the Backdoor register to all TIDs	 
	for(tidIdx=0 ; tidIdx < RXPP_MAX_TID_NUM ; tidIdx++)
	{
		// this initiates a go with no operation since swInstCtlReg was configured to no operation
		swInstAddrReg.bitFields.tid = tidIdx;
		RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);

		//rxPpWaitInstructionDone();

		// Write backdoor register to initiate a go on setting the valid bits in RXPP RAM
		RegAccess_Write(REG_RX_PP_CONTROL_BACKDOOR_UPDATE, controlBackdoorUpdateReg.val);

		// pool to check if command was completed
		poolReg.val = 0;
		do
		{			
			RegAccess_Read(REG_RX_PP_CONTROL_BACKDOOR_UPDATE, &poolReg.val);
		}
		while (!(poolReg.bitFields.controlUpdateDone));
	}

	// loop over TIDs- set the win size and SN
	for(tidIdx=0 ; tidIdx < RXPP_MAX_TID_NUM ; tidIdx++)
	{		
		// if the Station is Non QOS set window size to 0 else set winsize to 1
		rxPpUpdateWindowSize(staId,tidIdx, isQosStation);		
	}
		
	if(isXFilterOpen)
 	{
		forwardStatus = RXPP_FORWARD_ALL_TIDS_TO_HOSTIF;
	}	

	rxPpSetForwardStatus(staId, TID_BITMAP_ALL_TIDS, forwardStatus);
	rxPpSetFragEn(staId, TID_BITMAP_ALL_TIDS, RXPP_ENABLE_ALL_TIDS_FREG);

	rxPpResume();
}	

/**********************************************************************************
RxPp_OpenTs

Description:
------------	
	Configure the following traffic stream parameters (per StaId TID):
		Window Size - Sets the aggregation window size
		Sequence Number - Sets the sequence number.
	The RXPP module will execute the instruction after completion of the current MPDU processing.

Input:
-----
	StaId staId - station Index
	Tid_e tid - TID to set
	uint8 winSize - BA Window size
	uint16 seqNum - Lowest Sequence num of the window	
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_OpenTs(StaId staId, uint8 tid, uint16 winSize, uint16 ssn)
{
	uint16 sn;
	bool   snValid = FALSE;
	uint32 tidParamsWord0;
	uint32* pTidParamsWord0;

	rxPpHalt();

	pTidParamsWord0 = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid]);

	DEBUG_ASSERT((((uint32)pTidParamsWord0) & 0x3) == 0);

#ifdef ENET_INC_ARCH_WAVE600D2
	// TODO:  STA-Mode should be added here and remove w/a in FW
#endif
	tidParamsWord0 = *pTidParamsWord0;
	
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
	snValid = ((RxPpStaTidRam_t *)&tidParamsWord0)->snValid;

	if (snValid == TRUE)
	{
		DEBUG_ASSERT(RXPP_WIN_SIZE_ONE == ((RxPpStaTidRam_t *)&tidParamsWord0)->windowSize);
		
		if (ssn == sn)
		{
			rxPpUpdateWindowSn(staId, tid, ssn, RXPP_NON_BA_CLIENT_FORCE);
			rxPpUpdateWindowSize(staId, tid, winSize); //sn is valid, hence sn will be set as equal to sn+1 by the rxpp hw
		}
		else
		{
			rxPpUpdateWindowSn(staId, tid, (ssn-1), RXPP_NON_BA_CLIENT_FORCE);
			rxPpUpdateWindowSize(staId, tid, winSize); //sn is valid, hence sn will be set as equal to sn+1 (which will be equal to ssn) by the rxpp hw
		}
	}
	else
	{
		rxPpUpdateWindowSn(staId, tid, ssn, RXPP_NON_BA_CLIENT_FORCE);
		rxPpUpdateWindowSize(staId, tid, winSize); //sn is not valid, hence sn will be set as equal to sn by the rxpp hw
	}	

	/*Copy value to shadow*/
	tidParamsWord0 = *pTidParamsWord0;
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
#ifndef ENET_INC_ARCH_WAVE600D2 	
	RxPpLastSn[staId][tid] = sn;
#endif
	rxPpResume();
}	

/**********************************************************************************
RxPp_CloseTs

Description:
------------	
	Configure the Window Size of the specified Station TID to 1.
	The RXPP module will execute the instruction after completion of the current MPDU processing.

	1)Flush All RD's from requested Q
	2) Update the last RD  SN +1 oin intrenal RAm
	3) Updaet window size in internal Ram

Input:
-----
	StaId staId - station Index
	uint8    tid
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_CloseTs(StaId staId, uint8 tid)
{
	rxPpUpdateWindowSize(staId,tid,RXPP_WIN_SIZE_ONE);
}	


/**********************************************************************************
RxPp_UpdateWindowSn

Description:
------------	
	
Sets the current sequence number of the TS.
The RXPP module will execute the instruction after completion of the current MPDU processing. 

Input:
-----
	StaId staId - station Index
	Tid_e tid - TID to update
	uint16 seqNum - Lowest Sequence num of the window
	RxPpNonBaClientForce_e nonBaForce - Determines if update window is valid for non BA agreement clients
	
Output:
-------
	None

**********************************************************************************/
void RxPp_UpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce)
{
	rxPpUpdateWindowSn(staId,tid,seqNum,nonBaForce);
}	

/**********************************************************************************
RxPp_FlushTid 

Description:
------------
	
	Flashes the RDs of the specified Traffic stream.

	Called when:
		1) TS Manager Remove Station - flush is called with discard & Force (TID_BITMAP_ALL_TIDS).	
		2) TS inactivity internal timeout has expired. TS is not closed but flush is called without discard and without force.
Input:
-----
	uint8         		             - staID	
	uint8 tid				      - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_FlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e force)
{
	rxPpFlushTid(staId,tid,discard,force);
}	

/**********************************************************************************
RxPp_FlushMultiTid 

Description:
------------
	
	Flashes the RDs of the specified Traffic stream.
		Called when: TS Manager Remove Station - flush is called with discard & Force (TID_BITMAP_ALL_TIDS).	
				     TS inactivity internal timeout has expired. TS is not closed but flush is called without discard and without force.
Input:
-----
	uint8         		             - staID
	uint16 tidBitmap 		      - Bitmap of TIDs [0-8]. Valid only when mulTid = TRUE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_FlushMultiTid(StaId staId, uint16 tidBitmap,  RxPpDiscard_e discard, RxPpNonBaClientForce_e force)
{
	rxPpFlushMultiTid(staId,tidBitmap,discard,force);
}	

/**********************************************************************************
RxPp_SetForwardStatus 

Description:
------------
	set the status bit of the specified Sta/TID - Usually called with all 1's TID Bitmap (TID_BITMAP_ALL_TIDS). 
	The status bit determines if RDs should be forwarded to UMAC or to HostIf. 
	Status bit is reset when station is disconnected but we still get RDs from the station that we would
	like them to be handled by UMAC.
Input:
-----
	StaId staId - station Index
	TidBitmap_e tidBitmap - Bitmap of TIDs we want the instruction to modify	
	RxPpTidForwardStatus_e forwardStatusBitmap - Bitmap of forwatd status values: 
							1- forward to UMAC, 0- forward to HostIf.
							Bitmap is usually all 1's or all 0's
	RxPpUmacBit umacBit  = 1 - RDs are forwarded to UMAC	          
		   	      umacBit  = 0 - RDs are forwarded to Host (GSWIP/ Driver)
Output:
-------
	None
Returns:
--------
	void 

**********************************************************************************/
void RxPp_SetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap)
{
	rxPpSetForwardStatus(staId,tidBitmap,forwardStatusBitmap);
}

/**********************************************************************************
RxPp_SetFragEn 

Description:
------------

Output:
-------
	None
Returns:
--------
	void 

**********************************************************************************/

void RxPp_SetFragEn(StaId staId, TidBitmap_e tidBitmap, RxPpTidEnableFreg_e fragEnValBitmap)

{
	rxPpSetFragEn(staId,tidBitmap,fragEnValBitmap);
}



/**********************************************************************************
RxPp_Halt 

Description:
----------	
	Halts the HW from processing MPDUs that are not currently In the MPDU list.
	The HW finishes processing the MPDUs that are in the MPDU IN list and then halts.
Input:
-----
	RxPpHaltContinue_e command - halt the HW or continue its processing.
Output:
-------
	void
Returns:
--------
	void 

**********************************************************************************/
void RxPp_Halt(void)
{
	rxPpHalt();
}	

/**********************************************************************************
RxPp_Resume 

Description:
----------	
	Resumes the HW from processing MPDUs that are not currently In the MPDU list.
	The HW finishes processing the MPDUs that are in the MPDU IN list and then halts.
Input:
-----
	RxPpHaltContinue_e command - halt the HW or continue its processing.
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
void RxPp_Resume(void)
{
	rxPpResume();
}

/**********************************************************************************
RxPp_Rekey 

Description:
----------	
	called during Rekey - RXPP is already halted
	needs to flush all TIDs
Input:
-----
	STA ID
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
void RxPp_Rekey(StaId staId)
{
	rxPpFlushMultiTid(staId, TID_BITMAP_ALL_TIDS, RXPP_DONT_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
}

/**********************************************************************************
RxPp_SetNewPn 

Description:
----------	

Input:
-----

Output:
-------

Returns:
--------
 
**********************************************************************************/
void RxPp_SetNewPn(uint16 staId, uint8 *pn, uint8 keyId, bool rekeyDisableFlag, bool staModeEnable)
{
	RegRxPpSwRekeyInstLsb_u rxPpSwRekeyInstLsb;
	RegRxPpSwRekeyInstMsb_u rxPpSwRekeyInstMsb;
	rxPpSwRekeyInstLsb.val = 0;
	rxPpSwRekeyInstMsb.val = 0;

	// verify that for older prodcuts than WAV600D2 we reach this function only in rekey case
#ifndef ENET_INC_ARCH_WAVE600D2
	ASSERT(rekeyDisableFlag == FALSE);
#endif

	rxPpSwRekeyInstLsb.bitFields.rekeySta = staId;
#ifdef ENET_INC_ARCH_WAVE600D2
	rxPpSwRekeyInstLsb.bitFields.rekeyVapAccess = staModeEnable;
	rxPpSwRekeyInstLsb.bitFields.rekeyDataUpdateOnly = rekeyDisableFlag;
#else
	UNUSED_PARAM(staModeEnable);
#endif	
	rxPpSwRekeyInstLsb.bitFields.rekeyKey = keyId;
	rxPpSwRekeyInstLsb.bitFields.rekeyMgmtTidEn = TRUE;
	rxPpSwRekeyInstLsb.bitFields.rekeyPn015 = ((pn[0]) | (pn[1] << 8));

	rxPpSwRekeyInstMsb.bitFields.rekeyPn1647 = ((pn[2]) | (pn[3] << 8) | (pn[4] << 16) | (pn[5] << 24)); 
	
	// Note: SW_REKEY_INST_MSB must be configured after SW_REKEY_INST_LSB, since writing to SW_REKEY_INST_MSB triggers the HW to perform the Rekey sequence 
	RegAccess_Write(REG_RX_PP_SW_REKEY_INST_LSB, rxPpSwRekeyInstLsb.val);
	RegAccess_Write(REG_RX_PP_SW_REKEY_INST_MSB, rxPpSwRekeyInstMsb.val);

	rxPpWaitRekeyInstructionDone();	
}

/**********************************************************************************

rxPpWaitInstructionDone

Description:
------------
	1) 	1) Poll Instruction done bit to ensure  instruction has been processed
	2) If instruction process take too long ASSERT
	
Returns:
--------
	void
	
**********************************************************************************/
static void rxPpWaitRekeyInstructionDone(void)
{
	TX_INTERRUPT_SAVE_AREA;

	RegRxPpSwRekeyInstLsb_u rxPpSwRekeyInstLsb;
	uint32 startTime = 0;
	uint32 currentPollingTime = 0;
	
	startTime = Pac_TimGetTsfLow();
	do
	{
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		RegAccess_Read(REG_RX_PP_SW_REKEY_INST_LSB,&rxPpSwRekeyInstLsb.val);	
		
		currentPollingTime = Pac_TimGetTsfLow();
		ASSERT(((currentPollingTime - startTime) <= RX_PP_INSTRUCTION_PROCESS_MAX_TIME) || (rxPpSwRekeyInstLsb.bitFields.rekeyDone == RX_PP_SW_INSTR_DONE));

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}while(rxPpSwRekeyInstLsb.bitFields.rekeyDone != RX_PP_SW_INSTR_DONE);
}

/**********************************************************************************
RxPp_WindowChange

	
	
Description:
------------

	Update the new PN for a an active  station for all TIDs including management TID (Used When working in Sta mode )
	the RAM access is proteced by halt/resume to avoid races with HW
	
Input:
-----
	stationId - the station Id
	pn -  the address of the new PN
	
Returns:
--------
	void	
	
**********************************************************************************/
bool RxPp_WindowChange(StaId staId, uint8 tid)
{
#ifdef ENET_INC_ARCH_WAVE600D2
	bool activity = FALSE;
	activity = Pac_RxcRxPpGetAndClearActivity(staId, tid);
#ifdef DEBUG_RXPP	
	ILOG0_DDD("[RxPp_WindowChange], staId = %d, tid = %d, activity = %d", staId, tid, activity);
#endif
	return(activity);
#else
	uint16 sn, lastSn;
	uint32 tidParamsWord0;
	uint32* pTidParamsWord0;

	pTidParamsWord0 = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid]);
	// Verify that address is aligned to 4 Byte (word aligned, the valid values are 0 or 4)
	DEBUG_ASSERT((((uint32)pTidParamsWord0) & FOUR_BYTE_ALIGNED_MASK) == 0);
	tidParamsWord0 = *pTidParamsWord0;
	/*Read SN from RXPP RAM*/
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
	/*save previous last SN*/
	lastSn = RxPpLastSn[staId][tid];
	/*update last SN*/
	RxPpLastSn[staId][tid] = sn;
	/*Check if current SN is equal to Last SN*/
	if (lastSn == sn)
	{
		/*Window has not moved*/
		return (FALSE);
	}
	/*Window has moved*/
	return (TRUE);
#endif
}

/**********************************************************************************
RxPp_ConfigFragmentFifo 

Description:
------------
	Configures the Sharam fragments fifo - Base address and line size
Input:
-----
	void
Output:
-------
	None
Returns:
--------
       None
**********************************************************************************/	
void RxPp_ConfigFragmentFifo(void)
{
	RegRxPpShramFifoCfg_u 			shramFifoCfgReg;
	RegRxPpShramFifoBaseAdd_u 		shramFifoBaseAddrReg;
#ifdef ENET_INC_ARCH_WAVE600B 	
	RegRxPpShramFifoDepthMinusOne_u	rxPpShramFifoDepthMinusOneReg;
	RegRxPpShramFifoClearStrb_u		rxPpShramFifoClearStrbReg;
#endif
#ifdef ENET_INC_ARCH_WAVE600D2 	
	RegRxPpRxPpIntErrorClear_u		rxPpSIntErrorClearReg;
#endif	

	RxPpFragmentReadIndex = 0;	

	// init values to 0
	shramFifoCfgReg.val = 0;
#ifdef ENET_INC_ARCH_WAVE600B 	
	rxPpShramFifoDepthMinusOneReg.val = 0;
	rxPpShramFifoClearStrbReg.val = 0;
#endif

	// FIFO enable
	shramFifoCfgReg.bitFields.shramFifoEn = TRUE;

	// configure HW RXPP FIFO Base address and depth
	// the size is + 1 of what we write here ...
#ifdef ENET_INC_ARCH_WAVE600B 	
	rxPpShramFifoDepthMinusOneReg.bitFields.shramFifoDepthMinusOne = RXPP_FRAGMENT_FIFO_DEPTH - 1;
	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_DEPTH_MINUS_ONE, rxPpShramFifoDepthMinusOneReg.val);

#ifdef ENET_INC_ARCH_WAVE600D2 	
	rxPpShramFifoClearStrbReg.bitFields.shramFifoClearStrb = TRUE;
	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_CLEAR_STRB, rxPpShramFifoClearStrbReg.val);
	rxPpSIntErrorClearReg.bitFields.shramSynFifoClearOverflowStrb = TRUE;	
	rxPpSIntErrorClearReg.bitFields.shramFifoClearFullDropCtrStrb = TRUE;
	rxPpSIntErrorClearReg.bitFields.shramFifoClearDecLessThanZeroStrb = TRUE;
	RegAccess_Write(REG_RX_PP_RX_PP_INT_ERROR_CLEAR, rxPpSIntErrorClearReg.val);	
#else
	rxPpShramFifoClearStrbReg.bitFields.shramFifoClearStrb = TRUE;
	rxPpShramFifoClearStrbReg.bitFields.shramFifoClearFullDropCtrStrb = TRUE;
	rxPpShramFifoClearStrbReg.bitFields.shramFifoClearDecLessThanZeroStrb = TRUE;
	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_CLEAR_STRB, rxPpShramFifoClearStrbReg.val);
#endif

#else
	shramFifoCfgReg.bitFields.fifoLineSize = RXPP_FRAGMENT_FIFO_DEPTH - 1;
#endif

	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_CFG, shramFifoCfgReg.val);

	memset(&RxPpShramFragmentFifo[0], 0, (sizeof(RxPpFifoFragStructure_t)*RXPP_FRAGMENT_FIFO_DEPTH));
	shramFifoBaseAddrReg.bitFields.shramFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&(RxPpShramFragmentFifo[0]));
	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_BASE_ADD, shramFifoBaseAddrReg.val);
}


/**********************************************************************************
RxPp_FragmentFifoRead 

Description:
------------
	Reads a line from fragments fifo
	
Input:
-----
	void
Output:
-------
	None
Returns:
--------
       None
**********************************************************************************/
void RxPp_FragmentFifoRead(RxPpFifoFragLine_t* fifoFragLine)
{
	RxPpFifoFragStructure_t   *fifoLineReg; 

#ifdef ENET_INC_ARCH_WAVE600B
	RegRxPpShramFifoDebug_u				rxPpShramFifoDebugReg;
	RegRxPpShramFifoRdEntriesNum_u 		rxPpShramFifoRdEntriesNumReg;

	// init values
	rxPpShramFifoDebugReg.val 			= 0;
	rxPpShramFifoRdEntriesNumReg.val 	= 0;

	// read debug FIFO
	RegAccess_Read(REG_RX_PP_SHRAM_FIFO_DEBUG, &rxPpShramFifoDebugReg.val);

	if (rxPpShramFifoDebugReg.bitFields.shramFifoNotEmpty) // check that fifo is not empty.
	{
		fifoFragLine->valid = TRUE;
		
		// read the line that the read index points to
		fifoLineReg = &RxPpShramFragmentFifo[RxPpFragmentReadIndex];

		RxPpFragmentReadIndex++;
		if (RxPpFragmentReadIndex >= RXPP_FRAGMENT_FIFO_DEPTH)
		{
			RxPpFragmentReadIndex = 0;
		}

		// fill the returned struct
		fifoFragLine->stationId = fifoLineReg->sta;
		fifoFragLine->tid = fifoLineReg->tid;
		fifoFragLine->sn = fifoLineReg->sn;
		fifoFragLine->fragStart = fifoLineReg->fragOpen;

		// notify HW that FW read a line
		rxPpShramFifoRdEntriesNumReg.bitFields.shramFifoRdEntriesNum = 1;
		RegAccess_Write(REG_RX_PP_SHRAM_FIFO_RD_ENTRIES_NUM, rxPpShramFifoRdEntriesNumReg.val);	
	}
	else
	{
		fifoFragLine->valid = FALSE;
	}

	
#else	
	RegRxPpShramFifoSub_u     rxPpShramFifoSubReg; // the number of entries that were read
	RegRxPpShramFifoStatus_u  rxPpShramFifoStatusReg;

	// check if we had FIFO overflows
	RegAccess_Read(REG_RX_PP_SHRAM_FIFO_STATUS, &rxPpShramFifoStatusReg.val);

	ASSERT(!(rxPpShramFifoStatusReg.bitFields.shramFifoOverflow)); // Need to increase configured fifo size (RxPpManagementObj.fifoLineSize)

	rxPpShramFifoSubReg.val = 0;
	
	if (!(rxPpShramFifoStatusReg.bitFields.shramFifoEmpty)) // check that fifo is not empty.
	{
		fifoFragLine->valid = TRUE;

		// read the line that the read index points to
		fifoLineReg = &RxPpShramFragmentFifo[RxPpFragmentReadIndex];
		
		RxPpFragmentReadIndex++;
		if (RxPpFragmentReadIndex >= RXPP_FRAGMENT_FIFO_DEPTH)
		{
			RxPpFragmentReadIndex = 0;
		}

		// fill the returned struct
		fifoFragLine->stationId= fifoLineReg->sta;
		fifoFragLine->tid = fifoLineReg->tid;
		fifoFragLine->sn = fifoLineReg->sn;
		fifoFragLine->fragStart= fifoLineReg->fragOpen;
	
		// notify HW that FW read a line
		
		rxPpShramFifoSubReg.bitFields.shramFifoSub = 1;
		RegAccess_Write(REG_RX_PP_SHRAM_FIFO_SUB, rxPpShramFifoSubReg.val);		
	}
	else
	{
		fifoFragLine->valid = FALSE;
	}
#endif
}

/**********************************************************************************

RxPp_Init 

Description:
------------
Initialize Rx post processing HW. Configures the following parameters:
	Skip bit - configured per RD type and determines if the RD will be forwarded to the Reordering module or forwarded directly to the status bit check.
	Umac Bit - configured per RD type and determines if RD will be forwarded to status bit check or forwarded to UMAC.
	Dlm Lists - Configuration of the list indexes at the RX DLM lists of the following list types: MPDU in list, Error list, Umac done list, Drop src 0 - 7 lists, forward Rd type 0 -16 lists.

Input:
-----
	void

Output:
-------
	None

Returns:
--------
	void 

**********************************************************************************/
void RxPp_Init(void)
{
	RegRxPpRdTypeSkipCfg_u		rdTypeSkipCfg;
	RegRxPpRdTypeSecSkipCfg_u	rdTypeSecSkipCfg;
	RegRxPpRdTypeUmacCfg_u		rdTypeUmacCfg;
	RegRxPpMpduInList_u			mpduInListReg;
	RegRxPpErrList_u			errListReg;
	RegRxPpUmacDoneList_u		umacDoneListReg;
	RegRxPpLiberatorList_u		liberatorListReg;
	RegRxPpRxCountersCfg_u		rxCountersCfgReg;
	RegRxPpSwRekeyEn_u			swRekeyEnReg;
	int i = 0;
	
	// configure RD types forward list in a for loop.
	uint32* pForwardList = (uint32*)REG_RX_PP_RD_TYPE0_FORWARD_LIST;

	/* Initialize the pointer of the Rx Pp Ram */
	RxPpRam = (RxPpRam_t *)(RX_PP_BASE_ADDR + RXPP_RAM_OFFSET_FROM_BASE_ADDRESS);

	// config the global per RD type reordering skip 
	rdTypeSkipCfg.bitFields.skipBit = RX_PP_SKIP_REORDERING_RD_TYPES;
	RegAccess_Write(REG_RX_PP_RD_TYPE_SKIP_CFG, rdTypeSkipCfg.val);

	// config the global per RD type security skip (replay detection)
	rdTypeSecSkipCfg.bitFields.secSkipBit = RX_PP_SKIP_SECURITY_RD_TYPES;
	RegAccess_Write(REG_RX_PP_RD_TYPE_SEC_SKIP_CFG, rdTypeSecSkipCfg.val);

	// config the global per RD type umac bit
	rdTypeUmacCfg.bitFields.umacBit = RX_PP_SEND_TO_UMAC_ALL_DATA_RD_TYPES;
	RegAccess_Write(REG_RX_PP_RD_TYPE_UMAC_CFG, rdTypeUmacCfg.val);

	// config the DLM lists indexes
	mpduInListReg.val = 0;
	errListReg.val = 0;
	umacDoneListReg.val = 0;
	liberatorListReg.val = 0;

	// config MPDU_IN list index
	mpduInListReg.bitFields.mpduInListIdx = HW_Q_MANAGER_RX_POST_PROCESSING_INPUT_LIST;
	RegAccess_Write(REG_RX_PP_MPDU_IN_LIST, mpduInListReg.val);		

	// config ERROR list index
	errListReg.bitFields.errListIdx = HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS;
	RegAccess_Write(REG_RX_PP_ERR_LIST, errListReg.val);		

	// config the UMAC_DONE list index
	umacDoneListReg.bitFields.umacDoneListIdx = HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW;
	RegAccess_Write(REG_RX_PP_UMAC_DONE_LIST, umacDoneListReg.val);		

	// config the Libarator list index
	liberatorListReg.bitFields.liberatorListIdx = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	RegAccess_Write(REG_RX_PP_LIBERATOR_LIST, liberatorListReg.val);		

	// config the rd type forward lists: FW and Driver
	for (i=0 ; i < RD_TYPE_LAST; i++)
	{	
		RegAccess_Write((uint32)pForwardList, RxPpForwardRdTypeList[i]);
		pForwardList++;
	}

	// Enable Rx PP Counters
	rxCountersCfgReg.val =0;
	rxCountersCfgReg.bitFields.qosRxEn = TRUE;
	rxCountersCfgReg.bitFields.rdCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdDelayedCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdSwDropCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdDuplicateDropCountEn = TRUE;
	rxCountersCfgReg.bitFields.missingSnSwUpdateCountEn = TRUE;
	rxCountersCfgReg.bitFields.mpduUniOrMgmtEn = TRUE;
	rxCountersCfgReg.bitFields.mpduRetryEn = TRUE;
	rxCountersCfgReg.bitFields.amsduCountEn = TRUE;
	rxCountersCfgReg.bitFields.dropMpduEn = TRUE;
	rxCountersCfgReg.bitFields.mpduTypeNotSupportedEn = TRUE;
	rxCountersCfgReg.bitFields.barMpduCountEn = TRUE;
	rxCountersCfgReg.bitFields.replayEn = TRUE;
	rxCountersCfgReg.bitFields.replayMgmtEn = TRUE;
	rxCountersCfgReg.bitFields.tkipCountEn = TRUE;
	rxCountersCfgReg.bitFields.failureCountEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassAmsduBytesStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxcAmpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassMpduInAmpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassOctetMpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassDropStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassSecurMisStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxcCrcErrorStbEn = TRUE;
	rxCountersCfgReg.bitFields.delineatorCrcErrorStbEn = TRUE;
	rxCountersCfgReg.bitFields.bcMcCountEn = TRUE;
	rxCountersCfgReg.bitFields.qosRxDecEn = TRUE;						
	RegAccess_Write(REG_RX_PP_RX_COUNTERS_CFG, rxCountersCfgReg.val); 

	swRekeyEnReg.val = 0;
	swRekeyEnReg.bitFields.rekeyEn = TRUE;
#ifdef ENET_INC_ARCH_WAVE600D2
	swRekeyEnReg.bitFields.keyIdCheckEn = TRUE;
#endif
	RegAccess_Write(REG_RX_PP_SW_REKEY_EN, swRekeyEnReg.val); 	

	// Reset the RXPP RAM 
	memset32(RxPpRam, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxPpRam_t)));
#ifndef ENET_INC_ARCH_WAVE600D2
	memset(&RxPpLastSn, 0, sizeof(RxPpLastSn));
#endif
	RxPp_ConfigFragmentFifo();

	rxPpResume();
}

