/***********************************************************************************
 File:		RxPostProcessing.c
 Module:		RxPostProcessing
 Purpose:		
 Description:	FW driver for Rx Post Processing module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "RxPp_Api.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "RxDescriptor.h"
#include "RxPpRam.h"
#include "RxPpRegs.h"
#include "loggerAPI.h"
#include "OSAL_Interrupts.h"
#include "Pac_Api.h"
#include "ShramRxPpFifo.h"
#include "RxPp_Api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_RX_HW_ACCLERATORS
#define LOG_LOCAL_FID 0

#define RXPP_ZERO_SN			  0x0 // set SN to 0 when calling RxPp_ResetStation
#define RXPP_TID_BIT        	  0x1 // create the TID bitmap by shifting this bit by the offset specified as TID parameter
#define RXPP_WIN_SIZE_ONE   	  0x1 // Window size is set to 1 in RxPp_CloseTs() 

#define RXPP_SET_INSTRUCTION      0x1 // value used to set instruction type to the register. Instruction can be win size/sn/flash/status bit

#define RXPP_DROP_LISTS_NUM				8   // size of RxPpDropListConfigArr array

#define RXPP_MAX_TID_NUM				9

#define RX_PP_SW_INSTR_DONE 				(1)
#define RX_PP_ALL_SM_IDLE 					(0)

/* Halt Rx PP Definitions*/
#define RX_PP_SW_HALT   					(1)
#define RX_PP_SW_RESUME 					(0)

#define MAX_HALT_POLLING_DURATION			(200)
#define RX_PP_INSTRUCTION_PROCESS_MAX_TIME	(64)


typedef enum RxPpBackdoorUpdate
{
	RXPP_BACKDOOR_SN_VALID_BIT_ENABLE = 0x1,
	RXPP_BACKDOOR_PN_VALID_BIT_ENABLE = 0x2,
	RXPP_BACKDOOR_IN_FRAG_BIT_ENABLE = 0x8,
	RXPP_BACKDOOR_ALL_BITS_ENABLE = 0xF,
} RxPpBackdoorUpdate_e;



/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef union
{
	uint32 val;
	struct
	{
		uint32 dlmListIdx:5; // MPDU list index number at the RX lists DLM
		uint32 reserved0:27;
	} bitFields;
} RegRxPpConfigList_u;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void rxPpHalt(void);
static void rxPpResume(void);
static void rxPpUpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce);
static void rxPpUpdateWindowSize(StaId staId, uint8 tid, uint16 winSize);
static void rxPpFlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce);
static void rxPpFlushMultiTid(StaId staId, uint16 tidBitmap, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce);
static void rxPpSetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap);
static void rxPpSetStationPn(StaId staId, uint8 *pn);
static void rxPpWaitInstructionDone(void);

/*---------------------------------------------------------------------------------
/						Static variables Declaration									
/----------------------------------------------------------------------------------*/
static RxPpRam_t *RxPpRam;
static uint8 IntialRxPn[UMI_RSN_SEQ_NUM_LEN] = {0xFF,0xFF,0xFF,0xFF,0xFF,0xFF};
static uint32 RxPpFragmentReadIndex;
static uint16 RxPpLastSn[HW_NUM_OF_STATIONS][NUM_OF_TID];

#if defined (RX_LIST_DEBUG)
uint8 RxPpForwardRdTypeList[] =
{
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  							  //	0- RD_TYPE_UNICAST_QOS_DATA
	HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP, 	                          //	1- RD_TYPE_NDP					
	HW_Q_MANAGER_RX_READY_LIST_FW, 	  							  	  //	2- RD_TYPE_MULTICAST_DATA		
	HW_Q_MANAGER_RX_READY_LIST_DEBUG, 	  							  //	3- RD_TYPE_UNICAST_NON_QOS_DATA
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	4- RD_TYPE_CONTROL				
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	5- RD_TYPE_NOT_SUPPORTED			
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	6- RD_TYPE_UNICAST_MGMT_TYPE_1	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //	7- RD_TYPE_UNICAST_MGMT_TYPE_2	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //	8- RD_TYPE_MULTICAST_MGMT			
	HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP,                            //	9- RD_TYPE_DROP					
	HW_Q_MANAGER_RX_READY_LIST_FW,    	                              //	10- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1
	HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP,                            //	11- RD_TYPE_LOGGER	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,	                              //	12- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2
	HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP,                            //	13- RD_TYPE_MAGIC_PACKET
	HW_Q_MANAGER_RX_READY_LIST_DEBUG,       						  //	14- RD_TYPE_SNIFFER
	HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP                             //	15- RD_TYPE_RESERVED_0
};
#else
uint8 RxPpForwardRdTypeList[] =
{
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT, 	  //	0- RD_TYPE_UNICAST_QOS_DATA
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //	1- RD_TYPE_NDP					
	HW_Q_MANAGER_RX_READY_LIST_FW, 	  								  //	2- RD_TYPE_MULTICAST_DATA		
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT, 	  //	3- RD_TYPE_UNICAST_NON_QOS_DATA
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	4- RD_TYPE_CONTROL				
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	5- RD_TYPE_NOT_SUPPORTED			
	HW_Q_MANAGER_RX_READY_LIST_FW, 	                                  //	6- RD_TYPE_UNICAST_MGMT_TYPE_1	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //	7- RD_TYPE_UNICAST_MGMT_TYPE_2	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,                                //	8- RD_TYPE_MULTICAST_MGMT			
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //	9- RD_TYPE_DROP					
	HW_Q_MANAGER_RX_READY_LIST_FW,    	                              //	10- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_1
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //	11- RD_TYPE_LOGGER	
	HW_Q_MANAGER_RX_READY_LIST_DRIVER,	                              //	12- RD_TYPE_NON_ASSOCIATED_MGMT_TYPE_2
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST,                             //	13- RD_TYPE_MAGIC_PACKET
	HW_Q_MANAGER_RX_DONE_LIST_HOST_INTERFACE_ACCELERATOR_INPUT,       //	14- RD_TYPE_SNIFFER
	HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST                              //	15- RD_TYPE_RESERVED_0
};
#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

rxPpHalt

Description:
------------

	Halt The Rx PP State Machines

Returns:
--------
	void 
	
**********************************************************************************/\
static void rxPpHalt(void)
{
	RegRxPpSwHalt_u swHaltReg;
	RegRxPpSmStatus_u smStatusReg;
	uint32 startTime = 0;
	uint32 currentPollingTime = 0;

	swHaltReg.val = 0;

	swHaltReg.bitFields.swHalt = RX_PP_SW_HALT;
	RegAccess_Write(REG_RX_PP_SW_HALT,swHaltReg.val);
	
	startTime = Pac_TimGetTsfLow();

	do
	{
		currentPollingTime = Pac_TimGetTsfLow();
		ASSERT((currentPollingTime - startTime) <= MAX_HALT_POLLING_DURATION);

		/* Poll Rx PP  until all its SM are idle*/
		RegAccess_Read(REG_RX_PP_SM_STATUS,&smStatusReg.val);
	}
	while(smStatusReg.val != RX_PP_ALL_SM_IDLE);
}

/**********************************************************************************

rxPpResume

Description:
------------
	Resume The Rx PP State Machines

Returns:
--------
	void 

**********************************************************************************/
static void rxPpResume(void)
{
	RegRxPpSwHalt_u swHaltReg;

	swHaltReg.val = 0;

	swHaltReg.bitFields.swHalt = RX_PP_SW_RESUME;
	RegAccess_Write(REG_RX_PP_SW_HALT,swHaltReg.val);
}

/**********************************************************************************

rxPpUpdateWindowSn

Description:
------------
	Sets the current sequence number of the BA Agreement

	The RXPP module will execute the instruction after completion of the current MPDU processing. 

Input:
-----
	stationId - the station Id
	pn -  the address of the new PN
	
Returns:
--------
	void 
	
**********************************************************************************/
static void rxPpUpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce)
{
	RegRxPpSwInstCtl_u	swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;

	// Set update SN instruction bits values
	swInstCtlReg.bitFields.snUpdate = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.newSn = seqNum;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;

	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;
	
	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpUpdateWindowSize

Description:
------------
	Update The window size
	 	BA Agreement Exist -  window size is set according to Ba sise in agreement
	 	BA Agreement doesn't exist - Window size is set to 1
	 	Configure the Window Size of the specified Station TID to 1.	- 

	The RXPP module will execute the instruction after completion of the current MPDU processing. 

Input:
-----
	StaId staId - station Index
	Tid_e tid - TID to set
	uint8 winSize - BA Window size
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpUpdateWindowSize(StaId staId, uint8 tid, uint16 winSize)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;

	// Set win size instruction bits values
	swInstCtlReg.bitFields.winSizeUpdate = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.baWindowSize = winSize;
	// Set Instruction address
	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();	
}	



/**********************************************************************************

rxPpFlushTid

Description:
------------
	Flushes the RDs of the specified Traffic stream.
	And either discard RDs or forward to Host according to input

Input:
-----
	uint8         		             - staID	
	uint8 tid				      - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpFlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce)
{
	
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	// Set flush instruction bits values 	
	swInstCtlReg.bitFields.flush = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.clearDest = discard;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;
	// Set Instruction address
	swInstAddrReg.bitFields.tid = tid;
	swInstAddrReg.bitFields.sta = staId;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpFlushMultiTid

Description:
------------
	Flushes the RDs of the specified Traffic streams according to bitmap.
	And either discard RDs or forward to Host according to input

Input:
-----
	uint8         		             - staID	
	uint8 tidBitmap			 Bitmap of TIDs [0-8]. Valid only when mulTid = TRUE
	RxPpDiscard_e discard        - Determines if the flushed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
							 or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpFlushMultiTid(StaId staId, uint16 tidBitmap, RxPpDiscard_e discard, RxPpNonBaClientForce_e nonBaForce)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	// Set flush instruction bits values 	
	swInstCtlReg.bitFields.flush = RXPP_SET_INSTRUCTION;
	swInstCtlReg.bitFields.clearDest = discard;
	swInstCtlReg.bitFields.noBaForce = nonBaForce;
	swInstCtlReg.bitFields.mulTids = TRUE;
	// Set Instruction address
	swInstAddrReg.bitFields.sta = staId;
	swInstAddrReg.bitFields.mulTidsEn = tidBitmap;

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	
	rxPpWaitInstructionDone();
}

/**********************************************************************************

rxPpSetForwardStatus

Description:
------------
	set the status bit of the specified Sta/TID - Usually called with all 1's TID Bitmap (TID_BITMAP_ALL_TIDS). 
	The status bit determines if RDs should be forwarded to UMAC or to HostIf. 
	Status bit is reset when station is disconnected but we still get RDs from the station that we would
	like them to be handled by UMAC.

	 --Becasue Multi TID command doesn't work as expected the status settings is done for each TID seperatly

Input:
-----
	StaId staId - station Index
	TidBitmap_e tidBitmap - Bitmap of TIDs we want the instruction to modify 
	RxPpTidForwardStatus_e forwardStatusBitmap - Bitmap of forwatd status values: 
							0- RD are forwarded to HostIf  and from there to host(GSWIP/ Driver))					
							1- RDs are forwarded to UMAC

Returns:
--------
	void
	
**********************************************************************************/
void rxPpSetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap)
{
	RegRxPpSwInstCtl_u  swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;

	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	
	// set the status instruction bit field. 
	// Multi TID is set to FALSE becasue an HW bug  altough ususally all TIDs are set to same value
	swInstCtlReg.bitFields.statusBitUpdate = RXPP_SET_INSTRUCTION;	
	swInstCtlReg.bitFields.mulTids = TRUE; 
	// Set Instruction address
	swInstAddrReg.bitFields.sta = staId;
	swInstAddrReg.bitFields.mulTidsEn = tidBitmap; // MulTidsEn: Bitmap indicating which TIDs are modified
	swInstAddrReg.bitFields.mulTidsStatusBitValue = forwardStatusBitmap; // MulTidsStatusBitValue -Bitmap indicating the status value of the TIDs that are modified

	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);
	RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);
	rxPpWaitInstructionDone();
}	

/**********************************************************************************

rxPpSetStationPn

Description:
------------
	Set the new PN for a specific station for all TIDs including management TID

Input:
-----
	uint8         		             - staID	
	uint8 tid				      - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
static void rxPpSetStationPn(StaId staId, uint8 *pn)
{
	uint32 tid = 0 ;
	uint32 pn03 = 0;
	uint32 pn45 = 0;
	uint32* pUnicastStaTid0Word2Params;
	uint32* pMgmtStaTidParamsWord2;


	pMgmtStaTidParamsWord2 = (uint32 *)&(RxPpRam->rxppManagementRam.ManagementStaArr[staId].managementTidParameters.pn45);

	DEBUG_ASSERT((((uint32)pMgmtStaTidParamsWord2) & 0x3) == 0);

	pn03 = ((pn[0]) | (pn[1] << 8) | (pn[2] << 16) | (pn[3] << 24)); 
	pn45 = ((pn[4]) | (pn[5] << 8));

	/* Set data PNs */
	for(tid = 0 ; tid < NUM_OF_TID;tid ++)
	{
		RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid].pn03 = pn03;
		pUnicastStaTid0Word2Params = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid].pn45);
		DEBUG_ASSERT((((uint32)pUnicastStaTid0Word2Params) & 0x3) == 0);
		*pUnicastStaTid0Word2Params = pn45;
	}	

	/* Set management PNs */
	RxPpRam->rxppManagementRam.ManagementStaArr[staId].managementTidParameters.pn03 = pn03;
	*pMgmtStaTidParamsWord2 = pn45;

}

/**********************************************************************************

rxPpWaitInstructionDone

Description:
------------
	1) 	1) Poll Instruction done bit to ensure  instruction has been processed
	2) If instruction process take too long ASSERT
	
Returns:
--------
	void
	
**********************************************************************************/
static void rxPpWaitInstructionDone(void)
{
	TX_INTERRUPT_SAVE_AREA;

	RegRxPpSwInstAddr_u swInstrAddrReg;
	uint32 startTime = 0;
	uint32 currentPollingTime = 0;
	
	startTime = Pac_TimGetTsfLow();
	do
	{
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		RegAccess_Read(REG_RX_PP_SW_INST_ADDR,&swInstrAddrReg.val);	
		
		currentPollingTime = Pac_TimGetTsfLow();
		ASSERT(((currentPollingTime - startTime) <= RX_PP_INSTRUCTION_PROCESS_MAX_TIME) || (swInstrAddrReg.bitFields.swInstDone == RX_PP_SW_INSTR_DONE));

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}while(swInstrAddrReg.bitFields.swInstDone != RX_PP_SW_INSTR_DONE);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************
RxPp_ResetStation 

Description:
------------
	Configure the window size and SN when a new station connects. 
	If the station is QOS station then the window size is configured to 1 else(Non QOS station)
	the window size is configured to 0.
Input:
-----
	StaId staId - station ID
	RxPpQos_e qos - Indicates if the new station supports QOS or not.
	isXFilterOpen -  indicates if the X filter is open
Output:
-------
	None
Returns:
--------
	void 

**********************************************************************************/
void RxPp_ResetStation(StaId staId, uint8 isQosStation, uint8 isXFilterOpen)
{
	// Set Win size
	RegRxPpSwInstCtl_u	swInstCtlReg;
	RegRxPpSwInstAddr_u swInstAddrReg;
	RegRxPpControlBackdoorUpdate_u controlBackdoorUpdateReg;
	RegRxPpControlBackdoorUpdate_u poolReg;
	RxPpTidForwardStatus_e forwardStatus = RXPP_FORWARD_ALL_TIDS_TO_UMAC;
	uint8 tidIdx = 0;
	
	swInstCtlReg.val = 0;
	swInstAddrReg.val = 0;
	controlBackdoorUpdateReg.val = 0;
	poolReg.val = 0;

	rxPpHalt();

	/* Set all the PNs to invalid value so all packets will be dropped till the key is set */
	rxPpSetStationPn(staId,IntialRxPn);
	
	// reset INST_CTL register and write to it - 
	swInstCtlReg.val = 0;
	/* Reset the control register it will indicate no operation once the address register write 
		initiates a go operation on HW and won't influence the backdoor operation */
	RegAccess_Write(REG_RX_PP_SW_INST_CTL, swInstCtlReg.val);

	swInstAddrReg.bitFields.sta = staId;

	// we reset all the valid bits to indicate the PN,SN,In frag are Not valid
	controlBackdoorUpdateReg.val = 0;
	// We set the bits in Control_BE bits in order to update the bits
	controlBackdoorUpdateReg.bitFields.controlBe = RXPP_BACKDOOR_ALL_BITS_ENABLE;
	 /* Configure the PN to be valid - it will get the value of 0xFFFFFFFF ,0xFFFF */ 
	controlBackdoorUpdateReg.bitFields.pnNumValid = TRUE;

	// Set the valid SN and Valid PN through the Backdoor register to all TIDs	 
	for(tidIdx=0 ; tidIdx < RXPP_MAX_TID_NUM ; tidIdx++)
	{
		// this initiates a go with no operation since swInstCtlReg was configured to no operation
		swInstAddrReg.bitFields.tid = tidIdx;
		RegAccess_Write(REG_RX_PP_SW_INST_ADDR, swInstAddrReg.val);

		//rxPpWaitInstructionDone();

		// Write backdoor register to initiate a go on setting the valid bits in RXPP RAM
		RegAccess_Write(REG_RX_PP_CONTROL_BACKDOOR_UPDATE, controlBackdoorUpdateReg.val);

		// pool to check if command was completed
		poolReg.val = 0;
		do
		{			
			RegAccess_Read(REG_RX_PP_CONTROL_BACKDOOR_UPDATE, &poolReg.val);
		}
		while (!(poolReg.bitFields.controlUpdateDone));
	}

	// loop over TIDs- set the win size and SN
	for(tidIdx=0 ; tidIdx < RXPP_MAX_TID_NUM ; tidIdx++)
	{		
		// if the Station is Non QOS set window size to 0 else set winsize to 1
		rxPpUpdateWindowSize(staId,tidIdx, isQosStation);		
	}
		
	if(isXFilterOpen)
 	{
		forwardStatus = RXPP_FORWARD_ALL_TIDS_TO_HOSTIF;
	}	

	rxPpSetForwardStatus(staId, TID_BITMAP_ALL_TIDS, forwardStatus);

	rxPpResume();
}	

/**********************************************************************************
RxPp_OpenTs

Description:
------------	
	Configure the following traffic stream parameters (per StaId TID):
		Window Size - Sets the aggregation window size
		Sequence Number - Sets the sequence number.
	The RXPP module will execute the instruction after completion of the current MPDU processing.

Input:
-----
	StaId staId - station Index
	Tid_e tid - TID to set
	uint8 winSize - BA Window size
	uint16 seqNum - Lowest Sequence num of the window	
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_OpenTs(StaId staId, uint8 tid, uint16 winSize, uint16 seqNum)
{
	uint16 sn;
	bool   snValid = FALSE;
	uint16 seqNumPlusOne;
	uint32 tidParamsWord0;
	uint32* pTidParamsWord0;

	rxPpHalt();

	pTidParamsWord0 = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid]);

	DEBUG_ASSERT((((uint32)pTidParamsWord0) & 0x3) == 0);

	tidParamsWord0 = *pTidParamsWord0;
	
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
	snValid = ((RxPpStaTidRam_t *)&tidParamsWord0)->snValid;

	if (snValid == TRUE)
	{
		/* Check if Veriwave has an issue of ADDBA.SSN = 0 */
		if (seqNum != 0)
		{
			// increment the sn to overcome STA sending ADDBA with SSN already received  (JIRA 2028)
			seqNumPlusOne = sn + 1;
			
			// use rxPpUpdateWindowSn to update the sn to ssn+1
			rxPpUpdateWindowSn(staId, tid, seqNumPlusOne, RXPP_NON_BA_CLIENT_FORCE);			

			rxPpUpdateWindowSize(staId,tid,winSize);
			rxPpUpdateWindowSn(staId, tid, seqNum, RXPP_NON_BA_CLIENT_FORCE);
		}
		else
		{
			rxPpUpdateWindowSn(staId, tid,seqNum,RXPP_NON_BA_CLIENT_FORCE);
			rxPpUpdateWindowSize(staId,tid,winSize);		
		}
	}
	else
	{		
		rxPpUpdateWindowSize(staId,tid,winSize);		
		rxPpUpdateWindowSn(staId, tid,seqNum,RXPP_NON_BA_CLIENT_FORCE);
	}	

	/*Copy value to shadow*/
	tidParamsWord0 = *pTidParamsWord0;
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
	RxPpLastSn[staId][tid] = sn;

	rxPpResume();
}	

/**********************************************************************************
RxPp_CloseTs

Description:
------------	
	Configure the Window Size of the specified Station TID to 1.
	The RXPP module will execute the instruction after completion of the current MPDU processing.

	1)Flush All RD's from requested Q
	2) Update the last RD  SN +1 oin intrenal RAm
	3) Updaet window size in internal Ram

Input:
-----
	StaId staId - station Index
	Tid_e    tid
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_CloseTs(StaId staId, uint8 tid)
{
	rxPpUpdateWindowSize(staId,tid,RXPP_WIN_SIZE_ONE);
}	


/**********************************************************************************
RxPp_UpdateWindowSn

Description:
------------	
	
Sets the current sequence number of the TS.
The RXPP module will execute the instruction after completion of the current MPDU processing. 

Input:
-----
	StaId staId - station Index
	Tid_e tid - TID to update
	uint16 seqNum - Lowest Sequence num of the window
	RxPpNonBaClientForce_e nonBaForce - Determines if update window is valid for non BA agreement clients
	
Output:
-------
	None

**********************************************************************************/
void RxPp_UpdateWindowSn(StaId staId, uint8 tid, uint16 seqNum, RxPpNonBaClientForce_e nonBaForce)
{
	rxPpUpdateWindowSn(staId,tid,seqNum,nonBaForce);
}	

/**********************************************************************************
RxPp_FlushTid 

Description:
------------
	
	Flashes the RDs of the specified Traffic stream.

	Called when:
		1) TS Manager Remove Station - flush is called with discard & Force (TID_BITMAP_ALL_TIDS).	
		2) TS inactivity internal timeout has expired. TS is not closed but flush is called without discard and without force.
Input:
-----
	uint8         		             - staID	
	uint8 tid				      - single TID in case mulTid is FALSE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_FlushTid(StaId staId, uint8 tid, RxPpDiscard_e discard, RxPpNonBaClientForce_e force)
{
	rxPpFlushTid(staId,tid,discard,force);
}	

/**********************************************************************************
RxPp_FlushMultiTid 

Description:
------------
	
	Flashes the RDs of the specified Traffic stream.
		Called when: TS Manager Remove Station - flush is called with discard & Force (TID_BITMAP_ALL_TIDS).	
				     TS inactivity internal timeout has expired. TS is not closed but flush is called without discard and without force.
Input:
-----
	uint8         		             - staID
	uint16 tidBitmap 		      - Bitmap of TIDs [0-8]. Valid only when mulTid = TRUE		
	RxPpDiscard_e discard        - Determines if the flashed RDs will be forwarded to the Drop lists according to the RD source (discard = 1)
					                or Forwarded to UMAC/GSWIP/HostIf lists (discard = 0).
	RxPpNonBaClientForce_e force - Determines if flash is valid for non BA agreement clients
	
Returns:
--------
	void 

**********************************************************************************/
void RxPp_FlushMultiTid(StaId staId, uint16 tidBitmap,  RxPpDiscard_e discard, RxPpNonBaClientForce_e force)
{
	rxPpFlushMultiTid(staId,tidBitmap,discard,force);
}	

/**********************************************************************************
RxPp_SetForwardStatus 

Description:
------------
	set the status bit of the specified Sta/TID - Usually called with all 1's TID Bitmap (TID_BITMAP_ALL_TIDS). 
	The status bit determines if RDs should be forwarded to UMAC or to HostIf. 
	Status bit is reset when station is disconnected but we still get RDs from the station that we would
	like them to be handled by UMAC.
Input:
-----
	StaId staId - station Index
	TidBitmap_e tidBitmap - Bitmap of TIDs we want the instruction to modify	
	RxPpTidForwardStatus_e forwardStatusBitmap - Bitmap of forwatd status values: 
							1- forward to UMAC, 0- forward to HostIf.
							Bitmap is usually all 1's or all 0's
	RxPpUmacBit umacBit  = 1 - RDs are forwarded to UMAC	          
		   	      umacBit  = 0 - RDs are forwarded to Host (GSWIP/ Driver)
Output:
-------
	None
Returns:
--------
	void 

**********************************************************************************/
void RxPp_SetForwardStatus(StaId staId, TidBitmap_e tidBitmap, RxPpTidForwardStatus_e forwardStatusBitmap)
{
	rxPpSetForwardStatus(staId,tidBitmap,forwardStatusBitmap);
}	


/**********************************************************************************
RxPp_Halt 

Description:
----------	
	Halts the HW from processing MPDUs that are not currently In the MPDU list.
	The HW finishes processing the MPDUs that are in the MPDU IN list and then halts.
Input:
-----
	RxPpHaltContinue_e command - halt the HW or continue its processing.
Output:
-------
	void
Returns:
--------
	void 

**********************************************************************************/
void RxPp_Halt(void)
{
	rxPpHalt();
}	

/**********************************************************************************
RxPp_Resume 

Description:
----------	
	Resumes the HW from processing MPDUs that are not currently In the MPDU list.
	The HW finishes processing the MPDUs that are in the MPDU IN list and then halts.
Input:
-----
	RxPpHaltContinue_e command - halt the HW or continue its processing.
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
void RxPp_Resume(void)
{
	rxPpResume();
}

/**********************************************************************************
RxPp_Rekey 

Description:
----------	
	called during Rekey - RXPP is already halted
	needs to flush all TIDs
Input:
-----
	STA ID
Output:
-------
	None
Returns:
--------
	None 
**********************************************************************************/
void RxPp_Rekey(uint8 staId)
{
	rxPpFlushMultiTid(staId, TID_BITMAP_ALL_TIDS, RXPP_DONT_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
}

/**********************************************************************************
RxPp_SetStationPn

Description:
------------
	Set the new PN for a specific station for all TIDs including management TID
	
Input:
-----
	stationId - the station Id
	pn -  the address of the new PN
	
Returns:
--------
	void	
**********************************************************************************/
void RxPp_SetStationPn(StaId staId, uint8 *pn)
{
	rxPpSetStationPn(staId, pn);
}

/**********************************************************************************
RxPp_WindowChange


Description:
------------
	Checks if window has changed
	
Input:
-----
	stationId - the station Id
	tid - TID
	
Returns:
--------
	void	
	
**********************************************************************************/
bool RxPp_WindowChange(StaId staId, uint8 tid)
{
	uint16 sn, lastSn;
	uint32 tidParamsWord0;
	uint32* pTidParamsWord0;

	pTidParamsWord0 = (uint32 *)&(RxPpRam->rxppUnicastRam.UnicastStaArr[staId].tidsParameters[tid]);
	DEBUG_ASSERT((((uint32)pTidParamsWord0) & 0x3) == 0);
	tidParamsWord0 = *pTidParamsWord0;
	/*Read SN from RXPP RAM*/
	sn = ((RxPpStaTidRam_t *)&tidParamsWord0)->sn;
	/*save previous last SN*/
	lastSn = RxPpLastSn[staId][tid];
	/*update last SN*/
	RxPpLastSn[staId][tid] = sn;
	/*Check if current SN is equal to Last SN*/
	if (lastSn == sn)
	{
		/*Window has not moved*/
		return (FALSE);
	}
	/*Window has moved*/
	return (TRUE);
}

/**********************************************************************************
rxPpConfigFragmentFifo Fifo

Description:
------------
	Configures the Sharam fragments fifo - Base address and line size
Input:
-----
	void
Output:
-------
	None
Returns:
--------
       None
**********************************************************************************/
void RxPp_ConfigFragmentFifo(void)
{
	RegRxPpShramFifoCfg_u shramFifoCfgReg;

	/*Clear fragmentation FIFO*/
	memset32(RxPpShramFragmentFifo, 0, RXPP_FRAGMENT_FIFO_DEPTH);
	RxPpFragmentReadIndex = 0;	
	
	shramFifoCfgReg.val = 0;
	// configure HW RXPP FIFO Base address and depth
	// the size is + 1 of what we write here ...
	shramFifoCfgReg.bitFields.fifoLineSize = RXPP_FRAGMENT_FIFO_DEPTH - 1;
	shramFifoCfgReg.bitFields.shramFifoBaseAddr = (((uint32)RxPpShramFragmentFifo) >> 3);
	shramFifoCfgReg.bitFields.shramFifoEn = TRUE;
	
	RegAccess_Write(REG_RX_PP_SHRAM_FIFO_CFG, shramFifoCfgReg.val);
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
/**********************************************************************************
RxPp_FragmentFifoRead Fifo

Description:
------------
	Reads a line form fragments fifo
	
Input:
-----
	void
Output:
-------
	None
Returns:
--------
       None
**********************************************************************************/

void RxPp_FragmentFifoRead(RxPpFifoFragLine_t* fifoFragLine)
{
	RxPpFifoFragStructure_t   *fifoLineReg;
	RegRxPpShramFifoSub_u     rxPpShramFifoSubReg; // the number of entries that were read
	RegRxPpShramFifoStatus_u  rxPpShramFifoStatusReg;

	// check if we had FIFO overflows
	RegAccess_Read(REG_RX_PP_SHRAM_FIFO_STATUS, &rxPpShramFifoStatusReg.val);

	ASSERT(!(rxPpShramFifoStatusReg.bitFields.shramFifoOverflow)); // Need to increase configured fifo size (RxPpManagementObj.fifoLineSize)

	rxPpShramFifoSubReg.val = 0;
	
	if (!(rxPpShramFifoStatusReg.bitFields.shramFifoEmpty)) // check that fifo is not empty.
	{
		fifoFragLine->valid = TRUE;

		// read the line that the read index points to
		fifoLineReg = &RxPpShramFragmentFifo[RxPpFragmentReadIndex];
		
		RxPpFragmentReadIndex++;
		if (RxPpFragmentReadIndex >= RXPP_FRAGMENT_FIFO_DEPTH)
		{
			RxPpFragmentReadIndex = 0;
		}

		// fill the returned struct
		fifoFragLine->stationId = fifoLineReg->stationId;
		fifoFragLine->tid = fifoLineReg->tid;
		fifoFragLine->sn = fifoLineReg->sn;
		fifoFragLine->fragStart = fifoLineReg->fragStart;
	
		// notify HW that FW read a line
		
		rxPpShramFifoSubReg.bitFields.shramFifoSub = 1;
		RegAccess_Write(REG_RX_PP_SHRAM_FIFO_SUB, rxPpShramFifoSubReg.val);		
	}
	else
	{
		fifoFragLine->valid = FALSE;
	}

}


/**********************************************************************************

RxPp_Init 

Description:
------------
Initialize Rx post processing HW. Configures the following parameters:
	Skip bit - configured per RD type and determines if the RD will be forwarded to the Reordering module or forwarded directly to the status bit check.
	Umac Bit - configured per RD type and determines if RD will be forwarded to status bit check or forwarded to UMAC.
	Dlm Lists - Configuration of the list indexes at the RX DLM lists of the following list types: MPDU in list, Eror list, Umac done list, Drop src 0 - 7 lists, forward Rd type 0 -16 lists.

Input:
-----
	void

Output:
-------
	None

Returns:
--------
	void 

**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void RxPp_Init(void)
{
	RegRxPpRdTypeSkipUmacCfg_u	rdTypeSkipUmacCfg;  // RD_TYPE_SKIP_UMAC_CFG (offset 0x487c)
	RegRxPpConfigList_u 		dlmIdxCfg;	
	RegRxPpRxCountersCfg_u		rxCountersCfgReg;
	RegRxPpRxPpChickenBits_u	RegRxPpRxPpChickenBits;
	int i = 0;

	// configure RD types forward list in a for loop.
	uint32* pForwardList = (uint32*)REG_RX_PP_RD_TYPE0_FORWARD_LIST;

	/* Initialize the pointer of the Rx Pp Ram */
	RxPpRam = (RxPpRam_t *)(RX_PP_BASE_ADDRESS + RXPP_RAM_OFFSET_FROM_BASE_ADDRESS);

	// config the global per RD type parameters (1) skipBit  (2) umac bit
	rdTypeSkipUmacCfg.bitFields.skipBit = RX_PP_SKIP_REORDERING_RD_TYPES;
	rdTypeSkipUmacCfg.bitFields.umacBit = RX_PP_SEND_TO_UMAC_ALL_DATA_RD_TYPES;
	
    RegAccess_Write(REG_RX_PP_RD_TYPE_SKIP_UMAC_CFG, rdTypeSkipUmacCfg.val);

	// config the DLM lists indexes

	// config MPDU_IN list index
	dlmIdxCfg.bitFields.dlmListIdx = HW_Q_MANAGER_RX_POST_PROCESSING_INPUT_LIST;
	RegAccess_Write(REG_RX_PP_MPDU_IN_LIST, dlmIdxCfg.val);		

	// config ERROR list index
	dlmIdxCfg.bitFields.dlmListIdx = HW_Q_MANAGER_RX_READY_LIST_ERROR_RDS;
	RegAccess_Write(REG_RX_PP_ERR_LIST, dlmIdxCfg.val);		

	// config the UMAC_DONE list index
	dlmIdxCfg.bitFields.dlmListIdx = HW_Q_MANAGER_RX_READY_LIST_DATA_TEMPORARY_FORWARD_TO_FW;
	RegAccess_Write(REG_RX_PP_UMAC_DONE_LIST, dlmIdxCfg.val);		

	// config the Libarator list index
#if defined (RX_LIST_DEBUG)
	dlmIdxCfg.bitFields.dlmListIdx = HW_Q_MANAGER_RX_READY_LIST_DEBUG_DROP;
#else
	dlmIdxCfg.bitFields.dlmListIdx = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
#endif
	RegAccess_Write(REG_RX_PP_LIBERATOR_LIST, dlmIdxCfg.val);		

	// config the rd type forward lists: FW and Driver
	for (i=0 ; i < RD_TYPE_LAST; i++)
	{	
		RegAccess_Write((uint32)pForwardList, RxPpForwardRdTypeList[i]);
		pForwardList++;
	}

	// Enable Rx PP Counters
	rxCountersCfgReg.val =0;
	rxCountersCfgReg.bitFields.qosRxEn = TRUE;
	rxCountersCfgReg.bitFields.rdCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdDelayedCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdSwDropCountEn = TRUE;
	rxCountersCfgReg.bitFields.rdDuplicateDropCountEn = TRUE;
	rxCountersCfgReg.bitFields.missingSnSwUpdateCountEn = TRUE;
	rxCountersCfgReg.bitFields.mpduUniOrMgmtEn = TRUE;
	rxCountersCfgReg.bitFields.mpduRetryEn = TRUE;
	rxCountersCfgReg.bitFields.amsduCountEn = TRUE;
	rxCountersCfgReg.bitFields.dropMpduEn = TRUE;
	rxCountersCfgReg.bitFields.mpduTypeNotSupportedEn = TRUE;
	rxCountersCfgReg.bitFields.barMpduCountEn = TRUE;
	rxCountersCfgReg.bitFields.replayEn = TRUE;
	rxCountersCfgReg.bitFields.replayMgmtEn = TRUE;
	rxCountersCfgReg.bitFields.tkipCountEn = TRUE;
	rxCountersCfgReg.bitFields.failureCountEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassAmsduBytesStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxcAmpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassMpduInAmpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassOctetMpduStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassDropStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxClassSecurMisStbEn = TRUE;
	rxCountersCfgReg.bitFields.rxcCrcErrorStbEn = TRUE;
	rxCountersCfgReg.bitFields.delineatorCrcErrorStbEn = TRUE;
	rxCountersCfgReg.bitFields.bcMcCountEn = TRUE;
	rxCountersCfgReg.bitFields.qosRxDecEn = TRUE;						
	RegAccess_Write(REG_RX_PP_RX_COUNTERS_CFG,rxCountersCfgReg.val); 

	RegRxPpRxPpChickenBits.val = 0;
	RegRxPpRxPpChickenBits.bitFields.pnFixChickenBit = TRUE;
	RegAccess_Write(REG_RX_PP_RX_PP_CHICKEN_BITS,RegRxPpRxPpChickenBits.val);

	// Reset the RXPP RAM (RXPP RAM unicast = 0x0000 - 0x4000 , RXPP RAM Management 0x4000 -0x4800)
	memset32(RxPpRam, 0, CONVERT_BYTES_TO_WORDS(sizeof(RxPpRam_t)));

	memset(&RxPpLastSn, 0, sizeof(RxPpLastSn));
	
	RxPp_ConfigFragmentFifo();
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

