/***************************************************************
 File:		SenderInterface.c
 Module:	  	Sender interface
 Purpose: 	holds all the implementation of the  FW - sender interface module
 Description:  This module contains the implementation of the  FW - sender interface
 			module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "SenderInterface_Api.h"
#include "SenderInterface.h"
#include "MacGenriscTxRegs.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "ShramDeliaDescriptors.h"
#include "ShramSender.h"
#include "ShramStationDatabase.h"
#include "ShramVapDatabase.h"
#include "ShramGroupDatabase.h"
#include "ShramPacketDescriptors.h"
#include "ShramSecurity.h"
#include "ShramGenRiscMailboxes.h"
#include "HostInterface_API.h"
#include "ErrorHandler_Api.h"
#include "lm.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "TxSender_ScratchPadApi.h"
#include "ShramGenriscInitVectorStartAddr.h"
#include "ShramAtf.h"
/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_GENRISCS_INTERFACE
#define LOG_LOCAL_FID 3

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void txSenderInterface_InitFwTxSenderMailboxes(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
static FwTxSenderMailbox0FifoInParams_t		*pFwTxSenderMailbox0FifoInParams;
static FwTxSenderMailbox0FifoOutParams_t	*pFwTxSenderMailbox0FifoOutParams;
#endif //FW_TX_SENDER_MAILBOX_0_ENABLED
#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
static FwTxSenderMailbox1FifoInParams_t		*pFwTxSenderMailbox1FifoInParams;
static FwTxSenderMailbox1FifoOutParams_t	*pFwTxSenderMailbox1FifoOutParams;
#endif //FW_TX_SENDER_MAILBOX_1_ENABLED

uint8 SenderBand = BAND_5_2_GHZ;

/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

isr_Sender  


Description:
------------
interrupt routine that handles errors scenarions from the sender 



Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_Sender(void)
{
#if defined (ENET_INC_UMAC)
	RegMacGenriscTxGenriscUpperIrqClr_u irqClearRegister;
	RegMacGenriscTxGenriscUpperIrqStatus_u irqStatusRegister;
#elif defined (ENET_INC_LMAC)
	RegMacGenriscTxGenriscLowerIrqClr_u irqClearRegister;
	RegMacGenriscTxGenriscLowerIrqStatus_u irqStatusRegister;
#endif
	uint32 eventIndex;
#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
	FwTxSenderMailbox0FifoOutMessage_t *pBody0;
#endif //FW_TX_SENDER_MAILBOX_0_ENABLED
#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
	FwTxSenderMailbox1FifoOutMessage_t *pBody1;
#endif //FW_TX_SENDER_MAILBOX_1_ENABLED
#if defined FW_TX_SENDER_MAILBOX_0_ENABLED || defined FW_TX_SENDER_MAILBOX_1_ENABLED
	uint32 idx;
	K_MSG *pMsg;
#endif
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams = NULL;
	uint32 errorFileId;
	uint32 errorLine;
	uint32 state;
	UNUSED_PARAM(errorFileId);
	UNUSED_PARAM(errorLine);
	UNUSED_PARAM(state);

	irqClearRegister.val = 0;
	irqStatusRegister.val = 0;


#if defined (ENET_INC_UMAC)
	RegAccess_Read(REG_MAC_GENRISC_TX_GENRISC_UPPER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscUpperIrqClr = (0x1 << eventIndex);

	RegAccess_Write(REG_MAC_GENRISC_TX_GENRISC_UPPER_IRQ_CLR, irqClearRegister.val);
#elif defined (ENET_INC_LMAC)
	RegAccess_Read(REG_MAC_GENRISC_TX_GENRISC_LOWER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscLowerIrqClr = (0x1 << eventIndex);

	RegAccess_Write(REG_MAC_GENRISC_TX_GENRISC_LOWER_IRQ_CLR, irqClearRegister.val);
#else
	#error "either umac or lmac must be defined"
#endif
	
	switch (eventIndex)
	{
		case TX_SENDER_2_MIPS_INTERRUPTS_ERROR_START_BIT:
			pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
			errorFileId = pScratchPadApiParams->error_last_file_id;
			errorLine = pScratchPadApiParams->error_last_line;
			state = pScratchPadApiParams->state;
			ILOG0_DDD("Sender Error: file ID %d, line %d, state %d", errorFileId, errorLine, state);
			FATAL("Sender Error Start");
			break;

#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
		case TX_SENDER_2_MIPS_INTERRUPTS_MAILBOX_0_FIFO_OUT_NOT_EMPTY_START_BIT:
			idx = 0;
			while (idx < FW_GENRISC_TX_SENDER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES)
			{
				pMsg = TxSenderInterface_Mailbox0_Pop();
				if (pMsg)
				{
					pBody0 = (FwTxSenderMailbox0FifoOutMessage_t *)pK_MSG_DATA(pMsg);

					//send message to designated task
					switch(pBody0->messageId)
					{
                    case FW_TX_SENDER_MAILBOX_0_FIFO_OUT_STOP_TX_INDICATION:
                        OSAL_SEND_MESSAGE(ATF_BUFFERING_SENDER_INDICATION_STOPTX, TASK_ATF, pMsg, GET_DEFAULT_VAP_FOR_MY_BAND());
                        break;
					default:
						FATAL("Unsupported Message ID");
					}
				}
				else
				{
					break;
				}
				idx++;
			}
			break;
#endif //FW_TX_SENDER_MAILBOX_0_ENABLED
#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
		case TX_SENDER_2_MIPS_INTERRUPTS_MAILBOX_1_FIFO_OUT_NOT_EMPTY_START_BIT:
			idx = 0;
			while (idx < FW_GENRISC_TX_SENDER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES)
			{
				pMsg = TxSenderInterface_Mailbox1_Pop();

				if (pMsg)
				{
					pBody1 = (FwTxSenderMailbox1FifoOutMessage_t *)pK_MSG_DATA(pMsg);

					//send message to designated task
					switch(pBody1->messageId)
					{
					default:
						FATAL("Unsupported Message ID");
					}
				}
				else
				{
					break;
				}
				idx++;
			}
			break;
#endif //FW_TX_SENDER_MAILBOX_1_ENABLED
		default:
			FATAL("Sender Unknown Event");
	}
}

/**********************************************************************************

Sender_Halt  


Description:
------------
stop the sender

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 Sender_Halt(void)
{	
	RegMacGenriscTxStopOp_u haltRegister;
	RegMacGenriscTxLastPcExecuted_u lastPcExecuted;

	haltRegister.val = 0;


	/* Start the sender */
	haltRegister.bitFields.stopOp = SENDER_HALT_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_TX_STOP_OP, haltRegister.val);

	/* Read last pc executed */
	RegAccess_Read(REG_MAC_GENRISC_TX_LAST_PC_EXECUTED, &lastPcExecuted.val);

	return lastPcExecuted.bitFields.lastExecuted;
}


/**********************************************************************************

Sender_initialize  


Description:
------------
fill the initizlization structre of the sender and start the sender

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void Sender_Initialize()
{	
	RegMacGenriscTxStartOp_u startRegister;

	SenderInterfaceParams_t *pSenderInitializationStructure;

    uint32 timeStamp = 0;
	UNUSED_PARAM(timeStamp);
	
	txSenderInterface_InitFwTxSenderMailboxes();

	startRegister.val = 0;
	
	pSenderInitializationStructure = (SenderInterfaceParams_t *)CONVERT_TO_PHYSICAL_ADDR(&SenderInitializationStructure);
	GenriscsInitVectorStartAddr.txSenderInitVectorPtr = (uint32)pSenderInitializationStructure;
	
	memset((void*)&SenderInitializationStructure, 0, sizeof(SenderInitializationStructure));
	SenderInitializationStructure.securityDescriptors0BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(txSecurityDescriptors[0][0]));
	SenderInitializationStructure.securityDescriptors1BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(txSecurityDescriptors[1][0]));
	SenderInitializationStructure.securityDescriptors2BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(txSecurityDescriptors[2][0]));
	SenderInitializationStructure.securityDescriptors3BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(txSecurityDescriptors[3][0]));
	SenderInitializationStructure.deliaDescriptors0BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(DeliaDescriptors[0][0]));
	SenderInitializationStructure.deliaDescriptors1BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(DeliaDescriptors[1][0]));
	SenderInitializationStructure.deliaDescriptors2BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(DeliaDescriptors[2][0]));
	SenderInitializationStructure.deliaDescriptors3BaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&(DeliaDescriptors[3][0]));
	SenderInitializationStructure.stationDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(StaDbHwEntries);
	SenderInitializationStructure.stationDatabaseEntrySize = sizeof(StaDb_t);
	SenderInitializationStructure.stationDatabaseMacAddressOffset = FM_STRUCT_OFFSET(StaDb_t,ra);
	SenderInitializationStructure.stationDatabaseSequenceNumberOffset = FM_STRUCT_OFFSET(StaDb_t,tid);
	SenderInitializationStructure.stationDatabaseRaExtensionWpParamsOffset = FM_STRUCT_OFFSET(StaDb_t,raExtensionWpParams);
	SenderInitializationStructure.vapDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(VapDbHwEntries);
	SenderInitializationStructure.vapDatabaseEntrySize = sizeof(VapDb_t);
	SenderInitializationStructure.vapDatabaseMacAddressOffset = FM_STRUCT_OFFSET(VapDb_t,bssid);
	SenderInitializationStructure.vapDatabaseSequenceNumberOffset = FM_STRUCT_OFFSET(VapDb_t,tid); 
	SenderInitializationStructure.vapDatabaseTcrTemplateOffset = FM_STRUCT_OFFSET(VapDb_t,tcrTemplates);
	SenderInitializationStructure.groupDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(GroupDbHwEntries);
	SenderInitializationStructure.groupDatabaseEntrySize = sizeof(MuGrpDb_t);
	SenderInitializationStructure.pdPoolBaseAddress = (uint32)CONVERT_TO_PHYSICAL_ADDR(PacketDescriptorsPtr);
	SenderInitializationStructure.wlanInterface = 0; 				/* TBD add API to get wlan IF*/
	SenderInitializationStructure.loggerStream = 0;					/* TBD - Mechanism to change logger stream  - Based on mailbox*/
	SenderInitializationStructure.frequencyBand	= SenderBand; 		/*TBD - Add API to StaDB*/
    SenderInitializationStructure.AtfDatabaseBaseAddress = (uint32)CONVERT_TO_PHYSICAL_ADDR(&AtfShramDb);
	SenderInitializationStructure.enabledTasksBitmap.bitFields.suProbingHandlingEnabled = TRUE;
	SenderInitializationStructure.enabledTasksBitmap.bitFields.muProbingHandlingEnabled = TRUE;
	SenderInitializationStructure.enabledTasksBitmap.bitFields.autoFillRecoveryEnabled = TRUE;
#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
	SenderInitializationStructure.mailbox0_FifoInStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailbox0FifoInParams);
	SenderInitializationStructure.mailbox0_FifoInNumOfEntries = FW_GENRISC_TX_SENDER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES;
	SenderInitializationStructure.mailbox0_FifoOutStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailbox0FifoOutParams);
	SenderInitializationStructure.mailbox0_FifoOutNumOfEntries= FW_GENRISC_TX_SENDER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES;
#else
	SenderInitializationStructure.mailbox0_FifoInStructStartAddress = SENDER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	SenderInitializationStructure.mailbox0_FifoInNumOfEntries = 0x0;
	SenderInitializationStructure.mailbox0_FifoOutStructStartAddress = SENDER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	SenderInitializationStructure.mailbox0_FifoOutNumOfEntries= 0x0;
#endif
#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
	SenderInitializationStructure.mailbox1_FifoInStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailbox1FifoInParams);
	SenderInitializationStructure.mailbox1_FifoInNumOfEntries= FW_GENRISC_TX_SENDER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES;
	SenderInitializationStructure.mailbox1_FifoOutStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailbox1FifoOutParams);
	SenderInitializationStructure.mailbox1_FifoOutNumOfEntries = FW_GENRISC_TX_SENDER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES;
#else
	SenderInitializationStructure.mailbox1_FifoInStructStartAddress = SENDER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	SenderInitializationStructure.mailbox1_FifoInNumOfEntries = 0x0;
	SenderInitializationStructure.mailbox1_FifoOutStructStartAddress = SENDER_INIT_NOT_VALID_INPUT_EXPECTED_VALUE;
	SenderInitializationStructure.mailbox1_FifoOutNumOfEntries= 0x0;
#endif

	/* Start the sender */
	startRegister.bitFields.startOp = SENDER_START_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_TX_START_OP, startRegister.val);

	timeStamp = GET_TSF_TIMER_LOW();
	while(SENDER_COMPLETED_CHI_MAGIC != SenderInitializationStructure.chiMagic)
	{
		DEBUG_ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= SENDER_MAX_MICROSECONDS_FOR_INITIALIZATION); 
	}	

}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif
static void txSenderInterface_InitFwTxSenderMailboxes(void)
{
#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
	memset(&FwTxSenderMailbox0FifoInBuffer, 0x0, sizeof(FwTxSenderMailbox0FifoInBuffer));
	memset(&FwTxSenderMailbox0FifoOutBuffer, 0x0, sizeof(FwTxSenderMailbox0FifoOutBuffer));
	pFwTxSenderMailbox0FifoInParams = (FwTxSenderMailbox0FifoInParams_t	*)(&FwTxSenderMailbox0FifoInBuffer);
	pFwTxSenderMailbox0FifoOutParams = (FwTxSenderMailbox0FifoOutParams_t *)(&FwTxSenderMailbox0FifoOutBuffer);
#endif

#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
	memset(&FwTxSenderMailbox1FifoInBuffer, 0x0, sizeof(FwTxSenderMailbox1FifoInBuffer));
	memset(&FwTxSenderMailbox1FifoOutBuffer, 0x0, sizeof(FwTxSenderMailbox1FifoOutBuffer));
	pFwTxSenderMailbox1FifoInParams = (FwTxSenderMailbox1FifoInParams_t	*)(&FwTxSenderMailbox1FifoInBuffer);
	pFwTxSenderMailbox1FifoOutParams = (FwTxSenderMailbox1FifoOutParams_t *)(&FwTxSenderMailbox1FifoOutBuffer);
#endif
}
		
#ifdef FW_TX_SENDER_MAILBOX_0_ENABLED
void TxSenderInterface_Mailbox0_Push(FwTxSenderMailbox0FifoInMessage_t *message)
{
	RegMacGenriscTxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwTxSenderMailbox0FifoInParams->writeCounter - pFwTxSenderMailbox0FifoInParams->readCounter) < FW_GENRISC_TX_SENDER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES);
	if ((pFwTxSenderMailbox0FifoInParams->writeCounter - pFwTxSenderMailbox0FifoInParams->readCounter) < FW_GENRISC_TX_SENDER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES)
	{
		memcpy32(&(pFwTxSenderMailbox0FifoInParams->messageArray[(pFwTxSenderMailbox0FifoInParams->writeCounter % FW_GENRISC_TX_SENDER_MAILBOX_0_IN_FIFO_NUMBER_OF_ENTRIES)]), 
			message, 
			(sizeof(FwTxSenderMailbox0FifoInMessage_t) >> 0x2));

		pFwTxSenderMailbox0FifoInParams->writeCounter++;

		/* set interrupt to genrisc */
		irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x0); //bit 0 for mailbox 0
		RegAccess_Write(REG_MAC_GENRISC_TX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);
	}

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *TxSenderInterface_Mailbox0_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwTxSenderMailbox0FifoOutMessage_t *pBody;

	if (pFwTxSenderMailbox0FifoOutParams->readCounter != pFwTxSenderMailbox0FifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwTxSenderMailbox0FifoOutMessage_t));
		pBody = (FwTxSenderMailbox0FifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwTxSenderMailbox0FifoOutParams->messageArray[(pFwTxSenderMailbox0FifoOutParams->readCounter % FW_GENRISC_TX_SENDER_MAILBOX_0_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwTxSenderMailbox0FifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwTxSenderMailbox0FifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}

	return pMsg;
}


void TxSenderInterface_Send_Band(uint8 band)
{

    FwTxSenderMailbox0FifoInMessage_t message;	

	if (SenderBand != band)
	{
    	message.messageId = FW_TX_SENDER_MAILBOX_0_FIFO_IN_MESSAGE_ID_SET_BAND;
    	message.messageValue = band;   
    	TxSenderInterface_Mailbox0_Push(&message);
		SenderBand = band;
	}
}

#endif //FW_TX_SENDER_MAILBOX_0_ENABLED

#ifdef FW_TX_SENDER_MAILBOX_1_ENABLED
void TxSenderInterface_Mailbox1_Push(FwTxSenderMailbox1FifoInMessage_t *message)
{
	RegMacGenriscTxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwTxSenderMailbox1FifoInParams->writeCounter - pFwTxSenderMailbox1FifoInParams->readCounter) < FW_GENRISC_TX_SENDER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES);

	memcpy32(&(pFwTxSenderMailbox1FifoInParams->messageArray[(pFwTxSenderMailbox1FifoInParams->writeCounter % FW_GENRISC_TX_SENDER_MAILBOX_1_IN_FIFO_NUMBER_OF_ENTRIES)]), 
		message, 
		(sizeof(FwTxSenderMailbox1FifoInMessage_t) >> 0x2));

	pFwTxSenderMailbox1FifoInParams->writeCounter++;

	/* set interrupt to genrisc */
	irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x1); //bit 1 for mailbox 1
	RegAccess_Write(REG_MAC_GENRISC_TX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *TxSenderInterface_Mailbox1_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwTxSenderMailbox1FifoOutMessage_t *pBody;

	if (pFwTxSenderMailbox1FifoOutParams->readCounter != pFwTxSenderMailbox1FifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwTxSenderMailbox1FifoOutMessage_t));
		pBody = (FwTxSenderMailbox1FifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwTxSenderMailbox1FifoOutParams->messageArray[(pFwTxSenderMailbox1FifoOutParams->readCounter % FW_GENRISC_TX_SENDER_MAILBOX_1_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwTxSenderMailbox1FifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwTxSenderMailbox1FifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}

	return pMsg;
}

#endif //FW_TX_SENDER_MAILBOX_1_ENABLED













