/***************************************************************
 File:		SenderInterface.c
 Module:	  	Sender interface
 Purpose: 	holds all the implementation of the  FW - sender interface module
 Description:  This module contains the implementation of the  FW - sender interface
 			module
***************************************************************/

/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "SenderInterface_Api.h"
#include "SenderInterface.h"
#include "MacGenriscTxRegs.h"
#include "RegAccess_Api.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#include "enet_pas.h"
#include "ShramDeliaDescriptors.h"
#include "ShramSender.h"
#include "ShramStationDatabase.h"
#include "ShramVapDatabase.h"
#include "ShramGroupDatabase.h"
#include "ShramPacketDescriptors.h"
#include "ShramSecurity.h"
#include "ShramGenRiscMailboxes.h"
#include "HostInterface_API.h"
#include "ErrorHandler_Api.h"
#include "lm.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "TxSender_ScratchPadApi.h"
#include "ShramGenriscInitVectorStartAddr.h"
#include "ShramAtf.h"
#include "HwSecurity_API.h"
#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY
#include "ShramRxDescriptors.h"
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_GENRISCS_INTERFACE
#define LOG_LOCAL_FID 4

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void txSenderInterface_InitFwTxSenderMailboxes(void);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static FwTxSenderMailboxFifoInParams_t	*pFwTxSenderMailboxFifoInParams;
static FwTxSenderMailboxFifoOutParams_t	*pFwTxSenderMailboxFifoOutParams;
uint8 SenderBand = BAND_5_2_GHZ;

/*---------------------------------------------------------------------------------
/						Debug Section									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/




/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

isr_Sender  


Description:
------------
interrupt routine that handles errors scenarions from the sender 



Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID isr_Sender(void)
{
	RegMacGenriscTxGenriscUpperIrqClr_u irqClearRegister;
	RegMacGenriscTxGenriscUpperIrqStatus_u irqStatusRegister;
	uint32 eventIndex;
	FwTxSenderMailboxFifoOutMessage_t *pBody;
	uint32 idx;
	K_MSG *pMsg;
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams = NULL;
	uint32 errorFileId;
	uint32 errorLine;
	uint32 state;
	uint32 deliaStatus;

	irqClearRegister.val = 0;
	irqStatusRegister.val = 0;


	RegAccess_Read(REG_MAC_GENRISC_TX_GENRISC_UPPER_IRQ_STATUS, &irqStatusRegister.val);

	eventIndex = Utils_CountTrailingZeros(irqStatusRegister.val);
	irqClearRegister.bitFields.genriscUpperIrqClr = (0x1 << eventIndex);
	
	pScratchPadApiParams = (TxSender_ScratchPadApiParams_t *)(B0_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	
	errorFileId = pScratchPadApiParams->error_last_file_id;
	errorLine = pScratchPadApiParams->error_last_line;
	state = pScratchPadApiParams->state;
	deliaStatus = pScratchPadApiParams->deliaStatusError;

	RegAccess_Write(REG_MAC_GENRISC_TX_GENRISC_UPPER_IRQ_CLR, irqClearRegister.val);
	
	switch (eventIndex)
	{
		case TX_SENDER_2_MIPS_INTERRUPTS_GENERAL_ERROR_START_BIT:
			ILOG0_DDD("Sender Error: file ID %d, line %d, state %d", errorFileId, errorLine, state);
			FATAL("Sender Error Start");
			break;

		case TX_SENDER_2_MIPS_INTERRUPTS_DELIA_ERROR_IN_MIDDLE_OF_TX_START_BIT:
			Pac_DeliaTxEndErrorDuringTx(deliaStatus);
			break;

		case TX_SENDER_2_MIPS_INTERRUPTS_DELIA_ERROR_IN_END_OF_TX_START_BIT:
			Pac_DeliaTxEndErrorAtEndOfTx(deliaStatus);
			break;
			
		case TX_SENDER_2_MIPS_INTERRUPTS_SEC_ERROR_IN_END_OF_TX_START_BIT:
			HwSecurity_TxEndSecurityNotInIdleError(deliaStatus);
			break;
			
		case TX_SENDER_2_MIPS_INTERRUPTS_MAILBOX_FIFO_OUT_NOT_EMPTY_START_BIT:
			idx = 0;
			while (idx < FW_GENRISC_TX_SENDER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES)
			{
				pMsg = TxSenderInterface_Mailbox_Pop();
				if (pMsg)
				{
					pBody = (FwTxSenderMailboxFifoOutMessage_t *)pK_MSG_DATA(pMsg);

					//send message to designated task
					switch(pBody->messageId)
					{
                    case FW_TX_SENDER_MAILBOX_FIFO_OUT_STOP_TX_INDICATION:
                        OSAL_SEND_MESSAGE(ATF_BUFFERING_SENDER_INDICATION_STOPTX, TASK_ATF, pMsg, VAP_ID_DO_NOT_CARE);
                        break;
					default:
						FATAL("Unsupported Message ID");
					}
				}
				else
				{
					break;
				}
				idx++;
			}
			break;
		default:
			FATAL("Sender Unknown Event");
	}
}

/**********************************************************************************

Sender_Halt  


Description:
------------
stop the sender

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint16 Sender_Halt(void)
{	
	RegMacGenriscTxStopOp_u haltRegister;
	RegMacGenriscTxLastPcExecuted_u lastPcExecuted;

	haltRegister.val = 0;


	/* Start the sender */
	haltRegister.bitFields.stopOp = SENDER_HALT_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_TX_STOP_OP, haltRegister.val);

	/* Read last pc executed */
	RegAccess_Read(REG_MAC_GENRISC_TX_LAST_PC_EXECUTED, &lastPcExecuted.val);

	return lastPcExecuted.bitFields.lastExecuted;
}


/**********************************************************************************

Sender_initialize  


Description:
------------
fill the initizlization structre of the sender and start the sender

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void Sender_Initialize()
{	
	RegMacGenriscTxStartOp_u 	    startRegister;
    RegMacGenriscTxAbortCntLimit_u  abortCntLimitRegister;
    uint32 						    timeStamp = 0;
	TxSenderInitVector_t*		    pSenderInitializationStructure;	
	Std_t *std;

	//use std to be available in debugger
	UNUSED_PARAM(std);
	
	txSenderInterface_InitFwTxSenderMailboxes();

	startRegister.val = 0;
    abortCntLimitRegister.val = 0;

	// Each band use a different place in SHRAM for init vector
#if defined(ENET_INC_LMAC0) 
	pSenderInitializationStructure = (TxSenderInitVector_t *)(&SenderInitializationStructure);
	GenriscsInitVectorStartAddr.txSenderB0InitVectorPtr = (uint32)CONVERT_TO_PHYSICAL_ADDR(pSenderInitializationStructure);
#elif defined(ENET_INC_LMAC1) 
	pSenderInitializationStructure = (TxSenderInitVector_t *)(&SenderInitializationStructure_B1);
	GenriscsInitVectorStartAddr.txSenderB1InitVectorPtr = (uint32)CONVERT_TO_PHYSICAL_ADDR(pSenderInitializationStructure);
#endif

	memset((void*)pSenderInitializationStructure, 0, sizeof(TxSenderInitVector_t));

#if defined(ENET_INC_LMAC0) 
	pSenderInitializationStructure->deliaDescriptorsBaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&DeliaDataStructure);

#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY
	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_SINGLE_BAND)
	{		
		pSenderInitializationStructure->mpduDescriptorsFreeListOffset = B0_CONVERT_MPDU_DESC_TO_OFFSET(B0_MpduDescriptors.singleBandDescriptors.sB0_TxMpduDescriptor);
		pSenderInitializationStructure->totalNumOfMpduDescriptors = NUM_OF_TX_MPDU_DESCRIPTORS_B0_SINGLE_BAND;
	}
	else
	{		
		pSenderInitializationStructure->mpduDescriptorsFreeListOffset = B0_CONVERT_MPDU_DESC_TO_OFFSET(B0_MpduDescriptors.dualBandDescriptors.sB0_TxMpduDescriptor);
		pSenderInitializationStructure->totalNumOfMpduDescriptors = NUM_OF_TX_MPDU_DESCRIPTORS_B0_DUAL_BAND;
	}
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY

#else //(ENET_INC_LMAC1) 
	pSenderInitializationStructure->deliaDescriptorsBaseAddress = CONVERT_TO_PHYSICAL_ADDR((uint32)&DeliaDataStructure_B1);

#ifdef WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY
	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_SINGLE_BAND)
	{		
		pSenderInitializationStructure->mpduDescriptorsFreeListOffset = B1_CONVERT_MPDU_DESC_TO_OFFSET(B1_MpduDescriptors.singleBandDescriptors.sB1_TxMpduDescriptor);
		pSenderInitializationStructure->totalNumOfMpduDescriptors = NUM_OF_TX_MPDU_DESCRIPTORS_B1_SINGLE_BAND;
	}
	else
	{		
		pSenderInitializationStructure->mpduDescriptorsFreeListOffset = B1_CONVERT_MPDU_DESC_TO_OFFSET(B1_MpduDescriptors.dualBandDescriptors.sB1_TxMpduDescriptor);
		pSenderInitializationStructure->totalNumOfMpduDescriptors = NUM_OF_TX_MPDU_DESCRIPTORS_B1_DUAL_BAND;
	}
#endif //WORKAROUND_FOR_HW_BUG_IN_DELIA_POP_MPDU_LIST_IN_RECOVERY

#endif //ENET_INC_LMAC0

	pSenderInitializationStructure->stationDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(StaDbHwEntries);
	pSenderInitializationStructure->stationDatabaseEntrySize = sizeof(StaDb_t);
	pSenderInitializationStructure->stationDatabaseSeqNumOffset = FM_STRUCT_OFFSET(StaDb_t,tid);
	pSenderInitializationStructure->stationDatabaseWpParamsOffset = FM_STRUCT_OFFSET(StaDb_t, wp);
	pSenderInitializationStructure->vapDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(VapDbHwEntries);
	pSenderInitializationStructure->vapDatabaseEntrySize = sizeof(VapDb_t);
	pSenderInitializationStructure->vapDatabaseMacAddressOffset = FM_STRUCT_OFFSET(VapDb_t, bssid);
	pSenderInitializationStructure->vapDatabaseSeqNumOffset = FM_STRUCT_OFFSET(VapDb_t,tid);
	pSenderInitializationStructure->vapDatabaseTcrTemplateOffset = FM_STRUCT_OFFSET(VapDb_t, tcrTemplates);
	pSenderInitializationStructure->groupDatabaseBaseAddress = CONVERT_TO_PHYSICAL_ADDR(GroupDbHwEntries);
	pSenderInitializationStructure->groupDatabaseEntrySize = sizeof(MuGrpDb_t);	
	pSenderInitializationStructure->groupDatabaseWpParamsOffset= FM_STRUCT_OFFSET(MuGrpDb_t, wp);
	pSenderInitializationStructure->wlanInterface = 0; /* TBD add API to get wlan IF*/
	pSenderInitializationStructure->loggerStream = 0; /* TBD - Mechanism to change logger stream  - Based on mailbox*/
	pSenderInitializationStructure->frequencyBand = SenderBand; /*TBD - Add API to StaDB*/
	pSenderInitializationStructure->atfDatabaseStartAddress = (uint32)CONVERT_TO_PHYSICAL_ADDR(&AtfShramDb);
	pSenderInitializationStructure->autoFillRecovery = TRUE;
	pSenderInitializationStructure->bfReportHandling = FALSE;
	pSenderInitializationStructure->suProbingHandling = TRUE;
	pSenderInitializationStructure->vhtMuProbingHandling = FALSE;
	pSenderInitializationStructure->heMuProbingHandling = TRUE;
	pSenderInitializationStructure->mailboxFifoInStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailboxFifoInParams);
	pSenderInitializationStructure->mailboxFifoInNumOfEntries = FW_GENRISC_TX_SENDER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES;
	pSenderInitializationStructure->mailboxFifoOutStructStartAddress = CONVERT_TO_PHYSICAL_ADDR(pFwTxSenderMailboxFifoOutParams);
	pSenderInitializationStructure->mailboxFifoOutNumOfEntries = FW_GENRISC_TX_SENDER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES;

	/* Start the sender */
	startRegister.bitFields.startOp = SENDER_START_VALUE;
	RegAccess_Write(REG_MAC_GENRISC_TX_START_OP, startRegister.val);
#ifndef ARC_SIM
	timeStamp = GET_TSF_TIMER_LOW();
	while(SENDER_COMPLETED_CHI_MAGIC != pSenderInitializationStructure->chiMagic)
	{
		DEBUG_ASSERT((GET_TSF_TIMER_LOW() - timeStamp) <= SENDER_MAX_MICROSECONDS_FOR_INITIALIZATION); 
	}	
#endif

    /* Set abort counter limit */
    abortCntLimitRegister.bitFields.abortCntLimit = 0x7FF;
    RegAccess_Write(REG_MAC_GENRISC_TX_ABORT_CNT_LIMIT, abortCntLimitRegister.val);
}

static void txSenderInterface_InitFwTxSenderMailboxes(void)
{
#if defined(ENET_INC_LMAC0) 
	memset(&FwTxSenderMailboxFifoInBuffer, 0x0, sizeof(FwTxSenderMailboxFifoInBuffer));
	memset(&FwTxSenderMailboxFifoOutBuffer, 0x0, sizeof(FwTxSenderMailboxFifoOutBuffer));
	pFwTxSenderMailboxFifoInParams = (FwTxSenderMailboxFifoInParams_t	*)(&FwTxSenderMailboxFifoInBuffer);
	pFwTxSenderMailboxFifoOutParams = (FwTxSenderMailboxFifoOutParams_t *)(&FwTxSenderMailboxFifoOutBuffer);
#endif
			
#if defined(ENET_INC_LMAC1) 
	memset(&FwTxSenderMailboxFifoInBuffer_B1, 0x0, sizeof(FwTxSenderMailboxFifoInBuffer_B1));
	memset(&FwTxSenderMailboxFifoOutBuffer_B1, 0x0, sizeof(FwTxSenderMailboxFifoOutBuffer_B1));
	pFwTxSenderMailboxFifoInParams = (FwTxSenderMailboxFifoInParams_t	*)(&FwTxSenderMailboxFifoInBuffer_B1);
	pFwTxSenderMailboxFifoOutParams = (FwTxSenderMailboxFifoOutParams_t *)(&FwTxSenderMailboxFifoOutBuffer_B1);
#endif
}
		
void TxSenderInterface_Mailbox_Push(FwTxSenderMailboxFifoInMessage_t *message)
{
	RegMacGenriscTxMips2GenriscIrqSet_u irqSetRegister;
	TX_INTERRUPT_SAVE_AREA;

	irqSetRegister.val = 0;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	DEBUG_ASSERT((pFwTxSenderMailboxFifoInParams->writeCounter - pFwTxSenderMailboxFifoInParams->readCounter) < FW_GENRISC_TX_SENDER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES);
	if ((pFwTxSenderMailboxFifoInParams->writeCounter - pFwTxSenderMailboxFifoInParams->readCounter) < FW_GENRISC_TX_SENDER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES)
	{
		memcpy32(&(pFwTxSenderMailboxFifoInParams->messageArray[(pFwTxSenderMailboxFifoInParams->writeCounter % FW_GENRISC_TX_SENDER_MAILBOX_IN_FIFO_NUMBER_OF_ENTRIES)]), 
			message, 
			(sizeof(FwTxSenderMailboxFifoInMessage_t) >> 0x2));

		pFwTxSenderMailboxFifoInParams->writeCounter++;

		/* set interrupt to genrisc */
		irqSetRegister.bitFields.mips2GenriscIrqSet = (TRUE << 0x0); //bit 0 for mailbox
		RegAccess_Write(REG_MAC_GENRISC_TX_MIPS2GENRISC_IRQ_SET, irqSetRegister.val);
	}

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

K_MSG *TxSenderInterface_Mailbox_Pop(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *pMsg = NULL;
	FwTxSenderMailboxFifoOutMessage_t *pBody;

	if (pFwTxSenderMailboxFifoOutParams->readCounter != pFwTxSenderMailboxFifoOutParams->writeCounter)
	{
		pMsg = OSAL_GET_MESSAGE(sizeof(FwTxSenderMailboxFifoOutMessage_t));
		pBody = (FwTxSenderMailboxFifoOutMessage_t *)pK_MSG_DATA(pMsg);

		memcpy32(pBody,
			&(pFwTxSenderMailboxFifoOutParams->messageArray[(pFwTxSenderMailboxFifoOutParams->readCounter % FW_GENRISC_TX_SENDER_MAILBOX_OUT_FIFO_NUMBER_OF_ENTRIES)]),
			(sizeof(FwTxSenderMailboxFifoOutMessage_t) >> 0x2));

		OSAL_DISABLE_INTERRUPTS(&interrupt_save);

		pFwTxSenderMailboxFifoOutParams->readCounter++;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}

	return pMsg;
}


void TxSenderInterface_Send_Band(uint8 band)
{

    FwTxSenderMailboxFifoInMessage_t message;	

	if (SenderBand != band)
	{
   		message.messageId = FW_TX_SENDER_MAILBOX_FIFO_IN_MESSAGE_ID_SET_BAND;
    	message.messageValue = band;   
    	TxSenderInterface_Mailbox_Push(&message);
		SenderBand = band;
	}
}


