/***********************************************************************************
 File:			SmpsManager.c
 Module:		Smps Manager
 Purpose: 		handle Smps Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "TxPacketsClassifier_API.h"
#include "SmpsManager_Api.h"
#include "SmpsManager_StaManager.h"
#include "loggerAPI.h"
#include "CoC_Api.h"
#include "ChannelSwitchManager_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SMPS_MANAGER 
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define SMPS_MANAGER_INVALID_VAP	0xFF
#define SMPS_MANAGER_CONVERT_RX_ANT_TO_RX_NSS(x)	(x-1)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct SmpsManagerVapDb_s
{
	uint8					flags;	 /*VAP flags*/
	uint8					nextVap; /*We hold a linked list of VAPs that support static SMPS - we need to send Action Frames to STAs only if the VAP supports it*/
	uint8					prevVap;
	uint8					isActive;
} SmpsManagerVapDb_t;

typedef struct SmpsManagerDb_s
{
	uint8							vapHead;
	uint8							mode;
	uint8							numOfAntennas;
	uint8							maxRxAntsNum;		
	uint8							channelWidth;		
	uint8							numVaps;
} SmpsManagerDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void SmpsManager_AddVapToList(uint8 vapId);
static void SmpsManager_RemoveVapFromList(uint8 vapId);
static void SmpsManager_ClearVapEntry(uint8 vapId);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static SmpsManagerVapDb_t SmpsManager_VapDb[HW_NUM_OF_VAPS];
static SmpsManagerDb_t SmpsManager_Db;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

SmpsManager_AddVapToList 


Description:
------------
Utility function that add a VAP to the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_AddVapToList(uint8 vapId)
{
	if (SmpsManager_Db.vapHead != SMPS_MANAGER_INVALID_VAP)
	{
		SmpsManager_VapDb[vapId].nextVap = SmpsManager_Db.vapHead;
		SmpsManager_VapDb[SmpsManager_Db.vapHead].prevVap = vapId;
	}
	SmpsManager_Db.vapHead = vapId;
}

/**********************************************************************************

SmpsManager_RemoveVapFromList 


Description:
------------
Utility function that removes a VAP from the list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_RemoveVapFromList(uint8 vapId)
{
	uint8 prevVap = SmpsManager_VapDb[vapId].prevVap;
	uint8 nextVap = SmpsManager_VapDb[vapId].nextVap;

	if (prevVap == SMPS_MANAGER_INVALID_VAP)
	{
		SmpsManager_Db.vapHead = nextVap;
	}
	else
	{
		SmpsManager_VapDb[prevVap].nextVap = nextVap;
	}
	if (nextVap != SMPS_MANAGER_INVALID_VAP)
	{
		SmpsManager_VapDb[nextVap].prevVap = prevVap;
	}
}

/**********************************************************************************

SmpsManager_ClearVapEntry 


Description:
------------
Utility function that clears a VAP entry


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_ClearVapEntry(uint8 vapId)
{
	SmpsManager_VapDb[vapId].nextVap = SMPS_MANAGER_INVALID_VAP;
	SmpsManager_VapDb[vapId].prevVap = SMPS_MANAGER_INVALID_VAP;
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

SmpsManager_VapSmps 


Description:
------------
returns VAP SM mode


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 SmpsManager_VapSmps(uint8 vapId)
{
	uint8 smps = MTLK_BFIELD_GET(SmpsManager_VapDb[vapId].flags, VAP_ADD_FLAGS_SMPS);

	if (smps == TRUE)
	{
		return (VAP_SMPS_MODE_STATIC);
	}
 	return (VAP_SMPS_MODE_DISABLED);
}

/**********************************************************************************

SmpsManager_BuildOmn 


Description:
------------
Build Omn Frame

Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	
**********************************************************************************/

void SmpsManager_BuildOmn(TxPd_t *pd)
{
	TxPd_t *pdPointer = (TxPd_t *)pd;
	OperatingModeNotificationPayload_t *omnFramePayload = NULL;

	/*Build OMN frame*/
	omnFramePayload = (OperatingModeNotificationPayload_t *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	omnFramePayload->categoryCode = VHT_CATEGORY; // According to Standard
	omnFramePayload->actionCode = VHT_CATEGORY_OPERATING_MODE_NOTIFICATION; // According to Standard
	omnFramePayload->operatingModeField.rxNss = SMPS_MANAGER_CONVERT_RX_ANT_TO_RX_NSS(SmpsManager_Db.numOfAntennas);
	omnFramePayload->operatingModeField.rxNssType = 0; // According to Standard
	omnFramePayload->operatingModeField.channelWidth = SmpsManager_Db.channelWidth;
	omnFramePayload->operatingModeField.reserved = 0; // Clear reserved bits
	/*Set len in PD*/
	pdPointer->dataLength = sizeof(OperatingModeNotificationPayload_t) + frame_sizeOfNewManagementFrameHeader(FALSE);
	TxPacketsClassifier_SendManagementPacketFromFw(pd);	
}

/**********************************************************************************

SmpsManager_BuildSmps 


Description:
------------
Builds SMPS frame


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void SmpsManager_BuildSmps(TxPd_t *pd)
{
	TxPd_t *pdPointer = (TxPd_t *)pd;
	FM_PAYLOAD_ACTION_FRAME_GENERAL *smpsFramePayload = NULL;

	/*Build SMPS frame*/
	smpsFramePayload = (FM_PAYLOAD_ACTION_FRAME_GENERAL *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	smpsFramePayload->u8CategoryCode = HT_CATEGORY;
	smpsFramePayload->u8ActionCode = HT_CATEGORY_SM_POWER_SAVE;
	/*Just copy the mode here
	Following coding
	0 - Disabled
	1 - Static - sent
	3 - Dynamic - not supported by us
	*/
	smpsFramePayload->u8DialogToken = SmpsManager_Db.mode;
	/*Set len in PD*/
	pdPointer->dataLength = sizeof(FM_PAYLOAD_ACTION_FRAME_GENERAL) + frame_sizeOfNewManagementFrameHeader(FALSE);
	/*Put SMPS on STA queue*/
	TxPacketsClassifier_SendManagementPacketFromFw((TxPd_t *)pd);
}

/**********************************************************************************

SmpsManager_GetAntennaNumber 


Description:
------------
Returns current SMPS number of antennas


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 SmpsManager_GetAntennaNumber()
{
	return (SmpsManager_Db.numOfAntennas);
}

/**********************************************************************************

SmpsManager_GetMaxAntennaNumber 


Description:
------------
Returns max number of antennas


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 SmpsManager_GetMaxAntennaNumber(void)
{
	return (SmpsManager_Db.maxRxAntsNum);
}

/**********************************************************************************

SmpsManager_GetMode 


Description:
------------
Returns current SMPS mode


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 SmpsManager_GetMode()
{
	return (SmpsManager_Db.mode);
}

/**********************************************************************************

smpsManagerTxStartReq 


Description:
------------
SMPS TX START REQ from CoC


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerTxStartReq(K_MSG* smpsManagerMessage)
{
	SmpsManagerStartTxReqParams_t *txReqParams = (SmpsManagerStartTxReqParams_t *)pK_MSG_DATA(smpsManagerMessage);
	uint8 vapId = SmpsManager_Db.vapHead;

	/*Check if anything changed - this is also called when Channel Normal or CSA is set due to antenna mask per channel*/
	if ((SmpsManager_Db.mode == txReqParams->mode) && (SmpsManager_Db.numOfAntennas == txReqParams->numOfAntennas))
	{
		return;
	}	
	/*Store the new mode*/
	SmpsManager_Db.mode 			= txReqParams->mode;
	SmpsManager_Db.numOfAntennas	= txReqParams->numOfAntennas;	
	SmpsManager_Db.maxRxAntsNum		= txReqParams->maxRxAntsNum;	
	
	/*Now need to send event to registered STAs - only on VAPs that support */
	while (vapId != SMPS_MANAGER_INVALID_VAP)
	{
		/*Call SMPS manager STA manager to transmit SMPS action frames to registered STAs on this VAP*/
		SmpsManager_StaManagerTxStartReq(vapId);
		/*VAP can't not be removed during this processing so no need to store next before*/
		vapId = SmpsManager_VapDb[vapId].nextVap;
	}
}
	
/**********************************************************************************

smpsManagerVapOpen 


Description:
------------
SMPS VAP Open from VAP manager handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerVapOpen(K_MSG* smpsManagerMessage)
{
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	uint8 vapId; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(smpsManagerMessage)));
	UMI_SET_BSS* pSetBssMsg = (UMI_SET_BSS*) pK_MSG_DATA(vapMsg);

	vapId = pSetBssMsg->vapId;
	if (SmpsManager_VapDb[vapId].isActive == FALSE)
	{
		/*Increment number of VAPs*/
		SmpsManager_Db.numVaps++;
		SmpsManager_ClearVapEntry(vapId);
		/*Store flags in VAP DB*/
		SmpsManager_VapDb[vapId].flags = pSetBssMsg->flags;
		/*If VAP support static SM add it to list*/
		if (SmpsManager_VapSmps(vapId) == VAP_SMPS_MODE_STATIC)
		{
			SmpsManager_AddVapToList(vapId);
		}
		SmpsManager_VapDb[vapId].isActive = TRUE;		
		/*If this is first VAP, need to initialize number of antennas to max*/
		if (SmpsManager_Db.numVaps == 1)
		{
			SmpsManager_Db.numOfAntennas = SmpsManager_GetMaxAntennaNumber();
		}
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(smpsManagerMessage->abData));
	confirmEvent->vapId= vapId;
	confirmEvent->eventIndex = VAP_MANAGER_SET_BSS; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_SMPS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, smpsManagerMessage, vapId);
}

/**********************************************************************************

smpsManagerVapRemove 


Description:
------------
SMPS VAP Remove from VAP manager handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerVapRemove(K_MSG* smpsManagerMessage)
{
	uint8 vapId; 
	BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(smpsManagerMessage)));
	UMI_REMOVE_VAP* removeVapPtr = (UMI_REMOVE_VAP*) pK_MSG_DATA(vapMsg);
	
	vapId = removeVapPtr->vapId; 
	if (SmpsManager_VapDb[vapId].isActive == TRUE)
	{
		/*Decrement number of VAPs*/
		SmpsManager_Db.numVaps--;
		/*If VAP support static SM remove it from list*/
		if (SmpsManager_VapSmps(vapId) == VAP_SMPS_MODE_STATIC)
		{
			SmpsManager_RemoveVapFromList(vapId);
		}
		SmpsManager_ClearVapEntry(vapId);
		SmpsManager_VapDb[vapId].isActive = FALSE;
	}
	/*Send Confirmation to VAP manager*/
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT*)(&(smpsManagerMessage->abData));
	confirmEvent->vapId= vapId;
	confirmEvent->eventIndex = VAP_MANAGER_REMOVE_VAP; 
	confirmEvent->clientId = BSS_MANAGER_VAP_MANAGER_SMPS_MANAGER_CLIENT; 
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, smpsManagerMessage, vapId);
}

/**********************************************************************************

smpsManager_GetChannelWidth 


Description:
------------
Get the channelWidth information sent by the channel switch mannager and plant it in the SmpsManager_Db

global struct for future use.


Input: 
-----	

		
Output:
-------
	

Returns:
-------- 
	void - 
	
**********************************************************************************/
void smpsManager_GetChannelWidth(K_MSG* smpsManagerMessage)
{
	cocChannelWidth_t* cocChannelWidth_p = (cocChannelWidth_t*)pK_MSG_DATA(smpsManagerMessage); // SmpsManagerDb_p gets the message (channel width)

	SmpsManager_Db.channelWidth = cocChannelWidth_p->channelWidth; // move the channel width from message to global variable field
	OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_SET_NOTIFY_CFM,TASK_CHANNEL_SWITCH_MANAGER,smpsManagerMessage->header.vapId);
}

#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************

smpsManager_Init 


Description:
------------
SMPS Init


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManager_Init(void)
{
	uint8 i;

	SmpsManager_Db.vapHead = SMPS_MANAGER_INVALID_VAP;
	SmpsManager_Db.mode = SMPS_MODE_DISABLED;
	SmpsManager_Db.numOfAntennas = 0;
	SmpsManager_Db.numVaps = 0;
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		SmpsManager_VapDb[i].nextVap = SMPS_MANAGER_INVALID_VAP;
		SmpsManager_VapDb[i].prevVap = SMPS_MANAGER_INVALID_VAP;
		SmpsManager_VapDb[i].isActive = FALSE;
	}
	SmpsManager_StaManagerInit();
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif



