/***********************************************************************************
 File:			SmpsManager_StaManager.c
 Module:		SMPS Manager STA Manager
 Purpose: 		handle SMPS Manager STA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "SmpsManager_Api.h"
#include "SmpsManager.h"
#include "SmpsManager_StaManager.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_SMPS_MANAGER 
#define LOG_LOCAL_FID 0

/*To serialize Lock requests we reuse the PD request ID field*/
#define SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX				(INVALID_STA_INDEX)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
/*SMPS Manager STA Manager events*/
typedef enum
{
	SMPS_MANAGER_STA_MANAGER_ADD_STA,
	SMPS_MANAGER_STA_MANAGER_REMOVE_STA,
	SMPS_MANAGER_STA_MANAGER_START_TX_REQ,
	SMPS_MANAGER_STA_MANAGER_TX_CFM,
	SMPS_MANAGER_STA_MANAGER_PD_ALLOC,
	SMPS_MANAGER_STA_MANAGER_LOCK_CFM,
	SMPS_MANAGER_STA_MANAGER_NUM_EVENTS
} smpsManagerStaManager_ev;
	
/*SMPS Manager STA Manager states*/
typedef enum
{
	SMPS_MANAGER_STA_MANAGER_IDLE_STATE,
	SMPS_MANAGER_STA_MANAGER_OPEN_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE,
	SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE,
	SMPS_MANAGER_STA_MANAGER_NUM_STATES
} SmpsManagerStaManager_state;

typedef struct SmpsSManagerStaManagerDb_s
{
	K_MSG						*psMsg;
	TxPd_t						*pd;
	uint8						flags;
	uint8						sid;
	uint8						vapId;
	SmpsManagerStaManager_state	state;
	uint8						requestId;	/*From resource manager*/
	uint8						nextSid;
	uint8						prevSid;
	uint8						info;		/*This is the last mode / Antennas number we sent to the STA*/
} SmpsManagerStaManagerDb_t;

typedef struct SmpsManagerStaManagerVapDb_s
{
	StaId headSid;
	StaId tailSid;
} SmpsManagerStaManagerVapDb_t;

typedef struct SmpsManagerStaManagerGlobalDb_s
{
	uint8	lockHead;
	uint8 	lockTail;
	TxSelectorLockStatus_e lockStatus;	/*There can be only one pending lock request so we can store the status here*/
} SmpsManagerStaManagerGlobalDb_t;

typedef void (*SmpsManagerStaManager_Func)(SmpsManagerStaManagerDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/*Utility functions*/
static void SmpsManager_StaManagerChangeState(SmpsManagerStaManager_state state, SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerClearSta(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerAddStaToVap(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaFromVap(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerSendConfirm(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaDone(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerSendSmps(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerSendOmn(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerLockStaMng(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerHandlePendingLock(void);
static void SmpsManager_StaManagerUnlockStaMngQueue(SmpsManagerStaManagerDb_t *StaDb);
static bool SmpsManager_StaManagerRemovePd(SmpsManagerStaManagerDb_t *StaDb);
static bool SmpsManager_StaManagerCheckModeUpdate(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerTxStart(StaId sid);
/*Event handlers*/
static void SmpsManager_StaManagerAddStaInIdle(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaInOpen(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaInWaitPd(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerRemoveStaInWaitLock(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerStartTxInOpen(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerStartTxInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerStartTxInWaitPd(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerTxCfmInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerTxCfmInWaitTxCfmClose(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerTxCfmInWaitTxLockClose(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerPdAllocInWaitPd(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerPdAllocInWaitPdClose(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerLockCfmInWaitLock(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerLockCfmInWaitTxLockClose(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerLockCfmInWaitLockClose(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerIgnore(SmpsManagerStaManagerDb_t *StaDb);
static void SmpsManager_StaManagerFatal(SmpsManagerStaManagerDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*Smps Manager STA Manager Add STA Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerAddSta[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 SmpsManager_StaManagerAddStaInIdle, 			/* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	 SmpsManager_StaManagerFatal,					/* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
 	 SmpsManager_StaManagerFatal,					/* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	 SmpsManager_StaManagerFatal, 					/* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	 SmpsManager_StaManagerFatal, 					/* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	 SmpsManager_StaManagerFatal,					/* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	 SmpsManager_StaManagerFatal, 					/* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	 SmpsManager_StaManagerFatal,					/* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_LOCK_CLOSE_STATE */
	 SmpsManager_StaManagerFatal 					/* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

/*Smps Manager STA Manager Remove STA Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerRemoveSta[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SmpsManager_StaManagerFatal, 		  		   /* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	SmpsManager_StaManagerRemoveStaInOpen,		   /* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
	SmpsManager_StaManagerRemoveStaInWaitTxCfm,    /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	SmpsManager_StaManagerFatal,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	SmpsManager_StaManagerRemoveStaInWaitPd,	   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	SmpsManager_StaManagerFatal,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	SmpsManager_StaManagerRemoveStaInWaitLock,	   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	SmpsManager_StaManagerFatal,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE */
	SmpsManager_StaManagerFatal 		   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

/*Smps Manager STA Manager Start TX REQ Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerStartTxReq[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SmpsManager_StaManagerFatal, 		  		   /* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	SmpsManager_StaManagerStartTxInOpen,		   /* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
	SmpsManager_StaManagerStartTxInWaitTxCfm,      /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	SmpsManager_StaManagerIgnore,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	SmpsManager_StaManagerStartTxInWaitPd,	   	   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	SmpsManager_StaManagerIgnore,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	SmpsManager_StaManagerIgnore,	   			   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	SmpsManager_StaManagerIgnore,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE */
	SmpsManager_StaManagerIgnore 		   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

/*Smps Manager STA Manager TX CFM Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerTxCfm[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SmpsManager_StaManagerFatal, 		  		   /* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	SmpsManager_StaManagerFatal,		  		   /* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
	SmpsManager_StaManagerTxCfmInWaitTxCfm,        /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	SmpsManager_StaManagerTxCfmInWaitTxCfmClose,   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	SmpsManager_StaManagerFatal,	   	   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	SmpsManager_StaManagerFatal,				   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	SmpsManager_StaManagerIgnore,		   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	SmpsManager_StaManagerTxCfmInWaitTxLockClose,  /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE */
	SmpsManager_StaManagerFatal	  				   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

/*Smps Manager STA Manager PD Alloc Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerPdAlloc[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SmpsManager_StaManagerFatal, 		  		   /* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	SmpsManager_StaManagerFatal,		  		   /* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
	SmpsManager_StaManagerFatal,        		   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	SmpsManager_StaManagerFatal,   				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	SmpsManager_StaManagerPdAllocInWaitPd,	   	   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	SmpsManager_StaManagerPdAllocInWaitPdClose,    /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	SmpsManager_StaManagerFatal,		   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	SmpsManager_StaManagerFatal,  				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE */
	SmpsManager_StaManagerFatal	   				   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

/*Smps Manager STA Manager Lock CFM Handlers*/
static const SmpsManagerStaManager_Func SmpsManager_StaManagerLockCfm[SMPS_MANAGER_STA_MANAGER_NUM_STATES] =
{
	SmpsManager_StaManagerFatal, 		  		   /* SMPS_MANAGER_STA_MANAGER_IDLE_STATE */
	SmpsManager_StaManagerFatal,		  		   /* SMPS_MANAGER_STA_MANAGER_OPEN_STATE */
	SmpsManager_StaManagerFatal,        		   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE */
	SmpsManager_StaManagerFatal,   				   /* SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE */
	SmpsManager_StaManagerFatal,	   	   		   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE */
	SmpsManager_StaManagerFatal,    			   /* SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE */
	SmpsManager_StaManagerLockCfmInWaitLock,	   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE */
	SmpsManager_StaManagerLockCfmInWaitTxLockClose,/* SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE */
	SmpsManager_StaManagerLockCfmInWaitLockClose   /* SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE */
};

static SmpsManagerStaManagerDb_t SmpsManagerStaManager_Db[HW_NUM_OF_STATIONS];
static SmpsManagerStaManagerVapDb_t SmpsManagerStaManager_VapDb[HW_NUM_OF_VAPS];
static SmpsManagerStaManagerGlobalDb_t SmpsManagerStaManager_GlobalDb;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

SmpsManager_StaManagerChangeState 


Description:
------------
Utility function that changes STA state


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerChangeState(SmpsManagerStaManager_state state, SmpsManagerStaManagerDb_t *StaDb)
{
 	ILOG0_DDD("Smps Manager STA Manager, STA %d changed from %d to %d", StaDb->sid, StaDb->state, state);
	StaDb->state = state;
}

/**********************************************************************************

SmpsManager_StaManagerClearSta 


Description:
------------
Utility function that clear a STA entry


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerClearSta(SmpsManagerStaManagerDb_t *StaDb)
{
	StaDb->prevSid = INVALID_STA_INDEX;
	StaDb->nextSid = INVALID_STA_INDEX;
}

/**********************************************************************************

SmpsManager_StaManagerAddStaToVap 


Description:
------------
Utility function that adds a STA to the VAP list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerAddStaToVap(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 vapId = StaDb->vapId;

	/*First clear entry*/
	SmpsManager_StaManagerClearSta(StaDb);
	/*Add new to last*/
	if (SmpsManagerStaManager_VapDb[vapId].headSid == INVALID_STA_INDEX)
	{
		SmpsManagerStaManager_VapDb[vapId].headSid = StaDb->sid;
	}
	else
	{
		StaDb->prevSid = SmpsManagerStaManager_VapDb[vapId].tailSid;
		SmpsManagerStaManager_Db[SmpsManagerStaManager_VapDb[vapId].tailSid].nextSid = StaDb->sid;
	}
	SmpsManagerStaManager_VapDb[vapId].tailSid = StaDb->sid;
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaFromVap 


Description:
------------
Utility function that removes a STA from the VAP list


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaFromVap(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 vapId = StaDb->vapId;

	if (StaDb->prevSid == INVALID_STA_INDEX)
	{
		/*Entry is head*/
		SmpsManagerStaManager_VapDb[vapId].headSid = StaDb->nextSid;
	}
	else
	{
		SmpsManagerStaManager_Db[StaDb->prevSid].nextSid = StaDb->nextSid;
	}
	if (StaDb->nextSid == INVALID_STA_INDEX)
	{
		/*Entry is tail*/
		SmpsManagerStaManager_VapDb[vapId].tailSid = StaDb->prevSid;
	}
	else
	{
		SmpsManagerStaManager_Db[StaDb->nextSid].prevSid = StaDb->prevSid;
	}
	/*Now clear entry*/
	SmpsManager_StaManagerClearSta(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerSendConfirm 


Description:
------------
Utility function that sends confirmation to STA Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerSendConfirm(SmpsManagerStaManagerDb_t *StaDb)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_SMPS_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = StaDb->sid;
	/*Send confirmation message*/
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, StaDb->vapId);
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaDone 


Description:
------------
Utility function that does the necessary cleanup when STA remove is done


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaDone(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Clear message*/
	StaDb->psMsg = NULL;
	/*Set state to Idle*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_IDLE_STATE, StaDb); 
	/*Send Indication to Smps Manager*/
	SmpsManager_StaManagerRemoveStaFromVap(StaDb);
	/*Send confirmation to BSS Manager*/
	SmpsManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerSendSmps 


Description:
------------
Utility function that builds and sends the SMPS action frame


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerSendSmps(SmpsManagerStaManagerDb_t *StaDb)
{
	TxPd_t *pdPointer = (TxPd_t *)StaDb->pd;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	if (isStaQos == TRUE)
	{
		pdPointer->txQTid = IEEE802_1D_VO_2;
	}
	else
	{
		pdPointer->txQTid = IEEE802_1D_BE_1;
	}
	pdPointer->txQStaId= StaDb->sid;
	pdPointer->mgmtSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	pdPointer->mgmtActionCode = HT_CATEGORY;
	pdPointer->mgmtActionValue = HT_CATEGORY_SM_POWER_SAVE;
	/*Always unencrypted*/
	pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
	pdPointer->txQVapId = StaDb->vapId;
	pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = UNICAST;
	/*Build and sends Smps frame - sets also len in PD*/
	SmpsManager_BuildSmps(StaDb->pd);
	/*Change State to wait for TX CFM*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerSendOmn 


Description:
------------
Utility function that builds and sends the OMN action frame


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	 
	
**********************************************************************************/
static void SmpsManager_StaManagerSendOmn(SmpsManagerStaManagerDb_t *StaDb)
{
	TxPd_t *pdPointer = StaDb->pd;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	if (isStaQos == TRUE)
	{
		pdPointer->txQTid = IEEE802_1D_VO_2;
	}
	else
	{
		pdPointer->txQTid= IEEE802_1D_BE_1;
	}
	pdPointer->txQStaId = 			StaDb->sid;
	pdPointer->mgmtSubtype = 	MGMT_FRAME_SUBTYPE_ACTION;
	pdPointer->mgmtActionCode = 	VHT_CATEGORY; 
	pdPointer->mgmtActionValue = 	VHT_CATEGORY_OPERATING_MODE_NOTIFICATION;
	pdPointer->pdType = 			PD_TYPE_MANAGEMENT_UNENC;
	pdPointer->txQVapId = 			StaDb->vapId;
	pdPointer->txQGroupId = 		HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = 			UNICAST;
	SmpsManager_BuildOmn(StaDb->pd);
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_STATE, StaDb); 

}


/**********************************************************************************

SmpsManager_StaManagerLockStaMng 


Description:
------------
Utility function that locks the STA management queue in case we want to remove the DP from the queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerLockStaMng(SmpsManagerStaManagerDb_t *StaDb)
{
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos;

	/*Need to serialize lock requests*/
	/*first set next of entry to Invalid - use request ID - it is not used at this time*/
	StaDb->requestId = SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	if (SmpsManagerStaManager_GlobalDb.lockHead == SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		/*Set current entry as head*/
		SmpsManagerStaManager_GlobalDb.lockHead = StaDb->sid;
		isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
		/*Send lock request for STA Management queue*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to lock TID 7*/
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = SMPS_LOCK_CFM;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		if (isStaQos == TRUE)
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
		}
		else
		{
			/*For non-QoS everything is in Queue 0*/
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
		}
	}
	else
	{
		/*Link current entry to tail*/
		SmpsManagerStaManager_Db[SmpsManagerStaManager_GlobalDb.lockTail].requestId = StaDb->sid;
	}
	/*Set current entry as  tail*/
	SmpsManagerStaManager_GlobalDb.lockTail = StaDb->sid;
	/*Change State to Wait for Lock*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerHandleLock 


Description:
------------
Utility function that handles pending lock requests


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerHandlePendingLock()
{
	RequesterLockParams_t lockRequesterParams;
	uint8 isStaQos;
	SmpsManagerStaManagerDb_t *StaDb;

	/*get next entry*/
	SmpsManagerStaManager_GlobalDb.lockHead = SmpsManagerStaManager_Db[SmpsManagerStaManager_GlobalDb.lockHead].requestId;
	/*Do we have pending entry?*/
	if (SmpsManagerStaManager_GlobalDb.lockHead != SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX)
	{
		StaDb = &SmpsManagerStaManager_Db[SmpsManagerStaManager_GlobalDb.lockHead];
		isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
		/*Send lock request for STA Management queue*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to lock TID 7*/
		lockRequesterParams.callerContext = (void *)StaDb;
		lockRequesterParams.returnMsg = SMPS_LOCK_CFM;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		if (isStaQos == TRUE)
		{
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
		}
		else
		{
			/*For non-QoS everything is in Queue 0*/
			Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_BE_1, &lockRequesterParams);
		}
	}
	else
	{
		/*Set tail as invalid as well*/
		SmpsManagerStaManager_GlobalDb.lockTail = SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	}
}

/**********************************************************************************

SmpsManager_StaManagerUnlockStaMngQueue 


Description:
------------
Utility function that unlocks the STA management queue in case we want to remove teh PD from the queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerUnlockStaMngQueue(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);
	
	/*Unlock TID 7*/
	if (isStaQos == TRUE)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid, (0x1 << IEEE802_1D_VO_2));
	}
	else
	{
		/*In non-Qos everything goes to TID 0*/
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid, (0x1 << IEEE802_1D_BE_1));
	}
}

/**********************************************************************************

SmpsManager_StaManagerRemovePd


Description:
------------
Utility function that removes s SID from STA queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static bool SmpsManager_StaManagerRemovePd(SmpsManagerStaManagerDb_t *StaDb)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	bool found = FALSE;
	TxPd_t *prevPd = (TxPd_t *)NULL_PD;
	TxPd_t *pd;
	uint8 isStaQos = MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME);

	/*Flush Queue*/	
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	if (isStaQos == TRUE)
	{
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	}
	else
	{
		/*For non-QoS everything is on TID 0*/
		hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_BE_1;
	}
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	/*Search for PD in list*/
	if (hwQueueManagerRequestParams.pHeadDesc != (TxPd_t *)NULL_PD)
	{
		pd = hwQueueManagerRequestParams.pHeadDesc;
		while (pd != (TxPd_t *)NULL_PD)
		{
			if (pd == StaDb->pd)
			{
				/*found it!*/
				if (prevPd != (TxPd_t *)NULL_PD)
				{
					prevPd->nextPdPointer = SET_NEXT_PD(GET_NEXT_PD(pd));
				}
				else
				{
					/*PD was head of queue*/
					hwQueueManagerRequestParams.pHeadDesc = (void *)GET_NEXT_PD(pd);
				}
				if (pd->nextPdPointer == NEXT_PD_NULL)
				{
					/*PD was tail of queue*/
					hwQueueManagerRequestParams.pTailDesc = (void *)prevPd;
				}
				/*Restore DRAM pointer*/
				pd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(pd->packetPointer);  
				found = TRUE;
				break;
			}
			prevPd = pd;
			pd = (TxPd_t *)GET_NEXT_PD(pd);
		}
		/*Put List back to STA queue - we may have removed the only PD so check we have something in list*/
		if (hwQueueManagerRequestParams.pHeadDesc != (TxPd_t *)NULL_PD)
		{
			HwQManager_PushPacketlistToHead(&hwQueueManagerRequestParams); 
		}
	}
	return (found);
}

/**********************************************************************************

SmpsManager_StaManagerCheckModeUpdate



Description:
------------
Utility function that checks if we need to send an addditonal update


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static bool SmpsManager_StaManagerCheckModeUpdate(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Check mode*/	
	if ((MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT)) && (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_OMN)))
	{
		if (StaDb->info != SmpsManager_GetAntennaNumber())
		{	
			/*Set Antenna number to new Antenna number*/
			StaDb->info = SmpsManager_GetAntennaNumber();
			/*And transmit PD again - and stay in this state*/
			SmpsManager_StaManagerSendOmn(StaDb);
			/*Return that info was updated*/
			return (TRUE);
		}
	}
	else
	{
		if (StaDb->info != SmpsManager_GetMode())
		{
			/*Set mode to new mode*/
			StaDb->info = SmpsManager_GetMode();
			/*And transmit PD again - and stay in this state*/
			SmpsManager_StaManagerSendSmps(StaDb);
			/*Return that info was updated*/
			return (TRUE);
		}	
	}
	return (FALSE);
}

/**********************************************************************************

SmpsManager_StaManagerTxStart 


Description:
------------
Smps Start TX REQ from Channel Switch Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerTxStart(StaId sid)
{
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];
	
	/*Call handler*/
	SmpsManager_StaManagerStartTxReq[SmpsManagerStaManager_Db[sid].state](StaDb);
}


/**********************************************************************************

SmpsManager_StaManagerAddStaInIdle 


Description:
------------
STA ADD in Idle State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerAddStaInIdle(SmpsManagerStaManagerDb_t *StaDb)
{
	K_MSG *psMsg = StaDb->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);
	/*Store VAP*/
	StaDb->vapId = pAddSta->u8VapIndex;
	/*Clear message*/
	StaDb->psMsg = NULL;
							 
	/*Copy flags from message*/
	StaDb->flags = pAddSta->u8Flags;
	/*Set state to open*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
	/*Send confirmation to BSS Manager*/
	SmpsManager_StaManagerSendConfirm(StaDb);
	if ((MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT)) || (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_HT)))
	{
		/*Add STA to VAP linked list*/
		SmpsManager_StaManagerAddStaToVap(StaDb);
		/*If STA is VHT and supports OMN need to ckeck if current number of antennas is equal to what we report in Beacon*/
		if ((MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT)) && (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_OMN)))
		{
			/*Get current number of antennas*/
			StaDb->info = SmpsManager_GetAntennaNumber();
			/*Check if current number of antennas is equal to what we report in beacon*/
			if ((SmpsManager_VapSmps(StaDb->vapId) == VAP_SMPS_MODE_STATIC) && (StaDb->info != SmpsManager_GetMaxAntennaNumber()))
			{
				/*Current number of antennas is different that what we report in beacon, send OMN to STA*/
				SmpsManager_StaManagerStartTxInOpen(StaDb);
			}
		}
		else
		{
			/*STA is not VHT or does not support OMN*/
			/*Set mode to current mode*/
			StaDb->info = SmpsManager_GetMode();
			/*Check if we need to initiate transmission to STA*/
			if ((SmpsManager_VapSmps(StaDb->vapId) == VAP_SMPS_MODE_STATIC) &&	(StaDb->info == SMPS_MODE_STATIC))
			{
				SmpsManager_StaManagerStartTxInOpen(StaDb);
			}
		}
	}
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaInOpen 


Description:
------------
STA Remove in Open State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaInOpen(SmpsManagerStaManagerDb_t *StaDb)
{
	SmpsManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaInWaitTxCfm 


Description:
------------
STA Remove in Wait TX CFM State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Set state to wait TX CFM Close - no need top try and expedite removing the SMPS action frame from queue - 
	TX manager will flush all packets*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaInWaitPd 


Description:
------------
STA Remove in Wait PD State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaInWaitPd(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 removeStatus = FALSE;
	
	/*Tell Resource Manager we don't need the resource anymore*/
	removeStatus = ResourceManager_RemoveRequest(StaDb->requestId, DESC_POOL_MANAGEMENT_FROM_FW);
	if (removeStatus)
	{
		/*Request remove succesfully*/
		/*We are done*/
		SmpsManager_StaManagerRemoveStaDone(StaDb);
	}
	else
	{
		/*Request is on the way - wait for it - change state to wait PD Close State*/
		SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_PD_CLOSE_STATE, StaDb); 
	}	
}

/**********************************************************************************

SmpsManager_StaManagerRemoveStaInWaitLock 


Description:
------------
STA Remove in Wait LOCK State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerRemoveStaInWaitLock(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Set state to wait TX Lock Close - wa are waiting for two events TX CFM and Lock*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_TX_LOCK_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerStartTxInOpen 


Description:
------------
START TX In Open State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerStartTxInOpen(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	pdRequestFillParameters.returnTaskId = TASK_TX_MANAGER;
	pdRequestFillParameters.returnMsgType = SMPS_PD_ALLOC_CFM;
	pdRequestFillParameters.context = StaDb->sid;
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MANAGEMENT_FROM_FW, &pdRequestFillParameters);
	if (allocationStatus)
	{								  
		/*Store PD pointer*/
		StaDb->pd = pdRequestFillParameters.packetDescriptor;
		if ((MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT)) && (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_OMN)))
		{
			/* info field is the number of antennas */
			StaDb->info = SmpsManager_GetAntennaNumber();
			/*Build and send OMN*/					
			SmpsManager_StaManagerSendOmn(StaDb);
		}
		else 
		{
			/* info field is the operation mode */
			StaDb->info = SmpsManager_GetMode(); 
			/*Build and send SMPS*/					
			SmpsManager_StaManagerSendSmps(StaDb);
		}
	}
	else
	{
		/* PD allocation falied */		
		StaDb->requestId = pdRequestFillParameters.requestId;
		/*Change State to wait for PD*/
		SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_PD_STATE, StaDb); 
	}
}

/**********************************************************************************

SmpsManager_StaManagerStartTxInWaitTxCfm 


Description:
------------
START TX In Wait TX CFM State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerStartTxInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Check mode change*/	
	if (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT))
	{
		if (StaDb->info != SmpsManager_GetAntennaNumber())
		{
			/*Current mode is different from the one we are trying to send to the STA*/
			/*Try to remove PD from queue*/
			SmpsManager_StaManagerLockStaMng(StaDb);
		}
	}
	else if (StaDb->info != SmpsManager_GetMode())
	{
		/*Current mode is different from the one we are trying to send to the STA*/
		/*Try to remove PD from queue*/
		SmpsManager_StaManagerLockStaMng(StaDb);
	}

}

/**********************************************************************************

SmpsManager_StaManagerStartTxInWaitPd 


Description:
------------
START TX In Wait PD State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerStartTxInWaitPd(SmpsManagerStaManagerDb_t *StaDb)
{
	uint8 removeStatus = FALSE;
		
	/*Check mode*/	
	if (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_IS_VHT))
	{
		if (StaDb->info == SmpsManager_GetAntennaNumber())
		{
			/*If same number of antennas tell Resource Manager we don't need the resource anymore*/
			removeStatus = ResourceManager_RemoveRequest(StaDb->requestId, DESC_POOL_MANAGEMENT_FROM_FW);
			if (removeStatus)
			{
				/*Request remove succesfully*/
				/*Change State to OPEN*/
				SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
			}
			/*resource is on its way - check mode when PD gets here*/
		}
	}
	else if (StaDb->info == SmpsManager_GetMode())
	{
		/*If same mode tell Resource Manager we don't need the resource anymore*/
		removeStatus = ResourceManager_RemoveRequest(StaDb->requestId, DESC_POOL_MANAGEMENT_FROM_FW);
		if (removeStatus)
		{
			/*Request remove succesfully*/
			/*Change State to OPEN*/
			SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
		}
		/*resource is on its way - check mode when PD gets here*/
	}
}

/**********************************************************************************

SmpsManager_StaManagerTxCfmInWaitTxCfm 


Description:
------------
TX CFM In Wait TX CFM State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerTxCfmInWaitTxCfm(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Check if we need to send another update
	needed because of the following flow
	TX CFM state -> TX start with mode change -> Lock REQ -> Lock CFM -> PD not found -> TX CFM state */
	if (SmpsManager_StaManagerCheckModeUpdate(StaDb) == FALSE)
	{
		/*Release the PD*/
		ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
		/*Change State to Open*/
		SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
	}	
}

/**********************************************************************************

SmpsManager_StaManagerTxCfmInWaitTxCfmClose 


Description:
------------
TX CFM In Wait TX CFM Close State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerTxCfmInWaitTxCfmClose(SmpsManagerStaManagerDb_t *StaDb)
{
    /* Release the PD */
    ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
	/*We are done*/
	SmpsManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerTxCfmInWaitTxLockClose 


Description:
------------
TX CFM In Wait TX LOCK Close State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerTxCfmInWaitTxLockClose(SmpsManagerStaManagerDb_t *StaDb)
{
	/*In this state we are waiting for TX CFM AND LOCK CFM before removing STA*/
	/*TX CFM received first*/
    /* Release the PD */
    ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
	/*Change State - now we are waiting only for LOCK CFM*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_LOCK_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerPdAllocInWaitPd 


Description:
------------
PD Alloc In Wait PD State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerPdAllocInWaitPd(SmpsManagerStaManagerDb_t *StaDb)
{
	if (SmpsManager_StaManagerCheckModeUpdate(StaDb) == FALSE)
	{
		/*Release the PD*/
		ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
		/*Change State to Open*/
		SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_OPEN_STATE, StaDb); 
	}
}

/**********************************************************************************

SmpsManager_StaManagerPdAllocInWaitPdClose 


Description:
------------
PD Alloc In Wait PD Close State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerPdAllocInWaitPdClose(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Release the PD*/
	ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
	/*We are done*/
	SmpsManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerLockCfmInWaitLock 


Description:
------------
LOCK CFM In Wait LOCK State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerLockCfmInWaitLock(SmpsManagerStaManagerDb_t *StaDb)
{
	DEBUG_ASSERT(SmpsManagerStaManager_GlobalDb.lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED);
	/*First Look for the PD*/
	if (SmpsManager_StaManagerRemovePd(StaDb) == TRUE)
	{
		/*PD found - check if we still need to transmit*/
		if (SmpsManager_StaManagerCheckModeUpdate(StaDb) == FALSE)
		{
			/*We don't need to transmit - Release the PD*/
			ResourceManager_ReleaseDescriptor(StaDb->pd, DESC_POOL_MANAGEMENT_FROM_FW);
		}
	}
	/*Unlock queue*/
	SmpsManager_StaManagerUnlockStaMngQueue(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerLockCfmInWaitTxLockClose 


Description:
------------
LOCK CFM In Wait TX LOCK Close State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerLockCfmInWaitTxLockClose(SmpsManagerStaManagerDb_t *StaDb)
{
	/*In this state we are waiting for TX CFM AND LOCK CFM before removing STA*/
	/*LOCK CFM received first*/
	/*Unlock queue*/
	/*Queue lock may fail if queue is disabled - this can happen if queue was disabled during stop traffic procedure*/
	if (SmpsManagerStaManager_GlobalDb.lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		SmpsManager_StaManagerUnlockStaMngQueue(StaDb);
	}
	/*Now wait for TX CFM*/
	SmpsManager_StaManagerChangeState(SMPS_MANAGER_STA_MANAGER_WAIT_TX_CFM_CLOSE_STATE, StaDb); 
}

/**********************************************************************************

SmpsManager_StaManagerLockCfmInWaitLockClose 


Description:
------------
LOCK CFM In Wait LOCK Close State Handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerLockCfmInWaitLockClose(SmpsManagerStaManagerDb_t *StaDb)
{
	/*Unlock queue*/
	/*Queue lock may fail if queue is disabled - this can happen if queue was disabled during stop traffic procedure*/
	if (SmpsManagerStaManager_GlobalDb.lockStatus == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		SmpsManager_StaManagerUnlockStaMngQueue(StaDb);
	}
	/*We are done*/
	SmpsManager_StaManagerRemoveStaDone(StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerIgnore 


Description:
------------
Ignore event handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerIgnore(SmpsManagerStaManagerDb_t *StaDb)
{
}

/**********************************************************************************

SmpsManager_StaManagerFatal 


Description:
------------
Fatal event handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void SmpsManager_StaManagerFatal(SmpsManagerStaManagerDb_t *StaDb)
{
	FATAL(0);
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

smpsManagerStaOpen 


Description:
------------
STA ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerStaOpen(K_MSG* SmpsManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(SmpsManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);
	StaId sid = pAddSta->u16SID;
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];

	DEBUG_ASSERT(StaDb->psMsg == NULL);
	/*Store message*/
	StaDb->psMsg = psMsg;
	/*Call handler*/
	SmpsManager_StaManagerAddSta[StaDb->state](StaDb);
}

/**********************************************************************************

smpsManagerStaOpen 


Description:
------------
STA Remove message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerStaRemove(K_MSG* SmpsManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(SmpsManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pRemoveSta = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(psMsg);
	StaId sid = pRemoveSta->u16SID;
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];

	DEBUG_ASSERT(StaDb->psMsg == NULL);
	/*Store message*/
	StaDb->psMsg = psMsg;
	/*Call handler*/
	SmpsManager_StaManagerRemoveSta[StaDb->state](StaDb);
}

/**********************************************************************************

smpsManagerLockCfm






Description:
------------
Lock Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerLockCfm(K_MSG* SmpsManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(SmpsManagerMessage);
	StaId sid = lockReg->stationOrVapNum;
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];

	/*Store the status*/
	SmpsManagerStaManager_GlobalDb.lockStatus = lockReg->lockStatus;
	/*Handle pending lock requests first*/
	SmpsManager_StaManagerHandlePendingLock();
	/*Call handler*/
	SmpsManager_StaManagerLockCfm[SmpsManagerStaManager_Db[sid].state](StaDb);
}

/**********************************************************************************

smpsManagerTxCfm






Description:
------------
TX Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerTxCfm(K_MSG* SmpsManagerMessage)
{
	/*From PD get SID*/
	SmpsManagerPacketConfirmedMessage_t *SmpsMsg = (SmpsManagerPacketConfirmedMessage_t *)pK_MSG_DATA(SmpsManagerMessage);
	TxPd_t *pd = SmpsMsg->pd;
	StaId sid = pd->txQStaId;
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];

	/*Call handler*/
	SmpsManager_StaManagerTxCfm[SmpsManagerStaManager_Db[sid].state](StaDb);
}

/**********************************************************************************

smpsManagerPdAlloc







Description:
------------
PD Alloc Confirm message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void smpsManagerPdAlloc(K_MSG* SmpsManagerMessage)
{
	RmPdFreeDescResponse_t *pdMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(SmpsManagerMessage);
	StaId sid = pdMsg->context;
	SmpsManagerStaManagerDb_t *StaDb = &SmpsManagerStaManager_Db[sid];

	/*Store PD pointer*/
	StaDb->pd = pdMsg->packetDescriptor;
	/*Release the request*/
	ResourceManager_ReleaseRequest(StaDb->requestId, DESC_POOL_MANAGEMENT_FROM_FW);
	/*Call handler*/
	SmpsManager_StaManagerPdAlloc[SmpsManagerStaManager_Db[sid].state](StaDb);
}

/**********************************************************************************

SmpsManager_StaManagerTxStartReq







Description:
------------
Called by SMPS manager when CoC send SM mode update


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void SmpsManager_StaManagerTxStartReq(uint8 vapId)
{
	StaId sid = SmpsManagerStaManager_VapDb[vapId].headSid;

	while (sid != INVALID_STA_INDEX)
	{
		/*Send event to STA state machine*/
		SmpsManager_StaManagerTxStart(sid);
		/*TX REQ can not cause STA to be removed so no need to store next before*/
		sid = SmpsManagerStaManager_Db[sid].nextSid;
	}
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif


/**********************************************************************************

SmpsManager_Init



Description:
------------
Initiailize Smps MAnager VAP Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void SmpsManager_StaManagerInit()
{
	uint8 i;
	
	SmpsManagerStaManager_GlobalDb.lockHead = SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	SmpsManagerStaManager_GlobalDb.lockTail = SMPS_MANAGER_STA_MANAGER_INVALID_LOCK_IDX;
	/*Intiialize Databases*/
	for (i = 0; i < HW_NUM_OF_VAPS; i++)
	{
		SmpsManagerStaManager_VapDb[i].headSid = INVALID_STA_INDEX;
		SmpsManagerStaManager_VapDb[i].tailSid = INVALID_STA_INDEX;
	}
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		SmpsManagerStaManager_Db[i].sid = i;
		SmpsManagerStaManager_Db[i].state = SMPS_MANAGER_STA_MANAGER_IDLE_STATE;
		SmpsManagerStaManager_Db[i].pd = (TxPd_t *)NULL_PD;
		SmpsManagerStaManager_Db[i].psMsg = NULL;
		SmpsManagerStaManager_Db[i].prevSid = INVALID_STA_INDEX;
		SmpsManagerStaManager_Db[i].nextSid = INVALID_STA_INDEX;
		SmpsManagerStaManager_Db[i].info = SMPS_MODE_DISABLED;
	}
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif





