/***************************************************************
 File:		  StaDatabase_Api.h
 Module:	  	  StaDatabase
 Purpose: 	 
 Description:     This module is the API to the Station database module. Calls that access station database parameters 
 			  should be made through the api function located in this file. Direct access(Not through the api) to the
 			  station database parameters 
 			  is not permitted.
***************************************************************/
#ifndef StaDatabase_Api_H_
#define StaDatabase_Api_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------

#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"

#include "mhi_ieee_address.h"
#include "database.h"
#include "SharedDbTypes.h"
#include "ShramSecurity.h"
#include "StaDb.h"
#include "Std.h"



//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Defines 					
//---------------------------------------------------------------------------------

typedef enum StaState
{
	STA_STATE_FREE = 0,
	STA_STATE_SID_ALLOCATED,
	STA_STATE_CONNECTED,
	STA_STATE_NUM_OF_STATES,
	STA_STATE_MAX_STATE,
}	StaState_e;

/* Buffer size of 64 should be changed to 0 for the HW */

#ifdef ENET_INC_ARCH_WAVE600
#define STATION_DATABASE_BUFFER_SIZE_MASK 0xFF 
#else
#define STATION_DATABASE_BUFFER_SIZE_MASK 0x3F 
#endif
#define STATION_DATABASE_TX_MODE_AMSDU_ENABLE_SHIFT 1


#define MAX_SP_LENGTH_UNLIMITED 	(0xFF) // setting maximun value to 255 won't affect aggregation builder logic

#define MAX_AMPDU_LENGTH_EXP_VHT	(7)

#ifdef ENET_INC_ARCH_WAVE600
#define MANAGEMENT_TID_IN_STA_DB	(8)
#endif
//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------


typedef struct HtCapabilitiesInfo
{
    uint16  ldpcCodingCapability         :1; // The advanced coding capability field indicates support for advanced coding if its value is 1
    uint16  supportedChannelWidthSet     :1; // The supported channel width set indicates {20} if its value is 0, and {20, 40} if its value is 1. 
    uint16  smPowerSave                  :2; // The dynamic MIMO Power Save mode indicates that STA or AP asks for assistance to switch to 
    // MIMO from being in steady state of single Rx chain. Such a STA or AP needs that any MIMO 
    // sequence shall be preceded by RTS/CTS. The static MIMO power save mode indicates that station or
    // AP ask not to send MIMO sequences. 
    uint16  greenField                   :1; // The GF field indicates ability to receive frames with Green Field preamble. 
    uint16  shortGiFor20Mhz              :1; // The Short GI for 20 MHz field indicates support for 20 MHz packets if its value is 1
    uint16  shortGiFor40Mhz              :1; // The Short GI for 40 MHz field indicates support for 40 MHz packets if its value is 1. 
    uint16  txStbc                       :1; // The Rx STBC field indicates support STBC by transmitter.
    uint16  rxStbc                       :2; // The Rx STBC field indicates support STBC by receiver:
    // 0 = No Rx STBC Support
    // 1 = Rx support of one spatial stream
    // 2 = Rx support of one and two spatial streams
    // 3 = Rx support of one, two and three spatial streams
    uint16  htDelayedBa                  :1; // Delayed BA  0 not supported, 1 supported, should be agreed from both sides. 
    uint16  maximalAmsduSize             :1; // Maximal A-MSDU size  0- 3839 bytes, 1- 7935 bytes 
    uint16  dsssCckModeAt40MhzMode       :1; // DSSS/CCK rates in 40 MHz capable BSS: If this field is 0 in the beacon and probe response then in 
    // this BSS only HTI compatible Stations shall be associated, and 11g compatible devices shall not be
    // associated.  The beacon shall be sent under OFDM basic rate. If this field is 1 the 11g legacy devices
    // may be associated. The HTI station with this field 1 in association request shall be not associated with
    ///@todo This bit should affect all 11g stuff !!!
    // BSS that contains 0 in beacon or probe response.
    uint16  reserved					 :1; // The PSMP support field indicates support if set to 1. 
    uint16  fortyMhzIntolerant			 :1; // The STBC Control Frame support field indicates that STBC control frames are supported if set to 1.
    uint16  lSigTxopProtectionSupport    :1; // The L-SIG TXOP Protection support field indicates whether the HT QAP and the HT Station is 
    // capable of supporting the L-SIG field on specific frames which contain duration longer than the
    // remaining packet duration
} HtCapabilitiesInfo_t;


typedef struct AmpduParameters
{
	uint8 maximumAmpdulengthExponent :2; // b0 -b1  Maximum A-MPDU length exponent. integer in the 0-3. The length is 2 ^ (13 + field)  -1 octet
	uint8 minimumMpduStartSpacing    :3; // b2-b4	Minimum MPDU start spacing. 0 -no restriction. 1- 1/4 usec, 2- 1/2 usec , ... 7 -16usec
	uint8 reserved					 :3; 
} AmpduParameters_t;


typedef struct VHTMcsNssSet
{
	uint32 rxVhtMcsMap 							:16; 	// b0  -b15 	Max value of MCS that can be received at all channel widths for each number of spatial streams.
	uint32 rxHighestSupportedLongGIDataRate		:13; 	// b16-b28	Highest long GI VHT PPDU data rate that the STA is able to receive (in Mbit/Sec)
	uint32 reserved0							:3;		// b29-b31
	uint32 txVhtMcsMap 							:16; 	// b32-b47	Max value of MCS that can be transmitted at all channel widths for each number of spatial streams.
	uint32 txHighestSupportedLongGIDataRate		:13; 	// b48-b60	Highest long GI VHT PPDU data rate that the STA is able to transmit (in Mbit/Sec)
	uint32 reserved1							:3; 	// b61-b63
} VHTMcsNssSet_t;
 


	
typedef struct txBfCapabilities
{
	uint32 implicitTBfRxCapable 	:1;						//          B0       
	uint32 rxStaggeredSounding 		:1;					    //          B1       
	uint32 txStaggeredSounding 		:1;					    //          B2      
	uint32 rxNdpCapable 			:1;						//          B3      
	uint32 txNdpCapable 			:1;						//          B4      
	uint32 implicitTxBfCapable      :1;						//          B5       
	uint32 calibration 				:2;					    //          B6-B7    
	uint32 explicitCsiTxBf 			:1;						//          B8       
	uint32 explicitNonCompressedSteeringCapable 	   :1;  //          B9       
	uint32 explicitCompressedSteeringCapable	   	   :1; 	//          B10
	uint32 explicitTxBfCSIFeedback                     :2;  //          B11-B12  	
	uint32 explicitNonCompressedBfFeedbackCapable      :2;	//          B13-B14  
	uint32 explicitCompressedBfFeedbackCapable         :2;	//          B15-B16  
	uint32 minimalGrouping 							   :2;	//          B17-B18 
	uint32 csiNumberOfBfAntennasSupported 			   :2;	//	      B19-B20  
	uint32 uncompressedSteeringNumOfAntSupported       :2;	//          B21-B22  
	uint32 compressedSteeringNumOfAntSupported		   :2;	//          B23-B24  
	uint32 csiMaxNumberOfRowsBeamFormerSupported 	   :2;	//          B25-B26 
	uint32 channelEstimationCapability 				   :2;	//	      B27-B28 
	uint32 reserved 								   :3;	//	      B29-B31 Reserved
}	TxBfCapabilities_t;


typedef struct Coexistance2040Coex
{
	uint8 informationRequest			:1; //according to 7.3.2.60 in 802.11n-2009
	uint8 fortyMHzIntolerant			:1;
	uint8 twentyMHzBssWidthRequest		:1;
	uint8 obssScanningExemptionRequest	:1;
	uint8 obssScanningExemptionGrant	:1;
	uint8 reserved						:3;
}	Coexistance2040Coex_t;

// Parameters which are taken from the old STA DB in order to be copied to WAVE500 STADB
typedef struct AssocParams
{	
	uint32 supportedRates; 			            /* WARNING - needs to be changed- supported rates - in AR10 style- needs to be removed */  
} AssociateParameters_t;


typedef struct StaBaParameters
{
	uint8      winSize;
} StaBaParameters_t;


/* STA Database Software Struct contains parameters accesible only by software */
typedef struct StaDbSwInfo
{
	/* General parameters */
	IEEE_ADDR	macAddress;
	StaState_e  state; // used by the rx manager to know the allowed frame calass of the sta
	uint8       vapIndex;

	/* Each STA is a part of a double linked list of all STAs in the same VAP*/
	StaId	   nextSid; 			
	StaId	   previousSid;
}StaDbSwInfo_t;

typedef enum StaDbPpduTxMode
{
	STA_DB_PPDU_TX_MODE_LEGACY_ONLY = 0,
	STA_DB_PPDU_TX_MODE_LEGACY_AND_MPDU = 1,
	STA_DB_PPDU_TX_MODE_LEGACY_AND_AMSDU_NO_AMPDU = 2,
	STA_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR = 3,
}	StaDbPpduTxMode_e;

typedef struct StaTxTsParameters
{
	uint8      baEnable;
	uint8      ppduTxMode;
	uint8      winSize;
} StaTxTsParameters_t;

typedef struct RssiParams
{
	uint8 au8Addr[IEEE_ADDR_LEN]; /* WARNING: Special case! No padding here! This structure must be padded externally! */
	uint8 vapId;
	uint8 reserved[2];
}RssiParams_t;




#ifdef ENET_INC_ARCH_WAVE600D2

typedef struct StaHeErParameters
{
	HeSuStaResponseDurTimeout_e     heSuStaResponseMode;
	HeSuPhyModeBaAckRespType_e      heSuPhyModeBaAckRespType;
	HeSuErPhyModeBaAckRespType_e    heSuErPhyModeBaAckRespType;
} StaHeErParameters_t;


#endif //ENET_INC_ARCH_WAVE600D2



//---------------------------------------------------------------------------------
//						Globals
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Public Function Declaration 			
//---------------------------------------------------------------------------------

void	 StaDB_GetHTCapabilities(StaId staId, HtCapabilitiesInfo_t htCapabilities);
void 	 StaDB_SetHTCapabilities(StaId staId, HtCapabilitiesInfo_t htCapabilities);
#if defined (ENET_INC_UMAC)
void 	 StaDB_SetSecurityKey(UMI_SET_KEY *pSetKey, bool rekeyDisableFlag);
void 	 StaDB_SetTxPn(StaId stationId, uint8 *pnAddress);
#endif /* #ifdef ENET_INC_UMAC */
void 	 StaDB_SetDefaultKeyIndex(StaId staId, uint8 defaultKeyId, uint8 setForPeerAp);
void 	 StaDB_SetWepSecurityKeyIndex(StaId staId, uint8 keyIndex, uint32 keyIndexValue);
void	 StaDB_ClearSecurityKey(StaId staId, uint8 vapId);
void 	 StaDB_SetWepSecurityKey(StaId staId, uint8 vapId, uint8 cipherSuite);
#ifdef ENET_INC_ARCH_WAVE600
void StaDB_SetTidRRweight(StaId staId, uint8 tidNum, uint8 tidRrWeight);
void StaDB_SetHeMngAgg(StaId staId, MultiTidMngAgg_e setMngAgg);
void StaDB_SetHeBarAgg(StaId staId, bool setResetVal);
void StaDB_InitHeSuMultiTid(StaId staId, uint8 maxTids, uint8 ackEnabledSupport);
void StaDB_SetHeMultiTidSelectPolicy(StaId staId, MultiTidSelectPolicy_e multiTidSelectPolicy);
void StaDB_HeMultiTidAddTid(StaId staId, uint8 TidNum);
void StaDB_HeMultiTidRemoveTid(StaId staId, uint8 TidNum);
#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A after Jira WLANVLSIIP-2881 is resolved in B0
void  StaDB_heMultiTidStationAcPsTypesAreMixed(StaId staId, uint8 acPsTypeBitmap);
#endif //PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE
void     StaDB_OpenBaAgreement(StaId stationIndex, uint8 tid, uint16 bufferSize, uint8 amsduSupport);
#else
void     StaDB_OpenBaAgreement(StaId stationIndex, uint8 tid, uint8 bufferSize, uint8 amsduSupport);
#endif //ENET_INC_ARCH_WAVE600
void     StaDB_CloseBaAgreement(StaId stationIndex, uint8 tid, uint8 changePpduMode);
void StaDB_ChangePpduTxMode(StaId stationIndex, uint8 tid, uint8 PpduTxMode);
void     StaDB_UpdateLowSequenceNumber(StaId stationIndex, uint8 tid, uint16 newSequenceNumber);
void     StaDB_UpdateLowSequenceNumberFromCurrent(StaId stationIndex, uint8 tid);
#if defined(ENET_INC_ARCH_WAVE600)
void StaDB_UpdateLowSequenceNumberFromCurrentMngTid(StaId stationIndex);
#endif
StaDbPpduTxMode_e StaDb_GetMinPpduTxModePerStation(StaId stationIndex);

void	 StaDB_AddStaToDB(UMI_STA_ADD* pStaAdd);
void     StaDb_RemoveStationFromDB(StaId staId);
void 	 StaDb_GetVapId(StaId StaId, uint8* VapId);
StaId   StaDb_GetNextSid(StaId StaId);
void 	 StaDb_SetNextSid(StaId staId, StaId newNextSid);
StaId   StaDb_GetPreviousSid(StaId staId);
void 	 StaDb_SetPreviousSid(StaId staId, StaId newPreviousSid);
void 	 StaDB_SetSpLengthToMaximum(StaId stationId);
void 	 StaDB_GetMacAddress(StaId StaId, IEEE_ADDR* pMacAddress);
IEEE_ADDR* StaDB_GetMacAddressPtr(StaId StaId);
void 	 StaDB_SetMacAddress(StaId StaId, IEEE_ADDR* pMacAddress);


void	 StaDb_Init(void);
void 	 StaDbGetHTCapabilities(StaId StaId, HtCapabilitiesInfo_t* pHtCapabilities);
void 	 StaDbSetHTCapabilities(StaId StaId, HtCapabilitiesInfo_t* pHtCapabilities);
bool	 StaDbIsHtSupported(StaId StaId);
uint16   StaDB_GetBaWinSize(StaId staId, uint8 tid);
uint16   StaDB_GetBaWinSsn(StaId staId, uint8 tid);
void     StaDB_SetBaWinSsn(StaId staId, uint8 tid,uint16 winSsn);
uint8    StaDB_GetQosEnable(StaId staId);
void 	 StaDB_SetManagementEncrypted(StaId staId);
uint8    StaDB_IsManagementEncrypted(StaId staId);
uint8 	 StaDB_IsDataEncrypted(StaId staId);
void 	 StaDB_SetDataEncryptionEnable(StaId staId, bool dataEncryptionEnable);
uint8    StaDB_GetNegotiatedWds(StaId staId);
void     StaDB_SetNegotiatedWds(StaId staId, bool wds);
uint8    StaDB_GetWdsClientType(StaId staId);
uint8    StaDB_IsStationConnected(StaId staId);
uint8    StaDB_GetDataEncryptionType(StaId staId);
void 	 StaDB_SetDataEncryptionType(StaId staId, uint8 dataEncryptionType);
uint8 	 StaDB_GetManagementEncryptionType(StaId staId);
uint8	 StaDB_GetVapRtsCtsMethod(StaId staId);
StaDbTid_t* StaDB_GetTidParams(StaId staId , uint8 tid);
#ifdef ENET_INC_ARCH_WAVE600D2
void StaDB_SetMinHeFragLevel(StaId staId, uint8 heFragLevel, uint8 staTid);
void StaDB_SetHeSuExtParams(StaId staId, uint8 isHeExtSuDisable);
#elif defined (ENET_INC_ARCH_WAVE600)
void StaDB_SetMinHeFragLevel(StaId staId, uint8 heFragLevel);
#endif
uint32 StaDB_GetMaxPsduLimit(StaId staId);

uint16 StaDB_GetCurrentSpLengthLimit(StaId staId);
uint8 taDB_GetCurrentSpLengthLimit(StaId staId);
void StaDB_SetCurrentSpLengthLimit(StaId staId , uint8 value);
uint8 StaDB_GetNoAggInPs(StaId staId);
void StaDB_SetNoAggInPs(StaId staId, bool enable);
void StaDB_SetPsForceOneNss(StaId staId, bool enable);
#ifdef ENET_INC_ARCH_WAVE600
void StaDB_debug_SetBF(StaId staID, uint8 BFtype);
MultiTidAckEn_e	StaDb_GetStationAckEnabledSupport(StaId stationIndex);
#endif //ENET_INC_ARCH_WAVE600
#ifdef ENET_INC_ARCH_WAVE600B
void StaDB_debug_TriggerTomi(StaId staID,uint8 OMCtrl_RxNSS,uint8 OMCtrl_ChnlWidth);
#endif //ENET_INC_ARCH_WAVE600B
void StaDB_debug_SetBWlimit(StaId staID, uint8 dataLimit, uint8 managementLimit);

StaState_e StaDb_getStaState(StaId stationId);


uint8 StaDb_getNumOfConnectedStations(void);
void StaDb_setTxopMode(StaId stationIndex, uint8 mode);
uint8 StaDb_getTxopMode(StaId stationIndex);

void staDbCopyIeeeAddrToRaType(IEEE_ADDR* pMacAddress, StaDbRa_t* pRaMacAddress);
void staDbBssLoadIeAddSta(uint8 vapId);
void staDbBssLoadIeRemoveSta(uint8 vapId);
void StaDb_SetPsduTimeLimit(StaId stationIndex ,  uint32 maxPsduTransmissionTimeLimit);
void StaDB_SetStationAmpduLimit(StaId stationIndex, uint8 numOfMpdus);
void StaDb_setStaStat(StaId stationId, StaState_e state);
uint8 StaDb_getStaSupportedNss(StaId stationId); 
void StaDB_SetPrimaryStationSuTxopAndNumMpdus(StaId stationIndex, uint8 txopMode, uint8 numOfMpdus);
uint8 StaDB_CalcAmpduLimit (uint8 maxAmpduSize);
uint8 StaDB_SearchMaxWindowSizeOutOfTids (StaId stationIndex);
void StaDb_SetMaxStationTidLimitInAggregation(StaId stationIndex, uint8 maxTids);
uint8 StaDb_GetMaxStationTidLimitInAggregation(StaId stationIndex);
#endif //  StaDatabase_Api_H_
	

