/***************************************************************
 File:		  StaDatabase.c
 Module:	  	  Station Database
 Purpose: 	 
 Description:     This module is the implementation of the Station database module. Calls that access station database parameters 
 			  should be made through the api function located in this file. Direct access (Not through the api) to the
 			  station database parameters is not permitted.
***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "StaDatabase_Api.h"
#if defined (ENET_INC_UMAC)
#include "TxSelector_Api.h"
#include "RxPp_Api.h"
#endif //ENET_INC_UMAC
#include "VapDatabase_Api.h"
#include "StaDb.h"
#include "stringLibApi.h"
#include "TxSelector_Api.h"
#include "HwBeamforming_Api.h"
#include "ShramStationDatabase.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "RxManager_Api.h"
#include "Pac_Api.h"
#include "shramTxDesc.h"
#include "BaAnalyzer_Api.h"
#include "HwSemaphore_API.h"
#include "HwSemaphore_API.h"
#include "GroupManager_API.h"
#if defined(ENET_INC_ARCH_WAVE600)
#include "secManager_StaManager.h"
#endif
#include "TsManagerCommonDefinitions.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_STATION_DB
#define LOG_LOCAL_FID 0

// According to current System requirements - by default, MNG aggregation is disabled, remove this if decided differently
#define HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED
#define SU_HE_UNLIMITED_NUM_OF_MPDUS_IN_A_MPDU 0
#define MIN_NUM_OF_MPDUS_IN_A_MPDU 1
#define SU_HE_MAX_NUM_OF_MPDUS_IN_A_MPDU 196 // Take 3/4*256(max value)
#define VHT_MAX_NUM_OF_MPDUS_IN_A_MPDU 48 // Take 3/4*256(max value)

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

typedef struct StaDbObject
{
	uint8  		NumOfConnectedSta;
	StaDbSwInfo_t StaDbSwEntries[HW_NUM_OF_STATIONS];
	StaDb_t* pStaDbHwEntries; // [HW_NUM_OF_STATIONS];
} StaDbObject_t;

//---------------------------------------------------------------------------------
//						Globals					
//---------------------------------------------------------------------------------

StaDbObject_t	StationDatabase;
const uint8 au8convertMaxVhtMcsToMcsValue[NUM_OF_MAX_MCS_VALUES] =
{
	MCS_7,
	MCS_8,
	MCS_9
};

uint8 StaDb_TxopMode[HW_NUM_OF_STATIONS] = {UMI_TXOP_MODE_DISABLED};

#ifdef ENET_INC_ARCH_WAVE600 
MultiTidAckEn_e gackEnabledSupport[HW_NUM_OF_STATIONS] = {MULTI_TID_ACK_ENABLED_UNSUPPORTED};
uint8 heMultiTidAggSupportLimit[HW_NUM_OF_STATIONS] = {0x0}; //initial value is that multi TID is not supported

#ifdef ENET_INC_ARCH_WAVE600D2 
static StaHeErParameters_t heExtDefaultValues = 
{
	//default configurations for HE according to system
	HE_SU_STA_RESPONSE_DUR_TIMEOUT_LEGACY_BASED,
	HE_SU_PHY_MODE_BA_ACK_RESP_TYPE_11AG,
	HE_SU_ER_PHY_MODE_BA_ACK_RESP_TYPE_HE_ER_SU,
};
#endif //ENET_INC_ARCH_WAVE600D2
#endif

//---------------------------------------------------------------------------------
//						Private Function Declaration 			
//---------------------------------------------------------------------------------
#if defined (ENET_INC_UMAC)
void staDbFillDbEntry(UMI_STA_ADD* pStaAdd);
#endif

//---------------------------------------------------------------------------------
//						Function implementation				
//--------------------------------------------------------------------------------

/**********************************************************************************
staDbCopyIeeeAddrToRaType 

Description:
------------

Input:
-----

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void staDbCopyIeeeAddrToRaType(IEEE_ADDR* pMacAddress, StaDbRa_t* pRaMacAddress)
{
	uint32			  macAddr03 = 0;
	uint32			  macAddr45 = 0;

	macAddr03 = (pMacAddress->au8Addr[3] << 24) | 
				(pMacAddress->au8Addr[2] << 16) | 
		    	(pMacAddress->au8Addr[1] << 8)  |
		    	(pMacAddress->au8Addr[0]);
	macAddr45 = (pMacAddress->au8Addr[5] << 8) | (pMacAddress->au8Addr[4]);
	pRaMacAddress->macAddress03 = macAddr03;
	pRaMacAddress->macAddress45 = macAddr45;
}

void staDbBssLoadIeAddSta(uint8 vapId)
{
	BSS_LOAD_ELEMENT *bssLoadElement_p = &bssLoadIePayload[vapId].bssLoad;

	DEBUG_ASSERT(bssLoadElement_p->stationCount < HW_NUM_OF_STATIONS);
	bssLoadElement_p->stationCount++;
}

void staDbBssLoadIeRemoveSta(uint8 vapId)
{
	BSS_LOAD_ELEMENT *bssLoadElement_p = &bssLoadIePayload[vapId].bssLoad;

	DEBUG_ASSERT(bssLoadElement_p->stationCount != 0);
	bssLoadElement_p->stationCount--;
}

void StaDb_SetPsduTimeLimit(StaId stationIndex ,  uint32 maxPsduTransmissionTimeLimit)
 {
	StaDb_t* pStaDbHwEntries;
	uint8 tidIndex;
	StaDbTid_t* pStaTidEntry;
    TX_INTERRUPT_SAVE_AREA;
	
	DEBUG_ASSERT(stationIndex < HW_NUM_OF_STATIONS);
	pStaDbHwEntries = &StaDbHwEntries[stationIndex];
    //ILOG0_DD("StaDb_SetPsduTimeLimit stationIndex:%d maxPsduTransmissionTimeLimit:%d", stationIndex, maxPsduTransmissionTimeLimit);
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif

	for (tidIndex = 0; tidIndex < NUM_OF_TIDS; tidIndex++)
	{ 
   		pStaTidEntry = &(pStaDbHwEntries->tid[tidIndex]);
		pStaTidEntry->maxPsduTransmissionTimeLimit = (maxPsduTransmissionTimeLimit >> 1); // time resolution is 2microSec
	}
#if defined(ENET_INC_ARCH_WAVE600)
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}


#if defined (ENET_INC_UMAC)

/**********************************************************************************
StaDB_OpenBaAgreement 

Description:
------------
Enable agregations for a specific station for a specific TID

Input:
-----
stationIndex - the station for which to open BA agreement
tid - the TID for which to open BA agreement
bufferSize - maximum window size
amsduSupport - indicates if the station support A-MSDUs

Output:
-------

Returns:
--------


**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
void StaDB_OpenBaAgreement(StaId stationIndex, uint8 tid, uint16 bufferSize, uint8 amsduSupport)
#else
void StaDB_OpenBaAgreement(StaId stationIndex, uint8 tid, uint8 bufferSize, uint8 amsduSupport)
#endif //ENET_INC_ARCH_WAVE600
{
	uint8 maskedBufferSize = (bufferSize & STATION_DATABASE_BUFFER_SIZE_MASK) ;
	/* The first word of the TID parameters is used only by the sender. the second word is changed by the TS manager. 
       In case the LMAC need to change paramneters in the second word it should send a message to entity in UMAC
       that will do it for it to avoid race conditions */
    TX_INTERRUPT_SAVE_AREA;
	StaDbTid_t newTidParameters;
	uint32 *destinationAddress = NULL;
	uint32 *sourceAddress = NULL;
	uint8 tidIndex = 0;
	RegTxSelectorMuTidBitmap_u tidBitmap;
	uint8 maxWindowSize;
	uint8 ampduLimit;
    /* The TID parameters should be word aligned */
	memset32(&newTidParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(StaDbTid_t)));
	destinationAddress = (uint32 *)(&StationDatabase.pStaDbHwEntries[stationIndex].tid[tid]);
	sourceAddress  = (uint32 *)(&newTidParameters);
	/* writing only to the second word of the parameters */
	destinationAddress ++;
	sourceAddress ++;
	/* The disable/enable interrupts here is to prevent writing to this structure by 2 entities in UMAC. 
	   In case another entity than the TS manager needs to change TID parameters it should have 
	   disable/enable in its function */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	
	/* Get current values of the second word of the TID parameters */
	 *sourceAddress = *destinationAddress;
	/* Writing the new parameters */
	newTidParameters.baEnable = TRUE;
    newTidParameters.currentBaWindowLimit = maskedBufferSize;
	newTidParameters.ppduTxMode = STA_DB_PPDU_TX_MODE_LEGACY_AND_MPDU | (amsduSupport << STATION_DATABASE_TX_MODE_AMSDU_ENABLE_SHIFT);

	newTidParameters.txopMultipleDataEn = StaDb_TxopMode[stationIndex];
	
	/* Save new parameters in the station database */
	*destinationAddress = *sourceAddress;
	/*Search for the maximum baWindows in order to update the ampdu limit*/
	maxWindowSize = StaDB_SearchMaxWindowSizeOutOfTids(stationIndex);
	ampduLimit = StaDB_CalcAmpduLimit(maxWindowSize);
	ILOG0_DD("currentTidWindowSize, maxWindowSize = %d, ampduLimit = %d", maxWindowSize, ampduLimit);
	/*Update sta dB with new ampdu limit val*/	
	StationDatabase.pStaDbHwEntries[stationIndex].common.aMpduLimit = ampduLimit;
#if defined(MULTI_TID_ENABLED)
	// Add TID with BA agreement to multi-tid bitmap
	StaDB_HeMultiTidAddTid(stationIndex, tid);
#endif //MULTI_TID_ENABLED

#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	
	// the following loop enables MU TID bits of the station each time new BA Agreement of some TID is opened
	/*If STA is a member of MU groups need to set the PPDU TX mode in the Group DB*/
	/*Adding a TID can only decrease the PPDU TX mode-i.e. move ot from Two Level to One Level 
	- change the group PPDU TX mode before
	adding the TID to the selector so we dont transmit two level to this TID*/
#ifndef ENET_INC_ARCH_WAVE600
	// In gen6, HW takes the Tx PPDU mode from sta DB, no need to copy it to group DB
	GroupManager_SetPpduTxModePerStation(stationIndex);	
#endif
	tidBitmap.val = 0;
	for(tidIndex = 0; tidIndex < NUM_OF_TIDS; tidIndex ++)
	{
		if(StationDatabase.pStaDbHwEntries[stationIndex].tid[tidIndex].baEnable)
		{
			tidBitmap.val |= (1 << tidIndex);
		}
	}
	TxSelector_ConfigureStaMuEnabledTids(stationIndex, tidBitmap);

}

/**********************************************************************************
StaDB_CloseBaAgreement 

Description:
------------
disables agregations for a specific station for a specific TID

Input:
-----
stationIndex - the station for which to open BA agreement
tid - the TID for which to open BA agreement
amsdu support - whether we support A-MSDU in legacy mode

Output:
-------

Returns:
--------


**********************************************************************************/
void StaDB_CloseBaAgreement(StaId stationIndex, uint8 tid, uint8 changePpduMode)
{    
	/* The first word of the TID parameters is used only by the sender. the second word is changed by the TS manager. 
       In case the LMAC need to change paramneters in the second word it should send a message to entity in UMAC
       that will do it for it to avoid race conditions */
    TX_INTERRUPT_SAVE_AREA;
	StaDbTid_t newTidParameters;
	uint32 *destinationAddress = NULL;
	uint32 *sourceAddress = NULL;
		
	uint8 tidIndex = 0;
	RegTxSelectorMuTidBitmap_u tidBitmap;

    /* The TID parameters should be word aligned */
	memset32(&newTidParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(StaDbTid_t)));
	destinationAddress = (uint32 *)(&StationDatabase.pStaDbHwEntries[stationIndex].tid[tid]);
	sourceAddress  = (uint32 *)(&newTidParameters);
	/* writing only to the second word of the parameters */
	destinationAddress ++;
	sourceAddress ++;
	/* The disable/enable interrupts here is to prevent writing to this structure by 2 entities in UMAC. 
	   In case another entity than the TS manager needs to change TID parameters it should have 
	   disable/enable in its function */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	/* Get current values of the second word of the TID parameters */
	 *sourceAddress = *destinationAddress;
	/* Writing the new parameters */
	newTidParameters.baEnable = FALSE;
    newTidParameters.currentBaWindowLimit = 1;
	if(changePpduMode)
	{	
		//trun of only the AMPDU support
		newTidParameters.ppduTxMode =  newTidParameters.ppduTxMode & STA_DB_PPDU_TX_MODE_LEGACY_AND_AMSDU_NO_AMPDU;
	}

	newTidParameters.txopMultipleDataEn = FALSE;

	/* Save new parameters in the station database */
	*destinationAddress = *sourceAddress;

#if defined(MULTI_TID_ENABLED)
	// Remove TID with BA agreement to multi-tid bitmap only if STA doesn't support ack-enabled aggregation
	if(MULTI_TID_ACK_ENABLED_UNSUPPORTED == gackEnabledSupport[stationIndex])
	{
		StaDB_HeMultiTidRemoveTid(stationIndex, tid);
	}
#endif //MULTI_TID_ENABLED

#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

	/*Clear activity bit*/
	(void)Pac_RxcBaAgreementInitiatorGetAndClearActivity(stationIndex, tid);
	

	// the following loop disables MU TID bits of the station each time BA Agreement of some TID is closed
	tidBitmap.val = 0;
	for(tidIndex = 0; tidIndex < NUM_OF_TIDS; tidIndex ++)
	{
		if(StationDatabase.pStaDbHwEntries[stationIndex].tid[tidIndex].baEnable)
		{
			tidBitmap.val |= (1 << tidIndex);
		}
	}
	TxSelector_ConfigureStaMuEnabledTids(stationIndex, tidBitmap);
	/*If STA is a member of MU groups need to set the PPDU TX mode in the Group DB - we do it here because this function
	looks at the BA enable bit in the TID Params - closing a BA agreement can only increase the level. i.e.
	moving from one level to two level so there is no risk of transmitting to the remaaining TIDs more than what was
	negotiated*/
#ifndef ENET_INC_ARCH_WAVE600
	// In gen6, HW takes the Tx PPDU mode from sta DB, no need to copy it to group DB
	GroupManager_SetPpduTxModePerStation(stationIndex);
#endif
}


void StaDB_ChangePpduTxMode(StaId stationIndex, uint8 tid, uint8 PpduTxMode)
{   
	TX_INTERRUPT_SAVE_AREA;
	StaDbTid_t newTidParameters;
	uint32 *destinationAddress = NULL;
	uint32 *sourceAddress = NULL;
		

	/* The TID parameters should be word aligned */
	memset32(&newTidParameters, 0, CONVERT_BYTES_TO_WORDS(sizeof(StaDbTid_t)));
	destinationAddress = (uint32 *)(&StationDatabase.pStaDbHwEntries[stationIndex].tid[tid]);
	sourceAddress  = (uint32 *)(&newTidParameters);
	/* writing only to the second word of the parameters */
	destinationAddress ++;
	sourceAddress ++;
	/* The disable/enable interrupts here is to prevent writing to this structure by 2 entities in UMAC. 
	   In case another entity than the TS manager needs to change TID parameters it should have 
	   disable/enable in its function */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	/* Get current values of the second word of the TID parameters */
	 *sourceAddress = *destinationAddress;
	/* change the PPDU TX mode - trun of only the AMPDU support */
	newTidParameters.ppduTxMode = newTidParameters.ppduTxMode & PpduTxMode;

	/* Save new parameters in the station database */
	*destinationAddress = *sourceAddress;
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
	HW_SEMAPHORE_FREE(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif
	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

/**********************************************************************************
InitAssociateDBentry 

Description:
------------
The function initializes the statio DB entry after association is completed

Input:
-----

Output:
-------

Returns:
--------


**********************************************************************************/
void staDbFillDbEntry(UMI_STA_ADD* pStaAdd)
{
 	StaId				staId = pStaAdd->u16SID;
	StaId				previousFirstConnectedSta;
 	uint8				vapIndex = pStaAdd->u8VapIndex;
	StaDbSwInfo_t*    	pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	StaDb_t*          	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
#ifdef ENET_INC_ARCH_WAVE600 
	HE_MAC_PHY_CAPABILITIES_INFO *heMacPhyCapabilities = (HE_MAC_PHY_CAPABILITIES_INFO *)&pStaAdd->u8HE_Mac_Phy_Cap_Info;
	HE_MAC_CAPABILITIES_INFO *heMacCapabilities = (HE_MAC_CAPABILITIES_INFO *)&heMacPhyCapabilities->heMacCapInfo;
#endif
	/* Increment number of stations in VAP */
	VapDb_AddStationCounter(vapIndex);
	StationDatabase.NumOfConnectedSta++;

	staDbBssLoadIeAddSta(vapIndex);

	/*****************/
	/* INITIALIZE SW DB */
	/****************/
	/* General parameters */
	pStaDbSwEntry->state = STA_STATE_CONNECTED;
	StaDB_SetMacAddress(staId,&pStaAdd->sAddr);	
	pStaDbSwEntry->vapIndex = vapIndex;
    /* Insert the STA to the head of the double linked list of STAs in this VAP*/     
	previousFirstConnectedSta = VapDb_GetFirstConnectedStaIndex(vapIndex);
	StaDb_SetNextSid(staId, previousFirstConnectedSta);
    StaDb_SetPreviousSid(staId, DB_ASYNC_SID);
	if (previousFirstConnectedSta != DB_ASYNC_SID)
	{
		StaDb_SetPreviousSid(previousFirstConnectedSta, staId);
	}
	VapDb_SetFirstConnectedStaIndex(vapIndex, staId);
#if defined(ENET_INC_ARCH_WAVE600)
	if (MTLK_BFIELD_GET(pStaAdd->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE) == TRUE)
	{ //HE capabilities
		// Init SU Multi-TID parameters for this STA
		StaDB_InitHeSuMultiTid(staId, heMacCapabilities->HE_MAC_MULTI_TID_AGGREGATION_RX_SUPPORT, heMacCapabilities->HE_MAC_ACK_ENABLED_MULTI_TID_AGGREGATION_SUPPORT);
#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A after Jira WLANVLSIIP-2881 is resolved in B0
		StaDB_heMultiTidStationAcPsTypesAreMixed(staId, pStaAdd->u8UAPSD_Queues);
#endif //PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE

		StaDb_SetMaxStationTidLimitInAggregation(staId, StaDb_GetMaxStationTidLimitInAggregation(staId));
	}
#endif //ENET_INC_ARCH_WAVE600	
	staDbCopyIeeeAddrToRaType(&pStaAdd->sAddr, (StaDbRa_t*)&(pStaDbHwEntry->ra.word0));
	/*****************/
	/* INITIALIZE HW DB */
	/****************/
	/* Encryption parameters are set by set security key during the 4W handshake */
	pStaDbHwEntry->common.dataEncryptionEnable = DISABLED; 	  //  [26]   Data Encryption disabled
	pStaDbHwEntry->common._80211EncryptionEnable = DISABLED;//   [27]  Management Encryption disabled
	pStaDbHwEntry->common.dataKeyId = KEY_ID_0;			      //   [29:28]  
	pStaDbHwEntry->common._80211KeyId = KEY_ID_0;		      //   [31:30] 
#if defined(ENET_INC_ARCH_WAVE600)
	pStaDbHwEntry->common.pn03 = 0;
	pStaDbHwEntry->common.pn45 = 0;
#else 
	/* DW19  */
	pStaDbHwEntry->common.pn01 = 0;
	/* DW20  */
	pStaDbHwEntry->common.pn25 = 0;
#endif 
	/* DW21  */
	pStaDbHwEntry->common.keyIndex0 = SECURITY_INVALID_KEY_INDEX;
	pStaDbHwEntry->common.keyIndex1 = SECURITY_INVALID_KEY_INDEX;	
	/* DW22  */
	pStaDbHwEntry->common.keyIndex2 = SECURITY_INVALID_KEY_INDEX;
	pStaDbHwEntry->common.keyIndex3 = SECURITY_INVALID_KEY_INDEX;

#ifdef ENET_INC_ARCH_WAVE600D2
	//set HE ER default values
	StaDB_SetHeSuExtParams(staId, FALSE);
#endif //ENET_INC_ARCH_WAVE600D2
	
}

/**********************************************************************************
StaDB_AddStaToDB 

Description:
------------
The function adds a station to a VAP. The function receives the parameter ConnectionState 
which determines the initialization phase of the DB entry. 


Input:
-----
VapIndex - VAP Index the station associates to.
Sta ID - Station ID
connectionState_t - [Authenticate/Associate] Indicates the initialization phase

Output:
-------

Returns:
--------


**********************************************************************************/
void StaDB_AddStaToDB(UMI_STA_ADD* pStaAdd)
{
	staDbFillDbEntry(pStaAdd);
}
#endif

/**********************************************************************************
StaDb_RemoveStationFromDB 

Description:
------------
Clean the STA DB entry of the disconnected station. 

Input:
-----
Sta ID - Station ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void StaDb_RemoveStationFromDB(StaId staId)
 {
	StaDbSwInfo_t* pStaDbSwEntry = &(StationDatabase.StaDbSwEntries[staId]);
	StaDb_t* pStaDbHwEntry = NULL;
	uint8 vapIndex = pStaDbSwEntry->vapIndex;
 	StaId previousFirstConnectedSta;
	StaId previousSid = StaDb_GetPreviousSid(staId);
	StaId nextSid = StaDb_GetNextSid(staId);	
 
	ILOG2_D("StaDb_RemoveStationFromDB %X", staId);
	if (pStaDbSwEntry->state)
	{
		pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	    /* Remove the STA from the double linked list of STAs in this VAP*/     
		previousFirstConnectedSta = VapDb_GetFirstConnectedStaIndex(vapIndex);
		if (previousFirstConnectedSta == staId)
		{
			// STA is first in list. Set new head in VAP.
			VapDb_SetFirstConnectedStaIndex(vapIndex, nextSid);
		}
		else
		{
			// STA is not first. Connect previous STA to next STA (if exists)
			StaDb_SetNextSid(previousSid, nextSid);
		}
		if (nextSid != DB_ASYNC_SID)
		{
			// STA is not last. Connect next STA to previous STA
			StaDb_SetPreviousSid(nextSid, previousSid);
		}
		VapDb_DecreaseStationCounter(pStaDbSwEntry->vapIndex);
		/* Clear SW DB */
		memset(pStaDbSwEntry, 0, sizeof(StaDbSwInfo_t));
		memset(pStaDbHwEntry, 0, sizeof(StaDb_t));
		pStaDbSwEntry->state = STA_STATE_SID_ALLOCATED;
		StationDatabase.NumOfConnectedSta--;
		
		staDbBssLoadIeRemoveSta(vapIndex);
	}
}

#if defined (ENET_INC_UMAC)
/**********************************************************************************
StaDB_SetSecurityKey 

Description:
------------
This function sets the STA DB security entry according to the parameters passed from the driver. 

Input:
-----
pSetKey - a pointer to the structure that contains all the parameters


Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void StaDB_SetSecurityKey(UMI_SET_KEY *pSetKey, bool rekeyDisableFlag)
{
	uint16 staKeysOffset = 0;
	uint8 index = 0;
	uint8 keyId = 0;
	uint8 encryptionType;
#if defined(ENET_INC_ARCH_WAVE600)	
	bool  staModeEnable = FALSE;
#endif
	StaDb_t* pStaDbHwEntry = NULL;
 	StaId stationId = 0;
 
	stationId = pSetKey->u16Sid;
	encryptionType = pSetKey->u16CipherSuite;

	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationId]);

	keyId = pSetKey->u16KeyIndex;

#if defined(ENET_INC_ARCH_WAVE600)	
		//in wep remap is not used
		if((encryptionType != UMI_RSN_CIPHER_SUITE_WEP40) && (encryptionType != UMI_RSN_CIPHER_SUITE_WEP104))
		{
			pStaDbHwEntry->common.rxKeyId0Remapping = 0x0;
			pStaDbHwEntry->common.rxKeyId1Remapping = 0x1;
			pStaDbHwEntry->common.rxKeyId2Remapping = 0x0;
			pStaDbHwEntry->common.rxKeyId3Remapping = 0x1;
	
			keyId &= 0x1; //key IDs 0 & 2 mapped to 0, 1 & 3 mapped to 1
		}
#endif // ENET_INC_ARCH_WAVE600
	
	/* First copy the PNs before enabling the encryption in order to avoid races 
	   This code relies on the assumption that there is no rekey for unicast */
	StaDB_SetTxPn(stationId, pSetKey->au8TxSeqNum);
#if defined(ENET_INC_ARCH_WAVE600)
	RxPp_SetPn(stationId, pSetKey->au8RxSeqNum, keyId, rekeyDisableFlag, staModeEnable);
#else //ENET_INC_ARCH_WAVE500
	RxPp_SetStationPn(stationId, pSetKey->au8RxSeqNum);
#endif
	pStaDbHwEntry->common.dataEncryptionType = encryptionType;
	pStaDbHwEntry->common.dataEncryptionEnable = TRUE;
	/*
			Keys DB
		------------------------
		|VAP 0 keys				|
		|VAP 1 keys				|
		|	...					|
		|VAP 15/31(gen6) keys	|
		|STA 0 keys				|
		|STA 1 keys				|
		|	...					|
		|STA 127/255(gen6) keys	|	
		------------------------
	*/
	/* Offset of the key value from the start of the security Key values DB is measured by: 
	    Location of STA keys (After VAP keys) + OFFSET TO STA 4 KEYS + OFFSET BY KEY ID */
	staKeysOffset = (stationId * SECURITY_KEYS_NUM) + (HW_NUM_OF_VAPS * SECURITY_KEYS_NUM);
	staKeysOffset += keyId;
	/* shabtay - Add to STA HW DB the security indexes */
	switch (keyId)
	{
		case KEY_ID_0:
			pStaDbHwEntry->common.keyIndex0 = staKeysOffset; 
			break;
		case KEY_ID_1:
			pStaDbHwEntry->common.keyIndex1 = staKeysOffset;
			break;
		case KEY_ID_2:
			pStaDbHwEntry->common.keyIndex2 = staKeysOffset;
			break;
		case KEY_ID_3:
			pStaDbHwEntry->common.keyIndex3 = staKeysOffset; 
			break;
		default:
			ASSERT(0);
	}
	switch(encryptionType)
	{
#ifdef ENET_INC_ARCH_WAVE600	
		case UMI_RSN_CIPHER_SUITE_GCMP128:
			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId].actualKey[index] = pSetKey->au8Tk1[GCMP_TK_SIZE - (index+1)];
			}
			break;
		case UMI_RSN_CIPHER_SUITE_GCMP256:
			/* In GCMP256 there are 2 valid keys */
			if(keyId == KEY_ID_0)
			{
				pStaDbHwEntry->common.keyIndex2 = staKeysOffset + 2;
			}
			else
			{
				pStaDbHwEntry->common.keyIndex3 = staKeysOffset + 2;
			}

			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId].actualKey[index] = pSetKey->au8Tk2[GCMP_TK_SIZE - (index+1)];
			}
			// Set the 2nd part of the key in the MIC entry
			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId + 2].actualKey[index] = pSetKey->au8Tk1[GCMP_TK_SIZE - (index+1)];
			}
			break;
#endif // ENET_INC_ARCH_WAVE600			
		case UMI_RSN_CIPHER_SUITE_CCMP:
			for (index = 0 ; index < CCMP_TK_SIZE ; index++)
			{
				KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId].actualKey[index] = pSetKey->au8Tk1[CCMP_TK_SIZE - (index+1)];
			}
			break;
		case UMI_RSN_CIPHER_SUITE_TKIP:
			/* For CCMP need to copy MIC key also */
			MEMCPY(&KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId + 2], pSetKey->au8Tk2, sizeof(Key_t));
			/* In Tkip there are only key 0 and key 1 valid. key 2 and 3 are MIC keys */

			if(keyId == KEY_ID_0)
			{
				pStaDbHwEntry->common.keyIndex2 = staKeysOffset + 2;
			}
			else
			{
				pStaDbHwEntry->common.keyIndex3 = staKeysOffset + 2;
			}
			/* fall throught */
		case UMI_RSN_CIPHER_SUITE_WEP40: 
			/* fall through */
		case UMI_RSN_CIPHER_SUITE_WEP104:	
			MEMCPY(&KeysDatabase[HW_NUM_OF_VAPS + stationId].keyValues[keyId], pSetKey->au8Tk1, sizeof(Key_t));
			
			//WEP WDS can not support more then 460 Mbitsec. Need to reduce the PHY rate.   
			// TBD - Remove once rate adaptive is on and make sure this limitation is forced somewhere.
			//if (phyMode == PHY_MODE_11AC)
			//{	
				//TBD the link adaptation should support this - no point chaning the rates becasue the link adaptation will change them again
				//StaDB_SetAllDataRates(stationId, staRateType, selectedDataRate, phyMode);
			//}
			break;
		default:
			ASSERT(0);
	}
}
#endif /* #if defined (ENET_INC_UMAC) */

/**********************************************************************************
StaDB_SetDefaultKeyId 

Description:
------------
This function sets the default key ID in the STA DB according to the parameters passed from the driver. 

Input:
-----
StaId - Station ID
defaultKeyId - Default Key Id (0/1/2/3)

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
 void StaDB_SetDefaultKeyIndex(StaId staId, uint8 defaultKeyId, uint8 setForPeerAp)
 {
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);

	if ((pStaDbHwEntry->common.negotiatedWds == DISABLED) || 
		(pStaDbHwEntry->common.wdsClientType != PEER_AP) || 
		(setForPeerAp == TRUE))
	{
		pStaDbHwEntry->common.dataKeyId = defaultKeyId;
	}
}


#if defined (ENET_INC_UMAC)
/**********************************************************************************
StaDB_SetTxPn

Description:
------------
set the PN in the station DataBase
Input:
-----
stationId - the station Id
pnAddress - the address of the PN to copy


Output:
-------

Returns:
--------


**********************************************************************************/
#if defined(ENET_INC_ARCH_WAVE600)
void StaDB_SetTxPn(StaId stationId, uint8 *pnAddress)
{
	uint16 pn03 = 0;
	uint32 pn45 = 0;
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationId]);

	pn03 = ((pnAddress[0]) | (pnAddress[1] << 8) | (pnAddress[2] <<16) | (pnAddress[3] << 24));
	pn45 = ((pnAddress[4]) | (pnAddress[5] << 8));
	pStaDbHwEntry->common.pn03 = pn03;
	pStaDbHwEntry->common.pn45 = pn45;
}
#else //wave500 
void StaDB_SetTxPn(StaId stationId, uint8 *pnAddress)
{
	uint16 pn01 = 0;
	uint32 pn25 = 0;
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationId]);

	pn01 = ((pnAddress[0]) | (pnAddress[1] << 8));
	pn25 = ((pnAddress[2]) | (pnAddress[3] << 8) | (pnAddress[4] << 16) | (pnAddress[5] << 24));
	pStaDbHwEntry->common.pn01 = pn01;
	pStaDbHwEntry->common.pn25 = pn25;
}
#endif //ENET_INC_ARCH_WAVE600

#endif /* #if defined (ENET_INC_UMAC) */

/**********************************************************************************
StaDB_SetWepSecurityKeyIndex 

Description:
------------
This function sets the STA DB security entry according to the parameters passed from the driver. 

Input:
-----
StaId - Station ID
keyIndex
keyIndexValue - the index which match the vap entry of this WEP key

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void StaDB_SetWepSecurityKeyIndex(StaId staId, uint8 keyIndex, uint32 keyIndexValue)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);

	/*
			Keys DB
		---------------
		|VAP 0 keys		|
		|VAP 1 keys		|
		|	...			|
		|VAP15 keys		|
		|STA 0 keys		|
		|STA 1 keys		|
		|	...			|
		|STA 127 keys	|	
		---------------
	*/
	// Set the key index to point to the VAP part of the KeysDB.
	// For WDS - we don't do this. PeerAPs has their own WEP pairwise keys which are set separately.
	if ((pStaDbHwEntry->common.negotiatedWds == DISABLED) || (pStaDbHwEntry->common.wdsClientType != PEER_AP))
	{
		switch (keyIndex)
		{
			case KEY_ID_0:
				pStaDbHwEntry->common.keyIndex0 = keyIndexValue; 
				break;
			case KEY_ID_1:
				pStaDbHwEntry->common.keyIndex1 = keyIndexValue;
				break;
			case KEY_ID_2:
				pStaDbHwEntry->common.keyIndex2 = keyIndexValue;
				break;
			case KEY_ID_3:
				pStaDbHwEntry->common.keyIndex3 = keyIndexValue; 
				break;
			default:
				ASSERT(0);
		}
	}
}

/**********************************************************************************
StaDB_ClearSecurityKey 

Description:
------------
Clear security key. The function receives the key Index of the key to clear
and sets an Invalid index in the key entry of Key Indexes array in STA DB.

Input:
-----
StaId - Station ID
KeyId - The key ID of the key to clear

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void StaDB_ClearSecurityKey(StaId staId, uint8 vapId)
{
	StaDb_t* pStaDbHwEntry =	&(StationDatabase.pStaDbHwEntries[staId]);

	/*Clear key Indices*/
	pStaDbHwEntry->common.keyIndex0 = SECURITY_INVALID_KEY_INDEX; 	
	pStaDbHwEntry->common.keyIndex1 = SECURITY_INVALID_KEY_INDEX; 	
	pStaDbHwEntry->common.keyIndex2 = SECURITY_INVALID_KEY_INDEX; 	
	pStaDbHwEntry->common.keyIndex3 = SECURITY_INVALID_KEY_INDEX; 	
	/*If STA is a VSTA, need to clear the group Key. As STA may reconnect and start receiving multicast data with a new key*/
	if ((VapDb_GetVapMode(vapId) == VAP_MODE_STA) && (pStaDbHwEntry->common.dataEncryptionType != UMI_RSN_CIPHER_SUITE_WEP40) && (pStaDbHwEntry->common.dataEncryptionType != UMI_RSN_CIPHER_SUITE_WEP104))
	{
		VapDB_ClearSecurityKey(vapId);
		RxManager_ClearGroupKey(vapId);
		VapDB_DisableManagementEncryption(vapId);
		VapDB_SetDataEncryptionEnabled(vapId, FALSE);
	}
}

/**********************************************************************************
StaDB_SetWepSecurityKey 

Description:
------------
When STA connects, if VAP is WEP we need to copy the Key Indices from VAP DB to STA DB

Input:
-----
StaId - Station ID
VapID - VAP ID
CIpher Suite

Output:
-------

Returns:
--------

**********************************************************************************/
void StaDB_SetWepSecurityKey(StaId staId, uint8 vapId, uint8 cipherSuite)
{
	StaDb_t*			pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);

	pStaDbHwEntry->common.dataEncryptionType = cipherSuite;
	pStaDbHwEntry->common.dataEncryptionEnable = ENABLED;
	pStaDbHwEntry->common.dataKeyId = VapDB_GetDefaultKeyIndex(vapId);
	pStaDbHwEntry->common.keyIndex0 = VapDB_GetKeyIndexValue(vapId, KEY_ID_0);
	pStaDbHwEntry->common.keyIndex1 = VapDB_GetKeyIndexValue(vapId, KEY_ID_1);
	pStaDbHwEntry->common.keyIndex2 = VapDB_GetKeyIndexValue(vapId, KEY_ID_2);
	pStaDbHwEntry->common.keyIndex3 = VapDB_GetKeyIndexValue(vapId, KEY_ID_3);
}


///TBD to remove when removing wave500 BA handler
uint16 StaDB_GetBaWinSsn(StaId staId, uint8 tid)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);

	return (pStaDbHwEntry->tid[tid].lowSequenceNumber);
}
uint16 StaDB_GetBaWinSize(StaId staId, uint8 tid)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);

	return (pStaDbHwEntry->tid[tid].currentBaWindowLimit);
}

void StaDB_SetBaWinSsn(StaId staId, uint8 tid,uint16 offset)
{
	StaDb_t* pStaDbHwEntry =	&(StationDatabase.pStaDbHwEntries[staId]);
	pStaDbHwEntry->tid[tid].lowSequenceNumber += offset;
}

uint8 StaDB_GetQosEnable(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	return (pStaDbHwEntry->common.qosEnableCapability);
}

void StaDB_SetManagementEncrypted(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	/*Management is always CCMP*/
	pStaDbHwEntry->common._80211EncryptionType   = UMI_RSN_CIPHER_SUITE_CCMP;
	pStaDbHwEntry->common._80211EncryptionEnable = TRUE;
	pStaDbHwEntry->common._80211KeyId            = pStaDbHwEntry->common.dataKeyId;
}

uint8 StaDB_IsManagementEncrypted(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return (pStaDbHwEntry->common._80211EncryptionEnable);
}

uint8 StaDB_IsDataEncrypted(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return (pStaDbHwEntry->common.dataEncryptionEnable);
}

void StaDB_SetDataEncryptionEnable(StaId staId, bool dataEncryptionEnable)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.dataEncryptionEnable = dataEncryptionEnable;
}


uint8 StaDB_IsStationConnected(StaId staId)
{
	StaDbSwInfo_t* pStaDbSwEntry = &(StationDatabase.StaDbSwEntries[staId]);
	
	return (pStaDbSwEntry->state == STA_STATE_CONNECTED);
}

uint8 StaDB_GetNegotiatedWds(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return (pStaDbHwEntry->common.negotiatedWds);
}

void StaDB_SetNegotiatedWds(StaId staId, bool wds)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.negotiatedWds = wds;
}

uint8    StaDB_GetWdsClientType(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return pStaDbHwEntry->common.wdsClientType;
}


uint8 StaDB_GetDataEncryptionType(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return (pStaDbHwEntry->common.dataEncryptionType);
}

void StaDB_SetDataEncryptionType(StaId staId, uint8 dataEncryptionType)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.dataEncryptionType = dataEncryptionType;
}

uint8 StaDB_GetManagementEncryptionType(StaId staId)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	return (pStaDbHwEntry->common._80211EncryptionType);
}

StaDbPpduTxMode_e StaDb_GetMinPpduTxModePerStation(StaId stationIndex)
{
	uint32 tidIndex;
	StaDb_t *pStaDbHwEntry = NULL;
	StaDbPpduTxMode_e ppduTxMode = STA_DB_PPDU_TX_MODE_TWO_LEVEL_AGGR;

	pStaDbHwEntry = &StationDatabase.pStaDbHwEntries[stationIndex];
	for (tidIndex = 0; tidIndex < NUM_OF_TID; tidIndex++)
	{
		if (pStaDbHwEntry->tid[tidIndex].baEnable == TRUE)
		{
			if ((StaDbPpduTxMode_e)(pStaDbHwEntry->tid[tidIndex].ppduTxMode) < ppduTxMode)
			{
				ppduTxMode = (StaDbPpduTxMode_e)(pStaDbHwEntry->tid[tidIndex].ppduTxMode);
			}
		}
	}
	return (ppduTxMode);
}

#ifdef ENET_INC_ARCH_WAVE600
/**********************************************************************************
StaDB_SetTidRRweight

Description: Set RR Weight for a specific TID
------------

Input:
-----
staId - station index
tidNum - TID number
tidRrWeight - weight from 1-8

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_SetTidRRweight(StaId staId, uint8 tidNum, MultiTidRrWeight_e tidRrWeight)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->tid[tidNum].tidRrWeight = tidRrWeight;
}

/**********************************************************************************
StaDB_SetHeMngAgg

Description: Set or Clear managment aggregation TID bit in TIDs bitmap
------------

Input:
-----
staId - station index
setResetVal - TRUE for set, FALSE for clear

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_SetHeMngAgg(StaId staId, MultiTidMngAgg_e setMngAgg)
{
	StaDb_t* pStaDbHwEntry = &(StaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.heMngAgg = setMngAgg;

	// Set/clear mng bit in heSupportMultiTid bitmap
	if(MULTI_TID_MNG_AGGREGATION_ON == setMngAgg)
	{
		pStaDbHwEntry->common.heSupportMultiTid |= MULTI_TID_SET_MNG_BIT_8;
	}
	else
	{
		pStaDbHwEntry->common.heSupportMultiTid &= MULTI_TID_CLEAR_MNG_BIT_8;
	}
}

/**********************************************************************************
StaDB_SetHeBarAgg

Description: Enable/Disable BAR aggregation
------------

Input:
-----
staId - station index
setResetVal - TRUE for enable, FALSE for disable

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_SetHeBarAgg(StaId staId, bool setResetVal)
{
	UNUSED_PARAM(setResetVal);
	UNUSED_PARAM(staId);	
	// According to WiFi 802.11ax standard draft 2.0 - multi-TID with BAR aggregation is not allowed
#if 0
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.heBarAgg = setResetVal;
#endif
}

/**********************************************************************************
StaDb_GetMaxStationTidLimitInAggregation

Description: 
------------

Input:
-----


Returns:
--------

	
**********************************************************************************/
uint8 StaDb_GetMaxStationTidLimitInAggregation(StaId stationIndex)
{
	return heMultiTidAggSupportLimit[stationIndex];
}

/**********************************************************************************
StaDb_SetMaxStationTidLimitInAggregation

Description: 
------------

Input:
-----


Returns:
--------

	
**********************************************************************************/
void StaDb_SetMaxStationTidLimitInAggregation(StaId stationIndex, uint8 maxTids)
{
	StaDbHwEntries[stationIndex].common.heMultiTidAggSupport = maxTids;
}

/**********************************************************************************
StaDB_InitHeSuMultiTid

Description: Init multi-tid fields in STA connection
------------

Input:
-----
staId - station index
maxTids - Maximum TIDs supported for aggregation, received after association with STA
ackEnabledSupport


Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_InitHeSuMultiTid(StaId staId, uint8 maxTids, uint8 ackEnabledSupport)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	uint8 tidIndex = 0;
	MultiTidAckEn_e isAckEnabledSupport = (MultiTidAckEn_e)ackEnabledSupport;
	
	// Clear TIDs bitmap (each TID that opens BA Agreement its bit will be set and cleared in del. BA agreement)
	pStaDbHwEntry->common.heSupportMultiTid = MULTI_TID_CLEAR_ALL_TIDS;

	// According to WiFi 802.11ax standard draft 2.0 - multi-TID with BAR aggregation is not allowed - Do not touch this setting
	pStaDbHwEntry->common.heBarAgg = MULTI_TID_BAR_AGGREGATION_DEFAULT;

	gackEnabledSupport[staId] = isAckEnabledSupport;

#ifdef HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED
	// According to current System requirements - by default, MNG aggregation is disabled, remove this if decided differently
	StaDB_SetHeMngAgg(staId, MULTI_TID_MNG_AGGREGATION_OFF);
#else
	if(isAckEnabledSupport == MULTI_TID_ACK_ENABLED_SUPPORTED)
	{
		StaDB_SetHeMngAgg(staId, MULTI_TID_MNG_AGGREGATION_ON);
	}
	else
	{
		StaDB_SetHeMngAgg(staId, MULTI_TID_MNG_AGGREGATION_OFF);
	}
#endif //HE_SU_MULTI_TID_MNG_AGG_UNSUPPORTED

	// Set initial RR weight for all TIDs
	for(tidIndex = 0  ; tidIndex < NUM_OF_TOTAL_TIDS ; tidIndex++)
	{
		// By default, all TIDs have lowest priority
		pStaDbHwEntry->tid[tidIndex].tidRrWeight = MULTI_TID_RR_WEIGHT_8;
	}

	// Set default select policy
	pStaDbHwEntry->common.heMultiTidSelectPolicy = MULTI_TID_SELECT_DEFAULT;
	
	// Set maximum TIDs supported in aggregation, value received after association request from STA
	heMultiTidAggSupportLimit[staId] = maxTids;

}

/**********************************************************************************
StaDB_SetHeMultiTidSelectPolicy

Description: Set TID select policy
------------

Input:
-----
staId - station index
multiTidSelectPolicy - selection policy:
	"00" - Policy is Low to High
	"01" - Policy is High to Low
	"10" - Policy is ALL

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_SetHeMultiTidSelectPolicy(StaId staId, MultiTidSelectPolicy_e multiTidSelectPolicy)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	
	pStaDbHwEntry->common.heMultiTidSelectPolicy = multiTidSelectPolicy;
}


/**********************************************************************************
StaDB_HeMultiTidAddTid

Description: Add TID with block-ack agreement to the multi-tid bitmap
------------

Input:
-----
staId - station index
TidNum - TID number to add

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_HeMultiTidAddTid(StaId staId, uint8 TidNum)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	uint16 mask = 0;

	//set the bit in bitfield only if multi-tid agg is supported
	//(this bit also effects the decision made by the pre aggregator on which TCR DB to use for the transmissions and forces using Data TCRs)
	if (pStaDbHwEntry->common.heMultiTidAggSupport != 0)
	{
		mask = (1<<TidNum);
		pStaDbHwEntry->common.heSupportMultiTid |= mask;
	}
}

/**********************************************************************************
StaDB_HeMultiTidRemoveTid

Description: Remove TID with block-ack agreement to the multi-tid bitmap
------------

Input:
-----
staId - station index
TidNum - TID number to remove

Returns:
--------
	void - 
	
**********************************************************************************/
void StaDB_HeMultiTidRemoveTid(StaId staId, uint8 TidNum)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);
	uint16 mask = 0;

	mask = ~(1<<TidNum);
	pStaDbHwEntry->common.heSupportMultiTid &= mask;
}

#ifdef PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE // remove this W\A after Jira WLANVLSIIP-2881 is resolved in B0
/**********************************************************************************
StaDB_heMultiTidStationAcPsTypesAreMixed


Description: if TIDs power-save types are mixed, some Legacy and some WMM override multi-tid aggregation support
------------

Input:
-----
staId - station index
acPsTypeBitmap - tids PS bitmap

Returns:
--------
	void - 
	
**********************************************************************************/
void  StaDB_heMultiTidStationAcPsTypesAreMixed(StaId staId, uint8 acPsTypeBitmap)
{
	// only 4 LSB bits are relevant from this bitmap
 	uint8 psTypeBitmap = (acPsTypeBitmap & 0xF);
	// If 4 AC bits are not all '0's or all '1's, remove multi-tid support for this STA by setting the number of supported TIDs to 0
	if((psTypeBitmap != 0) && (psTypeBitmap != 0xF))
	{
		heMultiTidAggSupportLimit[staId] = 0;
	}
}
#endif // PS_SELECTION_IN_HE_SELECT_ALL_TIDS_WITH_SAME_PS_TYPE
#endif //ENET_INC_ARCH_WAVE600


/**********************************************************************************
StaDB_UpdateLowSequenceNumber 

Description:
------------
Updates the low sequence number - which is the oldest sequnce number still being 
used by the Tx queue

Input:
-----
stationIndex - the station for which to update window
tid - the TID for which to update window
newSequenceNumber - the new value for the lowest sequence number


Output:
-------

Returns:
--------


**********************************************************************************/
void StaDB_UpdateLowSequenceNumber(StaId stationIndex, uint8 tid, uint16 newSequenceNumber)
{   
	
	ASSERT(stationIndex < HW_NUM_OF_STATIONS);
	ASSERT(tid < NUM_OF_TIDS);

	/* This function is used when the queue is locked so there should not be race with HW */
	/* Disable/enable interrupts is not needed since only a sinlge thread can be accesing a queue at a single time */
	StationDatabase.pStaDbHwEntries[stationIndex].tid[tid].lowSequenceNumber = newSequenceNumber;
}

/**********************************************************************************
StaDB_UpdateLowSequenceNumberFromCurrent 

Description:
------------
Updates the low sequence number (which is the oldest sequnce number still being 
used by the Tx queue) from the current sequnce number(the sequence number that will 
be assign to the next new PD)

Input:
-----
stationIndex - the station for which to update window
tid - the TID for which to update window


Output:
-------

Returns:
--------


**********************************************************************************/
void StaDB_UpdateLowSequenceNumberFromCurrent(StaId stationIndex, uint8 tid)
{	
	ASSERT(stationIndex < HW_NUM_OF_STATIONS);
	ASSERT(tid < NUM_OF_TIDS);

	/* This function is used when the queue is locked so there should not be race with HW */
	/* Disable/enable interrupts is not needed since only a sinlge thread can be accesing a queue at a single time */
	StationDatabase.pStaDbHwEntries[stationIndex].tid[tid].lowSequenceNumber = StationDatabase.pStaDbHwEntries[stationIndex].tid[tid].currentSequenceNumber;
}

#if defined(ENET_INC_ARCH_WAVE600)
void StaDB_UpdateLowSequenceNumberFromCurrentMngTid(StaId stationIndex)
{	
	/* This function is used when the queue is locked so there should not be race with HW */
	/* Disable/enable interrupts is not needed since only a sinlge thread can be accesing a queue at a single time */
	StationDatabase.pStaDbHwEntries[stationIndex].tid[MANAGEMENT_TID_IN_STA_DB].lowSequenceNumber = StationDatabase.pStaDbHwEntries[stationIndex].tid[MANAGEMENT_TID_IN_STA_DB].currentSequenceNumber;
}
#endif


/**********************************************************************************
StaDB_GetMacAddress 

Description:
------------
Get the Station Mac address

Input:
-----
StaId - station ID
IEEE_ADDR* pMacAddress - Mac address of the station

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void StaDB_GetMacAddress(StaId staId, IEEE_ADDR* pMacAddress)
{
	vIEEE_ADDR_CopyAddr(pMacAddress, &(StationDatabase.StaDbSwEntries[staId].macAddress));
}

IEEE_ADDR* StaDB_GetMacAddressPtr(StaId staId)
{
	return (&StationDatabase.StaDbSwEntries[staId].macAddress);
}

void StaDB_SetMacAddress(StaId staId, IEEE_ADDR* pMacAddress)
{
	vIEEE_ADDR_CopyAddr(&(StationDatabase.StaDbSwEntries[staId].macAddress),pMacAddress);
}

void StaDb_GetVapId(StaId staId, uint8* VapId)
{
	StaDbSwInfo_t*	  pStaDbSwEntry;

	ASSERT(staId < HW_NUM_OF_STATIONS);
	pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	*VapId = pStaDbSwEntry->vapIndex;
}

/**********************************************************************************
StaDb_GetNextSid 

Description:
------------
Provide the next STA index in the double linked list of STAs in that VAP.

Input:
-----
staId - station ID

Output:
-------

Returns:
--------
The next SID

**********************************************************************************/
StaId StaDb_GetNextSid(StaId staId)
{
	StaDbSwInfo_t*	  pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	
	return (pStaDbSwEntry->nextSid);
}

/**********************************************************************************
StaDb_SetNextSid 

Description:
------------
Set the next STA index in the double linked list of STAs in that VAP.

Input:
-----
staId - station ID
uint16 newNextSid

Output:
-------

Returns:
--------


**********************************************************************************/
void StaDb_SetNextSid(StaId staId, StaId newNextSid)
{
	StaDbSwInfo_t*	  pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	
	pStaDbSwEntry->nextSid = newNextSid;
}

/**********************************************************************************
StaDb_GetPreviousSid 

Description:
------------
Provide the previous STA index in the double linked list of STAs in that VAP.

Input:
-----
staId - station ID

Output:
-------

Returns:
--------
The next SID

**********************************************************************************/
StaId StaDb_GetPreviousSid(StaId staId)
{
	StaDbSwInfo_t*	  pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	
	return (pStaDbSwEntry->previousSid);
}

/**********************************************************************************
StaDb_SetPreviousSid 

Description:
------------
Set the previous STA index in the double linked list of STAs in that VAP.

Input:
-----
staId - station ID
uint16 newPreviousSid

Output:
-------

Returns:
--------


**********************************************************************************/
void StaDb_SetPreviousSid(StaId staId, StaId newPreviousSid)
{
	StaDbSwInfo_t*	  pStaDbSwEntry =  &(StationDatabase.StaDbSwEntries[staId]);
	
	pStaDbSwEntry->previousSid = newPreviousSid;
}

/**********************************************************************************
StaDB_SetSpLengthToMaximum

Description:
------------
	Set current SP length to be the maximum SP Length
Input:
-----
	stationId - the station Id
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void StaDB_SetSpLengthToMaximum(StaId stationId)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationId]);
	
	/* If SP is unlimited(=0) set current SP to 255 -> this value won't limit aggregation builder in its calculations */
	if(pStaDbHwEntry->common.maxSpLength == 0)
	{
		pStaDbHwEntry->common.currSpLengthLimit = MAX_SP_LENGTH_UNLIMITED;
	}
	else
	{
		pStaDbHwEntry->common.currSpLengthLimit = pStaDbHwEntry->common.maxSpLength;
	}
}

uint8	 StaDB_GetVapRtsCtsMethod(StaId staId)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);

	return pStaDbHwEntry->common.rtsCtsTxMethod;
}


uint32 StaDB_GetMaxPsduLimit(StaId staId)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	return (pStaDbHwEntry->common.maxPsduLengthLimit);
}

uint8 taDB_GetCurrentSpLengthLimit(StaId staId)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	return (pStaDbHwEntry->common.currSpLengthLimit);
}

void StaDB_SetCurrentSpLengthLimit(StaId staId , uint8 value)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	pStaDbHwEntry->common.currSpLengthLimit = value;
}

uint8 StaDB_GetNoAggInPs(StaId staId)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	return (pStaDbHwEntry->common.noAggInPs);
}

void StaDB_SetNoAggInPs(StaId staId, bool enable)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	pStaDbHwEntry->common.noAggInPs = enable;
}

void StaDB_SetPsForceOneNss(StaId staId, bool enable)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	pStaDbHwEntry->common.psForceOneNss = enable;
}

StaDbTid_t* StaDB_GetTidParams(StaId staId , uint8 tid)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	return ((StaDbTid_t*)&(pStaDbHwEntry->tid[tid]));
}

#ifdef ENET_INC_ARCH_WAVE600D2

void StaDB_SetHeSuExtParams(StaId staId, uint8 isHeExtSuDisable)
{
	StaDb_t* pStaDbHwEntry = NULL;
	
	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staId]);
	
	if (isHeExtSuDisable)
	{
		//heSuExt disabled - set all values to LEGACY	
		pStaDbHwEntry->common.heSuStaResponseMode = HE_SU_STA_RESPONSE_DUR_TIMEOUT_LEGACY_BASED;
		pStaDbHwEntry->common.heSuPhyModeBaAckRespType = HE_SU_PHY_MODE_BA_ACK_RESP_TYPE_11AG;
		pStaDbHwEntry->common.heSuErPhyModeBaAckRespType = HE_SU_ER_PHY_MODE_BA_ACK_RESP_TYPE_11AG;
	}
	else
	{
		//set default configuration
		pStaDbHwEntry->common.heSuStaResponseMode = heExtDefaultValues.heSuStaResponseMode; 
		pStaDbHwEntry->common.heSuPhyModeBaAckRespType = heExtDefaultValues.heSuPhyModeBaAckRespType;
		pStaDbHwEntry->common.heSuErPhyModeBaAckRespType = heExtDefaultValues.heSuErPhyModeBaAckRespType;		
	}
}

#endif //ENET_INC_ARCH_WAVE600D2

/**********************************************************************************

StaDB_SetPrimaryStationSuTxopAndNumMpdus 


Description:
------------
this function changes TXOP mode and num of max mpdus in AMPDU in single user for provided station in 
station DB. 

Input: stationIndex - index of station to change its configuration
	   txopMode - requested txopMode for this station in SU 
	   numOfMpdus - max num of Mpdus in AMPDU for this station in SU 
		
Output:
-------

Returns:
--------
	
**********************************************************************************/
void StaDB_SetPrimaryStationSuTxopAndNumMpdus(StaId stationIndex, uint8 txopMode, uint8 numOfMpdus)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationIndex]);
	uint8 tidIndex = 0;
	StaDbTid_t* pStaTidEntry = NULL;
	TX_INTERRUPT_SAVE_AREA;

	// set new value for SU in ampduLimit
	pStaDbHwEntry->common.aMpduLimit = numOfMpdus;
	// for all enabled Tids for this station - change txop mode in SU
#if defined(ENET_INC_ARCH_WAVE600)
	for(tidIndex=0; tidIndex < NUM_OF_TOTAL_TIDS; tidIndex++)
#else
	for(tidIndex=0; tidIndex < NUM_OF_TIDS; tidIndex++)
#endif //ENET_INC_ARCH_WAVE600
	{
		pStaTidEntry = &(pStaDbHwEntry->tid[tidIndex]);
		/*Set TXOP only if BA enabled*/
		if (pStaTidEntry->baEnable == 0)
		{
			continue;
		}
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
#if defined(ENET_INC_ARCH_WAVE600)
	HW_SEMAPHORE_LOCK(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
		HW_SEMAPHORE_LOCK(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif //ENET_INC_ARCH_WAVE600
		pStaTidEntry->txopMultipleDataEn = txopMode;

#if defined(ENET_INC_ARCH_WAVE600)
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_TID_PARAMS_WORD_1);
#else
		HW_SEMAPHORE_FREE(HW_SEMAPHORE_17_1_TID_PARAMS_WORD_1_UM);
#endif //ENET_INC_ARCH_WAVE600
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
	}
	ILOG2_DDD("StaDB_SetPrimaryStationSuTxopAndNumMpdus: SID %d, txop %d, mpdus %d", stationIndex, txopMode, numOfMpdus);
}

/**********************************************************************************
StaDB_SetMinHeFragLevel 

For WAV600D2 fragLevel is given per TID

Description:
------------
For WAV600D2 fragLevel is given per TID

Input:
-----

Output:
-------

Returns:
--------
void

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
#ifdef ENET_INC_ARCH_WAVE600D2
void StaDB_SetMinHeFragLevel(StaId staId, uint8 heFragLevel, uint8 staTid)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);

	// We want to keep the minimum frag level among all TIDs. So if new frag level is smaller than current one - save it instead.
	// This was initiated upon association.
	if (pStaDbHwEntry->tid[staTid].fragLevel > heFragLevel)
	{
		pStaDbHwEntry->tid[staTid].fragLevel = heFragLevel;
	}
}
#else
void StaDB_SetMinHeFragLevel(StaId staId, uint8 heFragLevel)
{
	StaDb_t* pStaDbHwEntry = &(StationDatabase.pStaDbHwEntries[staId]);

	// We want to keep the minimum frag level among all TIDs. So if new frag level is smaller than current one - save it instead.
	// This was initiated upon association.
	// We have a waiver here - If this min TID is closed, we still keep the low frag level value. It is reset only upon association (its better then keeping 256*8 bytes database just for it).
	if (pStaDbHwEntry->common.fragLevel > heFragLevel)
	{
		pStaDbHwEntry->common.fragLevel = heFragLevel;
	}
}
#endif //ENET_INC_ARCH_WAVE600D2
#endif //ENET_INC_ARCH_WAVE600
/**********************************************************************************
StaDb_GetStationAckEnabledSupport 

Description:
------------


Input:
-----

Output:
-------

Returns:
--------
void

**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600
MultiTidAckEn_e	StaDb_GetStationAckEnabledSupport(StaId stationIndex)
{
	return gackEnabledSupport[stationIndex];

}
#endif //ENET_INC_ARCH_WAVE600

/**********************************************************************************
StaDb_Init 

Description:
------------
This function is the Station Database general reset and initialization function. 
It is called from UMAC initialization function and performs the following steps:
Reset DB entries - memset to 0 the DB entries. Initialize DB structure.
Set number of free entries to STA_DB_MAX
Set the number of connected stations to zero

Input:
-----

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void StaDb_Init(void)
{
#if defined (ENET_INC_UMAC)
 	StaId StaIndex;
#endif

	StationDatabase.pStaDbHwEntries = StaDbHwEntries; 
	StationDatabase.NumOfConnectedSta = 0;
#if defined (ENET_INC_UMAC)
	for (StaIndex = 0; StaIndex < HW_NUM_OF_STATIONS; StaIndex++)
	{
		/* Reset SW DB and mark all entries as free */
		memset(&(StationDatabase.StaDbSwEntries[StaIndex]), 0, sizeof(StaDbSwInfo_t));
		StationDatabase.StaDbSwEntries[StaIndex].state = STA_STATE_FREE;
		/* Reset HW DB */
		memset(&(StationDatabase.pStaDbHwEntries[StaIndex]), 0, sizeof(StaDb_t));
		/* Initialize security indeces array */
		StationDatabase.pStaDbHwEntries[StaIndex].common.keyIndex0 = SECURITY_INVALID_KEY_INDEX;
		StationDatabase.pStaDbHwEntries[StaIndex].common.keyIndex1 = SECURITY_INVALID_KEY_INDEX;
		StationDatabase.pStaDbHwEntries[StaIndex].common.keyIndex2 = SECURITY_INVALID_KEY_INDEX;
		StationDatabase.pStaDbHwEntries[StaIndex].common.keyIndex3 = SECURITY_INVALID_KEY_INDEX;
		StaDb_TxopMode[StaIndex] = UMI_TXOP_MODE_ENABLED;
	}
#endif
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

void StaDB_debug_SetBF(StaId staID,uint8 BFtype)
{
	StaDb_t *pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staID]);
#if defined(ENET_INC_ARCH_WAVE600)	
	FATAL("tbd - field names are wrong");

	pStaDbHwEntry->common.tcr0Data20MhzPhyMode = BFtype;
	pStaDbHwEntry->common.tcr0Data40MhzPhyMode = BFtype;
	pStaDbHwEntry->common.tcr0Data80MhzPhyMode = BFtype;
#else //wave500 
	pStaDbHwEntry->common.tcr020MhzBfModeData = BFtype;
	pStaDbHwEntry->common.tcr040MhzBfModeData = BFtype;
	pStaDbHwEntry->common.tcr080MhzBfModeData = BFtype;
#endif 
}

#ifdef ENET_INC_ARCH_WAVE600B
void StaDB_debug_TriggerTomi(StaId staID,uint8 OMCtrl_RxNSS,uint8 OMCtrl_ChnlWidth)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staID]);
	uint32	 htHeAControl = 0;

	ILOG0_DDD("[StaDB_debug_TriggerTomi] staId %d OMCtrl_RxNSS %d OMCtrl_ChnlWidth %d", staID, OMCtrl_RxNSS, OMCtrl_ChnlWidth);

	// Update STA DB with 
	htHeAControl = 3;
	htHeAControl |= (1 << 2);
	htHeAControl |= (OMCtrl_RxNSS << 6);
	htHeAControl |= (OMCtrl_ChnlWidth << 9);
	htHeAControl |= (3 << 12);

	pStaDbHwEntry->common.htHeAControl = htHeAControl;
	
	ILOG0_D("StaDB_debug_TriggerTomi. htHeAControl 0x%x", htHeAControl);
	
	TxSelector_SetAControl(staID, TX_SELECTOR_A_CONTROL_ENABLE);
}
#endif //ENET_INC_ARCH_WAVE600B

void StaDB_debug_SetBWlimit(StaId staID,uint8 dataLimit,uint8 managementLimit)
{
	StaDb_t*          	pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[staID]);

	pStaDbHwEntry->common.dataBwLimit = dataLimit;
	pStaDbHwEntry->common._80211BwLimit = managementLimit;
}

StaState_e StaDb_getStaState(StaId stationId)
{
	return StationDatabase.StaDbSwEntries[stationId].state;
}

uint8 StaDb_getNumOfConnectedStations(void)
{
	return StationDatabase.NumOfConnectedSta;
}

void StaDb_setStaStat(StaId stationId, StaState_e state)
{
	if (StationDatabase.StaDbSwEntries[stationId].state == STA_STATE_CONNECTED && state == STA_STATE_FREE)
	{
		FATAL("wrong new state");
	}
	StationDatabase.StaDbSwEntries[stationId].state = state;
}

uint8 StaDb_getStaSupportedNss(StaId stationId)
{
	return  StaDbHwEntries[stationId].common.maxSupportedNss;
}

void StaDb_setTxopMode(StaId stationIndex, uint8 mode)
 {
	BandId_e bandId;
	StaId staIdIterator = 0;
	StaId firstSidInBand = 0;
	uint16 numOfSidsInBand = 0;

	if(stationIndex == INVALID_STA_INDEX)
	{
		for (staIdIterator = 0; staIdIterator < HW_NUM_OF_STATIONS; staIdIterator++)
		{
			 StaDb_TxopMode[staIdIterator] = mode;
		}
	}
	else
	{
		bandId = ConfigurationManager_GetBandForStation(stationIndex);
		firstSidInBand = ConfigurationManager_GetFirstSidForBand(bandId);
		numOfSidsInBand = ConfigurationManager_GetNumOfSupportedStationsForBand(bandId); 
		for (staIdIterator = firstSidInBand ; staIdIterator < firstSidInBand + numOfSidsInBand; staIdIterator++)
		{
			 StaDb_TxopMode[staIdIterator] = mode;
		}
	}
}

uint8 StaDb_getTxopMode(StaId stationIndex)
{
	return	StaDb_TxopMode[stationIndex];
}

void StaDB_SetStationAmpduLimit(StaId stationIndex, uint8 numOfMpdus)
{
	StaDb_t* pStaDbHwEntry =  &(StationDatabase.pStaDbHwEntries[stationIndex]);
	pStaDbHwEntry->common.aMpduLimit = numOfMpdus;
}

uint8 StaDB_CalcAmpduLimit (uint8 maxAmpduSize)
{
	uint8 ampduLimit = 0;
	
#ifdef ENET_INC_ARCH_WAVE600
	if (maxAmpduSize == 0)
	{
		ampduLimit = SU_HE_MAX_NUM_OF_MPDUS_IN_A_MPDU;
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_128)
	{
		ampduLimit = 96;	
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_64)
	{
		ampduLimit = 48;
	}
#else
	if (maxAmpduSize == 0)
	{
		ampduLimit = VHT_MAX_NUM_OF_MPDUS_IN_A_MPDU;
	}
#endif
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_32)
	{
		ampduLimit = 24;
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_16)
	{
		ampduLimit = 12;
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_8)
	{
		ampduLimit = 6;
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_4)
	{
		ampduLimit = 3;
	}
	else if (maxAmpduSize == TS_MANAGER_WINDOW_SIZE_2)
	{
		ampduLimit = 1;
	}
	
	else
	{
		/*Not expected value*/
		ASSERT(0);
	}
	
	return ampduLimit;
}

uint8 StaDB_SearchMaxWindowSizeOutOfTids (StaId stationIndex)
{
	uint8 tidIndex = 0;
	StaDbTid_t* pStaDbTid = (StationDatabase.pStaDbHwEntries[stationIndex].tid); 
	uint8 currentTidWindowSize;
	uint8 windowSizeMaxValue = 0;

	for(tidIndex = 0; tidIndex < NUM_OF_TIDS; tidIndex ++)
	{
		if (pStaDbTid[tidIndex].baEnable == TRUE)
		{
			currentTidWindowSize = (uint8)pStaDbTid[tidIndex].currentBaWindowLimit - 1; //0 represent unlimited value, convert it to the highest value by subscribe 1 and get wrap arrounf to the highest uint8 value

			if (currentTidWindowSize > windowSizeMaxValue)
			{
				windowSizeMaxValue = currentTidWindowSize;
			}
		}
	}

	return windowSizeMaxValue + 1; //add 1 in order to revert to the original value
}
