/***********************************************************************************
 File:		StatisticsManager.c
 Module:		StatisticsManager
 Purpose: 	 
 Description:	 
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "StatisticsManager.h"
#include "ShramStatistics.h"
#include "HostInterface_API.h"
#include "RegAccess_Api.h"
#include "TpcClbrHndlr.h"
#include "ContinuousInterfererDetection.h"
#include "ShramPhyStatDb.h"
#include "StaDatabase_Api.h"
#include "stringLibApi.h"
#include "TxMulticastHandler_API.h"
#include "TxSender_ScratchPadApi.h"
#include "SenderInterface_Api.h"
#include "PacketTrafficArbitrator_api.h"
#include "TrainingManager_Api.h"
#include "GroupManager_API.h"
#include "StatisticsPhyHandler.h"
#include "ShramPacketDescriptors.h"
#include "shramTxDesc.h"
#include "Statistics_Descriptors.h"
#include "TwtManager_API.h"
#include "ConfigurationManager_api.h"
#include "PacExtrapolator_api.h"
#include "StatisticsManager_api.h"
#include "TsManager_API.h"
#include "ErrorHandler_Api.h"
#include "CommonRamStatistics.h"
#include "BsrcCntMem_Descriptors.h"
#include "stringLibApi.h"


#define LOG_LOCAL_GID   GID_STATISTICS_MANAGER
#define LOG_LOCAL_FID   5


/*---------------------------------------------------------------------------------
/						Externals								
/----------------------------------------------------------------------------------*/

extern TwtStatistics_t TwtStatistics;
extern PtaStatistics_t PtaStatistics;
extern BsrcCntMem_t* pBsrcCntMem;
extern PlanManagerStatistics_t PlanManagerStatistics;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void statisticsManagerAddVap(K_MSG *statisticsManagerMsg);
static void statisticsManagerRemoveVap(K_MSG *statisticsManagerMsg);
static void statisticsManagerAddSta(K_MSG *statisticsManagerMsg);
static void statisticsManagerRemoveSta(K_MSG *statisticsManagerMsg);
static void statisticsManagerSendConfirmToStaManager(StaId sid);
static void statisticsManagerSendConfirmToVapManager(uint8 vapId, vapManagerEvents_e eventIndex);
static void statisticsManagerUpdateGeneralStats(void);
static void statisticsManagerAddNewSid(K_MSG *statisticsManagerMsg);
static void statisticsManagerRemoveSid(K_MSG *statisticsManagerMsg);




//---------------------------------------------------------------------------------
//						Local Function Declaration				
//---------------------------------------------------------------------------------
void triggerCopyStatisticsManager(void);

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/
// Statistics Manager parameters data base
static statisticsManagerDb_t statisticsManagerDb;

// General statistics strcut
GeneralStatistics_t GeneralStatistics;

// Statistics Manager CPU load parameters data base
StatisticsCpuLoadDb_t statisticsManagerCpuLoadDb;


/*---------------------------------------------------------------------------------
/						Ex						
/----------------------------------------------------------------------------------*/
#if defined (ENABLE_PIE_TEST)
extern uint32 TxApplicationIdleCountGetUpperMacCpuLoad(void);
#endif
/*---------------------------------------------------------------------------------
/						Consts						
/----------------------------------------------------------------------------------*/


typedef struct StatisticsFunctionEntry
{
	void (* modeHandler)(void);	
} StatisticsFunctionEntry_t;

static const StatisticsFunctionEntry_t statisticsManagerCopyStatisticsContinue[STATISTICS_COPY_STATE_TOTAL] = 
{
    triggerCopyStatisticsManager,               // iramSplitInUse FALSE
    statisticsManagerCopyStatisticsHandler, 	// iramSplitInUse TRUE
};



/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const FunctionEntry_t afpTaskTable[TASK_STATISTICS_MANAGER_END - TASK_STATISTICS_MANAGER_START]=
{
    {statisticsManagerAddVap,                       DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_ADD_VAP)},
	{statisticsManagerRemoveVap,					DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_REMOVE_VAP)},
    {statisticsManagerAddSta,                       DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_ADD_STA)},
    {statisticsManagerRemoveSta,                    DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_REMOVE_STA)},  
    {PhyStatistics_MetricsFifoNotEmpty,             DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_PHY_STAT_METRICS_FIFO_NOT_EMPTY)},
    {statisticsManagerDmaDone,                      DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_FW_STAT_DMA_DONE)}, 
	{statisticsManagerGetStatisticsDataReq, 		DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_GET_STATISTICS_REQ)},
	{PhyStatistics_SetAntennaActiveBitmap,			DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_SET_ANTENNAS_BITMAP)},
	{statisticsManagerCpuLoadReq,					DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_CPU_LOAD_REQ)},
	{PhyStatistics_HdkStatisticsCfm,				DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_HDK_STATISTICS_CFM)},
	{PhyStatistics_ChannelLoadCalculated,			DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_CHANNEL_LOAD_CALCULATED)},
	{PhyStatistics_ChannelSwitchNotify,				DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_CHANNEL_SWITCH_NOTIFY)},
    {statisticsManagerAddNewSid,					DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_ADD_NEW_SID)},
	{statisticsManagerRemoveSid,					DOUBLE_CHECK_MSG_TYPE(STATISTICS_MANAGER_REMOVE_SID)},

		
};

/*---------------------------------------------------------------------------------
/						FW scope Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************
statisticsManagerAddVap



Description:
------------

	
Input:
-----

**********************************************************************************/
static void statisticsManagerAddVap(K_MSG *statisticsManagerMsg)
{
	UMI_ADD_VAP* addVapStructurePtr;
	uint8 vapId;
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(statisticsManagerMsg)));

	addVapStructurePtr = (UMI_ADD_VAP*) pK_MSG_DATA(vapMsg);
	vapId = addVapStructurePtr->vapId; 

    // Remove all statistics of the vapId slot (last added vap (with same id) statistics)
	statisticsManagerResetAllVapStatistics(vapId);

    statisticsManagerSendConfirmToVapManager(vapId, VAP_MANAGER_ADD_VAP);
}

    
/***********************************************************************
* statisticsManagerRemoveVap
* 
* Description:
* ------------
*  
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void statisticsManagerRemoveVap(K_MSG *statisticsManagerMsg)
{
	UMI_REMOVE_VAP* removeVapStructurePtr;
	uint8 vapId;
	K_MSG *vapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(statisticsManagerMsg)));

	removeVapStructurePtr = (UMI_REMOVE_VAP*) pK_MSG_DATA(vapMsg);
	vapId = removeVapStructurePtr->vapId; 

    // Remove all statistics of the vapId slot (last added vap (with same id) statistics)
	statisticsManagerResetAllVapStatistics(vapId);

    statisticsManagerSendConfirmToVapManager(vapId, VAP_MANAGER_REMOVE_VAP);	

}

/**********************************************************************************

statisticsManagerAddNewSid



Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg - UMI_REQUEST_SID
	
**********************************************************************************/
static void statisticsManagerAddNewSid(K_MSG *statisticsManagerMsg)
{
	UMI_REQUEST_SID * reqSid = (UMI_REQUEST_SID *)pK_MSG_DATA(statisticsManagerMsg);
	StaId staId = reqSid->u16SID;

	// Remove statistics of the stationId slot (last connected station (with same id) statistics)
	statisticsManagerResetAllStationStatistics(staId, TRUE);

	OSAL_SEND_MESSAGE(UMI_MC_MAN_REQUEST_SID_CFM, TASK_UM_IF_TASK, statisticsManagerMsg, statisticsManagerMsg->header.vapId);
	
}

/**********************************************************************************
statisticsManagerRemoveSid




Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg -
	
**********************************************************************************/
static void statisticsManagerRemoveSid(K_MSG *statisticsManagerMsg)
{
	UMI_REMOVE_SID * reqSid = (UMI_REMOVE_SID *)pK_MSG_DATA(statisticsManagerMsg);
	StaId staId = reqSid->u16SID;

	// Remove statistics of the stationId slot (last connected station (with same id) statistics)
	statisticsManagerResetAllStationStatistics(staId, FALSE);
	
	OSAL_SEND_MESSAGE(UMI_MC_MAN_REMOVE_SID_CFM, TASK_UM_IF_TASK, statisticsManagerMsg, statisticsManagerMsg->header.vapId);
}


/**********************************************************************************

statisticsManagerAddSta


Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg - Station attributes(Originated from Driver)
	
**********************************************************************************/
static void statisticsManagerAddSta(K_MSG *statisticsManagerMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(statisticsManagerMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	StaId staId = pAddSta->u16SID;
	uint8 maxRssi = pAddSta->rssi;

	// Remove phy statistics of the stationId slot (last connected station (with same id) statistics
	PhyStatistics_addSta(staId,maxRssi);
	
	statisticsManagerSendConfirmToStaManager(staId);	
}

/**********************************************************************************
StatisticsHandler_RxThRemoveSta


Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg - Station attributes(Originated from Driver)
	
**********************************************************************************/
static void statisticsManagerRemoveSta(K_MSG *statisticsManagerMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(statisticsManagerMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_REMOVE *pRemoveSta = (UMI_STA_REMOVE *)pK_MSG_DATA(pMsg);
	StaId staId = pRemoveSta->u16SID;	

	PhyStatistics_removeSta(staId);
	PhyStatistics_removeStaPhyRxStatus(staId);

	statisticsManagerSendConfirmToStaManager(staId);	
}

/**********************************************************************************

statisticsManagerSendConfirmToStaManager

Description:
------------
	Send confirmation to Sta Manager 
	
Input:
-----
	clientId	- Sta Manager client Id supplied in registration
	staId	- Station Index


**********************************************************************************/
static void statisticsManagerSendConfirmToStaManager(StaId staId)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_STAT_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = staId;
	/*Send confirmation message*/
	ILOG2_DD("Statistics Manager, Send Confirmation, Client %d, SID %d", confirmMessage->clientId, staId);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

statisticsManagerSendConfirmToVapManager

Description:
------------
	Send confirmation to Sta Manager 
	
Input:
-----
	clientId	- Sta Manager client Id supplied in registration
	vapId	- Vap Index


**********************************************************************************/
static void statisticsManagerSendConfirmToVapManager(uint8 vapId, vapManagerEvents_e eventIndex)
{
	K_MSG *pMsg;
	BSS_MANAGER_CONFIRM_EVENT *confirmMessage;
    
    pMsg = OSAL_GET_MESSAGE(sizeof(BSS_MANAGER_CONFIRM_EVENT));
    confirmMessage = (BSS_MANAGER_CONFIRM_EVENT*) pK_MSG_DATA(pMsg);
   	/*Set Client ID to the registered ID*/
	confirmMessage->eventIndex = eventIndex;
   	/*Set VAP ID*/
    confirmMessage->vapId = vapId;
	confirmMessage->clientId = BSS_MANAGER_VAP_MANAGER_STATISTICS_MANAGER_CLIENT;
   	/*Send confirmation message*/
	ILOG2_D("Statistics Manager, Send Confirmation, VapId %d", vapId);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, pMsg, vapId);
}

/**********************************************************************************

statisticsManagerUpdateGeneralStats


Description:
------------
	Update Genral Stats before processing Get Status Request
	
Input:
-----


**********************************************************************************/
static void statisticsManagerUpdateGeneralStats(void)
{
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams_b0 = NULL;
	TxSender_ScratchPadApiParams_t *pScratchPadApiParams_b1 = NULL;
	
	/* This is the counter of beacons */
	pScratchPadApiParams_b0 = (TxSender_ScratchPadApiParams_t *)(B0_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));
	pScratchPadApiParams_b1 = (TxSender_ScratchPadApiParams_t *)(B1_MAC_GENRISC_TX_SPRAM_BASE_ADDR + (SCPAD_ADDRESS_TX_SENDER_SCRATCHPAD_API_STRUCTURE_START << 0x2));

	GeneralStatistics.cyclicBufferOverflow[CONFIGURATION_MANAGER_BAND_0] = CyclicBufferOverflowCounter[CONFIGURATION_MANAGER_BAND_0];
	GeneralStatistics.cyclicBufferOverflow[CONFIGURATION_MANAGER_BAND_1] = CyclicBufferOverflowCounter[CONFIGURATION_MANAGER_BAND_1];

	GeneralStatistics.beaconsTransmitted[CONFIGURATION_MANAGER_BAND_0] = pScratchPadApiParams_b0->numOfBeaconsTransmitted;
	GeneralStatistics.beaconsTransmitted[CONFIGURATION_MANAGER_BAND_1] = pScratchPadApiParams_b1->numOfBeaconsTransmitted;	


	GeneralStatistics.calibrationMask[CONFIGURATION_MANAGER_BAND_0]			= SUPPORTED_OFFLINE_CALIBRATION_MASK;
	GeneralStatistics.calibrationMask[CONFIGURATION_MANAGER_BAND_1] 		= SUPPORTED_OFFLINE_CALIBRATION_MASK;

	GeneralStatistics.onlineCalibrationMask[CONFIGURATION_MANAGER_BAND_0]	= SUPPORTED_ONLINE_CALIBRATION_MASK;
	GeneralStatistics.onlineCalibrationMask[CONFIGURATION_MANAGER_BAND_1]	= SUPPORTED_ONLINE_CALIBRATION_MASK;
}

	
/***********************************************************************
* StatisticsSetPdStatusNackInList
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
uint32 StatisticsSetPdStatusNackInList(TxPd_t *headPacketDescriptor)
{
	TxPd_t *nextPd = headPacketDescriptor;
	uint32				count = 0;
	
	while (nextPd != (TxPd_t *)NULL_PD)
	{
		/* PD was not transmitted\ACKed: set status as NACK */
		nextPd->status = PD_STATUS_NACK;
	
		/* If this is an Aggregation skip to the tail and count all PDs */
		if (nextPd->aggregationIndication)
		{
			count += nextPd->pdCounter;
			nextPd = (TxPd_t *)CONVERT_OFFSET_TO_PD(nextPd->aMsduTailPointer);			
		}
		else
		{
			/* Single MSDU */
			count += 1;
		}		
		nextPd = (TxPd_t*)GET_NEXT_PD(nextPd);		
	}

	return count;
}

/**********************************************************************************
statisticsManagerTaskEntry 

Description:
------------
	Common handler for sceduling & queueing  Lock/Disable requstes
	
Input:
-----


**********************************************************************************/
void statisticsManagerTaskEntry(K_MSG *statisticsManagerMsg)
{
	/* Use common task switching and Table */
	vTaskDispatcher(statisticsManagerMsg, afpTaskTable, TASK_STATISTICS_MANAGER_START, TASK_STATISTICS_MANAGER_END);
}


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
triggerCopyStatisticsManager


Description:
------------
    
Input:
-----
	None
Output:
-------
    None
Returns:
--------
    Calculted offset from last entry to current one.
**********************************************************************************/
void triggerCopyStatisticsManager()
{
    statisticsManagerDb.triggerManager();
}


/**********************************************************************************
statisticsManagerCopyStatisticsStart

Description:
------------
	Set relevant values necessary to copy statistics
Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerCopyStatisticsStart()
{ 
	statisticsDmaParams_t *dmaParamsStatsMangDb_p = &statisticsManagerDb.dmaParams;

	
	dmaParamsStatsMangDb_p->dmaInUse = TRUE;
	statisticsManagerDb.statisticsCopyState = STATISTICS_TRIGGER_MANAGER;
	statisticsManagerDb.splitTimesRemain = 0;

	switch (statisticsManagerDb.statisticComponent)
 	{
		case STATISTICS_RAW_DATA:
		{
		   	dmaParamsStatsMangDb_p->dmaTypeTransaction = 0;
			dmaParamsStatsMangDb_p->dmaTotal += TOTAL_NUMBER_OF_DATA_RAW_TRANSACTIONS;
			statisticsManagerDb.triggerManager = &statisticsManagerCopyStatisticsRawData;
			statisticsManagerDb.triggerCopyToShramBuffer= &statisticsManagerCopyStatisticsRawDataShramHandler;
		    break;
		}
	    case STATISTICS_MANIPULATION_DATA:
		{
	    	dmaParamsStatsMangDb_p->dmaTotal += TOTAL_NUMBER_OF_DATA_MANIPULATION_TRANSACTIONS;
			statisticsManagerDb.triggerManager = &statisticsManagerCopyStatisticsManipulationData;
			statisticsManagerDb.triggerCopyToShramBuffer= &statisticsManagerCopyStatisticsManipulationDataShramHandler;

			//made for future flexibility to copy statistics to different DDR destinations
			FATAL("STATISTICS_MANIPULATION_DATA type is unsupported currently");
			break;
	    }
	}
	
	statisticsManagerDb.triggerManager();
}

/**********************************************************************************
statisticsManagerCopyStatisticsMng

Description:
------------
	Manage level for statistics table copy
Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerCopyStatisticsManipulationData()
{
	ASSERT(0);
	// itzhaki - currently we don't support manipulation data. (STATISTICS_MANIPULATION_DATA)
}


/**********************************************************************************
statisticsManagerCopyStatisticsManipulationDataShramHandler


Description:
------------
	Statistics table is located in IRAM, therefore need to be split to chunks into
	Shram buffer.
	
    * WAVE500B we reorder the table in driver side so it will apper as in shared file.

Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
uint32 statisticsManagerCopyStatisticsManipulationDataShramHandler(uint32 sourceCurr, uint16 sizeToCopy)
{
	FATAL("statisticsManagerCopyStatisticsManipulationDataShramHandler"); //not supported currenrly, need to make sure this is correct behaviour when STATISTICS_MANIPULATION_DATA would be supported
	//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
	MEMCPY(&StatisticsShramBuffer[0], (uint32*)sourceCurr, sizeToCopy); // Copy to Sharm

	return (sizeToCopy);
}


/**********************************************************************************
statisticsManagerCopyStatisticsRawData


Description:
------------
	Manage level for statistics data copy
Input:
-----
	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerCopyStatisticsRawData(void)
{
	statisticsDmaParams_t *dmaParamsStatsMangDb_p = &statisticsManagerDb.dmaParams;
	
	while (dmaParamsStatsMangDb_p->dmaTypeTransaction < TOTAL_NUMBER_OF_DATA_RAW_TRANSACTIONS)
	{
		/* Statistic from HW congest the DMA hence shall pass via SHRAM which has faster DMA transaction (WLANRTSYS-6287) */
		if (dmaParamsStatsMangDb_p->typesArraySourceType[dmaParamsStatsMangDb_p->dmaTypeTransaction] != SOURCE_SHRAM)
    	{
			statisticsManagerCopyStatisticsToShramHandlerInit(statisticsManagerDb.typesArraySize[dmaParamsStatsMangDb_p->dmaTypeTransaction],(uint32)(statisticsManagerDb.pTypeDataList[dmaParamsStatsMangDb_p->dmaTypeTransaction]));
			dmaParamsStatsMangDb_p->dmaTypeTransaction++;
			break;
		}
	    else
	    {
    	    statisticsManagerDmaTransaction((uint32)statisticsManagerDb.pTypeDataList[dmaParamsStatsMangDb_p->dmaTypeTransaction], dmaParamsStatsMangDb_p->ddrBufferAddress, statisticsManagerDb.typesArraySize[dmaParamsStatsMangDb_p->dmaTypeTransaction], dmaParamsStatsMangDb_p->typesArrayDmaDirection[dmaParamsStatsMangDb_p->dmaTypeTransaction]);
	 	}
		dmaParamsStatsMangDb_p->dmaTypeTransaction++;
	}
}


/**********************************************************************************
statisticsManagerCopyStatisticsRawDataShramHandler


Description:
------------
	
Input:
sourceCurr, Source to copy from.
sizeToCopy, Size of chunk from source.

-----
	
Output:
-------
	None
Returns:
--------
	NOT copied size; but the size to advance source.
**********************************************************************************/
uint32 statisticsManagerCopyStatisticsRawDataShramHandler(uint32 sourceCurr, uint16 sizeToCopy)
{
	//KW_FIX_FW_G MEMCPY macro points to memcpy_s, we cannot use banned memcpy function.
	MEMCPY(&StatisticsShramBuffer[0], (uint32*)sourceCurr, sizeToCopy); // Copy to Sharm

	return (sizeToCopy);
}


/**********************************************************************************
statisticsManagerCopyStatisticsToShramHandlerInit



Description:
------------
	
Input:
-----

Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerCopyStatisticsToShramHandlerInit(uint16 sizeToTransfer, uint32 source)
{
	statisticsDmaParams_t *dmaParamsStatsMangDb_p = &statisticsManagerDb.dmaParams;
	uint16 splitTimesRemain = 0;
	
	splitTimesRemain = (sizeToTransfer / STATISTICS_MANAGER_SHRAM_BUFFER_SIZE);

	dmaParamsStatsMangDb_p->sizeToTransfer = sizeToTransfer;	// Total size of this statistic table
	dmaParamsStatsMangDb_p->dmaTotal += splitTimesRemain; 		// additional DMA transactions are needed due to the split
	statisticsManagerDb.splitTimesRemain = splitTimesRemain;
	dmaParamsStatsMangDb_p->sourceCurr = source;

	statisticsManagerCopyStatisticsHandler();
}

/**********************************************************************************
statisticsManagerCopyStatisticsIramHandler


Description:
------------
	
Input:
-----

Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerCopyStatisticsHandler(void)
{
	uint16 sizeToCopy 			= 0;
	uint32 sourceCurr 			= 0;
    uint32 sizeToAddToSource    = 0;

	statisticsDmaParams_t *statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;
	sourceCurr = statsMangDbDmaParams_p->sourceCurr;

	// check if there is only 1 more DMA activation needed (this might be the first and only one)
	if (statisticsManagerDb.splitTimesRemain  == 0) 
	{
		statisticsManagerDb.statisticsCopyState = STATISTICS_TRIGGER_MANAGER;
		sizeToCopy = (statsMangDbDmaParams_p->sizeToTransfer % STATISTICS_MANAGER_SHRAM_BUFFER_SIZE); // last chunk may not be equal in size to previous chunks

		// check if we are done witht his table shram split
		if (sizeToCopy == 0)  //  last chunk is zero.
		{
			statsMangDbDmaParams_p->dmaTotal--;
			statisticsManagerDb.triggerManager();
			return;
		}
	}
	else
	{
		sizeToCopy = STATISTICS_MANAGER_SHRAM_BUFFER_SIZE;
		statisticsManagerDb.statisticsCopyState = STATISTICS_SPLIT_ON;

		statisticsManagerDb.splitTimesRemain--;
	}

	sizeToAddToSource = statisticsManagerDb.triggerCopyToShramBuffer(sourceCurr, sizeToCopy); // trigger the copy to shram functions

	statisticsManagerDmaTransaction((uint32)(&StatisticsShramBuffer[0]), statsMangDbDmaParams_p->ddrBufferAddress, sizeToCopy, DMA_DIRECTION_SHRAM_TO_HOST);
    
	//update the source
	statsMangDbDmaParams_p->sourceCurr += sizeToAddToSource;
}


/**********************************************************************************
statisticsManagerClearQosRxSta


Description:
------------
	QoS Rx subtype is the only HW counter 16bits wide - in wave600 access to HW counters RAM is 32bit wide only
	so for these counters we must read 32bits, modify wanted 16bits and write back
	read 32bits and use shifts to insert wanted num of '0's on lower side
Input:
-----
	dst - 32bits address to read and write
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerClearQosRxSta(void *pQosRxStaCounter)
{
	uint32 readVal = *(uint32*)pQosRxStaCounter;
	uint32* addressAligned = pQosRxStaCounter;
	uint32 alignment = 0;

	// Calculate the residual size of 4 bytes alignment
	// Algin (if needed) to lower dword address
	addressAligned = (uint32*)STAT_ALIGN_DWORD((uint32)pQosRxStaCounter);

	//Check if alignment was needed or not (alignment= 0|2)
	alignment = (uint32)pQosRxStaCounter - (uint32)addressAligned;
	// We need '0' for aligned or 2 for not aligned
	alignment = 2 - alignment;

	//Read the 32 bits value
	readVal = *addressAligned;

	//Mask only the word we want to save - upper word if no alignment was needed or lower if aligned
	readVal = readVal & ((uint32)(STAT_SHORT_MASK << (alignment * STAT_SIZE_OF_BYTE)));

	//Write back the modified 32bits
	*addressAligned = readVal;
}


/**********************************************************************************

statisticsManagerResetAllStationStatisticsHostIfCounters ()

Description:
------------
 
	in D2 only, the Host I/F counters were splited by the HW team into two different structures,
		1) Host Interface QoS counters	(struct HostIfQosCounters_t)
		2) Host Interface Counters (struct HostIfCounters_t)

		This function will reset all counters related to Stations ID in D2 of both structures.

Input:  station ID
-----
		
Output: :   reset all per station counters
-------
	
Returns:    --------
Address in memory of the statistic
	
**********************************************************************************/
void statisticsManagerResetAllStationStatisticsHostIfCounters(StaId stationId)
{
	
	HostIfCounters_t *hifc = (HostIfCounters_t*)statisticsGetHostIfCountersAddress();

	hifc->staCounts[stationId].rxOutStaNumOfBytes = 0;	
	
#ifdef ENET_INC_ARCH_WAVE600B
	hifc->staCounts[stationId].agerPdNoTransmitCountSta = 0; // This field exist in 600B only
#endif
}

/**********************************************************************************

statisticsManagerResetAllStationStatisticsRxCounters

Description:
------------
 
	in D2 only, the Rx counters were splited by the HW team into two different tructurs,
		1) Rx Coordinate counters  (struct RxCoordinatorCounters_t)
		2) Rx Counters (struct RxCounters_t)

		This function will reset all counters related to Stations ID in D2 of both structures.

Input:  station ID
-----
		
Output: :   reset all per station counters
-------
	
Returns:    --------
Address in memory of the statistic
	
**********************************************************************************/
void statisticsManagerResetAllStationStatisticsRxCounters(StaId stationId)
{

	RxCounters_t *rxc = (RxCounters_t*)statisticsGetRxCountersAddress();

#ifdef ENET_INC_ARCH_WAVE600D2
			/* In D2, the statistics counter structures were splited to two different structs */
	RxCoordinatorCounters_t *rxcc = (RxCoordinatorCounters_t*)statisticsGetRxCoordinatorCountersAddress();
	rxcc->rdCount[stationId] = 0;
#else
	rxc->rdCount[stationId] = 0;					
#endif


	rxc->staCounts[stationId].rddelayed = 0;				
	rxc->staCounts[stationId].swUpdateDrop = 0;			
	rxc->staCounts[stationId].rdDuplicateDrop = 0;			
	rxc->staCounts[stationId].missingSn = 0;	

	// QoS Rx subtype is the only HW counter 16bits wide - in wave600 access to HW counters RAM is 32bit wide only
	// so for these counters we must read 32bits, modify wanted 16bits and write back
	statisticsManagerClearQosRxSta((void*)&(rxc->qosRxSta[stationId]));
}


void statisticsManagerResetAllStationStatisticsBaaCounters(StaId stationId)
{
	BaaCounters_t *baac = (BaaCounters_t*)statisticsGetBaaCountersAddress();


	baac->staCounts[stationId].mpduFirstRetransmission = 0;		
	baac->staCounts[stationId].mpduTransmitted = 0;
	baac->staCounts[stationId].mpduByteTransmitted = 0;	
#ifdef ENET_INC_ARCH_WAVE600B
	baac->staCounts[stationId].mpduRetransmission = 0;			
	baac->staCounts[stationId].channelTransmitTime = 0;	
#endif
}

/**********************************************************************************
statisticsManagerResetAllStationStatistics


Description:
------------

Input:
-----

Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/	
void statisticsManagerResetAllStationStatistics(StaId stationId, bool resetHostIFCounters)
{    
	if (resetHostIFCounters == TRUE)
	{
		statisticsManagerResetAllStationStatisticsHostIfCounters(stationId);
	}
    statisticsManagerResetAllStationStatisticsRxCounters(stationId);
    statisticsManagerResetAllStationStatisticsBaaCounters(stationId);
	statisticsManagerResetAllStationStatisticsLinkAdaptationStatistics(stationId);
	statisticsManagerResetAllStationStatisticsPerClientStatistics(stationId);
	statisticsManagerResetAllStationStatisticsPlanMangerStatistics(stationId);
	statisticsManagerResetAllStationStatisticsUpLinkBsrcStatistics(stationId);
}

void statisticsManagerResetAllStationStatisticsUpLinkBsrcStatistics(StaId stationId)
{
	UplinkBsrcTidCnt_t *UplinkBsrcTidCnt = (UplinkBsrcTidCnt_t *)statisticsGetUpLinkBsrcCountersAddress();
	
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt0 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt1 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt2 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt3 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt4 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt5 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt6 = 0;
	UplinkBsrcTidCnt->BsrcPerTidCnt[stationId].bufStsCnt7 = 0;
}

void statisticsManagerResetAllStationStatisticsPlanMangerStatistics(StaId stationId)
{
	PlanManagerStatistics.downlinkMuStats[stationId].downlinkMuType = 0;
	PlanManagerStatistics.downlinkMuStats[stationId].dlRuSubChannels = 0;
	PlanManagerStatistics.downlinkMuStats[stationId].dlRuType = 0;

	
	PlanManagerStatistics.uplinkMuStats[stationId].uplinkMuType = 0;
	PlanManagerStatistics.uplinkMuStats[stationId].ulRuSubChannels = 0;
	PlanManagerStatistics.uplinkMuStats[stationId].ulRuType = 0;
}

void statisticsManagerResetAllStationStatisticsLinkAdaptationStatistics(StaId stationId)
{
	uint32 i = 0;

	for (i = 0; i < RATES_BIT_MASK_SIZE; i++)
	{
		LinkAdaptationStatistics.ratesMask[stationId][i] = 0;			
	}

	for (i = 0; i < NUM_OF_BW; i++)
	{		
		LinkAdaptationStatistics.DataPhyMode[stationId][i] = 0;	
		LinkAdaptationStatistics.ManagementPhyMode[stationId][i] = 0;
		LinkAdaptationStatistics.powerData[stationId][i] = 0;	
		LinkAdaptationStatistics.powerManagement[stationId][i] = 0;	
		LinkAdaptationStatistics.antennaSelectionData[stationId][i] = 0;
		LinkAdaptationStatistics.antennaSelectionManagement[stationId][i] = 0;
		LinkAdaptationStatistics.scpData[stationId][i] = 0;
		LinkAdaptationStatistics.nssData[stationId][i] = 0;
		LinkAdaptationStatistics.nssManagement[stationId][i] = 0;
		LinkAdaptationStatistics.mcsData[stationId][i] = 0;
		LinkAdaptationStatistics.mcsManagement[stationId][i] = 0;
	}
	
	LinkAdaptationStatistics.stationCapabilities[stationId] = 0;
	LinkAdaptationStatistics.dataBwLimit[stationId] = 0;
	LinkAdaptationStatistics.bfModeData[stationId] = 0;
	LinkAdaptationStatistics.stbcModeData[stationId] = 0;
	LinkAdaptationStatistics.raState[stationId] = 0;
	LinkAdaptationStatistics.raStability[stationId] = 0;
	LinkAdaptationStatistics.bw[stationId] = 0;				
	LinkAdaptationStatistics.transmittedAmpdu[stationId] = 0;
	LinkAdaptationStatistics.totalTxTime[stationId] = 0;
	LinkAdaptationStatistics.mpduInAmpdu[stationId] = 0;
	LinkAdaptationStatistics.ewmaTimeNsec[stationId] = 0;				
	LinkAdaptationStatistics.sumTimeNsec[stationId] = 0;
	LinkAdaptationStatistics.numMsdus[stationId] = 0;
}

void statisticsManagerResetAllStationStatisticsPerClientStatistics(StaId staionId)
{
	/* TX */
	PerClientStatistics.retryCount[staionId] = 0;		
	PerClientStatistics.successCount[staionId] = 0;
	PerClientStatistics.exhaustedCount[staionId] = 0;
	PerClientStatistics.clonedCount[staionId] = 0;
	PerClientStatistics.oneOrMoreRetryCount[staionId] = 0;
	PerClientStatistics.packetRetransCount[staionId] = 0;
	PerClientStatistics.dropCntReasonClassifier[staionId] = 0;
	PerClientStatistics.dropCntReasonDisconnect[staionId] = 0;
	PerClientStatistics.dropCntReasonATF[staionId] = 0;
	PerClientStatistics.dropCntReasonTSFlush[staionId] = 0;
	PerClientStatistics.dropCntReasonReKey[staionId] = 0;
	PerClientStatistics.dropCntReasonSetKey[staionId] = 0;
	PerClientStatistics.dropCntReasonDiscard[staionId] = 0;
	PerClientStatistics.dropCntReasonDsabled[staionId] = 0;
	PerClientStatistics.dropCntReasonAggError[staionId] = 0;
	/* RX */
	PerClientStatistics.mpduRetryCount[staionId] = 0;	
	PerClientStatistics.mpduInAmpdu[staionId] = 0;		
	PerClientStatistics.ampdu[staionId] = 0;	
}



void StatisticsManagerResetAllVapStatisticsHostIfCounters(uint8 vapId)
{
	HostIfCounters_t *hifc = (HostIfCounters_t *)statisticsGetHostIfCountersAddress();

#ifdef ENET_INC_ARCH_WAVE600D2
	HostIfQosCounters_t *hifqosc = (HostIfQosCounters_t *)statisticsGetHostIfQosCountersAddress();
	memset32(&(hifqosc->qosTxVap[vapId][0]), 0, ACCESS_CATEGORY_NUM);
#else
	memset32(&(hifc->qosTxVap[vapId][0]), 0, ACCESS_CATEGORY_NUM);
#endif


	hifc->vapCounts[vapId].txInUnicastHd = 0;
	hifc->vapCounts[vapId].txInMulticastHd = 0;
	hifc->vapCounts[vapId].txInBroadcastHd = 0;
	hifc->vapCounts[vapId].txInUnicastNumOfBytes = 0;
	hifc->vapCounts[vapId].txInMulticastNumOfBytes = 0;
	hifc->vapCounts[vapId].txInBroadcastNumOfBytes = 0;
	hifc->vapCounts[vapId].rxOutUnicatHd = 0;
	hifc->vapCounts[vapId].rxOutMulticastHd = 0;
	hifc->vapCounts[vapId].rxOutBroadcastHd = 0;
	hifc->vapCounts[vapId].rxOutUnicastNumOfBytes = 0;
	hifc->vapCounts[vapId].rxOutMulticastNumOfBytes = 0;
	hifc->vapCounts[vapId].rxOutBroadcastNumOfBytes = 0;
	hifc->vapCounts[vapId].agerCount = 0;
}


void StatisticsManagerResetAllVapStatisticsRxCounters(uint8 vapId)
{

	RxCounters_t *rxc = (RxCounters_t*)statisticsGetRxCountersAddress();

#ifdef ENET_INC_ARCH_WAVE600D2
	RxCoordinatorCounters_t *rxcc = (RxCoordinatorCounters_t*)statisticsGetRxCoordinatorCountersAddress();
	/* in coordinator struct */
	rxcc->vapCounts[vapId].amsdu = 0;
	rxcc->vapCounts[vapId].amsduBytes = 0;
	rxcc->vapCounts[vapId].dropCount = 0;
	rxcc->vapCounts[vapId].mpduInAmpdu = 0;
	rxcc->vapCounts[vapId].octetsInAmpdu = 0;
	rxcc->vapCounts[vapId].rxCoorSecurityMismatch = 0;
#else
	rxc->vapCounts[vapId].amsdu = 0;
	rxc->vapCounts[vapId].amsduBytes = 0;
	rxc->vapCounts[vapId].dropCount = 0;
	rxc->vapCounts[vapId].mpduInAmpdu = 0;
	rxc->vapCounts[vapId].octetsInAmpdu = 0;
	rxc->vapCounts[vapId].rxCoordinatorSecurityMismatch = 0;
#endif

	/* in rx counter struct */
	rxc->rxppVapCounts[vapId].mpduUnicastOrMngmnt = 0;
	rxc->rxppVapCounts[vapId].mpduRetryCount = 0;
	rxc->rxppVapCounts[vapId].dropMpdu = 0;
	rxc->rxppVapCounts[vapId].ampdu = 0;
	rxc->rxppVapCounts[vapId].mpduTypeNotSupported = 0;
	rxc->rxppVapCounts[vapId].replayData = 0;
	rxc->rxppVapCounts[vapId].replayMngmnt = 0;
	rxc->rxppVapCounts[vapId].bcMcCountVap = 0; 

	// those 2 counters changed place between wave600 and wave600B
#if defined (ENET_INC_ARCH_WAVE600D2)
	rxcc->vapCounts[vapId].tkipCount = 0;
	rxcc->vapCounts[vapId].securityFailure = 0;
#elif defined (ENET_INC_ARCH_WAVE600B)
	rxc->vapCounts[vapId].tkipCount = 0;
	rxc->vapCounts[vapId].securityFailure = 0;
#else
	rxc->rxppVapCounts[vapId].tkipCount = 0;
	rxc->rxppVapCounts[vapId].securityFailure = 0; 
#endif
}



void StatisticsManagerResetAllVapStatisBaaCounters(uint8 vapId)
{
	BaaCounters_t *baac = (BaaCounters_t*)statisticsGetBaaCountersAddress();


	baac->vapCounts[vapId].rtsSuccessCount = 0;
	baac->vapCounts[vapId].rtsFailure = 0;
	baac->vapCounts[vapId].transmitStreamRprtMSDUFailed = 0;
	baac->vapCounts[vapId].qosTransmittedFrames = 0;
	baac->vapCounts[vapId].transmittedAmsdu = 0;
	baac->vapCounts[vapId].transmittedOctetsInAmsdu = 0;
	baac->vapCounts[vapId].transmittedAmpdu = 0;
	baac->vapCounts[vapId].transmittedMpduInAmpdu = 0;
	baac->vapCounts[vapId].transmittedOctetsInAmpdu = 0;
	baac->vapCounts[vapId].beamformingFrames = 0;
	baac->vapCounts[vapId].ackFailure = 0;
	baac->vapCounts[vapId].failedAmsdu = 0;
	baac->vapCounts[vapId].retryAmsdu = 0;
	baac->vapCounts[vapId].multipleRetryAmsdu = 0;
	baac->vapCounts[vapId].amsduAckFailure = 0;
	baac->vapCounts[vapId].implicitBarFailure = 0;
	baac->vapCounts[vapId].explicitBarFailure = 0;
	baac->vapCounts[vapId].transmitStreamRprtMultipleRetryCount = 0;
	baac->vapCounts[vapId].transmitBw20 = 0;
	baac->vapCounts[vapId].transmitBw40 = 0;
	baac->vapCounts[vapId].transmitBw80 = 0;
	baac->vapCounts[vapId].transmitBw160 = 0;
	baac->rxGroupFrame[vapId] = 0;
	baac->txSenderError[vapId] = 0; 
}

    
/***********************************************************************
* statisticsManagerResetAllVapStatistics
* 
* Description:
* ------------
* This function resets all the statistics related to vap in Host I/F RX and BAA counters
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void statisticsManagerResetAllVapStatistics(uint8 vapId)
{
	StatisticsManagerResetAllVapStatisticsHostIfCounters(vapId);
	StatisticsManagerResetAllVapStatisticsRxCounters(vapId);
	StatisticsManagerResetAllVapStatisBaaCounters(vapId);
}


/* ----------------------------------------------- */
/* --------------------Massages------------------- */
/* ----------------------------------------------- */


/***********************************************************************
* statisticsManagerGetStatisticsDataReq
* 
* Description:
* ------------
* This function collects the phy stats via PhyStatistics_GetPhyStatusReq and when the results are returned it triggers DMA from SHRAM to DDR
* 
* Input:
* ------
* msg from Driver "UM_DBG_GET_STATISTICS_REQ"
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void statisticsManagerGetStatisticsDataReq(K_MSG* msg_p)
{
	UMI_GET_STATISTICS_DATA *GetStatisticsReqMsgData_p  = (UMI_GET_STATISTICS_DATA*)(pK_MSG_DATA(msg_p));
	statisticsManagerDb_t *statsMangDb_p = &statisticsManagerDb;
	statisticsDmaParams_t *statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;

	ASSERT(GetStatisticsReqMsgData_p->length >= statsMangDb_p->totalSizeToAllocate); // ASSERT when length of statistics is greater than allocated size in Driver
	ASSERT(statsMangDbDmaParams_p->pMcMsg == NULL);		// assert if previous message was not served 
	ASSERT(statsMangDbDmaParams_p->dmaInUse == FALSE);	// assert If getting a "get statistics" msg but didn't finished the last requast .

	statsMangDbDmaParams_p->pMcMsg = msg_p; // keep msg for confirm
	statsMangDbDmaParams_p->ddrBufferAddress 		= GetStatisticsReqMsgData_p->ddrBufferAddress; // Keep destanation address to copy statistics to Host (modifided on the fly)
	statsMangDbDmaParams_p->ddrBufferStartAddress 	= GetStatisticsReqMsgData_p->ddrBufferAddress; // Keep destanation address to copy statistics to Host

	statisticsManagerUpdateGeneralStats();
	
	statsMangDb_p->statisticComponent = STATISTICS_RAW_DATA;		

	PhyStatistics_GetPhyStatusReq();
}

/**********************************************************************************
statisticsManagerGetStatisticsCfm  

Description:
------------
	trigger DMA to copy statistics from Shram to DDR
Input:
-----
	msg from Driver "UM_DBG_GET_STATISTICS_REQ"
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerGetStatisticsDataCfm(K_MSG *msg_p, uint8 status)
{
	UMI_GET_STATISTICS_DATA *getStatisticsCfmMsgData_p  = (UMI_GET_STATISTICS_DATA*)(pK_MSG_DATA(msg_p));

	getStatisticsCfmMsgData_p->status = status;
	
	OSAL_SEND_MESSAGE(UMI_MC_MAN_GET_STATISTICS_CFM , TASK_UM_IF_TASK, msg_p, msg_p->header.vapId); // send confirmation to driver
}

    
/***********************************************************************
* statisticsManagerCpuLoadReq
* 
* Description:
* ------------
* Driver is requesting the CPU load of : UPPER MAC, LOWER MAC and 3 Genriscs.
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void statisticsManagerCpuLoadReq(K_MSG *pMsg)
{
	/* Send signal to umac task with CPU load values */
	UmiCpuLoadCfm_t *umiCpuLoadCfm = (UmiCpuLoadCfm_t*)pK_MSG_DATA(pMsg);

	umiCpuLoadCfm->upperMac 	= statisticsManagerCpuLoadDb.load[STATISTICS_CPU_LOAD_CPUS_UPPERMAC];
	umiCpuLoadCfm->lowerMac		= statisticsManagerCpuLoadDb.load[STATISTICS_CPU_LOAD_CPUS_LOWERMAC];

	OSAL_SEND_MESSAGE(UMI_MC_MAN_CPU_LOAD_CFM, TASK_UM_IF_TASK, pMsg, pMsg->header.vapId);
}



/* ----------------------------------------------- */
/* ----------------------DMA---------------------- */
/* ----------------------------------------------- */
/**********************************************************************************
statisticsManagerDmaTransaction

Description:
------------
	statistics Manager DMA transaction initiator
Input:
-----
	*source		 - > Pointer to the copied data address
	*destination - > FW Mips prespective on the address to copy to
	length       - > Length of data to copy from source to destination
	direction    - > an  "dmaDirection_e" type of copy direction to use (e.g. DMA_DIRECTION_SHRAM_TO_HOST).
Output:
-------
	None
Returns:
--------
	None
Notes:
--------
	Address conversion to DMA prespective (BAR0) is done in DMA driver according to 
	typesArrayDmaDirection[] that correspond to TypesTable;
**********************************************************************************/
void statisticsManagerDmaTransaction(uint32 source, uint32 destination, uint16 length ,uint8 direction)
{
	dmaDesc_t* pDmaDesc = NULL;
	statisticsManagerDb_t *statsMangDb_p = &statisticsManagerDb;
	statisticsDmaParams_t *statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;
    
	ASSERT((statsMangDbDmaParams_p->ddrBufferStartAddress + statsMangDb_p->totalSizeToAllocate) >= ((uint32)destination + length)); // Check there is no memory access violation 
	
	pDmaDesc = DmaManager_AllocateDesc();
	// fill in DMA descriptor
	pDmaDesc->clientId = statsMangDbDmaParams_p->dmaClientId;
	pDmaDesc->destination = destination;
	pDmaDesc->direction = direction;
	pDmaDesc->source = source;
	pDmaDesc->length = length;

	DmaManager_NewDmaRequest(pDmaDesc); // trigger DMA 	 
	
	// Update states
	statsMangDbDmaParams_p->dmaReq++;  // one more active transaction
	statsMangDbDmaParams_p->ddrBufferAddress += length; // Update next iteratation DDR location

}

/**********************************************************************************
statisticsManagerDmaDone

Description:
------------
	Statistics Manager DMA done, received after DMA transactions (one or many) are done.
	Pay attention: the msg is not for every transaction, passable to send for a couple.
Input:
-----
	msg DMA manager
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerDmaDone(K_MSG* msg_p)
{
	TX_INTERRUPT_SAVE_AREA;	
	
	uint32 dmaManagerTransferCounter = 0;
	statisticsDmaParams_t*	statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;
	UNUSED_PARAM(msg_p);
		
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	// Get current DMA counter (if more will be added, we will handle it in the next MSG)
	dmaManagerTransferCounter = DmaManager_GetTransferCount(statsMangDbDmaParams_p->dmaClientId);

	// Allow DMA Manager to send MSG again (it was disabled since the MSG was sent)
	EventManager_TurnOnEvent(EVENT_ID_FW_STAT_DMA_DONE);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	
	if ((statsMangDbDmaParams_p->dmaReq != statsMangDbDmaParams_p->dmaTotal)// Check if we already triggered all the DMA activations we need.
		&& (statsMangDbDmaParams_p->dmaReq == dmaManagerTransferCounter))	// Init next transaction only when all previous are done, to not override the shram buffer.
	{
		// continue to trigger more DMA transactions
		statisticsManagerCopyStatisticsContinue[statisticsManagerDb.statisticsCopyState].modeHandler();
	}
	else if (dmaManagerTransferCounter == statsMangDbDmaParams_p->dmaTotal) // All transaction are done DMAing
	{
		// Done using DMA
		statsMangDbDmaParams_p->dmaInUse = FALSE;

        // Confirm to HOST all DMA transactions are DONE.
		statisticsManagerGetStatisticsDataCfm(statsMangDbDmaParams_p->pMcMsg, UMI_OK);
        statsMangDbDmaParams_p->pMcMsg = NULL;

		// Put phy stats back to regular state
		PhyStatistics_DmaDone();
    }
}


/**********************************************************************************
statisticsManagerResetAllStatistics



Description:
------------
	Reset all statistics to ZERO
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void statisticsManagerResetAllStatistics(void)
{
	statisticsDmaParams_t *statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;
	uint32 i;

	
	for (i = STATISTIC_TYPE_FIRST_STATISTIC; i < STATISTIC_TYPE_TOTAL ; i++)
	{
		//HW mem supports only 4bytes access, SHRAM and IRAM not
		if((statsMangDbDmaParams_p->typesArraySourceType[i] == SOURCE_SHRAM) || (statsMangDbDmaParams_p->typesArraySourceType[i] == SOURCE_IRAM))
		{
            memset(statisticsManagerDb.pTypeDataList[i], 0x0, statisticsManagerDb.typesArraySize[i]);
		}
		else
		{
            memset32(statisticsManagerDb.pTypeDataList[i], 0x0, CONVERT_BYTES_TO_WORDS(statisticsManagerDb.typesArraySize[i]));
		}
	}
}


/**********************************************************************************
statisticsManagerPostInit

Description:
------------
	Register to DMA
Input:
-----
	
Output:
-------
	None
Returns:
--------
	DMA client ID, used for every new transaction
**********************************************************************************/
void statisticsManagerPostInit()
{
	dmaRegistration_t dmaRegister;
	statisticsDmaParams_t *statsMangDbDmaParams_p = &statisticsManagerDb.dmaParams;
	
	// DMA manager registration
	dmaRegister.priority = DMA_PRIORITY_0;
	dmaRegister.eventId = EVENT_ID_FW_STAT_DMA_DONE;
	
	statsMangDbDmaParams_p->dmaClientId = DmaManager_Register(&dmaRegister);
}


/**********************************************************************************
statisticsManagerInit

Description:
------------
	initialization


**********************************************************************************/
void statisticsManagerInit(void)
{
	uint8 i;
		
    // External Statistics modules
    StatisticsHandler_Init();
	PhyStatistics_Init();

	// init helpers to zeros
	memset(&statisticsManagerDb, 0, sizeof(statisticsManagerDb_t)); 

	memset(&CyclicBufferOverflowCounter, 0, sizeof(CyclicBufferOverflowCounter)); 
	    
	// point pTypeTableList to it relevent TypeTable, point pTypeDataList to it relevent Data
#ifdef ENET_INC_ARCH_WAVE600D2
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_HOSTIF_QOS_COUNTERS] = (uint8*) statisticsGetHostIfQosCountersAddress();
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_HOSTIF_QOS_COUNTERS] = sizeof(HostIfQosCounters_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_RX_COORDINATOR_COUNTERS] = (uint8*) statisticsGetRxCoordinatorCountersAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_RX_COORDINATOR_COUNTERS] = sizeof(RxCoordinatorCounters_t);
#endif
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_HOSTIF_COUNTERS] = (uint8*) statisticsGetHostIfCountersAddress();
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_HOSTIF_COUNTERS] = sizeof(HostIfCounters_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_RX_COUNTERS] = (uint8*) statisticsGetRxCountersAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_RX_COUNTERS] = sizeof(RxCounters_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_BAA_COUNTERS] = (uint8*) statisticsGetBaaCountersAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_BAA_COUNTERS] = sizeof(BaaCounters_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_UPLINK_BSRC_COUNTERS] = (uint8*) statisticsGetUpLinkBsrcCountersAddress();
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_UPLINK_BSRC_COUNTERS] = sizeof(UplinkBsrcTidCnt_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TS_MANAGER_INITIATOR_GLOBAL_TID_STATS] = (uint8*)tsManagerGetInitiatorTidGlobalStatisticsAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TS_MANAGER_INITIATOR_GLOBAL_TID_STATS] = sizeof(TsManagerInitiatorTidGlobalStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TS_MANAGER_INITIATOR_GLOBAL_STA_STATS] = (uint8*)tsManagerGetInitiatorStationGlobalStatisticsAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TS_MANAGER_INITIATOR_GLOBAL_STA_STATS] = sizeof(TsManagerInitiatorStationGlobalStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TS_MANAGER_RECIPIENT_GLOBAL_TID_STATS] = (uint8*)tsManagerGetRecipientTidGlobalStatisticsAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TS_MANAGER_RECIPIENT_GLOBAL_TID_STATS] = sizeof(TsManagerRecipientTidGlobalStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TS_MANAGER_RECIPIENT_GLOBAL_STA_STATS] = (uint8*)tsManagerGetRecipientStationGlobalStatisticsAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TS_MANAGER_RECIPIENT_GLOBAL_STA_STATS] = sizeof(TsManagerRecipientStationGlobalStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_LINK_ADAPTATION_STATS] =  (uint8*)(&LinkAdaptationStatistics);	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_LINK_ADAPTATION_STATS] = sizeof(LinkAdaptationStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_MULTICAST_STATS] = (uint8*)MulticastHandler_GetStatistics_Address();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_MULTICAST_STATS] = sizeof(MulticastStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TRAINING_MANAGER_STATS] = (uint8*)TrainingManagerGetStatisticsAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TRAINING_MANAGER_STATS] = sizeof(TrainingManagerStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_GROUP_MANAGER_STATS] = (uint8*)GroupManagerGetGroupDbAddress();	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_GROUP_MANAGER_STATS] = sizeof(GroupManagerStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_LINK_ADAPTATION_MU_STATS] = (uint8*)(&LinkAdaptationMuStatistics);	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_LINK_ADAPTATION_MU_STATS] = sizeof(LinkAdaptationMuStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_PTA] = (uint8*)&PtaStatistics;	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_PTA] = sizeof(PtaStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_TWT_STATS] = (uint8*)&TwtStatistics;	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_TWT_STATS] = sizeof(TwtStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_PLAN_MANAGER_STATS] = (uint8*)&PlanManagerStatistics;
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_PLAN_MANAGER_STATS] = sizeof(PlanManagerStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_ALPHA_FILTER_STATS] = (uint8*)(&pacExtrapolatorRssi);
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_ALPHA_FILTER_STATS] = sizeof(AlphaFilterStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_PER_CLIENT_TRANSMIT_COUNT] = (uint8*)(&PerClientStatistics);
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_PER_CLIENT_TRANSMIT_COUNT] = sizeof(PerClientStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_GENERAL_STATS] = (uint8*)(&GeneralStatistics);	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_GENERAL_STATS] = sizeof(GeneralStatistics_t);

	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_DYNAMIC_BW_STATS] = (uint8*)(&DynamicBwStatistics);	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_DYNAMIC_BW_STATS] = sizeof(DynamicBwStatistics_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_RX_PHY_STATUS_STATS] = (uint8*)(&phyRxStatusDb);	
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_RX_PHY_STATUS_STATS] = sizeof(wholePhyRxStatusDb_t);
	
	statisticsManagerDb.pTypeDataList[STATISTIC_TYPE_CHANNEL_STATS] = (uint8*)(&ChannelStatistics);
	statisticsManagerDb.typesArraySize[STATISTIC_TYPE_CHANNEL_STATS] = sizeof(channelStatistics_t);

	// Make sure each statistics struct is a 32 bit multiplication This is done for DMA alignment for 4 bytes copy.

	for (i = STATISTIC_TYPE_FIRST_STATISTIC; i < STATISTIC_TYPE_TOTAL; i++)
	{
		ASSERT((statisticsManagerDb.typesArraySize[i] & STAT_FOUR_BYTES_ALIGN_MASK) == 0); 

		statisticsManagerDb.totalSizeToAllocate += statisticsManagerDb.typesArraySize[i];      // Calculate: total statistics data size.
    
    	Utils_determineSourceTypeAndDmaDirection((uint32)statisticsManagerDb.pTypeDataList[i], &(statisticsManagerDb.dmaParams.typesArrayDmaDirection[i]), &(statisticsManagerDb.dmaParams.typesArraySourceType[i]));
	}
  
	StatisticsCalcLinkAdaptationOffsets();

 	statisticsManagerResetAllStatistics();

   	// Set debug to known value, to be checked in console.
   		
	for(uint8 bandId = CONFIGURATION_MANAGER_BAND_0; bandId < NUM_OF_CONFIGURATION_MANAGER_BANDS ; bandId++)
	{
		GeneralStatistics.debugStatistics[bandId] = DEBUG_STAT_INIT_VALUE;
	}	
	
}



