/***********************************************************************************
 File:		System_CoreTimers.c
 Module:		
 Purpose:		
 Description:	
************************************************************************************/


/*---------------------------------------------------------------------------------
/						Includes
/----------------------------------------------------------------------------------*/
#include "System_GlobalDefinitions.h"
#include "System_CoreTimersApi.h"
#if defined (CPU_ARC)
#include "InterruptManager_Api.h"
#include "ErrorHandler_Api.h"
#include "arc_reg.h"
#include "loggerAPI.h"
#include "stringLibApi.h"
/*---------------------------------------------------------------------------------
/						Defines
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_SYS
#define LOG_LOCAL_FID 	4

//interrupt numbers
#define INTERRUPT_NUM_TIMER_0 (16)
#define INTERRUPT_NUM_TIMER_1 (17)

//interrupt timers priority
#define INTERRUPT_TIMER_PRIORITY (0)

/*---------------------------------------------------------------------------------
/						Macros
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Data Type Definition
/----------------------------------------------------------------------------------*/

typedef struct CoreTimerDb
{
	System_coreTimerCallback	callBackFunction;
	bool 						isOneShotTimer;	
	bool						isTimerActive;
} CoreTimerDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration
/----------------------------------------------------------------------------------*/
void System_CoreTimersCallBack(systemCoreTimerId_e timerId);
void System_CoreTimersEnableTimer(systemCoreTimerId_e timerId);
void System_CoreTimersSetTimerLimit(systemCoreTimerId_e timerId,uint32 timerLimit);

/*---------------------------------------------------------------------------------
/						Static Variables
/----------------------------------------------------------------------------------*/
static CoreTimerDb_t coreTimerDb[SYSTEM_CORE_NUM_OF_TIMERS];

/*---------------------------------------------------------------------------------
/						Static Functions Definitions
/----------------------------------------------------------------------------------*/



/**********************************************************************************

System_CoreTimersSetTimerLimit


Description: update the timer limit
------------

Input: timerId (timer 0 or timer 1)
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersSetTimerLimit(systemCoreTimerId_e timerId, uint32 timerLimit)
{
	if (timerId == SYSTEM_CORE_TIMER_0)
	{
		_sr(timerLimit, REG_LIMIT0);
	}
	else
	{
		_sr(timerLimit, REG_LIMIT1);
	}
}
/**********************************************************************************

System_CoreTimersCallBack


Description: call to the timer caêlback function once the timer inerrupt occured 
------------

Input:timerId (timer 0 or timer 1)
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersCallBack(systemCoreTimerId_e timerId)
{
	if (coreTimerDb[timerId].callBackFunction)
	{
		coreTimerDb[timerId].callBackFunction();
	}
	else
	{
		ASSERT(0);
	}	
}
/**********************************************************************************

System_CoreTimersEnableTimer


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersEnableTimer(systemCoreTimerId_e timerId)
{
	
	if (timerId == SYSTEM_CORE_TIMER_0)
	{
		//enable timer 0 interrupt in non-halt mode
		_sr(REG_CONTROL0_IE_BIT | REG_CONTROL0_NH_BIT, REG_CONTROL0);
	}
	else
	{
		//enable timer 1 interrupt in non-halt mode
		_sr(REG_CONTROL1_IE_BIT | REG_CONTROL1_NH_BIT, REG_CONTROL1);
	}


}
/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
	
/**********************************************************************************

System_CoreTimersResetTimerCount


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersResetTimerCount(systemCoreTimerId_e timerId)
{
	if (timerId == SYSTEM_CORE_TIMER_0)
	{
		_sr(0, REG_COUNT0);
	}
	else
	{
		_sr(0, REG_COUNT1);
	}
}

/**********************************************************************************

System_CoreTimersDisableTimer


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersDisableTimer(systemCoreTimerId_e timerId)
{
	ASSERT(timerId < SYSTEM_CORE_NUM_OF_TIMERS);
	
	coreTimerDb[timerId].isTimerActive = FALSE;
	
	if (timerId == SYSTEM_CORE_TIMER_0)
	{
		_sr(0, REG_CONTROL0);
	}
	else
	{
		_sr(0, REG_CONTROL1);
	}
}

/**********************************************************************************

System_CoreTimersIsrHandlerTimer0


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
ISR_VOID System_CoreTimersIsrHandlerTimer0(void)
{


	if (coreTimerDb[SYSTEM_CORE_TIMER_0].isOneShotTimer == FALSE)
	{		
		_sr(0, REG_COUNT0);
		_sr(REG_CONTROL0_IE_BIT | REG_CONTROL0_NH_BIT, REG_CONTROL0);
	}
	else
	{
		_sr(0, REG_CONTROL0);
	}
	System_CoreTimersCallBack(SYSTEM_CORE_TIMER_0);
}
/**********************************************************************************

System_CoreTimersIsrHandlerTimer1


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
ISR_VOID System_CoreTimersIsrHandlerTimer1(void)
{
	if (coreTimerDb[SYSTEM_CORE_TIMER_1].isOneShotTimer == FALSE)
	{		
		_sr(0, REG_COUNT1);
		_sr(REG_CONTROL1_IE_BIT | REG_CONTROL1_NH_BIT, REG_CONTROL1);
	}
	else
	{
		_sr(0, REG_CONTROL1);	
	}
	System_CoreTimersCallBack(SYSTEM_CORE_TIMER_1);

}
/**********************************************************************************

System_CoreTimersSetTimer


Description: 
------------

Input: 	timer id (0 or 1)
		oneShot indication 
		timer duration in MS the max timer duration is limited to 13000 ms in csase of working with 320Mhz
		callback function (the callback function should send MSG to the releavent task its in interrupt context)
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersSetTimer(systemCoreTimerId_e timerId, bool oneShot, uint32 durationInMs, System_coreTimerCallback callbackFunction)
{

	ASSERT(timerId < SYSTEM_CORE_NUM_OF_TIMERS);

	if (coreTimerDb[timerId].isTimerActive == FALSE)
	{
		coreTimerDb[timerId].isTimerActive = TRUE;
		coreTimerDb[timerId].isOneShotTimer = oneShot;
		coreTimerDb[timerId].callBackFunction= callbackFunction;

		System_CoreTimersSetTimerLimit(timerId,CONVERT_MS_TO_SYS_TICK(durationInMs));
		System_CoreTimersResetTimerCount(timerId);
		System_CoreTimersEnableTimer(timerId);
	}
	else
	{
		ASSERT(0);
	}
}
/**********************************************************************************

System_CoreTimersInit


Description:
------------

Input:
-----

Output:
-------

Returns:
--------

**********************************************************************************/
void System_CoreTimersInit(void)
{
 	memset(coreTimerDb,0x0,sizeof(CoreTimerDb_t)*SYSTEM_CORE_NUM_OF_TIMERS);
	
	//disable time 0
	_sr(0, REG_CONTROL0);
	//disable time 1
	_sr(0, REG_CONTROL1);

	//set interrupt timer 0 priority
	_sr(INTERRUPT_NUM_TIMER_0, IRQ_INTERRUPT);
	_sr(INTERRUPT_TIMER_PRIORITY, IRQ_PRIORITY);

	//set interrupt timer 1 priority
	_sr(INTERRUPT_NUM_TIMER_1, IRQ_INTERRUPT);
	_sr(INTERRUPT_TIMER_PRIORITY, IRQ_PRIORITY);

}


#endif //defined (CPU_ARC)



