/************************************************************************************
*    
*    File:		  System_Timers.c
*    Class/Module: 
*    Description:  A detailed description of the Module, its  purpose attributes and
*		 	  whatever information the user & maintainer might find valuable.
*
*    COPYRIGHT: 
*        (C) Lantiq Israel Ltd.
*        All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*        copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/


/*************************************************************************/
/***                        Include Files                              ***/
/*************************************************************************/
#include "System_GlobalDefinitions.h"
#include "Debug_Trace.h"
#include "OSAL_Api.h"
#include "System_Timers.h"
#include "InterruptManager_Api.h"
#include "lm.h"
#include "CpuLoad_Api.h"
#include "MT_MAC_HT_extensions_regs.h"
#include "MT_Emerald_Env_regs.h"
#include "MacHtExtensionsRegs.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "Pac_Api.h"
#include "lm_statistics.h"
#include "InterruptManager_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "lmtimerUtility.h"
#if defined (ENET_INC_UMAC)
#include "PacketTrafficArbitrator_api.h"
#endif
/*************************************************************************/
/***                        Local Defines                              ***/
/*************************************************************************/
#define LOG_LOCAL_GID   GLOBAL_GID_SYS
#define LOG_LOCAL_FID 6


#define US_TO_PRESCALER_FACTOR   (2)  //prescaler is configured to 0.5 microsec. desired time is uSec should be multiplied by 2.
#define US_TO_PRESCALER_UNIT(us)   (uint32)(us*US_TO_PRESCALER_FACTOR)  //prescaler configured to 0.5 microsec

#define CPU_LOAD_MIN_TIME_BETWEEN_COMPUTATIONS	(25000)	/* Prevent CPU load calculation at periods below 25 mSec. */


/*************************************************************************/
/***               Public Functions                                    ***/
/*************************************************************************/


/**********************************************************************************
MT_SetModeTimer


Description:
------------
Set the mode (one shot / multi shot) and enable/disable a timer.

Input:
-----
	timer ID
	Enable / Disable
	Shot mode (one shot / multi shot)
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_SetModeTimer(uint32 timerId, uint32 enableTimer, uint32 shotMode)	
{
	TX_INTERRUPT_SAVE_AREA;

	uint32 data;
	uint32 timerOneshotMode;
	uint32 timerEnable;


	// Calculate mask for one shot mode for this timer
	timerOneshotMode = 0x1 << timerId;
	// Calculate mask for "enable bit" for this timer
	timerEnable = 0x10 << timerId;

	
	if (enableTimer == MT_TIMER_ENABLE)
	{
		if (shotMode == MT_TIMER_MULT_SHOT) /* enable timer in multi shot mode */
		{
		  data =  timerEnable | timerOneshotMode;
		}
		else  /* enable timer in one shot mode */
		{
          data = timerEnable;
		}
	}
    else
    {
	    data = 0; /* disable timer */
    }

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	RegAccess_WriteMasked(REG_EMERALD_ENV_TIMERS_CONFIG, (timerEnable | timerOneshotMode), data);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************
MT_PollTimer

Description:
------------
Poll on a timer until it expires (not recommanded!)

Input:
-----
	timer ID
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_PollTimer(uint32 timerId)
{
	uint32 timerExpireMask;

	timerExpireMask = 0x1 << timerId;	
	while( (RegAccess_ReadImmediate(REG_EMERALD_ENV_TIMERS_STATUS) & timerExpireMask) == 0 );
}



/**********************************************************************************
MT_SetTimeTimer

Description:
------------
Set time period for timer

Input:
-----
	timer ID
	Expiration time
Output:
-------
	
Returns:
--------
		
**********************************************************************************/
 void MT_SetTimeTimer(uint32 timerId, uint32 maxCount)	
{
	RegEmeraldEnvTimer0MaxCount_u 	timerMaxCount;
	uint32							timerMaxCountReg;

	/* On Gen 5 all timers has prescalers*/
	timerMaxCount.bitFields.timer0MaxCount = US_TO_PRESCALER_UNIT(maxCount);	

    //Configure the timer period
	timerMaxCountReg = REG_EMERALD_ENV_TIMER0_MAX_COUNT + CONVERT_WORDS_TO_BYTES(timerId);
    RegAccess_Write(timerMaxCountReg, timerMaxCount.val);
}

/****************************************************************************
 **
 ** NAME:           MT_ClearIsrTimer
 **
 ** PARAMETERS:     timer id,
 **                 
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:    Clear interrupt source for  timer 
 ****************************************************************************/
void MT_ClearIsrTimer(uint32 timerId)	
{
	uint32 timerExpireMask;

	timerExpireMask = 0x1 << timerId;
	//Clear interrupt source (clear by write '1' to relevant bit). We must not touch the other bits since it can clear other interrupts.
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_STATUS, timerExpireMask);
}


#ifdef ENET_INC_UMAC
/**********************************************************************************

isr_Timers_TimerA


Description:
------------
Handle the event that timerA has expired (timer0 and/or timer1)

Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
ISR_VOID isr_Timers_TimerA(void)
{
	RegEmeraldEnvTimersStatus_u timersStatusRegister;
	RegEmeraldEnvTimersStatus_u timersStatusCopyRegister;

	timersStatusRegister.val = 0;
	timersStatusCopyRegister.val = 0; 
	 
    RegAccess_Read(REG_EMERALD_ENV_TIMERS_STATUS, &timersStatusRegister.val);
	timersStatusCopyRegister.val = timersStatusRegister.val;
	timersStatusRegister.val = 0;
	
    if(timersStatusCopyRegister.bitFields.timer0Expired)
    {
#if defined RADAR_DETECTION_ENABLED
		timersStatusRegister.bitFields.timer0Expired = TRUE;
        OSAL_SEND_NO_DATA_MESSAGE(HDK_RADAR_DETECTION_TIMER, TASK_HDK, VAP_ID_DO_NOT_CARE);
#else
		FATAL("Radar Detection Timer isn't used in 2.4G");
#endif
	}
    if(timersStatusCopyRegister.bitFields.timer1Expired)
    {
#ifdef PTA_BUILD_IN_PLAT
		timersStatusRegister.bitFields.timer1Expired = TRUE;
		Isr_MipsTimer1();
#endif		
    }    
    RegAccess_Write(REG_EMERALD_ENV_TIMERS_STATUS, timersStatusRegister.val);
}

#endif /* #ifdef ENET_INC_UMAC */

/**********************************************************************************
MT_enableDisableGpTimerIrq

Description:
------------
Enable  Disable the interrupt for a specfic timer

Input:
-----
	timer ID
	Enable / Disable IRQ
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_enableDisableGpTimerIrq(uint32 timerID, bool isEnable)
{
	RegEmeraldEnvTimersIrqEn_u regEmeraldEnvTimersIrqEn;


	RegAccess_Read(REG_EMERALD_ENV_TIMERS_IRQ_EN, &regEmeraldEnvTimersIrqEn.val);

	if (isEnable == 1)
	{
		/*Enable timer IRQ*/
		regEmeraldEnvTimersIrqEn.val |=  (1 << timerID);
	}
	else
	{
		/*Disable timer IRQ*/
		regEmeraldEnvTimersIrqEn.val &= (~(1 << timerID));
		
	}
	
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_IRQ_EN, regEmeraldEnvTimersIrqEn.val);
}

/*------------------------------------------------------------------------------
* Function name:	MT_vINTTimer1_PromptScheduler
* Description:	  dummy function - does nothing.
*					the only reason for this interrupt is to make CPU move out
*					of "wait" state and run scheduler again.
*					this should take care of a known issue in :
*					there is no atomic "enable interrupts and enter wait state"
*					assembly commands. 
*					in our scheduler, interrupts are used to queue messages in 
*					scheduler's queue. the scheduler, which is implemented as a 
*					busy loop, checks repeatedly for any queued messages. if 
*					there aren't any, scheduler calls the "wait" command.
*					the problem may rise when an interrupt occurs after interrupts
*					are enabled, which puts a message in queue to be scheduled, 
*					but since scheduler already checked for pending messages, 
*					"wait" will still be called.
*					by setting a multi-shot timer to pop every 10msec we make sure
*					CPU doesn't stay in "wait" for too long.
*					
* PARAMETERS:	   none
* RETURN VALUES:	none
------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC)
ISR_VOID MT_vINTTimer1_PromptScheduler(void)
{
	MT_ClearIsrTimer(MT_TIMER_1);

	ACCUMULATE_CPU_IDLE_TIME();
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_MT_vINTTimer1_PromptScheduler,0,0);
	//interrupt does nothing - it's only here to wake up CPU
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_MT_vINTTimer1_PromptScheduler,1,1);
}
#endif


/**********************************************************************************
updatePrescalersForFullAndReducedFrequencyMode

Description:
------------
Initializae the prescalers of all 4 timers to be 0.5 usec resolution

Input:
-----

Output:
-------

Returns:
--------
		
**********************************************************************************/
void updatePrescalersForFullAndReducedFrequencyMode()
{
    uint32 volatile presclarValueForReducedFreq;
    uint32 volatile presclarValueForFullFreq; 
	uint32 systemFrequency = 0; 
	RegEmeraldEnvTimer2Prescaler_u emeraldEnvTimePrescalerReg;

	emeraldEnvTimePrescalerReg.val = 0;
    systemFrequency = System_GetMainCoreFrequency();
	
    /* Wave500 mips frequency =240MHZ or 320 Mhz */
	/* (240/2)-1 ~=119	--> 0.5 millisecond */
	/* (320/2)-1 ~=159	--> 0.5 millisecond */
    DEBUG_ASSERT((systemFrequency == 240) ||(systemFrequency == 320));               
    presclarValueForFullFreq 	= (systemFrequency >> 1) - 1;
	presclarValueForReducedFreq 	= (systemFrequency >> 2) - 1;

 	emeraldEnvTimePrescalerReg.bitFields.timer2Prescaler	= presclarValueForFullFreq;
  	emeraldEnvTimePrescalerReg.bitFields.rfT2PrescaleValue	= presclarValueForReducedFreq;
	RegAccess_Write(REG_EMERALD_ENV_TIMER0_PRESCALER,emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER1_PRESCALER,emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER2_PRESCALER,emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER3_PRESCALER,emeraldEnvTimePrescalerReg.val);
}


/**********************************************************************************
isReducedFrequency

Description:
------------
API to check if we are currently in reduced frequency. In wave600 -it's not supported.

Input:
-----

Output:
-------

Returns:
--------
		
**********************************************************************************/
bool isReducedFrequency(void)
{
	RegMacHtExtensionsReducedFreqMode_u reducedFreqMode; 

	RegAccess_Read(REG_MAC_HT_EXTENSIONS_REDUCED_FREQ_MODE, &reducedFreqMode.val);

	return reducedFreqMode.bitFields.reducedFreqMode;
}
#if defined(ENET_INC_LMAC)
ISR_VOID GpTimerB_ISR()
{
    RegEmeraldEnvTimersStatus_u timerIndication;

	RegAccess_Read(REG_EMERALD_ENV_TIMERS_STATUS, &timerIndication.val);
    if(timerIndication.bitFields.timer3Expired == TRUE)
    {
		//timer3 indication
		MT_ClearIsrTimer(MT_TIMER_3);
        TimerUtiltyInterruptHandler();
    }
    if(timerIndication.bitFields.timer2Expired == TRUE)
    {
        //timer2 indication
		//MT_ClearIsrTimer(MT_TIMER_2);
		ASSERT(0); // Currently we don't use this timer, so it shouldn't expire. If we want to use it - replace assert with implementation, and uncomment the line above.
    }  
}
#endif


/****************************************************************************
 **
 ** NAME:           LmvKNLintrTimer
 **
 ** PARAMETERS:     frame ticks passed
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:    Sends number of tick periods elapsed to the Timer
 **                 Tick task
 **
 ****************************************************************************/
#if defined(CALCULATE_CPU_IDLE_TIME) 

void LmvKNLintrTimer(void)
{
	uint32 deltaTime = (GET_TSF_TIMER_LOW() - CalcCPUIdle_LastPrint);
	
	ACCUMULATE_CPU_IDLE_TIME();

	/* Make sure we do not get here too soon causing the denominator to equal zero */
	if (deltaTime > CPU_LOAD_MIN_TIME_BETWEEN_COMPUTATIONS)
	{	
		cpuLoadArrayLm[cpuLoadArrayLmIndex] = (CalcCPUIdle_totalSleepTime*100) / deltaTime; 

		cpuLoadArrayLmIndex++;

		if (cpuLoadArrayLmIndex == CPU_IDLE_ARRAY_SIZE)
		{
			cpuLoadArrayLmIndex= 0;
		}
		
		CalcCPUIdle_totalSleepTime = 0;

		CalcCPUIdle_LastPrint = GET_TSF_TIMER_LOW();
	}
}
#endif//CALCULATE_CPU_IDLE_TIME


/**********************************************************************************
MT_InitTimers

Description:
Initialize the timers- disable them, set to one shot mode, and set prescaler to 0.5usec)

Input:
-----

Output:
-------
	
Returns:
--------
		
**********************************************************************************/
#if (defined (ENET_INC_UMAC)  || defined (ENET_INC_LMAC))
#pragma ghs section text=".initialization" 
#endif

void MT_InitTimers(void)
{
	uint32 timerId = 0;


	for (timerId = MT_TIMER_0 ; timerId  <= MT_TIMER_3 ; timerId++)
	{
	    /* Disable Timer */
	    MT_SetModeTimer(timerId, MT_TIMER_DISABLE, MT_TIMER_ONE_SHOT); 

	    /* clear ISR for Timer */
	    MT_ClearIsrTimer(timerId);
	}

	updatePrescalersForFullAndReducedFrequencyMode();
}

#if (defined (ENET_INC_UMAC)  || defined (ENET_INC_LMAC))
#pragma ghs section text=default
#endif



