/************************************************************************************
*    
*    File:		  System_Timers.c
*    Class/Module: 
*    Description:  A detailed description of the Module, its  purpose attributes and
*		 	  whatever information the user & maintainer might find valuable.
*
*    COPYRIGHT: 
*        (C) Lantiq Israel Ltd.
*        All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*        copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/


/*************************************************************************/
/***                        Include Files                              ***/
/*************************************************************************/
#include "System_GlobalDefinitions.h"
#include "Debug_Trace.h"
#include "OSAL_Api.h"
#include "System_Timers.h"
#include "InterruptManager_Api.h"
#include "lm.h"
#include "CpuLoad_Api.h"
#include "MacHtExtensionsRegs.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "Pac_Api.h"
#include "lm_statistics.h"
#include "InterruptManager_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "lmtimerUtility.h"
#if defined (ENET_INC_UMAC)
#include "PacketTrafficArbitrator_api.h"
#endif
/*************************************************************************/
/***                        Local Defines                              ***/
/*************************************************************************/
#define LOG_LOCAL_GID   GLOBAL_GID_SYS
#define LOG_LOCAL_FID 7


#define US_TO_PRESCALER_FACTOR   (2)  //prescaler is configured to 0.5 microsec. desired time is uSec should be multiplied by 2.
#define US_TO_PRESCALER_UNIT(us)   (uint32)(us*US_TO_PRESCALER_FACTOR)  //prescaler configured to 0.5 microsec


/*************************************************************************/
/***               Public Functions                                    ***/
/*************************************************************************/


/**********************************************************************************
MT_SetModeTimer


Description:
------------
Set the mode (one shot / multi shot) and enable/disable a timer.

Input:
-----
	timer ID
	Enable / Disable
	Shot mode (one shot / multi shot)
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_SetModeTimer(uint32 timerId, uint32 enableTimer, uint32 shotMode)	
{
	TX_INTERRUPT_SAVE_AREA;

	RegEmeraldEnvTimersConfig_u		timersConfigReg;
	RegEmeraldEnvTimersControl_u	timersControlReg;

	/* Init timer mode to Single Shot */
	timersConfigReg.val = 0;
	/* Init the timer as Disabled */
	timersControlReg.val = 0;

	/* Assert if the timer does not exists */
	DEBUG_ASSERT(timerId <= MT_TIMER_3)
		
	/* For Multi Shot set the value to '1' */
	if (shotMode == MT_TIMER_MULT_SHOT)
	{
		timersConfigReg.val = 0x1 << timerId;
	}

	// configure shot mode.
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_CONFIG, timersConfigReg.val);

	
	if (enableTimer == MT_TIMER_ENABLE)
	{

		// Calculate mask for "enable bit" for this timer
		timersControlReg.val = 0x1 << timerId;
	}
    else
    {
		// Calculate mask for "disable bit" for this timer
	    timersControlReg.val = 0x10 << timerId;; 
    }

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	// Enable / Disable the timer (write only register)
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_CONTROL, timersControlReg.val);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

}


/**********************************************************************************
MT_PollTimer

Description:
------------
Poll on a timer until it expires (not recommanded!)

Input:
-----
	timer ID
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_PollTimer(uint32 timerId)
{
	uint32 timerExpireMask;

	timerExpireMask = 0x1 << timerId;	
	while( (RegAccess_ReadImmediate(REG_EMERALD_ENV_TIMERS_EXPIRATION_STATUS) & timerExpireMask) == 0);
}


/**********************************************************************************
MT_SetTimeTimer

Description:
------------
Set time period for timer

Input:
-----
	timer ID
	Expiration time
Output:
-------
	
Returns:
--------
		
**********************************************************************************/
 void MT_SetTimeTimer(uint32 timerId, uint32 maxCount)	
{
	RegEmeraldEnvTimer0MaxCount_u 	timerMaxCount;
	uint32							timerMaxCountReg;

	/* All timers has prescalers*/
	timerMaxCount.bitFields.timer0MaxCount = US_TO_PRESCALER_UNIT(maxCount);	

    //Configure the timer period
	timerMaxCountReg = REG_EMERALD_ENV_TIMER0_MAX_COUNT + CONVERT_WORDS_TO_BYTES(timerId);
    RegAccess_Write(timerMaxCountReg, timerMaxCount.val);
}

/**********************************************************************************
MT_ClearIsrTimer

Description:
	Clear the interrupt of the timer (should be called from ISR)
	The timer is NOT DISABLED! If it's multi shot - it will keep running.

Input:
-----
	timer ID
Output:
-------
	
Returns:
--------
		
**********************************************************************************/
void MT_ClearIsrTimer(uint32 timerId)	
{
	RegEmeraldEnvTimersExpirationStatus_u	timersExpirationStatusReg;

	timersExpirationStatusReg.val = 0;
	timersExpirationStatusReg.val = 0x1 << timerId;
	//Clear interrupt source (clear by write '1' to relevant bit). We must not touch the other bits since it can clear other interrupts.
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_EXPIRATION_STATUS, timersExpirationStatusReg.val);
}



#ifdef ENET_INC_LMAC
/**********************************************************************************
isr_Timers_Timer0

Description:
------------
Handle the event that timer0 has expired

Input:
-----

Output:
-------
	

Returns:
--------
		
**********************************************************************************/
ISR_VOID isr_Timers_Timer0(void)
{
#if defined RADAR_DETECTION_ENABLED
	OSAL_SEND_NO_DATA_MESSAGE(HDK_RADAR_DETECTION_TIMER, TASK_HDK, VAP_ID_DO_NOT_CARE);

	MT_ClearIsrTimer(MT_TIMER_0);
#else
	FATAL("Radar Detection Timer isn't used in 2.4G");
#endif
}

#endif /* #ifdef ENET_INC_LMAC */

#ifdef ENET_INC_UMAC
ISR_VOID isr_Timer0()
{
	MT_ClearIsrTimer(MT_TIMER_0);
	ASSERT(0);
}

ISR_VOID isr_Timer1()
{
	MT_ClearIsrTimer(MT_TIMER_1);
	PtaHandleTimer1Expiry();
}

/**********************************************************************************

isr_Timers_TimerA


Description:
------------
Handle the event that timerA has expired (timer0 and/or timer1)

Input:
-----


Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
//ISR_VOID isr_Timers_TimerA(void)
//{
//	RegEmeraldEnvTimersStatus_u timersStatusRegister;
//	RegEmeraldEnvTimersStatus_u timersStatusCopyRegister;
//
//	timersStatusRegister.val = 0;
//	timersStatusCopyRegister.val = 0; 
//	 
//    RegAccess_Read(REG_EMERALD_ENV_TIMERS_STATUS, &timersStatusRegister.val);
//	timersStatusCopyRegister.val = timersStatusRegister.val;
//	timersStatusRegister.val = 0;
//	
//    if(timersStatusCopyRegister.bitFields.timer0Active == FALSE)
//    {
//		timersStatusRegister.bitFields.timer0Active = FALSE;
//        OSAL_SEND_NO_DATA_MESSAGE(HDK_RADAR_DETECTION_TIMER, TASK_HDK, VAP_ID_DO_NOT_CARE);
//    }
//    if(timersStatusCopyRegister.bitFields.timer1Active == FALSE)
//    {
//		/* This is an un-used timer in WAVE600 (in WAVE500 it is used for PTA) */
//		ASSERT(0);
//    }    
//    RegAccess_Write(REG_EMERALD_ENV_TIMERS_STATUS, timersStatusRegister.val);
//}

#endif /* #ifdef ENET_INC_UMAC */


/**********************************************************************************
MT_enableDisableGpTimerIrq

Description:
------------
Enable  Disable the interrupt for a specfic timer

Input:
-----
	timer ID
	Enable / Disable IRQ
Output:
-------
	

Returns:
--------
		
**********************************************************************************/
void MT_enableDisableGpTimerIrq(uint32 timerID, bool isEnable)
{
	RegEmeraldEnvTimersIrqEn_u regEmeraldEnvTimersIrqEn;


	RegAccess_Read(REG_EMERALD_ENV_TIMERS_IRQ_EN, &regEmeraldEnvTimersIrqEn.val);

	if (isEnable == 1)
	{
		/*Enable timer IRQ*/
		regEmeraldEnvTimersIrqEn.val |=  (1 << timerID);
	}
	else
	{
		/*Disable timer IRQ*/
		regEmeraldEnvTimersIrqEn.val &= (~(1 << timerID));
		
	}
	
	RegAccess_Write(REG_EMERALD_ENV_TIMERS_IRQ_EN, regEmeraldEnvTimersIrqEn.val);
}
/**********************************************************************************
updatePrescalersForFullAndReducedFrequencyMode

Description:
------------
Initializae the prescalers of all 4 timers to be 0.5 usec resolution

Input:
-----

Output:
-------

Returns:
--------
		
**********************************************************************************/
void updatePrescalersForFullAndReducedFrequencyMode()
{
    uint32 presclarValueForFullFreq; 
	uint32 systemFrequency = 0; 
	RegEmeraldEnvTimer0Prescaler_u emeraldEnvTimePrescalerReg;

	emeraldEnvTimePrescalerReg.val = 0;
    systemFrequency = System_GetMainCoreFrequency();
	
    /* Wave600 frequency = 640MHZ*/
	/* (640/2)-1 ~=319	--> 0.5 millisecond */
    ASSERT(systemFrequency == 640);               
    presclarValueForFullFreq = (systemFrequency >> 1) - 1;

 	emeraldEnvTimePrescalerReg.bitFields.timer0Prescaler = presclarValueForFullFreq;
	RegAccess_Write(REG_EMERALD_ENV_TIMER0_PRESCALER, emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER1_PRESCALER, emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER2_PRESCALER, emeraldEnvTimePrescalerReg.val);
	RegAccess_Write(REG_EMERALD_ENV_TIMER3_PRESCALER, emeraldEnvTimePrescalerReg.val);
}


/**********************************************************************************
isReducedFrequency

Description:
------------
API to check if we are currently in reduced frequency. In wave600 -it's not supported.

Input:
-----

Output:
-------

Returns:
--------
		
**********************************************************************************/
bool isReducedFrequency(void)
{
	return 0;
}

#if defined(ENET_INC_LMAC)
ISR_VOID isr_Timer1()
{
	MT_ClearIsrTimer(MT_TIMER_1);
	ASSERT(0);
	//PtaHandleMipsTimer1Expiry(); => PTA is in upper mac
}

ISR_VOID isr_Timer2()
{
	MT_ClearIsrTimer(MT_TIMER_2);
	ASSERT(0); // Currently we don't use this timer, so it shouldn't expire. If we want to use it - replace assert with implementation.
}

ISR_VOID isr_Timer3()
{
	MT_ClearIsrTimer(MT_TIMER_3);
    TimerUtiltyInterruptHandler();
}
#endif // ENET_INC_LMAC


/**********************************************************************************
MT_InitTimers

Description:
Initialize the timers- disable them, set to one shot mode, and set prescaler to 0.5usec)

Input:
-----

Output:
-------
	
Returns:
--------
		
**********************************************************************************/
void MT_InitTimers(void)
{
	uint32 timerId = 0;


	for (timerId = MT_TIMER_0 ; timerId  <= MT_TIMER_3 ; timerId++)
	{
	    /* Disable Timer */
	    MT_SetModeTimer(timerId, MT_TIMER_DISABLE, MT_TIMER_ONE_SHOT); 

	    /* clear ISR for Timer */
	    MT_ClearIsrTimer(timerId);
	}

	updatePrescalersForFullAndReducedFrequencyMode();
}


