/*******************************************************************************
*    
*   Source File: tx_application_func.c
*	
*	AUTHOR: Omer Modlin
*
*   Description: 
*       
*   Copyright: 
*       
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "tx_api.h"
#include "tx_application_vars.h"
#include "tx_application_define.h"
#include "tx_application_Api.h"
#include "database.h"
#include "stringLibApi.h"
#include "fw_utils_queue.h"
#ifndef ENET_INC_ARCH_WAVE600
#include "MT_MAC_HT_extensions_regs.h"
#endif
#include "tx_application_trace.h"
#include "ErrorHandler_Api.h"
#include "shram_interruptSection.h"
#include "loggerAPI.h"
#include "OSAL_Interrupts.h"

#define LOG_LOCAL_GID GLOBAL_GID_TX_APP
#define LOG_LOCAL_FID 3



static TX_THREAD	*pStackOverFlowThread = NULL;

#ifdef CPU_MIPS
#pragma alignvar (64)
#endif
extern 	uint32 SharedRamIntSection[NUM_OF_SHRAM_INT];
extern const K_TASKID taskConverterTable[NUM_OF_TASKS_IN_SYSTEM];


/******************************************************************************
 **
 ** Name:		TxApp_SendMessage
 **
 ** Purpose:	send a message from a thread/intrrupt to a thread's queue.
 **				the function puts a pointer to a message, that is allocated in
 **				a msg pool, in a thread's queue.
 **				
 ** Input:		K_MSG_TYPE msgType	- type is set inside the msg
 ** 			K_TASKID bTo		- task id is set inside the msg
 **				K_MSG *psKmsg		- pointer to msg in msg pool
 **				uint32 wait			- TX_WAIT_FOREVER / TX_NO_WAIT 
 **									  (no wait from interrupt)
 **
 ** Output:		None
 **
 ******************************************************************************/

void TxApp_SendMessage(K_MSG_TYPE msgType, K_TASKID bTo, K_MSG *psKmsg, uint32 wait, uint8 vapId)
{
	uint16 status;
	TxApp_interThreadMsg_t interThreadMsg;
	TxApp_thread_e target_thread;
	K_TASKID kTaskId;
	K_TASKID bFrom = TASK_INVALID; //KW_FIX_FW_G
	
#if (THREADX_DEBUG_ON)
	TX_INTERRUPT_SAVE_AREA;
#endif
	TxApp_thread_e from_thread = TX_APP_THRD_SYSTEM;

	TX_THREAD	*pThread = tx_thread_identify();
	if (pThread)
	{
		from_thread = (TxApp_thread_e)(pThread->tx_thread_entry_parameter); //entry parameter holds thread enum
		if (from_thread < TX_APP_THRD_NUM) //KW_FIX_FW_
		{
			bFrom = TxApp_threadPeripheralInfoArray[from_thread].taskId;
		}
	}

	psKmsg->header.tKMsgType 	= msgType;
	psKmsg->header.sTo.taskID 	= bTo;
	psKmsg->header.sTo.stub 	= 0;
	psKmsg->header.sFrom.taskID = bFrom;
	psKmsg->header.sFrom.stub 	= 0;

	if (vapId != VAP_ID_DO_NOT_CARE)
		psKmsg->header.vapId = vapId;

	/*The messgae shall not pass through more than 7 threads */
	if(!(psKmsg->header.bPersistentMsg))
	{
		ASSERT(bLNARC_GET_REF_COUNT(psKmsg->header.bListNoAndRefCount) != 7);
		psKmsg->header.bListNoAndRefCount++; 
	}


	//ILOG0_DD("TxApp_SendMessage: 0x%x, %d", msgType, bTo);

	//ASSERT(psKmsg != PNULL);

	interThreadMsg.p = psKmsg;
	ASSERT(bTo < NUM_OF_TASKS_IN_SYSTEM);

	kTaskId = taskConverterTable[bTo];
	ASSERT(kTaskId < NUM_OF_TASKS_IN_SYSTEM);//KW_FIX_FW_G

	target_thread = TxApp_translatationTableFromTaskIdToThreadId[kTaskId];
	
	ASSERT(target_thread != TX_APP_THRD_NULL_VALUE);

	/*send msg to the specific thread's queue*/

//	ILOG0_DDD("TxApp_SendMessage msgType:%d, from thread: %d  to thread: %d ",msgType, from_thread, target_thread);
#if (THREADX_DEBUG_ON)
	if ((msgType == KMSG_LOGGER_DBG_TRACE_IND) ||
		(msgType == HIM_RINGS_DMA_DONE) ||
		(msgType == HIM_NEW_HDS_IN_HOST_RING) ||
		(msgType == RX_MANAGER_RDS_LIST_NOT_EMPTY))
	{
		// ignore
	}
	else
	{
//		SLOG0(0, 0, K_MSG, psKmsg);
	}
#endif
	THREADX_TRACE(THREADX_SEND, from_thread, target_thread);
	ASSERT(target_thread < TX_APP_THRD_NUM);
	status = tx_queue_send(
				&(TxApp_threadInfoArray[target_thread].queue),
				&interThreadMsg,
				wait);

	THREADX_TRACE(THREADX_SEND_RESUME, from_thread,msgType);

	ASSERT(status == TX_SUCCESS);
}

/******************************************************************************
 **
 ** Name:		TxApp_SendFrontMessage
 **
 ** Purpose:	send a message from a thread/intrrupt to the front of a thread's queue.
 **				the function puts a pointer to a message, that is allocated in
 **				a msg pool, in a thread's queue.
 **				
 ** Input:		K_MSG_TYPE msgType	- type is set inside the msg
 ** 			K_TASKID bTo		- task id is set inside the msg
 **				K_MSG *psKmsg		- pointer to msg in msg pool
 **				uint32 wait			- TX_WAIT_FOREVER / TX_NO_WAIT 
 **									  (no wait from interrupt)
 **
 ** Output:		None
 **
 ******************************************************************************/

void TxApp_SendFrontMessage(K_MSG_TYPE msgType, K_TASKID bTo, K_MSG *psKmsg, uint32 wait, uint8 vapId)
{
	uint16 status;
	TxApp_interThreadMsg_t interThreadMsg;
	TxApp_thread_e target_thread;
	K_TASKID kTaskId;
#if (THREADX_DEBUG_ON)
	TX_INTERRUPT_SAVE_AREA;
	TxApp_thread_e from_thread = TX_APP_THRD_SYSTEM;

	TX_THREAD	*pThread = tx_thread_identify();	
	if (pThread)
	{
		from_thread = (TxApp_thread_e)(pThread->tx_thread_entry_parameter); //entry parameter holds thread enum
	}
	else
	{
		from_thread = TX_APP_THRD_SYSTEM;
	}
#endif

	psKmsg->header.tKMsgType 	= msgType;
	psKmsg->header.sTo.taskID 	= bTo;

	if (vapId != VAP_ID_DO_NOT_CARE)
		psKmsg->header.vapId = vapId;

	/*The messgae shall not pass through more than 7 threads */
	if(!(psKmsg->header.bPersistentMsg))
	{
		ASSERT(bLNARC_GET_REF_COUNT(psKmsg->header.bListNoAndRefCount) != 7);
		psKmsg->header.bListNoAndRefCount++; 
	}


//	ILOG0_DD("TxApp_SendFrontMessage: 0x%x, %d", msgType, bTo);

	//ASSERT(psKmsg != PNULL);

	interThreadMsg.p = psKmsg;
	interThreadMsg.p->header.tKMsgType = msgType;
	interThreadMsg.p->header.sTo.taskID = bTo;
	interThreadMsg.p->header.sTo.stub = 0;
#if (THREADX_DEBUG_ON)
	interThreadMsg.p->header.sFrom.taskID = from_thread;
	interThreadMsg.p->header.sFrom.stub = 0;
#endif		
	ASSERT(bTo < NUM_OF_TASKS_IN_SYSTEM);

	kTaskId = taskConverterTable[bTo];
	ASSERT(kTaskId < NUM_OF_TASKS_IN_SYSTEM);//KW_FIX_FW_G

	target_thread = TxApp_translatationTableFromTaskIdToThreadId[kTaskId];
	
	ASSERT(target_thread != TX_APP_THRD_NULL_VALUE);

	/*send msg to the specific thread's queue*/

//	ILOG0_DDD("TxApp_SendFrontMessage msgType:%d, from thread: %d  to thread: %d ",msgType, from_thread, target_thread);
#if (THREADX_DEBUG_ON)
	if ((msgType == KMSG_LOGGER_DBG_TRACE_IND) ||
		(msgType == HIM_RINGS_DMA_DONE) ||
		(msgType == HIM_NEW_HDS_IN_HOST_RING) ||
		(msgType == RX_MANAGER_RDS_LIST_NOT_EMPTY))
	{
		// ignore
	}
	else
	{
//		SLOG0(0, 0, K_MSG, psKmsg);
	}
#endif
	THREADX_TRACE(THREADX_SEND, from_thread, target_thread);
	ASSERT(target_thread < TX_APP_THRD_NUM);//KW_FIX_FW_G
	status = tx_queue_front_send(
				&(TxApp_threadInfoArray[target_thread].queue),
				&interThreadMsg,
				wait);

	THREADX_TRACE(THREADX_SEND_RESUME, from_thread,msgType);

	ASSERT(status == TX_SUCCESS);
}


/******************************************************************************
 **
 ** Name:		TxApp_KernelEnter
 **
 ** Purpose:	wrapper for tx_kernel_enter
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization_start" 
#endif
void TxApp_KernelEnter(void)
{ 
	tx_kernel_enter();
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default 
#endif

/******************************************************************************
 **
 ** Name:		TxApp_SetMessageOriginatorTaskId
 **
 ** Purpose:	set originator for msg according to thread peripheral info
 ** 			array
 **				
 ** Input:		K_MSG *psMsg
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_SetMessageOriginatorTaskId(K_MSG *psMsg)
{
	K_TASKID taskId;
	TX_THREAD *pThread = tx_thread_identify();

	if (pThread)
	{
		ASSERT(pThread->tx_thread_entry_parameter < TX_APP_THRD_NUM);//KW_FIX_FW_G
		taskId = TxApp_threadPeripheralInfoArray[(TxApp_thread_e)(pThread->tx_thread_entry_parameter)].taskId; //entry parameter holds thread enum
	}
	else
	{
		taskId = TX_APP_THRD_SYSTEM;

	}

	ASSERT( psMsg != NULL );
	psMsg->header.sFrom.taskID = taskId;
	psMsg->header.sFrom.stub = 0;
}

/******************************************************************************
 **
 ** Name:		TxApp_GetMessageOriginator
 **
 ** Purpose:	get message originator task id
 **				
 ** Input:		K_MSG *psMsg
 **				K_TASKID *pTask
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_GetMessageOriginator(K_MSG *psMsg, K_TASKID *pTask)
{
    ASSERT((psMsg != NULL) && (pTask != NULL));

    *pTask = psMsg->header.sFrom.taskID;
}
/******************************************************************************
 **
 ** Name:		TxApp_EnableTraceX
 **
 ** Purpose:		Enable TRACEX buffer and initialize it 
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_EnableTraceX(VOID)
{
#ifdef TX_ENABLE_EVENT_TRACE
	/*	Initialize	TRACEX	dump file */
	uint16 status = 0;
	status = tx_trace_enable(&tracexDumpFile,SIZE_OF_TRACEX_DUMP, 1000);
	ASSERT(status == TX_SUCCESS);
#endif // TX_ENABLE_EVENT_TRACE
}

/******************************************************************************
 **
 ** Name:		TxApp_DisableTraceX
 **
 ** Purpose:		Disable TraceX event trace by a FW module call
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_DisableTraceX(VOID)
{
#ifdef TX_ENABLE_EVENT_TRACE
	tx_trace_disable();
#endif // TX_ENABLE_EVENT_TRACE
}


/******************************************************************************
 **
 ** Name:		TxApp_Error
 **
 ** Purpose:		Callback function 
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_Error(TX_THREAD *thread_ptr)
{
#ifdef TX_ENABLE_STACK_CHECKING  
	TX_INTERRUPT_SAVE_AREA;

	pStackOverFlowThread = thread_ptr; 
#if defined (ANALYZE_DISABLE_INTERRUPT)
	InterruptManager_DisableAllInterrupts(&interrupt_save, __FILE__, __LINE__);
#else
	InterruptManager_DisableAllInterrupts(&interrupt_save);
#endif
	SharedRamInt_EnableInt(SHRAM_INT_TX_APP_ERROR);
	SharedRamIntSection[SHRAM_INT_TX_APP_ERROR] = 0xFF;  
#if defined (ANALYZE_DISABLE_INTERRUPT)
	InterruptManager_EnableAllInterrupts(interrupt_save, __FILE__, __LINE__); 
#else
	InterruptManager_EnableAllInterrupts(interrupt_save); 
#endif
#else
	UNUSED_PARAM(thread_ptr);
	ASSERT(0);
#endif
}


/******************************************************************************
 **
 ** Name:		TxApp_HandleShramErrorIsr
 **
 ** Purpose:		Callback function from shram interrupt, used as an indication for stack overflow 
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/

void TxApp_HandleShramErrorIsr(void)
{
	ASSERT(pStackOverFlowThread == NULL); //Verify the existance of a stack overflowed thread
	ASSERT(0); //This assert means No thread has stack overflow, so we got here by mistake.
}


/******************************************************************************
 **
 ** Name:		full_buffer_callback
 **
 ** Purpose:		 This routine is being called by threadX when TRACEX buffer is full
 **				  Defined only if EVENT TRACE is enable
 **				
 ** Input:		None
 **
 ** Output:		None
 **
 ******************************************************************************/
#ifdef TX_ENABLE_EVENT_TRACE

VOID full_buffer_callback(VOID)
{
	/* In debug mode-http://www.barby.co.il/TemplatesPage/showcat.aspx?id2=753 we would like the system to halt when buffer is full*/
	ASSERT(0);
}

#endif // TX_ENABLE_EVENT_TRACE


bool TxApp_IsThreadsQueueEmpty(uint32 taskId)
{
	uint32 threadId;
	ASSERT(taskId < NUM_OF_TASKS_IN_SYSTEM);//KW_FIX_FW_G
	threadId = TxApp_translatationTableFromTaskIdToThreadId[taskId]; 
	ASSERT(threadId < TX_APP_THRD_NUM); //KW_FIX_FW_G
	if(TxApp_threadInfoArray[threadId].queue.tx_queue_enqueued > 0)
	{
		return FALSE; 
	}
	else
	{
		return TRUE; 
	}
}

void TxApp_Thread_Relinquish(void) 
{
#if (THREADX_DEBUG_ON)
		TX_INTERRUPT_SAVE_AREA;
		TxApp_thread_e from_thread = TX_APP_THRD_SYSTEM;
	
		TX_THREAD	*pThread = tx_thread_identify();	
		if (pThread)
		{
			from_thread = (TxApp_thread_e)(pThread->tx_thread_entry_parameter); //entry parameter holds thread enum
		}
		else
		{
			from_thread = TX_APP_THRD_SYSTEM;
		}
		THREADX_TRACE(THREADX_THREAD_RELINQUISH, from_thread, TX_APP_THRD_SYSTEM);
#endif

	tx_thread_relinquish();

}


/**********************************************************************************

TxApp_ChangeThreadPriority 


Description:
------------
Changes priority of a specific thread


Input: 
-----
taskId - the Id of the task
newPriority - new priority value
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
void TxApp_ChangeThreadPriority(uint8 taskId, uint8 newPriority)
{
	uint8 threadId = 0;
	UINT oldPriority = 0;
	TX_THREAD *threadPointer = NULL;
	uint16 status = TX_SUCCESS;

	ASSERT(taskId < NUM_OF_TASKS_IN_SYSTEM);

	threadId = TxApp_translatationTableFromTaskIdToThreadId[taskId];
	ASSERT(threadId < TX_APP_THRD_NUM); //KW_FIX_FW_G
	threadPointer = &TxApp_threadInfoArray[threadId].thread;
	status = tx_thread_priority_change(threadPointer, newPriority, &oldPriority);
	ASSERT(status == TX_SUCCESS);
}






