/*******************************************************************************
*    
*   Source File: tx_application_msg_pool.c
*	
*	AUTHOR: Omer Modlin
*
*   Description: 
*       
*   Copyright: 
*       
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "tx_application_Api.h"
#include "tx_application_trace.h"
#include "tx_application_vars.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"

#define LOG_LOCAL_GID GLOBAL_GID_TX_APP
#define LOG_LOCAL_FID 5



void TxApp_FreeFastMessage(K_MSG *msg);

/******************************************************************************
 **
 ** Name:		TxApp_InitPersistentMsg
 **
 ** Purpose:	mark message as persistant
 **				
 ** Input:		K_MSG *msg
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_InitPersistentMsg(K_MSG *kMsg_p, K_LEN tK_LenLength)
{
	ASSERT(kMsg_p != PNULL);
	
	/* Mark message as being used by someone (1) and as a user allocated message */
	kMsg_p->header.bPersistentMsg = TRUE;

	kMsg_p->header.payloadSize = (uint16)tK_LenLength;
}

/******************************************************************************
 **
 ** Name:		TxApp_FreeFastMessage
 **
 ** Purpose:	free fast message back to block pool
 **				
 ** Input:		K_MSG *msg
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_FreeFastMessage(K_MSG *msg)
{
	uint16 status;
#if (THREADX_DEBUG_ON)
	TX_INTERRUPT_SAVE_AREA;

	TX_THREAD	*pThread = tx_thread_identify();
	TxApp_thread_e from_thread;
	
	if (pThread)
	{
		from_thread = (TxApp_thread_e)(pThread->tx_thread_entry_parameter); //entry parameter holds thread enum
	}
	else
	{
		from_thread = TX_APP_THRD_SYSTEM;
	}
#endif
	
	THREADX_TRACE(THREADX_FREE_FAST_MSG_WAIT, from_thread, 0);

	status = tx_block_release(msg);
	ASSERT(status == TX_SUCCESS);

	THREADX_TRACE(THREADX_FREE_FAST_MSG_RESUME, from_thread, 0);
}

/******************************************************************************
 **
 ** Name:		TxApp_GetFastMessage
 **
 ** Purpose:	get fast message from block pool
 **				
 ** Input:		len - length of msg body - must be smaller than K_FAST_MSG_SIZE
 **
 ** Output:		K_MSG *
 **
 ******************************************************************************/
K_MSG *TxApp_GetFastMessage(uint16 len)
{
	uint16 status;
	K_MSG *pMsg;

	ASSERT(len <= K_FAST_MSG_SIZE + sizeof(K_MSG_HEADER));
	
	status = tx_block_allocate(&TxApp_blockPool_fastMsgs, (void **)&pMsg, TX_NO_WAIT);
	if (status != TX_SUCCESS)
	{
		ASSERT(0);
	}

	pMsg->header.bListNoAndRefCount = bLNARC_FAST_POOL | bLNARC_REF_COUNT(1); //mark refcount as 1
	pMsg->header.bPersistentMsg	 = FALSE;
	pMsg->header.vapId = INVALID_VAP;
	pMsg->header.payloadSize = 0;
	return pMsg;
}

/******************************************************************************
 **
 ** Name:		TxApp_GetDefaultMessage
 **
 ** Purpose:	get default message from byte pool, of size header + len
 **				
 ** Input:		uint16 len
 **
 ** Output:		K_MSG *msg
 **
 ******************************************************************************/
K_MSG *TxApp_GetDefaultMessage(uint16 len)
{
	uint16 status;
	K_MSG *pMsg = PNULL;

	ASSERT(len <= K_FAST_MSG_SIZE);
	status = tx_block_allocate(&TxApp_blockPool_fastMsgs, (void **)&pMsg, TX_NO_WAIT);
	if (status != TX_SUCCESS)
	{
		ASSERT(0);
	}
	pMsg->header.payloadSize = (uint32)len;
	pMsg->header.bListNoAndRefCount = bLNARC_FAST_POOL | bLNARC_REF_COUNT(1); //mark refcount as 1
	pMsg->header.bPersistentMsg	 = FALSE;
	pMsg->header.vapId = INVALID_VAP;
	return pMsg;

}

/******************************************************************************
 **
 ** Name:		TxApp_SendNoDataFastMessage
 **
 ** Purpose:	allocate a fast message and send to thread according to task id
 **				
 ** Input:		K_MSG_TYPE msgType
 **				K_TASKID taskId
 **				uint32 wait
 **
 ** Output:		None
 **
 ******************************************************************************/
void TxApp_SendNoDataFastMessage(K_MSG_TYPE msgType, K_TASKID taskId, uint32 wait,uint8 vapId)
{
	K_MSG * pMsg;

	pMsg = TxApp_GetFastMessage(K_NO_DATA);

	TxApp_SendMessage(msgType, taskId, pMsg, wait,vapId);
}

/******************************************************************************
 **
 ** Name:		TxApp_FreeMessage
 **
 ** Purpose:	general free function which calls relevant free function
 **				according to where the msg was allocated 
 **				
 ** Input:		K_MSG *msg
 **				byte bPersistentMsg
 **
 ** Output:		None
 **
 ******************************************************************************/ 

void TxApp_FreeMessage(K_MSG *psMsg, byte bPersistentMsg)
{
	byte bPoolId;

    /* If we have a persistent message we do not remove it under any circumstances */
    if (bPersistentMsg == TRUE)
    {	
        return;
    }

    /* Zero ref count here is a fatal error */
    ASSERT(bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) > 0);
	psMsg->header.bListNoAndRefCount--; 

    /* The message is released if the counter is 1 or 0 : the refernce is 1 when it was sent, the
	   refernce is 0 when the message was allocated and released without being sent. the latest case
	   should be fixed in all models that use the message in this way, then this fix can be removed */
    if (bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) <= 1)
    {
		bPoolId = bLNARC_GET_POOL_ID(psMsg->header.bListNoAndRefCount);

		switch (bPoolId)
		{
			case bLNARC_FAST_POOL:
				psMsg->header.vapId = INVALID_VAP;
				TxApp_FreeFastMessage(psMsg);
				break;
			default:
				FATAL("attempt to release a non allocated message");
        }
    }
}


/****************************************************************************
 **
 ** NAME:           TxApp_DecMsgRefCount
 **
 ** PARAMETERS:     pointer to message
 **
 ** RETURN VALUES:  None
 **
 ** DESCRIPTION:    Decrements the reference count of a message buffer.
 **                 Message buffers are given a reference count of 1 when
 **                 allocated and the count is decremented when freed.
 **
 **                 Decrementing the reference count allows the message
 **                 block being released to the pool if the count reaches 0.
 **
 ****************************************************************************/

void TxApp_DecMsgRefCount(K_MSG *psMsg)
{
    /* If we have a persistent message count is irrelevant */
	TX_INTERRUPT_SAVE_AREA;
    if (psMsg->header.bPersistentMsg == TRUE)
        return;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

    if (bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) > 0)
    {
        /* NOTE: Relies on ref count being in bottom bits */
        psMsg->header.bListNoAndRefCount--;
    }
    else
    {
        FATAL("Tried to decrease a reference count past zero");
    }

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/****************************************************************************
 **
 ** NAME:           TxApp_IncMsgRefCount
 **
 ** PARAMETERS:     pointer to message
 **
 ** RETURN VALUES:  None
 **
 ** DESCRIPTION:    Increments the reference count of a message buffer.
 **                 Message buffers are given a reference count of 1 when
 **                 allocated and the count is decremented when freed.
 **
 **                 Incrementing the reference count avoids the message
 **                 block being released to the pool.
 **
 ****************************************************************************/

void TxApp_IncMsgRefCount(K_MSG *psMsg)
{
    /* If we have a persistent message count is irrelevant */
	TX_INTERRUPT_SAVE_AREA;
    if (psMsg->header.bPersistentMsg == TRUE)
        return;
	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);

	/* NOTE: Relies on ref count being in bottom bits */
    if (bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) < K_MAX_REF_COUNT)
    {
		psMsg->header.bListNoAndRefCount++;        
    }
    else
    {
	    FATAL("Tried to increase a reference count beyond the limit");
    }

	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}




