/**
	@file	interrupt_eic.c

	@brief	External interrupt controller implementation

	@date 10/2010

	@author Uri Shkolnik, Goeny Ltd

	<b> Copyright (c) 2009-2010 Goeny Ltd. </b>\n
	43 Hamelacha street, P.O. Box 8786, Poleg Industrial Park, Netanaya, ZIP 42505 Israel\n
	All rights reserved\n\n
	Proprietary rights of Goeny Ltd are involved in the
	subject matter of this material. All manufacturing, reproduction,
	use, and sales rights pertaining to this subject matter are governed
	by the license agreement. The recipient of this software implicitly
	accepts the terms of the license. This source code is the unpublished
	property and trade secret of Goeny Ltd.
	It is to be utilized solely under license from Goeny Ltd and it
	is to be maintained on a confidential basis for internal company use
	only. It is to be protected from disclosure to unauthorized parties,
	both within the Licensee company and outside, in a manner not less stringent
	than that utilized for Licensee's own proprietary internal information.
	No copies of the source or object code are to leave the premises of
	Licensee's business except in strict accordance with the license
	agreement signed by Licensee with Goeny Ltd.\n\n

	For more details - http://zoro-sw.com <or> http://goeny.com
	email: info@zoro-sw.com
*/


#include <string.h>
#include "zr_common.h"
#include "interrupt_eic.h"


////////////////////////////////////////////////////////////////////////////////
static void intr_eic_map_unset_all( void );
extern void __EnableInt(uint32 mask);

void (*nmi_handler)(void);

////////////////////////////////////////////////////////////////////////////////
P_ISR irq_handlers[TOTAL_IRQ_NUM];

////////////////////////////////////////////////////////////////////////////////
void intr_init(void)
{
	memset(irq_handlers, 0, sizeof(irq_handlers));
	intr_eic_map_unset_all();
	REG32(IRQ_EIC_REG_IRQ_ADDRESS) = 0;

	// Configure IPL=0 to enable all interrupts (=all that have priority
	// higher than 0)
	__EnableInt(0);

	// Put all IRQs on shadow set 1. in this register bit n is ss number (either 0 or 1) for IRQ number n.
	REG32(IRQ_EIC_IRQ_SS_ADDRESS) = 0xffffffff;
}

////////////////////////////////////////////////////////////////////////////////
static void intr_eic_map_unset_all( void )
{
    int i;
	for (i = 0 ; i < IRQ_MAP_REG_NUM ; i++)
		REG32(IRQ_EIC_MAP_BASE_ADDRESS + (i<<2)) = 0;
}

////////////////////////////////////////////////////////////////////////////////
int intr_eic_map_irq( uint32 ic_line, uint32 map_num)
{
	uint32 mask, map_reg_num, bit_pos,addr;

	if ( ic_line >= TOTAL_IRQ_NUM || map_num > IRQ_EIC_INT_MASK)
		return -1;

	map_reg_num		= (ic_line-1)/ IRQ_LINE_PER_REG;
	bit_pos			= ((ic_line-1) % IRQ_LINE_PER_REG) * IRQ_EIC_INT_MASK_WIDTH;
	addr = IRQ_EIC_MAP_BASE_ADDRESS + (map_reg_num<<2);
	mask = ((1<<IRQ_EIC_INT_MASK_WIDTH)-1) << bit_pos;
	REG32(addr) &= ~mask;
	REG32(addr) |= map_num<<bit_pos;

	return 0;
}

////////////////////////////////////////////////////////////////////////////////
void intr_enable_int(uint32 map_num)
{
	if (map_num <= IRQ_EIC_INT_MASK)
		SET_BIT(ADDR32(IRQ_EIC_REG_IRQ_ADDRESS), map_num);
}

////////////////////////////////////////////////////////////////////////////////
void intr_disable_int(uint32 map_num)
{
	if (map_num <= IRQ_EIC_INT_MASK)
		CLEAR_BIT(ADDR32(IRQ_EIC_REG_IRQ_ADDRESS), map_num);
}

////////////////////////////////////////////////////////////////////////////////
int intr_register_isr(uint32 map_num, P_ISR isr_func)
{
	int status = 0;

	if ((map_num <= IRQ_EIC_INT_MASK) && (isr_func != NULL))
		irq_handlers[map_num] = isr_func;
	else
		status = -1;

	return status;
}

////////////////////////////////////////////////////////////////////////////////
void intr_enable_interrupts()
{
#if defined (CPU_MIPS)	
	asm("ei");
	asm("ehb");
#endif
}

////////////////////////////////////////////////////////////////////////////////
void intr_disable_interrupts()
{
#if defined (CPU_MIPS)	
	asm("di");
	asm("ehb");
#endif
}

