/*****************************************************************************
#*    
#*   MODULE NAME:
#*       boot.s
#*   DESCRIPTION: 
#*       Mips Operating system interrupt handling assembler module
#*       and BootLoader.
#*       2010-2011:
#*       Added support for ThreadX OS, copying the ROM to RAM, configuring PLL,
#*       flash bootloading, NMI handling.
#*   AUTHOR:  
#*       Ziv Avital, 24/04/2000
#*       Ishai Asa, Goeny Ltd., December 2010
#*   COPYRIGHT: 
#*       (C) Metalink Ltd.
#*       All rights are strictly reserved. Reproduction or divulgence in any 
#*       form whatsoever is not permitted without written authority from the 
#*       copyright owner. Issued by Metalink Transmission Devices Ltd in 
#*       Israel - 11/94.
#*   REVISION HISTORY: 
#*       MetalinkRevision: 
*****************************************************************************/

.define ROM_VERSION                 ,0x00051901

.include "mnemonics.hs"

.define GP_INIT_VALUE				,0
.define CACHED_RAM                  ,0x80000000
.define NON_CACHED_RAM              ,0xa0000000


/*
 * The upper 2 bytes of the address of the HW register block.
 * The full address is 0xA018:0000
 */
.define REGISTER_BLOCK_UPPER        ,0xA018

.define USB_DCTL_REG				,0xA7100804
.define USB_DCTL_SOFT_DISC			,0x2
.define FULL_CHIP_RESET_REG			,0xa71c0008
.define FULL_CHIP_RESET_BIT			,~0x00010000

.define WAIT_STATE_REG 				,0xa71c0240

/* 2sec in timer counter cycles = 2 * 100 * 10ms in cycles */
.define TWO_SEC_COUNT				,2*100*1200000

/* ExtNV definitions */ 

/* Address of the flash in memory */
.define FLASH_BASE_ADDR             ,0xA8000000

/* The type of the bootloader TLV entry in Flash */
.define TLV_GUARD_BOOT              ,0x544F4F42

/* The size of the ROM in bytes */
.define ROM_SIZE                    ,0xB000 /* 45KBytes, no need to copy the rest of the ROM */

.define SRS_CTL_NO_PSS		        ,0xFFFFFC3F
.define STATUS_NMI_SET		        ,0x00080000
.define CAUSE_DC_0_UPPER	        ,0x0800
/* eFUSE PLL configuration */

.define EFUSE_REG_BASE				,0xa71c0120
.define EFUSE_TIMER1_CFG_REG_ADDR	,(EFUSE_REG_BASE+0x10)
.define EFUSE_TIMER2_CFG_REG_ADDR	,(EFUSE_REG_BASE+0x14)

.define EFUSE_TIMER1_36MHZ_DEFAULT	,0x000800C0
.define EFUSE_TIMER2_36MHZ_DEFAULT	,0x00080010

.define FREQ_240_MHZ				,(240 * 1000 * 1000)
.define FREQ_36_MHZ					,(36 * 1000 * 1000)


	
/*****************************************************************************
#* Function Name:    boot_code
#* Description: 
#*   Boot program
#*   The CPU start at address 0xbfc00000 (in ROM mode)
#*   this is minimal code to jump to KSEG0
#*****************************************************************************/

.extern main
.extern main_c
.extern config_wave400_pll
.extern efuse_set_wait_state_240mhz

    .section ".boot", .text
    .set noreorder

.section ".base_inst", "ax"

main:
boot_code:
base_inst_start:
start:

    /* Disable interrupts, as interrupts have undefined reset value, and wait */
    di
    ehb

    /* check if an alternative bootloader resides on the Flash */
    li   t0, FLASH_BASE_ADDR+8
    
    /* check if the boot entry magic number is there */
    li   t2, TLV_GUARD_BOOT
    lw   t1, -8(t0)
    bne  t1, t2, normal_boot
    nop
    jr   t0
    nop

normal_boot:
	
    /* Check for NMI exception */
    mfc0 t0, $12
    li   t1, STATUS_NMI_SET
    and  t0, t0, t1
    beqz t0, init_pll
    nop
    
    li   t0, handle_nmi
    jalr t0
    nop
	
init_pll:
	/* set ROM wait state to 1 */
	li   t0, WAIT_STATE_REG
	li   t1, 1
	sw   t1, (t0)
	
    /* Initialize the PLL */    
    li   t0, config_wave400_pll
    jalr t0
    nop


	/* Set eFuse wait state to work with 240MHz PLL */
	li   t0,  efuse_set_wait_state_240mhz
	jalr t0
	nop

	/* Fix RAM memories, if any needs fixing */
	li   t0, update_memory_redundency
	jalr t0
	nop

first_shadow_check:
	/* Make sure we're on the first shadow set */
	
	/* Read SRSCtl */
	mfc0 t0, $12, 2
	ehb
	/* If CSS=0 then we're on the first shadow - continue */
	andi t3, t0, 0xf
	beq  t3, zero, clear_gprs
	nop

	/* Else, move to first shadow */	
	/* Set continuation address */
	la   ra, clear_gprs
	mtc0 ra, $14
	mtc0 ra, $30
	ehb

	/* set PSS=0, to be copied to CSS on eret */
	li   t1, SRS_CTL_NO_PSS
	and  t0, t0, t1
	mtc0 t0, $12, 2
	ehb
	eret
	nop

clear_gprs:	
    /*# Initialize the register file*/
   	li   t0, init_gprs
   	jalr t0
    nop

    li   t0, clean_cache
   	jalr t0
    nop
    li   t0, shadowFileInit
   	jalr t0
    nop

/*################################ 1 ################################

################################ 3 ################################*/

    /*# Disable watch exceptions*/
    mtc0    $0, $18

    /*# Clear Watch Status bits*/
    li      $11, 0x7
    mtc0    $11, $19

    /*# Clear WP bit to avoid watch exception upon user code entry
    # Clear IV bit - Interrupts go to general exception vector
    # Clear software interrupts*/
    mtc0    $0, $13

/*################################ 3 ################################ */

	/* 
		Set KSeg0 as cacheable
		write value 3 K0 field of register 16, select 0
		K0=3 means: Cacheable, noncoherent, write-back, write allocate
	*/
	mfc0	$10, $16
	li	$11, 0x7
	not	$11
	and	$10, $11
.if COPY_BOOTROM_TO_RAM==0
	or	$10, 0x3 /* run from cached, copy is done by extrnal mean */
.else
    or  $10, 0x2 /* run from un-cached, copy is done following code,
                    later on we will move to cahce */
.endif
	mtc0	$10, $16
/*################################ 5 ################################*/
    /*# Clear timer Count register*/
    mtc0    $0, $9

    /*# Set timer compare to -1 to delay 1st count=compare
    # Also, clears timer interrupt*/
    li    $10, -1
    mtc0    $10, $11

    /* Use the special interrupt vector at 0xXXXX:X200 in VI */
    lui     t0,0x0080
    mtc0    t0,C0_CAUSE
    
	/* Initialize the stack pointer (so calling C functions is possible)
	   and the global pointer */
    li       $gp, GP_INIT_VALUE
    lui      $sp, %hi(__ghsend_stack)    
    addiu    $sp, $sp, %lo(__ghsend_stack)

/*##--------------------------------------------------------------------
## Configure the Status register:
## Enable all coprocessors, and clear the BEV bit so the
## interrupts will be at 0x80000x00 ,
## Unmask all interrupts but dont enable IE bit. Work in Kernel mode.
##--------------------------------------------------------------------*/
nnext:
    lui  t0,0xf000 
    ori  t0,0x0000  #<E.Z> changed from 0xff00 - for EIC          
    mtc0 t0,C0_SR

/*##--------------------------------------------------------------------
## Select the vectore spacing:
## Spacing of 0 is equivalent to having no vectored interrupts and 
## interrupts fall to 0x80000200
## Minimum vector spacing is 0x20       
##--------------------------------------------------------------------*/

    lui  t0,0x0000
    ori  t0,0x0,0x0020    #<E.Z> changed from 0x0000 - for EIC
    mtc0 t0,$12,1

.if COPY_BOOTROM_TO_RAM==1

    /* we're out of space in the few bytes before the interrupt vector, so continue in the general .text */
    bal copy_to_ram
    nop

    .section .text
/*
 * Copy the copying code to the cached RAM, so that it will perform quickly
 */
copy_to_ram:
    /* call copy_mem(dst: copying_code-base_inst_start+NON_CACHED_RAM , src:copying_code, length: copying_code_end-copying_code)*/
    
    /* s0 is the difference between the actual pc and the address the program was compiled to,
       e.g. 0xa002:0900 - 0xa000:0900 = 0x0002:0000
     */
    bal  4
    nop
    move s0, ra
    li   t3, 8
    subu s0, t3
    la   t3, copy_to_ram
    sub  s0, s0, t3
    li   t0, NON_CACHED_RAM
    la   t1, base_inst_start
    add  t1, t1, s0

    /* param2: src */
    la   a1, copying_code
    add  a1, a1, s0

    /* param1: dst */
    subu a0, a1, t1
    addu a0, a0, t0

    /* param3: length */
    la   a2, copying_code_end
    add  a2, a2, s0
    subu a2, a2, a1
    bal  copy_mem
    nop

    /* jump to the copied code in cached RAM */
    li   t0, CACHED_RAM
    li   t1, NON_CACHED_RAM
    subu a0, a0, t1
    addu a0, a0, t0
    jr   a0
    nop

copying_code:
    /*
     * This code is being executed from kseg0.
     * copy the first part: base_inst_start to copying_code
     */
     /* call copy_mem(dst: NON_CACHED_RAM, src: base_inst_start, length: copying_code-base_inst_start) */
    li   t0, NON_CACHED_RAM
    la   t1, base_inst_start

    /* param2: src */
    add a1, t1, s0    

    /* param1: dst */
    move a0, t0

    /* param3: length */
    la   a2, copying_code
    add  a2, a2, s0
    subu a2, a2, a1
    bal  copy_mem
    nop


    /*
     * Now copy the 2nd part: copying_code_end to end of ROM
     * copy_mem(copying_code_end-base_inst_start+NON_CACHED_RAM , copying_code_end , ROM_SIZE-(copying_code_end-base_inst_start)
     */
    li   t0, NON_CACHED_RAM
    la   t1, base_inst_start

    /* param2: src */
    la   a1, copying_code_end
    add  a1, a1, s0

    /* param1: dst */
    subu a0, a1, t1
    addu a0, a0, t0
    sub  a0, a0, s0

    /* param3: length */
    li   a2, ROM_SIZE
    add  a2, a2, s0
    addu a2, a2, t1
    subu a2, a2, a1
    bal  copy_mem
    nop

    /* change to cache mode just after the copy done
       to elimante instruction cache error due to
       un-initilize memory  */
    mfc0    $10, $16
    li  $11, 0x7
    not $11
    and $10, $11
    or  $10, 0x3
    mtc0    $10, $16

    /* jump over internal code */
    b copying_code_end
    nop

    /* copy_mem (a0: dst, a1: src, a2: length in multiples of 4) */
copy_mem:
    li   t0, 4
    move t2, a0
copy_mem_loop:
    lw   t1, a1
    sw   t1, t2
    addu a1, a1, t0
    addu t2, t2, t0
    subu a2, a2, t0
    bnez a2, copy_mem_loop
    nop
    jr   ra
    nop

copying_code_end:

.endif /* COPY_BOOTROM_TO_RAM */



/*##--------------------------------------------------------------------
## Jump to main C function
##--------------------------------------------------------------------*/

    li $10, main_c
    jr $10
    nop

.section ".rom_version", .data
rom_version:
    .word ROM_VERSION

/*
#* Function Name:    Exception handler
#* Description: 
#*   This is the code in the exceptions intterupt vector (0x8000180)
#*   all it does is to stay in loop so at debugging we can catch this exception
#*   and fix it.
#*/

  .section ".exception", .text    /*# Put this code in the Int vector > 0x80000180*/
  .set noreorder

	wait
	nop
	nop
	nop
	nop
    nop



/*
#* Interrupt Vector
#*/
    .section ".vector", .text    /*# Put this code in the Int vector > 0x80000200*/
    .set noreorder
    .extern irq_handlers /*this is the ISR table "C"*/
    
default_handler:
/*####<E.Z> all unmapped interrupts fall here so just return from interrupt#####*/
    eret
    nop

	/*
	   The following are the 31 interrupt handlers which are implemented
	   according to the following template, depending on N - the interrupt number:

	       .section ".vector_irqN", .text
	   irqN:
	       li      k0,irq_handlers
	       lw      a0,N*4(k0)
	       j       shadowSet_handler
	       nop
	 */

   .section ".vector_irq1", .text
irq1:
   li      k0,irq_handlers
   lw      a0,1*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq2", .text
irq2:
   li      k0,irq_handlers
   lw      a0,2*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq3", .text
irq3:
   li      k0,irq_handlers
   lw      a0,3*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq4", .text
irq4:
   li      k0,irq_handlers
   lw      a0,4*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq5", .text
irq5:
   li      k0,irq_handlers
   lw      a0,5*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq6", .text
irq6:
   li      k0,irq_handlers
   lw      a0,6*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq7", .text
irq7:
   li      k0,irq_handlers
   lw      a0,7*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq8", .text
irq8:
   li      k0,irq_handlers
   lw      a0,8*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq9", .text
irq9:
   li      k0,irq_handlers
   lw      a0,9*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq10", .text
irq10:
   li      k0,irq_handlers
   lw      a0,10*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq11", .text
irq11:
   li      k0,irq_handlers
   lw      a0,11*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq12", .text
irq12:
   li      k0,irq_handlers
   lw      a0,12*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq13", .text
irq13:
   li      k0,irq_handlers
   lw      a0,13*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq14", .text
irq14:
   li      k0,irq_handlers
   lw      a0,14*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq15", .text
irq15:
   li      k0,irq_handlers
   lw      a0,15*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq16", .text
irq16:
   li      k0,irq_handlers
   lw      a0,16*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq17", .text
irq17:
   li      k0,irq_handlers
   lw      a0,17*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq18", .text
irq18:
   li      k0,irq_handlers
   lw      a0,18*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq19", .text
irq19:
   li      k0,irq_handlers
   lw      a0,19*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq20", .text
irq20:
   li      k0,irq_handlers
   lw      a0,20*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq21", .text
irq21:
   li      k0,irq_handlers
   lw      a0,21*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq22", .text
irq22:
   li      k0,irq_handlers
   lw      a0,22*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq23", .text
irq23:
   li      k0,irq_handlers
   lw      a0,23*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq24", .text
irq24:
   li      k0,irq_handlers
   lw      a0,24*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq25", .text
irq25:
   li      k0,irq_handlers
   lw      a0,25*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq26", .text
irq26:
   li      k0,irq_handlers
   lw      a0,26*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq27", .text
irq27:
   li      k0,irq_handlers
   lw      a0,27*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq28", .text
irq28:
   li      k0,irq_handlers
   lw      a0,28*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq29", .text
irq29:
   li      k0,irq_handlers
   lw      a0,29*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq30", .text
irq30:
   li      k0,irq_handlers
   lw      a0,30*4(k0)
   j       shadowSet_handler
   nop


   .section ".vector_irq31", .text
irq31:
   li      k0,irq_handlers
   lw      a0,31*4(k0)
   j       shadowSet_handler
   nop



    .text
/*
#* Function Name:    shadowSet_handler
#* Description: 
#*   Called from the interrupt vector.
#*   Saves  the registers in use for each int calls the ISR and restore registers
#*   this function uses the second register file and thus only special registers need to be 
#*     saved.
#*   The function does not return.
#* Parameters: k0 -C0_EPC, k1- ISR address
#*/
    .globl shadowSet_handler

shadowSet_handler:
    /* Save current registers */
    jal     _tx_thread_context_save
    nop

    /* Execute the IRQ handler */
    jalr    ra,a0
    nop

    /* Restore previous registers and return from interrupt */
    j         _tx_thread_context_restore
    nop
    
    .end shadowSet_handler
 

.set reorder
.set at    /*#end of assembler code now we may use AT*/




/**************************************************************************
 ROM Code only.
 These functions are being called when we're still in ROM,
 and are not used when we're in RAM.
 From hereafter the code will reside in text_rom section.
 **************************************************************************/
	.section ".text_rom" .text

init_gprs:
    or   $1,$0, $0
    or   $2,$0, $0
    or   $3,$0, $0
    or   $4,$0, $0
    or   $5,$0, $0
    or   $6,$0, $0
    or   $7,$0, $0
    or   $8,$0, $0
    or   $9,$0, $0
    or   $10,$0, $0
    or   $11,$0, $0
    or   $12,$0, $0
    or   $13,$0, $0
    or   $14,$0, $0
    or   $15,$0, $0
    or   $16,$0, $0
    or   $17,$0, $0
    or   $18,$0, $0
    or   $19,$0, $0
    or   $20,$0, $0
    or   $21,$0, $0
    or   $22,$0, $0
    or   $23,$0, $0
    or   $24,$0, $0
    or   $25,$0, $0
    or   $26,$0, $0
    or   $27,$0, $0
    or   $30,$0, $0
    jr   $31
    nop


/*
#*Function name: shadowFileInit 
#*Description: This code is part of intialization process,
#*               it's called immediately after Initialize normal shadow register file is completed, 
#*               first we set shadow     register file 1 as PSS - so we can access it 
#*                and then reset all registers using wrpgpr
#*/
shadowFileInit:
.set noreorder
    li        k0,0x40
    nop
    mtc0    k0,$12,2 /*#SRSCtrl */
    nop                

.set reorder
/*#now set 0 to all shadow file GPRs*/
    wrpgpr  $0,$0
    wrpgpr  $1,$0
    wrpgpr  $2,$0
    wrpgpr  $3,$0
    wrpgpr  $4,$0
    wrpgpr  $5,$0
    wrpgpr  $6,$0
    wrpgpr  $7,$0
    wrpgpr  $8,$0
    wrpgpr  $9,$0
    wrpgpr  $10,$0
    wrpgpr  $11,$0
    wrpgpr  $12,$0
    wrpgpr  $13,$0
    wrpgpr  $14,$0
    wrpgpr  $15,$0
    wrpgpr  $16,$0
    wrpgpr  $17,$0
    wrpgpr  $18,$0
    wrpgpr  $19,$0
    wrpgpr  $20,$0
    wrpgpr  $21,$0
    wrpgpr  $22,$0
    wrpgpr  $23,$0
    wrpgpr  $24,$0
    wrpgpr  $25,$0
    wrpgpr  $26,$0
    wrpgpr  $27,$0
    wrpgpr  $28,$0
    wrpgpr  $29,$0
    wrpgpr  $30,$0
    wrpgpr  $31,$0
    mtc0    $0,$12,2 
    jr      ra /*#return */
    or      k0,$0, $0    /*##takes advantage of jump hazards(next instruction executed)*/
    

clean_cache:    
.set noreorder

        mfc0    $10, $16, 1            /* # .word 0x400a8001 */
        nop
        /* # Isolate I$ Line Size */
        sll     $11, $10, 10
        srl     $11, 29

        /* # Skip ahead if No I$ */
        beq     $11, $0, no_icache
        nop
        li      $14, 2
        sllv    $11, $14, $11           /* # Now have true I$ line size in bytes */

        sll     $12, $10, 7
        srl     $12, 29
        li      $14, 64
        sllv    $12, $14, $12           /* # I$ Sets per way */

        sll     $13, $10, 13
        srl     $13, 29                 /* # I$ Assoc (-1) */
        add     $13, 1
        mul     $12, $12, $13           /* # Total number of sets */

        lui     $14, 0x8000             /* # Get a KSeg0 address for cacheops */

        /* # Clear TagLo/TagHi registers */
        mtc0    $0, $28
        mtc0    $0, $29
        mtc0    $0, $28, 2
        mtc0    $0, $29, 2

        move    $15, $12

        /* # Index Store Tag Cache Op */
        /* # Will invalidate the tag entry, clear the lock bit, and clear the LRF bit */
ic_clean_loop:    
        cache   0x8, 0($14)
        add     $15, -1                 /* # Decrement set counter */
        add     $14, $11                /* # Get next line address */
        bne     $15, $0, ic_clean_loop
        nop
no_icache:
        /* # Isolate D$ Line Size */
        sll     $11, $10, 19
        srl     $11, 29

        /* # Skip ahead if No D$ */
        beq     $11, $0, no_dcache
        nop
        li      $14, 2
        sllv    $11, $14, $11           /* # Now have true D$ line size in bytes */

        sll     $12, $10, 16
        srl     $12, 29
        li      $14, 64
        sllv    $12, $14, $12           /* # D$ Sets per way */

        sll     $13, $10, 22
        srl     $13, 29                 /* # D$ Assoc (-1) */
        add     $13, 1

        mul     $12, $12, $13           /* # Get total number of sets */

        lui     $14, 0x8000             /* # Get a KSeg0 address for cacheops */

        /* # Clear TagLo/TagHi registers */
        mtc0    $0, $28
        mtc0    $0, $29

        move    $15, $12

       /* # Index Store Tag Cache Op */
       /* # Will invalidate the tag entry, clear the lock bit, and clear the LRF bit */
dc_clean_loop:        
        cache   0x9, 0($14)
        add     $15, -1                 /* # Decrement set counter */
        add     $14, $11                /* # Get next line address */
        bne     $15, $0, dc_clean_loop
        nop
no_dcache:

    jr ra
    nop
    
.set reorder


handle_nmi:

	/* We're in NMI */
    /* According to Amir R., there is no need to disable watchdog,
    	including after Watchdog NMI event */

	/* nofity the firmware handler, if exists */
//	li   t0, call_nmi_handler
//	jalr t0
//	nop
	
	/* take down the USB PHY */ 
	li   t0, USB_DCTL_REG
	lw   t1, (t0)
	ori  t1, t1, USB_DCTL_SOFT_DISC
	sw   t1, (t0)
	
	/* wait for 2 seconds so the PC will recognize disconnection */
	
	/* Enable counting - Set Cause_DC to zero */
	mfc0 t0, $13
	lui  t1, CAUSE_DC_0_UPPER
	not  t1, t1
	and  t0, t0, t1
	mtc0 t0, $13
	ehb
	
    li   t0, TWO_SEC_COUNT                   # 2 seconds value - see above
    mtc0 t0, $11                             # Set compare register, reset count reg.
    mtc0 zero, $9                            # Initialize CP0 count register to zero
    ehb
reset_busy_loop:
	mfc0 t1, $9
	ehb
	sub  t1, t1, t0
	blez t1, reset_busy_loop
	nop

	/* Issue a full chip reset */
	li   t0, FULL_CHIP_RESET_REG
	li   t1, FULL_CHIP_RESET_BIT
	lw   t2, (t0)
	and  t2, t2, t1
	sw   t2, (t0)
	wait
	nop

/**
	Currently there are two registers in the shared registers that youre configuring.
	The default values of both registers are:
						 npu_efuse_timer_cfg_1      <= 32'h80064;
						 npu_efuse_timer_cfg_2      <= 32'h80008;
	the two registers are separated into two fields each:
	npu_efuse_timer_cfg_1/2[15:0] and npu_efuse_timer_cfg_1/2[31:16]
	the default values are related to 36Mhz clock.
	If you intend to increase the clock frequency you should multiply each field by
	(NEW_FREQ/36mhz) * npu_efuse_timer_cfg_1/2[15:0] and (NEW_FREQ/36mhz) * npu_efuse_timer_cfg_1/2[31:16]
*/
efuse_set_wait_state_240mhz:
	li   t0, EFUSE_TIMER1_CFG_REG_ADDR
	li   t1, ((FREQ_240_MHZ/FREQ_36_MHZ) * (EFUSE_TIMER1_36MHZ_DEFAULT & 0xFFFF)) + (((FREQ_240_MHZ/FREQ_36_MHZ) * (EFUSE_TIMER1_36MHZ_DEFAULT >> 16)) << 16);
	sw   t1, (t0)
	li   t0, EFUSE_TIMER2_CFG_REG_ADDR
	li   t1, ((FREQ_240_MHZ/FREQ_36_MHZ) * (EFUSE_TIMER2_36MHZ_DEFAULT & 0xFFFF)) + (((FREQ_240_MHZ/FREQ_36_MHZ) * (EFUSE_TIMER2_36MHZ_DEFAULT >> 16)) << 16);
	sw   t1, (t0)
    jr ra
    nop

