/**
 @file    pll.s

 @brief    Assembly file to configure PLL of wave400 chip

 @date December, 2010
 @author Amir Rosenblum, Lantiq
 @author Ishai Asa, Goeny Ltd
 */

    .section ".text_rom" .text

/*
 * Configure Wave400 PLL
 */
    .globl config_wave400_pll
	
    /* System Interface definitions */

    .define    PLL_LOCK_TIME_IN_US,                       1000
    .define    PLL_LOCK_TIME_IN_US_rstq1,                 1200
    .define    PLL_LOCK_TIME_IN_US_rstq2,                 1600
    .define    PLL_LOCK_TIME_IN_US_rstq3,                 2400
    .define    MAC_CLOCK_FREQ_IN_MHZ,                     36
    .define    FAST_CLK_DIV,                              2

    .define    sys_if_base_address,                       0xa018
    .define    rxtd_base_address,                         0xaf30
    .define    ht_ext_base,                               0xa01f

    .define    secure_write_addr,                         (0x1A << 2)

    .define    pll_mul_div_addr,                          (0x10 << 2)
    .define    pll_n_field,                               0
    .define    pll_refdiv_field,                          9

    .define    pll_control_reg,                           (0x11 << 2 )
    .define    pll_control_pll_off_field,                 3
    .define    pll_control_pll_bypass_field,              4
    .define    pll_control_pll_refdly_field,              0

    .define    pll_param_reg,                             (0x14 << 2)
    .define    pll_param_vco_cal_field,                   9
    .define    pll_param_vco_field,                       4
    .define    pll_param_filter_field,                    2
    .define    pll_param_pllcp_field,                     0

    .define    cpu_clk_select_reg,                        (0x16 << 2)
    .define    phy_clk_select_mac_reg,                    (0x2f << 2)
    .define    mac2phy_rst_reg,                           (0x45 << 2)
    .define    cpu_clock_gen_reg,                         (0x1e << 2)
    .define    cpu_clock_gen_cpu_phy_div_enable_field,    5
    .define    cpu_clock_gen_cpu_phy_div_field,           3
    .define    cpu_clock_gen_modem_pre_div_field,         2
    .define    cpu_clock_gen_modem_xo_sel_field,          1
    .define    cpu_clock_gen_lvds_xo_sel_field,           0

    .define    digital_lpbk_reg,                          ( 0x25 << 2)
    .define    digital_lpbk_rx0_polarity_sel_field,       0
    .define    digital_lpbk_rx1_polarity_sel_field_field, 1
    .define    digital_lpbk_rx2_polarity_sel_field_field, 2
    .define    digital_lpbk_tx0_polarity_sel_field,       3
    .define    digital_lpbk_tx1_polarity_sel_field,       4
    .define    digital_lpbk_ch0_lpbk_clk_src_field,       5
    .define    digital_lpbk_ch1_lpbk_clk_src_field,       6
    .define    digital_lpbk_rx0_lpbk_select_field,        7
    .define    digital_lpbk_rx1_lpbk_select_field,        9
    .define    digital_lpbk_rx2_lpbk_select_field,        11
    .define    digital_lpbk_rx_aux_polarity_sel_field,    13

    .define    pll_en_div_reg,                            (0x10 << 2)
    .define    mac_clk_div_d2a_field,                     0
    .define    mac_clk_en_d2a_field,                      3
    .define    tbus_clk_div_d2a_field,                    4
    .define    tbus_clk_en_d2a_field,                     7
    .define    phy_clk_div_d2a_field,                     8
    .define    phy_clk_en_d2a_field,                      11
    .define    adc_clk_div_d2a_field,                     12
    .define    adc_clk_en_d2a_field,                      15

/* ## MAC Clock Div */
    .define    mac_clk_div_d2a_320MHz,                    4
    .define    mac_clk_div_d2a_480MHz,                    0
    .define    mac_clk_div_d2a_240MHz,                    1
    .define    mac_clk_div_d2a_80MHz,                     6
    .define    mac_clk_div_d2a_120MHz,                    2
    .define    mac_clk_div_d2a_160MHz,                    5
    .define    mac_clk_div_d2a_40MHz,                     7
    .define    mac_clk_div_d2a_60MHz,                     3

/* ## Phy Clock Div */
    .define    phy_clk_div_d2a_240MHz,                    0
    .define    phy_clk_div_d2a_480MHz,                    1
    .define    phy_clk_div_d2a_60MHz,                     2
    .define    phy_clk_div_d2a_120MHz,                    3


    .define    pll_ctl_reg,                               (0x12 << 2)
    .define    bbpll_config_d2a_field,                    0
    .define    bbpll_pup_d2a_field,                       8
    .define    bbpll_ldodis_d2a_field,                    9
    .define    bbpll_rstq_cnt_d2a_field,                  10
    .define    fcsi_ldo_pd_d2a_field,                     13

    .define    pll_lock_reg,                              (0x14 << 2)
    .define    not_in_cached_value,                       0
    .define    in_cached_value,                           1


    .define    tmr_prescale_addr,                         0x0e << 2
    .define    tmr_enable_addr,                           0x0f << 2
    .define    tmr_disable_addr,                          0x3c << 2
    .define    mac_ht_extensions_tmr_irq ,                ( 0x0034 )
    .define    tmr_00_init_addr,                          0x10 << 2
    .define    tmr_00_cnt_addr,                           0x20 << 2

    /***************************************************/
    /* Clock Manipulation 240 (default), 160 , 120 MHz  */
    /***************************************************/

    .define    sys_clk_fast_divider_reg,                  (0x13 << 2)
    .define    pll_clk_sel_reg,                           (0x2f << 2)

    .define    fast_mode_mac_div,                         mac_clk_div_d2a_480MHz
    .define    slow_mode_mac_div,                         mac_clk_div_d2a_240MHz
    .define    seperate_mac_phy_clks_equ,                 0

config_wave400_pll:
    .define    SHARED_REG_BASE,                           0xa71c0000

    .define    EFUSE_DATA,                                0x0120
    .define    EFUSE_ADDRESS,                             0x0124
    .define    EFUSE_CONTROL,                             0x0128
    .define    EFUSE_STATUS,                              0x012c
    .define    EFUSE_TIMER_CONFIG1,                       0x0130
    .define    EFUSE_TIMER_CONFIG2,                       0x0134

    .define    EFUSE_BUSY,                                0x1

    /* Hyperion 4 PLL */
    move    $30, $31
    lui     $10, sys_if_base_address

    /* Modify the MAC clock divider field */
    li      $4, 0xa7000000
    lw      $3, 0x0060($4)
    li      $4, 0x1<<19 /* CPU fast mode indication @ NPU system I/F */

    /* Store fast mode indication in $4: 1 = fast mode, 0 = slow mode */
    and     $3, $3, $4
    srl     $4, $3, 19

    /* PLL turning on + PLL config */

    /* Read offsets 50, 51 from efuse array */

    /* Verify efuse control machine is not busy */
    li      $5, SHARED_REG_BASE
poll_efuse_machine_busy:
    lw      $13, EFUSE_STATUS($5)
    andi    $13, EFUSE_BUSY
    bne     $13, $0, poll_efuse_machine_busy
    nop

    /* Write address to read from */
    li      $13, 0x32<<3
    sw      $13, EFUSE_ADDRESS($5)
    /* Set READ bit - also triggers machine operation */
    sw      $0, EFUSE_CONTROL($5)

    /* Wait for machine ready */
wait_if_busy:
    lw      $13, EFUSE_STATUS($5)
    andi    $13, EFUSE_BUSY
    bne     $13, $0, wait_if_busy
    nop

    /* Load read data into $12 */
    lw      $12, EFUSE_DATA($5)

    /* Write address to read from */
    li      $13, 0x33<<3
    sw      $13, EFUSE_ADDRESS($5)
    /* Set READ bit - also triggers machine operation */
    sw      $0, EFUSE_CONTROL($5)

    /* Wait for machine ready */
wait_if_busy2:
    lw      $13, EFUSE_STATUS($5)
    andi    $13, EFUSE_BUSY
    bne     $13, $0, wait_if_busy2
    nop

    /* Load read data into $12 */
    lw      $13, EFUSE_DATA($5)
	andi	$14,$13,0x8 /* $14 will hold bit[3] which is "1" if we have a delay loop for PLL lock of "0" if we want to rely on PLL lock indication */
    andi    $13, 0x7
    or      $5, $13, $0 /* Copy RST_Q to $5 */
    sll     $13, 10
    or      $12, $12, $13
    li      $13, 0xffffe300

    lw      $2, pll_ctl_reg($10)
    and     $2, $2, $13
    li      $3, ( 1 << bbpll_pup_d2a_field)
    or      $2, $3, $2
    or	$2,$2,$12
    bal     secure_write
    nop

    sw      $2, pll_ctl_reg($10)

    li      $13, (MAC_CLOCK_FREQ_IN_MHZ - 1)
    beq     $4, $0, fast_mode_delay_continue
    nop

    /* If fast mode, override prev. value */
    li      $13, (MAC_CLOCK_FREQ_IN_MHZ/FAST_CLK_DIV - 1)

fast_mode_delay_continue:
	bne	$14,$0,do_pll_delay_loop
    nop
pll_lock_poll_loop:
	lw	$2,pll_lock_reg($10)
	beq	$2,$0,pll_lock_poll_loop
    nop
	b	configure_pll_clocks
    nop

do_pll_delay_loop:
        bal     power_up_delay        
    nop

configure_pll_clocks:
    /* Update the AFE clock according to equ algo_afe_clock_freq_div*/
    /* Updated to Feronia V32 */
    /* for 80MHz algo_afe_clock_freq_div = 6  */
    /* for 40MHz algo_afe_clock_freq_div = 12 */

    /* Configuring the PLL clocks (0(AFE)=80M, 1(LVDS)= 480M, 2(PHY)=240M, 3(MAC)=160M) */
    lw      $2, pll_en_div_reg($10)

    /* We need to modify MAC clock divider's default value value set epending on fast mode and mac freq */
    beq     $4, $0, set_slow_mode_mac_divider
    nop

    li      $3, 0xfffffff8
    and     $2, $2, $3
set_fast_mode_mac_divider:
    li      $3, (fast_mode_mac_div   << mac_clk_div_d2a_field)
    b       set_mac_clock_div
    nop

set_slow_mode_mac_divider:
    li      $3, (slow_mode_mac_div   << mac_clk_div_d2a_field)
set_mac_clock_div:
    or      $2, $2, $3
    li      $3, ((1 << adc_clk_en_d2a_field) | (1 << phy_clk_en_d2a_field) | (1 << tbus_clk_en_d2a_field) | (1 << mac_clk_en_d2a_field))
    or      $2, $3, $2
    bal     secure_write
    nop

    sw      $2, pll_en_div_reg($10)


switch_to_pll:	
    /*Select fast clock (sel=2) */
    bal     secure_write
    nop

    li      $2, 0x00000002
    sw      $2, cpu_clk_select_reg($10)

    /* check if need to seperate the mac and phy clocks */
    li      $2, seperate_mac_phy_clks_equ
    beq     $2, $0, phy_mac_synchronized
    nop

    /* sepearte mac and phy */
    /* Start Phy clock*/
    lw      $2, cpu_clock_gen_reg($10)
    ori     $2, $2, ( 1 << cpu_clock_gen_cpu_phy_div_enable_field)
    sw      $2, cpu_clock_gen_reg($10)

    /* Hold the phy in reset before changing the clock mux */
    bal     secure_write
    nop

    sw      $0, mac2phy_rst_reg($10)

    /* Select PHY clock from PHY not MAC */
    li      $2, 0x0
    sw      $2, phy_clk_select_mac_reg($10)

    /* Release Phy from reset */
    bal    secure_write
    nop

    li      $2, 0x1
    sw      $2, mac2phy_rst_reg($10)


phy_mac_synchronized:
    li      $9, rxtd_base_address

    b       std_init_cont
    nop

secure_write:
    li      $12, 0x0000aaaa
    sw      $12, secure_write_addr($10)
    li      $12, 0x00005555
    sw      $12, secure_write_addr($10)
    jr      $31
    nop

power_up_delay:

    /* Enable GCLK to HT Extensions units */
    lui     $20, ht_ext_base
    li      $11, 0x000000ff
    sw      $11, 0x0000($20)

    /* Cofigure prescaler */
    srl     $14, $13, 1       # load to $14 prescaler value for reduced freq (half of $13)
    sll     $14, $14, 16      # reduced freq value resides in upper 16ms      # reduced freq value resides in upper 16msbb
    or      $13, $14, $13
    sw      $13, tmr_prescale_addr($20)


    /* Configure all timers */
    li      $11, PLL_LOCK_TIME_IN_US
    /* We start with 1000us. Then, we decrement $5 which is rst_q and increase lock time accordingly */
    beq     $5, $0, do_configure
    nop

    addiu   $5, -1
    li      $11, PLL_LOCK_TIME_IN_US_rstq1
    beq     $5, $0, do_configure
    nop

    addiu   $5, -1
    li      $11, PLL_LOCK_TIME_IN_US_rstq2
    beq     $5, $0, do_configure
    nop

    li      $11, PLL_LOCK_TIME_IN_US_rstq3

do_configure:
    sw      $11, tmr_00_init_addr($20)

    /* Enable timer 0 */
    li      $11, 0x00000001
    sw      $11, tmr_enable_addr($20)

wait_timer_expired:
    lw      $11, tmr_00_cnt_addr($20)
    bne     $11, $0, wait_timer_expired
    nop

     /* clear  timer irq 0 */
    li      $11, 0x00000001
    sw      $11, mac_ht_extensions_tmr_irq($20)
     /* disable GCLK to HT Extensions units */
    li      $11, 0x00000000
    sw      $11, 0x0000($20)
    jr      $31
    nop

std_init_cont:
    nop
    nop
    nop
    nop
    nop

    move    $31, $30
    jr      $31
    nop
