/*
 *                ThreadX C/C++ Library Support
 *
 *      Copyright 1983-2005 Green Hills Software,Inc.
 *
 *  This program is the property of Green Hills Software, Inc,
 *  its contents are proprietary information and no part of it
 *  is to be disclosed to anyone except employees of Green Hills
 *  Software, Inc., or as agreed in writing signed by the President
 *  of Green Hills Software, Inc.
 */
#include "tx_ghs.h"
#ifndef TX_DISABLE_ERROR_CHECKING
#define TX_DISABLE_ERROR_CHECKING
#endif
#include "tx_api.h"
#include <setjmp.h>
#include <string.h>
/* Allow these routines to access the following ThreadX global variables.  */
extern ULONG      _tx_thread_created_count;
extern TX_THREAD *_tx_thread_created_ptr;
extern TX_THREAD *_tx_thread_current_ptr;

/*
 * ThreadX C and C++ thread-safe library support routines.
 *
 * This implementation merely tries to guarantee thread safety within
 * individual C library calls such as malloc() and free(), but it does
 * not attempt to solve the problems associated with the following
 * multithreaded issues:
 *
 * 1. Use of errno.  This can be made thread-safe by a Premium (source)
 *    licensee of ThreadX by adding errno to TX_THREAD_PORT_EXTENSION and using
 *    that within a modified version of libsrc/ind_errno.c.
 *
 * 2. Thread safety ACROSS library calls.  Certain C library calls either
 *    return pointers to statically-allocated data structures or maintain
 *    state across calls.  These include strtok(), asctime(), gmtime(),
 *    tmpnam(NULL), signal().  To make such C library routines thread-safe
 *    would require adding a ThreadLocalStorage struct to the thread control
 *    block TX_THREAD.  Since relatively few applications make use of these
 *    library routines, the implementation provided here uses a single, global
 *    ThreadLocalStorage data structure rather than greatly increasing the size
 *    of the thread control block TX_THREAD.
 *
 *    The ThreadX global variable _tx_thread_current_ptr points to the
 *    current thread's control block TX_THREAD. If a ThreadLocalStorage struct
 *    called tx_tls is placed in TX_THREAD, the function GetThreadLocalStorage
 *    should be modified to return &(_tx_thread_current_ptr->tx_tls).
 */

ThreadLocalStorage GlobalTLS;

ThreadLocalStorage *GetThreadLocalStorage()
{
    return &GlobalTLS;
}

/*
 * Use a global ThreadX mutex to implement thread safety within C and C++
 * library routines.
 *
 */
TX_MUTEX __ghLockMutex;

/*
 * Use tx_mutex_get to implement __ghsLock
 */
void __ghsLock(void)
{
    tx_mutex_get(&__ghLockMutex, TX_WAIT_FOREVER);
}

/*
 * Use tx_mutex_put to implement __ghsUnlock
 */
void __ghsUnlock(void)
{
    tx_mutex_put(&__ghLockMutex);
}

/*
 * Minimal implementations for saving and restoring the Signal context.
 */
int __ghs_SaveSignalContext(jmp_buf jmpbuf)
{
    return 0;
}

void __ghs_RestoreSignalContext(jmp_buf jmpbuf)
{
}

/* ThreadX Initialization function prototype.  */
void _tx_initialize_kernel_setup(void);

void __gh_lock_init(void)
{
    /* Initialize the low-level portions of ThreadX. */
    _tx_initialize_kernel_setup();

    /* Create the global thread lock mutex.  */
    tx_mutex_create(&__ghLockMutex, "__ghLockMutex", TX_NO_INHERIT);
}

/*
 * return the thread-specific pointer for C++ exception handling.
 *
 * This pointer is initialized by the routine __cpp_except_init().
 *
 */
void *__get_eh_globals(void) 
{
    if (_tx_thread_current_ptr)

        /* Return thread-specific __eh_globals pointer.  */
        return(_tx_thread_current_ptr->tx_thread_eh_globals);
    else
        /* Return the global __eh_globals pointer.  */
        return(GlobalTLS.__eh_globals);
}

/*
 * Use the global lock to serialize file I/O accesses.
 */
void __ghs_flock_file(void *addr)
{
    tx_mutex_get((TX_MUTEX *)addr, TX_WAIT_FOREVER);
}

void __ghs_funlock_file(void *addr)
{
    tx_mutex_put((TX_MUTEX *)addr);
}
int __ghs_ftrylock_file(void *addr)
{
    return -1;
}
void __ghs_flock_create(void **addr)
{
    *addr = (void *)(&__ghLockMutex);
}
void __ghs_flock_destroy(void *addr) {}


/*
 * ThreadX Peak Stack Checking support routines.
 *
 * All of these routines are called by MULTI's ThreadX-aware debugging
 * package to determine the peak stack use for one thread or for all threads.
 *
 * These routines are included in this file in order to guarantee that they will
 * be available while debugging with MULTI.  These routines are not referenced by
 * any other part of the ThreadX system.
 *
 * _txs_thread_stack_check:         return the peak stack usage for a thread.
 *
 * _txs_thread_stack_check_2:       store the peak stack usage for all threads
 *                                  in the tx_stack_size field of each thread
 *                                  control block, TX_THREAD.  This routine takes
 *                                  advantage of the redundancy within the TX_THREAD
 *                                  structure since tx_stack_size can be computed
 *                                  from the tx_stack_start and tx_stack_end
 *                                  fields of TX_THREAD.
 *
 * _txs_thread_stack_check_2_fixup: clean up from the _txs_thread_stack_check_2
 *                                  call by computing the stack size for each
 *                                  thread and storing the result in the
 *                                  tx_stack_size field of each thread control
 *                                  block TX_THREAD.
 *
 * These three routines do not support architectures such as i960 or StarCore
 * where the stack grows up instead of down.
 *
 */
#ifndef TX_DISABLE_STACK_FILLING

ULONG _txs_thread_stack_check(TX_THREAD *thread_ptr)
{
    
UCHAR      *cp;                  /* Pointer inside thread's stack.  */

    /* Search through the thread's stack to find the highest address modified.  */
    for ( cp = (UCHAR *)thread_ptr->tx_thread_stack_start;
        cp <= (UCHAR *)thread_ptr->tx_thread_stack_end; ++cp ) {

        /* Check if this byte in the stack contains something other than TX_STACK_FILL.  */
        if (*cp != (UCHAR)TX_STACK_FILL) {

            /* Assume cp points to the locating marking the peak stack use.
               Return the number of bytes from cp up to and including the
               end of the stack.  */
            return (((ULONG)thread_ptr->tx_thread_stack_end) - (ULONG)cp + 1);
        }
    }
    return thread_ptr->tx_thread_stack_size;
}


int _txs_thread_stack_check_2(void) {
    UCHAR       *cp;                  /* Pointer inside thread's stack.  */
    TX_THREAD   *tp;                  /* Pointer to each thread.  */

    /* If no threads are created, return immediately.  */
    if (!_tx_thread_created_count)
        return 0;

    /* Start iterating through the threads in the system.  Assume that we always
       have at least one thread (the system timer thread) in the system.  */
    tp = _tx_thread_created_ptr;

    do {

        /* Search through the thread's stack to find the highest address modified.  */
        for ( cp = (UCHAR *)tp->tx_thread_stack_start; cp <= (UCHAR *)tp->tx_thread_stack_end;
        ++cp ) {

            /* Check if this byte in the stack contains something other than TX_STACK_FILL.  */
            if (*cp != (UCHAR)TX_STACK_FILL) {

                /* Assume cp points to the locating marking the peak stack use.
                   Store the number of bytes from cp up to and including the
                   end of the stack in the tx_stack_size field.  */
                tp->tx_thread_stack_size = ((ULONG)tp->tx_thread_stack_end) - (ULONG)cp + 1;
                break;
            }

        }

        /* Continue with the next thread.  */
        tp = tp->tx_thread_created_next;

    /* Loop until we point to the first thread again.  */
    } while ( tp != _tx_thread_created_ptr );

    return 0;
}

int _txs_thread_stack_check_2_fixup(void) {
    TX_THREAD   *tp;                  /* Pointer to each thread.  */

    /* If no threads are created, return immediately.  */
    if (!_tx_thread_created_count)
        return 0;

    /* Start iterating through the threads in the system.  Assume that we always
       have at least one thread (the system timer thread) in the system.  */
    tp = _tx_thread_created_ptr;

    do {

        /* Compute the tx_stack_size field by using the tx_stack_end and
           tx_stack_start fields.  */
        tp->tx_thread_stack_size = (ULONG)tp->tx_thread_stack_end-(ULONG)tp->tx_thread_stack_start+1;

        /* Continue with the next thread.  */
        tp = tp->tx_thread_created_next;

    /* Loop until we point to the first thread again.  */
    } while ( tp != _tx_thread_created_ptr );

    return 0;
}


void _tx_bring_in_ghs_support(void)
{
    /* This routine is never called, it is just there to make
       sure the stack checking services are brought in by the
       library.  */
    _txs_thread_stack_check(TX_NULL);
}

#endif /* TX_DISABLE_STACK_FILLING */
