/*! \addtogroup zoro_common_memory_group1
*  @{
*/

/**
    @file    zr_memory.h

		heap management: allocate and free variable length chunks.

    @brief   Allocate/free memory interface

    @date December, 2010

    @author Ishai Asa, Goeny Ltd

    <b> Copyright (c) 2009-2010 Goeny Ltd. </b>\n
    43 Hamelacha street, P.O. Box 8786, Poleg Industrial Park, Netanaya, ZIP 42505 Israel\n
    All rights reserved\n\n
    Proprietary rights of Goeny Ltd are involved in the
    subject matter of this material. All manufacturing, reproduction,
    use, and sales rights pertaining to this subject matter are governed
    by the license agreement. The recipient of this software implicitly
    accepts the terms of the license. This source code is the unpublished
    property and trade secret of Goeny Ltd.
    It is to be utilized solely under license from Goeny Ltd and it
    is to be maintained on a confidential basis for internal company use
    only. It is to be protected from disclosure to unauthorized parties,
    both within the Licensee company and outside, in a manner not less stringent
    than that utilized for Licensee's own proprietary internal information.
    No copies of the source or object code are to leave the premises of
    Licensee's business except in strict accordance with the license
    agreement signed by Licensee with Goeny Ltd.\n\n

    For more details - http://zoro-sw.com <or> http://goeny.com
    email: info@zoro-sw.com
*/
#ifndef _ZR_MEMORY_H_
#define _ZR_MEMORY_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "zr_common.h"

// convert application processor address to USB internal DMA address
    //    BB CPU to IRAM: 0x0000_0000 (256KB)
    //    BB CPU to Shared RAM: 0x0600_0000
    //    BB CPU to Shared RAM cyclic buffer: 0x0610_0000
    //    USB Internal DMA to IRAM: 0x0624_0000 (256KB)
    //    USB Internal DMA to Shared RAM: 0x0600_0000
    //    USB Internal DMA to Shared RAM cyclic buffer: 0x0610_0000

// MIPS 4KEc memory map
#define MIPS_SHRD_LINR_RAM_BASE			(0xA6000000)
#define MIPS_SHRD_CICL_RAM_BASE			(0xA6100000)
#define MIPS_IRAM_CAC_BASE				(0x80000000)
#define MIPS_IRAM_UNCAC_BASE			(0xA0000000)

// USB internal DMA memory map
#define USB_IRAM_BASE           	 	(0x06240000)
#define USB_SHRD_LINR_BASE         	 	(0x06000000)
#define USB_SHRD_CICL_BASE         	 	(0x06100000)

// Memory regions masks
#define IRAM_MASK						(0x0003FFFF) /* 256 KB */
#define SHRD_MASK						(0x0001FFFF) /* 128 KB */
#define SHRD_CS_MASK					(0x06100000)

// Convert IRAM cache <--> un-cached addresses
#define TO_PHYS_MASK            		(0x0FFFFFFF)

#define TO_CAC(x)																	\
		((uint32)x < (uint32)MIPS_SHRD_LINR_RAM_BASE)							?	\
		((uint32)MIPS_IRAM_CAC_BASE   | ((uint32)(x) & (uint32)TO_PHYS_MASK))	:	\
		((uint32)x)

#define TO_CAC_PTR(x)			(void *)(TO_CAC(x))


#define TO_UNCAC(x)																	\
		((uint32)x < (uint32)MIPS_SHRD_LINR_RAM_BASE)							?	\
		((uint32)MIPS_IRAM_UNCAC_BASE | ((uint32)(x) & (uint32)TO_PHYS_MASK))	:	\
		((uint32)x)

#define TO_UNCAC_PTR(x)			(void *)(TO_UNCAC(x))

// Convert MIPS' address to USB internal DMA' address
#define MIPS_TO_USB(x)														\
		((uint32)x < (uint32)MIPS_SHRD_LINR_RAM_BASE)					?	\
		(uint32)USB_IRAM_BASE | ((uint32)x & (uint32)IRAM_MASK)			:	\
		((uint32)x  & (uint32)SHRD_CS_MASK )  | 							\
			((uint32)x & (uint32)SHRD_MASK)

/**
 * Initializes the memory SW module for future allocations and deallocations.
 * This function must be called first before any other zr_* memory function.
 */
void zr_memory_init(void);

/**
 * Terminates the memory SW module.
 * No other zr_* memory function may be called after this function call.
 */
void zr_memory_term(void);

/**
 * Allocates memory of size bytes long.
 * Returns a pointer to the allocated memory.
 */
void *zr_malloc(size_t size);

/**
 * Frees up a previously allocated memory.
 */
void zr_free(void *ptr);

/**
 * Returns the number of bytes available on the heap.
 */
uint32 zr_memory_available(void);


/**
 * Macros to allocate/free buffer to be used by USB AHB master. This buffer
 * resides within un-cached memory, and we should take care of cache-write
 * burst cycles, which may overwrite the data which is stored un-cached
 */
#define CACHE_WRITE_BURST_SIZE			(16)

void *zr_malloc_usb_master(size_t size);
void zr_free_usb_master(void *ptr);


#ifdef __cplusplus
}
#endif
#endif /* _ZR_MEMORY_H_ */

/*! @} */
