/************************************************************************************
*	
*	File:		  TinyKernel_Messages.c
*	Class/Module: 
*	Description:  A detailed description of the Module, its  purpose attributes and
*		 	  whatever information the user & maintainer might find valuable.
*
*	COPYRIGHT: 
*		(C) Lantiq Israel Ltd.
*		All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*		copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/

   /*************************************************************************/
   /***						Include Files							  ***/
   /*************************************************************************/
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "OSAL_Api.h"
#include "TinyKernel_Api.h"
#include "TinyKernel_Messages.h"
#include "TinyKernel_Scheduler.h"
#include "System_MainApi.h"
#include "Debug_Trace.h"
#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_TINY_KERNEL
#define LOG_LOCAL_FID 0

/*************************************************************************/
/***						Local defines							  ***/
/*************************************************************************/

/* The following are masks that are applied to the bListNoAndRefCount */
/* element of the K_MSG structure.									*/
/* Bits 7-4 are used for message pool information					 */
/* Bit  3 is reserved for trapping reference count over/underflow	 */
/* Bits 2-0 are used for the reference count (0-7: 0 means 'free')	*/

#define bLNARC_POOL_MASK			 0xf0
#define bLNARC_POOL_OVERFLOW_BIT	 0x08
#define bLNARC_REF_COUNT_MASK		0x07

/* Define a default limit on ref count. A smaller value can be specified
 * in the kernel configuration file to allow a build to specify that it is
 * only intended to use a smaller reference count. */

#define K_MAX_REF_COUNT (7)

#define bLNARC_FAST_POOL			 0xe0

#define bLNARC_GET_POOL_ID(x)		((x) & bLNARC_POOL_MASK)
#define bLNARC_GET_REF_COUNT(x)	  ((x) & bLNARC_REF_COUNT_MASK)

/*************************************************************************/
/***			   Fast message pool								   ***/
/*************************************************************************/

typedef struct K_FAST_MSG
{
	K_MSG_HEADER	sMsgHeader;
	byte			abData[K_FAST_MSG_SIZE];
} K_FAST_MSG;

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
K_FAST_MSG	*TinyKernel_FastMsgPool;
#else
K_FAST_MSG	TinyKernel_FastMsgPool[K_NO_OF_FAST_MSG];

#endif

K_MSG		*TinyKernel_FastMsgFreeList;

/*************************************************************************/
/***				Static Functions								   ***/
/*************************************************************************/
static void			TinyKernel_InitFastMessagePool(void);
static void			TinyKernel_FreeFastMessage(K_MSG *);


/****************************************************************************
 **
 ** NAME:		   TinyKernel_InitMessages
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	initialise the KNL message data structures,
 **
 ****************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif
void TinyKernel_InitMessages(void)
{
	/*************************************/
	/* Fast pool initialisation		  */
	/*************************************/
	TinyKernel_InitFastMessagePool();
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif

/****************************************************************************
 **
 ** NAME:		   TinyKernel_InitPersistentMsg
 **
 ** PARAMETERS:	 Pointer to message defined in user space (outwith kernel)
 **				 Nominal length of message.
 **
 ** RETURN VALUES:  
 **
 ** DESCRIPTION:	Sets up the header of the structure passed so that it may
 **				 be passed to other kernel functions as if it were a kernel
 **				 allocated message, and is never deleted by the kernel.
 **
 ****************************************************************************/
void TinyKernel_InitPersistentMsg(K_MSG *kMsg_p, K_LEN tK_LenLength)
{
	ASSERT(kMsg_p != PNULL);
	
	/* Mark message as being used by someone (1) and as a user allocated message */
	kMsg_p->header.bPersistentMsg = TRUE;

	kMsg_p->header.payloadSize = (uint16)tK_LenLength;
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_GetFastMessage
 **
 ** PARAMETERS:	 length of user message space in bytes
 **
 ** RETURN VALUES:  pointer to a K_MSG
 **
 ** DESCRIPTION:	Allocates a K_MSG from the fast message pool.
 **
 ****************************************************************************/
K_MSG *TinyKernel_GetFastMessage(K_LEN tK_LenLength)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *allocatedMsg_p;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	/* check that required length is not too long for a fast message */
	ASSERT(tK_LenLength <= K_FAST_MSG_SIZE);

	/* Critical Section */
	{
		allocatedMsg_p = TinyKernel_FastMsgFreeList;

		AAA_DEBUG_KCC_MSG_MACRO(TinyKernel_GetRunningTaskId(),(uint32)allocatedMsg_p);

		/* make sure fast message pool is not empty*/
		ASSERT(allocatedMsg_p != K_MSG_NULL);
		TinyKernel_FastMsgFreeList = allocatedMsg_p->header.psNext;

		/* This is a free message - the reference count must be zero */
		ASSERT( bLNARC_GET_REF_COUNT(allocatedMsg_p->header.bListNoAndRefCount) == 0 );

		/* Increase reference count to 1 */
		allocatedMsg_p->header.bListNoAndRefCount++;
	}

	allocatedMsg_p->header.payloadSize = (uint32)tK_LenLength;

	OSAL_ENABLE_INTERRUPTS(interrupt_save);

	return allocatedMsg_p;
}



/****************************************************************************
 **
 ** NAME:		   TinyKernel_SendMessage
 **
 ** PARAMETERS:	 Task to send message to
 **				 message to send
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	General send message function
 **
 ****************************************************************************/
void TinyKernel_SendMessage(K_MSG_TYPE msgType, K_TASKID bTo, K_MSG *kMsg_p, uint8 vapId)
{
	OSAL_SET_MESSAGE_ORIGINATOR(kMsg_p);
	kMsg_p->header.tKMsgType = msgType;
	kMsg_p->header.sTo.taskID = bTo;
	if (vapId != VAP_ID_DO_NOT_CARE)
		kMsg_p->header.vapId = vapId;
	
	TinyKernel_PutMessage(kMsg_p);
}



/****************************************************************************
 **
 ** NAME:		   TinyKernel_SetMessageOriginator
 **
 ** PARAMETERS:	 Message to set the originator of
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Sets the originator of the message to the current task
 **				 and ID.
 **
 ****************************************************************************/

void TinyKernel_SetMessageOriginator( K_MSG *psMsg )
{
	ASSERT( psMsg != NULL );

	psMsg->header.sFrom.taskID = TinyKernel_GetRunningTaskId();
}


/****************************************************************************
 **
 ** NAME:		   TinyKernel_GetMessageOriginator
 **
 ** PARAMETERS:	 Message containing the originators address
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Extracts the originating task ID and instance from the
 **				 message, and returns it to the caller.
 **
 ****************************************************************************/

void TinyKernel_GetMessageOriginator( K_MSG *psMsg, K_TASKID *pTask)
{
	ASSERT( psMsg != NULL );
	ASSERT( pTask != NULL );

	*pTask = psMsg->header.sFrom.taskID;
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_FreeMessage
 **
 ** PARAMETERS:	 pointer to K_MSG to free
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Frees up the kernel message
 **
 ****************************************************************************/

void TinyKernel_FreeMessage(K_MSG *psMsg)
{
	if ((psMsg == NULL) || (psMsg->header.bPersistentMsg == TRUE))
		return;

	/* Zero ref count here is a fatal error */
	ASSERT(bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) != 0);

	/* If reference count is exactly 1, then no other tasks have a claim
	 * on the message, so free it, otherwise don't, and don't change the
	 * ref count either. */
	if (bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) == 1)
	{
		byte bPoolId;

		/* Decrease the reference count to 0 */
		psMsg->header.bListNoAndRefCount--;

		bPoolId = bLNARC_GET_POOL_ID(psMsg->header.bListNoAndRefCount);

		ASSERT(bPoolId == bLNARC_FAST_POOL); // asserts if attempt to free a non freeable message

		TinyKernel_FreeFastMessage(psMsg);
	}
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_InitQueue
 **
 ** PARAMETERS:	 pointer to buffer
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Initialises a message queue
 **
 ****************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization_start" 
#endif
void TinyKernel_InitQueue( K_MSG_QUEUE *psQueue )
{
	psQueue->psHead = psQueue->psTail = K_MSG_NULL;
}

#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

/****************************************************************************
 **
 ** NAME:		   TinyKernel_QueueMsg
 **
 ** PARAMETERS:	 pointer to message queue and to the message to buffer
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Places a message on a queue. The function increments the
 **				 reference count to avoid the message being deleted.
 **
 ****************************************************************************/
void TinyKernel_QueueMsg( K_MSG_QUEUE *psQueue, K_MSG *psMsg )
{
	TX_INTERRUPT_SAVE_AREA;
	psMsg->header.psNext = K_MSG_NULL;
	TinyKernel_IncMsgRefCount( psMsg );

	/* Critical Section */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	{
		if (psQueue->psTail == K_MSG_NULL)
		{
			/* list is empty */
			psQueue->psTail = psQueue->psHead = psMsg;
		}
		else
		{
			/* list is not empty */
			psQueue->psTail->header.psNext = psMsg;

			psQueue->psTail = psMsg;
		}
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}



/****************************************************************************
 **
 ** NAME:		   TinyKernel_DequeueMsg
 **
 ** PARAMETERS:	 pointer to message queue
 **
 ** RETURN VALUES:  returns pointer to first-in message or NULL if queue is empty
 **
 ** DESCRIPTION:	removes first message from the queue, and decrements its
 **				 reference count, so that it'll get deleted if it doesn't get
 **				 queued again.
 **
 ****************************************************************************/

K_MSG *TinyKernel_DequeueMsg( K_MSG_QUEUE *psQueue)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG *psMsg;

	/* Critical Section */	
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	{
		psMsg = psQueue->psHead;
		if (psMsg != K_MSG_NULL)
		{
			TinyKernel_DecMsgRefCount( psMsg );
			psQueue->psHead = psMsg->header.psNext;
			if (psQueue->psHead == K_MSG_NULL)
			{
				psQueue->psTail = K_MSG_NULL;
			}
		}
	}
	OSAL_ENABLE_INTERRUPTS(interrupt_save);

	return psMsg;
}

/****************************************************************************
 **
 ** NAME:		   oboKNLbufferEmpty
 **
 ** PARAMETERS:	 pointer to message queue
 **
 ** RETURN VALUES:  returns TRUE if buffer empty
 **
 ** DESCRIPTION:	examines buffer for empty condition
 **
 ****************************************************************************/

bool TinyKernel_IsQueueEmpty( K_MSG_QUEUE *psQueue )
{
	if (psQueue->psHead != K_MSG_NULL)
	{
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_IncMsgRefCount
 **
 ** PARAMETERS:	 pointer to message
 **
 ** RETURN VALUES:  None
 **
 ** DESCRIPTION:	Increments the reference count of a message buffer.
 **				 Message buffers are given a reference count of 1 when
 **				 allocated and the count is decremented when freed.
 **
 **				 Incrementing the reference count prevents the message
 **				 block being released to the pool.
 **
 ****************************************************************************/

void TinyKernel_IncMsgRefCount(K_MSG * psMsg)
{
	/* If we have a persistent message count is irrelevant */
	if (psMsg->header.bPersistentMsg == TRUE)
	{
		return;
	}

	ASSERT(bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) < K_MAX_REF_COUNT); // asserts if Tried to increase a reference count past limit
	/* NOTE: Relies on ref count being in bottom bits */
	psMsg->header.bListNoAndRefCount++;
}


/****************************************************************************
 **
 ** NAME:		   TinyKernel_DecMsgRefCount
 **
 ** PARAMETERS:	 pointer to message
 **
 ** RETURN VALUES:  None
 **
 ** DESCRIPTION:	Decrements the reference count of a message buffer.
 **				 Message buffers are given a reference count of 1 when
 **				 allocated and the count is decremented when freed.
 **
 **				 Decrementing the reference count allows the message
 **				 block being released to the pool if the count reaches 0.
 **
 ****************************************************************************/

void TinyKernel_DecMsgRefCount(K_MSG * psMsg)
{
	/* If we have a persistent message count is irrelevant */
	if (psMsg->header.bPersistentMsg == TRUE)
	{
		return;
	}

	ASSERT(bLNARC_GET_REF_COUNT(psMsg->header.bListNoAndRefCount) > 0); // asserts if Tried to decrease a reference count past zero
	/* NOTE: Relies on ref count being in bottom bits */
	psMsg->header.bListNoAndRefCount--;
}


/*************************************************************************/
/***						Static Functions						   ***/
/*************************************************************************/

/****************************************************************************
 **
 ** NAME:		   TinyKernel_InitFastMessagePool
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Initialises the fast message pool
 **
 ****************************************************************************/

#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization_start" 
#endif
static void TinyKernel_InitFastMessagePool(void)
{
	byte bIndex;
	K_MSG *pMsg;


#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
	System_MainAllocInitializationMemory((uint8 **)&TinyKernel_FastMsgPool, (sizeof(K_FAST_MSG)*K_NO_OF_FAST_MSG));
#endif

	TinyKernel_FastMsgFreeList = (K_MSG *)TinyKernel_FastMsgPool;

	for (bIndex = 0; bIndex < K_NO_OF_FAST_MSG; bIndex++)
	{
		pMsg = (K_MSG *)&(TinyKernel_FastMsgPool[bIndex]);

		pMsg->header.bListNoAndRefCount = bLNARC_FAST_POOL;
		pMsg->header.bPersistentMsg = FALSE;

		if (bIndex == (K_NO_OF_FAST_MSG - 1))
		{
			pMsg->header.psNext = K_MSG_NULL;
		}
		else
		{
			pMsg->header.psNext = (K_MSG *)&TinyKernel_FastMsgPool[bIndex + 1];
		}
	}
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

/****************************************************************************
 **
 ** NAME:		   TinyKernel_FreeFastMessage
 **
 ** PARAMETERS:	 pointer to K_MSG to free
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Returns message to the free fast message pool
 **
 ****************************************************************************/
static void TinyKernel_FreeFastMessage(K_MSG *psMsg)
{
	TX_INTERRUPT_SAVE_AREA;

	/* Start Critical Section */
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	psMsg->header.psNext = TinyKernel_FastMsgFreeList;
	TinyKernel_FastMsgFreeList = psMsg;

	/* End Critical Section */
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

