/************************************************************************************
*	
*	File:		  TinyKernel_Scheduler.c
*	Class/Module: 
*	Description:  A detailed description of the Module, its  purpose attributes and
*		 	  whatever information the user & maintainer might find valuable.
*
*	COPYRIGHT: 
*		(C) Lantiq Israel Ltd.
*		All rights are strictly reserved. Reproduction or divulgence in any   
* 	  form whatsoever is not permitted without written authority from the 
*		copyright owner. Issued by Lantiq Israel Ltd
*
**************************************************************************************/
/*************************************************************************/
/***						Include Files							  ***/
/*************************************************************************/
#include "System_GlobalDefinitions.h"
#include "System_MainApi.h"
#include "OSAL_Api.h"
#include "TinyKernel_Api.h"
#include "TinyKernel_Scheduler.h"
#include "TinyKernel_Messages.h"
#include "stringLibApi.h"
#include "ErrorHandler_Api.h"
#include "lmtimerUtility.h"
#include "System_Timers.h"
#include "OSAL_Tasks.h"
#include "Debug_Trace.h"
#include "BeaconHandler_api.h"
#include "Utils_Api.h"
#include "CpuLoad_Api.h"
#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_TINY_KERNEL
#define LOG_LOCAL_FID 1

//-=-=-=-=-=-=-=-=-
// Definitions
//-=-=-=-=-=-=-=-=-
#define TINY_KERNEL_10mSec_IN_MIPS_TICKS	(10000 * SYSTEM_NUMBER_OF_MIPS_TICKS_IN_1uSec)
#define TINY_KERNEL_NO_TASKS_PENDING		(-1)

extern char __ghsbegin_initialization[], __ghsend_initialization[];
extern uint32 initstartAdress;
extern uint32 endinitAddress;
extern uint32 sizeOfInitSection;


typedef enum _TinyKernel_Priorites_e
{
	TINY_KERNEL_LOWEST_PRIORITY,
	TINY_KERNEL_MID_PRIORITY,
	TINY_KERNEL_HIGHEST_PRIORITY,
	TINY_KERNEL_NUM_OF_PRIORITIES,
} TinyKernel_Priorites_e;

typedef struct _TinyKernel_RunningTaskInfo_t
{
	K_MSG		*psMsg;
	K_TASKID	taskId;
} TinyKernel_RunningTaskInfo_t;

//-=-=-=-=-=-=-=-=-=-
// Static Variables
//-=-=-=-=-=-=-=-=-=-

/* This variable contains one bit per task priority indicating if there */
/* are any messages pending at that priority. Bit0=highest priority	 */
static uint32 TinyKernel_PendingTasksBitmap;

/* Information about the current task scheduled at each priority		*/
static TinyKernel_RunningTaskInfo_t TinyKernel_RunningTaskInfo;

/* Each task priority has a single message queue shared between all	 */
/* tasks at the same level. The queue structure has head and tail	   */
/* pointers only as each message in it has a next pointer			   */
static K_MSG_QUEUE TinyKernel_PendingTasks[TINY_KERNEL_NUM_OF_PRIORITIES];

//-=-=-=-=-=-=-=-=-=-
// Static Prototypes
//-=-=-=-=-=-=-=-=-=-
static void TinyKernel_Scheduler(void);
#if defined (CPU_MIPS)	
#pragma ghs nomips16e
void TinyKernel_MipsSleep(void);
#pragma ghs mips16e
#endif
const K_TASKID taskConverterTable[NUM_OF_TASKS_IN_SYSTEM]=
{
	TASK_INVALID,					//TASK_KNL_TIMER_TICK,
	TASK_INVALID,					//TASK_KNL_TIMER_HIGH,
	TASK_IPC_2,						//TASK_IPC_2,
	TASK_LINK_ADAPTATION,			//TASK_LINK_ADAPTATION,
	TASK_PAC_MANAGER,				//TASK_PAC_MANAGER,
	TASK_INVALID,					//TASK_KNL_TIMER_MID,
	TASK_INVALID,					//TASK_KNL_TIMER_LOW,
	TASK_IPC_2,						//TASK_UM_IF_TASK,
	TASK_IPC_2,						//TASK_BSS_MANAGE,
	TASK_IPC_2,						//TASK_TS_MANAGER,
	TASK_IPC_2,						//TASK_QOS,
	TASK_IPC_2,						//TASK_ENCRYPTION,
	TASK_IPC_2,						//TASK_LOGGER,
	TASK_INVALID,					//TASK_IPC,
#ifdef PTA_BUILD_IN_PLAT	
	TASK_IPC_2,						//TASK_PTA
#endif
	TASK_IPC_2,						//TASK_DUT,
	TASK_IPC_2,						//TASK_INTERFERER_DETECTION,
	TASK_IPC_2,						//TASK_COC,	
	TASK_IPC_2,						//TASK_HDK,
	TASK_IPC_2,						//TASK_HIM,
	TASK_IPC_2,						//TASK_RX_MANAGER,
	TASK_IPC_2,						//TASK_TX_MANAGER,	
	TASK_IPC_2,						//TASK_BSS_MANAGER,
	TASK_IPC_2, 					//TASK_CHANNEL_SWITCH_MANAGER,
#ifdef USE_AGER_EMULATOR
	TASK_IPC_2, 					//TASK_AGER_EMULATOR,
#endif
	TASK_IPC_2,						//TASK_STATISTICS_MANAGER,
	TASK_IPC_2,						//TASK_PROCESS_MANAGER,
	TASK_IPC_2,						//TASK_GROUP_MANAGER,
	TASK_IPC_2,						//TASK_ATF,
	TASK_HDK_CDB_MANAGER,			//TASK_HDK_CDB_MANAGER,
#ifdef HDK_CDB_SUPPORT
	TASK_IPC_2						//TASK_PROCESS_MANAGER_CDB	
#endif
};

/****************************************************************************
 **
 ** NAME:		   TinyKernel_Scheduler
 **
 ** PARAMETERS:	 priority of currently running task
 **
 ** RETURN VALUES:  bool - true if a task was scheduled
 **
 ** DESCRIPTION:	schedules task according to scheduling rules. If no tasks
 **				 are running, should be called with parameter
 **				 TINY_KERNEL_NUM_OF_PRIORITIES
 **
 ****************************************************************************/
static void TinyKernel_Scheduler(void)
{
	TX_INTERRUPT_SAVE_AREA;
	K_MSG			*pPendingMsg;
	int32			highestPriorityOfPendingTask;
	K_TASKID		taskID;

	for (;;)
	{
		/* Start Critical Section. To protect access to head pointer of task queue */
		/* that can be modified by zyx an ISR main body?? */
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		if (TinyKernel_PendingTasksBitmap != 0)
		{
            highestPriorityOfPendingTask = Utils_FindFirstSet(TinyKernel_PendingTasksBitmap);
			//pop message in head of queue for the highest priority pending task
			pPendingMsg = TinyKernel_DequeueMsg(&(TinyKernel_PendingTasks[highestPriorityOfPendingTask]));
			if (TinyKernel_IsQueueEmpty(&(TinyKernel_PendingTasks[highestPriorityOfPendingTask])) == TRUE)
			{
				/* no more messages pending at this priority - clear bit */
				TinyKernel_PendingTasksBitmap &= ~(1 << highestPriorityOfPendingTask);
			}
			taskID = taskConverterTable[pPendingMsg->header.sTo.taskID];
			/* Record task+instance and message in the RUNNING_TASK structure for the current priority */
			TinyKernel_RunningTaskInfo.psMsg = pPendingMsg;
			TinyKernel_RunningTaskInfo.taskId = taskID;
			/* End Critical Section */
			OSAL_ENABLE_INTERRUPTS(interrupt_save);		
#ifdef TLOG_FLAG
			TLOG0(pPendingMsg, FALSE);
#endif			
			/* Start profiler - temp use of single task */
			AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_kernelBefore_scheduledCall, taskID, pPendingMsg->header.tKMsgType);
			(*asAPP_TaskTable[taskID].rFunction)(pPendingMsg);
			AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_kernelAfter_scheduledCall,taskID, pPendingMsg->header.tKMsgType);
			TinyKernel_FreeMessage(pPendingMsg);
		}
		else
		{
#if defined (ENET_INC_LMAC)
			/* set timer so that CPU won't be in "wait" state for more than TINY_KERNEL_10mSec_IN_MIPS_TICKS */
			//MT_SetModeTimer(MT_TIMER_1, MT_TIMER_ENABLE, MT_TIMER_MULT_SHOT);
#endif
#if defined(CALCULATE_CPU_IDLE_TIME) 
			if (CalcCPUIdle_isInSleep == 0)
			{
				CalcCPUIdle_enterSleepTime = GET_TSF_TIMER_LOW();		
				CalcCPUIdle_isInSleep = 1;
			}
#endif
			//AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_KNL_MIPS_SLEEP,0,0);
			/* End Critical Section */
			OSAL_ENABLE_INTERRUPTS(interrupt_save);
			//put MIPS in sleep
			/*Remove mips sleep 
			  Timer patch does not helps here because it configure on 10 msec and we have higher interrupt frequency (less than 3 msec)
			*/
			//TinyKernel_MipsSleep(); //remove mips sleep so we can support high rate interrupt (less than 3 msec between interrupt)
			//AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_KNL_MIPS_SLEEP,1,1);
#if defined (ENET_INC_LMAC)
			//MT_SetModeTimer(MT_TIMER_1, MT_TIMER_DISABLE, MT_TIMER_MULT_SHOT);
#endif
			break;
		}
	}
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_PutMessage
 **
 ** PARAMETERS:	 Destination Task Id, pointer to K_MSG
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	
 **
 ****************************************************************************/
void TinyKernel_PutMessage(K_MSG *psSendMsg)
{
	TinyKernel_Priorites_e priority;
	K_TASKID	taskId;
	TX_INTERRUPT_SAVE_AREA;

	taskId = psSendMsg->header.sTo.taskID;
	AAA_DEBUG_KERNEL_SCHEDULER_MACRO(AAA_DebugKernelSchduler_vKNLputMessageHigh,(psSendMsg->header.sTo.taskID),(psSendMsg->header.tKMsgType));
	if (OSAL_IS_HIGH_PRIORITY_TASK(taskId))
	{
		priority = TINY_KERNEL_HIGHEST_PRIORITY;
	}
	else if (OSAL_IS_MID_PRIORITY_TASK(taskId))
	{
		priority = TINY_KERNEL_MID_PRIORITY;
	}
	else
	{
		priority = TINY_KERNEL_LOWEST_PRIORITY;
	}
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	TinyKernel_QueueMsg(&(TinyKernel_PendingTasks[priority]), psSendMsg);
	TinyKernel_PendingTasksBitmap |= (1 << priority);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_GetRunningTaskId
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  task identity
 **
 ** DESCRIPTION:	gets the current task's identity
 **
 ****************************************************************************/

K_TASKID TinyKernel_GetRunningTaskId(void)
{
	if (InterruptManager_IsInInterruptContext() == TRUE)
	{
		return TASK_KNL_INTERRUPT_CONTEXT;
	}
	return TinyKernel_RunningTaskInfo.taskId;
}

/****************************************************************************
 **
 ** NAME:		   TinyKernel_MipsSleep
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	Put MIPS in sleep mode
 **
 ****************************************************************************/
#if defined (CPU_MIPS)	
#pragma ghs nomips16e
void TinyKernel_MipsSleep(void)
{
	//// - MT_LOG_EVENT32_1(LOGGER_EVENT_GROUP_BUGS,19,"TinyKernel_MipsSleep", 0);//Color_Slate_Gray
	asm("	wait");
}
#pragma ghs mips16e
#endif

/****************************************************************************
 **
 ** NAME:		   TinyKernel_StartScheduler
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	main loop of firmware
 **
 ****************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif
void TinyKernel_StartScheduler(void)
{
	// The following inits are done here since they do overlays with iniy code. This is equivalent to "post init" of the threadX.
	System_MainPostInit();
	
	while(TRUE)
	{	
		ErrorHadler_CheckAssertReqFromHost();
		TinyKernel_Scheduler();
	}
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif



//-=-=-=-=-=-=-=-=-=-
// Public Functions
//-=-=-=-=-=-=-=-=-=-
/****************************************************************************
 **
 ** NAME:		   TinyKernel_InitScheduler
 **
 ** PARAMETERS:	 none
 **
 ** RETURN VALUES:  none
 **
 ** DESCRIPTION:	initialise the kernel Scheduler
 **
 ****************************************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif
void TinyKernel_InitScheduler(void)
{
	uint32 i;

	TinyKernel_PendingTasksBitmap = 0x0;//no tasks pending
	for (i = 0; i < TINY_KERNEL_NUM_OF_PRIORITIES; i++)
	{
		TinyKernel_InitQueue(&(TinyKernel_PendingTasks[i]));
	}
	memset(&TinyKernel_RunningTaskInfo, 0, sizeof(TinyKernel_RunningTaskInfo_t));
#if defined (ENET_INC_LMAC)
	/* Configure the timer period */
	MT_SetTimeTimer(MT_TIMER_1, TINY_KERNEL_10mSec_IN_MIPS_TICKS);
#endif
}
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif


