/***********************************************************************************
 File:			TrainingManager.h
 Module:		TrainingManager
 Purpose: 		To handle MU training sequence
 Description:	
************************************************************************************/
#ifndef TRAINING_MANAGER_H_
#define TRAINING_MANAGER_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "TrainingManager_Api.h"
#include "GroupManager_API.h"
#include "MuEffectiveRateDbRam.h"
#ifdef TRAINING_FIRST_PHASE_POOL
#include "CalibrationsDefines.h"
#endif //TRAINING_FIRST_PHASE_POOL

#if defined (DEBUG_TRAINING_MANAGER)
#include "Pac_Api.h"
#endif //DEBUG_TRAINING_MANAGER


//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------

// Training Manager events
typedef enum
{
	TRAINING_MANAGER_EVENT_START_TRAINING_REQ,		// 0x0
	TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ,		// 0x1
	TRAINING_MANAGER_EVENT_ADD_VAP_REQ,				// 0x2
	TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ,			// 0x3
	TRAINING_MANAGER_EVENT_ADD_STA_REQ,				// 0x4
	TRAINING_MANAGER_EVENT_REMOVE_STA_REQ,			// 0x5
	TRAINING_MANAGER_EVENT_HALT_STA_REQ,			// 0x6
	TRAINING_MANAGER_EVENT_STOP_STA_REQ,			// 0x7
	TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED,	// 0x8
	TRAINING_MANAGER_EVENT_NDPA_CFM,				// 0x9	
	TRAINING_MANAGER_EVENT_NUM_EVENTS				// 0xA
} trainingManagerEvents_e;
	
//Training Manager states
typedef enum
{
	TRAINING_MANAGER_STATE_IDLE,					// 0x0
	TRAINING_MANAGER_STATE_WAIT_FOR_LA_INFO,		// 0x1
	TRAINING_MANAGER_STATE_WAIT_FOR_RESULTS,		// 0x2
	TRAINING_MANAGER_STATE_STOP_WAIT_FOR_LA_INFO,	// 0x3
	TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA,		// 0x4
	TRAINING_MANAGER_STATE_NUM_STATES,				// 0x5
	TRAINING_MANAGER_MAX_STATE = 0xFF				
} trainingManagerStates_e;

//test plan policies
typedef enum
{
	SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY,			// 0x0
	SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY,			// 0x1
	DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY,			// 0x2
	DOUBLE_STA_COMBINATION_SEGMENT_TWO_NSTS_ONLY_POLICY,			// 0x3
	INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY,	// 0x4
	INCLUDE_THREE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY,	// 0x5
	FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY,			// 0x6
	NUM_OF_POLICIES,												// 0x7
	INVALID_POLICY = 0xFF
} testPlanPolicies_e;


//pool Ids
typedef enum
{
	FIRST_POOL,
	FIRST_PHASE_POOL_ID = FIRST_POOL,	// 0x0
	NUM_OF_POOLS,						// 0x1
	MAX_NUM_OF_POOLS = 8
} trainingManagerPools_e;


//pool Ids
typedef enum
{
	VAP_NOT_SELECTED,		// 0x0
	VAP_SELECTED			// 0x1
} vapSelectionStatus_e;


#define MAX_NUM_OF_TEST_PLAN_LINES				(MU_EFFECTIVE_RATE_DB_RAM_NUM_ROWS) //1024

#if (MAX_NUM_OF_TEST_PLAN_LINES > MU_EFFECTIVE_RATE_DB_RAM_NUM_ROWS)
#error Max Num of Test Plan lines is not supported by HW
#endif


#define MAX_NUM_OF_STA_PRIORITY_HIGH			(16) //unlimited
#define MAX_NUM_OF_STA_PRIORITY_MID				(16) //unlimited
#define MAX_NUM_OF_STA_PRIORITY_LOW				(16) //unlimited

#define POOL_MU_STAS_LOWER_LIMIT				(2)
#define POOL_MU_STAS_UPPER_LIMIT				(16) //unlimitted


#define TRAINING_MANAGER_INVALID_VAP			(0xFF)

#ifdef ENET_INC_ARCH_WAVE600
#define VAP_BITMAP_MASK							(0xFFFFFFFF) //((0x1 << HW_NUM_OF_VAPS) - 1)
#else
#define VAP_BITMAP_MASK							((0x1 << HW_NUM_OF_VAPS) - 1)
#endif //ENET_INC_ARCH_WAVE600


//group calculation weights (%) - sum must be 100%
#define GROUP_CALCULATION_PHY_THROUGHPUT_WEIGHT	(60)
#define GROUP_CALCULATION_NUM_OF_STAS_WEIGHT	(20)
#define GROUP_CALCULATION_NUM_OF_NSTS_WEIGHT	(20)

//minimal phy validation group mark
#define MINIMNAL_PHY_VALIDATION_GROUP_MARK		(0) //TBD get the value from system



//num of station per nsts segment
#define NUM_OF_STATIONS_SEGMENT_1XXX			(1)
#define NUM_OF_STATIONS_SEGMENT_2XXX			(1)
#define NUM_OF_STATIONS_SEGMENT_22XX			(2)
#define NUM_OF_STATIONS_SEGMENT_21XX			(2)
#define NUM_OF_STATIONS_SEGMENT_12XX			(2)
#define NUM_OF_STATIONS_SEGMENT_11XX			(2)
#define NUM_OF_STATIONS_SEGMENT_111X			(3)
#define NUM_OF_STATIONS_SEGMENT_211X			(3)
#define NUM_OF_STATIONS_SEGMENT_121X			(3)
#define NUM_OF_STATIONS_SEGMENT_112X			(3)
#define NUM_OF_STATIONS_SEGMENT_1111			(4)


//num of nsts' per nsts segment
#define TOTAL_NUM_OF_NSTS_SEGMENT_1XXX			(1)
#define TOTAL_NUM_OF_NSTS_SEGMENT_2XXX			(2)
#define TOTAL_NUM_OF_NSTS_SEGMENT_22XX			(4)
#define TOTAL_NUM_OF_NSTS_SEGMENT_21XX			(3)
#define TOTAL_NUM_OF_NSTS_SEGMENT_12XX			(3)
#define TOTAL_NUM_OF_NSTS_SEGMENT_11XX			(2)
#define TOTAL_NUM_OF_NSTS_SEGMENT_111X			(3)
#define TOTAL_NUM_OF_NSTS_SEGMENT_211X			(4)
#define TOTAL_NUM_OF_NSTS_SEGMENT_121X			(4)
#define TOTAL_NUM_OF_NSTS_SEGMENT_112X			(4)
#define TOTAL_NUM_OF_NSTS_SEGMENT_1111			(4)



#if ((GROUP_CALCULATION_PHY_THROUGHPUT_WEIGHT + GROUP_CALCULATION_NUM_OF_STAS_WEIGHT + GROUP_CALCULATION_NUM_OF_NSTS_WEIGHT) != 100)
#error Group calculation weights sum must be 100%.
#endif


//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------

/*Bitmap manipulations for up to 32 bits bitmaps */
#define CLR_BIT_IN_SINGLE_BITMAP(bitmap, offset) (bitmap &= ~(1 << (offset)))
#define SET_BIT_IN_SINGLE_BITMAP(bitmap, offset) (bitmap |=  (1 << (offset)))
#define GET_BIT_IN_SINGLE_BITMAP(bitmap, offset) (bitmap & (1 << (offset)))

#if defined (DEBUG_TRAINING_MANAGER)

#define FILL_TRAINING_DEBUG_STRUCT(currentState, event) 													\
	AAA_TRAINING_MANAGER[AAA_TRAINING_MANAGER_INDEX].tsf = GET_TSF_TIMER_LOW();								\
	AAA_TRAINING_MANAGER[AAA_TRAINING_MANAGER_INDEX].state = currentState;									\
	AAA_TRAINING_MANAGER[AAA_TRAINING_MANAGER_INDEX].event = event;											\
	AAA_TRAINING_MANAGER_INDEX = (AAA_TRAINING_MANAGER_INDEX + 1) % AAA_TRAINING_MANAGER_DEBUG_ARRAY_SIZE;	\
	
#else

#define FILL_TRAINING_DEBUG_STRUCT(currentState, event)	\

#endif /* DEBUG_TRAINING_MANAGER */



//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------



typedef struct _GroupMarkCalculationWeights_t
{
	uint8 phyThroughputWeight;
	uint8 numOfStationsWeight;
	uint8 numOfNstsWeight;							
} GroupMarkCalculationWeights_t;

#if defined (TRAINING_FIRST_PHASE_POOL) || defined (TRAINING_DEBUG_CLI)

typedef struct _TrainingChangableCliParams_t
{
#ifdef TRAINING_DEBUG_CLI
	uint16 minimalPhyValidationGroupMark;					
	GroupMarkCalculationWeights_t groupMarkCalculationWeights;
	bool isNewGroupMarkConfiguration;	//only if TRUE - there is new Group Mark configurations to update (i.e. the parameter above should be used)
	bool isNewWeightsConfiguration;		//only if TRUE - there is new weights configurations to update (i.e. the parameter above should be used)
#endif //TRAINING_DEBUG_CLI
#ifdef TRAINING_FIRST_PHASE_POOL
	mcs_e minimalMcsForFirstPhasePool; 
	SpatialStreamNum_e maximalNssForFirstPhasePool;
	bool isNewMinimalMcsConfiguration;	//only if TRUE - there is new minimal mcs configuration to update (i.e. the parameter above should be used)
	bool isNewMaximalNssConfiguration;	//only if TRUE - there is new minimal nss configuration to update (i.e. the parameter above should be used)
#endif //TRAINING_FIRST_PHASE_POOL
} TrainingChangableCliParams_t;

#endif //(TRAINING_FIRST_PHASE_POOL) || (TRAINING_DEBUG_CLI)


typedef struct _TrainingManagerStaticParams_t
{
	uint16 muActiveVapsBitmap; 																 			//bitmap of current active vaps (which supports VHT)
	uint16 activeVapsBitmap;																 			//bitmap of all active VAPs - used as a flag for handling SET_BSS message only once per VAP
	uint8 selectedVap; 																		 			//last selected VAP
	TrainingManagerStaBitmap_t connectedStaBitmap[HW_NUM_OF_VAPS]; 						 			//connected stations per vap
	TrainingManagerStaBitmap_t waitingForRemoveStaBitmap; 									 			//bitmap of stations waiting to be removed and can't be removed immediately
	uint16 minimalPhyValidationGroupMark;													 			//can be configured through CLI
	GroupMarkCalculationWeights_t groupMarkCalculationWeights;								 			//can be configured through CLI
	GroupManagerTrainingBitmap_t *groupManagerTrainingBitmaps; 								 			//pointers to bitmaps in Group Manager's DB
#ifdef TRAINING_FIRST_PHASE_POOL
	testPlanPolicies_e policyPriorityPerNumOfAntennas[MAX_NUM_OF_DIGITAL_ANTENNAS][NUM_OF_POLICIES]; 	//diffenet priorities per num of antennas
#endif //TRAINING_FIRST_PHASE_POOL
#if defined (TRAINING_FIRST_PHASE_POOL) || defined (TRAINING_DEBUG_CLI)
	TrainingChangableCliParams_t debugWaitingForUpdateParams;											//params that can be configured through the CLI and are waiting to be updated in the DB
#endif //(TRAINING_FIRST_PHASE_POOL) || (TRAINING_DEBUG_CLI)
	trainingManagerStates_e state; 																		//state machine's state
} TrainingManagerStaticParams_t;





typedef struct _TrainingDynamicParams_t
{
	uint8 currentTrainingVectorIdx;																			//current training vector index
	uint8 activePoolsInTrainingBitmap;																		//active pools in current training
	TrainingManagerStaBitmap_t phyTrainingStaBitmap; 														//Bitmap of stations participating in current training
	StaId trainingVector[GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING]; 									//Final training vector
	uint16 locationBitmap[NUM_OF_POOLS]; 																	//Per pool bitmap which represents the locations of STA from the same pool in the training vector
	uint8 staLocationTable[HW_NUM_OF_STATIONS]; 														//location of the station in the phy training vector
	uint8 numberOfStasInTrainingFromPool[NUM_OF_POOLS]; 													//Number of stations in current training per pool
	uint16 currentTestPlanIdx; 																				//next free entry in test plan
	uint16 currentRestoredTestPlanIdx; 																		//next entry to read from phyDB output (results are in the same order as the original test plan)
	GroupCombinationTableEntry_t groupCombinationTable[GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING]; 		//Group Combination table for the Group Manager	
	Bandwidth_e currentBw;																					//BW of stations participating in the training
	uint8 numOfAntennas;																					//num of Tx supported antennas (1 to 4).
} TrainingDynamicParams_t;




typedef struct _TrainingManagerDb_t
{
	TrainingManagerStaticParams_t 	staticParams; 				//valid forever
	TrainingDynamicParams_t 		dynamicParams; 				//valid only during the training	
} TrainingManagerDb_t;



typedef union
{
uint32 val;
	struct
	{
		uint32 nss:4;	// bit0 - NSS1, bit3 - NSS4 (nss_e)
		uint32 mcs:10; 	//bit0 - MCS0, bit9 - MCS9  (mcs_e)
		uint32 bw:3;	//bit0 - 20MHZ, bit2 - 80MHZ
		uint32 reserved0:15;
	} bitFields;	
} PoolFeatureBitmap_u;



typedef struct _StaticPoolInformation_t
{
	uint8 selectablePoolsBitmap; 						                        //bitmap of selectable pools for current pool (each bit represents pool from trainingManagerPools_e)
	uint8 testPlanPoliciesBitmap; 						                        //bitmap of supported policies for current pool (each bit represents policy from testPlanPolicies_e)
	uint8 poolMuSTAsLowerLimit; 						                       	//minimum stations from pool in order to choose it for training
	uint8 poolMuSTAsUpperLimit; 						                        //maximum stations can be chosen for training from the pool
	uint8 maxNumOfStaPerPriority[GROUP_MANAGER_NUMBER_OF_STATION_PRIORITIES]; 	//maximum allowed number of stations per priority
	PoolFeatureBitmap_u poolFeatureBitmap; 				                        //bitmap of supported features for pool
} StaticPoolInformation_t;                   



typedef struct _PoolInformation_t
{
	TrainingManagerStaBitmap_t stationsInPoolBitmap; 	//bitmap of stations in current pool
	uint8 numOfStationsInPool; 							//num of stations in current pool
} PoolInformation_t;


typedef struct _DynamicPoolInformation_t
{
	uint8 activePoolsBitmap; 							//bitmap of active pools (includes more than minimum stations)
	PoolInformation_t poolInformation[NUM_OF_POOLS];	//contains detailes regarding stations in pool
} DynamicPoolInformation_t;


typedef struct _PoolDb_t
{
StaticPoolInformation_t staticParams[NUM_OF_POOLS]; 	//valid forever - pools properties
DynamicPoolInformation_t dynamicParams; 				//valid only during the training
	
} PoolDb_t;



// Function pointer for SM
typedef void (*TrainingManager_Func)(void *parameter);

// Function pointer for building Test Plan per policy
typedef void (*TrainingManagerTestPlanPolicy_Func)(trainingManagerPools_e poolId);


// Training Manager State Machine
typedef struct _TrainingManagerSm_t
{
	TrainingManager_Func handler[TRAINING_MANAGER_EVENT_NUM_EVENTS];
} TrainingManagerSm_t;


// Training Manager Test Plan Policy Handler
typedef struct _TrainingManagerTestPlanPolicyHandler_t
{
	TrainingManagerTestPlanPolicy_Func handler;
} TrainingManagerTestPlanPolicyHandler_t;



// Training Manager Test Plan Policy Handler
typedef struct _TrainingVectorDebugStruct_t
{
	uint8 trainingVector[GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING];
} TrainingVectorDebugStruct_t;





//---------------------------------------------------------------------------------
//						Static Function Declaration				
//---------------------------------------------------------------------------------

static void TrainingManager_RunStateMachine(trainingManagerEvents_e event, void *parameter);
static void TrainingManager_ChangeState(trainingManagerStates_e state);
static void TrainingManager_IllegalEvent(void *parameter);
static void TrainingManager_SendLaInfoReq(void *parameter);
static void TrainingManager_StartTraining(void *parameter);

static void TrainingManager_CheckNdpaConfirmationStatus(void *parameter);

#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_StopTrainingDuringTraining(void *parameter);
static void TrainingManager_RemoveStaDuringTraining(void *parameter);
static void TrainingManager_HaltStaDuringTraining(void *parameter);
static void TrainingManager_StopStaDuringTraining(void *parameter);
#endif

static void TrainingManager_StopTrainingDuringWaitingForLaInfo(void *parameter);
static void TrainingManager_IgnoreStopTraining(void *parameter);
static void TrainingManager_AddVap(void *parameter);
static void TrainingManager_RemoveVap(void *parameter);
static void TrainingManager_AddSta(void *parameter);
static void TrainingManager_RemoveStaNotDuringTraining(void *parameter);
static void TrainingManager_RemoveStaDuringStopping(void *parameter);
static void TrainingManager_HaltStaNotDuringTraining(void *parameter);
static void TrainingManager_HaltStaDuringStopping(void *parameter);
static void TrainingManager_SendTrainingStoppedConfirmation(void *parameter);
static void TrainingManager_StartResultsProcessing(void);
static vapSelectionStatus_e TrainingManager_BuildPools(TrainingManagerLinkAdaptiveInfoCfm_t *pLinkAdaptiveTrainingInfo);
static void TrainingManager_SelectStations(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlan(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTable(trainingManagerPools_e poolId);
static void TrainingManager_InitializeGroupCombinationTable(void);
static uint32 TrainingManager_CalculateGroupMark(MuEffectiveRateDbRamEntry_u testPlanOutputEntry, uint8 totalNsts, uint8 numOfStations);
static void TrainingManager_CheckAndUpdateGroupCombinationTable(uint8 tableIndex, MuEffectiveRateDbRamEntry_u testPlanOutputEntry, uint32 currentGroupMark, StaId groupCombinationSids[SPATIAL_STREAM_NUM]);


 //test plan per policy handlers
static void TrainingManager_BuildTestPlanForPolicy1_SingleStaCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy2_SingleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy4_DoubleStaCombinationSegmentTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForPolicy7_FourStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);

 //test plan per segment handlers
static void TrainingManager_BuildTestPlanForSegment_1xxx(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_2xxx(trainingManagerPools_e poolId);
#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_BuildTestPlanForSegment_22xx(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_21xx(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_12xx(trainingManagerPools_e poolId);
#endif
static void TrainingManager_BuildTestPlanForSegment_11xx(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_111x(trainingManagerPools_e poolId);
#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_BuildTestPlanForSegment_211x(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_121x(trainingManagerPools_e poolId);
static void TrainingManager_BuildTestPlanForSegment_112x(trainingManagerPools_e poolId);
#endif
static void TrainingManager_BuildTestPlanForSegment_1111(trainingManagerPools_e poolId);


 //restore test plan and update group combination table per policy handlers
static void TrainingManager_UpdateGroupCombinationTableForPolicy1_SingleStaCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy2_SingleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy4_DoubleStaCombinationSegmentTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForPolicy7_FourStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId);
 

 //restore test plan and update group combination table per segment handlers
static void TrainingManager_UpdateGroupCombinationTableForSegment_1xxx(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_2xxx(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_22xx(trainingManagerPools_e poolId);
#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_UpdateGroupCombinationTableForSegment_21xx(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_211x(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_121x(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_112x(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_12xx(trainingManagerPools_e poolId);
#endif
static void TrainingManager_UpdateGroupCombinationTableForSegment_11xx(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_111x(trainingManagerPools_e poolId);
static void TrainingManager_UpdateGroupCombinationTableForSegment_1111(trainingManagerPools_e poolId);
static inline void TrainingManager_StaBitmapBitwiseAND(TrainingManagerStaBitmap_t* resultBitmap, TrainingManagerStaBitmap_t* bitmap1, TrainingManagerStaBitmap_t* bitmap2);
static inline void TrainingManager_StaBitmapBitwiseNOT(TrainingManagerStaBitmap_t* resultBitmap, TrainingManagerStaBitmap_t* originalBitmap);

#ifdef TRAINING_FIRST_PHASE_POOL
static void TrainingManagerUpdateMinimalMcsCriteria(trainingManagerPools_e poolId, mcs_e mcs);
static void TrainingManagerUpdateMaximalNssCriteria(trainingManagerPools_e poolId, SpatialStreamNum_e nss);
#endif //TRAINING_FIRST_PHASE_POOL



#endif /* TRAINING_MANAGER_H_ */
