/***********************************************************************************
 File:			TrainingManager.c
 Module:		Training Manager
 Purpose: 		build and trigger MU training sequence
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "stringLibApi.h"
#include "HwMemoryMap.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "GroupManager_API.h"
#include "TrainingManager_Api.h"
#include "TrainingManager.h"
#include "loggerAPI.h"
#include "Utils_Api.h"
#include "StaDatabase_Api.h"
#include "SharedDbTypes.h"
#include "StaDatabase_Api.h"
#include "BSSmanager_API.h"
#include "MuEffectiveRateDbRam.h"
#include "ShramStationDatabase.h"
#ifndef TRAINING_WAVE600_Z0
#include "NdpaManager_Api.h"
#endif
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "CoC_Api.h"
#include "Hdk_Api.h"

#ifdef TRAINING_WAVE600_Z0
#include "Pac_Api.h"
#endif


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GID_TRAINING_MANAGER 
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Debug Code									
/----------------------------------------------------------------------------------*/

// Turn on/off training debug by comment/uncomment:

//#define TRAINING_MANAGER_DEBUG 

#if defined (DEBUG_TRAINING_MANAGER)

#define AAA_TRAINING_MANAGER_DEBUG_ARRAY_SIZE (128)	// Debug array size

typedef struct _aaa_trainingManager_
{
	uint32 					tsf;
	trainingManagerStates_e state;
	trainingManagerEvents_e event;	
}	aaa_trainingManager_t;

aaa_trainingManager_t 		AAA_TRAINING_MANAGER[AAA_TRAINING_MANAGER_DEBUG_ARRAY_SIZE] = {0}; 	// Debug array
uint32 						AAA_TRAINING_MANAGER_INDEX = 0;										// Debug array index

#endif //DEBUG_TRAINING_MANAGER

TrainingManagerStatistics_t TrainingStatistics;


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static TrainingManagerSm_t TrainingManagerSm[TRAINING_MANAGER_STATE_NUM_STATES] = 
{
	// TRAINING_MANAGER_STATE_IDLE
	{
		TrainingManager_SendLaInfoReq, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ		
		TrainingManager_SendTrainingStoppedConfirmation,	// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_AddVap,								// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_RemoveVap,							// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ			
		TrainingManager_AddSta,								// TRAINING_MANAGER_EVENT_ADD_STA_REQ				
		TrainingManager_RemoveStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ			
		TrainingManager_HaltStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},
	// TRAINING_MANAGER_STATE_WAIT_FOR_LA_INFO
	{
		TrainingManager_IllegalEvent, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ	
		TrainingManager_StopTrainingDuringWaitingForLaInfo,	// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_AddVap,								// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_RemoveVap,							// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ		
		TrainingManager_AddSta,								// TRAINING_MANAGER_EVENT_ADD_STA_REQ			
		TrainingManager_RemoveStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ
		TrainingManager_HaltStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_StartTraining,						// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},
#ifdef TRAINING_WAVE600_Z0
	// TRAINING_MANAGER_STATE_WAIT_FOR_RESULTS
	{
		TrainingManager_IllegalEvent, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ	
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ		
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_ADD_STA_REQ			
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},	
#else
	// TRAINING_MANAGER_STATE_WAIT_FOR_RESULTS
	{
		TrainingManager_IllegalEvent, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ	
		TrainingManager_StopTrainingDuringTraining,			// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_AddVap,								// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_RemoveVap,							// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ		
		TrainingManager_AddSta,								// TRAINING_MANAGER_EVENT_ADD_STA_REQ			
		TrainingManager_RemoveStaDuringTraining,			// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ
		TrainingManager_HaltStaDuringTraining,				// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_StopStaDuringTraining,				// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_CheckNdpaConfirmationStatus,		// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},	
#endif
	// TRAINING_MANAGER_STATE_STOP_WAIT_FOR_LA_INFO
	{
		TrainingManager_IllegalEvent, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ	
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_AddVap,								// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_RemoveVap,							// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ		
		TrainingManager_AddSta,								// TRAINING_MANAGER_EVENT_ADD_STA_REQ			
		TrainingManager_RemoveStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ
		TrainingManager_HaltStaNotDuringTraining,			// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_SendTrainingStoppedConfirmation,	// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},	
	// TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA
	{
		TrainingManager_IllegalEvent, 						// TRAINING_MANAGER_EVENT_START_TRAINIG_REQ	
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ
		TrainingManager_AddVap,								// TRAINING_MANAGER_EVENT_ADD_VAP_REQ			
		TrainingManager_RemoveVap,							// TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ		
		TrainingManager_AddSta,								// TRAINING_MANAGER_EVENT_ADD_STA_REQ			
		TrainingManager_RemoveStaDuringStopping,			// TRAINING_MANAGER_EVENT_REMOVE_STA_REQ
		TrainingManager_HaltStaDuringStopping,				// TRAINING_MANAGER_EVENT_HALT_STA_REQ
		TrainingManager_IgnoreStopTraining,					// TRAINING_MANAGER_EVENT_STOP_STA_REQ
		TrainingManager_IllegalEvent,						// TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED
		TrainingManager_SendTrainingStoppedConfirmation,	// TRAINING_MANAGER_EVENT_NDPA_CFM		
	},	
};


static TrainingManagerTestPlanPolicyHandler_t TestPlanPolicyHandler[NUM_OF_POLICIES] = 
{
	TrainingManager_BuildTestPlanForPolicy1_SingleStaCombinationSegmentSingleNstsOnly, 		     //SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
	TrainingManager_BuildTestPlanForPolicy2_SingleStaCombinationSegmentUpToTwoNsts,  			 //SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY	
	TrainingManager_BuildTestPlanForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts,				 //DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY
	TrainingManager_BuildTestPlanForPolicy4_DoubleStaCombinationSegmentTwoNsts,					 //DOUBLE_STA_COMBINATION_SEGMENT_TWO_NSTS_ONLY_POLICY
	TrainingManager_BuildTestPlanForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly, //INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
	TrainingManager_BuildTestPlanForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts,	 //INCLUDE_THREE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY
	TrainingManager_BuildTestPlanForPolicy7_FourStationCombinationSegmentSingleNstsOnly,		 //FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
};


static TrainingManagerTestPlanPolicyHandler_t UpdateGroupCombinationTablePolicyHandler[NUM_OF_POLICIES] = 
{
	TrainingManager_UpdateGroupCombinationTableForPolicy1_SingleStaCombinationSegmentSingleNstsOnly, 		   //SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
	TrainingManager_UpdateGroupCombinationTableForPolicy2_SingleStaCombinationSegmentUpToTwoNsts,  		   	   //SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY	
	TrainingManager_UpdateGroupCombinationTableForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts,			   //DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY
	TrainingManager_UpdateGroupCombinationTableForPolicy4_DoubleStaCombinationSegmentTwoNsts,				   //DOUBLE_STA_COMBINATION_SEGMENT_TWO_NSTS_ONLY_POLICY
	TrainingManager_UpdateGroupCombinationTableForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly, //INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
	TrainingManager_UpdateGroupCombinationTableForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts,	   //INCLUDE_THREE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY
	TrainingManager_UpdateGroupCombinationTableForPolicy7_FourStationCombinationSegmentSingleNstsOnly,		   //FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY
};

//conversion from MCS with 1 NSS to Phy rate in 80MHz Short CP
static const uint16 phyRate80MhzScp1Nss[NUM_OF_MCS] =

/*	VHT mcs,	80MHzSCP phyRate   */	
{			
	 /*0*/      32  ,
	 /*1*/      65  ,
	 /*2*/      97  ,
	 /*3*/     130  ,
	 /*4*/     195  ,
	 /*5*/     260  ,
	 /*6*/     292  ,
	 /*7*/     325  ,
	 /*8*/     390  ,
	 /*9*/     433  ,

};



// Training Manager Database
static TrainingManagerDb_t	TrainingManagerDb;
static PoolDb_t TrainingManagerPoolDb;

//pointer to the phy BF test plan Input/Output Ram 0xA6C3B800
static MuEffectiveRateDbRam_t *TrainingManagerTestPlanRam;

#ifdef TRAINING_WAVE600_Z0
	static MuEffectiveRateDbRam_t TrainingManagerTestPlanLocalDb;
#endif



/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

#ifdef TRAINING_WAVE600_Z0
static uint8 TrainingManager_get_rand_mcs()
{
	uint32 tsfLow = GET_TSF_TIMER_LOW();
	uint8 seed = (uint8)(tsfLow & 0xFF); // get lower 8 bits which change more frequently to be a seed

	uint8 random_mcs = (seed % MCS_5 ) + MCS_5; // this will provide MCS in range between MCS 5 and MCS 9

	return random_mcs; 
}

#endif


/**********************************************************************************
TrainingManager_RunStateMachine


Description:
------------

	Run state machine for all events recieved

	
Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_RunStateMachine(trainingManagerEvents_e event, void *parameter)
{
	trainingManagerStates_e currentState;

	DEBUG_ASSERT(event < TRAINING_MANAGER_EVENT_NUM_EVENTS);

	// Get current transmission state
	currentState = TrainingManagerDb.staticParams.state;

	FILL_TRAINING_DEBUG_STRUCT(currentState, event);
#ifdef TRAINING_MANAGER_DEBUG
	ILOG0_DD("TrainingManager_RunStateMachine: CurrentState = %d, event = %d", currentState, event);
#endif

	// Send the event to the state machine		
	TrainingManagerSm[currentState].handler[event](parameter);	
}

static void TrainingManager_ChangeState(trainingManagerStates_e state)
{
#ifdef TRAINING_MANAGER_DEBUG
	ILOG0_D("TrainingManager_ChangeState: newState = %d", state);
#endif

	//when going to idle, change the CLI requested parameters for debug
	if (state == TRAINING_MANAGER_STATE_IDLE)
	{
#ifdef TRAINING_DEBUG_CLI
		if (TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewGroupMarkConfiguration == TRUE)
		{
			TrainingManagerDb.staticParams.minimalPhyValidationGroupMark = TrainingManagerDb.staticParams.debugWaitingForUpdateParams.minimalPhyValidationGroupMark;
			TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewGroupMarkConfiguration = FALSE;
		}

		if (TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewWeightsConfiguration == TRUE)
		{
			TrainingManagerDb.staticParams.groupMarkCalculationWeights = TrainingManagerDb.staticParams.debugWaitingForUpdateParams.groupMarkCalculationWeights;
			TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewWeightsConfiguration = FALSE;
		}
#endif //TRAINING_DEBUG_CLI
#ifdef TRAINING_FIRST_PHASE_POOL
		if (TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMinimalMcsConfiguration == TRUE)
		{
			TrainingManagerUpdateMinimalMcsCriteria(FIRST_PHASE_POOL_ID, TrainingManagerDb.staticParams.debugWaitingForUpdateParams.minimalMcsForFirstPhasePool);
			TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMinimalMcsConfiguration = FALSE;
		}		
		if (TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMaximalNssConfiguration == TRUE)
		{
			TrainingManagerUpdateMaximalNssCriteria(FIRST_PHASE_POOL_ID, TrainingManagerDb.staticParams.debugWaitingForUpdateParams.maximalNssForFirstPhasePool);
			TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMaximalNssConfiguration = FALSE;
		}		
#endif //TRAINING_FIRST_PHASE_POOL		
	}


	TrainingManagerDb.staticParams.state = state;
}



/**********************************************************************************
TrainingManager_IllegalEvent



Description:
------------

	Illegal event in state machine - the recieved event is not axpected to be recieved in current state

	
Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_IllegalEvent(void *parameter)
{
	UNUSED_PARAM(parameter);	
	FATAL("TrainingManager_IllegalEvent");
}



/**********************************************************************************
TrainingManager_SendLaInfoReq


Description:
------------

	Recieved Start training trigger from Group Manager - 
	If there is at least one active VAP - firstly ask for active station bitmap from Link adaptive and wait for response

	If no active VAP - just send failure confirmation to Group Manager

Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_SendLaInfoReq(void *parameter)
{
	UNUSED_PARAM(parameter);
	ILOG2_D("TrainingManager_SendLaInfoReq, muActiveVapBitmap = 0x%04X", TrainingManagerDb.staticParams.muActiveVapsBitmap);

#ifdef TRAINING_STATISTICS
	//increment timer expired statistic counter
	TrainingStatistics.trainingTimerExpiredCounter++;
#endif //TRAINING_STATISTICS
	
	if(TrainingManagerDb.staticParams.muActiveVapsBitmap == 0)
	{

#ifdef TRAINING_STATISTICS
		//increment training not started statistic counter
		TrainingStatistics.trainingNotStartedCounter++;
#endif //TRAINING_STATISTICS

		GroupManager_TrainingFailed();

		/*Do not change state in this case*/
	}	
	else
	{		
		//send L/A info request to Access manager (Group Link adaptive) 
		OSAL_SEND_NO_DATA_MESSAGE(LINK_ADAPTATION_GET_TRAINIG_INFO, TASK_LINK_ADAPTATION, TrainingManagerDb.staticParams.selectedVap);

		//change state
		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_WAIT_FOR_LA_INFO);

	}
}



/**********************************************************************************
TrainingManager_CheckNdpaConfirmationStatus


Description:
------------

	Confirmation from NDPA recieved - check if the re are valid results and start result processing if yes.

Input: NDPA confirmation status
------
	
Returns: none
--------
	 
**********************************************************************************/
static void TrainingManager_CheckNdpaConfirmationStatus(void *parameter)
{
	TrainingManagerNdpaCfm_t* ndpaCfmParams = (TrainingManagerNdpaCfm_t*)parameter;

	ILOG2_D("TrainingManager_CheckNdpaConfirmationStatus, Status = %d", ndpaCfmParams->status);

	if(ndpaCfmParams->status == TRAINING_MANAGER_NDPA_STATUS_FAIL)
	{

#ifdef TRAINING_STATISTICS
		//increment unsucessfull training statistic counter
		TrainingStatistics.trainingFinishedUnsuccessfullyCounter++;
#endif //TRAINING_STATISTICS
		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_IDLE);
		//send group manager training failed
		GroupManager_TrainingFailed();
	} 
	else
	{
		GroupManagerTrainingResults_t trainingResults;
		
		//start results processing
		TrainingManager_StartResultsProcessing();

#ifdef TRAINING_STATISTICS
		//increment training successfull statistic counter
		TrainingStatistics.trainingFinishedSuccessfullyCounter++;
#endif //TRAINING_STATISTICS

		trainingResults.pGroupCombinationTable = TrainingManagerDb.dynamicParams.groupCombinationTable;
		trainingResults.resultsBw = TrainingManagerDb.dynamicParams.currentBw;

		ILOG2_D("Group Manager result BW = %d", trainingResults.resultsBw);
		SLOG2(0, 0, GroupCombinationTableDebug_t, trainingResults.pGroupCombinationTable);

		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_IDLE);
		GroupManager_TrainingResultsWereReceived(&trainingResults);
	}	
}


/**********************************************************************************
TrainingManager_StopTrainingDuringTraining



Description:
------------

	Stop training during ongoing training - which means we are waiting for results and NDPA confirmation.
	Send NDPA stop request and change state in order to ignore the reuslts when recieved

Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/
#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_StopTrainingDuringTraining(void *parameter)
{
	ILOG2_V("TrainingManager_StopTrainingDuringTraining");
	UNUSED_PARAM(parameter);	
	//change state
	TrainingManager_ChangeState(TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA);

	//send NDPA stop request
	NdpaManager_StopTxReq();
}

#endif

/**********************************************************************************
TrainingManager_StopTrainingDuringWaitingForLaInfo


Description:
------------

	We didn't started training yet - only waiting for parameters from link adaptive
	So just change state and when the confirmation will be recieved it will be ignored and Confirmation will be sent to the Group manager

Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_StopTrainingDuringWaitingForLaInfo(void *parameter)
{
	UNUSED_PARAM(parameter);
	ILOG2_V("TrainingManager_StopTrainingDuringWaitingForLaInfo");
	
	//change state
	TrainingManager_ChangeState(TRAINING_MANAGER_STATE_STOP_WAIT_FOR_LA_INFO);
}



/**********************************************************************************
TrainingManager_AddVap




Description:
------------

	Add VAP to active VAP bitmap parameter in training database if supports MU

	From Training Manager perspective the VAP is only Added when the SET BSS command is received
	However, the Set BSS command may also be sent when VAP is already up - e.g. set slot time.
	Training manager should possibly add the VAP only when the Set BSS is received during VAP bringup


Input: UMI_SET_BSS
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_AddVap(void *parameter)
{
	K_MSG* trainingManagerMessage = (K_MSG*)parameter;
	UMI_SET_BSS* pSetBssMsg = (UMI_SET_BSS*) EXTRACT_VAP_MANAGER_MSG(trainingManagerMessage);

	uint8 vapId = pSetBssMsg->vapId;
#ifdef TRAINING_MANAGER_DEBUG
	ILOG0_D("TrainingManager_AddVap, new VAP id = %d", vapId);
#endif
				
	/*From Training Manager perspective the VAP is only Added when the SET BSS command is received
	However, the Set BSS command may also be sent when VAP is already up - e.g. set slot time.
	Training manager should possibly add the VAP only when the Set BSS is received during VAP bringup*/
	if (!GET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.staticParams.activeVapsBitmap, vapId)) // if vap is not active (i.e. this is the first set bss message for this vap)
	{
		//set bit in active vap bitmap
		SET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.staticParams.activeVapsBitmap, vapId);
	}			
	if (MTLK_BFIELD_GET(pSetBssMsg->flags, VAP_ADD_FLAGS_VHT)) //if supports VHT, set Vap in muActiveVapBitmap
	{
		//VAP supports MU - set in MU active vaps bitmap
		SET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.staticParams.muActiveVapsBitmap, vapId);		
	}	
	/*Send Confirmation to VAP manager*/
	/*return message*/
	FILL_VAP_MANAGER_CONFIRM_MSG(trainingManagerMessage, vapId, VAP_MANAGER_SET_BSS, BSS_MANAGER_VAP_MANAGER_TRAINING_MANAGER_CLIENT);
	
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, trainingManagerMessage, vapId);

	/*Do not change state in this event*/

}




/**********************************************************************************
TrainingManager_RemoveVap



Description:
------------

	Remove VAP from active VAP bitmap parameter in training database.

	The removal done based on UMI_STOP_VAP_TRAFFIC event - there is no point to send training on VAP which started removal process3

	Stop traffic can be recieved more than one on the same VAP, maybe be a case where the VAP is not in out bitmaps already but the event is recieved - 
	this is normal case and we will not ASSERT on it - just clear the bit again


Input: UMI_STOP_VAP_TRAFFIC
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_RemoveVap(void *parameter)
{
	K_MSG* trainingManagerMessage = (K_MSG*)parameter;	
	UMI_STOP_VAP_TRAFFIC* stopTrafficMsg = (UMI_STOP_VAP_TRAFFIC*) EXTRACT_VAP_MANAGER_MSG(trainingManagerMessage);

	uint8 vapId = stopTrafficMsg->vapId;

	//zero bit in bitmaps
	CLR_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.staticParams.activeVapsBitmap, vapId);
	CLR_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.staticParams.muActiveVapsBitmap, vapId);

	FILL_VAP_MANAGER_CONFIRM_MSG(trainingManagerMessage,vapId,VAP_MANAGER_STOP_TRAFFIC, BSS_MANAGER_VAP_MANAGER_TRAINING_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, trainingManagerMessage, vapId);

	/*Do not change state in this event*/
}



/**********************************************************************************
TrainingManager_AddSta


Description:
------------

	Add station to per VAP bitmap according to its VAP

	Only stations which supports VHT and MU will be recieved here


Input: UMI_STA_ADD
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_AddSta(void *parameter)
{
	UMI_STA_ADD* addStaMsgParams = (UMI_STA_ADD*)parameter;
	
	StaId stationIndex = addStaMsgParams->u16SID;
	uint8 vapIndex = addStaMsgParams->u8VapIndex;

	//set bit in active sta bitmap per vap
	 Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.connectedStaBitmap[vapIndex]), stationIndex);

	/*Do not change state in this event*/

	//no confirmation message for this event, Group Manager assumes that the add stations handler is done when the function ends.

}




/**********************************************************************************
TrainingManager_RemoveStaNotDuringTraining




Description:
------------

	Remove it from per VAP DB bitmaps

	We are not during trainign -  just send immidiate confirmation to Group Manager


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_RemoveStaNotDuringTraining(void *parameter)
{
	uint8 vapIndex;
	TrainingManagerRemoveStaParams_t* removeStaParams = (TrainingManagerRemoveStaParams_t *)parameter;
	StaId stationIndex = removeStaParams->staId;

	ILOG2_D("TrainingManager_RemoveStaNotDuringTraining, sta Id = 0x%X", removeStaParams->staId);

	//zero sta bit in sta bitmaps for all vaps
	for(vapIndex = 0; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		Utils_ZeroBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.connectedStaBitmap[vapIndex]),stationIndex);
	}

	//send immidiate confirmation to Group Manager
	GroupManager_StationIsNotInTraining(stationIndex);	

	/*Do not change state in this event*/
}



/**********************************************************************************
TrainingManager_RemoveStaDuringTraining




Description:
------------

	Remove it from per VAP DB bitmaps

	If the station is part of the training and we are during training - Send stop Tx to NDPA ,and save the station in pending for confirmation bitmap

	If not a part of training - send immidiate response with station id.


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/
#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_RemoveStaDuringTraining(void *parameter)
{
	uint8 vapIndex;
	TrainingManagerRemoveStaParams_t* removeStaParams = (TrainingManagerRemoveStaParams_t *)parameter;
	StaId stationIndex = removeStaParams->staId;

	ILOG2_D("TrainingManager_RemoveStaDuringTraining, sta Id = 0x%X", removeStaParams->staId);
	
	//zero sta bit in sta bitmaps for all vaps
	for(vapIndex = 0; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		Utils_ZeroBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.connectedStaBitmap[vapIndex]), stationIndex);
	}

	//check whether the station is a part of the current training
	if (Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), stationIndex))
	{
		/*station is a part of current training*/

		//save staId in waiting for remove station bitmap
		 Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.waitingForRemoveStaBitmap), stationIndex);

		//change state
		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA);

		//send NDPA stop request
		NdpaManager_StopTxReq();
	}
	else
	{
		//station is not a part of current training - send immidiate confirmation to Group Manager
		GroupManager_StationIsNotInTraining(stationIndex);	

		/*Do not change state in this case*/
	}
}

#endif

/**********************************************************************************
TrainingManager_RemoveStaDuringStopping




Description:
------------

	Remove it from per VAP DB bitmaps

	If the station is part of the training and we are already in stopping state 
	- save station in the pending stations bitmap and wait for the confirmation from NDPA Manager before confirming to Group Manager

	If not a part of training - send immidiate response with station id.


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_RemoveStaDuringStopping(void *parameter)
{
	uint8 vapIndex;
	TrainingManagerRemoveStaParams_t* removeStaParams = (TrainingManagerRemoveStaParams_t *)parameter;
	StaId stationIndex = removeStaParams->staId;

	ILOG2_D("TrainingManager_RemoveStaDuringStopping, sta Id = 0x%X", removeStaParams->staId);
	
	//zero sta bit in sta bitmaps for all vaps
	for(vapIndex = 0; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		Utils_ZeroBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.connectedStaBitmap[vapIndex]), stationIndex);
	}
	
	//check whether the station is a part of the current training
	if (Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), stationIndex))
	{
		/*station is a part of current training*/

		//save staId in waiting for remove station bitmap
		 Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.waitingForRemoveStaBitmap), stationIndex);

		/*do not send confirmation to Group Manager now, need to wait for NDPA comfirmation*/
	}
	else
	{
		//station is not a part of current training - send immidiate confirmation to Group Manager
		GroupManager_StationIsNotInTraining(stationIndex); 
	}	

	/*Do not change state in this event*/
}



/**********************************************************************************
TrainingManager_HaltStaNotDuringTraining



Description:
------------
	We are not during trainign -  just send immidiate confirmation to Group Manager


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_HaltStaNotDuringTraining(void *parameter)
{
	TrainingManagerHaltStaParams_t *haltStaParams = (TrainingManagerHaltStaParams_t*)parameter;

	ILOG2_D("TrainingManager_HaltStaNotDuringTraining, sta Id = 0x%X", haltStaParams->staId);

	//send immidiate confirmation to Group Manager (no need to zero bitmaps in this case, station is temporary haltted, but not removed)
	GroupManager_StationIsNotInTraining(haltStaParams->staId);	

	/*Do not change state in this event*/
}



/**********************************************************************************
TrainingManager_HaltStaDuringTraining



Description:
------------
	If the station is part of the training and we are during training - Send stop Tx to NDPA and save the station in pending for confirmation bitmap

	If not a part of training - send immidiate response with station id.


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/
#ifndef TRAINING_WAVE600_Z0

static void TrainingManager_HaltStaDuringTraining(void *parameter)
{	
	TrainingManagerHaltStaParams_t *haltStaParams = (TrainingManagerHaltStaParams_t*)parameter;

	ILOG2_D("TrainingManager_HaltStaDuringTraining, sta Id = 0x%X", haltStaParams->staId);
	
	//check whether the station is a part of the current training
	if (Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), haltStaParams->staId))
	{
		/*station is a part of current training*/

		//save staId in waiting for remove station bitmap
		 Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.waitingForRemoveStaBitmap), haltStaParams->staId);

		//change state
		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA);

		//send NDPA stop request
		NdpaManager_StopTxReq();
	}
	else
	{
		//station is not a part of current training - send immidiate confirmation to Group Manager
		GroupManager_StationIsNotInTraining(haltStaParams->staId); 

		/*Do not change state in this case*/
	}
	
}


/**********************************************************************************
TrainingManager_StopStaDuringTraining





Description:
------------
	If the station is part of the training and we are during training - Send stop Tx to NDPA without saving the station in pending for confirmation bitmap

	If not a part of training - do nothing


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_StopStaDuringTraining(void *parameter)
{	
	TrainingManagerStopStaParams_t *stopStaParams = (TrainingManagerStopStaParams_t*)parameter;

	ILOG2_D("TrainingManager_StopStaDuringTraining, sta Id = 0x%X", stopStaParams->staId);
	
	//check whether the station is a part of the current training
	if (Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), stopStaParams->staId))
	{
		/*station is a part of current training*/

		/* do not save staId in waiting for remove station bitmap*/

		//change state
		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_STOP_WAIT_FOR_NDPA);

		//send NDPA stop request
		NdpaManager_StopTxReq();
	}

		/*if station is not a part of current training - DO NOT send immidiate confirmation to Group Manager*/

		/*Do not change state if sta not a part of the current training*/	
}

#endif // TRAINING_WAVE600_Z0


/**********************************************************************************
TrainingManager_HaltStaDuringStopping



Description:
------------
	If the station is part of the training and we are already in stopping state 
	- save station in the pending stations bitmap and wait for the confirmation from NDPA Manager before confirming to Group Manager

	If not a part of training - send immidiate response with station id.


Input: STA ID
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_HaltStaDuringStopping(void *parameter)
{
	TrainingManagerHaltStaParams_t *haltStaParams = (TrainingManagerHaltStaParams_t*)parameter;

	
	//check whether the station is a part of the current training
	if (Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), haltStaParams->staId))
	{
		/*station is a part of current training*/

		//save staId in waiting for remove station bitmap
		 Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.staticParams.waitingForRemoveStaBitmap), haltStaParams->staId);
		
		/*do not send confirmation to Group Manager now, need to wait for NDPA comfirmation*/
	}	
	else
	{
		//station is not a part of current training - send immidiate confirmation to Group Manager
		GroupManager_StationIsNotInTraining(haltStaParams->staId); 
	}
	
	/*Do not change state in this event*/
}



/**********************************************************************************
TrainingManager_SendTrainingStoppedConfirmation


Description:
------------
	Send training stopped confirmation with station bitmap of the resuested stations.
	The bitmap may be empty in case stop training is requested without specified station and no more stop/remove stations where recieved.

	The confirmation status from NDPA is not relevant here since �� there are ready results we will ignore them anyway


Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_SendTrainingStoppedConfirmation(void *parameter)
{
	uint32 cfmStaBitmap[GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS];	
	UNUSED_PARAM(parameter);
	memcpy32(cfmStaBitmap, &TrainingManagerDb.staticParams.waitingForRemoveStaBitmap, (CONVERT_BYTES_TO_WORDS(sizeof(TrainingManagerStaBitmap_t))));
	// TODO: change to original log when stations num is 128
	//ILOG2_DDDD("TrainingManager_SendTrainingStoppedConfirmation, cfmBitmap = 0x%08X, 0x%08X, 0x%08X, 0x%08X", cfmStaBitmap[0], cfmStaBitmap[1], cfmStaBitmap[2], cfmStaBitmap[3]);
	ILOG2_DD("TrainingManager_SendTrainingStoppedConfirmation, cfmBitmap = 0x%08X, 0x%08X", cfmStaBitmap[0], cfmStaBitmap[1]);

	//clear wait for remove bitmap in training DB
	memset(&(TrainingManagerDb.staticParams.waitingForRemoveStaBitmap), 0, sizeof(TrainingManagerStaBitmap_t));
		
	GroupManager_TrainingWasStoppedConfirmation(cfmStaBitmap);

	TrainingManager_ChangeState(TRAINING_MANAGER_STATE_IDLE);
}



/**********************************************************************************
TrainingManager_IgnoreStopTraining



Description:
------------
	Do nothing - trainig Manager already during stopping or we should ignore the request 
	(in case that stop station recieved during waiting for LA info - it means that nss or bw were changed for the station, 
	but since the training manager didn't choose stations yet, they will be chosen based on the new parameters and it is ok)


Input: none
------
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_IgnoreStopTraining(void *parameter)
{
	UNUSED_PARAM(parameter);	
	ILOG2_V("TrainingManager_IgnoreStopTraining");
}



/**********************************************************************************
TrainingManager_StartTraining


Description:
------------
	Start training - 
	1. building pools
	2. choosing stations for training
	3. building test plan

	invoked after recieving the link adaptation parameters

Input: active station bitmap from link adaptive
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_StartTraining(void *parameter)
{
	uint32 currentPoolBitmap;
	trainingManagerPools_e selectedPoolId;
	MuEffectiveRateDbRamEntry_u invalidTestPlanEntry;
#ifndef TRAINING_WAVE600_Z0
	NdpaManagerStartTxReq_t ndpaStartTxParams;
#endif // TRAINING_WAVE600_Z0
	TrainingManagerLinkAdaptiveInfoCfm_t *pLinkAdaptiveTrainingInfo = (TrainingManagerLinkAdaptiveInfoCfm_t *)parameter;
#ifdef TRAINING_WAVE600_Z0
	TrainingManagerNdpaCfm_t ndpaCfmParams;
#endif
#ifdef TRAINING_MANAGER_DEBUG
	TrainingVectorDebugStruct_t *trainingVectorDebug = (TrainingVectorDebugStruct_t *)(ndpaStartTxParams.trainingVector);
	uint8 numOfPools = NUM_OF_POOLS;
	uint8 numOfConnectedStation = StaDb_getNumOfConnectedStations(); // TBD Gen6 - this is total num of connected STAs for BOTH bands. Is this OK?

	UNUSED_PARAM(numOfPools);
	ILOG2_DDDD("TrainingManager_StartTraining, num of connected station = %d, All active vaps bitmap = 0x%04X, MU active vaps bitmap = 0x%04X, num of Pools = %d", numOfConnectedStation, TrainingManagerDb.staticParams.activeVapsBitmap, TrainingManagerDb.staticParams.muActiveVapsBitmap, numOfPools);
#endif //TRAINING_MANAGER_DEBUG

	if (TrainingManager_BuildPools(pLinkAdaptiveTrainingInfo) == VAP_SELECTED)
	{	
#ifdef TRAINING_MANAGER_DEBUG		
		ILOG0_D("VAP SELECTED!!!, VAP ID = %d", TrainingManagerDb.staticParams.selectedVap);
#endif

#ifdef TRAINING_STATISTICS
		//increment training started statistic counter
		TrainingStatistics.trainingStartedCounter++;
#endif //TRAINING_STATISTICS
		
		//initialize all training DB dynamic params
		memset(&(TrainingManagerDb.dynamicParams), 0, sizeof(TrainingDynamicParams_t)); 

		TrainingManagerDb.dynamicParams.numOfAntennas = groupManagerGetCoCTxAntsNum();

		//select first pool
		selectedPoolId = Utils_CountTrailingZeros(TrainingManagerPoolDb.dynamicParams.activePoolsBitmap); //find first lsb set

		//set pools current BW (all other pools which will be selected must be with the same BW)
		TrainingManagerDb.dynamicParams.currentBw = Utils_CountTrailingZeros(TrainingManagerPoolDb.staticParams[selectedPoolId].poolFeatureBitmap.bitFields.bw);
		

		//find final pool bitmap according to first selected pool and its selectable pools bitmap
		currentPoolBitmap = (TrainingManagerPoolDb.staticParams[selectedPoolId].selectablePoolsBitmap) & (TrainingManagerPoolDb.dynamicParams.activePoolsBitmap);

		while((currentPoolBitmap!=0) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES) && (TrainingManagerDb.dynamicParams.currentTrainingVectorIdx < GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING))
		{
			//select pool
			selectedPoolId = Utils_FindLastSetAndClear(&currentPoolBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentPoolBitmap)));

			//select stations from current pool and build location bitmap
			TrainingManager_SelectStations(selectedPoolId);

			//set pool in active pool bitmap for current training
			SET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.dynamicParams.activePoolsInTrainingBitmap, selectedPoolId);

			TrainingManager_BuildTestPlan(selectedPoolId);
		}

		
		//Fill the rest test plan entries if exist with invalid entry indication
		invalidTestPlanEntry.val = 0;
		invalidTestPlanEntry.inputEntryBitFields.invalidEntry = TRUE;

		if(TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES)
		{
			memset32(&(TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx]), invalidTestPlanEntry.val, (MAX_NUM_OF_TEST_PLAN_LINES - TrainingManagerDb.dynamicParams.currentTestPlanIdx));
		}

		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_WAIT_FOR_RESULTS);
#ifndef TRAINING_WAVE600_Z0	

		/*Fill ndpa start tx request and send it to NDPA Manager - 
		Number of stations in current training is same as TrainingManagerDb.dynamicParams.currentTrainingVectorIdx*/
		ndpaStartTxParams.numOfStations = TrainingManagerDb.dynamicParams.currentTrainingVectorIdx; 
		MEMCPY(ndpaStartTxParams.trainingVector, TrainingManagerDb.dynamicParams.trainingVector, sizeof(ndpaStartTxParams.trainingVector));
		ndpaStartTxParams.vapId = TrainingManagerDb.staticParams.selectedVap;
		ndpaStartTxParams.currentBw = TrainingManagerDb.dynamicParams.currentBw;
		
#ifdef TRAINING_MANAGER_DEBUG
		ILOG0_DD("Build test plan and Training Vector finished, send Start Tx to NDPA Manager, num of station = %d, VAP ID = %d", ndpaStartTxParams.numOfStations, ndpaStartTxParams.vapId);
		SLOG0(0, 0, TrainingVectorDebugStruct_t, trainingVectorDebug);
#endif		
		
		NdpaManager_StartTxReq(&ndpaStartTxParams);
#else // TRAINING_WAVE600_Z0
		ndpaCfmParams.status = TRAINING_MANAGER_NDPA_STATUS_SUCCESS;
		TrainingManager_CheckNdpaConfirmationStatus(&ndpaCfmParams);
#endif
	}
	else
	{
#ifdef TRAINING_MANAGER_DEBUG		
		ILOG0_V("VAP NOT SELECTED!!!");
#endif
		/*VAP not selected - there was no VAP with at lest one active pool (pool with minimal number of stations)*/

#ifdef TRAINING_STATISTICS
		//increment training not started statistic counter
		TrainingStatistics.trainingNotStartedCounter++;
#endif //TRAINING_STATISTICS

		//send Trainig failed indication to Group manager
		GroupManager_TrainingFailed();

		TrainingManager_ChangeState(TRAINING_MANAGER_STATE_IDLE);	
	}		

}



/**********************************************************************************
TrainingManager_BuildPools




Description:
------------
	Building pools for all VAPs in Round Robin till found VAP with active pool.
	Active pool is pool with minimal number of stations, initialized in Pool DB static params

Input: active station bitmap - stations that can participate in Training according to Link adaptive analyzing
		(have good performance in SU)
-----
	
Returns: indication whether VAP selected or not - 
		if VAP not selected the training will end with failure and indication will be sent to Group Manager
--------
	 
**********************************************************************************/

static vapSelectionStatus_e TrainingManager_BuildPools(TrainingManagerLinkAdaptiveInfoCfm_t *pLinkAdaptiveTrainingInfo)
{
	TrainingManagerStaBitmap_t finalActiveStaBitmap, currentStaBitmap; 
	uint32 muActiveVapBitmap = TrainingManagerDb.staticParams.muActiveVapsBitmap;
	PoolFeatureBitmap_u currentStaFeaturesBitmap = {0};
	StaId staId;
#ifdef ENET_INC_ARCH_WAVE600
	uint32 checkedVapBitmap = 0;
#else
	uint16 checkedVapBitmap = 0;
	Tcr3_t *pTcr3ForCurrentBw;
#endif //ENET_INC_ARCH_WAVE600
	uint8 bwLimit;
	trainingManagerPools_e currentPoolId;
	trainingManagerPools_e poolId;
	uint8*	tcrPerBwAdd;	
#ifdef TRAINING_MANAGER_DEBUG
	TrainingManagerStaBitmap_t debugActiveStaBitmap;
#endif
	
	ILOG2_V("TrainingManager_BuildPools, input is active sta bitmap from Link adaptive:");
	SLOG2(0, 0, TrainingManagerStaBitmap_t, &(pLinkAdaptiveTrainingInfo->activeStaBitmap));

	if (TrainingManagerDb.staticParams.muActiveVapsBitmap != 0)
	{		
		//find final active sta bitmap
		TrainingManager_StaBitmapBitwiseNOT(&finalActiveStaBitmap, (TrainingManagerStaBitmap_t *)(&(TrainingManagerDb.staticParams.groupManagerTrainingBitmaps->haltStationsBitmap)));
		TrainingManager_StaBitmapBitwiseAND(&finalActiveStaBitmap, &finalActiveStaBitmap, &(pLinkAdaptiveTrainingInfo->activeStaBitmap));

		ILOG2_V("Final Active STA bitmap");
		SLOG2(0, 0, TrainingManagerStaBitmap_t, &finalActiveStaBitmap);

#ifdef TRAINING_MANAGER_DEBUG
		//print group manager state for each active station for debug
		memcpy32(&debugActiveStaBitmap, &finalActiveStaBitmap, GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS);
		while(Utils_IsBitmapNotEmpty(debugActiveStaBitmap.staBitmap, HW_NUM_OF_STATIONS))
		{
			staId = Utils_FindLastSetAndClear(debugActiveStaBitmap.staBitmap, HW_NUM_OF_STATIONS);
			//ILOG0_DD("STA ID %d, State = %d", staId, GroupManager_GetStaState(staId));
		}
#endif	
		while (checkedVapBitmap != VAP_BITMAP_MASK)
		{
			TrainingManagerDb.staticParams.selectedVap = (TrainingManagerDb.staticParams.selectedVap + 1) % HW_NUM_OF_VAPS;

			//set selected vap bit in checked bitmap
			SET_BIT_IN_SINGLE_BITMAP(checkedVapBitmap, TrainingManagerDb.staticParams.selectedVap);

			ILOG2_D("current VAP Id = %d", TrainingManagerDb.staticParams.selectedVap);

			//if VAP supports MU - start building pool for this VAP
			if (Utils_GetBitFromBitmap(&muActiveVapBitmap, TrainingManagerDb.staticParams.selectedVap))
			{
				TrainingManager_StaBitmapBitwiseAND(&currentStaBitmap, &(TrainingManagerDb.staticParams.connectedStaBitmap[TrainingManagerDb.staticParams.selectedVap]), &finalActiveStaBitmap);

				//reset pool DB dynamic params
				memset(&(TrainingManagerPoolDb.dynamicParams), 0, sizeof(DynamicPoolInformation_t));

#ifdef TRAINING_FIRST_PHASE_POOL
				TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolFeatureBitmap.bitFields.bw = (0x1 << pLinkAdaptiveTrainingInfo->maxWorkingBandwidth[TrainingManagerDb.staticParams.selectedVap]);
#endif //TRAINING_FIRST_PHASE_POOL

				while(Utils_IsBitmapNotEmpty(currentStaBitmap.staBitmap, HW_NUM_OF_STATIONS))
				{
					//find first lsb set and clear
					staId = Utils_FindLastSetAndClear(currentStaBitmap.staBitmap, HW_NUM_OF_STATIONS);

					//build sta features bitmap
					bwLimit = StaDbHwEntries[staId].common.dataBwLimit;
					currentStaFeaturesBitmap.bitFields.bw = (1 << bwLimit);
#ifdef ENET_INC_ARCH_WAVE600					
					tcrPerBwAdd = (uint8*)&(StaDbHwEntries[staId].common.word24);
					
					currentStaFeaturesBitmap.bitFields.nss = (1 << EXTRACT_NSS_FROM_VHT_HE_RATE(tcrPerBwAdd[bwLimit]));
					currentStaFeaturesBitmap.bitFields.mcs = (1 << EXTRACT_MCS_FROM_VHT_HE_RATE(tcrPerBwAdd[bwLimit]));
#else
					tcrPerBwAdd = (uint8*)&(StaDbHwEntries[staId].common.word5) + bwLimit * (FM_STRUCT_OFFSET(StaDbCommon_t,word7) - FM_STRUCT_OFFSET(StaDbCommon_t,word5));
					pTcr3ForCurrentBw = (Tcr3_t*)(tcrPerBwAdd);
					
					currentStaFeaturesBitmap.bitFields.nss = (1 << EXTRACT_NSS_FROM_VHT_HE_RATE(pTcr3ForCurrentBw->wpRate));
					currentStaFeaturesBitmap.bitFields.mcs = (1 << EXTRACT_MCS_FROM_VHT_HE_RATE(pTcr3ForCurrentBw->wpRate));
#endif //ENET_INC_ARCH_WAVE600

					ILOG2_DD("Current feature bitmap for station %d = 0x%08X", staId, currentStaFeaturesBitmap.val);

					//initialize pool Id before entring to loop
					currentPoolId = FIRST_POOL;

					while(currentPoolId < NUM_OF_POOLS)
					{	
						//check if sta fits pool's bitmap criteria
						if(currentStaFeaturesBitmap.val == ((currentStaFeaturesBitmap.val) & (TrainingManagerPoolDb.staticParams[currentPoolId].poolFeatureBitmap.val)))
						{
							//sta fits the creteria - add station to current pool
							Utils_SetBitInBitmap((uint32*)&(TrainingManagerPoolDb.dynamicParams.poolInformation[currentPoolId].stationsInPoolBitmap), staId);							

							//increment num of stations in pool
							TrainingManagerPoolDb.dynamicParams.poolInformation[currentPoolId].numOfStationsInPool++;
						}						
						
						currentPoolId++;
					}				
				}	

				//update active pools bitmaps for current VAP
				for(poolId = FIRST_POOL; poolId < NUM_OF_POOLS; poolId++)
				{
					if(TrainingManagerPoolDb.dynamicParams.poolInformation[poolId].numOfStationsInPool >=  TrainingManagerPoolDb.staticParams[poolId].poolMuSTAsLowerLimit)
					{
						//update pool in active pool bitmap since we have minimum number of station in the pool
						SET_BIT_IN_SINGLE_BITMAP(TrainingManagerPoolDb.dynamicParams.activePoolsBitmap, poolId);
					}
				}	

				if (TrainingManagerPoolDb.dynamicParams.activePoolsBitmap != 0)
				{
					return VAP_SELECTED;
				}					
			}
		}

		//if we got here - we went over all VAPs and didn't find active VAP yet
		return VAP_NOT_SELECTED;		
	}
	else
	{
		//active VAP is not found
		TrainingManagerDb.staticParams.selectedVap = TRAINING_MANAGER_INVALID_VAP;

		return VAP_NOT_SELECTED;
	}
}



/**********************************************************************************
TrainingManager_SelectStations



Description:
------------
	Select stations from current pool and update the training vector

	The selection is done according to priority restrictions and pool limitations 
	that are initialized in PoolDB static params

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_SelectStations(trainingManagerPools_e poolId)
{
	uint8 priorityIndex = 0;
	TrainingManagerStaBitmap_t currentStaBitmap;
	TrainingManagerStaBitmap_t priorityStaBitmap;
	uint8 numOfStationsInCurrentPriority;
	StaId staIndex;

	ILOG2_D("Start select stations!!! , pool id = %d", poolId);

	 //start with the original station in pool bitmap and clear each selected station until one of conditions occur
	 memcpy32(&currentStaBitmap, &TrainingManagerPoolDb.dynamicParams.poolInformation[poolId].stationsInPoolBitmap, (CONVERT_BYTES_TO_WORDS(sizeof(TrainingManagerStaBitmap_t))));

	/*conditions to enter into the loop of choosing stations per priority:
	1. There are stations which are not selected yet
	2. We didn't reach maximum stations for training (16)
	3. we didn't reach the maximum stations from pool
	4. The priority index is valid
	*/
	while (Utils_IsBitmapNotEmpty((uint32*)(&currentStaBitmap), HW_NUM_OF_STATIONS) && (TrainingManagerDb.dynamicParams.currentTrainingVectorIdx < GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING)  && (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] <= TrainingManagerPoolDb.staticParams[poolId].poolMuSTAsUpperLimit) && (priorityIndex < GROUP_MANAGER_NUMBER_OF_STATION_PRIORITIES))
	{
		SLOG2(0, 0, GroupManagerStationBitmap_t, &(TrainingManagerDb.staticParams.groupManagerTrainingBitmaps->stationPriorityBitmap[priorityIndex]));
		//find all stations from pool with the current priority
		TrainingManager_StaBitmapBitwiseAND(&priorityStaBitmap, &(TrainingManagerPoolDb.dynamicParams.poolInformation[poolId].stationsInPoolBitmap), (TrainingManagerStaBitmap_t *)(&(TrainingManagerDb.staticParams.groupManagerTrainingBitmaps->stationPriorityBitmap[priorityIndex])));
		
		numOfStationsInCurrentPriority = 0;

		/*conditions to enter into the loop of choosing stations for current priority:
		1. There are stations which are not selected yet in current priority
		2. We didn't reach maximum stations from current priority
		3. We didn't reach maximum stations for training (16)
		4. we didn't reach the maximum stations from pool
		*/
		while (Utils_IsBitmapNotEmpty((uint32*)(&priorityStaBitmap), HW_NUM_OF_STATIONS) && 
			(numOfStationsInCurrentPriority < TrainingManagerPoolDb.staticParams[poolId].maxNumOfStaPerPriority[priorityIndex]) && 
			(TrainingManagerDb.dynamicParams.currentTrainingVectorIdx < GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING) && 
			(TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] <= TrainingManagerPoolDb.staticParams[poolId].poolMuSTAsUpperLimit))
		{
			//find first set lsb and clear
			staIndex = Utils_FindLastSetAndClear(priorityStaBitmap.staBitmap, HW_NUM_OF_STATIONS);


			/*Make sure the bitmap station bit in bitmap is on, otherwize station is being assigned with more that one priority (appears in more than one bitmap)*/
			DEBUG_ASSERT(Utils_GetBitFromBitmap((uint32*)(&currentStaBitmap), staIndex) == 1);

			/*clear staId bit also from the currentStaBitmap bitmap which contains all stations which are not selected meanwhile 
			(need in order to stop in the first loop in case there are no more stations)*/
			Utils_ZeroBitInBitmap((uint32*)(&currentStaBitmap), staIndex);

			//check if station is already selected for current training
			if (!Utils_GetBitFromBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), staIndex))
			{	
				/*station is not selected yet for current training - 
				select it and set bit to indicate that station is selected to be a part of current training*/
				Utils_SetBitInBitmap((uint32*)&(TrainingManagerDb.dynamicParams.phyTrainingStaBitmap), staIndex);
				
				//update training vector with current station
				TrainingManagerDb.dynamicParams.trainingVector[TrainingManagerDb.dynamicParams.currentTrainingVectorIdx] = staIndex;

				/*update station location table in the training vector to use it in case we will try to choose this station again from another pool - 
				the location should be updated in the location bitmap of the other pool*/
				TrainingManagerDb.dynamicParams.staLocationTable[staIndex] = TrainingManagerDb.dynamicParams.currentTrainingVectorIdx;

				//update pool location bitmap
				SET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.dynamicParams.locationBitmap[poolId], TrainingManagerDb.dynamicParams.currentTrainingVectorIdx);

				//increment training vector index
				TrainingManagerDb.dynamicParams.currentTrainingVectorIdx++;
			}

			else
			{
				//station already exists in training vector - update pool location bitmap
				SET_BIT_IN_SINGLE_BITMAP(TrainingManagerDb.dynamicParams.locationBitmap[poolId], TrainingManagerDb.dynamicParams.staLocationTable[staIndex]);
			}

			ILOG2_D("STA index selected = %X", staIndex);

			numOfStationsInCurrentPriority++;
			TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId]++;
		}
		priorityIndex++;
	}
}




/**********************************************************************************
TrainingManager_BuildTestPlan


Description:
------------
	Start building test plan for specific pool after choosing all stations participating in current training for the pool

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_BuildTestPlan(trainingManagerPools_e poolId)
{

#ifdef TRAINING_FIRST_PHASE_POOL
	testPlanPolicies_e currentPolicy;
	uint8 currentPolicyBitmap = TrainingManagerPoolDb.staticParams[poolId].testPlanPoliciesBitmap;
	uint8 priorityIndex = 0;
	
	ILOG2_DDD("Start Building Test Plan!!! , pool id = %d, current policies bitmap = 0x%04X, num of atennas = %d", poolId, currentPolicyBitmap , TrainingManagerDb.dynamicParams.numOfAntennas);
	while((currentPolicyBitmap!=0) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES))
	{
		currentPolicy = TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[TrainingManagerDb.dynamicParams.numOfAntennas - 1][priorityIndex];

		if(currentPolicy == INVALID_POLICY)
		{
			break;
		}
		
		//check if policy in current priority is enabled in policyBitmap
		if (GET_BIT_IN_SINGLE_BITMAP(currentPolicyBitmap, currentPolicy))
		{			
			CLR_BIT_IN_SINGLE_BITMAP(currentPolicyBitmap, currentPolicy);
			
			ILOG2_DD("TrainingManager_BuildTestPlan: currentPlicyBitmap = 0x%04X, currentPolicy = %d", currentPolicyBitmap, currentPolicy);
			TestPlanPolicyHandler[currentPolicy].handler(poolId);
		}
		priorityIndex++;
	}
#else

	testPlanPolicies_e currentPolicy;
	uint32 currentPolicyBitmap = TrainingManagerPoolDb.staticParams[poolId].testPlanPoliciesBitmap;

	ILOG2_D("Start Building Test Plan!!! , pool id = %d", poolId);
	while((currentPolicyBitmap!=0) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES))
	{
		//find first set lsb - each index in the bitmap represents test plan policy id (testPlanPolicies_e)
		currentPolicy = (testPlanPolicies_e)Utils_FindLastSetAndClear(&currentPolicyBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentPolicyBitmap)));
		//ILOG0_DD("TrainingManager_BuildTestPlan: currentPlicyBitmap %d, currentPolicy %d", currentPolicyBitmap, currentPolicy);
		TestPlanPolicyHandler[currentPolicy].handler(poolId);
	}
#endif //TRAINING_FIRST_PHASE_POOL	
}




/**********************************************************************************
TrainingManager_StartResultsProcessing



Description:
------------
	Start results processing after recieving successful indication from NDPA Manager and we are not during stopping

	This function calls the per policy and per segment functions which restores the test plan lines and process the output per line to update group combination table

Input: none
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_StartResultsProcessing(void)
{
	trainingManagerPools_e poolId;
	uint8 trainingVectorIndex;
	uint32 currentPoolBitmap = TrainingManagerDb.dynamicParams.activePoolsInTrainingBitmap;

	ILOG2_V("TrainingManager_StartResultsProcessing");

	//initialize Group Combination Table
	TrainingManager_InitializeGroupCombinationTable();

	/*1. go over all active pools in training (do not need to search for pools again through "selectable pools", etc. since the pools are already selected and we can use this info
	  2. Stop when there are no more pools or we get to last test plan line that was filled during building the test plan (can be less than 1024 so the loop will be shorter*/
	while((currentPoolBitmap != 0) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx))
	{
		poolId =  Utils_FindLastSetAndClear(&currentPoolBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentPoolBitmap)));
		TrainingManager_UpdateGroupCombinationTable(poolId);
	}

	//update STA ID for all Group combination Table entries
	for(trainingVectorIndex = 0; trainingVectorIndex < TrainingManagerDb.dynamicParams.currentTrainingVectorIdx; trainingVectorIndex++)
	{
		TrainingManagerDb.dynamicParams.groupCombinationTable[trainingVectorIndex].staId = TrainingManagerDb.dynamicParams.trainingVector[trainingVectorIndex];

	}
}



/**********************************************************************************
TrainingManager_UpdateGroupCombinationTable


Description:
------------
	Restore test plan line per policy for current pool ID and process it to update group combination table entries for all stations in the group.

	This function is invoked from the Result processing main func and calls the specific handlers per policy and per Segment that defined below

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_UpdateGroupCombinationTable(trainingManagerPools_e poolId)
{
#ifdef TRAINING_FIRST_PHASE_POOL
	testPlanPolicies_e currentPolicy;
	uint8 currentPolicyBitmap = TrainingManagerPoolDb.staticParams[poolId].testPlanPoliciesBitmap;
	uint8 priorityIndex = 0;

	ILOG2_DDD("TrainingManager_UpdateGroupCombinationTable, poolId = %d, testPlanPoliciesBitmap = 0x%04X, num of antennas = %d", poolId, currentPolicyBitmap, TrainingManagerDb.dynamicParams.numOfAntennas);

	while((currentPolicyBitmap!=0) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx))
	{
		currentPolicy = TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[TrainingManagerDb.dynamicParams.numOfAntennas - 1][priorityIndex];

		if(currentPolicy == INVALID_POLICY)
		{
			break;
		}
		
		//check if policy in current priority is enabled in policyBitmap
		if (GET_BIT_IN_SINGLE_BITMAP(currentPolicyBitmap, currentPolicy))
		{			
			CLR_BIT_IN_SINGLE_BITMAP(currentPolicyBitmap, currentPolicy);
			ILOG2_DD("TrainingManager_UpdateGroupCombinationTable: currentPlicyBitmap = 0x%04X, currentPolicy = %d", currentPolicyBitmap, currentPolicy);
	
			UpdateGroupCombinationTablePolicyHandler[currentPolicy].handler(poolId);
		}
		priorityIndex++;
	}
#else
	testPlanPolicies_e currentPolicy;
	uint32 currentPolicyBitmap = TrainingManagerPoolDb.staticParams[poolId].testPlanPoliciesBitmap;

	ILOG2_D("TrainingManager_UpdateGroupCombinationTable, poolId = %d", poolId);

	while((currentPolicyBitmap!=0) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx))
	{
		//find first set lsb - each index in the bitmap represents test plan policy id (testPlanPolicies_e)
		currentPolicy = (testPlanPolicies_e)Utils_FindLastSetAndClear(&currentPolicyBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentPolicyBitmap)));
		
		UpdateGroupCombinationTablePolicyHandler[currentPolicy].handler(poolId);
	}
#endif //TRAINING_FIRST_PHASE_POOL
}



/**********************************************************************************
TrainingManager_BuildTestPlanForPolicy[POLICY TYPE]


Description:
------------
	Handlers per policy for building test plan input lines
	Each function calls the Segment funcs that corresponds to the policy id.

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/


static void TrainingManager_BuildTestPlanForPolicy1_SingleStaCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 0");
	TrainingManager_BuildTestPlanForSegment_1xxx(poolId);
}

static void TrainingManager_BuildTestPlanForPolicy2_SingleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 1");
	TrainingManager_BuildTestPlanForSegment_1xxx(poolId);
	TrainingManager_BuildTestPlanForSegment_2xxx(poolId);
}

static void TrainingManager_BuildTestPlanForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 2");

#ifndef TRAINING_WAVE600_Z0
	TrainingManager_BuildTestPlanForSegment_22xx(poolId);
	TrainingManager_BuildTestPlanForSegment_12xx(poolId);
	TrainingManager_BuildTestPlanForSegment_21xx(poolId);
#endif
	TrainingManager_BuildTestPlanForSegment_11xx(poolId);	
}

static void TrainingManager_BuildTestPlanForPolicy4_DoubleStaCombinationSegmentTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 3");
#ifndef TRAINING_WAVE600_Z0
	TrainingManager_BuildTestPlanForSegment_22xx(poolId);
#else
	UNUSED_PARAM(poolId);	
#endif
}

static void TrainingManager_BuildTestPlanForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 4");
	TrainingManager_BuildTestPlanForSegment_111x(poolId);
}

static void TrainingManager_BuildTestPlanForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 5");
	TrainingManager_BuildTestPlanForSegment_111x(poolId);
#ifndef TRAINING_WAVE600_Z0
	TrainingManager_BuildTestPlanForSegment_211x(poolId);
	TrainingManager_BuildTestPlanForSegment_121x(poolId);
	TrainingManager_BuildTestPlanForSegment_112x(poolId);
#endif
}

static void TrainingManager_BuildTestPlanForPolicy7_FourStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Bulding Test Plan for policy 6");
	TrainingManager_BuildTestPlanForSegment_1111(poolId);
}


/******************END Of TrainingManager_BuildTestPlanForPolicy[POLICY TYPE] ******************************/




/**********************************************************************************
TrainingManager_BuildTestPlanForSegment_[SEGMENT_STRUCTURE]


Description:
------------
	Handlers per segment for building test plan input lines

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/


static void TrainingManager_BuildTestPlanForSegment_1xxx(trainingManagerPools_e poolId)
{
	uint8 i;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG
	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_1)
	{
		ILOG2_V("Bulding Test Plan for Segment 1xxx");

		//while we have more stations in pool and there are still free test plan lines
		while((currentLocationBitmap!=0) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES))
		{
			//find first lsb set - this is the location station from current pool in the training vector
			i = Utils_FindLastSetAndClear(&currentLocationBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentLocationBitmap)));

			//update test plan line with the station location, total Nsts = 1
			testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
			testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_1;
			testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
			TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

			SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

			//increment current test plan index
			TrainingManagerDb.dynamicParams.currentTestPlanIdx++;

#ifdef TRAINING_MANAGER_DEBUG
			debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
			
		}

#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 1XXX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}


static void TrainingManager_BuildTestPlanForSegment_2xxx(trainingManagerPools_e poolId)
{
	uint8 i;
	uint8 maxSupportedNss = 0; 
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_2)
	{
		ILOG2_V("Bulding Test Plan for Segment 2xxx");

		//while we have more stations in pool and there are still free test plan lines
		while((currentLocationBitmap!=0) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES))
		{
			//find first lsb set - this is the location station from current pool in the training vector
			i = Utils_FindLastSetAndClear(&currentLocationBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentLocationBitmap)));

			//check if the station supports 2 nss
			maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
			if (maxSupportedNss >= SPATIAL_STREAM_2)
			{
				//update test plan line with the station location, total Nsts = 2
				testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
				testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = i;
				testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_2;
				testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
				TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

				SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

				//increment current test plan index
				TrainingManagerDb.dynamicParams.currentTestPlanIdx++;

#ifdef TRAINING_MANAGER_DEBUG
				debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
			}
		}

#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 2XXX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}

#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_BuildTestPlanForSegment_22xx(trainingManagerPools_e poolId)
{

	uint8 startIndex, stopIndex, i, j;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Bulding Test Plan for Segment 22xx");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if both stations support 2 nss
							maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);

							if ((maxSupportedNss >= SPATIAL_STREAM_2) && (maxSupportedNss >= SPATIAL_STREAM_2))
							{
								//update test plan line with the station location, total Nsts = 4
								testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
								testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = i;
								testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = j;
								testPlanInputLine.inputEntryBitFields.userStream3OrderIdx = j;
								testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_4;
								testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
								
								TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentTestPlanIdx++;

#ifdef TRAINING_MANAGER_DEBUG
								debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 22XX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}	

static void TrainingManager_BuildTestPlanForSegment_21xx(trainingManagerPools_e poolId)
{

	uint8 startIndex, stopIndex, i, j;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG


	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Bulding Test Plan for Segment 21xx");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if station in place i supports 2 nss
							maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
							if (maxSupportedNss >= SPATIAL_STREAM_2)
							{
								//update test plan line with the station location, total Nsts = 4
								testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
								testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = i;
								testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = j;
								testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_3;
								testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
								
								TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
								debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 21XX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}	


static void TrainingManager_BuildTestPlanForSegment_12xx(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Bulding Test Plan for Segment 12xx");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if station in place j supports 2 nss
							maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[j]); 
							if (maxSupportedNss >= SPATIAL_STREAM_2)
							{
								//update test plan line with the station location, total Nsts = 4
								testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
								testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
								testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = j;
								testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_3;
								testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
								
								TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
								debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 12XX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}	
#endif // TRAINING_WAVE600_Z0

static void TrainingManager_BuildTestPlanForSegment_11xx(trainingManagerPools_e poolId)
{

	uint8 startIndex, stopIndex, i, j;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_2)
	{
		ILOG2_V("Bulding Test Plan for Segment 11xx");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{						
							//update test plan line with the station location, total Nsts = 4
							
#ifdef TRAINING_WAVE600_Z0

							testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = TrainingManager_get_rand_mcs();
							testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = TrainingManager_get_rand_mcs();
#else
							testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
							testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
#endif
							testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_2;
							testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;		
							
							TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

							SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

							//increment current test plan index
							TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
							debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 11XX segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}	


static void TrainingManager_BuildTestPlanForSegment_111x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Bulding Test Plan for Segment 111x");

		//if at least three stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {	

#ifdef TRAINING_WAVE600_Z0
									testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = TrainingManager_get_rand_mcs();
									testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = TrainingManager_get_rand_mcs();
									testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = TrainingManager_get_rand_mcs();
#else
									//update test plan line with the station location, total Nsts = 3
									testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
									testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
									testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
#endif

									testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_3;
									testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;

									
									TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

									SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

									//increment current test plan index
									TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
									debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
								}
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 111X segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}

#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_BuildTestPlanForSegment_211x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Bulding Test Plan for Segment 211x");

		//if at least three stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	//check if station supports 2 nss
								 	maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
									if (maxSupportedNss >= SPATIAL_STREAM_2)
									{
										//update test plan line with the station location, total Nsts = 4
										testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
										testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = i;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = j;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
										testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_4;
										testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;
										
										TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
										debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
									}
								}
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 211X segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}


static void TrainingManager_BuildTestPlanForSegment_121x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Bulding Test Plan for Segment 121x");

		//if at least three stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	//check if station supports 2 nss 	
								 	maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[j]);
									if (maxSupportedNss >= SPATIAL_STREAM_2)
									{
										//update test plan line with the station location, total Nsts = 4
										testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
										testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = j;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
										testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_4;
										testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;
										
										TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
										debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
									}
								}
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 121X segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}


static void TrainingManager_BuildTestPlanForSegment_112x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0; 
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Bulding Test Plan for Segment 112x");

		//if at least three stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	//check if station supports 2 nss
								 	maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[k]);
									if (maxSupportedNss >= SPATIAL_STREAM_2)
									{
										//update test plan line with the station location, total Nsts = 4
										testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
										testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
										testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
										testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_4;
										testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;

										TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
										debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
									}
								}
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 112X segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}
#endif // TRAINING_WAVE600_Z0


static void TrainingManager_BuildTestPlanForSegment_1111(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k, l;
	MuEffectiveRateDbRamEntry_u testPlanInputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
#ifdef TRAINING_MANAGER_DEBUG
	uint8 debugCounter = 0;
#endif //TRAINING_MANAGER_DEBUG

	testPlanInputLine.val = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Bulding Test Plan for Segment 1111");

		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 4)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentTestPlanIdx < MAX_NUM_OF_TEST_PLAN_LINES); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	for (l=k+1; l <= stopIndex; l++)
									{
										if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, l))
										 {


#ifdef TRAINING_WAVE600_Z0
											
									 		//update test plan line with the station location, total Nsts = 4
											testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = TrainingManager_get_rand_mcs();
											testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = TrainingManager_get_rand_mcs();
											testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = TrainingManager_get_rand_mcs();
											testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = TrainingManager_get_rand_mcs();

#else
									 		//update test plan line with the station location, total Nsts = 4
											testPlanInputLine.inputEntryBitFields.userStream0OrderIdx = i;
											testPlanInputLine.inputEntryBitFields.userStream1OrderIdx = j;
											testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = k;
											testPlanInputLine.inputEntryBitFields.userStream2OrderIdx = l;
#endif
											testPlanInputLine.inputEntryBitFields.totalNsts = SPATIAL_STREAM_4;
											testPlanInputLine.inputEntryBitFields.invalidEntry = FALSE;

											TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentTestPlanIdx].val = testPlanInputLine.val;

											SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanInputLine);

											//increment current test plan index
											TrainingManagerDb.dynamicParams.currentTestPlanIdx++;
#ifdef TRAINING_MANAGER_DEBUG
											debugCounter++;
#endif //TRAINING_MANAGER_DEBUG
										}
									}
								}
							}
						}
			   		}
			  	}
			}
		}
#ifdef TRAINING_MANAGER_DEBUG
		ILOG2_D("Num of combinations in 1111 segment is %d", debugCounter);
#endif //TRAINING_MANAGER_DEBUG
	}
}

/*************END of TrainingManager_BuildTestPlanForSegment_[SEGMENT_STRUCTURE] *******************************/




/**********************************************************************************
TrainingManager_UpdateGroupCombinationTableForPolicy[POLICY TYPE]


Description:
------------
	Handlers per policy for processing phy results output lines.
	Each function calls the Segment funcs that corresponds to the policy id
	The function is similar to function of building the test plan for the same policy since we need to restore the test plan 
	input lines to associate with the corresponding outputs

Input: pool ID
-----
	
Returns: none
--------
	 
**********************************************************************************/



static void TrainingManager_UpdateGroupCombinationTableForPolicy1_SingleStaCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 0");
	TrainingManager_UpdateGroupCombinationTableForSegment_1xxx(poolId);
}


static void TrainingManager_UpdateGroupCombinationTableForPolicy2_SingleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
 	ILOG2_V("Update Group Combination table for policy 1");
	TrainingManager_UpdateGroupCombinationTableForSegment_1xxx(poolId);
	TrainingManager_UpdateGroupCombinationTableForSegment_2xxx(poolId);

}

static void TrainingManager_UpdateGroupCombinationTableForPolicy3_DoubleStaCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 2");
#ifndef TRAINING_WAVE600_Z0
	TrainingManager_UpdateGroupCombinationTableForSegment_22xx(poolId);
	TrainingManager_UpdateGroupCombinationTableForSegment_21xx(poolId);
	TrainingManager_UpdateGroupCombinationTableForSegment_12xx(poolId);
#endif
	TrainingManager_UpdateGroupCombinationTableForSegment_11xx(poolId);
}


static void TrainingManager_UpdateGroupCombinationTableForPolicy4_DoubleStaCombinationSegmentTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 3");
	TrainingManager_UpdateGroupCombinationTableForSegment_22xx(poolId);
}


static void TrainingManager_UpdateGroupCombinationTableForPolicy5_IncludeThreeStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 4");
	TrainingManager_UpdateGroupCombinationTableForSegment_111x(poolId);
}


static void TrainingManager_UpdateGroupCombinationTableForPolicy6_IncludeThreeStationCombinationSegmentUpToTwoNsts(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 5");
	TrainingManager_UpdateGroupCombinationTableForSegment_111x(poolId);
#ifndef TRAINING_WAVE600_Z0
	TrainingManager_UpdateGroupCombinationTableForSegment_211x(poolId);
	TrainingManager_UpdateGroupCombinationTableForSegment_121x(poolId);
	TrainingManager_UpdateGroupCombinationTableForSegment_112x(poolId);	
#endif
}


static void TrainingManager_UpdateGroupCombinationTableForPolicy7_FourStationCombinationSegmentSingleNstsOnly(trainingManagerPools_e poolId)
{
	ILOG2_V("Update Group Combination table for policy 6");
	TrainingManager_UpdateGroupCombinationTableForSegment_1111(poolId);
}
/*********END of TrainingManager_UpdateGroupCombinationTableForPolicy[POLICY TYPE]***********/


 /**********************************************************************************
 TrainingManager_UpdateGroupCombinationTableForSegment_[SEGMENT_STRUCTURE]
 
 
 Description:
 ------------
	 Handlers per segment for processing phy results output line and decide whether need to update group combination table or not

	 The function is similar to function of building the test plan for the same segment since we need to restore the test plan 
	 input lines to associate with the corresponding outputs
 
 Input: pool ID
 -----
	 
 Returns: none
 --------
	  
 **********************************************************************************/



static void TrainingManager_UpdateGroupCombinationTableForSegment_1xxx(trainingManagerPools_e poolId)
{
	uint8 i;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];	

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_1)
	{
		ILOG2_V("Update Group Combination table for Segment 1xxx, output lines:");

		//while we have more stations in pool and there are still valid test plan output lines
		while((currentLocationBitmap!=0) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx))
		{
			//find first lsb set - this is the location station from current pool in the training vector
			i = Utils_FindLastSetAndClear(&currentLocationBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentLocationBitmap)));

			testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

			SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

			//check if stream 0 includes valid output and update group combination table only for valid outputs
			if (testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE)
			{
				TrainingManagerDb.dynamicParams.groupCombinationTable[i].singleNstsPhyRate = MIN(testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate, TrainingManagerDb.dynamicParams.groupCombinationTable[i].singleNstsPhyRate);

			}

			//increment current test plan index
			TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
		}
	}
}


static void TrainingManager_UpdateGroupCombinationTableForSegment_2xxx(trainingManagerPools_e poolId)
{
	uint8 i;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];
	uint8 maxSupportedNss = 0;

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_2)
	{
		ILOG2_V("Update Group Combination table for Segment 2xxx, output lines:");

		//while we have more stations in pool and there are still valid test plan output lines
		while((currentLocationBitmap!=0) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx))
		{
			//find first lsb set - this is the location station from current pool in the training vector
			i = Utils_FindLastSetAndClear(&currentLocationBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(currentLocationBitmap)));

			//check if the station supports 2 nss
			maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
			if (maxSupportedNss >= SPATIAL_STREAM_2)
			{
				testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

				SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);
				
				//check if streams 0 and 1 includes valid output and update group combination table only for valid outputs
				if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
				{
					//take Minimum(minimum(userStream0EffectiveRate, group combination table doubleNstsPhyRate), userStream1EffectiveRate)
					TrainingManagerDb.dynamicParams.groupCombinationTable[i].doubleNstsPhyRate = MIN(testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate, TrainingManagerDb.dynamicParams.groupCombinationTable[i].doubleNstsPhyRate);
					TrainingManagerDb.dynamicParams.groupCombinationTable[i].doubleNstsPhyRate = MIN(testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate, TrainingManagerDb.dynamicParams.groupCombinationTable[i].doubleNstsPhyRate);
				}

				//increment current test plan index
				TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
			}
		}
	}
}


static void TrainingManager_UpdateGroupCombinationTableForSegment_22xx(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j;
	uint8 maxSupportedNss_i,maxSupportedNss_j; 
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Update Group Combination table for Segment 22xx, output lines:");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if both stations support 2 nss
							maxSupportedNss_i = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
							maxSupportedNss_j = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[j]);

							if ((maxSupportedNss_i >= SPATIAL_STREAM_2) && (maxSupportedNss_j >= SPATIAL_STREAM_2))
							{
								testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

								//check if all streams includes valid outputs and update group combination table only for valid outputs
								if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream3EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
								{
									currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_22XX, NUM_OF_STATIONS_SEGMENT_22XX);
									if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
									{
										StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
										StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
										StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];

										groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
										groupCombinationSIDs[SPATIAL_STREAM_2] = staId1;
										groupCombinationSIDs[SPATIAL_STREAM_3] = staId2;
										groupCombinationSIDs[SPATIAL_STREAM_4] = staId2;

										TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
										TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
									}								
								}						

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
							}
						}
			   		}
			  	}
			}
		}
	}
}

#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_UpdateGroupCombinationTableForSegment_21xx(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j;
	uint8 maxSupportedNss = 0; 
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Update Group Combination table for Segment 21xx, output lines:");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if station in place i supports 2 nss
							maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
							if (maxSupportedNss >= SPATIAL_STREAM_2) 
							{
								testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

								//check if all streams includes valid outputs and update group combination table only for valid outputs
								if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
								{
									currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_21XX, NUM_OF_STATIONS_SEGMENT_21XX);
									if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
									{
										StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
										StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
										StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];

										groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
										groupCombinationSIDs[SPATIAL_STREAM_2] = staId1;
										groupCombinationSIDs[SPATIAL_STREAM_3] = staId2;
										groupCombinationSIDs[SPATIAL_STREAM_4] = INVALID_STA_INDEX;

										TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
										TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
									}								
								}						

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
							}
						}
			   		}
			  	}
			}
		}
	}
}

static void TrainingManager_UpdateGroupCombinationTableForSegment_12xx(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j;
	uint8 maxSupportedNss = 0;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Update Group Combination table for Segment 12xx, output lines:");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							//check if station in place j supports 2 nss
							maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[j]);
							if (maxSupportedNss >= SPATIAL_STREAM_2) 
							{
								testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

								SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

								//check if all streams includes valid outputs and update group combination table only for valid outputs
								if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
								{
									currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_12XX, NUM_OF_STATIONS_SEGMENT_12XX);
									if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
									{
										StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
										StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
										StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];

										groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
										groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
										groupCombinationSIDs[SPATIAL_STREAM_3] = staId2;
										groupCombinationSIDs[SPATIAL_STREAM_4] = INVALID_STA_INDEX;

										TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
										TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
									}								
								}						

								//increment current test plan index
								TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
							}
						}
			   		}
			  	}
			}
		}
	}
}
#endif // TRAINING_WAVE600_Z0


static void TrainingManager_UpdateGroupCombinationTableForSegment_11xx(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_2)
	{
		ILOG2_V("Update Group Combination table for Segment 11xx, output lines:");

		//if at least two stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 2)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{						
							testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

							SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

							//check if all streams includes valid outputs and update group combination table only for valid outputs
							if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
							{
								currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_11XX, NUM_OF_STATIONS_SEGMENT_11XX);
								if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
								{
									StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
									StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
									StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];

									groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
									groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
									groupCombinationSIDs[SPATIAL_STREAM_3] = INVALID_STA_INDEX;
									groupCombinationSIDs[SPATIAL_STREAM_4] = INVALID_STA_INDEX;

									TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
									TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
								}								
							}						

							//increment current test plan index
							TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;						
						}
			   		}
			  	}
			}
		}
	}
}

 
static void TrainingManager_UpdateGroupCombinationTableForSegment_111x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_3)
	{
		ILOG2_V("Update Group Combination table for Segment 111x, output lines:");

		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								{							 	
									testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

									SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

									//check if all streams includes valid outputs and update group combination table only for valid outputs
									if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
									{
										currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_111X, NUM_OF_STATIONS_SEGMENT_111X);
										if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
										{
											StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
											StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
											StaId staId3 = TrainingManagerDb.dynamicParams.trainingVector[k];
											StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];
											
											groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
											groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
											groupCombinationSIDs[SPATIAL_STREAM_3] = staId3;
											groupCombinationSIDs[SPATIAL_STREAM_4] = INVALID_STA_INDEX;										
								
											TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
											TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
											TrainingManager_CheckAndUpdateGroupCombinationTable(k, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
										}								
									}						

									//increment current test plan index
									TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
								}
							}
						}
			   		}
			  	}
			}
		}
	}
}

#ifndef TRAINING_WAVE600_Z0
static void TrainingManager_UpdateGroupCombinationTableForSegment_211x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	uint8 maxSupportedNss = 0; 
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Update Group Combination table for Segment 211x, output lines:");

		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	//check if station supports 2 nss
								 	maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[i]);
									if (maxSupportedNss >= SPATIAL_STREAM_2)
									{
										testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

										//check if all streams includes valid outputs and update group combination table only for valid outputs
										if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream3EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
										{
											currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_211X, NUM_OF_STATIONS_SEGMENT_211X);
											if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
											{
												StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
												StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
												StaId staId3 = TrainingManagerDb.dynamicParams.trainingVector[k];
												StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];
											
												groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
												groupCombinationSIDs[SPATIAL_STREAM_2] = staId1;
												groupCombinationSIDs[SPATIAL_STREAM_3] = staId2;
												groupCombinationSIDs[SPATIAL_STREAM_4] = staId3;		
												
												TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
												TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
												TrainingManager_CheckAndUpdateGroupCombinationTable(k, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
											}								
										}						

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
									}
								}
							}
						}
			   		}
			  	}
			}
		}
	}
}


static void TrainingManager_UpdateGroupCombinationTableForSegment_121x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	uint8 maxSupportedNss = 0;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];


	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Update Group Combination table for Segment 121x, output lines:");

		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
									 maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[j]);
								 	//check if station supports 2 nss
									if (maxSupportedNss  >= SPATIAL_STREAM_2)
									{
										testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

										//check if all streams includes valid outputs and update group combination table only for valid outputs
										if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream3EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
										{
											currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_121X, NUM_OF_STATIONS_SEGMENT_121X);
											if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
											{
												StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
												StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
												StaId staId3 = TrainingManagerDb.dynamicParams.trainingVector[k];
												StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];
											
												groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
												groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
												groupCombinationSIDs[SPATIAL_STREAM_3] = staId2;
												groupCombinationSIDs[SPATIAL_STREAM_4] = staId3;							
												
												TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
												TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
												TrainingManager_CheckAndUpdateGroupCombinationTable(k, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
											}								
										}						

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
									}
								}
							}
						}
			   		}
			  	}
			}
		}
	}
}



static void TrainingManager_UpdateGroupCombinationTableForSegment_112x(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k;
	uint8 maxSupportedNss = 0; 
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Update Group Combination table for Segment 112x, output lines:");

		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 3)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	//check if station supports 2 nss
								 	maxSupportedNss = StaDb_getStaSupportedNss(TrainingManagerDb.dynamicParams.trainingVector[k]);
									if (maxSupportedNss >= SPATIAL_STREAM_2)
									{
										testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

										SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

										//check if all streams includes valid outputs and update group combination table only for valid outputs
										if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream3EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
										{
											currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_112X, NUM_OF_STATIONS_SEGMENT_112X);
											if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
											{
												StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
												StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
												StaId staId3 = TrainingManagerDb.dynamicParams.trainingVector[k];
												StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];
											
												groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
												groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
												groupCombinationSIDs[SPATIAL_STREAM_3] = staId3;
												groupCombinationSIDs[SPATIAL_STREAM_4] = staId3;
											
												TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
												TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
												TrainingManager_CheckAndUpdateGroupCombinationTable(k, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);	
											}
										}						

										//increment current test plan index
										TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
									}
								}
							}
						}
			   		}
			  	}
			}
		}
	}
}
#endif


static void TrainingManager_UpdateGroupCombinationTableForSegment_1111(trainingManagerPools_e poolId)
{
	uint8 startIndex, stopIndex, i, j, k, l;
	MuEffectiveRateDbRamEntry_u testPlanOutputLine;
	uint32 currentGroupMark = 0;
	uint32 currentLocationBitmap = TrainingManagerDb.dynamicParams.locationBitmap[poolId];

	if (TrainingManagerDb.dynamicParams.numOfAntennas > SPATIAL_STREAM_4)
	{
		ILOG2_V("Update Group Combination table for Segment 1111, output lines:");

		//if at least four stations from the pool is participating in current training
		if (TrainingManagerDb.dynamicParams.numberOfStasInTrainingFromPool[poolId] >= 4)
		{
			//first set lsb
			startIndex = Utils_CountTrailingZeros(currentLocationBitmap);
			
			//last set lsb (first set msb)
			stopIndex = Utils_FindFirstSet(currentLocationBitmap);
			
			for (i=startIndex; (i <= stopIndex) && (TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx < TrainingManagerDb.dynamicParams.currentTestPlanIdx); i++)
			{
				if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, i))
				{
					for (j=i+1; j <= stopIndex; j++)
			  		{
			   			if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, j))
						{
							for (k=j+1; k <= stopIndex; k++)
							{
								 if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, k))
								 {
								 	for (l=k+1; l <= stopIndex; l++)
									{
										if (GET_BIT_IN_SINGLE_BITMAP(currentLocationBitmap, l))
										 {
									 		testPlanOutputLine.val = TrainingManagerTestPlanRam->muEffectiveRateDbEntry[TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx].val;

											SLOG2(0, 0, MuEffectiveRateDbRamEntry_u, &testPlanOutputLine);

											//check if all streams includes valid outputs and update group combination table only for valid outputs
											if ((testPlanOutputLine.outputEntryBitFields.userStream0EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream1EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream2EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE) && (testPlanOutputLine.outputEntryBitFields.userStream3EffectiveRate != MU_EFFECTIVE_RATE_DB_INVALID_OUTPUT_VALUE))
											{
												currentGroupMark = TrainingManager_CalculateGroupMark(testPlanOutputLine, TOTAL_NUM_OF_NSTS_SEGMENT_1111, NUM_OF_STATIONS_SEGMENT_1111);
												if (currentGroupMark >= TrainingManagerDb.staticParams.minimalPhyValidationGroupMark)
												{
													StaId staId1 = TrainingManagerDb.dynamicParams.trainingVector[i];
													StaId staId2 = TrainingManagerDb.dynamicParams.trainingVector[j];
													StaId staId3 = TrainingManagerDb.dynamicParams.trainingVector[k];
													StaId staId4 = TrainingManagerDb.dynamicParams.trainingVector[l];
													StaId groupCombinationSIDs[SPATIAL_STREAM_NUM];
											
													groupCombinationSIDs[SPATIAL_STREAM_1] = staId1;
													groupCombinationSIDs[SPATIAL_STREAM_2] = staId2;
													groupCombinationSIDs[SPATIAL_STREAM_3] = staId3;
													groupCombinationSIDs[SPATIAL_STREAM_4] = staId4;												
											
													TrainingManager_CheckAndUpdateGroupCombinationTable(i, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);
													TrainingManager_CheckAndUpdateGroupCombinationTable(j, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
													TrainingManager_CheckAndUpdateGroupCombinationTable(k, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
													TrainingManager_CheckAndUpdateGroupCombinationTable(l, testPlanOutputLine, currentGroupMark, groupCombinationSIDs);									
												}								
											}						

											//increment current test plan index
											TrainingManagerDb.dynamicParams.currentRestoredTestPlanIdx++;
										}
									}
								}
							}
						}
			   		}
			  	}
			}
		}
	}
}


/*********END of  TrainingManager_UpdateGroupCombinationTableForSegment_[SEGMENT_STRUCTURE] handlers***************/



/**********************************************************************************
TrainingManager_InitializeGroupCombinationTable


Description:
------------
	Initialize Group cmbination table with invalid values. 
	This function is invoked before each result processing start since 
	the table is filled out of order and we can't know at the end which entries are invalid

Input: none
-----
	
Returns: none
--------
	 
**********************************************************************************/

static void TrainingManager_InitializeGroupCombinationTable(void)
{
	GroupCombinationTableEntry_t initialGroupCombinationTableEntry;
	uint8 index;

	initialGroupCombinationTableEntry.staId = INVALID_STA_INDEX;
#ifdef ENET_INC_ARCH_WAVE600
	for (index = 0; index < SPATIAL_STREAM_NUM; index++)
	{
		initialGroupCombinationTableEntry.groupCombinationSIDs[index] = INVALID_STA_INDEX;
	}
#else
	memset(initialGroupCombinationTableEntry.groupCombinationSIDs, INVALID_STA_INDEX, SPATIAL_STREAM_NUM);
#endif //ENET_INC_ARCH_WAVE600
	memset(initialGroupCombinationTableEntry.groupCombinationPhyRates, INVALID_STA_INDEX, SPATIAL_STREAM_NUM);
	initialGroupCombinationTableEntry.groupMark = 0;
	initialGroupCombinationTableEntry.singleNstsPhyRate = GROUP_MANAGER_INVALID_RATE_VALUE;
	initialGroupCombinationTableEntry.doubleNstsPhyRate = GROUP_MANAGER_INVALID_RATE_VALUE;

	for(index = 0; index < GROUP_MANAGER_MAX_NUM_OF_STA_FOR_MU_TRAINING; index++)
	{
		TrainingManagerDb.dynamicParams.groupCombinationTable[index] = initialGroupCombinationTableEntry;
	}	
}


/**********************************************************************************
TrainingManager_CalculateGroupMark


Description:
------------
	Group Mark calculation according to the output from the phy.

	Group Mark = Total_TP_Weight*TotalTP + Total_Nss_Weight * TotalNss + Total_numOfStations_Weight * TotalNumOfStations

Input:
-----
MuEffectiveRateDbRamEntry_u testPlanOutputEntry - output mcs indexes per station (per spatial stream)
uint8 totalNsts - total NSS that was requested
uint8 numOfStations - num of stations in current group


Returns: calculated Group Mark
--------
	 
**********************************************************************************/

static uint32 TrainingManager_CalculateGroupMark(MuEffectiveRateDbRamEntry_u testPlanOutputEntry, uint8 totalNsts, uint8 numOfStations)
{
	uint32 groupMark = 0;
	uint16 totalPhyThroughput = 0;
	uint8 outputMcsPerNssArray[SPATIAL_STREAM_NUM];
	uint8 nssIndex;

	//set array with output entry values
	outputMcsPerNssArray[SPATIAL_STREAM_1] = testPlanOutputEntry.outputEntryBitFields.userStream0EffectiveRate;
	outputMcsPerNssArray[SPATIAL_STREAM_2] = testPlanOutputEntry.outputEntryBitFields.userStream1EffectiveRate;
	outputMcsPerNssArray[SPATIAL_STREAM_3] = testPlanOutputEntry.outputEntryBitFields.userStream2EffectiveRate;
	outputMcsPerNssArray[SPATIAL_STREAM_4] = testPlanOutputEntry.outputEntryBitFields.userStream3EffectiveRate;

	for(nssIndex= 0 ; nssIndex < totalNsts; nssIndex++)
	{
	
		totalPhyThroughput += phyRate80MhzScp1Nss[outputMcsPerNssArray[nssIndex]];
	}

	groupMark = ((totalPhyThroughput * TrainingManagerDb.staticParams.groupMarkCalculationWeights.phyThroughputWeight) 	+ \
				(numOfStations * TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfStationsWeight)		+ \
				(totalNsts * TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfNstsWeight));

	ILOG2_D("Calculated group mark = %d", groupMark);				

	return groupMark;
}



/**********************************************************************************
TrainingManager_CheckAndUpdateGroupCombinationTable



Description:
------------
	Update Group combination table on specified index with the parameters of the Group
	This function is invoked if the Group passed group mark minimal threshold
	The Group will be updated in the table only if it is better group than group that already exists in current entry

Input:
-----
uint8 tableIndex - table index is representing station in the same order that is in training vector
MuEffectiveRateDbRamEntry_u testPlanOutputEntry - output mcs indexes per station (per spatial stream)
uint16 currentGroupMark - Group mark of current group
uint8 groupCombinationSids[SPATIAL_STREAM_NUM] - Group SIDs per NSS (maybe same SID for some NSS's)


Returns: none
--------
	 
**********************************************************************************/
static void TrainingManager_CheckAndUpdateGroupCombinationTable(uint8 tableIndex, MuEffectiveRateDbRamEntry_u testPlanOutputEntry, uint32 currentGroupMark, StaId groupCombinationSids[SPATIAL_STREAM_NUM])
{
	if (currentGroupMark > TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupMark)
	{
		/*set all parameters for current table entry*/

		//set the best group combination for current station
		MEMCPY(TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationSIDs, groupCombinationSids, sizeof(TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationSIDs));
		
		//set the rates for current combination according to phy results
		TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationPhyRates[SPATIAL_STREAM_1] = testPlanOutputEntry.outputEntryBitFields.userStream0EffectiveRate;
		TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationPhyRates[SPATIAL_STREAM_2] = testPlanOutputEntry.outputEntryBitFields.userStream1EffectiveRate;
		TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationPhyRates[SPATIAL_STREAM_3] = testPlanOutputEntry.outputEntryBitFields.userStream2EffectiveRate;
		TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupCombinationPhyRates[SPATIAL_STREAM_4] = testPlanOutputEntry.outputEntryBitFields.userStream3EffectiveRate;

		//set group mark calculated for the current combination
		TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex].groupMark = currentGroupMark;		

		ILOG2_D("Group combination table for table index %d is updated!!!", tableIndex);
		SLOG2(0, 0, GroupCombinationTableEntry_t, &(TrainingManagerDb.dynamicParams.groupCombinationTable[tableIndex]));
	}
}




/**********************************************************************************
TrainingManager_StaBitmapBitwiseAND



Description:
------------
	Bitwise AND between STA bitmaps:

	ResultBitmap = bitmap1 & bitmap2
	
Input:
-----

result bitmap - pointer to bitmap which will hold the result
bitmap1 - pointer to first input bitmap
bitmap2 - pointer to the second input bitmap

Returns: none
--------
	 
**********************************************************************************/

static inline void TrainingManager_StaBitmapBitwiseAND(TrainingManagerStaBitmap_t* resultBitmap, TrainingManagerStaBitmap_t* bitmap1, TrainingManagerStaBitmap_t* bitmap2) 
{
	uint8 index;
	
	for(index = 0; index < GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS; index++)	
	{																				
		resultBitmap->staBitmap[index] = bitmap1->staBitmap[index] & bitmap2->staBitmap[index];
	}	
}




/**********************************************************************************
TrainingManager_StaBitmapBitwiseAND



Description:
------------
	Bitwise NOT for STA bitmap:

	ResultBitmap ~originalBitmap
	
Input:
-----

result bitmap - pointer to bitmap which will holds the result
originalBitmap - pointer to the original bitmap

Returns: none
--------
	 
**********************************************************************************/

static inline void TrainingManager_StaBitmapBitwiseNOT(TrainingManagerStaBitmap_t* resultBitmap, TrainingManagerStaBitmap_t* originalBitmap)
{
	uint8 index;
	
	for(index = 0; index < GROUP_MANAGER_SIZE_OF_STATION_BITMAP_IN_WORDS; index++)
	{																				
		resultBitmap->staBitmap[index] = ~(originalBitmap->staBitmap[index]);										
	}	
}



#ifdef TRAINING_FIRST_PHASE_POOL

/**********************************************************************************
TrainingManagerUpdateMinimalMcsCriteria



Description:
------------
	Update Training minimal mcs criteria for specified pool Id in Training Pool DB

	
Input: poolId, mcs
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerUpdateMinimalMcsCriteria(trainingManagerPools_e poolId, mcs_e mcs)
{
	uint8 i;
	TrainingManagerPoolDb.staticParams[poolId].poolFeatureBitmap.bitFields.mcs = 0;
	
	for(i = mcs; i < NUM_OF_MCS; i++)
	{
		TrainingManagerPoolDb.staticParams[poolId].poolFeatureBitmap.bitFields.mcs |= (1 << i);
	}
}


/**********************************************************************************
TrainingManagerUpdateMaximalNssCriteria




Description:
------------
	Update Training maximal nss criteria for specified pool Id in Training Pool DB

	
Input: poolId, mcs
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerUpdateMaximalNssCriteria(trainingManagerPools_e poolId, SpatialStreamNum_e nss)
{
	int8 i;
	TrainingManagerPoolDb.staticParams[poolId].poolFeatureBitmap.bitFields.nss = 0;
	
	for(i = nss; i >= SPATIAL_STREAM_1; i--)
	{
		TrainingManagerPoolDb.staticParams[poolId].poolFeatureBitmap.bitFields.nss |= (1 << i);
	}
}
#endif //TRAINING_FIRST_PHASE_POOL

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************
TrainingManagerStartTrainingReq


Description:
------------
	Start Training request from Group Manager API

Input: none
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerStartTrainingReq(void)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_START_TRAINING_REQ, NULL);
}


/**********************************************************************************
TrainingManagerStopTrainingReq


Description:
------------
	Stop Training request from Group Manager API

Input: none
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerStopTrainingReq(void)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_STOP_TRAINING_REQ, NULL);
}


/**********************************************************************************
TrainingManagerAddStaReq



Description:
------------
	Add Station request from Group Manager API 
	Group manager registers to ADD_STA, handle it and transfer to training only stations which support MU

Input: UMI_STA_ADD* pStaAdd
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerAddStaReq(UMI_STA_ADD* pStaAdd)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_ADD_STA_REQ, pStaAdd);	
}


/**********************************************************************************
TrainingManagerRemoveStaReq



Description:
------------
	Remove Station request from Group Manager API 
	Group manager registers to REMOVE_STA, and transfer to Training - if the station is part of ongoing training, the training will be stopped

Input: STA ID
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerRemoveStaReq(TrainingManagerRemoveStaParams_t* removeStaParams)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_REMOVE_STA_REQ, removeStaParams);	
}


/**********************************************************************************
TrainingManagerHaltStaReq



Description:
------------
	Halt Station request from Group Manager API 
	Request to stop current training if the station is part of it but 
	DO NOT remove it from active station bitmap since it is in HALT and can participate in future trainings (according to halt station bitmap in Group Manager)
	

Input: STA ID
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerHaltStaReq(TrainingManagerHaltStaParams_t *haltStaParams)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_HALT_STA_REQ, haltStaParams);	
}


/**********************************************************************************
TrainingManagerStopStaReq



Description:
------------
	Stop Station request from Group Manager API 
	Request to stop current training if the station is part of it but 
	DO NOT remove it from active station bitmap and DO NOT send confirmation on it.

	This can happen in case of BW change or NSS change of the station.
	

Input: STA ID
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerStopStaReq(TrainingManagerStopStaParams_t *stopStaParams)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_STOP_STA_REQ, stopStaParams);	
}


/**********************************************************************************
TrainingManagerAddVapReq



Description:
------------
	Add VAP request from VAP Manager API 
	Recieved as a result of registration to ADD_VAP event

Input: UMI_VAP_ADD* pStaAdd
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerAddVapReq(K_MSG* trainingManagerMessage)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_ADD_VAP_REQ, trainingManagerMessage);	
}



/**********************************************************************************
TrainingManagerRemoveVapReq





Description:
------------
	Remove VAP request from VAP Manager API 
	Recieved as a result of registration to REMOVE_VAP event

Input: UMI_REMOVE_ADD* pStaAdd
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerRemoveVapReq(K_MSG* trainingManagerMessage)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_REMOVE_VAP_REQ, trainingManagerMessage);	
}



/**********************************************************************************
TrainingManagerNdpaCfm



Description:
------------
	NDPA Manager confirmation on Start Tx request

Input: Tx status
-----	

Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerNdpaCfm(TrainingManagerNdpaCfm_t *ndpaCfmParams)
{
	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_NDPA_CFM, ndpaCfmParams);	
}


/**********************************************************************************
TrainingManagerLaInfoCfm



Description:
------------
	Link Adaptive confirmation on params request from Training
	
Input: Active Stations bitmap for training
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerLaInfoCfm(K_MSG *trainingManagerMessage)
{
	TrainingManagerLinkAdaptiveInfoCfm_t*	linkadaptiveInfo = (TrainingManagerLinkAdaptiveInfoCfm_t *)pK_MSG_DATA(trainingManagerMessage);

	// Run the state machine
	TrainingManager_RunStateMachine(TRAINING_MANAGER_EVENT_LA_INFO_REQ_CONFIRMED, linkadaptiveInfo);	
}



/**********************************************************************************

TrainingManagerGetStatisticsAddress



Description:
------------
Allocate memory address for statistics


Input: 
-----	
	
Returns:
--------
	address of statistics 
	
**********************************************************************************/

uint32* TrainingManagerGetStatisticsAddress(void)
{
	return ((uint32*)&TrainingStatistics);
}




/********************************CLI APIs***********************************************/


#ifdef TRAINING_DEBUG_CLI


/**********************************************************************************
TrainingManagerSetGroupMarkWeights



Description:
------------
	change minimal group mark weights from CLI
	
Input: phyThroughputWeight, numOfStationsWeight, numOfNstsWeight
-----	
	
Returns: none
--------
	 
**********************************************************************************/

void TrainingManagerSetGroupMarkWeights(uint8 phyThroughputWeight, uint8 numOfStationsWeight, uint8 numOfNstsWeight)
{
	if(TrainingManagerDb.staticParams.state == TRAINING_MANAGER_STATE_IDLE)
	{
		TrainingManagerDb.staticParams.groupMarkCalculationWeights.phyThroughputWeight = phyThroughputWeight;
		TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfStationsWeight= numOfStationsWeight;
		TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfNstsWeight= numOfNstsWeight;
	}
	else
	{		
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.groupMarkCalculationWeights.phyThroughputWeight = phyThroughputWeight;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.groupMarkCalculationWeights.numOfStationsWeight= numOfStationsWeight;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.groupMarkCalculationWeights.numOfNstsWeight= numOfNstsWeight;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewWeightsConfiguration = TRUE;
	}
}



/**********************************************************************************
TrainingManagerSetMinimalGroupMark



Description:
------------
	change Training minimal group mark for sending group candidates to Group Manager
	
Input: newMinimalGroupMark
-----	
	
Returns: none
--------
	 
**********************************************************************************/
void TrainingManagerSetMinimalGroupMark(uint16 newMinimalGroupMark)
{
	if(TrainingManagerDb.staticParams.state == TRAINING_MANAGER_STATE_IDLE)
	{
		TrainingManagerDb.staticParams.minimalPhyValidationGroupMark = newMinimalGroupMark;
	}
	else
	{		
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.minimalPhyValidationGroupMark = newMinimalGroupMark;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewGroupMarkConfiguration = TRUE;
	}
}

#endif //TRAINING_DEBUG_CLI



#ifdef TRAINING_FIRST_PHASE_POOL

/**********************************************************************************
TrainingManagerSetMinimalMcsForFirstPhasePool



Description:
------------
	change Training minimal MCS criteria for FIRST_PHASE_POOL_ID
	
Input: minimalMcs
-----	
	
Returns: none
--------
	 
**********************************************************************************/
void TrainingManagerSetMinimalMcsForFirstPhasePool(mcs_e minimalMcs)
{
	if(TrainingManagerDb.staticParams.state == TRAINING_MANAGER_STATE_IDLE)
	{
		TrainingManagerUpdateMinimalMcsCriteria(FIRST_PHASE_POOL_ID, minimalMcs);
	}
	else
	{		
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.minimalMcsForFirstPhasePool= minimalMcs;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMinimalMcsConfiguration= TRUE;
	}
}



/**********************************************************************************
TrainingManagerSetMaximalNssForFirstPhasePool



Description:
------------
	change Training maximal NSS criteria for FIRST_PHASE_POOL_ID
	
Input: maximalNss
-----	
	
Returns: none
--------
	 
**********************************************************************************/
void TrainingManagerSetMaximalNssForFirstPhasePool(SpatialStreamNum_e maximalNss)
{
	if(TrainingManagerDb.staticParams.state == TRAINING_MANAGER_STATE_IDLE)
	{
		TrainingManagerUpdateMaximalNssCriteria(FIRST_PHASE_POOL_ID, maximalNss);
	}
	else
	{		
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.maximalNssForFirstPhasePool= maximalNss;
		TrainingManagerDb.staticParams.debugWaitingForUpdateParams.isNewMaximalNssConfiguration= TRUE;
	}
}


#endif //TRAINING_FIRST_PHASE_POOL
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

/*********************************************************************************************************/


/**********************************************************************************
TrainingManager_Init

Description:
------------
	Initialize Training Manager DB and Pool DB params.

Input: none
-----	

Returns: none
--------
	 
**********************************************************************************/
void TrainingManager_Init(void)
{
#ifdef TRAINING_FIRST_PHASE_POOL
	uint32 maxNumAnts;
#endif
	
	//zero all parameters first
	memset(&TrainingManagerDb, 0, sizeof(TrainingManagerDb_t));

	
	//initialize static params
	
	TrainingManagerDb.staticParams.groupMarkCalculationWeights.phyThroughputWeight = GROUP_CALCULATION_PHY_THROUGHPUT_WEIGHT;
	TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfStationsWeight = GROUP_CALCULATION_NUM_OF_STAS_WEIGHT;
	TrainingManagerDb.staticParams.groupMarkCalculationWeights.numOfNstsWeight = GROUP_CALCULATION_NUM_OF_NSTS_WEIGHT;	
	TrainingManagerDb.staticParams.minimalPhyValidationGroupMark = MINIMNAL_PHY_VALIDATION_GROUP_MARK;
	TrainingManagerDb.staticParams.selectedVap = TRAINING_MANAGER_INVALID_VAP;

#ifdef TRAINING_FIRST_PHASE_POOL

	/*Set policy priorities per num of antennas */
	maxNumAnts = (MAX_NUM_OF_ANTENNAS - 1);
	switch (maxNumAnts)
	{
	case ANTENNA_3:
		// 4 antennas
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][0] = SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][1] = SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][2] = INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][3] = INCLUDE_THREE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][4] = DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][5] = DOUBLE_STA_COMBINATION_SEGMENT_TWO_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_3][6] = FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
	case ANTENNA_2:
		// 3 antennas
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][0] = SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][1] = SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][2] = DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][3] = DOUBLE_STA_COMBINATION_SEGMENT_TWO_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][4] = INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][5] = INCLUDE_THREE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_2][6] = FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
	case ANTENNA_1:
		// 2 antennas
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][0] = SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][1] = SINGLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][2] = DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][3] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][4] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][5] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_1][6] = INVALID_POLICY;
	case ANTENNA_0:
		// 1 antennas
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][0] = SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][1] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][2] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][3] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][4] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][5] = INVALID_POLICY;
		TrainingManagerDb.staticParams.policyPriorityPerNumOfAntennas[ANTENNA_0][6] = INVALID_POLICY;
		break;
	default:
		FATAL("TrainingManager_Init");
		break;
	}

#endif //TRAINING_FIRST_PHASE_POOL

	TrainingManagerDb.staticParams.state = TRAINING_MANAGER_STATE_IDLE;
	
	//get group manager bitmaps address

	TrainingManagerDb.staticParams.groupManagerTrainingBitmaps = GroupManager_GetTrainingBitmaps();
	

	//*****initialize pools (1 pool for first phase)*****/

	//zero all Pool database first
	memset(&TrainingManagerPoolDb, 0, sizeof(PoolDb_t));

	//configure max number of stations per priority
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].maxNumOfStaPerPriority[GROUP_MANAGER_STATION_PRIORITY_LOW]	 = MAX_NUM_OF_STA_PRIORITY_HIGH;
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].maxNumOfStaPerPriority[GROUP_MANAGER_STATION_PRIORITY_MID] 	 = MAX_NUM_OF_STA_PRIORITY_MID;
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].maxNumOfStaPerPriority[GROUP_MANAGER_STATION_PRIORITY_HIGH]  = MAX_NUM_OF_STA_PRIORITY_LOW;

	//rate params
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolFeatureBitmap.bitFields.nss = ((1 << SPATIAL_STREAM_1) | (1 << SPATIAL_STREAM_2) | (1 << SPATIAL_STREAM_3) | (1 << SPATIAL_STREAM_4));
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolFeatureBitmap.bitFields.mcs = ((1 << MCS_4) | (1 << MCS_5) | (1 << MCS_6) | (1 << MCS_7) | (1 << MCS_8) | (1 << MCS_9));
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolFeatureBitmap.bitFields.bw = (1 << BANDWIDTH_EIGHTY);

	//selectable pools bitmap, including the current pool itself
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].selectablePoolsBitmap = (1 << FIRST_PHASE_POOL_ID);

	//MU STA selection limits
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolMuSTAsLowerLimit = POOL_MU_STAS_LOWER_LIMIT;
	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].poolMuSTAsUpperLimit = POOL_MU_STAS_UPPER_LIMIT;

	//test plan policies

	TrainingManagerPoolDb.staticParams[FIRST_PHASE_POOL_ID].testPlanPoliciesBitmap = ((1 << DOUBLE_STA_COMBINATION_SEGMENT_UP_TO_TWO_NSTS_POLICY) 			| \
																					(1 << SINGLE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY) 			| \
																					(1 << INCLUDE_THREE_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY) 	| \
																					(1 << FOUR_STA_COMBINATION_SEGMENT_SINGLE_NSTS_ONLY_POLICY));

	//pointer to Phy test plan ram
#if defined (ENET_INC_ARCH_WAVE600)
#ifndef TRAINING_WAVE600_Z0
	TrainingManagerTestPlanRam = (MuEffectiveRateDbRam_t *)(B0_PHY_RX_FD_BASE_ADDR + MU_EFFECTIVE_RATE_DB_RAM_ADDR_OFFSET);
#else
	TrainingManagerTestPlanRam = &TrainingManagerTestPlanLocalDb;
#endif 
#else
	TrainingManagerTestPlanRam = (MuEffectiveRateDbRam_t *)(PHY_FD_BASE_ADDR + MU_EFFECTIVE_RATE_DB_RAM_ADDR_OFFSET);
#endif

}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


