/***********************************************************************************
 File:			TsManagerInitiator.h
 Module:		Ts Manager 
 Purpose: 		To handle BA agreements
 Description:	This file contains all definitions and the structures of the Ts manager initiator 
 				which is responsible of handling BA agreements from the initiator side
************************************************************************************/
#ifndef TS_MANAGER_INITIATOR_H_
#define TS_MANAGER_INITIATOR_H_
//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "RxDescriptor.h"
#include "PacketDescriptor.h"
#include "TsManagerCommonDefinitions.h"
#include "TsManager_API.h"
#include "frame.h"
#include "ieee80211.h"
#include "TxSelector_Api.h"
#include "Statistics_Descriptors.h"
#include "StatisticsManager_api.h"
#include "shramTxDesc.h"

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define TS_MANAGER_INITIATOR_FIRST_DIALOG_TOKEN (1)
#define TS_MANAGER_INITIATOR_DEFAULT_NUMBER_OF_CALENDAR_CYCLES (10)
#define TS_MANAGER_INITIATOR_INVALID_REQUEST_ID		((1 << TS_MANAGER_INITIATOR_PD_REQUEST_ID_WIDTH) - 1)

/*TS_MANAGER_INITIATOR_PD_OFFSET_WIDTH must accomodate NUM_OF_TS_MANAGER_DESCRIPTORS * sizeof(TxPd_t)/4*/

#if ((NUM_OF_TS_MANAGER_DESCRIPTORS * 8) > (1<< TS_MANAGER_INITIATOR_PD_OFFSET_WIDTH))
#error "TS_MANAGER_INITIATOR_PD_OFFSET_WIDTH must accomodate NUM_OF_TS_MANAGER_DESCRIPTORS * sizeof(TxPd_t)/4"
#endif

#define TS_MANAGER_INITIATOR_PD_NULL 		  	((1 << TS_MANAGER_INITIATOR_PD_OFFSET_WIDTH) - 1)
#define TS_MANAGER_INITIATOR_PACKET_DISCARDED 	(TS_MANAGER_INITIATOR_PD_NULL - 1)

#define TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT (250)
#define TS_MANAGER_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_MULTIPLY_BY_SHIFT (1)

#define TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS (TS_MANAGER_CONVERT_MILLISECONDS_INTO_250_MILLISECONDS_UNITS(TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT))

//---------------------------------------------------------------------------------
//						Macros						
//---------------------------------------------------------------------------------


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

typedef enum
{
	TS_MANAGER_INITIATOR_TID_STATE_IDLE = 0,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSED,
	TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED,
	TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN,
	TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD,
	TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST,
	TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION,	
	TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT,
	TS_MANAGER_INITIATOR_TID_STATE_OPENED,
	TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR,
	TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK,
	TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSING,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA,
	TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING,
	TS_MANAGER_INITIATOR_TID_STATE_REMOVING,
	TS_MANAGER_INITIATOR_NUM_OF_TID_STATES,
	TS_HANDLER_INITIATOR_MAX_NUM_OF_TID_STATES = 0xFF,	
} TsManagerInitiatorTidState_e;

typedef enum
{
	TS_MANAGER_INITIATOR_TID_EVENT_ADD_STA = 0,
	TS_MANAGER_INITIATOR_TID_EVENT_REMOVE_STA,
	TS_MANAGER_INITIATOR_TID_EVENT_ADDBA_RSP_RECEIVED,
	TS_MANAGER_INITIATOR_TID_EVENT_DELBA_RECEIVED,
	TS_MANAGER_INITIATOR_TID_EVENT_ADDBA_REQ_CONFIRMED,
	TS_MANAGER_INITIATOR_TID_EVENT_DELBA_CONFIRMED,
	TS_MANAGER_INITIATOR_TID_EVENT_BAR_CONFIRMED,
	TS_MANAGER_INITIATOR_TID_EVENT_BAR_STATUS,
	TS_MANAGER_INITIATOR_TID_EVENT_PD_ALLOCATED,
	TS_MANAGER_INITIATOR_TID_EVENT_TIMER,
	TS_MANAGER_INITIATOR_TID_EVENT_DISCARDED_PACKET,
	TS_MANAGER_INITIATOR_TID_EVENT_QUEUE_LOCK,
	TS_MANAGER_INITIATOR_TID_EVENT_CLOSE_TID,
	TS_MANAGER_INITIATOR_TID_EVENT_OPEN_TID,
	TS_MANAGER_INITIATOR_NUM_OF_TID_EVENTS,
	TS_HANDLER_INITIATOR_MAX_NUM_OF_TID_EVENTS = 0xFF,	
} TsManagerInitiatorTidEvent_e;

typedef struct TsManagerInitiatorStationTidParams
{
	/* word 0-2 */
	TsManagerCommonParameters_t commonPart;
}TsManagerInitiatorStationTidParams_t;


typedef struct TsManagerInitiatorVapParams
{
	uint8 amsduSupportPerTidArray[NUM_OF_TID];
	uint8 bufferSizePerTidArray[NUM_OF_TID];
	
	uint16 maxBaAgreementPerVap;
	uint16 openedBaAgreements;

	uint8 maxInactivityTimeout;
	uint8 maxBaAgreementsPerStation;
	uint8 addbaRequestTimeout;
	uint8 deleteBaAgreementOnInactivity;
} TsManagerInitiatorVapParams_t;


typedef struct TsManagerInitiatorGlobalParams
{
	uint16 maxGlobalBaAgreements;
	uint16 numOfGlobalOpenBaAgreements;
} TsManagerInitiatorGlobalParams_t;


//---------------------------------------------------------------------------------
//						Public Function Declaration				
//---------------------------------------------------------------------------------
void TsManagerInitiator_Initialize(void);
void TsManagerInitiator_AddStation(StaId stationIndex, uint8 allowed, uint8 pbac);
void TsManagerInitiator_RemoveStation(StaId stationIndex);
void TsManagerInitiator_AddbaResponseReceived(StaId stationIndex, AddbaResponseFrame_t *addbaResponseFrame);
void TsManagerInitiator_DelbaReceived(StaId stationIndex, uint8 tid);
void TsManagerInitiator_AddbaRequestConfirmation(TxPd_t *packetDescriptor);
void TsManagerInitiator_DelbaConfirmation(StaId stationIndex, uint8 tid);
void TsManagerInitiator_BarConfirmed(TxPd_t *packetDescriptor);
void TsManagerInitiator_BarStatus(TxPd_t *packetDescriptor);
void TsManagerInitiator_PdAllocated(StaId stationIndex, uint8 tid, TxPd_t *packetDescriptor);
void TsManagerInitiator_TimerExpired(void);
void TsManagerInitiator_DiscardedPacket(StaId stationIndex, uint8 tid);
void TsManagerInitiator_QueueIsLocked(StaId stationIndex, uint8 tid, TxSelectorLockStatus_e lockStatus);
void TsManagerInitiator_CloseTids(StaId stationIndex, uint8 tidsBitmap);
void TsManagerInitiator_OpenTids(StaId stationIndex, uint8 tidsBitmap);
void TsManagerInitiator_ConfigureVapParameters(uint8 vapIndex, TsManagerInitiatorVapConfigurationParams_t *vapConfigurationParameters);


#endif /* TS_MANAGER_INITIATOR_H_ */

