/***********************************************************************************
 File:			TsManagerRecipient.c
 Module:		Ts Manager
 Purpose: 		To handle BA agreements
 Description:   This file is the implementation of the Ts manager recipient which is responsible 
 				of handling BA agreement from the recipient side
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_MainApi.h"
#include "TsManagerRecipient.h"
#include "TsManager_API.h"
#include "TsManager.h"
#include "PacketDescriptor.h"
#include "database.h"
#include "HwGlobalDefinitions.h"
#include "ResourceManager_API.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "stringLibApi.h"
#include "mt_sysdefs.h"
#include "RxPp_Api.h"
#include "RxMpduStructure.h"
#include "HwMemoryMap.h"
#include "StaDatabase_Api.h"
#include "Utils_Api.h"
#include "Pac_Api.h"
#include "ShramRxfFifo.h"
#include "EventsManager_api.h"
#include "Statistics_Descriptors.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TS_MANAGER 
#define LOG_LOCAL_FID 2


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef void (*TsManagerRecipientGeneralFunc)(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void tsManagerRecipientIgnoreEvent(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientFatal(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientChangeState(TsManagerRecipientStationTidParams_t *TsDb, TsManagerRecipientTidState_e newState);
static void tsManagerRecipientLogEvent(TsManagerRecipientStationTidParams_t *TsDb, TsManagerRecipientTidEvent_e event);
static bool tsManagerRecipientRmRequest(TsManagerRecipientStationTidParams_t *TsDb, RmPdRequestFillParameters_t *pdRequestFillParameters);
static bool tsManagerRecipientRmRemove(TsManagerRecipientStationTidParams_t *TsDb);
/* Remove functions */
static void tsManagerRecipientRemoveInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInClosingWaitingForPdForDelba (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInClosingWaitingForPdForAddbaResponse(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientRemoveInClosingWaitingForConfirmation(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* ADDBA request received functions */
static void tsManagerRecipientAddbaRequestReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientAddbaRequestReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientAddbaRequestReceivedInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* DELBA received functions */
static void tsManagerRecipientDelbaReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaReceivedInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaReceivedInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* ADDBA response confirmation */
static void tsManagerRecipientAddbaResponseConfirmedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientAddbaResponseConfirmedInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

static void tsManagerRecipientAddbaResponseConfirmedInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientAddbaResponseConfirmedInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientAddbaResponseConfirmedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* DELBA response confirmation */
static void tsManagerRecipientDelbaConfirmedInClosing(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientDelbaConfirmedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* Illgal packet functions */
static void tsManagerRecipientIllegalPacketReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientIllegalPacketReceivedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientIllegalPacketReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* PD allocation functions */
static void tsManagerRecipientPdAllocatedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientPdAllocatedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientPdAllocatedInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientPdAllocatedInClosingWaitingForPdForDelba (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientPdAllocatedInClosingWaitingForPdForAddbaResponse (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientPdAllocatedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* BAR received functions */
static void tsManagerRecipientBarReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInOpened(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInClosing(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientBarReceivedInClosingWaitingForConfirmation(TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* Close TID functions */
static void tsManagerRecipientCloseInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientCloseInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientCloseInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientCloseInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);
static void tsManagerRecipientCloseInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters);

/* Service function */
static void tsManagerRecipientTryToSendDelba (TsManagerRecipientStationTidParams_t *TsDb);
static void tsManagerRecipientCloseBaAgreement (TsManagerRecipientStationTidParams_t *TsDb);
static void tsManagerRecipientRemoveRequestForPdBeforeRemovingTid (TsManagerRecipientStationTidParams_t *TsDb);
static void tsManagerRecipientProcessAddbaRequest (TsManagerRecipientStationTidParams_t *TsDb, FM_PAYLOAD_ADDBA_REQ *addbaRequestPayload);
static void tsManagerRecipientTimerEvent(TsManagerRecipientStationTidParams_t *stationTidEntry);
static void tsManagerRecipientBaAgreementOpenedUpdateCounters(StaId stationIndex);
static void tsManagerRecipientBaAgreementClosedUpdateCounter(StaId stationIndex);
static uint8 tsManagerRecipientIsAllowedToOpenBaAgreement(StaId stationIndex);
static void tsManagerRecipientProcessBar (TsManagerRecipientStationTidParams_t *TsDb, BarPayload_t *barPayload);

/*---------------------------------------------------------------------------------
/						External Variables									
/----------------------------------------------------------------------------------*/

extern TsManagerGlobalDb_t TsManager_GlobalDb;


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_UMAC)
static uint16 *TsManagerRecipientCalendarWheelSlots;
static TsManagerRecipientVapParams_t *TsManagerRecipientVapDatabase;
#else
static uint16 TsManagerRecipientCalendarWheelSlots[TS_MANAGER_CALENDAR_WHEEL_SIZE];
static TsManagerRecipientVapParams_t TsManagerRecipientVapDatabase[HW_NUM_OF_VAPS];
#endif
static CalendarWheelParameters_t TsManagerRecipientCalendarWheel;
static TsManagerRecipientStationTidParams_t TsManagerRecipientStationTidDatabase[HW_NUM_OF_STATIONS][NUM_OF_TID];
static uint8 TsManagerRecipientStationOpenedBaAgreements[HW_NUM_OF_STATIONS];

static TsManagerRecipientGlobalParams_t TsManagerRecipientGlobalParameters;

static const TsManagerRecipientGeneralFunc TsManagerRecipientRemoveTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientFatal 	                             		 , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientRemoveInClosed                                , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientRemoveInClosedWaitingForMessage               , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientRemoveInOpeningWaitingForPd                   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */	 
	 tsManagerRecipientRemoveInOpening                               , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientRemoveInOpened                                , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientRemoveInOpenedSending						 , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientRemoveInOpenedWaitingPd						 , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientRemoveInClosing                               , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientRemoveInClosingWaitingForPdForDelba           , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientRemoveInClosingWaitingForPdForAddbaResponse   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientRemoveInClosingWaitingForConfirmation         , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientFatal                                		 , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */
};
	
static const TsManagerRecipientGeneralFunc TsManagerRecipientAddbaRequestReceivedTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientIgnoreEvent 							  , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientAddbaRequestReceivedInClosed   		  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientIgnoreEvent						      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientIgnoreEvent							  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */	
	 tsManagerRecipientAddbaRequestReceivedInOpening  		  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientAddbaRequestReceivedInOpened	  		  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientIgnoreEvent			  				  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientIgnoreEvent	 	  					  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientIgnoreEvent				              , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientIgnoreEvent	                          , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientIgnoreEvent	                          , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
 	 tsManagerRecipientIgnoreEvent				              , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
 	 tsManagerRecipientIgnoreEvent 							  , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */	
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientDelbaReceivedTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientIgnoreEvent 				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientIgnoreEvent                                   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientIgnoreEvent				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientDelbaReceivedInOpeningWaitingForPd            , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientDelbaReceivedInOpening                        , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientDelbaReceivedInOpened                         , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientDelbaInOpenedSending			  				 , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientDelbaInOpenedWaitingPd		  				 , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientIgnoreEvent				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientIgnoreEvent				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientIgnoreEvent				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientDelbaReceivedInClosingWaitingForConfirmation	 , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientIgnoreEvent				                     , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */ 
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientAddbaResponseConfirmationdTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientFatal 					                       			, /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientFatal				                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientFatal				                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientFatal                                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientAddbaResponseConfirmedInOpening                 			, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientFatal                                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientAddbaResponseConfirmedInOpenedSending 					, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientFatal						 							, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientAddbaResponseConfirmedInClosing                 			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientFatal				                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientFatal				                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientAddbaResponseConfirmedInClosingWaitingForConfirmation	, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientAddbaResponseConfirmedInRemoving			               	, /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */ 	
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientDelbaConfirmationdTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientFatal 					                  , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientFatal				                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientFatal				                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientFatal                                      , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientFatal                                      , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientFatal                                      , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientFatal									  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientFatal									  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientDelbaConfirmedInClosing             		  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientFatal				                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientFatal				                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientFatal                                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientDelbaConfirmedInRemoving			   		  , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */ 			   		
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientIllegalPacketReceivedsTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
   	 tsManagerRecipientIgnoreEvent 					                            , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientIllegalPacketReceivedInClosed				    		, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientIllegalPacketReceivedInClosedWaitingForMessage			, /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientIllegalPacketReceivedInOpeningWaitingForPd       		, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientFatal                                           			, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientFatal                                                    , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientFatal									  				, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientFatal									  				, /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientIgnoreEvent                                              , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientIgnoreEvent				                                , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientIgnoreEvent				                                , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientFatal                                                    , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientIgnoreEvent			                                    , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */ 				
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientPdAllocatedTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
	 tsManagerRecipientFatal 							                  , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientFatal						                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientPdAllocatedInClosedWaitingForMessage               , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientPdAllocatedInOpeningWaitingForPd                   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientFatal 							                  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientFatal 							                  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientFatal									  		  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientPdAllocatedInOpenedWaitingPd				  		  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientFatal						                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientPdAllocatedInClosingWaitingForPdForDelba			  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientPdAllocatedInClosingWaitingForPdForAddbaResponse	  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientFatal 							                  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientPdAllocatedInRemoving			                  , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */  
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientBarReceivedTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
     tsManagerRecipientIgnoreEvent 						           , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientBarReceivedInClosed						   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientBarReceivedInClosedWaitingForMessage 	   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientBarReceivedInOpeningWaitingForPd 	       , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientBarReceivedInOpening 					   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientBarReceivedInOpened 						   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientBarReceivedInOpened						   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientBarReceivedInOpened 						   , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientBarReceivedInClosing						   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba	   , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientIgnoreEvent						           , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientBarReceivedInClosingWaitingForConfirmation  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientIgnoreEvent						           , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */	
};

static const TsManagerRecipientGeneralFunc TsManagerRecipientCloseTidTable[TS_MANAGER_RECIPIENT_NUM_OF_TID_STATES] =
{
     tsManagerRecipientIgnoreEvent 						  , /* TS_MANAGER_RECIPIENT_TID_STATE_IDLE */
	 tsManagerRecipientIgnoreEvent						  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED */
	 tsManagerRecipientIgnoreEvent 	                      , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER */
	 tsManagerRecipientCloseInOpeningWaitingForPd 	      , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD */   
	 tsManagerRecipientCloseInOpening 	                  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENING */
	 tsManagerRecipientCloseInOpened 				      , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED */
	 tsManagerRecipientCloseInOpenedSending				  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING */
	 tsManagerRecipientCloseInOpenedWaitingPd			  , /* TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD */
	 tsManagerRecipientIgnoreEvent						  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING */
	 tsManagerRecipientIgnoreEvent						  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA */
	 tsManagerRecipientIgnoreEvent						  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE */
	 tsManagerRecipientIgnoreEvent 						  , /* TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA */
	 tsManagerRecipientIgnoreEvent						  , /* TS_MANAGER_RECIPIENT_TID_STATE_REMOVING */	
};

/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
TsManagerRecipientTidGlobalStatistics_t TsManagerRecipientTidGlobalStatistics;
TsManagerRecipientStationGlobalStatistics_t TsManagerRecipientStationGlobalStatistics;
	

#if defined (RXPP_TID_DEBUG)
RxPpStaTidRam_t managementTidParametersDebug;
RxPpStaTidRam_t tidsParametersDebug[NUM_OF_TID];
#endif

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

tsManagerRecipientIgnoreEvent 


Description:
------------
Empty Event handler

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientIgnoreEvent(TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
}

/**********************************************************************************

tsManagerRecipientFatal 


Description:
------------
Fatal event handler

Input: 
-----
TsDb
state - new state

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientFatal(TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
#if defined (RXPP_TID_DEBUG)
	RxPpRam_t *RxPpRam;
	uint8 k;
	
	/* Initialize the pointer of the Rx Pp Ram */
	RxPpRam = (RxPpRam_t *)(RX_PP_BASE_ADDRESS + RXPP_RAM_OFFSET_FROM_BASE_ADDRESS);

	memcpy32(&managementTidParametersDebug, &RxPpRam->rxppManagementRam.ManagementStaArr[TsDb->commonPart.stationId].managementTidParameters, sizeof(RxPpStaTidRam_t) >> 2);
	for (k = 0; k < NUM_OF_TID; k++)
	{
		memcpy32(&tidsParametersDebug[k], &RxPpRam->rxppUnicastRam.UnicastStaArr[TsDb->commonPart.stationId].tidsParameters[k], sizeof(RxPpStaTidRam_t) >> 2);
	}
#endif

	FATAL("TS Manager Recipient Fatal");
}

/**********************************************************************************

tsManagerRecipientChangeState 


Description:
------------
Utility function to change state

Input: 
-----
stationIndex - station index
tid - the TID number
new State

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientChangeState(TsManagerRecipientStationTidParams_t *TsDb, TsManagerRecipientTidState_e newState)
{
	ILOG0_DDDD("TS Manager Recipient SID %d, TID %d, Change State from %d to %d", TsDb->commonPart.stationId, TsDb->commonPart.tid, TsDb->commonPart.state, newState);
	TsDb->commonPart.state = newState;
}

/**********************************************************************************

tsManagerRecipientLogEvent 


Description:
------------
Utility function to log event

Input: 
-----
TsDb
event

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientLogEvent(TsManagerRecipientStationTidParams_t *TsDb, TsManagerRecipientTidEvent_e event)
{
	ILOG0_DDDD("TS Manager Recipient Event SID %d, TID %d, State %d Event %d", TsDb->commonPart.stationId, TsDb->commonPart.tid, TsDb->state, event);
}

/**********************************************************************************

tsManagerRecipientRmRequest 


Description:
------------
Utility function to build RM rquest

Input: 
-----
stationIndex - station index
tid - the TID number
pdRequestFillParameters - request structure

		
Output:
-------
allocation status
	

Returns:
--------
	void - 
	
**********************************************************************************/
static bool tsManagerRecipientRmRequest(TsManagerRecipientStationTidParams_t *TsDb, RmPdRequestFillParameters_t *pdRequestFillParameters)
{
	bool allocationStatus = FALSE;

	pdRequestFillParameters->returnMsgType = TS_MANAGER_PACKET_DESCRIPTOR_ALLOCATED;
	pdRequestFillParameters->returnTaskId = TASK_TS_MANAGER;
	pdRequestFillParameters->context = TS_MANAGER_BUILD_RM_REQUEST(TsDb->commonPart.stationId, TsDb->commonPart.tid, TS_MANAGER_RM_REQUEST_RECIPIENT);
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_TS_MANAGER, pdRequestFillParameters);

	return (allocationStatus);
}

/**********************************************************************************

tsManagerRecipientRmRemove 


Description:
------------
Utility function to remove RM rquest

Input: 
-----
TsDb

		
Output:
-------
remove status
	

Returns:
--------
	bool - 
	
**********************************************************************************/
static bool tsManagerRecipientRmRemove(TsManagerRecipientStationTidParams_t *TsDb)
{
	bool removeStatus = TRUE;

	/*Check if RM returned a valid ID*/
	if (TsDb->commonPart.pdRequestId != TS_MANAGER_RECIPIENT_INVALID_REQUEST_ID)
	{
		removeStatus = ResourceManager_RemoveRequest(TsDb->commonPart.pdRequestId, DESC_POOL_TS_MANAGER);
	}

	return (removeStatus);
}


/**********************************************************************************

tsManagerRecipientRemoveInClosed 


Description:
------------
This function handles remove tid when the state is closed. It moves immidiately to Idle
and updtae counters

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerRecipientRemoveInClosedWaitingForMessage 


Description:
------------
This function handles remove TID when the state is 'closed wiating for message from
resource manager'.
Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerRecipientRemoveInOpeningWaitingForPd  


Description:
------------
This function hanldes remove TID when the state is 'opening waiting for PD'.
This function tries to remove the request for PD

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInOpeningWaitingForPd  (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{	
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	//("tsManagerRecipientRemoveInOpeningWaitingForPd station index %d tid %d", stationIndex, tid);
	tsManagerRecipientRemoveRequestForPdBeforeRemovingTid(TsDb);
}


/**********************************************************************************

tsManagerRecipientRemoveInOpening 


Description:
------------
This function hanldes remove TID when the state is opening. This function close the 
BA agreement and moves to removing state.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerRecipientRemoveInOpened 


Description:
------------
This function handles remove TID when the state is opened.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
   
    /* Remove entry from calendar wheel */	
	TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerRecipientRemoveInOpenedSending 


Description:
------------
This function handles remove TID when the state is opened sending

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);

	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
}
 
/**********************************************************************************

tsManagerRecipientRemoveInOpenedWaitingPd 


Description:
------------
This function handles remove TID when the state is opened waiting PD

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
   
	//("tsManagerRecipientRemoveInOpeningWaitingForPd station index %d tid %d", stationIndex, tid);
	tsManagerRecipientRemoveRequestForPdBeforeRemovingTid(TsDb);
}


/**********************************************************************************

tsManagerRecipientRemoveInClosing 


Description:
------------
This function hanldes remove TID when the state is closing. This function close the 
BA agreement and moves to removing state.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerRecipientRemoveInClosingWaitingForPdForDelba 


Description:
------------
This function handles remove TID when the state is 'closing waiting for PD doe DELBA'
This fucntion close the BA agreement and tries to remove the request for PD.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosingWaitingForPdForDelba (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
	tsManagerRecipientRemoveRequestForPdBeforeRemovingTid(TsDb);
}


/**********************************************************************************

tsManagerRecipientRemoveInClosingWaitingForPdForAddbaResponse 


Description:
------------
This function handles remove TID when the state is 'closing waiting for PD for ADDBA request'
This fucntion close the BA agreement and tries to remove the request for PD.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosingWaitingForPdForAddbaResponse (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientRemoveRequestForPdBeforeRemovingTid(TsDb);
}

/**********************************************************************************

tsManagerRecipientRemoveInClosingWaitingForConfirmation 


Description:
------------
This function handles remove TID when the state is 'closing waiting for confirmation to
send DELBA'
This fucntion close the BA agreement and tries to remove the request for PD.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientRemoveInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	RxPp_FlushTid(TsDb->commonPart.stationId, TsDb->commonPart.tid, RXPP_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);
    tsManagerRecipientCloseBaAgreement(TsDb);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
}


/**********************************************************************************

tsManagerRecipientAddbaRequestReceivedInClosed 


Description:
------------
This function handles ADDBA request in closed state.
This function handles ADDBA request when  BA agreement is currently not opened

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - pointer to the ADDBA request payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientAddbaRequestReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientAddbaRequestReceivedInClosed stationIndex %d tid %d ", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientProcessAddbaRequest(TsDb, (FM_PAYLOAD_ADDBA_REQ *)parameters);
}
/**********************************************************************************

tsManagerRecipientAddbaRequestReceivedInOpening 


Description:
------------
This function handles ADDBA request in opening state.

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - pointer to the ADDBA request payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientAddbaRequestReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	FM_PAYLOAD_ADDBA_REQ *addbaRequestPayload = NULL;

	addbaRequestPayload = (FM_PAYLOAD_ADDBA_REQ *)parameters;

	ILOG2_V("tsManagerRecipientAddbaRequestReceivedInOpeningthe s");
	/*if PBAC negotiated with STA need to update SEQ number*/
	if((addbaRequestPayload->u8DialogToken == TsDb->commonPart.dialogToken) &&
	    (TsDb->commonPart.pbac == TRUE))
	{
		uint16 sequenceNumber = 0;

		ILOG2_D("tsManagerRecipientAddbaRequestReceivedInOpeningthe same dialog token %d", addbaRequestPayload->u8DialogToken);
			
		sequenceNumber = addbaRequestPayload->u16BlockAckStartSn.BlockAckStartSn ;
		RxPp_UpdateWindowSn(TsDb->commonPart.stationId, TsDb->commonPart.tid, sequenceNumber, RXPP_NON_BA_CLIENT_DONT_FORCE);
	}
}

/**********************************************************************************

tsManagerRecipientAddbaRequestReceivedInOpened 


Description:
------------
This function handles ADDBA request in opened state.
This function handles ADDBA request when BA agreement is already opened

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - pointer to the ADDBA request payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientAddbaRequestReceivedInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	uint16 timeout = 0;
	FM_PAYLOAD_ADDBA_REQ *addbaRequestPayload = NULL;
	uint8 vapIndex = 0;

	addbaRequestPayload = (FM_PAYLOAD_ADDBA_REQ *)parameters;

	ILOG2_V("tsManagerRecipientAddbaRequestReceivedInOpenedthe s");
	if(addbaRequestPayload->u8DialogToken == TsDb->commonPart.dialogToken)
	{
		ILOG2_D("tsManagerRecipientAddbaRequestReceivedInOpenedthe same dialog token %d", addbaRequestPayload->u8DialogToken);
		StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
		/* BA agreement exist - reset time */
		timeout = addbaRequestPayload->u16BlockAckTimeOut;
		if(0 == timeout)
		{
			/* Initiator wants to disable inactivity mechanism */
			TsDb->commonPart.deleteBaAgreementOnInactivity = FALSE;
			timeout = TsManagerRecipientVapDatabase[vapIndex].maxInactivityTimeout;
		}
		else
		{
			timeout = TS_MANAGER_CONVERT_TIME_UNITS_INTO_250_MILLISECONDS_UINTS(timeout);
			/* Taking the minimum between internal timeout and the initiator timeout */
			timeout = MIN(timeout,TsManagerRecipientVapDatabase[vapIndex].maxInactivityTimeout);
			TsDb->commonPart.deleteBaAgreementOnInactivity = TRUE;
		}
				
		TsDb->commonPart.timeout = timeout;
		
		/* Reschedule entry in the calendar wheel */
		TsManager_RescheduleInCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart, timeout);

		if (TsDb->commonPart.pbac == FALSE)
		{
			uint8 allocationStatus = FALSE;
			RmPdRequestFillParameters_t pdRequestFillParameters;
			
			memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));
			/*if not PBAC send ADDBA RSP*/
			allocationStatus = tsManagerRecipientRmRequest(TsDb, &pdRequestFillParameters);
			if(allocationStatus)
			{
				/* Remove entry from calendar wheel */	
				TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
				/*PD allocated*/
				TsManager_TransmitAddbaResponse(TsDb, pdRequestFillParameters.packetDescriptor);
				tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING);
			}
			else
			{
				TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
				if (pdRequestFillParameters.requestId != INVALID_FREE_PENDING_LIST_ID)
				{
					/* Remove entry from calendar wheel */	
					TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
					tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENED_WAITING_PD);
				}
			}
		}
		else
		{
			/*if PBAC negotiated with STA need to update SEQ number*/
			uint16 sequenceNumber = 0;
			
			sequenceNumber = addbaRequestPayload->u16BlockAckStartSn.BlockAckStartSn ;
			RxPp_UpdateWindowSn(TsDb->commonPart.stationId, TsDb->commonPart.tid, sequenceNumber, RXPP_NON_BA_CLIENT_DONT_FORCE);
		}
		return;
	}

    /* New request to BA agreement - first close BA agreement , flush all. Then receive a desicion
       as if the BA agreement was closed from the begining */
       
	tsManagerRecipientCloseBaAgreement(TsDb);

    /* Update counters since the BA agreement is being closed */
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);

	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
	
	tsManagerRecipientProcessAddbaRequest(TsDb, addbaRequestPayload);
}


/**********************************************************************************

tsManagerRecipientDelbaReceivedInClosingWaitingForConfirmation 


Description:
------------
This function handles DELBA received in the state 'closing - waiting for confirmation'
This function move to closing state without further operations

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaReceivedInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
}


/**********************************************************************************

tsManagerRecipientDelbaReceivedInOpeningWaitingForPd 


Description:
------------
This function handles DELBA received in the state 'opening waiting for PD'. In this 
state the TID waits for PD to open BA agreement


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	bool removeStatus = FALSE;

	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	removeStatus= tsManagerRecipientRmRemove(TsDb);
	
	ILOG2_D("tsManagerRecipientDelbaReceivedInOpeningWaitingForPd remove status %d", removeStatus);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaReceived++;
	
	if(removeStatus)
	{
		/* remove succeeded -continue to allowed */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
	}
	else
	{
		/* remove failed - wait for message */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER);
	}
}


/**********************************************************************************

tsManagerRecipientDelbaReceivedInOpening 


Description:
------------
This function handles DELBA received in the state opening (waiting for ADDBA response 
confirmation).

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_V("tsManagerRecipientDelbaReceivedInOpening ");
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaReceived++;
	
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerRecipientDelbaReceivedInOpened 


Description:
------------
This function handles DELBA received in opened state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaReceivedInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_V("tsManagerRecipientDelbaReceivedInOpened ");
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaReceived++;
	
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerRecipientCloseBaAgreement(TsDb);
	
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
}


/**********************************************************************************

tsManagerRecipientDelbaInOpenedSending 


Description:
------------
This function handles DELBA received in opened sending state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaReceived++;
	
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
}


/**********************************************************************************

tsManagerRecipientDelbaInOpenedWaitingPd 


Description:
------------
This function handles DELBA received in opened waiting PD state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientDelbaInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	bool removeStatus = FALSE;

	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerRecipientCloseBaAgreement(TsDb);
	
	removeStatus= tsManagerRecipientRmRemove(TsDb);
	
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaReceived++;
	
	if(removeStatus)
	{
		/* remove succeeded -continue to allowed */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
	}
	else
	{
		/* remove failed - wait for message */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED_WAITING_FOR_MESSAGE_FROM_RESOURCE_MANAGER);
	}
}


/**********************************************************************************

tsManagerRecipientAddbaResponseConfirmedInOpening 


Description:
------------
This function handles ADDBA response confirmation in opening state -
move to state opened

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientAddbaResponseConfirmedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	/* Add entry to the calendar wheel */
	ILOG2_DD("tsManagerRecipientAddbaResponseConfirmedInOpening station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManager_AddToCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENED);
}


/**********************************************************************************

tsManagerRecipientAddbaResponseConfirmedInOpenedSending 


Description:
------------
This function handles ADDBA response confirmation in opened sending state -
move to state opened

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientAddbaResponseConfirmedInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	/* Add entry to the calendar wheel */
	TsManager_AddToCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENED);
}

/**********************************************************************************

tsManagerRecipientAddbaResponseConfirmedInClosing 


Description:
------------
This function handles ADDBA response confirmed in the state closing

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientAddbaResponseConfirmedInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientAddbaResponseConfirmedInClosing station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientCloseBaAgreement(TsDb);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
}


/**********************************************************************************

tsManagerRecipientAddbaResponseConfirmedInRemoving 


Description:
------------
This function handles ADDBA response confirmation on removing state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientAddbaResponseConfirmedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	//("tsManagerRecipientAddbaResponseConfirmedInRemoving station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerRecipientDelbaConfirmedInClosing 


Description:
------------
This function handles DELBA confirmed in the state closing

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientDelbaConfirmedInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientDelbaConfirmedInClosing station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientCloseBaAgreement(TsDb);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
}


/**********************************************************************************

tsManagerRecipientDelbaConfirmedInRemoving 


Description:
------------
This function handles DELBA confirmation on removing state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientDelbaConfirmedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	//("tsManagerRecipientDelbaConfirmedInRemoving station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerRecipientAddbaResponseConfirmedInClosingWaitingForConfirmation 


Description:
------------
This function handles ADDBA response confirmation in the state 'closing waiting for
confirmation'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientAddbaResponseConfirmedInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_V("tsManagerRecipientAddbaResponseConfirmedInClosingWaitingForConfirmation ");
	/* Try to allocate new PD for DELBA - the confirmed PD is returned by the general TS manager */ 
	tsManagerRecipientTryToSendDelba(TsDb);
}


/**********************************************************************************

tsManagerRecipientIllegalPacketReceivedInClosed 


Description:
------------
This function handles the event that aggregation was received in closed state -
send DELBA

Input: 
-----
stationIndex - station index
tid - the TID number
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientIllegalPacketReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{	
	ILOG2_DD("tsManagerRecipientIllegalPacketReceivedInClosed station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientTryToSendDelba(TsDb);
}



/**********************************************************************************

tsManagerRecipientIllegalPacketReceivedInClosedWaitingForMessage 


Description:
------------
This function handles the event that aggregation was received in the state 'closed -
waiting for message (from the resource manager)'.


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientIllegalPacketReceivedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientIllegalPacketReceivedInClosedWaitingForMessage station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA);
}

/**********************************************************************************

tsManagerRecipientIllegalPacketReceivedInOpeningWaitingForPd 


Description:
------------
This function handles the event that aggregation was received in the state 'opening -
waiting for PD to send ADDBA request'.


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientIllegalPacketReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientIllegalPacketReceivedInOpeningWaitingForPd station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA);
}

/**********************************************************************************

tsManagerRecipientPdAllocatedInClosedWaitingForMessage 


Description:
------------
This function handles PD allocation from resource manager on 'closed - waiting 
for message from resource manager' state

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientPdAllocatedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	TxPd_t *PdPointer = NULL;

	PdPointer = (TxPd_t *)parameters;
	
	ILOG2_DD("tsManagerRecipientPdAllocatedInClosedWaitingForMessage station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	ResourceManager_ReleaseDescriptor(PdPointer, DESC_POOL_TS_MANAGER);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
}

/**********************************************************************************

tsManagerRecipientPdAllocatedInOpeningWaitingForPd 


Description:
------------
This function handles PD allocation from resource manager on the state 'opening - 
waiting for PD to send ADDBA response'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientPdAllocatedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	uint16 winSize = 0;
	
	ILOG2_DD("tsManagerRecipientPdAllocatedInOpeningWaitingForPd station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaResponseWasSentWithStatusSuccess++;
	
	winSize = TsManagerBufferSizeToWindowSize[TsDb->commonPart.bufferSize];
	/*A little ugly but We stored the SSN in the calendar wheel entry to save space*/
	RxPp_OpenTs(TsDb->commonPart.stationId, TsDb->commonPart.tid, winSize, TsDb->commonPart.calendarWheelEntry.nextIndex);
    TsManager_TransmitAddbaResponse(TsDb, (TxPd_t *)parameters);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENING);
}


/**********************************************************************************

tsManagerRecipientPdAllocatedInOpenedWaitingPd 


Description:
------------
This function handles PD allocation from resource manager on the state 'opened waiting PD - 
waiting for PD to send ADDBA response

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientPdAllocatedInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaResponseWasSentWithStatusSuccess++;
	
    TsManager_TransmitAddbaResponse(TsDb, (TxPd_t *)parameters);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENED_SENDING);
}


/**********************************************************************************

tsManagerRecipientPdAllocatedInClosingWaitingForPdForDelba 


Description:
------------
This function handles PD allocation from resource manager in the state 'closing - 
waiting for PD to send DELBA'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientPdAllocatedInClosingWaitingForPdForDelba (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientPdAllocatedInClosingWaitingForPdForDelba station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaWasSent++;
	
    TsManager_TransmitDelba(&TsDb->commonPart, (TxPd_t *)parameters);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
}


/**********************************************************************************

tsManagerRecipientPdAllocatedInClosingWaitingForPdForAddbaResponse 


Description:
------------
This function handles PD allocation from resource manager in the state 'closing - 
waiting for PD to send ADDBA response'


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientPdAllocatedInClosingWaitingForPdForAddbaResponse (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientPdAllocatedInClosingWaitingForPdForAddbaResponse station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaResponseWasSentWithStatusFailure++;
	
    TsManager_TransmitAddbaResponse(TsDb, (TxPd_t *)parameters);
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerRecipientPdAllocatedInRemoving 


Description:
------------
This function handles PD allocation from resource manager in the state removing

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the PD that was allocated from the resource manager

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientPdAllocatedInRemoving (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	TxPd_t *PdPointer = NULL;

	PdPointer = (TxPd_t *)parameters;

	//("tsManagerRecipientPdAllocatedInRemoving station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	ResourceManager_ReleaseDescriptor(PdPointer, DESC_POOL_TS_MANAGER);
	
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerRecipientBarReceivedInClosed 


Description:
------------
This function handles the event that BAR was received in closed state -
send DELBA

Input: 
-----
stationIndex - station index
tid - the TID number
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInClosed (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{	
	ILOG2_DD("tsManagerRecipientBarReceivedInClosed station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientTryToSendDelba(TsDb);
}

/**********************************************************************************

tsManagerRecipientBarReceivedInClosedWaitingForMessage 


Description:
------------
This function handles the event that BAR was received in the state 'closed -
waiting for message (from the resource manager)'.


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInClosedWaitingForMessage (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInClosedWaitingForMessage station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA);
}

/**********************************************************************************

tsManagerRecipientBarReceivedInOpeningWaitingForPd 


Description:
------------
This function handles the event that BAR was received in the state 'opening -
waiting for PD to send ADDBA request'.


Input: 
-----
stationIndex - station index
tid - the TID number
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInOpeningWaitingForPd station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventIllegalAggregationWasReceived++;
	TsDb->commonPart.reason = TS_MANAGER_UNKNOWN_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA);
}


/**********************************************************************************

tsManagerRecipientBarReceivedInOpening 


Description:
------------
This function handles a BAR received in the state opening

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInOpening station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid); 
	tsManagerRecipientProcessBar(TsDb, (BarPayload_t *)parameters);
}

/**********************************************************************************

tsManagerRecipientBarReceivedInOpened 


Description:
------------
This function handles a BAR received in the state opened

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInOpened station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientProcessBar(TsDb, (BarPayload_t *)parameters);
}

/**********************************************************************************

tsManagerRecipientBarReceivedInClosing 


Description:
------------
This function handles a BAR received in the state closing

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInClosing (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInClosing station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientProcessBar(TsDb, (BarPayload_t *)parameters);
}


/**********************************************************************************

tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba 


Description:
------------
This function handles a BAR received in the state 'closing - waiting for PD for DELBA'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientProcessBar(TsDb, (BarPayload_t *)parameters);
}


/**********************************************************************************

tsManagerRecipientBarReceivedInClosingWaitingForConfirmation 


Description:
------------
This function handles a BAR received in the state 'closing - waiting for confirmation'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientBarReceivedInClosingWaitingForConfirmation (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	ILOG2_DD("tsManagerRecipientBarReceivedInClosingWaitingForPdForDelba station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientProcessBar(TsDb, (BarPayload_t *)parameters);

}

/**********************************************************************************

tsManagerRecipientCloseInOpeningWaitingForPd 


Description:
------------
This function handles close TID event on the state 'opening - waiting for PD'

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientCloseInOpeningWaitingForPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	//("tsManagerRecipientCloseInOpeningWaitingForPd station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart..tid);
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsDb->commonPart.status = TS_MANAGER_UNSPECIFIED_STATUS;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE);
}

/**********************************************************************************

tsManagerRecipientCloseInOpening 


Description:
------------
This function handles close TID event on the state opening 

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientCloseInOpening (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	//("tsManagerRecipientCloseInOpening station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA);
}


/**********************************************************************************

tsManagerRecipientCloseInOpened 


Description:
------------
This function handles close TID event on opened state 

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientCloseInOpened (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	//("tsManagerRecipientCloseInOpened station index %d tid % d", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerRecipientCalendarWheel, &TsDb->commonPart);
	
	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	tsManagerRecipientTryToSendDelba(TsDb);
}


/**********************************************************************************

tsManagerRecipientCloseInOpenedSending 


Description:
------------
This function handles close TID event on opened sending state 

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientCloseInOpenedSending (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_FOR_CONFIRMATION_TO_SEND_DELBA);
}


/**********************************************************************************

tsManagerRecipientCloseInOpenedWaitingPd 


Description:
------------
This function handles close TID event on opened waiting PD state 

Input: 
-----
stationIndex - station index
tid - the TID number
parameters - the BAR payload address
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientCloseInOpenedWaitingPd (TsManagerRecipientStationTidParams_t *TsDb, void *parameters)
{
	tsManagerRecipientBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsDb->commonPart.status = TS_MANAGER_UNSPECIFIED_STATUS;
	tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE);
}


/**********************************************************************************

tsManagerRecipientTryToSendDelba 


Description:
------------
This function tries to send DELBA

Input: 
-----
stationIndex - station index
tid - the TID number
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientTryToSendDelba (TsManagerRecipientStationTidParams_t *TsDb)
{
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));
	allocationStatus = tsManagerRecipientRmRequest(TsDb, &pdRequestFillParameters);
	//("tsManagerRecipientTryToSendDelba allocation status %d",allocationStatus);
	if(allocationStatus)
	{
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventDelbaWasSent++;
		TsManager_TransmitDelba(&TsDb->commonPart, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
	}
 	else
 	{
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
			tsManagerRecipientDelbaConfirmedInClosing(TsDb, NULL);
		}
		else
		{
			tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_DELBA);
		}
 	}
}

/**********************************************************************************

tsManagerRecipientCloseBaAgreement 


Description:
------------
This function close the BA agreement

Input: 
-----
stationIndex - station index
tid - the TID number
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientCloseBaAgreement (TsManagerRecipientStationTidParams_t *TsDb)
{
	
	ILOG2_DD("tsManagerRecipientCloseBaAgreement stationIndex %d tid %d ", TsDb->commonPart.stationId, TsDb->commonPart.tid);
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventBaAgreementClosed++;
	RxPp_CloseTs(TsDb->commonPart.stationId, TsDb->commonPart.tid);
	/*Clear activity now*/
	Pac_RxcBaAgreementRecipientGetAndClearActivity(TsDb->commonPart.stationId, TsDb->commonPart.tid);
}

/**********************************************************************************

tsManagerRecipientRemoveRequestForPdBeforeRemovingTid 


Description:
------------
This function tries to remove the request for PD from the resource manager before 
removing the TID

Input: 
-----
stationIndex - station index
tid - the TID number
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientRemoveRequestForPdBeforeRemovingTid (TsManagerRecipientStationTidParams_t *TsDb)
{
    bool removeStatus = FALSE;
	
	removeStatus = tsManagerRecipientRmRemove(TsDb);
	if(removeStatus)
	{
		/* Request was removed successfully - move to Idle */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
		TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
	}
	else
	{
		/* Remove request failed - wait for message */
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_REMOVING);
	}	
}

/**********************************************************************************

tsManagerRecipientProcessAddbaRequest 


Description:
------------
This function processes ADDBA request in order to determine if it is accepted or not

Input: 
-----
stationIndex - station index
tid - the TID number
addbaRequestPayload - pointer to the ADDBA request payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientProcessAddbaRequest (TsManagerRecipientStationTidParams_t *TsDb, FM_PAYLOAD_ADDBA_REQ *addbaRequestPayload)
{
	uint8 allocationStatus = FALSE;
	uint8 dialogToken = 0;
	uint8 isAddbaRequestAccpted = TRUE;
	uint16 bufferSize = 0;
	uint8 statusCode = TS_MANAGER_SUCCESS_STATUS;
	uint16 timeout = 0;
	uint16 sequenceNumber = 0;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	uint8 vapIndex = 0;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));
	
	dialogToken = addbaRequestPayload->u8DialogToken;
	bufferSize = addbaRequestPayload->u16BlockAckParameterSet.BlockAckBufferSize;
		
    if((dialogToken == 0) || (bufferSize > TS_MANAGER_MAX_BUFFER_SIZE) || ((bufferSize < TS_MANAGER_MIN_BUFFER_SIZE) && (bufferSize !=0)))
	{
		statusCode = TS_MANAGER_INVALID_PARAMETERS_STATUS;
		ILOG2_V("tsManagerRecipientProcessAddbaRequest invalid parameters ");
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaRequestWasReceivedWithInvalidParameters++;
    }
	else if(!addbaRequestPayload->u16BlockAckParameterSet.BlockAckPolicy)
	{
		/* Currently delayed block ACK is not supported */
		statusCode = TS_MANAGER_INVALID_PARAMETERS_STATUS;	
		ILOG2_V("tsManagerRecipientProcessAddbaRequest invalid parameters ");
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaRequestWasReceivedWithInvalidParameters++;
	}
	else if(!TsDb->commonPart.allowed)
	{
		/* The TID currently is not allowed to open BA agreement */
		statusCode = TS_MANAGER_UNSPECIFIED_STATUS;
	}
	else if (TsManager_GlobalDb.enableBa == FALSE)
	{
		/* Globally - currently it is not allowed to open BA agreement */
		statusCode = TS_MANAGER_UNSPECIFIED_STATUS;

	}

	if(statusCode == TS_MANAGER_SUCCESS_STATUS)
	{
		/* Check internal restriciton on opening BA agreements */
		isAddbaRequestAccpted = tsManagerRecipientIsAllowedToOpenBaAgreement(TsDb->commonPart.stationId);	
		if (isAddbaRequestAccpted == FALSE)
		{
			statusCode = TS_MANAGER_UNSPECIFIED_STATUS;
		}
	}
	else
	{
		isAddbaRequestAccpted = FALSE;		
	}

    /* Save status  */
	TsDb->commonPart.status = statusCode;
	TsDb->commonPart.dialogToken = dialogToken;
	
	allocationStatus = tsManagerRecipientRmRequest(TsDb, &pdRequestFillParameters);
	
	if(isAddbaRequestAccpted)
	{
		/* Check if the station sent a "dont care" value in bitmap length: */		
		if (0 == bufferSize)
		{
			/* The ADDBA initiator indicates (802.11-2016): "the Buffer Size field of an ADDBA Response frame is in the range 1 to 64". */
			bufferSize = TS_MANAGER_WINDOW_SIZE_64;
		}
		
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaRequestAccepted++;
		
		StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
		if(bufferSize > TsManagerRecipientVapDatabase[vapIndex].bufferSizePerTidArray[TsDb->commonPart.tid])
		{
			/* In case the internal restriction is smaller than the buffer size or the buffer size is 0 
			   we take the internal restriction */
		    bufferSize = TsManagerRecipientVapDatabase[vapIndex].bufferSizePerTidArray[TsDb->commonPart.tid];   
		}

		timeout = addbaRequestPayload->u16BlockAckTimeOut;
		if(0 == timeout)
		{
			/* Initiator wants to disable inactivity mechanism */
			TsDb->commonPart.deleteBaAgreementOnInactivity = FALSE;
			timeout = TsManagerRecipientVapDatabase[vapIndex].maxInactivityTimeout;
		}
		else
		{
			timeout = TS_MANAGER_CONVERT_TIME_UNITS_INTO_250_MILLISECONDS_UINTS(timeout);
			/* Taking the minimum between internal timeout and the initiator timeout */
			timeout = MIN(timeout,TsManagerRecipientVapDatabase[vapIndex].maxInactivityTimeout);
			TsDb->commonPart.deleteBaAgreementOnInactivity = TRUE;
		}
		
		tsManagerRecipientBaAgreementOpenedUpdateCounters(TsDb->commonPart.stationId);
		
		/* Save parameters  */
		TsDb->commonPart.bufferSize = TsManagerConvertWinSizeToBufferSize(bufferSize);
		/*Convert the buffer size back - this is needed in case the buffer size was not a 2^ value*/
		bufferSize = TsManagerBufferSizeToWindowSize[TsDb->commonPart.bufferSize];
		TsDb->commonPart.timeout = timeout;	
		sequenceNumber = addbaRequestPayload->u16BlockAckStartSn.BlockAckStartSn;

		/*A little ugly but it is a waste to store the SSN as it takes too much space*/
		TsDb->commonPart.calendarWheelEntry.nextIndex = sequenceNumber;
	
		/*Only if the station declared A-MSDU support and the TS manager recipient support A-MSDU the ADDBA response will indicate A-MSDU support */
		TsDb->commonPart.amsduSupport = addbaRequestPayload->u16BlockAckParameterSet.BlockAckAmsduSupported && TsManagerRecipientVapDatabase[vapIndex].amsduSupportPerTidArray[TsDb->commonPart.tid];

		/* The ADDBA request is accepted */
		if(allocationStatus)
		{
			TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventBaAgreementOpened++;
			/* PD allocation succeeded */
			RxPp_OpenTs(TsDb->commonPart.stationId, TsDb->commonPart.tid, bufferSize, sequenceNumber);
			TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaResponseWasSentWithStatusSuccess++;
			TsManager_TransmitAddbaResponse(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
			tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENING);
		}
		else
		{
			/* PD allocation failed */		
			TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
			if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
			{
				tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
				tsManagerRecipientAddbaResponseConfirmedInClosing(TsDb, NULL);
			}
			else
			{
				tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_OPENING_WAITING_PD);
			}
		}		
	}
 	else
 	{	
		ILOG2_D("tsManagerRecipientProcessAddbaRequest ADDBA request declined allocation status %d", allocationStatus);
		/* The ADDBA request is rejected */
		if(allocationStatus)
		{
			/* PD allocation succeeded */
			TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventAddbaResponseWasSentWithStatusFailure++;
			TsManager_TransmitAddbaResponse(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
			tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
		}
		else
		{
			/* PD allocation failed */
			TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
			if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
			{
				tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING);
				tsManagerRecipientAddbaResponseConfirmedInClosing(TsDb, NULL);
			}
			else
			{
				tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSING_WAITING_PD_FOR_ADDBA_RESPONSE);
			}
		}		
 	}

}


/**********************************************************************************

tsManagerRecipientTimerEvent 


Description:
------------
This function checks if there was activity in a specific station and TID

Input: 
-----
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void tsManagerRecipientTimerEvent(TsManagerRecipientStationTidParams_t *stationTidEntry)
{
	StaId stationIndex = 0;
	uint8 tid = 0;
	uint8 activity = FALSE;
	uint8 windowChange = FALSE;
	TsManagerRecipientStationTidParams_t *TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	
	stationIndex = stationTidEntry->commonPart.stationId;
	tid = stationTidEntry->commonPart.tid;
	TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
        
	DEBUG_ASSERT(TsDb->commonPart.state == TS_MANAGER_RECIPIENT_TID_STATE_OPENED);
				
	activity = Pac_RxcBaAgreementRecipientGetAndClearActivity(stationIndex, tid);

	windowChange = RxPp_WindowChange(stationIndex, tid);
	
	if((activity)&&(windowChange))
	{
		/* Return the entry to calendar wheel to check inactivity again */
		TsManager_AddToCalendarWheel(&TsManagerRecipientCalendarWheel, 
									 &TsDb->commonPart, 
									 TsDb->commonPart.timeout);		
	}
	else
	{
		tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_TIMER);
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventInactivityTimeout++;
		
		/* There wasnt activity - flush queue and if needed delete BA agreement */
		if(TsDb->commonPart.deleteBaAgreementOnInactivity)
		{
			TsDb->commonPart.reason = TS_MANAGER_TIMEOUT_REASON;
			tsManagerRecipientBaAgreementClosedUpdateCounter(stationIndex);
			/* should not be flushed yet - all closing actions will be done in the DELBA confirmation */
			tsManagerRecipientTryToSendDelba(TsDb);
		}
		else
		{
			RxPp_FlushTid(stationIndex, tid, RXPP_DONT_DISCARD, RXPP_NON_BA_CLIENT_DONT_FORCE);			
			/* Return the entry to calendar wheel to check inactivity again */
			TsManager_AddToCalendarWheel(&TsManagerRecipientCalendarWheel, 
									     &TsDb->commonPart, 
									     TsDb->commonPart.timeout);
		}
	}
}


/**********************************************************************************

tsManagerRecipientBaAgreementOpenedUpdateCounters 


Description:
------------
this function updates the counters when new BA agreement was opened in the recipient

Input: 
-----
stationIndex - station index

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientBaAgreementOpenedUpdateCounters(StaId stationIndex)
{
	uint8 vapIndex = 0;
	
	TsManagerRecipientGlobalParameters.numOfGlobalOpenBaAgreements++;
	TsManagerRecipientStationOpenedBaAgreements[stationIndex]++;

	StaDb_GetVapId(stationIndex, &vapIndex);
	TsManagerRecipientVapDatabase[vapIndex].openedBaAgreements++;
}


/**********************************************************************************

tsManagerRecipientBaAgreementClosedUpdateCounter 


Description:
------------
this function updates the counters when agreement was closed in the recipient


Input: 
-----
stationIndex - station index


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerRecipientBaAgreementClosedUpdateCounter(StaId stationIndex)
{
	uint8 vapIndex = 0;
	
	TsManagerRecipientGlobalParameters.numOfGlobalOpenBaAgreements--;
	TsManagerRecipientStationOpenedBaAgreements[stationIndex]--;

	StaDb_GetVapId(stationIndex, &vapIndex);
	TsManagerRecipientVapDatabase[vapIndex].openedBaAgreements--;
}

/**********************************************************************************

tsManagerRecipientIsAllowedToOpenBaAgreement 


Description:
------------
this function returns if a specific station is currently allowed to open new BA agreement
in the recipient side

Input: 
-----
stationIndex - the station to open BA agreement for


	
		
Output:
-------
	

Returns:
--------
	allowed - indicates if the specific station is allowed to open new BA agreement 
	
**********************************************************************************/

static uint8 tsManagerRecipientIsAllowedToOpenBaAgreement(StaId stationIndex)
{
	uint8 vapIndex = 0;

	StaDb_GetVapId(stationIndex, &vapIndex);
	
	/* First check global limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if(TsManagerRecipientGlobalParameters.numOfGlobalOpenBaAgreements >= TsManagerRecipientGlobalParameters.maxGlobalBaAgreements)
	{
		return FALSE;
	}

	/* Check Vap limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if(TsManagerRecipientVapDatabase[vapIndex].openedBaAgreements >= TsManagerRecipientVapDatabase[vapIndex].maxBaAgreementPerVap)
	{
		return FALSE;
	}
	
	/* Check station limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if(TsManagerRecipientStationOpenedBaAgreements[stationIndex] >= TsManagerRecipientVapDatabase[vapIndex].maxBaAgreementsPerStation)
	{
		return FALSE;
	}
	
	return TRUE;
}

/**********************************************************************************

tsManagerRecipientProcessBar 


Description:
------------
This function processes BAR (when it is received in the states that BA agreement 
might be opened)

Input: 
-----
stationIndex - station index
tid - the TID number
addbaRequestPayload - pointer to the ADDBA request payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRecipientProcessBar (TsManagerRecipientStationTidParams_t *TsDb, BarPayload_t *barPayload)
{
	uint16 sequenceNumber = 0;
	
	TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventBarWasReceivedWithBaAgreementOpened++;

    sequenceNumber = barPayload->u16BlockAckStartSn.BlockAckStartSn ;
    RxPp_UpdateWindowSn(TsDb->commonPart.stationId, TsDb->commonPart.tid, sequenceNumber, RXPP_NON_BA_CLIENT_DONT_FORCE);
}

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

TsManager_ChangeRecipientStationAllowed 


Description:
------------
change the allowed flag for all TIDs of specific station in the recipient

Input: 
-----
stationIndex - the specific station to change its TIDs flags 
allowed - the new value for the flag
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_ChangeRecipientStationAllowed(StaId stationIndex, uint8 allowed)
{
	uint32 i = 0;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	for(i = 0; i < NUM_OF_TID; i++)		
	{
		TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][i];
		TsDb->commonPart.allowed = allowed;
	}
}


/**********************************************************************************

TsManagerRecipient_AddStation 


Description:
------------
add new station to the TS manager recipient

Input: 
-----
stationIndex - station index
allowed - flag that indicates if the station is allowed to open BA agreement
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_AddStation(StaId stationIndex, uint8 allowed, uint8 pbac)
{
    uint32 i = 0;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;
	
	TsManagerRecipientStationGlobalStatistics.RecipientStationStatisticsEventAdd++;
	

	for(i = 0; i < NUM_OF_TID; i++)		
	{
		TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][i];
		DEBUG_ASSERT(TsDb->commonPart.state == TS_MANAGER_RECIPIENT_TID_STATE_IDLE);
		TsDb->commonPart.dialogToken = 0;
		tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_ADD_STA);
		tsManagerRecipientChangeState(TsDb, TS_MANAGER_RECIPIENT_TID_STATE_CLOSED);
		TsDb->commonPart.pbac = pbac;
		TsDb->commonPart.allowed = allowed;
	}
}

/**********************************************************************************

TsManagerRecipient_RemoveStation 


Description:
------------
remove all station TIDs in the recipient

Input: 
-----
stationIndex - station index
tid - the TID to remove
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_RemoveStation(StaId stationIndex)
{
   	uint32 i = 0;
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	//("TsManagerRecipient_RemoveStation stationIndex %d ", stationIndex);

	TsManagerRecipientStationGlobalStatistics.RecipientStationStatisticsEventRemove++;

	for(i = 0; i < NUM_OF_TID; i++)
	{
		TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][i];
		tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_REMOVE_STA);
		state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
		TsManagerRecipientRemoveTable[state](TsDb, NULL);
	}
}

/**********************************************************************************

TsManagerRecipient_AddbaRequestReceived 


Description:
------------
handle ADDBA request

Input: 
-----
stationIndex - the station for which the packet was received
addbaRequestFrame - the pointer to the frame (without the phy metrics)

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_AddbaRequestReceived(StaId stationIndex, AddbaRequestFrame_t *addbaRequestFrame)
{
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	uint8 tid = 0;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;
	
	tid = addbaRequestFrame->addbaRequestPayload.u16BlockAckParameterSet.BlockAckTid;
	if(tid >= NUM_OF_TID)
	{
		/* Should be bigger than 7 - numbers above 7 meant for traffic stream agreement which is not supported */
		TsManagerRecipientStationGlobalStatistics.RecipientStationStatisticsEventAddbaRequestReceivedWithIllegalTid++;
		return;
	}
	TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_ADDBA_REQ_RECEIVED);

	state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
	TsManagerRecipientAddbaRequestReceivedTable[state](TsDb, &addbaRequestFrame->addbaRequestPayload);
}

/**********************************************************************************

TsManagerRecipient_DelbaReceived 


Description:
------------
handle Delba received

Input: 
-----
stationIndex - station index
tid - the TID to remove
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_DelbaReceived(StaId stationIndex, uint8 tid)
{
	TsManagerRecipientStationTidParams_t *TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;

	tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_DELBA_RECEIVED);
    state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
	TsManagerRecipientDelbaReceivedTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerRecipient_BarReceived 


Description:
------------
handle BAR received

Input: 
-----
rd - address of the Rd which contains the recevied packet

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_BarReceived(Rd_t  *rxDescriptor)
{
	RxMpdu_t *rxMpdu = NULL; 
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	StaId stationIndex = 0;
	uint8 tid = 0;
	BarFrame_t *barFrame = NULL;
	uint8 discard = FALSE;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	rxMpdu = (RxMpdu_t *)(CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR((CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset))); 

    barFrame = (BarFrame_t *)rxMpdu->frame;

	if(barFrame->barPayload.bar_ack_policy || barFrame->barPayload.multi_tid || !barFrame->barPayload.compressed_bitmap)
	{
		/* Currently not immidiate block ack policy ,multi tid , not compressed block ack are not supported */
        discard = TRUE;
	}

	tid = barFrame->barPayload.TID;
	if(tid >= NUM_OF_TID)
	{
		/* Should be bigger than 7 - numbers above 7 meant for traffic stream agreement which is not supported */
		discard = TRUE;
	}  

	/*Get Station Index from RD*/
	stationIndex = rxDescriptor->rxQueueStaId;

	TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];

	/*if PBAC was negotiated, BAR should not be used*/
	if (TsDb->commonPart.pbac == TRUE)
	{
		discard = TRUE;
	}
	
	if(discard)
	{
		//TBD discard the BAR - statisitcs or error
		ILOG2_V("TsManagerRecipient_BarReceived illegal BAR");
		return;
	}

	tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_BAR_RECEIVED);
    state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;

	/* The sequence number is passed here as a pointer to keep unified parameters type for all the tables and
	   not create special type of function just for BAR */
	TsManagerRecipientBarReceivedTable[state](TsDb, &barFrame->barPayload);
}

/**********************************************************************************

TsManagerRecipient_AddbaResponseConfirmation 


Description:
------------
handles ADDBA response confirmation

Input: 
-----
packetDescriptor - the PD of the ADDBA response

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_AddbaResponseConfirmation(TxPd_t *packetDescriptor)
{
	uint8 tid = 0;
	StaId stationIndex = 0;
	AddbaResponseFrame_t *addbaResponseFrame = NULL;
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	addbaResponseFrame = (AddbaResponseFrame_t *)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)packetDescriptor->packetPointer);
	stationIndex = packetDescriptor->txQStaId;
	tid = addbaResponseFrame->addbaResponsePayload.u16BlockAckParameterSet.BlockAckTid;
	TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_ADDBA_RSP_CONFIRMED);

    state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
	TsManagerRecipientAddbaResponseConfirmationdTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerRecipient_DelbaConfirmation 


Description:
------------
handles DELBA confirmation

Input: 
-----
stationIndex - station index
tid - the TID to remove
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_DelbaConfirmation(StaId stationIndex, uint8 tid)
{
	TsManagerRecipientStationTidParams_t *TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;

	tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_DELBA_CONFIRMED);
    state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
    TsManagerRecipientDelbaConfirmationdTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerRecipient_IlegalPacketReceived 


Description:
------------
Handles the event that illegal packet was received - BAR or aggregation were received
and BA agreement doesnt exist. This function processes the Rxf FIFO in the SHRAM.


Input: 
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_IlegalPacketReceived(K_MSG* psMsg)
{
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	StaId stationIndex = 0;
	uint8 tid = 0;
	uint8 counter = 0;
	RxfFifoParameters_t fifoParameters;
	RegRxfFifoLine_u singleFifoLine;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	singleFifoLine.val = 0;
	memset(&fifoParameters, 0, sizeof(RxfFifoParameters_t));
	Pac_RxfGetFifoParameters(&fifoParameters);

	for(counter = 0; counter < fifoParameters.numberOfEntries; counter++)
	{
		singleFifoLine.val = RxfAmpduNoBaAggrementFifo[fifoParameters.currentIndexInFifo].val;
		tid = singleFifoLine.bitFields.tid;
		stationIndex = singleFifoLine.bitFields.address2Index;
		TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
		tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_ILLEGAL_PACKET);
		
		state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
    	TsManagerRecipientIllegalPacketReceivedsTable[state](TsDb, NULL);
		
		fifoParameters.currentIndexInFifo++;
		fifoParameters.currentIndexInFifo &= RXF_FIFO_AMPDU_NO_BA_AGGREMENT_DEPTH_MASK;
	}  

	Pac_RxfSetFifoParameters(&fifoParameters);

	EventManager_TurnOnEvent(EVENT_ID_RXF_FIFO_NOT_EMPTY);	
}

/**********************************************************************************

TsManagerRecipient_TimerExpired 


Description:
------------
Handles the event that TS manager timer has expired - process the entries in the 
calendar wheel

Input: 
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_TimerExpired(void)
{
	uint16 currentIndex = CALENDAR_WHEEL_NULL_ENTRY_INDEX;
	TsManagerRecipientStationTidParams_t *entry = NULL;

	currentIndex = CalendarWheel_TimerEvent(&TsManagerRecipientCalendarWheel);
	while(CALENDAR_WHEEL_NULL_ENTRY_INDEX != currentIndex)
	{	
		entry = (TsManagerRecipientStationTidParams_t *)TsManagerRecipientStationTidDatabase + currentIndex;
		DEBUG_ASSERT(entry->commonPart.entryState == TS_MANAGER_CALENDAR_WHEEL_ENTRY_ENQUEUED);
		entry->commonPart.entryState = TS_MANAGER_CALENDAR_WHEEL_ENTRY_FREE;
		/* Save next before inserting to another list */
        currentIndex = entry->commonPart.calendarWheelEntry.nextIndex;
		
		tsManagerRecipientTimerEvent(entry);
	}
}

/**********************************************************************************

TsManagerRecipient_PdAllocated 


Description:
------------
handles Pd allocated from the resourec manager

Input: 
-----
stationTidEntry - the address of the station/TID entry
packetDescriptor - the address of the PD that was allocated for this station/TID

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_PdAllocated(StaId stationIndex, uint8 tid, TxPd_t *packetDescriptor)
{
	TsManagerRecipientStationTidParams_t *TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;

    state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
	ILOG2_DDD("TsManagerRecipient_PdAllocated stationIndex %d tid %d state %d", stationIndex, tid, state);
	ResourceManager_ReleaseRequest(TsDb->commonPart.pdRequestId, DESC_POOL_TS_MANAGER);
    TsManagerRecipientPdAllocatedTable[state](TsDb, packetDescriptor);
}

/**********************************************************************************

TsManagerRecipient_CloseTids 


Description:
------------
handles close TID event

Input: 
-----
stationIndex - station index
tidsBitmap - bitmap that indicates which TIDs to handle
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_CloseTids(StaId stationIndex, uint8 tidsBitmap)
{
	TsManagerRecipientTidState_e state = TS_MANAGER_RECIPIENT_TID_STATE_IDLE;
	uint8 tid = 0;
	uint32 tempBitmap = 0;
	TsManagerRecipientStationTidParams_t *TsDb = NULL;

	/* Using temp variable since the function uses uint32 */
	tempBitmap = tidsBitmap;

    while(tempBitmap)
    {
		tid = Utils_FindFirstSetAndClear(&tempBitmap);
		TsDb = &TsManagerRecipientStationTidDatabase[stationIndex][tid];
		tsManagerRecipientLogEvent(TsDb, TS_MANAGER_RECIPIENT_TID_EVENT_CLOSE_TID);
		state = (TsManagerRecipientTidState_e)TsDb->commonPart.state;
		
		TsManagerRecipientTidGlobalStatistics.RecipientTidStatisticsEventCloseTid++;
		
		TsManagerRecipientCloseTidTable[state](TsDb, NULL);
    } 
}


/**********************************************************************************

TsManagerRecipient_ConfigureVapParameters 


Description:
------------
configures the VAP parameters in the recipient VAP database

Input: 
-----
vapIndex - the VAP index to configure 
vapConfigurationParameters - a pointer to the structure of the parameters

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerRecipient_ConfigureVapParameters(uint8 vapIndex, TsManagerRecipientVapConfigurationParams_t *vapConfigurationParameters)
{
	uint8 tid = 0;

	TsManagerRecipientVapDatabase[vapIndex].maxBaAgreementPerVap= vapConfigurationParameters->maxBaAgreementsPerVap;
	TsManagerRecipientVapDatabase[vapIndex].maxInactivityTimeout = TS_MANAGER_CONVERT_TIME_UNITS_INTO_250_MILLISECONDS_UINTS(vapConfigurationParameters->maxInactivityTimeout);
	TsManagerRecipientVapDatabase[vapIndex].maxBaAgreementsPerStation = vapConfigurationParameters->maxBaAgreementsPerStation;
	
    /* Fill structures for the initiator VAP database and the recipient VAP database */
	for(tid = 0; tid < NUM_OF_TID; tid++)		
	{
		TsManagerRecipientVapDatabase[vapIndex].amsduSupportPerTidArray[tid] = vapConfigurationParameters->amsduSupportPerTidArray[tid];		
		TsManagerRecipientVapDatabase[vapIndex].bufferSizePerTidArray[tid] = vapConfigurationParameters->maxBufferSizePerTidArray[tid];		
	}   
}


/**********************************************************************************

TsManager_TsManagerRecipientTidGlobalStatistics_Address 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic
**********************************************************************************/
uint32* tsManagerGetRecipientTidGlobalStatisticsAddress()
{
	return ((uint32*)&TsManagerRecipientTidGlobalStatistics);
}

/**********************************************************************************

TsManager_TsManagerRecipientStationGlobalStatistics_Address 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic	
**********************************************************************************/
uint16* tsManagerGetRecipientStationGlobalStatisticsAddress()
{
	return ((uint16*)&TsManagerRecipientStationGlobalStatistics);
}


/**********************************************************************************

TsManagerRecipient_Initialize 


Description:
------------
initialize internal databases and structures of the Ts manager recipient

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization_start" 
#endif

void TsManagerRecipient_Initialize()
{
	uint32 stationIndex = 0;
	uint32 tidIndex = 0;

#if defined (ENET_INC_UMAC)
	System_MainAllocInitializationMemory((uint8 **)&TsManagerRecipientCalendarWheelSlots, (sizeof(uint16)*TS_MANAGER_CALENDAR_WHEEL_SIZE));
	System_MainAllocInitializationMemory((uint8 **)&TsManagerRecipientVapDatabase, (sizeof(TsManagerRecipientVapParams_t)*HW_NUM_OF_VAPS));
#endif	

	/* Initialize Calendar wheel - timer will be set later */
	CalendarWheel_Initialize(&TsManagerRecipientCalendarWheel, 
							 TS_MANAGER_CALENDAR_WHEEL_SIZE, 
							 TsManagerRecipientCalendarWheelSlots, 
							 (uint32)TsManagerRecipientStationTidDatabase, 
							 sizeof(TsManagerRecipientStationTidParams_t));
		
	/* Initialize the internal structures - the databases are word alligned */
	memset32(TsManagerRecipientStationTidDatabase, 0, CONVERT_BYTES_TO_WORDS(sizeof(TsManagerRecipientStationTidDatabase)));
	memset(TsManagerRecipientStationOpenedBaAgreements, 0, sizeof(TsManagerRecipientStationOpenedBaAgreements));
	memset(TsManagerRecipientVapDatabase, 0, sizeof(TsManagerRecipientVapParams_t)*HW_NUM_OF_VAPS);
	memset(&TsManagerRecipientGlobalParameters, 0, sizeof(TsManagerRecipientGlobalParams_t));


	/* This initialization is done in case the global parameters message wont be sent from the driver */
	TsManagerRecipientGlobalParameters.maxGlobalBaAgreements = TS_MANAGER_MAX_BA_AGREEMENT_RECIPIENT_DEFAULT_VALUE;

	/* Initialize the recipient station/TID database */
    for(stationIndex = 0 ; stationIndex < HW_NUM_OF_STATIONS; stationIndex++)
    {
		for(tidIndex = 0; tidIndex < NUM_OF_TID; tidIndex++)
		{
			TsManagerRecipientStationTidDatabase[stationIndex][tidIndex].commonPart.stationId = stationIndex;
			TsManagerRecipientStationTidDatabase[stationIndex][tidIndex].commonPart.tid = tidIndex;
			TsManagerRecipientStationTidDatabase[stationIndex][tidIndex].commonPart.entryType = TS_MANAGER_ENTRY_TYPE_RECIPIENT;
		}
    }

	memset(&TsManagerRecipientTidGlobalStatistics, 0, sizeof(TsManagerRecipientTidGlobalStatistics_t));
	memset(&TsManagerRecipientStationGlobalStatistics, 0, sizeof(TsManagerRecipientStationGlobalStatistics_t));
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif