/***********************************************************************************
 File:			TsManager.c
 Module:		Ts Manager
 Purpose: 		To handle BA agreements
 Description:   This file is the implementation of the Ts manager which is responsible 
 				to configure the Ts manager recipeient and the Ts manager initiator
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_MainApi.h"
#include "TsManager_API.h"
#include "TsManager.h"
#include "TsManagerRecipient.h"
#include "TsManagerInitiator.h"
#include "PacketDescriptor.h"
#include "HwGlobalDefinitions.h"
#include "RxDescriptor.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "ResourceManager_API.h"
#include "stringLibApi.h"
#include "HwQManager_API.h"
#include "RxMpduStructure.h"
#include "HwMemoryMap.h"
#include "CalendarWheel_Api.h"
#include "ShramPacketDescriptors.h"
#include "BSSmanager_API.h"
#include "StaDatabase_Api.h"
#include "TxPacketsClassifier_API.h"
#include "Locker_Api.h"
#include "frame.h"
#include "shramTxDesc.h"
#include "shram_man_msgs.h"
#include "init_ifmsg.h"
#include "Dut_Api.h"
#include "Utils_Api.h"
#include "ShramStationDatabase.h"
#include "VapDatabase_Api.h"
#include "queue_utility.h"
#include "RegAccess_Api.h"
#include "HeGroupManager_API.h"
#include "linkAdaptation_api.h"
#include "StaDatabase_Api.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TS_MANAGER 
#define LOG_LOCAL_FID 5


#define TS_ADDBA_REQ_FRAME_MIN_LENGTH		(sizeof(FM_PAYLOAD_ADDBA_REQ) + sizeof(MANAGEMENT_FRAME_HEADER))
#define TS_ADDBA_RES_FRAME_MIN_LENGTH		(sizeof(FM_PAYLOAD_ADDBA_RES) + sizeof(MANAGEMENT_FRAME_HEADER))
#define TS_DELBA_FRAME_MIN_LENGTH			(sizeof(FM_PAYLOAD_DELBA) + sizeof(MANAGEMENT_FRAME_HEADER))
#define TS_ACTION_FRAME_MIN_LENGTH			(MIN(TS_DELBA_FRAME_MIN_LENGTH, MIN(TS_ADDBA_REQ_FRAME_MIN_LENGTH, TS_ADDBA_RES_FRAME_MIN_LENGTH)))


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef struct TsManagerStaDb_s
{
	uint8 activeTids;
	uint8 isStaHt:1;
	uint8 isStaVht:1;
	uint8 isStaHe:1;
	uint8 isStaMfp:1;
	uint8 isXFilterOpen:1;
	uint8 reserved:3;
} TsManagerStaDb_t;

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void tsManagerAddStation(K_MSG *tsManagerMessage); 
static void tsManagerSetFilter(K_MSG *tsManagerMessage); 
static void tsManagerRemoveStation(K_MSG *tsManagerMessage);
static void tsManagerActionPacketReceived(K_MSG *tsManagerMessage);
static void tsManagerBarPacketReceived(K_MSG *tsManagerMessage);
static void tsManagerActionPacketConfirmed(K_MSG *tsManagerMessage);
static void tsManagerAddbaRequestConfirmed(K_MSG *tsManagerMessage);
static void tsManagerBarConfirmed(K_MSG *tsManagerMessage);
static void tsManagerBarStatusReceived(K_MSG *tsManagerMessage);
static void tsManagerDataPacketDiscarded(K_MSG *tsManagerMessage);
static void tsManagerDataPacketDiscardedBitmap(K_MSG *tsManagerMessage);
static void tsManagerTimerExpired(K_MSG *tsManagerMessage);
static void tsManagerPdAllocated(K_MSG *tsManagerMessage);
static void tsManagerIllegalPacketReceived(K_MSG *tsManagerMessage);
static void tsManagerCloseTids(K_MSG *tsManagerMessage);
static void tsManagerOpenTids(K_MSG *tsManagerMessage);
static void tsManagerQueueIsLocked(K_MSG *tsManagerMessage);
static void tsManager_StaManagerSendConfirm(StaId sid);
static void tsManagerAddVap(K_MSG* psMsg);
static void tsManagerRemoveVap(K_MSG* psMsg);
static void tsManagerConfigureVap(K_MSG* psMsg);
static void tsManagerConfigureTxop(K_MSG* tsManagerMessage);


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
TsManagerStaDb_t TsManagerStaDb[HW_NUM_OF_STATIONS];
TsManagerVapConfigurationParams_t TsManagerVapConfiguration[HW_NUM_OF_VAPS];
TsManagerGlobalDb_t TsManager_GlobalDb;

static const FunctionEntry_t afpTaskTable[TASK_TS_MANAGER_END - TASK_TS_MANAGER_START]=
{
	{tsManagerAddStation,				DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ADD_STATION)},
	{tsManagerSetFilter,				DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_SET_FILTER)},
	{tsManagerRemoveStation,			DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_REMOVE_STATION)},
	{tsManagerActionPacketReceived,		DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ACTION_PACKET_RECEIVED)},
	{tsManagerBarPacketReceived,		DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_BAR_PACKET_RECEIVED)},	
	{tsManagerActionPacketConfirmed,    DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ACTION_PACKET_CONFIRMED)},
	{tsManagerAddbaRequestConfirmed,    DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ADDBA_REQUEST_CONFIRMED)},		
	{tsManagerBarConfirmed,			    DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_BAR_CONFIRMED)},
	{tsManagerBarStatusReceived,		DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_BAR_STATUS_RECEIVED)},	
	{tsManagerDataPacketDiscarded,		DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_PACKET_DISCARDED)},		
	{tsManagerDataPacketDiscardedBitmap,DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_PACKET_DISCARDED_BITMAP)},		
	{tsManagerTimerExpired,			    DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_TIMER_EXPIRED)},
	{tsManagerPdAllocated,	            DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_PACKET_DESCRIPTOR_ALLOCATED)},
	{tsManagerIllegalPacketReceived,	DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ILLEGAL_PACKET)},
	{tsManagerCloseTids,	            DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_CLOSE_TIDS)},
	{tsManagerOpenTids,	                DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_OPEN_TIDS)},
	{tsManagerQueueIsLocked,	        DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_QUEUE_IS_LOCKED)},
	{tsManagerAddVap,					DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_ADD_VAP)},
	{tsManagerRemoveVap,				DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_REMOVE_VAP)},
	{tsManagerConfigureVap,				DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_CONFIGURE_VAP)},
	{tsManagerConfigureTxop, 			DOUBLE_CHECK_MSG_TYPE(TS_MANAGER_CONFIGURE_TXOP_MODE)},
};


static uint16 TsManagerReasonToReasonCode[TS_MANAGER_NUM_REASON] = {
	TS_MANAGER_TIMEOUT_REASON_CODE,
	TS_MANAGER_END_BA_REASON_CODE,
	TS_MANAGER_UNKNOWN_BA_REASON_CODE
};

static uint16 TsManagerStatusToStatusCode[TS_MANAGER_NUM_STATUS] = {
	TS_MANAGER_SUCCESS_STATUS_CODE,
	TS_MANAGER_UNSPECIFIED_STATUS_CODE,
	TS_MANAGER_INVALID_PARAMETERS_STATUS_CODE
};

uint16 TsManagerBufferSizeToWindowSize[TS_MANAGER_NUM_BUFFER_SIZE] = {
	TS_MANAGER_WINDOW_SIZE_2,    
	TS_MANAGER_WINDOW_SIZE_4,
	TS_MANAGER_WINDOW_SIZE_8,		
	TS_MANAGER_WINDOW_SIZE_16,		
	TS_MANAGER_WINDOW_SIZE_32,		
	TS_MANAGER_WINDOW_SIZE_64,
	TS_MANAGER_WINDOW_SIZE_128,
	TS_MANAGER_WINDOW_SIZE_256
};

// How many TIDs can the Auto Response handle in one HE MU agg accoding to the max win size among all TIDs of a specific STAs.
// The auto response bitmap is 512 bits long and it must be divided to equal size parts (2*256 / 4*128 / 8*64)
uint16 TsManagerBufferSizeToMaxTidInHeMuAgg[TS_MANAGER_NUM_BUFFER_SIZE] = {
	8,	// TS_MANAGER_BUFFER_SIZE_2 - 	8*2 < 512	
	8,	// TS_MANAGER_BUFFER_SIZE_4 - 	8*4 < 512	
	8,	// TS_MANAGER_BUFFER_SIZE_8 - 	8*8 < 512	
	8,	// TS_MANAGER_BUFFER_SIZE_16 - 	8*16 < 512	
	8,	// TS_MANAGER_BUFFER_SIZE_32 - 	8*32 < 512
	8,	// TS_MANAGER_BUFFER_SIZE_64 -	8*64   = 512
	4,	// TS_MANAGER_BUFFER_SIZE_128 -	4*128 = 512
	2	// TS_MANAGER_BUFFER_SIZE_256 -	2*256 = 512
};


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

uint32 *TsManagerPdBasePtr;

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

tsManagerAddStation 


Description:
------------
add new station to the TS manager recipient and to the TS manager initiator

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and HT mode	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerAddStation(K_MSG *tsManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(tsManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	StaId sid = pAddSta->u16SID;
	uint8 allowed = FALSE;
	uint8 isXFilterOpen = FALSE;
	uint8 pbac = FALSE;

	TsManagerStaDb[sid].activeTids = TS_MANAGER_NUMBER_OF_INITIATOR_AND_RECIPIENT_TIDS;
	/*Store flags for later use*/
	TsManagerStaDb[sid].isStaHt = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_HT);
	TsManagerStaDb[sid].isStaVht = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_VHT);
	TsManagerStaDb[sid].isStaHe = MTLK_BFIELD_GET(pAddSta->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE);
	TsManagerStaDb[sid].isStaMfp = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_MFP);
	TsManagerStaDb[sid].isXFilterOpen = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);

	allowed = TsManagerStaDb[sid].isStaHt | TsManagerStaDb[sid].isStaVht | TsManagerStaDb[sid].isStaHe;
	isXFilterOpen = TsManagerStaDb[sid].isXFilterOpen;
	pbac = MTLK_BFIELD_GET(pAddSta->u8FlagsExt, STA_ADD_FLAGS_EXT_PBAC);

	ILOG2_DDD("Ts Manager, ADD STA, SID %d, allowed %d, Filter %d", sid, allowed, isXFilterOpen);

    TsManagerInitiator_AddStation(sid, allowed, pbac);
	TsManagerRecipient_AddStation(sid, allowed, pbac);

	if ((isXFilterOpen == TRUE) && (allowed == TRUE) && (TsManager_GlobalDb.enableBa == TRUE))
	{
		TsManagerInitiator_OpenTids(sid, TID_BITMAP_ALL_DATA_TIDS);
	}
	tsManager_StaManagerSendConfirm(sid);
}

/**********************************************************************************

tsManagerAddVap 


Description:
------------
add new Vap to the TS manager recipient and to the TS manager initiator

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and HT mode	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerAddVap(K_MSG* psMsg)
{
	uint8 vapId; 
	UMI_ADD_VAP* addVapStructurePtr = (UMI_ADD_VAP*) EXTRACT_VAP_MANAGER_MSG(psMsg);

	vapId = addVapStructurePtr->vapId; 
	
	TsManager_ConfigureVapParameters(vapId, &TsManagerVapConfiguration[vapId]);
	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_ADD_VAP, BSS_MANAGER_VAP_MANAGER_TS_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);


}

/**********************************************************************************

tsManagerRemoveVap 


Description:
------------
Remove Vap configuration from  the TS manager recipient and to the TS manager initiator

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and HT mode	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRemoveVap(K_MSG* psMsg)
{
	uint8 vapId; 
	TsManagerVapConfigurationParams_t tsManagerVapParameters;
	UMI_REMOVE_VAP* removeVapStructurePtr = (UMI_REMOVE_VAP*) EXTRACT_VAP_MANAGER_MSG(psMsg);

	vapId = removeVapStructurePtr->vapId; 
	
    memset(&tsManagerVapParameters, 0, sizeof(TsManagerVapConfigurationParams_t));
	TsManager_ConfigureVapParameters(vapId, &tsManagerVapParameters); 
	ILOG2_V("tsManagerRemoveVap");

	FILL_VAP_MANAGER_CONFIRM_MSG(psMsg,vapId,VAP_MANAGER_REMOVE_VAP, BSS_MANAGER_VAP_MANAGER_TS_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);


}


/**********************************************************************************

tsManagerConfigureVap 


Description:
------------
Configure VAP parameters
Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and HT mode	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerConfigureVap(K_MSG* tsManagerMessage)
{
	UMI_TS_VAP_CONFIGURE *pTsVapConfig = (UMI_TS_VAP_CONFIGURE *)pK_MSG_DATA(tsManagerMessage);
	uint8 vapIndex = 0;
	StaId staId = 0;
	uint8 tid = 0;
	
	vapIndex = pTsVapConfig->vapId;
	if (pTsVapConfig->getSetOperation == API_GET_OPERATION)
	{
		pTsVapConfig->enableBa = TsManager_GlobalDb.enableBa;
		pTsVapConfig->amsduSupport = TsManagerVapConfiguration[vapIndex].initiatorVapParameters.amsduSupportPerTidArray[FIRST_TID];
		pTsVapConfig->windowSize = TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxBufferSizePerTidArray[FIRST_TID];
	}
	else
	{
		TsManager_GlobalDb.enableBa = pTsVapConfig->enableBa;
	
		/*For now host can configure AMSDU support only*/
	    ASSERT(pTsVapConfig->amsduSupport < 2); // (TRUE || FALSE)
		for(tid = 0; tid < NUM_OF_TID; tid++)
		{
			TsManagerVapConfiguration[vapIndex].recipientVapParameters.amsduSupportPerTidArray[tid] = pTsVapConfig->amsduSupport;
			TsManagerVapConfiguration[vapIndex].initiatorVapParameters.amsduSupportPerTidArray[tid] = pTsVapConfig->amsduSupport;
			if (pTsVapConfig->windowSize != WINDOW_SIZE_NO_CHANGE)
			{
				TsManagerVapConfiguration[vapIndex].recipientVapParameters.maxBufferSizePerTidArray[tid] = pTsVapConfig->windowSize;
				TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxBufferSizePerTidArray[tid] = pTsVapConfig->windowSize;
			}
		}
		/*configure initiator and recipient*/
		TsManager_ConfigureVapParameters(vapIndex, &TsManagerVapConfiguration[vapIndex]);


		// Choose if to use AMSDUs or not when the TS is closed. Sometimes it can be opened and sometimes not.
		if (TsManager_GlobalDb.enableBa == TRUE)
		{
			// BA is enabled
			TsManager_GlobalDb.ppduTxModeNoAmpdu = STA_DB_PPDU_TX_MODE_LEGACY_AND_MPDU | (pTsVapConfig->amsduSupport << STATION_DATABASE_TX_MODE_AMSDU_ENABLE_SHIFT);
		}
		else
		{
			// AMPDU is turned off. We check if we still support AMSDU or not. Store it for later use when we close the BA Agreement
			TsManager_GlobalDb.ppduTxModeNoAmpdu = (pTsVapConfig->amsduSupport << STATION_DATABASE_TX_MODE_AMSDU_ENABLE_SHIFT); 
		}
		
		/*Now we need to close the TIDs - on Initiator they will be reopened after a timeout*/
		staId = VapDb_GetFirstConnectedStaIndex(vapIndex);
		while (staId != DB_ASYNC_SID)
		{
			// Close the BA Agreements and re-open with new parametes if needed
			TsManagerInitiator_CloseTids(staId, TID_BITMAP_ALL_DATA_TIDS);
			if (TsManager_GlobalDb.enableBa == TRUE)
			{
				
				TsManagerInitiator_OpenTids(staId, TID_BITMAP_ALL_DATA_TIDS);
			}
			TsManagerRecipient_CloseTids(staId, TID_BITMAP_ALL_DATA_TIDS);
			/*Go to next STA in VAP*/
			staId = StaDb_GetNextSid(staId);
		}
	}
	
	OSAL_SEND_MESSAGE(LINK_ADAPTATION_SET_PPDU_TX_MODE_REQ, TASK_LINK_ADAPTATION, tsManagerMessage, vapIndex);	
}

/**********************************************************************************

tsManagerConfigureTxop 


Description:
------------
Configure TxOP mode

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and HT mode	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerConfigureTxop(K_MSG* tsManagerMessage)
{
	TxopModeMsg_t *pSetTxopConfig = (TxopModeMsg_t *)pK_MSG_DATA(tsManagerMessage);

	StaDb_setTxopMode(pSetTxopConfig->staId, pSetTxopConfig->mode); 
}

/**********************************************************************************

tsManagerSetFilter 


Description:
------------
called when a set filter request is received

Input: 
-----
tsManagerMessage - pointer to the message 	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerSetFilter(K_MSG *tsManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(tsManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_802_1X_FILTER *pSetFilter = (UMI_802_1X_FILTER *)pK_MSG_DATA(pMsg);
	StaId sid = pSetFilter->u16SID;
	uint8 allowed = (TsManagerStaDb[sid].isStaHt | TsManagerStaDb[sid].isStaVht |  TsManagerStaDb[sid].isStaHe);

	ILOG2_D("Ts Manager, Set Filter, SID %d", sid);

	TsManagerStaDb[sid].isXFilterOpen = TRUE;
	if ((allowed == TRUE) && (TsManager_GlobalDb.enableBa == TRUE))
	{
		TsManagerInitiator_OpenTids(sid, TID_BITMAP_ALL_DATA_TIDS);
	}
	tsManager_StaManagerSendConfirm(sid);
}

/**********************************************************************************

tsManagerRemoveStation 


Description:
------------
removes station from the TS manager recipient and from the TS manager initiator

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerRemoveStation(K_MSG *tsManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(tsManagerMessage);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(pMsg);
	StaId sid = pStopTraffic->u16SID;

	ILOG2_D("Ts Manager, Remove STA, SID %d", sid);

	TsManagerInitiator_RemoveStation(sid);
	TsManagerRecipient_RemoveStation(sid);
}

/**********************************************************************************

tsManagerActionPacketReceived 


Description:
------------
handles an action packet received - ADDBA request, ADDBA response and DELBA

Input: 
-----
tsManagerMessage - pointer to the message that contains the pointer of the RD	

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerActionPacketReceived(K_MSG *tsManagerMessage)
{
	RxMpdu_t*							rxMpdu = NULL;
	Rd_t*								rxDescriptor  = NULL;
	TsManagerPacketReceivedMessage_t*	tsManagerMessageParameters = NULL;
	TsManPayload_t*						tsManagementPacket = NULL;
	TsManagerAddbaParams_t				addbaParams;
	uint8 								action = 0;
	StaId stationIndex = 0;
	uint8 								tid = 0;  	
	HwQueueManagerRequestParams_t		 hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	tsManagerMessageParameters = ((TsManagerPacketReceivedMessage_t *)tsManagerMessage->abData);
	rxDescriptor = tsManagerMessageParameters->rxDescriptor;
	stationIndex = rxDescriptor->rxQueueStaId;

	rxMpdu = (RxMpdu_t *)(CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR((CONVERT_DRAM_POINTER_TO_BYTE_ADDRESS(rxDescriptor->dramPointer) + rxDescriptor->dramByteOffset)));
	tsManagementPacket = (TsManPayload_t *)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER*)rxMpdu->frame);

	if (rxDescriptor->dataLength >= ACTION_FRAME_MIN_LENGTH)
	{
		// Prepare a structure with the addba payload pointer and the length not including header and CRC.
		addbaParams.addbaPayload = tsManagementPacket;
		addbaParams.payloadLength = rxDescriptor->dataLength - frame_sizeOfExistingManagementFrameHeader((MANAGEMENT_BASIC_FRAME_HEADER *)rxMpdu->frame);

		/* It can be assumed that all packet here are action management of category block ack */
		/* All TS maangement packets have the block ack action field in the same offset */
		action = tsManagementPacket->PacketPayload.AddBaReqResource.u8ActionCode;
	}
	else // length too short - ignore by setting invalid action code (max num)
	{
		ILOG0_D("tsManagerActionPacketReceived: length too short - %d", rxDescriptor->dataLength);
		action = BA_AGREEMENT_MAX_NUM_OF_ACTION_CODES;
	}

	switch(action)
	{
		case BA_AGREEMENT_ACTION_CODE_ADDBA_REQUEST:	
			TsManagerRecipient_AddbaRequestReceived(stationIndex, &addbaParams);
			break;
			
		case BA_AGREEMENT_ACTION_CODE_ADDBA_RESPONSE:
			TsManagerInitiator_AddbaResponseReceived(stationIndex, &addbaParams);
			break;
			
		case BA_AGREEMENT_ACTION_CODE_DELBA:	
			tid = ((FM_PAYLOAD_DELBA *)tsManagementPacket)->u16DelBaParameterSet.BlockAckTid;
			
			if(tid < NUM_OF_TID)
			{
				if(((FM_PAYLOAD_DELBA *)tsManagementPacket)->u16DelBaParameterSet.BlockAckInitiator) 
				{
					TsManagerRecipient_DelbaReceived(stationIndex ,tid);
				}
				else
				{			
					TsManagerInitiator_DelbaReceived(stationIndex ,tid);
				}
			}
		    break;
			
		default:
			break;
	}

	/* retrun the RD to the liberator which will put it in the proper free RD pool according to 
	   its RD source field */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	hwQueueManagerRequestParams.pHeadDesc = rxDescriptor;
	HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);	
}

/**********************************************************************************

tsManagerBarPacketReceived 


Description:
------------
handles received BAR 

Input: 
-----
tsManagerMessage - pointer to the message that contains the pointer of the RD	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerBarPacketReceived(K_MSG *tsManagerMessage)
{
	Rd_t  *rxDescriptor  = NULL;
	TsManagerPacketReceivedMessage_t *tsManagerMessageParameters = NULL; 	
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));	
	
	tsManagerMessageParameters = ((TsManagerPacketReceivedMessage_t *)tsManagerMessage->abData);
	rxDescriptor = tsManagerMessageParameters->rxDescriptor;
	TsManagerRecipient_BarReceived(rxDescriptor);
 
	/* retrun the RD to the liberator which will put it in the proper free RD pool according to 
	   its RD source field */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST; 
	hwQueueManagerRequestParams.pHeadDesc = rxDescriptor; 
	HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);	
}


/**********************************************************************************

tsManagerActionPacketConfirmed 


Description:
------------
handles an action packet that was confirmed - transmission ended

Input: 
-----
tsManagerMessage - pointer to the message that contains the pointer of the PD	

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerActionPacketConfirmed(K_MSG *tsManagerMessage)
{
	TxPd_t *packetDescriptor = NULL;
	TsManagerActionPacketConfirmedMessage_t * tsManagerMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 tid;
	uint8 initiator = FALSE;

	FM_PAYLOAD_DELBA *delbaFramePayload = NULL;

	tsManagerMessageParameters = ((TsManagerActionPacketConfirmedMessage_t *)tsManagerMessage->abData);
	packetDescriptor = tsManagerMessageParameters->packetDescriptor;

	switch (packetDescriptor->mgmtActionValue)
	{
		case BA_AGREEMENT_ACTION_CODE_ADDBA_RESPONSE:
			TsManagerRecipient_AddbaResponseConfirmation(packetDescriptor);
			break;
		case BA_AGREEMENT_ACTION_CODE_DELBA:
			stationIndex = packetDescriptor->txQStaId;
			delbaFramePayload = (FM_PAYLOAD_DELBA *)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)packetDescriptor->packetPointer);
			tid = delbaFramePayload->u16DelBaParameterSet.BlockAckTid;		
			initiator = delbaFramePayload->u16DelBaParameterSet.BlockAckInitiator;
			if (initiator)
			{
				TsManagerInitiator_DelbaConfirmation(stationIndex, tid);
			}
			else
			{
				TsManagerRecipient_DelbaConfirmation(stationIndex, tid);
			}
			break;
	    default:
			FATAL("TS Manager Action Packet Confirmed - Unknown Action");		   	
			break;
	}
	
   /* Release the PD */
   ResourceManager_ReleaseDescriptor((TxPd_t *)packetDescriptor, DESC_POOL_TS_MANAGER);
}


/**********************************************************************************

tsManagerAddbaRequestConfirmed 


Description:
------------
handles an  ADDBA request confirmation

Input: 
-----
tsManagerMessage - pointer to the message that contains the pointer of the PD	

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerAddbaRequestConfirmed(K_MSG *tsManagerMessage)
{
	TxPd_t *packetDescriptor = NULL;
	TsManagerActionPacketConfirmedMessage_t * tsManagerMessageParameters = NULL;

	tsManagerMessageParameters = ((TsManagerActionPacketConfirmedMessage_t *)tsManagerMessage->abData);
	packetDescriptor = tsManagerMessageParameters->packetDescriptor;
	
	TsManagerInitiator_AddbaRequestConfirmation(packetDescriptor);

   /* Release the PD */
   ResourceManager_ReleaseDescriptor((TxPd_t *)packetDescriptor, DESC_POOL_TS_MANAGER);
}



/**********************************************************************************

tsManagerBarConfirmed 


Description:
------------
handles a confirmed BAR - only in case it was discarded 

Input: 
-----
tsManagerMessage - a pointer to the message that contains the PD of the BAR
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerBarConfirmed(K_MSG *tsManagerMessage)
{
	TxPd_t *packetDescriptor = NULL;
	TsManagerBarConfirmedMessage_t *tsManagerMessageParameters = NULL;

	tsManagerMessageParameters = ((TsManagerBarConfirmedMessage_t *)tsManagerMessage->abData);
    packetDescriptor = tsManagerMessageParameters->packetDescriptor;

	ILOG0_DD("tsManagerBarConfirmed, staId=%d, tid=%d", packetDescriptor->txQStaId,packetDescriptor->txQTid);
	
	TsManagerInitiator_BarConfirmed(packetDescriptor);

	heGroupManagerTSManagerEnableTid(packetDescriptor->txQStaId, packetDescriptor->txQTid);  /*  Enable HE MU on the correspond TID, We disabled it following BAR transmission , 
																								as we musn't send BAR as part of HE MU sequence */
    /* Release the PD */
    ResourceManager_ReleaseDescriptor((TxPd_t *)packetDescriptor, DESC_POOL_TS_MANAGER);
}

/**********************************************************************************

tsManagerBarStatusReceived 


Description:
------------
handles received BAR status

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and the TID	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerBarStatusReceived(K_MSG *tsManagerMessage)
{
	TxPd_t *packetDescriptor = NULL;
	TsManagerBarStatusMessage_t *tsManagerMessageParameters = NULL;

	tsManagerMessageParameters = ((TsManagerBarStatusMessage_t *)tsManagerMessage->abData);
	packetDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(tsManagerMessageParameters->pdOffset);

	packetDescriptor->packetPointer = (CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(packetDescriptor->packetPointer));
	
	DEBUG_ASSERT(PD_TYPE_BAR == packetDescriptor->pdType);
	
	if(tsManagerMessageParameters->isStatusAck)
	{
		packetDescriptor->status = PD_STATUS_ACK_RECEIVED;
	}
    else
    {
		packetDescriptor->status = PD_STATUS_BLOCK_ACK_RECEIVED;
    }

	heGroupManagerTSManagerEnableTid(packetDescriptor->txQStaId, packetDescriptor->txQTid);  /*  Enable HE MU on the correspond TID, We disabled it fowwling BAR transmission , as we musn't send BAR as part of HE MU sequence */

	ILOG0_DD("tsManagerBarStatusReceived, staId=%d, tid=%d", packetDescriptor->txQStaId,packetDescriptor->txQTid);

	TsManagerInitiator_BarStatus(packetDescriptor);

    /* Release the PD */
    ResourceManager_ReleaseDescriptor((TxPd_t *)packetDescriptor, DESC_POOL_TS_MANAGER);
}


/**********************************************************************************

tsManagerDataPacketDiscarded 


Description:
------------
handles a data packet that was discarded (only in the initiator)

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and the TID	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerDataPacketDiscarded(K_MSG *tsManagerMessage)
{
    TsManagerDataPacketDiscardedMessage_t *tsManagerMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 tid = 0;
	
    tsManagerMessageParameters = ((TsManagerDataPacketDiscardedMessage_t *)tsManagerMessage->abData);
	stationIndex = tsManagerMessageParameters->stationId;
    tid  = tsManagerMessageParameters->tid;
	TsManagerInitiator_DiscardedPacket(stationIndex, tid);
}

/**********************************************************************************

tsManagerDataPacketDiscarded 


Description:
------------
handles a data packet that was discarded (only in the initiator)

Input: 
-----
tsManagerMessage - pointer to the message that contains the station id and the TID	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerDataPacketDiscardedBitmap(K_MSG *tsManagerMessage)
{
    TsManagerDataPacketDiscardedBitmapMessage_t *tsManagerMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 tid = 0;
	uint8 locationInBitmap;
	uint32 tempBitmap = 0;
	
    tsManagerMessageParameters = ((TsManagerDataPacketDiscardedBitmapMessage_t *)tsManagerMessage->abData);
    tid  = tsManagerMessageParameters->tid;
	for (locationInBitmap = 0; locationInBitmap < TS_MANAGER_STA_DISCARD_BITMAP_SIZE; locationInBitmap++)
	{
		tempBitmap = tsManagerMessageParameters->stationBitmap[locationInBitmap];
		while (tempBitmap)
		{
			stationIndex = Utils_FindFirstSetAndClear(&tempBitmap);
			stationIndex += CONVERT_WORDS_TO_BIT_INDEX(locationInBitmap);
			TsManagerInitiator_DiscardedPacket(stationIndex, tid);
		}
	}
}


/**********************************************************************************

tsManagerTimerExpired 


Description:
------------
handles a timer event

Input: 
-----
tsManagerMessage 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerTimerExpired(K_MSG *tsManagerMessage)
{
	UNUSED_PARAM(tsManagerMessage);	
	TsManagerInitiator_TimerExpired();
	TsManagerRecipient_TimerExpired();	
	OSAL_SET_TIMER_EXPLICIT(TS_MANAGER_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(TS_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_TS_MANAGER);
}

/**********************************************************************************

tsManagerPdAllocated 


Description:
------------
handles 'Pd allocated' event from the resource manager

Input: 
-----
tsManagerMessage - pointer to the message 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerPdAllocated(K_MSG *tsManagerMessage)
{
	RmPdFreeDescResponse_t *tsManagerMessageParameters = ((RmPdFreeDescResponse_t *)tsManagerMessage->abData);
	uint16 context = tsManagerMessageParameters->context;
	TsManagerRmType_e type = (TsManagerRmType_e)TS_MANAGER_RM_REQUEST_EXTRACT_TYPE(context);
	StaId stationIndex = TS_MANAGER_RM_REQUEST_EXTRACT_SID(context);
	uint8 tid = TS_MANAGER_RM_REQUEST_EXTRACT_TID(context);

	if(TS_MANAGER_RM_REQUEST_INITIATOR == type)
	{
		TsManagerInitiator_PdAllocated(stationIndex, tid, tsManagerMessageParameters->packetDescriptor);
	}
	else
	{
		TsManagerRecipient_PdAllocated(stationIndex, tid, tsManagerMessageParameters->packetDescriptor);
	}
}

/**********************************************************************************

tsManagerIllegalPacketReceived 


Description:
------------
Handles the event that illegal packet was received - BAR or aggregation were received
and BA agreement doesnt exist

Input: 
-----
tsManagerMessage 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerIllegalPacketReceived(K_MSG *tsManagerMessage)
{
	TsManagerRecipient_IlegalPacketReceived(tsManagerMessage);
}

/**********************************************************************************

tsManagerCloseTids 


Description:
------------
handles 'close TID' event

Input: 
-----
tsManagerMessage - pointer to the message 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerCloseTids(K_MSG *tsManagerMessage)
{
	TsManagerCloseTidMessage_t *tsManagerMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 tidsBitmap = 0;
	uint8 initiator = FALSE;

	tsManagerMessageParameters = ((TsManagerCloseTidMessage_t *)tsManagerMessage->abData);
	stationIndex = tsManagerMessageParameters->stationId;
	tidsBitmap = tsManagerMessageParameters->tidsBitmap;
	initiator = tsManagerMessageParameters->initiator;
	if(initiator)
	{
		TsManagerInitiator_CloseTids(stationIndex, tidsBitmap);
	}
	else
	{
		TsManagerRecipient_CloseTids(stationIndex, tidsBitmap);
	}
}

/**********************************************************************************

tsManagerOpenTids 


Description:
------------
handles 'open TIDs' event

Input: 
-----
tsManagerMessage - pointer to the message 	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerOpenTids(K_MSG *tsManagerMessage)
{
	TsManagerOpenTidsMessage_t *tsManagerMessageParameters = NULL;
	StaId stationIndex = 0;
	uint8 tidsBitmap = 0;
	uint8 allowed;

	tsManagerMessageParameters = ((TsManagerOpenTidsMessage_t *)tsManagerMessage->abData);
	stationIndex = tsManagerMessageParameters->stationId;
	tidsBitmap = tsManagerMessageParameters->tidsBitmap;
	allowed = (TsManagerStaDb[stationIndex].isStaHt | TsManagerStaDb[stationIndex].isStaVht |  TsManagerStaDb[stationIndex].isStaHe);

	if ((allowed == TRUE) && (TsManager_GlobalDb.enableBa == TRUE))
	{
		TsManagerInitiator_OpenTids(stationIndex, tidsBitmap);
	}
}

/**********************************************************************************

tsManagerQueueIsLocked 


Description:
------------
handles 'queue is locked' event (only in the initiator)

Input: 
-----
tsManagerMessage - pointer to the message that contains the station index and the tid	


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void tsManagerQueueIsLocked(K_MSG *tsManagerMessage)
{
	LockReqCb_t* lockReqCb;
	StaId stationIndex = 0;
	uint8 tid = 0;

	lockReqCb = ((LockReqCb_t *)tsManagerMessage->abData);
	stationIndex = lockReqCb->stationOrVapNum;
	tid = lockReqCb->tid;
	TsManagerInitiator_QueueIsLocked(stationIndex, tid, lockReqCb->lockStatus);
}

/**********************************************************************************

tsManager_StaManagerSendConfirm 


Description:
------------
Sends Confirmation message to STA manager

Input: 
-----


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManager_StaManagerSendConfirm(StaId sid)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_TS_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = sid;
	/*Send confirmation message*/
	ILOG2_DD("Ts Manager, Send Confirmation, Client %d, SID %d", confirmMessage->clientId, sid);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, VAP_ID_DO_NOT_CARE);
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

TsManagerConvertWinSizeToBufferSize



Description:
------------
This function converrts window size to buffer size encoding

Input:
-----
Window Size


		
Output:
-------
Encoded Buffer Size

Returns:
--------
	void - 
	
**********************************************************************************/
uint8 TsManagerConvertWinSizeToBufferSize(uint16 winSize)
{
	uint8 bufferSize = 0;


	if (winSize >= TS_MANAGER_WINDOW_SIZE_256)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_256;
	}

#ifndef BLOCK_ACK_AGREEMENT_PREVENT_BUFFER_SIZE_BETWEEN_65_128 // this #ifndef should be removed once https://jira-chd.intel.com/browse/WLANVLSIIP-2852 is resolved in B0.the code inside the #ifndef should remain.

	else if (winSize >= TS_MANAGER_WINDOW_SIZE_128)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_128;
	}
#endif //BLOCK_ACK_AGREEMENT_PREVENT_BUFFER_SIZE_BETWEEN_65_128

	else if (winSize >= TS_MANAGER_WINDOW_SIZE_64)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_64;
	}
	else if (winSize >= TS_MANAGER_WINDOW_SIZE_32)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_32;
	}
	else if (winSize >= TS_MANAGER_WINDOW_SIZE_16)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_16;
	}
	else if (winSize >= TS_MANAGER_WINDOW_SIZE_8)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_8;
	}
    else if (winSize >= TS_MANAGER_WINDOW_SIZE_4)
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_4;
	}
	else
	{
		bufferSize = TS_MANAGER_BUFFER_SIZE_2;
	}

	return (bufferSize);
}


/**********************************************************************************

TsManager_TransmitAddbaRequest


Description:
------------
This function transmits ADDBA request

Input:
-----
stationTidEntry - pointer to the station/TID entry
pdPointer - pointer to the PD that was allocated for transmission
timeout - the timeout the initiator transmits in calendar wheel units



		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_TransmitAddbaRequest(TsManagerInitiatorStationTidParams_t *stationTidEntry, TxPd_t *pdPointer, uint16 timeout)
{
	sHT_BA_ParameterSet 				baParametersSet;
	AddbaRequestFrame_t*				addbaRequestFrame = NULL;
	FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT	addbaExtensionIe;
	AddbaCapabilities_t*				addbaCapabilities;
	StaId 								stationIndex = 0;
	uint8 								vapIndex = 0;
	uint8								isStaHe;
#ifdef WORKAROUND_FOR_HW_BUG_IN_ADDBA_REQ_ENCRYPTED
	uint8 								mfp = TsManagerStaDb[stationTidEntry->commonPart.stationId].isStaMfp;
	uint8 								isFilterOpen = TsManagerStaDb[stationTidEntry->commonPart.stationId].isXFilterOpen;
#endif

	addbaRequestFrame = (AddbaRequestFrame_t *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	memset(&baParametersSet, 0, sizeof(sHT_BA_ParameterSet));
	memset(&addbaRequestFrame->addbaRequestPayload, 0, sizeof(FM_PAYLOAD_ADDBA_REQ)); //TBD check if needed
	
	stationIndex = stationTidEntry->commonPart.stationId;
	isStaHe = TsManagerStaDb[stationIndex].isStaHe;

	/* The parameters that are byte long are written directly to the SHRAM */
	addbaRequestFrame->addbaRequestPayload.u8CategoryCode = BLOCK_ACK_CATEGORY;
	addbaRequestFrame->addbaRequestPayload.u8ActionCode = BA_AGREEMENT_ACTION_CODE_ADDBA_REQUEST;
	addbaRequestFrame->addbaRequestPayload.u8DialogToken = stationTidEntry->commonPart.dialogToken;

	/* The parameters that are bit fields are copied first to internal vairable and then copied to the SHRAM
	   this is done in order to avoid many manipulation on the SHRAM */
	baParametersSet.BlockAckAmsduSupported = stationTidEntry->commonPart.amsduSupport;
	baParametersSet.BlockAckPolicy = IMMEDIATE_BA;
	baParametersSet.BlockAckTid = stationTidEntry->commonPart.tid;
	baParametersSet.BlockAckBufferSize = TsManagerBufferSizeToWindowSize[stationTidEntry->commonPart.bufferSize];
    MEMCPY((void *)&(addbaRequestFrame->addbaRequestPayload.u16BlockAckParameterSet), &baParametersSet, sizeof(sHT_BA_ParameterSet));

	if (stationTidEntry->commonPart.deleteBaAgreementOnInactivity == FALSE)
	{
		addbaRequestFrame->addbaRequestPayload.u16BlockAckTimeOut = TS_MANAGER_INFINITE_TIMEOUT;	
	}
	else
	{
		addbaRequestFrame->addbaRequestPayload.u16BlockAckTimeOut = TS_MANAGER_CONVERT_CALENDAR_WHEEL_UNITS_INTO_TIME_UINTS(timeout);	
	}

	ILOG0_D("[TsManager_TransmitAddbaRequest], isStaHe = %d", isStaHe);
	// Check if this STA supprts HE. If so - we want to add ADDBA Extension IE.
	if (isStaHe == TRUE)
	{
		addbaExtensionIe.u8ElementId = FM_EL_ADDBA_EXTENSION;
		addbaExtensionIe.u8Length = sizeof(AddbaCapabilities_t);
		addbaExtensionIe.u8AddbaCapabilities = 0;
		addbaCapabilities = (AddbaCapabilities_t*)&addbaExtensionIe.u8AddbaCapabilities;

		// We do not support non-HE fragmentation. for HE STAs - it's don't care.
		addbaCapabilities->no_fragmentation = TS_MANAGER_NON_HE_FRAGMENTED_MSDUS_DISABLED;

		// Use dynamic fragmentation level for HE Fragmentation Operation subfield:  
		addbaCapabilities->he_fragmentation_operation = TS_MANAGER_INITIATOR_DYNAMIC_FRAG_LEVEL_SUPPORTED_0;

		// Copy to SHRAM
		MEMCPY((void *)&addbaRequestFrame->AddbaExtension, &addbaExtensionIe, sizeof(FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT));

		// Set data length with ADDBA extension
		pdPointer->dataLength = sizeof(AddbaRequestFrame_t) + frame_sizeOfNewManagementFrameHeader(FALSE);
	}
	else
	{
		// Set data length without ADDBA extension
		pdPointer->dataLength = sizeof(AddbaRequestFrame_t) + frame_sizeOfNewManagementFrameHeader(FALSE) - sizeof(FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT);
	}

	pdPointer->txQStaId= stationIndex;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	pdPointer->mgmtActionCode = BLOCK_ACK_CATEGORY;
	pdPointer->mgmtActionValue = BA_AGREEMENT_ACTION_CODE_ADDBA_REQUEST;
#ifdef WORKAROUND_FOR_HW_BUG_IN_ADDBA_REQ_ENCRYPTED
	// there is a bug in OTFA - it doesnt encrypt PD_TYPE_ADDBA_REQ. 
	// So we abuse a PD TYPE that we don't need (PD_TYPE_COLOR_SWITCH_ANNOUNCEMENT_ENC) in order for HW and Sender to use encryption.
	if ((mfp == TRUE) && (isFilterOpen == TRUE))
	{
		pdPointer->pdType = PD_TYPE_ADDBA_REQ_ENC;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_ADDBA_REQ;
	}
#else
		pdPointer->pdType = PD_TYPE_ADDBA_REQ;
#endif
	

	
	StaDb_GetVapId(stationIndex, &vapIndex);
	pdPointer->txQVapId = vapIndex;

	pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = UNICAST;

	TxPacketsClassifier_SendManagementPacketFromFw(pdPointer);
}


/**********************************************************************************

TsManager_TransmitAddbaResponse 


Description:
------------
This function transmits ADDBA response

Input:
-----
stationTidEntry - pointer to the station/TID entry
pdPointer - pointer to the PD that was allocated for transmission




		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_TransmitAddbaResponse(TsManagerRecipientStationTidParams_t *stationTidEntry, TxPd_t *pdPointer)
{
	sHT_BA_ParameterSet 				baParametersSet;
	FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT	addbaExtensionIe;
	AddbaCapabilities_t*				addbaCapabilities;
	AddbaResponseFrame_t*				addbaResponseFrame = NULL;
	StaId 								stationIndex = 0;
	uint8 								vapIndex = 0;
	uint8 								mfp = TsManagerStaDb[stationTidEntry->commonPart.stationId].isStaMfp;
	uint8 								isFilterOpen = TsManagerStaDb[stationTidEntry->commonPart.stationId].isXFilterOpen;


	addbaResponseFrame = (AddbaResponseFrame_t *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	memset(&baParametersSet, 							0, sizeof(sHT_BA_ParameterSet));
	memset(&addbaResponseFrame->addbaResponsePayload,	0, sizeof(FM_PAYLOAD_ADDBA_RES));//TBD check if needed
	memset(&addbaResponseFrame->AddbaExtension, 		0, sizeof(FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT));//TBD check if needed

	stationIndex = stationTidEntry->commonPart.stationId;

	/* The parameters that are byte long are written directly to the SHRAM */  
	addbaResponseFrame->addbaResponsePayload.u8CategoryCode = BLOCK_ACK_CATEGORY;
	addbaResponseFrame->addbaResponsePayload.u8ActionCode = BA_AGREEMENT_ACTION_CODE_ADDBA_RESPONSE;
	addbaResponseFrame->addbaResponsePayload.u8DialogToken = stationTidEntry->commonPart.dialogToken;
	addbaResponseFrame->addbaResponsePayload.u16StatusCode = TsManagerStatusToStatusCode[stationTidEntry->commonPart.status];

	/* The parameters that are bit fields are copied first to internal vairable and then copied to the SHRAM
	   this is done in order to avoid many manipulation on the SHRAM */
	baParametersSet.BlockAckAmsduSupported = stationTidEntry->commonPart.amsduSupport;
	baParametersSet.BlockAckPolicy = IMMEDIATE_BA;
	baParametersSet.BlockAckTid = stationTidEntry->commonPart.tid;
	baParametersSet.BlockAckBufferSize = TsManagerBufferSizeToWindowSize[stationTidEntry->commonPart.bufferSize];
    MEMCPY((void *)&addbaResponseFrame->addbaResponsePayload.u16BlockAckParameterSet, &baParametersSet, sizeof(sHT_BA_ParameterSet));
	
	if (stationTidEntry->commonPart.deleteBaAgreementOnInactivity == FALSE)
	{
		addbaResponseFrame->addbaResponsePayload.u16BlockAckTimeOut = TS_MANAGER_INFINITE_TIMEOUT;
	}
	else
	{
		addbaResponseFrame->addbaResponsePayload.u16BlockAckTimeOut = TS_MANAGER_CONVERT_CALENDAR_WHEEL_UNITS_INTO_TIME_UINTS(stationTidEntry->commonPart.timeout);
	}

	// Check if ADDBA Extension IE has to be added (we add it if it was used in the ADDBA REQ)
	if (stationTidEntry->fragSupportIndicated == TRUE)
	{
		addbaExtensionIe.u8ElementId = FM_EL_ADDBA_EXTENSION;
		addbaExtensionIe.u8Length = sizeof(AddbaCapabilities_t);
		addbaExtensionIe.u8AddbaCapabilities = 0;
		addbaCapabilities = (AddbaCapabilities_t*)&addbaExtensionIe.u8AddbaCapabilities;

		// We use the same value the STA used for non-HE fragmentation support. for HE STAs - it's don't care.
		// addbaCapabilities->no_fragmentation = !stationTidEntry->nonHeFragsupport;
		/* In A0 we do not support reception of fragmentation */
		addbaCapabilities->no_fragmentation = TS_MANAGER_NON_HE_FRAGMENTED_MSDUS_DISABLED;

		// For dynamic fragmentation level we choose the lower value between our support level and the STAs support level
		addbaCapabilities->he_fragmentation_operation = MIN(stationTidEntry->nonHeFragsupport, TS_MANAGER_RECIPIENT_DYNAMIC_FRAG_LEVEL_SUPPORTED_0);

		// Copy to SHRAM
		MEMCPY((void *)&addbaResponseFrame->AddbaExtension, &addbaExtensionIe, sizeof(FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT));

		// Set data length with ADDBA extension
		pdPointer->dataLength = sizeof(AddbaResponseFrame_t) + frame_sizeOfNewManagementFrameHeader(FALSE);
	}
	else
	{
		// Set data length without ADDBA extension
		pdPointer->dataLength = sizeof(AddbaResponseFrame_t) + frame_sizeOfNewManagementFrameHeader(FALSE) - sizeof(FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT);
	}
	
		
	pdPointer->txQStaId = stationIndex;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	pdPointer->mgmtActionCode = BLOCK_ACK_CATEGORY;
	pdPointer->mgmtActionValue = BA_AGREEMENT_ACTION_CODE_ADDBA_RESPONSE;
	if ((mfp == TRUE) && (isFilterOpen == TRUE))
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_ENC;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
	}
	StaDb_GetVapId(stationIndex, &vapIndex);
	pdPointer->txQVapId = vapIndex;

	pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = UNICAST;
	
	TxPacketsClassifier_SendManagementPacketFromFw(pdPointer);
}

/**********************************************************************************

TsManager_TransmitDelba 


Description:
------------
This function transmits DELBA

Input:
-----
stationTidEntry - pointer to the station/TID entry
pdPointer - pointer to the PD that was allocated for transmission


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_TransmitDelba(TsManagerCommonParameters_t *stationTidEntry, TxPd_t *pdPointer)
{
	sHT_DELBA_ParameterSet delbaParametersSet;
	FM_PAYLOAD_DELBA *delbaFramePayload = NULL;
	StaId stationIndex = 0;
	uint8 vapIndex = 0;
	uint8 mfp = TsManagerStaDb[stationTidEntry->stationId].isStaMfp;
	uint8 isFilterOpen = TsManagerStaDb[stationTidEntry->stationId].isXFilterOpen;
	
	delbaFramePayload = (FM_PAYLOAD_DELBA *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
	memset(&delbaParametersSet, 0, sizeof(sHT_DELBA_ParameterSet));
	memset(delbaFramePayload, 0, sizeof(FM_PAYLOAD_DELBA));//to check if needed

    stationIndex = stationTidEntry->stationId;

	/* The parameters that are byte long are written directly to the SHRAM */  
	delbaFramePayload->u8CategoryCode = BLOCK_ACK_CATEGORY;
	delbaFramePayload->u8ActionCode = BA_AGREEMENT_ACTION_CODE_DELBA;

	/* The parameters that are bit fields are copied first to internal vairable and then copied to the SHRAM
	   this is done in order to avoid many manipulation on the SHRAM */
	delbaParametersSet.BlockAckInitiator = stationTidEntry->entryType;	
	
	delbaParametersSet.BlockAckTid = stationTidEntry->tid;
	MEMCPY(&delbaFramePayload->u16DelBaParameterSet, &delbaParametersSet, sizeof(sHT_DELBA_ParameterSet));
	delbaFramePayload->u16ResonCode = TsManagerReasonToReasonCode[stationTidEntry->reason];
			
	pdPointer->dataLength = sizeof(FM_PAYLOAD_DELBA) + frame_sizeOfNewManagementFrameHeader(FALSE);	
	pdPointer->txQStaId = stationIndex;
	pdPointer->txQTid = MANAGEMENT_TID;
	pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
	pdPointer->mgmtActionCode = BLOCK_ACK_CATEGORY;
	pdPointer->mgmtActionValue = BA_AGREEMENT_ACTION_CODE_DELBA;
	if ((mfp == TRUE) && (isFilterOpen == TRUE))
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_ENC;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
	}
	StaDb_GetVapId(stationIndex, &vapIndex);
	pdPointer->txQVapId = vapIndex;

		
	pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = UNICAST;

	//("TsManager_TransmitDelba: stationIndex=%d tid=%d",stationTidEntry->stationId,stationTidEntry->tid);

	TxPacketsClassifier_SendManagementPacketFromFw(pdPointer);
}

/**********************************************************************************

TsManager_TransmitBar 


Description:
------------
This function transmits BAR

Input:
-----
stationTidEntry - pointer to the station/TID entry
pdPointer - pointer to the PD that was allocated for transmission


		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_TransmitBar(TsManagerCommonParameters_t *stationTidEntry, TxPd_t *pdPointer)
{
	BarPayload_t barPayload;
	BarFrame_t *barFrame = NULL;
	uint8 tid = 0;
	StaId stationIndex = 0;
	uint8 vapIndex = 0;
	
	barFrame = (BarFrame_t *)pdPointer->packetPointer;
	memset(&barPayload, 0, sizeof(BarPayload_t));
	memset(&barFrame->barPayload, 0, sizeof(BarPayload_t));//to check if needed

	tid = stationTidEntry->tid;
	stationIndex = stationTidEntry->stationId; 

	/* The parameters that are bit fields are copied first to internal vairable and then copied to the SHRAM
	   this is done in order to avoid many manipulation on the SHRAM */
	barPayload.compressed_bitmap = TRUE;
	barPayload.TID = tid;
	MEMCPY(&barFrame->barPayload, &barPayload, sizeof(BarPayload_t));
		
	pdPointer->dataLength = sizeof(BarFrame_t);
	pdPointer->txQTid = tid;
	pdPointer->txQStaId = stationIndex;

	pdPointer->pdType = PD_TYPE_BAR;
	StaDb_GetVapId(stationIndex, &vapIndex);
	pdPointer->txQVapId = vapIndex;

	pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
	pdPointer->mcUnicast = UNICAST;

	//("TsManager_TransmitBar: stationIndex=%d tid=%d, pdPointer:0x%x",stationTidEntry->stationId,stationTidEntry->tid, pdPointer);

	ILOG0_DD("TsManager_TransmitBar, staId=%d, TID=%d",stationIndex ,tid);

	heGroupManagerTSManagerDisableTid(stationIndex, tid);  //Disbale HE MU on the correspond TID, as we musn't send BAR as part of HE MU sequence 

	TxPacketsClassifier_SendBar(pdPointer);
}



/**********************************************************************************

TsManager_TidMovedToIdleCallback 


Description:
------------
decrease number of active TIDs of the station and checks if the removing 
proccess has ended

Input: 
-----
stationIndex - station index


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void TsManager_TidMovedToIdleCallback(StaId stationIndex)
{
	TsManagerStaDb[stationIndex].activeTids-- ;
	if(TsManagerStaDb[stationIndex].activeTids == 0)
	{
		tsManager_StaManagerSendConfirm(stationIndex);
	}
}


/**********************************************************************************

TsManager_SetTimer 


Description:
------------
set the timer of the TS calendar wheel

Input: 
-----
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_SetTimer(void)
{
	/* Start TS calendar wheel timer */
	OSAL_SET_TIMER_EXPLICIT(TS_MANAGER_TIMER_EXPIRED, OSAL_TIMERS_MS_TO_K_TICKS(TS_MANAGER_CALENDAR_WHEEL_TIMER_IN_MILLISECONDS), TASK_TS_MANAGER);
}


/**********************************************************************************

TsManager_ConfigureVapParameters 


Description:
------------
Configures VAP parameters

Input: 
-----
vapIndex - the VAP index to configure 
vapConfigurationParameters - a pointer to the structure of the parameters
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_ConfigureVapParameters(uint8 vapIndex, TsManagerVapConfigurationParams_t *vapConfigurationParameters)
{
	TsManagerRecipient_ConfigureVapParameters(vapIndex, &vapConfigurationParameters->recipientVapParameters);
	TsManagerInitiator_ConfigureVapParameters(vapIndex, &vapConfigurationParameters->initiatorVapParameters);
}

/**********************************************************************************

TsManager_CloseOrOpenStationBaAgreements 


Description:
------------
Close or open specific station BA agreements

Input: 
-----
stationIndex - the station to configure
direction - initiator or recipient
allowed - a flag that indicates if to close or to open BA agrrements
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_CloseOrOpenStationBaAgreements(StaId stationIndex, uint8 direction, uint8 allowed)
{
	TsManagerCloseTidMessage_t *tsManagerCloseTidsMessageParameters = NULL;
	TsManagerOpenTidsMessage_t *tsManagerOpenTidsMessageParameters = NULL;
	K_MSG* tsManagerMessage = NULL; 
	
	DEBUG_ASSERT(direction < BA_AGREEMENT_NUM_OF_DIRECTIONS);
	DEBUG_ASSERT(allowed < BA_AGREEMENT_NUM_OF_ALLOWED_VALUES);
    DEBUG_ASSERT(stationIndex < HW_NUM_OF_STATIONS);	
	
	if(BA_AGREEMENT_ALLOWED_VALUE_NOT_ALLOWED == allowed)
	{
		if(BA_AGREEMENT_DIRECTION_RECIPIENT == direction)
		{
			TsManager_ChangeRecipientStationAllowed(stationIndex, BA_AGREEMENT_ALLOWED_VALUE_NOT_ALLOWED);
		}
		tsManagerMessage = OSAL_GET_MESSAGE(sizeof(TsManagerCloseTidMessage_t));
		tsManagerCloseTidsMessageParameters = ((TsManagerCloseTidMessage_t *)tsManagerMessage->abData);
		tsManagerCloseTidsMessageParameters->stationId = stationIndex;
		tsManagerCloseTidsMessageParameters->tidsBitmap = TID_BITMAP_ALL_DATA_TIDS;
		tsManagerCloseTidsMessageParameters->initiator = direction;
		OSAL_SEND_MESSAGE(TS_MANAGER_CLOSE_TIDS, TASK_TS_MANAGER, tsManagerMessage, VAP_ID_DO_NOT_CARE);	
	}
	else
	{
		if(BA_AGREEMENT_DIRECTION_INITIATOR == direction)
		{
			tsManagerMessage = OSAL_GET_MESSAGE(sizeof(TsManagerOpenTidsMessage_t));
			tsManagerOpenTidsMessageParameters = ((TsManagerOpenTidsMessage_t *)tsManagerMessage->abData);
			tsManagerOpenTidsMessageParameters->stationId = stationIndex;
			tsManagerOpenTidsMessageParameters->tidsBitmap = TID_BITMAP_ALL_DATA_TIDS;
			OSAL_SEND_MESSAGE(TS_MANAGER_OPEN_TIDS, TASK_TS_MANAGER, tsManagerMessage, VAP_ID_DO_NOT_CARE);	
		}
		else
		{
			TsManager_ChangeRecipientStationAllowed(stationIndex, BA_AGREEMENT_ALLOWED_VALUE_ALLOWED);
		}
	}
}

/**********************************************************************************

TsManager_RemoveFromCalendarWheel


Description:
------------
This function removes an entry from the calendar wheel. This function is used
for debug purposes 

Input: 
-----
calendarWheel - the address of the calendar wheel
entry - the address of the enrty to be removed


		
Output:
-------
	

Returns:
--------
	 
	
**********************************************************************************/
void TsManager_RemoveFromCalendarWheel (CalendarWheelParameters_t *calendarWheel, TsManagerCommonParameters_t *entry)
{
	DEBUG_ASSERT(entry->entryState == TS_MANAGER_CALENDAR_WHEEL_ENTRY_ENQUEUED);
	entry->entryState = TS_MANAGER_CALENDAR_WHEEL_ENTRY_FREE;
		
	CalendarWheel_RemoveEntry(calendarWheel, (CalendarWheelEntry_t *)entry);
}

/**********************************************************************************

TsManager_AddToCalendarWheel


Description:
------------
This function adds an entry to the calendar wheel This function is usedfor debug 
purposes 

Input: 
-----
calendarWheel - the address of the calendar wheel
entry - the address of the enrty to be added
timeout - the number of slots need to be count


		
Output:
-------
	

Returns:
--------
	 
	
**********************************************************************************/
void TsManager_AddToCalendarWheel (CalendarWheelParameters_t *calendarWheel, TsManagerCommonParameters_t *entry,uint16 timeout)
{
	DEBUG_ASSERT(entry->entryState == TS_MANAGER_CALENDAR_WHEEL_ENTRY_FREE);
	entry->entryState = TS_MANAGER_CALENDAR_WHEEL_ENTRY_ENQUEUED;
	
	CalendarWheel_AddEntry(calendarWheel, (CalendarWheelEntry_t *)entry, timeout);		
}

/**********************************************************************************

TsManager_RescheduleInCalendarWheel


Description:
------------
This function reschedules an entry in the calendar wheel This function is used for 
debug purposes 

Input: 
-----
calendarWheel - the address of the calendar wheel
entry - the address of the enrty to be rescheduled
timeout - the number of slots need to be count



		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
void TsManager_RescheduleInCalendarWheel (CalendarWheelParameters_t *calendarWheel, TsManagerCommonParameters_t *entry,uint16 timeout)
{
	CalendarWheel_RescheduleEntry(calendarWheel, (CalendarWheelEntry_t *)entry, timeout);		
}


/**********************************************************************************

TsManager_TaskEntry 


Description:
------------
the entry point of the TS manager task

Input: 
-----
tsManagerMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_TaskEntry(K_MSG *tsManagerMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(tsManagerMessage, afpTaskTable, TASK_TS_MANAGER_START, TASK_TS_MANAGER_END);
}


/**********************************************************************************

TsManager_PostInit 


Description:
------------
Post Initialization of TS Manager

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_PostInit()
{
	/*	Set the TS manager timer */
	TsManager_SetTimer(); 
}


bool TsManager_IsStaHe(StaId staId)
{
	DEBUG_ASSERT(staId < HW_NUM_OF_STATIONS);
	
	return TsManagerStaDb[staId].isStaHe;
}



/**********************************************************************************

TsManager_Initialize 


Description:
------------
initialize the TS manager initiator and TS manager recipient internal databases and structures

Input:
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManager_Initialize()
{
	uint8 vapIndex, tid;
	
	TsManagerInitiator_Initialize();
	TsManagerRecipient_Initialize();

	/* Initialize Sta DB*/		
	memset(TsManagerStaDb, 0, sizeof(TsManagerStaDb));

	memset(&TsManager_GlobalDb, 0, sizeof(TsManagerGlobalDb_t));
	TsManager_GlobalDb.enableBa = TRUE;
	TsManager_GlobalDb.ppduTxModeNoAmpdu = STA_DB_PPDU_TX_MODE_LEGACY_ONLY;
	
	/*Initialze the PD descriptors used for TS*/
	TsManagerPdBasePtr = (uint32 *)&PdDescPools.TsManDescPool;

	/*Initialize TS manager configuration*/
	for (vapIndex = 0 ; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		for(tid = 0; tid < NUM_OF_TID; tid++)
		{
			TsManagerVapConfiguration[vapIndex].recipientVapParameters.amsduSupportPerTidArray[tid] = TRUE;
			TsManagerVapConfiguration[vapIndex].initiatorVapParameters.amsduSupportPerTidArray[tid] = TRUE;
			TsManagerVapConfiguration[vapIndex].recipientVapParameters.maxBufferSizePerTidArray[tid] = TS_MANAGER_MAX_BUFFER_SIZE;
			TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxBufferSizePerTidArray[tid] = TS_MANAGER_MAX_BUFFER_SIZE;
		}
		TsManagerVapConfiguration[vapIndex].recipientVapParameters.maxBaAgreementsPerVap = TS_MANAGER_MAX_BA_AGREEMENT_RECIPIENT_DEFAULT_VALUE;
		TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxBaAgreementsPerVap = TS_MANAGER_MAX_BA_AGREEMENT_INITIATOR_DEFAULT_VALUE;
		TsManagerVapConfiguration[vapIndex].recipientVapParameters.maxInactivityTimeout = 4096;
		TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxInactivityTimeout = 4096;
		TsManagerVapConfiguration[vapIndex].recipientVapParameters.maxBaAgreementsPerStation = NUM_OF_TID;
		TsManagerVapConfiguration[vapIndex].initiatorVapParameters.maxBaAgreementsPerStation = NUM_OF_TID;
		TsManagerVapConfiguration[vapIndex].initiatorVapParameters.addbaRequestTimeout = 500;
		TsManagerVapConfiguration[vapIndex].initiatorVapParameters.deleteBaAgreementOnInactivity = FALSE;
	}
}

