/***********************************************************************************
 File:			TsManagerInitiator.c
 Module:		Ts Manager
 Purpose: 		To handle BA agreements
 Description:   This file is the implementation of the Ts manager initiator which is responsible 
 				of handling BA agreements from the initiator side
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "System_MainApi.h"
#include "TsManagerInitiator.h"
#include "TsManager_API.h"
#include "TsManager.h"
#include "PacketDescriptor.h"
#include "database.h"
#include "HwGlobalDefinitions.h"
#include "ResourceManager_API.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"
#include "stringLibApi.h"
#include "mt_sysdefs.h"
#include "RxMpduStructure.h"
#include "HwMemoryMap.h"
#include "StaDatabase_Api.h"
#include "Utils_Api.h"
#include "ShramPacketDescriptors.h"
#include "HwQManager_API.h"
#include "Pac_Api.h"
#include "Locker_Api.h"
#include "shramTxDesc.h"
#include "BaAnalyzer_Api.h"
#include "Statistics_Descriptors.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TS_MANAGER 
#define LOG_LOCAL_FID 3


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
/*To serialize Lock requests we reuse the PD request ID field*/
#define TS_MANAGER_INITIATOR_CALC_LOCK_INDEX(sta, tid)		(sta << 3 | tid)
#define TS_MANAGER_INITIATOR_GET_STA_FROM_LOCK_IDX(idx)		(idx >> 3)
#define TS_MANAGER_INITIATOR_GET_TID_FROM_LOCK_IDX(idx)		(idx & 0x7)
#define TS_MANAGER_INITIATOR_INVALID_LOCK_IDX				(0xFFF)

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/
typedef void (*TsManagerInitiatorGeneralFunc)(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void tsManagerInitiatorIgnoreEvent(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorFatal(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorChangeState(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerInitiatorTidState_e newState);
static void tsManagerInitiatorLogEvent(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerInitiatorTidEvent_e event);
static bool tsManagerInitiatorRmRequest(TsManagerInitiatorStationTidParams_t *TsDb, RmPdRequestFillParameters_t *pdRequestFillParameters);
static bool tsManagerInitiatorRmRemove(TsManagerInitiatorStationTidParams_t *TsDb);
/* Remove functions */
static void tsManagerInitiatorMoveToIdle(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInTryingToOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInTryingToOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInTryingSendingAddbaRequest(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInTryingWaitingForConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorMoveToRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInTryingToOpenCountingTimeout(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInOpenWaitingForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInClosingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorRemoveInClosingLockBeforeSendingDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* ADDBA response received functions */
static void tsManagerInitiatorAddbaResponseRecievedInTryingSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaResponseRecievedInTryingCounting(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* DELBA received functions */
static void tsManagerInitiatorDelbaRecievedInTryingWaitConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInClosingWaitForEventToSend(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInClosingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorDelbaRecievedInClosingWaitLockBeforeDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* ADDBA request confirmation functions */
static void tsManagerInitiatorAddbaRequestConfirmedInTryingSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInOpenedSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInWaitingToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInWaitingBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorAddbaRequestConfirmedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* DELBA confirmation functions */
static void tsManagerInitiatorDelbaConfirmedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* BAR confirmation functions */
static void tsManagerInitiatorBarConfirmedInOpenedSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarConfirmedInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarConfirmedInClosingWaitForEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarConfirmedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* BAR status functions */
static void tsManagerInitiatorBarStatusInOpenedSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarStatusInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarStatusInClosingWaitForEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorBarStatusInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Pd allocated functions functions */
static void tsManagerInitiatorPdAllocatedInTryingWaitingPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPdAllocatedInOpenedWaitingPdToSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPdAllocatedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPdAllocatedInClosingWaitingForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPdAllocatedInClosingWaitingEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPdAllocatedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Timer event functions */
static void tsManagerInitiatorTimerInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorTimerInTryingCountingTimeout(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorTimerInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorTimerInOpenWaitingForPdToSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorTimerInOpenWaitingForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorTimerInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Data Packet discarded functions */
static void tsManagerInitiatorPacketDiscardedInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorPacketDiscardedInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Queue is locked functions */
static void tsManagerInitiatorQueueLockedInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorQueueLockedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorQueueLockedInClosingWaitForLockToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorQueueLockedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Close Tid functions */
static void tsManagerInitiatorCloseTidInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorCloseTidInTryingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorCloseTidInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorCloseTidInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorCloseTidInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorCloseTidInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* Open Tid functions */
static void tsManagerInitiatorSetAllowedFlag(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorOpenTidInClosed(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
static void tsManagerInitiatorOpenTidInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters);
/* General functions */
static void tsManagerInitiatorLockQueue(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiator_HandlePendingLock(void);
static void tsManagerInitiatorFlushIllegalPackets(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiatorTryToOpenBaAgreement(TsManagerInitiatorStationTidParams_t *TsDb, TxPd_t *packetDescriptor);
static void tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiatorSaveAddbaResponseParameters(TsManagerInitiatorStationTidParams_t *TsDb, FM_PAYLOAD_ADDBA_RES *addbaResponsePayload);
static void tsManagerInitiatorTryToAllocatePdForDelba(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiatorTryToAllocatePdForBar(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiatorBaAgreementOpenedUpdateCounters(StaId stationIndex);
static void tsManagerInitiatorBaAgreementClosedUpdateCounter(StaId stationIndex);
static uint8 tsManagerInitiatorIsAllowedToOpenBaAgreement(StaId stationIndex);
static void tsManagerInitiatorIncreaseTimeForNextRetry(TsManagerInitiatorStationTidParams_t *TsDb);
static void tsManagerInitiatortProcessAddbaResExtension(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerAddbaParams_t* addbaParams);


/*---------------------------------------------------------------------------------
/						External Variables									
/----------------------------------------------------------------------------------*/

extern TsManagerGlobalDb_t TsManager_GlobalDb;
extern PerClientStatistics_t PerClientStatistics;


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
static uint16 TsManagerInitiatorCalendarWheelSlots[TS_MANAGER_CALENDAR_WHEEL_SIZE];
static CalendarWheelParameters_t TsManagerInitiatorCalendarWheel;
static TsManagerInitiatorStationTidParams_t TsManagerInitiatorStationTidDatabase[HW_NUM_OF_STATIONS][NUM_OF_TID];
static uint8 TsManagerInitiatorStationOpenedBaAgreements[HW_NUM_OF_STATIONS];
static TsManagerInitiatorVapParams_t TsManagerInitiatorVapDatabase[HW_NUM_OF_VAPS];
static TsManagerInitiatorGlobalParams_t TsManagerInitiatorGlobalParameters;
static uint16 TsManagerInitiator_LockHead, TsManagerInitiator_LockTail;
/*Remove STA table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorRemoveTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                  , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorMoveToIdle                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorMoveToIdle                             , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorRemoveInTryingToOpen                   , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorRemoveInTryingToOpenWaitForPd          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorRemoveInTryingSendingAddbaRequest	  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorRemoveInTryingWaitingForConfirmation   , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorRemoveInTryingToOpenCountingTimeout    , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorRemoveInOpen                           , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorRemoveInOpenWaitForPd                  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorRemoveInOpenWaitingForLock             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorRemoveInOpenSendingBar                 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorMoveToRemoving                         , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorRemoveInClosingWaitForEventToSendDelba , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorRemoveInClosingWaitForPd               , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorRemoveInClosingLockBeforeSendingDelba  , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorMoveToRemoving                         , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorFatal                                  , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*ADDBA RSP Received table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorAddbaResponseReceivedTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorAddbaResponseRecievedInTryingSending	  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorAddbaResponseRecievedInTryingCounting  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorIgnoreEvent                            , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*DELBA Received table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorDelbaReceivedTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorIgnoreEvent	                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorDelbaRecievedInTryingWaitConfirmation      , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorDelbaRecievedInOpen                        , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorDelbaRecievedInOpenWaitForPd               , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorDelbaRecievedInOpenWaitForLock             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorDelbaRecievedInOpenSendingBar              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorDelbaRecievedInClosingWaitForEventToSend   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorDelbaRecievedInClosingWaitForPd            , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorDelbaRecievedInClosingWaitLockBeforeDelba  , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorIgnoreEvent                                , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*ADDBA REQ Confirmation Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorAddbaRequestConfirmationTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorAddbaRequestConfirmedInTryingSending	          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorAddbaRequestConfirmedInOpenedSendBar           , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorAddbaRequestConfirmedInClosing                 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorAddbaRequestConfirmedInWaitingToSendDelba      , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                          , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorAddbaRequestConfirmedInWaitingBeforeLock       , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorAddbaRequestConfirmedInRemoving                , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*DELBA Confirmation Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorDelbaConfirmationTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                         , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorDelbaConfirmedInClosing   		 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorFatal                             , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorMoveToIdle                		 , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*BAR Confirmation Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorBarConfirmationTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                                             , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorBarConfirmedInOpenedSending                   		 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorBarConfirmedInClosingWaitForEventToSendDelba 		 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                                 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorBarConfirmedInClosingWaitForEventBeforeLock  		 , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorBarConfirmedInRemoving                        		, /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*BAR Status Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorBarStatusTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                                          , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorBarStatusInOpenedSending                   		  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorBarStatusInClosingWaitForEventToSendDelba  		  , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorBarStatusInClosingWaitForEventBeforeLock   		  , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorBarStatusInRemoving                        		  , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*PD Allocation Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorPdAllocatedTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorPdAllocatedInTryingWaitingPd                 		, /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                                            , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorPdAllocatedInOpenedWaitingPdToSendBar        		, /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorPdAllocatedInClosing                         		, /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorPdAllocatedInClosingWaitingForPd             		, /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                                , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorPdAllocatedInClosingWaitingEventBeforeLock   		, /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorPdAllocatedInRemoving                        		, /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*Timer Event Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorTimerEventTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorTimerInTrying                     		 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                                 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorTimerInTryingCountingTimeout      		 , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorTimerInOpen                       		 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorTimerInOpenWaitingForPdToSendBar  		 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorTimerInOpenWaitingForLock         		 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorTimerInOpenSendingBar             		 , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorFatal                                     , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*Packet Discarded Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorPacketDiscardedTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorIgnoreEvent	                            , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorPacketDiscardedInOpen            	    , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorPacketDiscardedInOpenSendingBar  		, /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorIgnoreEvent                              , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*Queue Locked Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorQueueIsLockedTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorFatal	                                           , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorQueueLockedInOpenWaitForLock                		   , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorQueueLockedInClosing                        		   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorQueueLockedInClosingWaitForLockToSendDelba  		   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorFatal                                               , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorQueueLockedInRemoving                       		   , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*Close TID Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorCloseTidTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorCloseTidInTrying           		  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorCloseTidInTryingWaitForPd  		  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorIgnoreEvent	                      , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorCloseTidInOpen             		  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorCloseTidInOpenWaitForPd    		  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorCloseTidInOpenWaitForLock  		  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorCloseTidInOpenSendingBar   		  , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorIgnoreEvent                        , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};
/*Open TID Table*/
static const TsManagerInitiatorGeneralFunc TsManagerInitiatorOpenTidTable[TS_MANAGER_INITIATOR_NUM_OF_TID_STATES] =
{
	 tsManagerInitiatorIgnoreEvent      , /* TS_MANAGER_INITIATOR_TID_STATE_IDLE */
	 tsManagerInitiatorOpenTidInClosed  , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSED */
	 tsManagerInitiatorIgnoreEvent      , /* TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED */
	 tsManagerInitiatorOpenTidInTrying  , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD */
	 tsManagerInitiatorSetAllowedFlag	, /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR */ 
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR */	 
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING */ 
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA */
	 tsManagerInitiatorSetAllowedFlag   , /* TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING */
     tsManagerInitiatorIgnoreEvent      , /* TS_MANAGER_INITIATOR_TID_STATE_REMOVING */ 
};


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/
TsManagerInitiatorTidGlobalStatistics_t TsManagerInitiatorTidGlobalStatistics;
TsManagerInitiatorStationGlobalStatistics_t TsManagerInitiatorStationGlobalStatistics;



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

tsManagerInitiatorIgnoreEvent 


Description:
------------
Empty Event handler

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorIgnoreEvent(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(TsDb);	
	UNUSED_PARAM(parameters);	
}

/**********************************************************************************

tsManagerInitiatorFatal 


Description:
------------
Fatal event handler

Input: 
-----
TsDb
state - new state

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorFatal(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(TsDb);	
	UNUSED_PARAM(parameters);	
	FATAL("TS Manager Initiator Fatal");
}

/**********************************************************************************

tsManagerInitiatorChangeState 


Description:
------------
Utility function to change state

Input: 
-----
TsDb
state - new state

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorChangeState(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerInitiatorTidState_e newState)
{
	ILOG0_DDDD("TS Manager Initiator Change SID %d, TID %d, State from %d to %d", TsDb->commonPart.stationId, TsDb->commonPart.tid, TsDb->commonPart.state, newState);
	TsDb->commonPart.state = newState;
}

/**********************************************************************************

tsManagerInitiatorLogEvent 


Description:
------------
Utility function to log event

Input: 
-----
TsDb
event

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorLogEvent(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerInitiatorTidEvent_e event)
{
	UNUSED_PARAM(TsDb);
	UNUSED_PARAM(event);
	ILOG0_DDDD("TS Manager Initiator Event SID %d, TID %d, State %d Event %d", TsDb->commonPart.stationId, TsDb->commonPart.tid, TsDb->commonPart.state, event);
}


static void tsManagerInitiatorChangeStateToClosed(TsManagerInitiatorStationTidParams_t *TsDb)
{
	ILOG0_DDDD("TS Manager Initiator Change SID %d, TID %d, State from %d to CLOSED. ppduTxMode = %d", TsDb->commonPart.stationId, TsDb->commonPart.tid, TsDb->commonPart.state, TsManager_GlobalDb.ppduTxModeNoAmpdu);	
	TsDb->commonPart.state = TS_MANAGER_INITIATOR_TID_STATE_CLOSED;
	StaDB_ChangePpduTxMode(TsDb->commonPart.stationId, TsDb->commonPart.tid, STA_DB_PPDU_TX_MODE_LEGACY_AND_AMSDU_NO_AMPDU);
}


/**********************************************************************************

tsManagerInitiatorRmRequest 


Description:
------------
Utility function to build RM rquest

Input: 
-----
TsDb
pdRequestFillParameters - request structure

		
Output:
-------
allocation status
	

Returns:
--------
	bool
	
**********************************************************************************/
static bool tsManagerInitiatorRmRequest(TsManagerInitiatorStationTidParams_t *TsDb, RmPdRequestFillParameters_t *pdRequestFillParameters)
{
	bool allocationStatus = FALSE;

	pdRequestFillParameters->returnMsgType = TS_MANAGER_PACKET_DESCRIPTOR_ALLOCATED;
	pdRequestFillParameters->returnTaskId = TASK_TS_MANAGER;
	pdRequestFillParameters->context = TS_MANAGER_BUILD_RM_REQUEST(TsDb->commonPart.stationId, TsDb->commonPart.tid, TS_MANAGER_RM_REQUEST_INITIATOR);
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_TS_MANAGER, pdRequestFillParameters);
	return (allocationStatus);
}

/**********************************************************************************

tsManagerInitiatorRmRemove 


Description:
------------
Utility function to remove RM rquest

Input: 
-----
TsDb

		
Output:
-------
remove status
	

Returns:
--------
	bool - 
	
**********************************************************************************/
static bool tsManagerInitiatorRmRemove(TsManagerInitiatorStationTidParams_t *TsDb)
{
	bool removeStatus = TRUE;

	/*Check if RM returned a valid ID*/
	if (TsDb->commonPart.pdRequestId != TS_MANAGER_INITIATOR_INVALID_REQUEST_ID)
	{
		removeStatus = ResourceManager_RemoveRequest(TsDb->commonPart.pdRequestId, DESC_POOL_TS_MANAGER);
	}

	return (removeStatus);
}

/**********************************************************************************

tsManagerInitiatorMoveToIdle 


Description:
------------
This function move to Idle state and reduce the number of active TIDs

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorMoveToIdle(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(TsDb);	
	UNUSED_PARAM(parameters);	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorRemoveReqeustInTryingToOpen 


Description:
------------
This function remove the entry from the calendar wheel and move to Idle state 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInTryingToOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	/*Change State to Idle*/
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	/*Call Callback*/
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorRemoveInTryingToOpenWaitForPd 


Description:
------------
This function tries to remove the request for PD and move to Idle

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInTryingToOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation(TsDb);
}

/**********************************************************************************

tsManagerInitiatorRemoveInTryingSendingAddbaRequest 


Description:
------------
This function decreases the number of open BA and moves to removing state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInTryingSendingAddbaRequest(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerInitiatorRemoveInTryingWaitingForConfirmation 


Description:
------------
This function decreases the number of open BA and moves to removing state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInTryingWaitingForConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerInitiatorMoveToRemoving 


Description:
------------
This function move to removing state without further operations

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorMoveToRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerInitiatorRemoveInTryingToOpenCountingTimeout 


Description:
------------
This function removes entry from calendar wheel , unlock queue after sending ADDBA 
request and moves to ilde state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInTryingToOpenCountingTimeout(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);	
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
	if(IEEE802_1D_VO_2 != TsDb->commonPart.tid)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}
#endif	
    /* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorRemoveInOpen 


Description:
------------
This function close BA agreement remove from calendar wheel and moves immidiately 
to Idle

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);	
	/*Close BA Agreement in STA DB*/
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId, TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);	
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorRemoveInopenWaitForPd 


Description:
------------
This function tries to remove request for PD, close BA and move to Idle

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	UNUSED_PARAM(parameters);
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId, TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	
	tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation(TsDb);
}

/**********************************************************************************

tsManagerInitiatorRemoveInOpenWaitingForLock 


Description:
------------
This function release the PD save in the TID data base, closes the BA agreement, 
remove the entry from calendar wheel and move to removing state 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInOpenWaitingForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	TxPd_t* pd = NULL;
	UNUSED_PARAM(parameters);

    /* Release the PD saved in the TID database */
	pd = (TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset);
	ResourceManager_ReleaseDescriptor(pd, DESC_POOL_TS_MANAGER);
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
	/*Close BA Agreement in STA DB*/
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId, TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerInitiatorRemoveInOpenSendingBar 


Description:
------------
This function closes the BA agreement, removes the entry from calendar wheel 
and moves to removing state 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId, TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}


/**********************************************************************************

tsManagerInitiatorRemoveInClosingWaitForEventToSendDelba 


Description:
------------
This function closes the BA agreement and moves to removing state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}


/**********************************************************************************

tsManagerInitiatorRemoveInClosingWaitForPd 


Description:
------------
This function closes the BA agreement and trying to remove the PD request in order
to remove

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInClosingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{		
	UNUSED_PARAM(parameters);
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid, FALSE);
	tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation(TsDb);
}

/**********************************************************************************

tsManagerInitiatorRemoveInClosingLockBeforeSendingDelba 


Description:
------------
This function closes the BA agreement , release the PD and moves to removing to
wait for lock confirmation to remove

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorRemoveInClosingLockBeforeSendingDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TxPd_t* pd = NULL;
	UNUSED_PARAM(parameters);

    /* Release the PD saved in the TID database */
	pd = (TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset);
	ResourceManager_ReleaseDescriptor(pd, DESC_POOL_TS_MANAGER);
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
	
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid, FALSE);
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
}

/**********************************************************************************

tsManagerInitiatorAddbaResponseRecievedInTryingSending 


Description:
------------
This function Handles ADDBA response received when still waiting for confirmation

Input: 
-----
TsDb
parameters - ADDBA response payload address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaResponseRecievedInTryingSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	uint16 					bufferSize = 0;
	uint16 					statusCode = TS_MANAGER_SUCCESS_STATUS;
	TsManagerAddbaParams_t* addbaParams = (TsManagerAddbaParams_t*)parameters;
	FM_PAYLOAD_ADDBA_RES*	addbaResponsePayload = NULL;
	
	addbaResponsePayload = (FM_PAYLOAD_ADDBA_RES *)&(addbaParams->addbaPayload->PacketPayload);

	if(addbaResponsePayload->u8DialogToken == TsDb->commonPart.dialogToken)
	{	
		statusCode = addbaResponsePayload->u16StatusCode;

		if(TS_MANAGER_SUCCESS_STATUS_CODE == statusCode)
		{	
			statusCode = TS_MANAGER_SUCCESS_STATUS;
			bufferSize = addbaResponsePayload->u16BlockAckParameterSet.BlockAckBufferSize;
			if(bufferSize > TS_MANAGER_MAX_BUFFER_SIZE || (bufferSize < TS_MANAGER_MIN_BUFFER_SIZE))
			{
				statusCode = TS_MANAGER_UNSPECIFIED_STATUS;	/*Set unspecified so we send DELBA afterwards*/
			}
			else if(!addbaResponsePayload->u16BlockAckParameterSet.BlockAckPolicy)
			{
				/* Currently delayed block ACK is not supported */
				statusCode = TS_MANAGER_UNSPECIFIED_STATUS; /*Set unspecified so we send DELBA afterwards*/
			}
			if(TS_MANAGER_SUCCESS_STATUS == statusCode)
			{
				TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusSuccessAndValid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				tsManagerInitiatorSaveAddbaResponseParameters(TsDb, addbaResponsePayload);

				// Process the IEs after the regular payload. Look for ADDBA Extension IE.
				// The TS is not opened yet since we are waiting for the ADDBA REQ confirmation but no harm to store the frag params here since we don't want to save them for later.
				tsManagerInitiatortProcessAddbaResExtension(TsDb, addbaParams);
			}
			else
			{
				TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusSuccessAndInvalid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
			}
		}
		else
		{		
			TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusFailure[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
			tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);

            if(statusCode == TS_MANAGER_INVALID_PARAMETERS_STATUS_CODE)
			{
				statusCode = TS_MANAGER_INVALID_PARAMETERS_STATUS;
			}
			else
			{
				statusCode = TS_MANAGER_UNSPECIFIED_STATUS;
			}
		}
		TsDb->commonPart.status = statusCode;
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_CONFIRMATION);
	}
}

/**********************************************************************************

tsManagerInitiatorAddbaResponseRecievedInTryingCounting 


Description:
------------
This function Handles ADDBA response received when counting timeout

Input: 
-----
TsDb
parameters - ADDBA response payload address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaResponseRecievedInTryingCounting(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	uint16 					bufferSize = 0;
	uint16 					statusCode = TS_MANAGER_SUCCESS_STATUS;
	FM_PAYLOAD_ADDBA_RES*	addbaResponsePayload = NULL;
	uint16 					winsize = 0;
	TsManagerAddbaParams_t* addbaParams = (TsManagerAddbaParams_t*)parameters;
	
	addbaResponsePayload = (FM_PAYLOAD_ADDBA_RES *)&(addbaParams->addbaPayload->PacketPayload);

	if(addbaResponsePayload->u8DialogToken == TsDb->commonPart.dialogToken)
	{	
		/* Remove entry from calendar wheel */
		TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
		statusCode = addbaResponsePayload->u16StatusCode;
		if(TS_MANAGER_SUCCESS_STATUS_CODE == statusCode)
		{
			statusCode = TS_MANAGER_SUCCESS_STATUS;
			bufferSize = addbaResponsePayload->u16BlockAckParameterSet.BlockAckBufferSize;
            /*in ADDBA RSP buffer size 0 is not allowed, checking min size covers it*/
			if(bufferSize > TS_MANAGER_MAX_BUFFER_SIZE || bufferSize < TS_MANAGER_MIN_BUFFER_SIZE)
			{
				statusCode = TS_MANAGER_INVALID_PARAMETERS_STATUS;
			}
			else if(!addbaResponsePayload->u16BlockAckParameterSet.BlockAckPolicy)
			{
				/* Currently delayed block ACK is not supported */
				statusCode = TS_MANAGER_INVALID_PARAMETERS_STATUS;
			}
			if(TS_MANAGER_SUCCESS_STATUS == statusCode && TsDb->commonPart.allowed)
			{
				tsManagerInitiatorSaveAddbaResponseParameters(TsDb, addbaResponsePayload);
				
				TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);

				winsize = TsManagerBufferSizeToWindowSize[TsDb->commonPart.bufferSize];
				/*Open BA Agreement in STA DB*/
				StaDB_OpenBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid,  winsize, TsDb->commonPart.amsduSupport);

				// Process the IEs after the regular payload. Look for ADDBA Extension IE.
				tsManagerInitiatortProcessAddbaResExtension(TsDb, addbaParams);

				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);
				TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusSuccessAndValid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBaAgreementOpened[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
				if(IEEE802_1D_VO_2 == TsDb->commonPart.tid)
#endif					
				{
					tsManagerInitiatorTryToAllocatePdForBar(TsDb);
				}
			}
			else
			{
				if(TS_MANAGER_SUCCESS_STATUS == statusCode)
				{
					/* In case of success reset the timeout for the case 'Open tid' request will be received during closing and then the TID tries to open again */
					TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS; 
					TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusSuccessAndValid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				}
				else
				{
					tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
					TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusSuccessAndInvalid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				}

                tsManagerInitiatorTryToAllocatePdForDelba(TsDb);
			}
			
		}
		else
		{			
			TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseReceivedWithStatusFailure[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

			if(TS_MANAGER_REQUEST_DECLINED_STATUS_CODE == statusCode)
			{
				tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
                tsManagerInitiatorTryToAllocatePdForDelba(TsDb);
			}
			else
			{
				tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
				
				if(TsDb->commonPart.allowed)
				{			
					tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
					
					TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
					tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
				}
				else
				{
					tsManagerInitiatorChangeStateToClosed(TsDb);
				}
			}
		}
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)			
		if(IEEE802_1D_VO_2 != TsDb->commonPart.tid)
		{
			Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
		}
#endif		
	}
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInTryingWaitConfirmation 


Description:
------------
This function moves to closing state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInTryingWaitConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInOpen 


Description:
------------
This function removes the entry from the calendar wheel, closes the BA agreement ,
sends lock request and moves to closing state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);

	tsManagerInitiatorLockQueue(TsDb);
	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInOpenWaitForPd 


Description:
------------
This function removes the entry from the calendar wheel, closes the BA agreement ,
and tries to remove the PD request

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	bool removeStatus = FALSE;
	UNUSED_PARAM(parameters);

	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	removeStatus = tsManagerInitiatorRmRemove(TsDb);
	
	//("tsManagerInitiatorDelbaRecievedInOpenWaitForPd remove status %d", removeStatus);
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	
	if(removeStatus)
	{
		/* Request was removed successfully - send lock request and move to closing */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);

		tsManagerInitiatorLockQueue(TsDb);
	}
	else
	{
		/* Remove request failed - wait for allocation */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING);
	}
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInOpenWaitForLock 


Description:
------------
This function removes the entry from the calendar wheel, closes the BA agreement,
release PD and move to closing

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TxPd_t* pd = NULL;
	UNUSED_PARAM(parameters);
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);

	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	/* Release the PD saved in the TID database */
	pd = (TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset);
	ResourceManager_ReleaseDescriptor(pd, DESC_POOL_TS_MANAGER);
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}


/**********************************************************************************

tsManagerInitiatorDelbaRecievedInOpenSendingBar 


Description:
------------
This function removes the entry from the calendar wheel, closes the BA agreement,
and move to closing - waiting for event before locking

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	/* Remove entry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING);
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInClosingWaitForEventToSend 


Description:
------------
This function closes the BA agreement, and moves to closing - waiting for event before locking

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInClosingWaitForEventToSend(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING);
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInClosingWaitForPd 


Description:
------------
This function closes the BA agreement, and tries to remove request for PD

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInClosingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	bool removeStatus = FALSE;
	UNUSED_PARAM(parameters);
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	
	removeStatus = tsManagerInitiatorRmRemove(TsDb);
	if(removeStatus)
	{
		/* Request was removed successfully - need to lock before flushing and closing BA */
	    tsManagerInitiatorLockQueue(TsDb);
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
	}
	else
	{
		/* Remove request failed - wait for allocation before locking, flushing and closing BA*/
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_BEFORE_LOCKING);
	}
}

/**********************************************************************************

tsManagerInitiatorDelbaRecievedInClosingWaitLockBeforeDelba 


Description:
------------
This function closes the BA agreement, release the PD and move to closing

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaRecievedInClosingWaitLockBeforeDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TxPd_t* pd = NULL;
	UNUSED_PARAM(parameters);

	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaReceived[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	/* Release the PD saved in the TID database */
	pd = (TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset);
	ResourceManager_ReleaseDescriptor(pd, DESC_POOL_TS_MANAGER);
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;

	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInTryingSending 


Description:
------------
This function handles confirmation of ADDBA request when sending ADDBA request

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInTryingSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 vapIndex = 0;

	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
	if(((TxPd_t *)parameters)->retransmissionIndication)
	{
	 	TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsManagerInitiatorVapDatabase[vapIndex].addbaRequestTimeout);
	 	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_COUNTING_ADDBA_REQUEST_TIMEOUT);
	}
	else
	{
		/* The packet was not transmitted at all - it is considered as failure to open */
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
		
		if(TsDb->commonPart.allowed)
		{
			/* This failure doesnt increase the timeout since it was internal failure */
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
		}
		else
		{
			tsManagerInitiatorChangeStateToClosed(TsDb);
		}
	}	
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation 


Description:
------------
This function handles confirmation of ADDBA request after already receiving ADDBA 
response

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 wasPacketTransmitted = FALSE;
	uint16 winsize = 0;

    wasPacketTransmitted = ((TxPd_t *)parameters)->retransmissionIndication;
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	//("tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation TsDb->commonPart.stationId = %d tid = %d", TsDb->commonPart.stationId, tid);

    if((TS_MANAGER_SUCCESS_STATUS == TsDb->commonPart.status ||
	   TS_MANAGER_UNSPECIFIED_STATUS == TsDb->commonPart.status)&&
	   wasPacketTransmitted)
    {
		/* ADDBA response received with status == success */

		if(TsDb->commonPart.allowed && 
		   TS_MANAGER_SUCCESS_STATUS == TsDb->commonPart.status)
		{
			//("tsManagerInitiatorAddbaRequestConfirmedInTryingWaitConfirmation open BA agreement");
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
			winsize = TsManagerBufferSizeToWindowSize[TsDb->commonPart.bufferSize];
			StaDB_OpenBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid, winsize, TsDb->commonPart.amsduSupport);
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);	
			TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBaAgreementOpened[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
			if(IEEE802_1D_VO_2 == TsDb->commonPart.tid)
#endif
			{
				tsManagerInitiatorTryToAllocatePdForBar(TsDb);
			}
		}
		else
		{
			if(TS_MANAGER_SUCCESS_STATUS == TsDb->commonPart.status)
			{
				/* Reset the timeout after it was overrun when receiving the ADDBA request - there was success and
				   in case 'Open TID' request will be received should start from minimum value */
				TsDb->commonPart.timeout =  TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
			}
			tsManagerInitiatorTryToAllocatePdForDelba(TsDb);
		}
    }
	else
	{
		/* The trial to open was rejected by the recipient - check allowed flag */
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
		if (TsDb->commonPart.allowed)
		{
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
		}
		else
		{
			tsManagerInitiatorChangeStateToClosed(TsDb);
		}
	}
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	if(wasPacketTransmitted && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));			
	}
#endif	
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInOpenedSendBar 


Description:
------------
This function handles confirmation of ADDBA request in opened sending BAR - used in PBAC
response

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInOpenedSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	uint8 wasPacketTransmitted = FALSE;
#else
	UNUSED_PARAM(parameters);	
#endif

	/*Should only happen in PBAC*/
	ASSERT(TsDb->commonPart.pbac == TRUE); 
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
    wasPacketTransmitted = ((TxPd_t *)parameters)->retransmissionIndication;
#endif
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	/*If ADDBA REQ was transmitted and TID != 7 Sender locked the queue*/
	if((wasPacketTransmitted) && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}
#endif	
	/*Check if additional packets were discarded while ADDBA REQ was being transmitted*/
	if(TS_MANAGER_INITIATOR_PD_NULL == TsDb->commonPart.pdOffset)
	{
		/* No event of discarded packet continue with normal flow */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);
	}
	else
	{
		tsManagerInitiatorTryToAllocatePdForBar(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInClosing 


Description:
------------
This function handles confirmation of ADDBA request when closing 

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	/* When the retransmission bit is not set it indicates that the packet was discarded before transmitted even once - the queue was not locked by the sender */
	if((((TxPd_t *)parameters)->retransmissionIndication) && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}
#else
	UNUSED_PARAM(parameters);	
#endif	
	/* closing process was finished - check allowed flag */		
	if(TsDb->commonPart.allowed)
	{
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
	}
	else
	{
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInWaitingToSendDelba 


Description:
------------
This function handles confirmation of ADDBA request when waiting to send DELBA
Could happen if ADDBA REQ was sent in PBAC - in this case if we have inactivity timeout we will wait for
the PD confitrmation before sending DELBA

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInWaitingToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
	uint8 wasPacketTransmitted = FALSE;
#endif
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;

#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
    wasPacketTransmitted = ((TxPd_t *)parameters)->retransmissionIndication;
#else
	UNUSED_PARAM(parameters);	
#endif
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);	
	if (allocationStatus)
	{
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)			
		/*If ADDBA REQ was transmitted and TID != 7 Sender locked the queue*/
		if((wasPacketTransmitted) && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
		{
			tsManagerInitiatorFlushIllegalPackets(TsDb);
			Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));

			TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++; 

			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
			TsManager_TransmitDelba(&TsDb->commonPart, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
		}
		else
#endif			
		{
			TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);
			//Send Lock Request
			tsManagerInitiatorLockQueue(TsDb);
				
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
		}
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)			
		if((wasPacketTransmitted)&& (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
		{
			Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
		}
#endif		
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			/*If resource manager did not have an ID available, no point in waiting for PD*/
			/*Lock the queue now*/
			tsManagerInitiatorLockQueue(TsDb);
			/*Move to closing - when queue is locked, it will be flushed and we reschedule*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
		}
	}
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInWaitingBeforeLock 


Description:
------------
This function handles confirmation of ADDBA request when waiting for PD before Lock
Could happen if ADDBA REQ was sent in PBAC - in this case if we receive DELBA we will wait for
the PD confitrmation before Locking the queue

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInWaitingBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
	uint8 wasPacketTransmitted = FALSE;

    wasPacketTransmitted = ((TxPd_t *)parameters)->retransmissionIndication;
#else
	UNUSED_PARAM(parameters);	
#endif	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	if((wasPacketTransmitted) && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
	{
		/*If ADDBA REQ was transmitted and TID != 7 Sender locked the queue*/
		tsManagerInitiatorFlushIllegalPackets(TsDb);
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
		
		if(TsDb->commonPart.allowed)
		{		
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
		}
		else
		{
			tsManagerInitiatorChangeStateToClosed(TsDb);
		}
	}
	else
#endif		
	{
		//Send Lock Request
	    tsManagerInitiatorLockQueue(TsDb);
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
	}	
}

/**********************************************************************************

tsManagerInitiatorAddbaRequestConfirmedInRemoving 


Description:
------------
This function handles confirmation of ADDBA request when removing 

Input: 
-----
TsDb
parameters - PD of the ADDBA request

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorAddbaRequestConfirmedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestConfiremd[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;

#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	/* When the retransmission bit is not set it indicates that the packet was discarded before transmitted even once - the queue was not locked by the sender */
	if((((TxPd_t *)parameters)->retransmissionIndication) && (IEEE802_1D_VO_2 != TsDb->commonPart.tid))
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}
#else
	UNUSED_PARAM(parameters);	
#endif
	/* Remove the tid */		
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorDelbaConfirmedInClosing 


Description:
------------
This function handles confirmation of Delba when closing 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorDelbaConfirmedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	/* closing process was finished - check allowed flag */		
	if(TsDb->commonPart.allowed)
	{
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
	}
	else
	{
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorBarConfirmedInOpenedSending 


Description:
------------
This function handles BAR confirmation in while BA agreement is opened 

Input: 
-----
TsDb
parameters - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarConfirmedInOpenedSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	if(TS_MANAGER_INITIATOR_PD_NULL == TsDb->commonPart.pdOffset)
	{
		/* No event of discarded packet continue with normal flow */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);	
	}
	else
	{
		tsManagerInitiatorTryToAllocatePdForBar(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorBarConfirmedInClosingWaitForEventToSendDelba 


Description:
------------
This function handles BAR confirmation when waiting to send DELBA

Input: 
-----
TsDb
parameters - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarConfirmedInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);	
	if(allocationStatus)
	{
		TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);

	    tsManagerInitiatorLockQueue(TsDb);
			
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			/*If resource manager did not have an ID available, no point in waiting for PD*/
			/*Lock the queue now*/
			tsManagerInitiatorLockQueue(TsDb);
			/*Move to closing so when queue is locked, it is flushed and we reschedule*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
		}
	}
}

/**********************************************************************************

tsManagerInitiatorBarConfirmedInClosingWaitForEventBeforeLock 


Description:
------------
This function handles BAR confirmation when waiting for event to lock

Input: 
-----
TsDb
parameters - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarConfirmedInClosingWaitForEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{		
	UNUSED_PARAM(parameters);
    tsManagerInitiatorLockQueue(TsDb);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}


/**********************************************************************************

tsManagerInitiatorBarConfirmedInRemoving 


Description:
------------
This function handles BAR confirmation when removing the station

Input: 
-----
TsDb
parameters - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarConfirmedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorBarStatusInOpenedSending 


Description:
------------
This function handles BAR status when BA agreement is opened

Input: 
-----
TsDb
parameters - the address of the status

		
Output:
-------

	
Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarStatusInOpenedSending(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	
	ILOG2_D("tsManagerInitiatorBarStatusInOpenedSending status %d", ((TxPd_t *)parameters)->status);

    if(PD_STATUS_ACK_RECEIVED == ((TxPd_t *)parameters)->status)
    {
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBarWasAnsweredWithAck[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		/* The recipient indicates that the BA agreeemnt is closed */
		
		tsManagerInitiatorFlushIllegalPackets(TsDb);
		
		TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
		TsManager_RescheduleInCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS);
					
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
		
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
		
    }
	else
	{
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBarWasAnsweredWithBa[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		if(TS_MANAGER_INITIATOR_PD_NULL == TsDb->commonPart.pdOffset)
		{
			/* No event of discarded packet continue with normal flow */
			ILOG2_V("tsManagerInitiatorBarStatusInOpenedSending normal flow");
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);
		}
		else
		{
			allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	
			//("tsManagerInitiatorBarStatusInOpenedSending  allocation status %d ", allocationStatus);
	
			if(allocationStatus)
			{
				TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBarWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
				TsManager_TransmitBar(&TsDb->commonPart, 
				                  (TxPd_t *)pdRequestFillParameters.packetDescriptor);
				TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;		
			}
			else
			{
				/* PD allocation falied */		
				TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
				if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
				{
					/*If resource manager did not have an ID available, no point in waiting for PD*/
					/*Clear PD offset*/
					TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
					/*Move back to opened*/
					tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);
				}
				else
				{
					tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR);
				}
			}
		}	
	}
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
}

/**********************************************************************************

tsManagerInitiatorBarStatusInClosingWaitForEventToSendDelba 


Description:
------------
This function handles BAR status when waiting for event to send DELBA
Input: 
-----
TsDb
parameters - the address of the status

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarStatusInClosingWaitForEventToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;	
	RmPdRequestFillParameters_t pdRequestFillParameters;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
    if(PD_STATUS_ACK_RECEIVED == ((TxPd_t *)parameters)->status)
    {
		/* The recipient indicates that the BA agreeemnt is closed - no need to send DELBA */
		
		tsManagerInitiatorFlushIllegalPackets(TsDb);
			
		if(TsDb->commonPart.allowed)
		{
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
		}
		else
		{
			tsManagerInitiatorChangeStateToClosed(TsDb);
		}
    }
	else
	{
		allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
		//("tsManagerInitiatorBarStatusInClosingWaitForEventToSendDelba normal flow allocation status = %d ", allocationStatus);
			
		if(allocationStatus)
		{
			tsManagerInitiatorFlushIllegalPackets(TsDb);
			
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
			TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++; 
			TsManager_TransmitDelba(&TsDb->commonPart, (TxPd_t *)pdRequestFillParameters.packetDescriptor);	
		}
		else
		{
			TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
			/* PD allocation falied */
			if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
			{
				/*If resource manager did not have an ID available, no point in waiting for PD*/
				tsManagerInitiatorFlushIllegalPackets(TsDb);
				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
				tsManagerInitiatorDelbaConfirmedInClosing(TsDb, NULL);
			}
			else
			{
				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
			}
		}	
	}
	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
}

/**********************************************************************************

tsManagerInitiatorBarStatusInClosingWaitForEventBeforeLock 


Description:
------------
This function handles BAR status when waiting for event to lock before closing

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarStatusInClosingWaitForEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	tsManagerInitiatorFlushIllegalPackets(TsDb);
	UNUSED_PARAM(parameters);

	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));

	if(TsDb->commonPart.allowed)
	{			
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);

		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
	}
	else
	{
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorBarStatusInRemoving 


Description:
------------
This function handles BAR status when removing the station

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBarStatusInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	UNUSED_PARAM(parameters);

	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}



/**********************************************************************************

tsManagerInitiatorPdAllocatedInTryingWaitingPd 


Description:
------------
This function handles PD allocated when waiting to send ADDBA request

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInTryingWaitingPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	tsManagerInitiatorTryToOpenBaAgreement(TsDb, (TxPd_t *)parameters);	
}


/**********************************************************************************

tsManagerInitiatorPdAllocatedInOpenedWaitingPdToSendBar 


Description:
------------
This function handles PD allocated when waiting to send BAR (BA agreement is opened)

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInOpenedWaitingPdToSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(parameters);

	if (TsDb->commonPart.pbac == FALSE)
	{
		//Send Lock Request
	    tsManagerInitiatorLockQueue(TsDb);
			
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK);
	}
	else
	{
		uint8 vapIndex = 0;
		
		StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
		/*Use the timeout we already negotiated*/
		TsManager_TransmitAddbaRequest(TsDb, (TxPd_t *)parameters, TsDb->commonPart.timeout);
		/*Skip a few states to Wait for ADDBA REQ confirmation*/
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR);
	}
}

/**********************************************************************************

tsManagerInitiatorPdAllocatedInClosing 


Description:
------------
This function handles PD allocated when closing

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	/* No need to release request since it was released when trying to remove request */
	ResourceManager_ReleaseDescriptor((TxPd_t *)parameters, DESC_POOL_TS_MANAGER);
	//("tsManagerInitiatorPdAllocatedInClosing allowed = %d ", TsDb->commonPart.allowed);
	if(TsDb->commonPart.allowed)
	{			
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);

		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
	}
	else
	{
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorPdAllocatedInClosingWaitingForPd 


Description:
------------
This function handles PD allocated when closing and waiting for PD to send DELBA

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInClosingWaitingForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(parameters);

	tsManagerInitiatorLockQueue(TsDb);

	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
}

/**********************************************************************************

tsManagerInitiatorPdAllocatedInClosingWaitingEventBeforeLock 


Description:
------------
This function handles PD allocated when closing and waiting for event before lock

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInClosingWaitingEventBeforeLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{
	/* No need to release request since it was released when trying to remove request */
	ResourceManager_ReleaseDescriptor((TxPd_t *)parameters, DESC_POOL_TS_MANAGER);
	
	tsManagerInitiatorLockQueue(TsDb);
    
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerInitiatorPdAllocatedInRemoving 


Description:
------------
This function handles PD allocated when closing and waiting for event before lock

Input: 
-----
TsDb
parameters - the PD address

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPdAllocatedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	/* No need to release request since it was released when trying to remove request */
	ResourceManager_ReleaseDescriptor((TxPd_t *)parameters, DESC_POOL_TS_MANAGER);

 	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorTimerInTrying 


Description:
------------
This function handles timer event in trying state (start process of trying to open
BA agreement)

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 isAllowedToOpen = FALSE;
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);

	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	isAllowedToOpen = tsManagerInitiatorIsAllowedToOpenBaAgreement(TsDb->commonPart.stationId);
	if (isAllowedToOpen)
	{
		tsManagerInitiatorBaAgreementOpenedUpdateCounters(TsDb->commonPart.stationId);
		allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
		if (allocationStatus)
		{
			tsManagerInitiatorTryToOpenBaAgreement(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
		}
		else
		{
			/* PD allocation falied */		
			TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
			if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
			{
				/*If resource manager did not have an ID available, no point in waiting for PD*/
				/*Reschedule*/
				tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
				TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);		
			}
			else
			{
				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD);
			}
		}
	}
	else
	{
		tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);		
	}
}

/**********************************************************************************

tsManagerInitiatorTimerInTryingCountingTimeout 


Description:
------------
This function handles timer event in trying to open - counting timeout state - open 
BA agreement failed

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInTryingCountingTimeout(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
	if(IEEE802_1D_VO_2 != TsDb->commonPart.tid)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}
#endif	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaResponseTimeout[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
	tsManagerInitiatorTryToAllocatePdForDelba(TsDb);	
}

/**********************************************************************************

tsManagerInitiatorTimerInOpen 


Description:
------------
This function handles timer event in opened state 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 activity = FALSE;
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	activity = Pac_RxcBaAgreementInitiatorGetAndClearActivity(TsDb->commonPart.stationId, TsDb->commonPart.tid);

	if((activity) || (TsDb->commonPart.deleteBaAgreementOnInactivity == FALSE))
	{	
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);		
	}
	else
	{
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);		
		TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;		
		TsDb->commonPart.reason = TS_MANAGER_TIMEOUT_REASON;
		allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);		
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventInactivityTimeout[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		if(allocationStatus)
		{	
			TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);
			
			tsManagerInitiatorLockQueue(TsDb);
			
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
		}
		else
		{
			/* PD allocation falied */		
			TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
			if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
			{
				/*If resource manager did not have an ID available, no point in waiting for PD*/
				/*Lock the queue now*/
				tsManagerInitiatorLockQueue(TsDb);
				/*Move to closing - when queue is locked, it will be flushed and we reschedule*/
				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
			}
			else
			{
				tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
			}
		}	
	}
}

/**********************************************************************************

tsManagerInitiatorTimerInOpenWaitingForPdToSendBar 


Description:
------------
This function handles timer event in opened - waiting for PD to send BAR state 

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInOpenWaitingForPdToSendBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 activity = FALSE;
	UNUSED_PARAM(parameters);

	activity = Pac_RxcBaAgreementInitiatorGetAndClearActivity(TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	if((activity) || (TsDb->commonPart.deleteBaAgreementOnInactivity == FALSE))
	{
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout); 	
	}
	else
	{
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventInactivityTimeout[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
		TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
		
		TsDb->commonPart.reason = TS_MANAGER_TIMEOUT_REASON;

		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
	}
}

/**********************************************************************************

tsManagerInitiatorTimerInOpenWaitingForLock 


Description:
------------
This function handles timer event in opened - waiting for lock state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInOpenWaitingForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 activity = FALSE;
	UNUSED_PARAM(parameters);

	activity = Pac_RxcBaAgreementInitiatorGetAndClearActivity(TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	if((activity) || (TsDb->commonPart.deleteBaAgreementOnInactivity == FALSE))
	{
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout); 
	}
	else
	{
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventInactivityTimeout[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
		
		TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
		
		TsDb->commonPart.reason = TS_MANAGER_TIMEOUT_REASON;

		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
	}
}

/**********************************************************************************

tsManagerInitiatorTimerInOpenSendingBar 


Description:
------------
This function handles timer event in opened - sending BAR state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTimerInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 activity = FALSE;
	UNUSED_PARAM(parameters);

	activity = Pac_RxcBaAgreementInitiatorGetAndClearActivity(TsDb->commonPart.stationId, TsDb->commonPart.tid);
	
	if((activity) || (TsDb->commonPart.deleteBaAgreementOnInactivity == FALSE))
	{
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
	}
	else
	{
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_TIMER);
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventInactivityTimeout[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
		tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
		
		TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
		
		TsDb->commonPart.reason = TS_MANAGER_TIMEOUT_REASON;

		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA);
	}
}

/**********************************************************************************

tsManagerInitiatorPacketDiscardedInOpen 


Description:
------------
This function handles packet discard in open state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPacketDiscardedInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	

	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	//("tsManagerInitiatorPacketDiscardedInOpen allocation status %d", allocationStatus);
	if(allocationStatus)
	{
		TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);

		/*If PBAC is negotiated need to send ADDBA REQ instead*/
		if (TsDb->commonPart.pbac == FALSE)
		{
			tsManagerInitiatorLockQueue(TsDb);
			
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK);
		}
		else
		{
			uint8 vapIndex = 0;
			
			StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
			TsManager_TransmitAddbaRequest(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor, TsDb->commonPart.timeout);
			/*Skip a few states to Wait for ADDBA REQ confirmation*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR);
		}
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId != INVALID_FREE_PENDING_LIST_ID)
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR);
		}
	}
}

/**********************************************************************************

tsManagerInitiatorPacketDiscardedInOpenSendingBar 


Description:
------------
This function handles packet discarded while sending BAR (BA agreement is opened)

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorPacketDiscardedInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PACKET_DISCARDED;
}

/**********************************************************************************

tsManagerInitiatorQueueLockedInOpenWaitForLock 


Description:
------------
This function handles 'queue is locked' event when waiting for lock to send BAR

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorQueueLockedInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBarWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
	DEBUG_ASSERT(*((TxSelectorLockStatus_e *)parameters) == TX_SELECTOR_LOCK_STATUS_LOCKED);
	TsManager_TransmitBar(&TsDb->commonPart, 
						  (TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset));
	TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR);
}

/**********************************************************************************

tsManagerInitiatorQueueLockedInClosing 


Description:
------------
This function handles 'queue is locked' event when waiting for lock to finish closing

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorQueueLockedInClosing(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	tsManagerInitiatorFlushIllegalPackets(TsDb);
	DEBUG_ASSERT(*((TxSelectorLockStatus_e *)parameters) == TX_SELECTOR_LOCK_STATUS_LOCKED);
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	//("tsManagerInitiatorQueueLockedInClosing allowed %d", TsDb->commonPart.allowed);

	if(TsDb->commonPart.allowed)
	{			
		TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN);
	}
	else
	{
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
}

/**********************************************************************************

tsManagerInitiatorQueueLockedInClosingWaitForLockToSendDelba 


Description:
------------
This function handles 'queue is locked' event when waiting for lock to send DELBA 
(when closing BA agreement)

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorQueueLockedInClosingWaitForLockToSendDelba(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	tsManagerInitiatorFlushIllegalPackets(TsDb);

	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++; 
	TsManager_TransmitDelba(&TsDb->commonPart,(TxPd_t *)TS_MANAGER_CONVERT_OFFSET_TO_PD(TsDb->commonPart.pdOffset));
	DEBUG_ASSERT(*((TxSelectorLockStatus_e *)parameters) == TX_SELECTOR_LOCK_STATUS_LOCKED);
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
}

/**********************************************************************************

tsManagerInitiatorQueueLockedInRemoving 


Description:
------------
This function handles 'queue is locked' event when removing the station

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorQueueLockedInRemoving(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	if(*((TxSelectorLockStatus_e *)parameters) == TX_SELECTOR_LOCK_STATUS_LOCKED)
	{
		Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId, (0x1 << TsDb->commonPart.tid));
	}

	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
	TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
}

/**********************************************************************************

tsManagerInitiatorCloseTidInTrying 


Description:
------------
This function handles 'close TID' request in trying state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	/* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	tsManagerInitiatorChangeStateToClosed(TsDb);
}

/**********************************************************************************

tsManagerInitiatorCloseTidInTryingWaitForPd 


Description:
------------
This function handles 'close TID' request in trying - waiting for PD state.
This function tries to remove the request for PD

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInTryingWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	bool removeStatus = FALSE;
	UNUSED_PARAM(parameters);
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
	removeStatus = tsManagerInitiatorRmRemove(TsDb);
	//("tsManagerInitiatorCloseTidInTryingWaitForPd remove status %d", removeStatus);

	if(removeStatus)
	{
		/* Request was removed successfully - move to Idle */
		tsManagerInitiatorChangeStateToClosed(TsDb);
	}
	else
	{
		/* Remove request failed - wait for message */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
	}	
}

/**********************************************************************************

tsManagerInitiatorCloseTidInOpen 


Description:
------------
This function handles 'close TID' request in opened state.

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInOpen(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	
	/* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;

	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;	
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	
	//("tsManagerInitiatorCloseTidInOpen allocation status %d", allocationStatus);
	if(allocationStatus)
	{
		TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);
		
	    tsManagerInitiatorLockQueue(TsDb);
		
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			/*If resource manager did not have an ID available, no point in waiting for PD*/
			/*Lock the queue now*/
			tsManagerInitiatorLockQueue(TsDb);
			/*Move to closing - when queue is locked, it will be flushed and we reschedule*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
		}
	}
}

/**********************************************************************************

tsManagerInitiatorCloseTidInOpenWaitForPd 


Description:
------------
This function handles 'close TID' request in opened - wait for PD state.

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInOpenWaitForPd(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);
	/* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
    TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
}


/**********************************************************************************

tsManagerInitiatorCloseTidInOpenWaitForLock 


Description:
------------
This function handles 'close TID' request in opened - wait for lock state.

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInOpenWaitForLock(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);	
	/* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	
    TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_LOCK_BEFORE_SENDING_DELBA);
}

/**********************************************************************************

tsManagerInitiatorCloseTidInOpenSendingBar 


Description:
------------
This function handles 'close TID' request in opened - sending BAR state.

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorCloseTidInOpenSendingBar(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);	
	/* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);

	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;	

	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_EVENT_TO_SEND_DELBA);
}

/**********************************************************************************

tsManagerInitiatorSetAllowedFlag 


Description:
------------
This function sets the 'allowed' flag when 'open TID' request is received

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorSetAllowedFlag(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	UNUSED_PARAM(parameters);	
	TsDb->commonPart.allowed = TRUE;
}

/**********************************************************************************

tsManagerInitiatorOpenTidInClosed 


Description:
------------
This function handles 'open TID' request in closed state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorOpenTidInClosed(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	

	TsDb->commonPart.allowed = TRUE;
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	tsManagerInitiatorBaAgreementOpenedUpdateCounters(TsDb->commonPart.stationId);
	TsDb->commonPart.timeout = 0;
	if (allocationStatus)
	{
		tsManagerInitiatorTryToOpenBaAgreement(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			/*If resource manager did not have an ID available, no point in waiting for PD*/
			/*Reschedule*/
			tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);		
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD);
		}
	}	
}

/**********************************************************************************

tsManagerInitiatorOpenTidInTrying 


Description:
------------
This function handles 'open TID' request in trying state

Input: 
-----
TsDb
parameters - empty

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorOpenTidInTrying(TsManagerInitiatorStationTidParams_t *TsDb, void *parameters)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	UNUSED_PARAM(parameters);
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	

    /* Remove enetry from calendar wheel */
	TsManager_RemoveFromCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart);
	TsDb->commonPart.timeout = TS_MANAGER_MINIMUM_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_IN_CALENDAR_WHEEL_UNITS;
	
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	tsManagerInitiatorBaAgreementOpenedUpdateCounters(TsDb->commonPart.stationId);
	/* No need to set allowed flag - if the allowed flag was cleared the tid wasnt not in trying state */
	if(allocationStatus)
	{
		tsManagerInitiatorTryToOpenBaAgreement(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			/*If resource manager did not have an ID available, no point in waiting for PD*/
			/*Reschedule*/
			tsManagerInitiatorIncreaseTimeForNextRetry(TsDb);
			TsManager_AddToCalendarWheel(&TsManagerInitiatorCalendarWheel, &TsDb->commonPart, TsDb->commonPart.timeout);		
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_WAITING_FOR_PD);
		}
	}	
}

/**********************************************************************************

tsManagerInitiatorLockQueue 


Description:
------------
This function sends lock requests

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorLockQueue(TsManagerInitiatorStationTidParams_t *TsDb)
{
	RequesterLockParams_t requesterLockParams;
	uint16 idx = TS_MANAGER_INITIATOR_CALC_LOCK_INDEX(TsDb->commonPart.stationId,TsDb->commonPart.tid);
	uint8 tailStationIndex;
	uint8 tailTid;

	/*Link to NULL*/
	TsDb->commonPart.pdRequestId = TS_MANAGER_INITIATOR_INVALID_LOCK_IDX;
	/*Serailize lock requests from TS Manager Initiator*/
	/*Do we have a pending entry*/
	if (TsManagerInitiator_LockHead == TS_MANAGER_INITIATOR_INVALID_LOCK_IDX)
	{
		/*Set new index as pending for lock*/
		TsManagerInitiator_LockHead = idx;		
		requesterLockParams.callerContext = NULL;
		requesterLockParams.returnMsg = TS_MANAGER_QUEUE_IS_LOCKED;
		requesterLockParams.returnTask = TASK_TS_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID,TsDb->commonPart.stationId,TsDb->commonPart.tid,&requesterLockParams);
	}
	else
	{
		/*Link new index to tail index*/
		tailStationIndex = TS_MANAGER_INITIATOR_GET_STA_FROM_LOCK_IDX(TsManagerInitiator_LockTail);
		tailTid = TS_MANAGER_INITIATOR_GET_TID_FROM_LOCK_IDX(TsManagerInitiator_LockTail);
		TsManagerInitiatorStationTidDatabase[tailStationIndex][tailTid].commonPart.pdRequestId = idx;
	}
	/*Set new index as tail*/
	TsManagerInitiator_LockTail = idx;
}

/**********************************************************************************

tsManagerInitiator_HandlePendingLock 


Description:
------------
This function handles pending lock requests

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiator_HandlePendingLock()
{
	RequesterLockParams_t requesterLockParams;
	uint16 idx = TsManagerInitiator_LockHead;
	StaId stationIndex = TS_MANAGER_INITIATOR_GET_STA_FROM_LOCK_IDX(idx);
	uint8 tid = TS_MANAGER_INITIATOR_GET_TID_FROM_LOCK_IDX(idx);
	
	/*Get entry from pending*/
	ASSERT(idx != TS_MANAGER_INITIATOR_INVALID_LOCK_IDX);
	/*Adjust head*/
	TsManagerInitiator_LockHead = TsManagerInitiatorStationTidDatabase[stationIndex][tid].commonPart.pdRequestId;
	/*Do we have more entries*/
	if (TsManagerInitiator_LockHead != TS_MANAGER_INITIATOR_INVALID_LOCK_IDX)
	{
		/*Get new head STA and TID*/
		idx = TsManagerInitiator_LockHead;
		stationIndex = TS_MANAGER_INITIATOR_GET_STA_FROM_LOCK_IDX(idx);
		tid = TS_MANAGER_INITIATOR_GET_TID_FROM_LOCK_IDX(idx);				
		requesterLockParams.callerContext = NULL;
		requesterLockParams.returnMsg = TS_MANAGER_QUEUE_IS_LOCKED;
		requesterLockParams.returnTask = TASK_TS_MANAGER;
	
		Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID,stationIndex,tid,&requesterLockParams);
	}
	else
	{
		/*Clear Tail as well*/
		TsManagerInitiator_LockTail = TS_MANAGER_INITIATOR_INVALID_LOCK_IDX;
	}
}

/**********************************************************************************

tsManagerInitiatorFlushIllegalPackets 


Description:
------------
This function flsuhes all data packets with retransmission bit set

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorFlushIllegalPackets(TsManagerInitiatorStationTidParams_t *TsDb)
{	
	TxPd_t *currentPacketDescriptor = NULL;
	TxPd_t *lastPacketDescriptorToDiscard = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint32	count = 0;

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = TsDb->commonPart.stationId;
	hwQueueManagerRequestParams.secondaryAddr = TsDb->commonPart.tid;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
    currentPacketDescriptor = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
	
    while(((TxPd_t *)NULL_PD != currentPacketDescriptor) && 
		  (currentPacketDescriptor->pdType == PD_TYPE_DATA) &&
		  (currentPacketDescriptor->retransmissionIndication))
    {
		/* Illegal PD: set status as NACK*/
		currentPacketDescriptor->status = PD_STATUS_NACK;
		if(currentPacketDescriptor->aggregationIndication)
		{
			count += currentPacketDescriptor->pdCounter;
			
			/* A-MSDU - discard the whole A-MSDU. There shouldnt be A-MSDU of 1 MSDU */
			currentPacketDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(currentPacketDescriptor->aMsduTailPointer);
		}
		else
		{
			count += 1;
		}
		lastPacketDescriptorToDiscard = currentPacketDescriptor;
		currentPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPacketDescriptor);
    }

	if(lastPacketDescriptorToDiscard != NULL)
	{
		hwQueueManagerRequestParams.pTailDesc = lastPacketDescriptorToDiscard;
		HwQManager_PopListFromHead(&hwQueueManagerRequestParams);

		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		GeneralStatistics.dropReasonTSFlush[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)] += count;
		PerClientStatistics.dropCntReasonTSFlush[TsDb->commonPart.stationId] += count;
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);		
		/* If PD with retry bit was discarded the low sequence number should be update */
		StaDB_UpdateLowSequenceNumberFromCurrent(TsDb->commonPart.stationId, TsDb->commonPart.tid);
	}
	/*Close agreement here - we want to disable A-MSDU support and we need to do it only after we ensured that there are no retries in queue*/
	StaDB_CloseBaAgreement(TsDb->commonPart.stationId,TsDb->commonPart.tid, TRUE);
}


/**********************************************************************************

tsManagerInitiatorTryToOpenBaAgreement


Description:
------------
This function prepares the entry for ADDBA request and transmit the ADDBA request

Input: 
-----
TsDb
packetDescriptor -  the PD of the ADDBA request
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTryToOpenBaAgreement(TsManagerInitiatorStationTidParams_t *TsDb, TxPd_t *packetDescriptor)
{	
	uint8 vapIndex = 0;
	uint16 timeout = 0;
	
	/* Prepare the entry for ADDBA request - return all the fields to default */
	TsDb->commonPart.dialogToken++;
	if(0 == TsDb->commonPart.dialogToken) //KLOCWORK IGNORE. klocwork does not know that dialogtoken can have 0xFFFFFFFF while getting into this function.
	{
		/* Dialog token cant be 0 */
		TsDb->commonPart.dialogToken = TS_MANAGER_INITIATOR_FIRST_DIALOG_TOKEN;
	}
	StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
	TsDb->commonPart.amsduSupport = TsManagerInitiatorVapDatabase[vapIndex].amsduSupportPerTidArray[TsDb->commonPart.tid];
	TsDb->commonPart.bufferSize = TsManagerConvertWinSizeToBufferSize(TsManagerInitiatorVapDatabase[vapIndex].bufferSizePerTidArray[TsDb->commonPart.tid]);
	//if sta doesn't support HE -> limit buffer size to 64
	if (TsManager_IsStaHe(TsDb->commonPart.stationId) == FALSE)
	{
		TsDb->commonPart.bufferSize = MIN(TS_MANAGER_BUFFER_SIZE_64, TsDb->commonPart.bufferSize);
	}
	TsDb->commonPart.deleteBaAgreementOnInactivity = TsManagerInitiatorVapDatabase[vapIndex].deleteBaAgreementOnInactivity;
	timeout = TsManagerInitiatorVapDatabase[vapIndex].maxInactivityTimeout;
	if (TsDb->commonPart.deleteBaAgreementOnInactivity == FALSE)
	{
		/*Try to negotiate infinite timeout*/
		timeout = 0;
	}

	ILOG0_DDD("[tsManagerInitiatorTryToOpenBaAgreement], TsManager_IsStaHe(TsDb->commonPart.stationId)=%d, TsDb->commonPart.amsduSupport = %d, TsDb->commonPart.bufferSize = %d", TsManager_IsStaHe(TsDb->commonPart.stationId), TsDb->commonPart.amsduSupport, TsDb->commonPart.bufferSize);
    TsManager_TransmitAddbaRequest(TsDb, packetDescriptor, timeout);	
	tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_TRYING_TO_OPEN_SENDING_ADDBA_REQUEST);	
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventAddbaRequestWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
}

/**********************************************************************************

tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation


Description:
------------
This function Tries to remove PD request when removing the station

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTryToRemovePdRequestWhenRemovingStation(TsManagerInitiatorStationTidParams_t *TsDb)
{	
	bool removeStatus = FALSE;
	
	removeStatus = tsManagerInitiatorRmRemove(TsDb);
	if (removeStatus)
	{
		/* Request was removed successfully - move to Idle */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_IDLE);
		TsManager_TidMovedToIdleCallback(TsDb->commonPart.stationId);
	}
	else
	{
		/* Remove request failed - wait for message */
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_REMOVING);
	}	
}

/**********************************************************************************

tsManagerInitiatorSaveAddbaResponseParameters


Description:
------------
This function saves ADDBA response parameters in case of success

Input: 
-----
TsDb
addbaResponsePayload - the address of the ADDBA response payload

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorSaveAddbaResponseParameters(TsManagerInitiatorStationTidParams_t *TsDb, FM_PAYLOAD_ADDBA_RES *addbaResponsePayload)
{	
	uint16 timeout  = 0;
	uint8 vapIndex = 0;
	
	StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);

	//TBD what to do in case the recipient sent buffer size bigger than the limitation ?
	TsDb->commonPart.bufferSize = TsManagerConvertWinSizeToBufferSize(MIN(addbaResponsePayload->u16BlockAckParameterSet.BlockAckBufferSize, TsManagerInitiatorVapDatabase[vapIndex].bufferSizePerTidArray[TsDb->commonPart.tid]));

	TsDb->commonPart.amsduSupport = addbaResponsePayload->u16BlockAckParameterSet.BlockAckAmsduSupported & 
									TsManagerInitiatorVapDatabase[vapIndex].amsduSupportPerTidArray[TsDb->commonPart.tid];
		
	timeout = addbaResponsePayload->u16BlockAckTimeOut;
	if (0 == timeout)//TBD what to do in this case???
	{
		/* Recipient wants to disable inactivity mechanism */
		timeout = TsManagerInitiatorVapDatabase[vapIndex].maxInactivityTimeout;
		/*If STA is OK with 0 mark TID according to our VAP configuration*/
		TsDb->commonPart.deleteBaAgreementOnInactivity = TsManagerInitiatorVapDatabase[vapIndex].deleteBaAgreementOnInactivity;
	}
	else
	{
		timeout = TS_MANAGER_CONVERT_TIME_UNITS_INTO_250_MILLISECONDS_UINTS(timeout);
		/* Taking the minimum between internal timeout and the recipient timeout */
		timeout = MIN(timeout,TsManagerInitiatorVapDatabase[vapIndex].maxInactivityTimeout);
		/*STA sent timeout, no matter what we is the VAP configuration , mark the TID to delete the BA Agreement on timeout*/
		TsDb->commonPart.deleteBaAgreementOnInactivity = TRUE;
	}
	/* This field is used for BA agreement open retries - overrun this field only in case of success,
	in case the TID moves to open use the calculated timeout in case of closed the value will be 
	reset when openeing again */
	TsDb->commonPart.timeout = timeout;
}

/**********************************************************************************

tsManagerInitiatorTryToAllocatePdForDelba


Description:
------------
This function tries to allocate a PD for sending DELBA. in case it succeeds it sends
the DELBA

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTryToAllocatePdForDelba(TsManagerInitiatorStationTidParams_t *TsDb)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	
	tsManagerInitiatorBaAgreementClosedUpdateCounter(TsDb->commonPart.stationId);
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	TsDb->commonPart.reason = TS_MANAGER_END_BA_REASON;
	if (allocationStatus)
	{
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventDelbaWasSent[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++; 
		tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
		TsManager_TransmitDelba(&TsDb->commonPart, (TxPd_t *)pdRequestFillParameters.packetDescriptor);
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId == INVALID_FREE_PENDING_LIST_ID)
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING);
			tsManagerInitiatorDelbaConfirmedInClosing(TsDb, NULL);
		}
		else
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_CLOSING_WAITING_FOR_PD_TO_SEND_DELBA);
		}
	}
}

/**********************************************************************************

tsManagerInitiatorTryToAllocatePdForBar


Description:
------------
This function tries to allocate PD for BAR

Input: 
-----
TsDb

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorTryToAllocatePdForBar(TsManagerInitiatorStationTidParams_t *TsDb)
{	
	uint8 allocationStatus = FALSE;
	RmPdRequestFillParameters_t pdRequestFillParameters;
		
	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));	
	allocationStatus = tsManagerInitiatorRmRequest(TsDb, &pdRequestFillParameters);
	if (allocationStatus)
	{
		TsDb->commonPart.pdOffset = TS_MANAGER_CONVERT_PD_TO_OFFSET(pdRequestFillParameters.packetDescriptor);

		/*If PBAC is negotiated need to send ADDBA REQ instead*/
		if (TsDb->commonPart.pbac == FALSE)
		{
			//Send Lock Request
			tsManagerInitiatorLockQueue(TsDb);

			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_LOCK);
		}
		else
		{
			uint8 vapIndex = 0;
			
			StaDb_GetVapId(TsDb->commonPart.stationId, &vapIndex);
			/*Usde the timeout already negotiated*/
			TsManager_TransmitAddbaRequest(TsDb, (TxPd_t *)pdRequestFillParameters.packetDescriptor, TsDb->commonPart.timeout);
			/*Skip a few states to Wait for ADDBA REQ confirmation*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_SENDING_BAR);
		}
	}
	else
	{
		/* PD allocation falied */		
		TsDb->commonPart.pdRequestId = pdRequestFillParameters.requestId;
		if (pdRequestFillParameters.requestId != INVALID_FREE_PENDING_LIST_ID)
		{
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED_WAITING_FOR_PD_TO_SEND_BAR);	
		}
		else
		{
			/*No PD and no resource ID*/
			/*Change State to Opened as we may get here if BAR or ADDBA REQ is confirmed during Sending BAR*/
			tsManagerInitiatorChangeState(TsDb, TS_MANAGER_INITIATOR_TID_STATE_OPENED);
			/*Clear PD offset - may be set to discarded if more packets were discarded while we were waiting for BAR transmission*/
			TsDb->commonPart.pdOffset = TS_MANAGER_INITIATOR_PD_NULL;
		}
	}
}

/**********************************************************************************

tsManagerInitiatorBaAgreementOpenedUpdateCounters 


Description:
------------
this function updates the counters when new BA agreement was opened in the initiator

Input: 
-----
TsDb

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBaAgreementOpenedUpdateCounters(StaId stationIndex)
{
	uint8 vapIndex = 0;
	
	TsManagerInitiatorGlobalParameters.numOfGlobalOpenBaAgreements++;
	TsManagerInitiatorStationOpenedBaAgreements[stationIndex]++;

	StaDb_GetVapId(stationIndex, &vapIndex);
	TsManagerInitiatorVapDatabase[vapIndex].openedBaAgreements++;
}


/**********************************************************************************

tsManagerInitiatorBaAgreementClosedUpdateCounter 


Description:
------------
this function updates the counters when agreement was closed in the initiator


Input: 
-----
TsDb


	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void tsManagerInitiatorBaAgreementClosedUpdateCounter(StaId stationIndex)
{
	uint8 vapIndex = 0;
	
	TsManagerInitiatorGlobalParameters.numOfGlobalOpenBaAgreements--;
	TsManagerInitiatorStationOpenedBaAgreements[stationIndex]--;

	StaDb_GetVapId(stationIndex, &vapIndex);
	TsManagerInitiatorVapDatabase[vapIndex].openedBaAgreements--;
}

/**********************************************************************************

tsManagerInitiatorIsAllowedToOpenBaAgreement 


Description:
------------
this function returns if a specific station is currently allowed to open new BA agreement
in the initiator side

Input: 
-----
TsDb->commonPart.stationId - the station to open BA agreement for


	
		
Output:
-------
	

Returns:
--------
	allowed - indicates if the specific station is allowed to open new BA agreement 
	
**********************************************************************************/

static uint8 tsManagerInitiatorIsAllowedToOpenBaAgreement(StaId stationIndex)
{
	uint8 vapIndex = 0;

	StaDb_GetVapId(stationIndex, &vapIndex);
	/* First check global limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if (TsManagerInitiatorGlobalParameters.numOfGlobalOpenBaAgreements >= TsManagerInitiatorGlobalParameters.maxGlobalBaAgreements)
	{
		return FALSE;
	}
	/* Check Vap limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if (TsManagerInitiatorVapDatabase[vapIndex].openedBaAgreements >= TsManagerInitiatorVapDatabase[vapIndex].maxBaAgreementPerVap)
	{
		return FALSE;
	}	
	/* Check station limitation. >= since the limitation can be changed on the fly - reduced to smaller number */
	if (TsManagerInitiatorStationOpenedBaAgreements[stationIndex] >= TsManagerInitiatorVapDatabase[vapIndex].maxBaAgreementsPerStation)
	{
		return FALSE;
	}

	/* Check Global limitation since the limitation can be changed on the fly by driver */
	if(TsManager_GlobalDb.enableBa == FALSE)
	{
		return FALSE;
	}	

	return TRUE;
}


/**********************************************************************************

tsManagerInitiatorIncreaseTimeForNextRetry 


Description:
------------
This function increases the time for next trial of opening BA agreement in case of
opening BA agreement failure

Input: 
-----
TsDb->commonPart.stationId - the station index
tid - the TID for which to increase the time
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/

static void tsManagerInitiatorIncreaseTimeForNextRetry(TsManagerInitiatorStationTidParams_t *TsDb)
{
#if defined (ENET_INC_HW_FPGA)
	UNUSED_PARAM(TsDb);
	return; // we don't want to increase timeout in palladium or fpga
#else

	/* Increase time to next retry - multiply by 2 and add 1 in order to save the form of 2^n - 1 */
	/* No need to check wrap arounf since the field is uint8 - automatic wrap around */

	//("tsManagerInitiatorIncreaseTimeForNextRetry timeout %d", TsDb->commonPart.timeout);
	/* Do not increase delays since the delay in FPGA is already long */
	TsDb->commonPart.timeout <<= TS_MANAGER_TIME_TO_TRY_TO_OPEN_BA_AGREEMENT_MULTIPLY_BY_SHIFT;
	TsDb->commonPart.timeout ++;
	
#endif
}


/**********************************************************************************

tsManagerRecipientProcessAddbaReqExtension 


Description:
------------
Process the extension part of the ADDBA REQ. It contains IEs.

Input: 
-----
		
Output:
-------
	
Returns:
--------

	
**********************************************************************************/
static void tsManagerInitiatortProcessAddbaResExtension(TsManagerInitiatorStationTidParams_t *TsDb, TsManagerAddbaParams_t* addbaParams)
{
	FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT* AddbaExtensionIe = &(addbaParams->addbaPayload->AddbaExtension);
	AddbaCapabilities_t*				addbaCapabilities;
	int32								length;
	uint8								noFragmentation = TRUE;			// Init value - no frags are supported for non-HE STAs.
	uint8								heFragmentationOperation = 0; 	// Init value - frag level 0 (no frags) for HE STAs.

	// Calculate the length of the IEs that follow that regular ADDBA RES payload
	length = addbaParams->payloadLength - sizeof(FM_PAYLOAD_ADDBA_REQ);

	// There can be several IEs after the ADDBA payload (GCR / Multi band / TCLAS / ADDBA EXT / etc...)
	// Currently we only support the ADDBA EXTENSION, so we look for it, and skip the others if exist.
	while (length > 0)
	{
		// We assume that all elements begins with element ID and length. 
		// We use the addba extension structure, even if it's not addba extension, just to extract the ID and length.
		if (AddbaExtensionIe->u8ElementId != FM_EL_ADDBA_EXTENSION)
		{
			// This IE is not ADDBA EXTENSION. Skip over it. (take length + 2 bytes of u8ElementId and u8Length itself)
			AddbaExtensionIe = (FM_PAYLOAD_ADDBA_EXTENSION_ELEMENT*)((uint32)AddbaExtensionIe + AddbaExtensionIe->u8Length + 2);

			// Getting length = 0 will get us stuck in an endless loop
			if(AddbaExtensionIe->u8Length == 0)
			{
				// during gen6 ramp up we should have this assert to catch bad behaviour of ourselves in WDS and Other HE STAs.
				// When ramp up is done, the assert should be removed and replaced with the log below, since we can't do assert when the STA do something wrong.
				// Currently we keep the ASSERT only for HE STAs which are also still under development.
				if (TsManager_IsStaHe(TsDb->commonPart.stationId))
				{
					ASSERT(0);
				}
				ILOG0_V("tsManagerInitiatortProcessAddbaResExtension - IE lenght is zero, break ADDBA extension parsing "); 
				break; 
			}			
			// substract the length of this unknown IE
			length = length - (AddbaExtensionIe->u8Length + 2);
		}
		else
		{
			//We found an ADDBA EXTENSION.
			addbaCapabilities = (AddbaCapabilities_t*)&(AddbaExtensionIe->u8AddbaCapabilities);

			// Extract the fragmentation support information for non-HE client. (the field is true when frags are not supported)
			noFragmentation = addbaCapabilities->no_fragmentation;

			// Extract the dynamic fragmentation level support information for HE client.
			heFragmentationOperation = addbaCapabilities->he_fragmentation_operation;	
			break;
		}
	}

	// Set the fragmentation level in the STA DB.
#ifdef ENET_INC_ARCH_WAVE600D2
	StaDB_SetMinHeFragLevel(TsDb->commonPart.stationId, heFragmentationOperation, TsDb->commonPart.tid);
#elif defined (ENET_INC_ARCH_WAVE600)
	StaDB_SetMinHeFragLevel(TsDb->commonPart.stationId, heFragmentationOperation);
#endif //ENET_INC_ARCH_WAVE600D2
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

TsManagerInitiator_AddStation 


Description:
------------
add new station to the TS manager initiator

Input: 
-----
TsDb
allowed - flag that inidcates if the station is allowed to open BA agreements
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_AddStation(StaId stationIndex, uint8 allowed, uint8 pbac)
{
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
    uint32 i = 0;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_CLOSED;
  
  TsManagerInitiatorStationGlobalStatistics.InitiatorStationStatisticsEventAdd[ConfigurationManager_GetBandForStation(stationIndex)]++;

	if(!allowed)
	{
		state = TS_MANAGER_INITIATOR_TID_STATE_NOT_ALLOWED;
	}

	for(i = 0; i < NUM_OF_TID; i++)		
	{
		TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][i];
		DEBUG_ASSERT(TsDb->commonPart.state == TS_MANAGER_INITIATOR_TID_STATE_IDLE);
		/* Dialog token cannot be 0 - it will be changed when sending the first ADDBA request*/
		TsDb->commonPart.dialogToken = 0;
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_ADD_STA);
		tsManagerInitiatorChangeState(TsDb, state);
	    TsDb->commonPart.pbac = pbac;
	}
}

/**********************************************************************************

TsManagerInitiator_RemoveStation 


Description:
------------
remove all station TIDs in the initiator

Input: 
-----
TsDb
tid - the TID to remove
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_RemoveStation(StaId stationIndex)
{
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
   	uint8 i = 0;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;
	//("TsManagerInitiator_RemoveStation uint16 stationIndex %d ", uint16 stationIndex);

	TsManagerInitiatorStationGlobalStatistics.InitiatorStationStatisticsEventRemove[ConfigurationManager_GetBandForStation(stationIndex)]++; 

	for(i = 0; i < NUM_OF_TID; i++)
	{
		TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][i];
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_REMOVE_STA);
		state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
		TsManagerInitiatorRemoveTable[state](TsDb, NULL);
	}
}

/**********************************************************************************

TsManagerInitiator_AddbaResponseReceived 


Description:
------------
handle ADDBA response receivedt

Input: 
-----
TsDb->commonPart.stationId - the station for which the packet was received
addbaResponseFrame - the pointer to the frame (without the phy metrics)

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_AddbaResponseReceived(StaId stationIndex, TsManagerAddbaParams_t* addbaParams)
{
	TsManagerInitiatorTidState_e 			state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;
	uint8 									tid = 0;
	TsManagerInitiatorStationTidParams_t*	TsDb = NULL;
	AddbaResponseFrame_t*					addbaResponseFrame = (AddbaResponseFrame_t*)addbaParams->addbaPayload;
	
	tid = addbaResponseFrame->addbaResponsePayload.u16BlockAckParameterSet.BlockAckTid;
	if(tid >= NUM_OF_TID)
	{
		/* Should not be bigger than TID 7 - numbers above 7 meant for traffic stream agreement which is not supported */
		TsManagerInitiatorStationGlobalStatistics.InitiatorStationStatisticsEventAddbaResponseReceivedWithIllegalTid[ConfigurationManager_GetBandForStation(stationIndex)]++;
		return;
	}
	TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_ADDBA_RSP_RECEIVED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
	TsManagerInitiatorAddbaResponseReceivedTable[state](TsDb, addbaParams);
}

/**********************************************************************************

TsManagerInitiator_DelbaReceived 


Description:
------------
handle Delba received

Input: 
-----
TsDb
tid - the TID for which the DELBA was received
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_DelbaReceived(StaId stationIndex, uint8 tid)
{
	TsManagerInitiatorStationTidParams_t *TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_DELBA_RECEIVED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
	TsManagerInitiatorDelbaReceivedTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerInitiator_AddbaRequestConfirmation 


Description:
------------
handles ADDBA request confirmation

Input: 
-----
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_AddbaRequestConfirmation(TxPd_t *packetDescriptor)
{
	AddbaRequestFrame_t *addbaRequestFrame = NULL;
	uint8 tid = 0;
	StaId stationIndex = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	addbaRequestFrame = (AddbaRequestFrame_t *)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)packetDescriptor->packetPointer);
	stationIndex = packetDescriptor->txQStaId;
	tid = addbaRequestFrame->addbaRequestPayload.u16BlockAckParameterSet.BlockAckTid;
	TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_ADDBA_REQ_CONFIRMED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
	TsManagerInitiatorAddbaRequestConfirmationTable[state](TsDb, packetDescriptor);
}

/**********************************************************************************

TsManagerInitiator_DelbaConfirmation 


Description:
------------
handles DELBA confirmation

Input: 
-----
TsDb
tid - the TID for which the DELBA was sent
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_DelbaConfirmation(StaId stationIndex, uint8 tid)
{
	TsManagerInitiatorStationTidParams_t *TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_DELBA_CONFIRMED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
    TsManagerInitiatorDelbaConfirmationTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerInitiator_BarConfirmed 


Description:
------------
handles BAR confirmation

Input: 
-----
packetDescriptor - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_BarConfirmed(TxPd_t *packetDescriptor)
{
	StaId stationIndex = 0;
	uint8 tid = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	stationIndex = packetDescriptor->txQStaId;
    tid = packetDescriptor->txQTid;
	TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_BAR_CONFIRMED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
	TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventBarWasDiscarded[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
    TsManagerInitiatorBarConfirmationTable[state](TsDb, packetDescriptor);
}


/**********************************************************************************

TsManagerInitiator_BarStatus

Description:
------------
handles DELBA confirmation

Input: 
-----
packetDescriptor - the PD of the BAR

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_BarStatus(TxPd_t *packetDescriptor)
{
	StaId stationIndex = 0;
	uint8 tid = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	stationIndex = packetDescriptor->txQStaId;
    tid = packetDescriptor->txQTid;
	TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_BAR_STATUS);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
    TsManagerInitiatorBarStatusTable[state](TsDb, packetDescriptor);
}

/**********************************************************************************

TsManagerInitiator_PdAllocated 


Description:
------------
handles Pd allocated from the resourec manager

Input: 
-----
stationTidEntry - the address of the station/TID entry
packetDescriptor - the address of the PD that was allocated for this station/TID

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_PdAllocated(StaId stationIndex, uint8 tid, TxPd_t *packetDescriptor)
{
	TsManagerInitiatorStationTidParams_t *TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_PD_ALLOCATED);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
	ResourceManager_ReleaseRequest(TsDb->commonPart.pdRequestId, DESC_POOL_TS_MANAGER);
    TsManagerInitiatorPdAllocatedTable[state](TsDb, packetDescriptor);
}

/**********************************************************************************

TsManagerInitiator_TimerExpired 


Description:
------------
Handles the event that TS manager timer has expired - process the entries in the 
calendar wheel

Input: 
-----


Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_TimerExpired(void)
{
	uint16 currentIndex = CALENDAR_WHEEL_NULL_ENTRY_INDEX;
	TsManagerInitiatorStationTidParams_t *entry = NULL;
	StaId stationIndex = 0;
	uint8 tid = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	currentIndex = CalendarWheel_TimerEvent(&TsManagerInitiatorCalendarWheel);
	while(CALENDAR_WHEEL_NULL_ENTRY_INDEX != currentIndex)
	{	
		entry = (TsManagerInitiatorStationTidParams_t *)TsManagerInitiatorStationTidDatabase + currentIndex;
		DEBUG_ASSERT(entry->commonPart.entryState == TS_MANAGER_CALENDAR_WHEEL_ENTRY_ENQUEUED);
		entry->commonPart.entryState = TS_MANAGER_CALENDAR_WHEEL_ENTRY_FREE;
		/* Save next before inserting to another list */
        currentIndex = entry->commonPart.calendarWheelEntry.nextIndex;

		stationIndex = entry->commonPart.stationId;
		tid = entry->commonPart.tid;
		TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
		state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;

		TsManagerInitiatorTimerEventTable[state](TsDb, NULL);
	}
}

/**********************************************************************************

TsManagerInitiator_DiscardedPacket 


Description:
------------
handles a data packet discarded

Input: 
-----
TsDb
tid - the TID to which the data packet is belong
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_DiscardedPacket(StaId stationIndex, uint8 tid)
{
	TsManagerInitiatorStationTidParams_t *TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_DISCARDED_PACKET);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
    TsManagerInitiatorPacketDiscardedTable[state](TsDb, NULL);
}

/**********************************************************************************

TsManagerInitiator_QueueIsLocked 


Description:
------------
handles a data packet discarded

Input: 
-----
stationIndex - SID
tid - the TID to which the data packet is belong
lockStatus - indicates if the lock succeeded
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_QueueIsLocked(StaId stationIndex, uint8 tid, TxSelectorLockStatus_e lockStatus)
{
	TsManagerInitiatorStationTidParams_t *TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;

	/*Handle Pending Lock Requests - do it before handling the lock event as we may request a PD therefore overruning the lock index*/
	tsManagerInitiator_HandlePendingLock();
	tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_QUEUE_LOCK);
    state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
    TsManagerInitiatorQueueIsLockedTable[state](TsDb, &lockStatus);
}

/**********************************************************************************

TsManagerInitiator_CloseTids 


Description:
------------
handles close TID request

Input: 
-----
TsDb
tidsBitmap - bitmap of the TIDS to be closed

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_CloseTids(StaId stationIndex, uint8 tidsBitmap)
{
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;
	uint8 tid = 0;
	uint32 tempBitmap = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;

	/* Using temp variable since the function uses uint32 */
	tempBitmap = tidsBitmap;
    while (tempBitmap)
    {
		tid = Utils_FindFirstSetAndClear(&tempBitmap);
		TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_CLOSE_TID);
		state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventCloseTid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		TsDb->commonPart.allowed = FALSE;
	    TsManagerInitiatorCloseTidTable[state](TsDb, NULL);
    }
}

/**********************************************************************************

TsManagerInitiator_OpenTids 


Description:
------------
handles open TID request

Input: 
-----
TsDb
tidsBitmap - bitmap of the TIDS to be opened

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_OpenTids(StaId stationIndex, uint8 tidsBitmap)
{
	TsManagerInitiatorTidState_e state = TS_MANAGER_INITIATOR_TID_STATE_IDLE;
	uint8 tid = 0;
	uint32 tempBitmap = 0;
	TsManagerInitiatorStationTidParams_t *TsDb = NULL;

	/* Using temp variable since the function uses uint32 */
	tempBitmap = tidsBitmap;
    while (tempBitmap)
    {
		tid = Utils_FindLastSetAndClear(&tempBitmap, CONVERT_BYTES_TO_BIT_INDEX(sizeof(uint32)));
		TsDb = &TsManagerInitiatorStationTidDatabase[stationIndex][tid];
		tsManagerInitiatorLogEvent(TsDb, TS_MANAGER_INITIATOR_TID_EVENT_OPEN_TID);
		state = (TsManagerInitiatorTidState_e)TsDb->commonPart.state;
		TsManagerInitiatorTidGlobalStatistics.InitiatorTidStatisticsEventOpenTid[ConfigurationManager_GetBandForStation(TsDb->commonPart.stationId)]++;
		
	    TsManagerInitiatorOpenTidTable[state](TsDb, NULL);
    }
}


/**********************************************************************************

TsManagerInitiator_ConfigureVapParameters 


Description:
------------
configures the VAP parameters in the initiator VAP database

Input: 
-----
vapIndex - the VAP index to configure 
vapConfigurationParameters - a pointer to the structure of the parameters

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_ConfigureVapParameters(uint8 vapIndex, TsManagerInitiatorVapConfigurationParams_t *vapConfigurationParameters)
{
	uint8 tid = 0;

	TsManagerInitiatorVapDatabase[vapIndex].maxBaAgreementPerVap= vapConfigurationParameters->maxBaAgreementsPerVap;
	TsManagerInitiatorVapDatabase[vapIndex].maxInactivityTimeout = TS_MANAGER_CONVERT_TIME_UNITS_INTO_250_MILLISECONDS_UINTS(vapConfigurationParameters->maxInactivityTimeout);
	TsManagerInitiatorVapDatabase[vapIndex].maxBaAgreementsPerStation = vapConfigurationParameters->maxBaAgreementsPerStation;
	TsManagerInitiatorVapDatabase[vapIndex].addbaRequestTimeout =  TS_MANAGER_CONVERT_MILLISECONDS_INTO_250_MILLISECONDS_UNITS(vapConfigurationParameters->addbaRequestTimeout);
	TsManagerInitiatorVapDatabase[vapIndex].deleteBaAgreementOnInactivity = vapConfigurationParameters->deleteBaAgreementOnInactivity;
    /* Fill structures for the initiator VAP database and the Initiator VAP database */
	for(tid = 0; tid < NUM_OF_TID; tid++)		
	{
		TsManagerInitiatorVapDatabase[vapIndex].amsduSupportPerTidArray[tid] = vapConfigurationParameters->amsduSupportPerTidArray[tid];		
		TsManagerInitiatorVapDatabase[vapIndex].bufferSizePerTidArray[tid] = vapConfigurationParameters->maxBufferSizePerTidArray[tid];		
	}   
}


/**********************************************************************************

TsManager_TsManagerInitiatorTidGlobalStatistics_Address 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic
	
**********************************************************************************/
uint32* tsManagerGetInitiatorTidGlobalStatisticsAddress()
{
	return ((uint32*)&TsManagerInitiatorTidGlobalStatistics);
}


/**********************************************************************************

TsManager_TsManagerInitiatorStationGlobalStatistics_Address 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic

**********************************************************************************/
uint16* tsManagerGetInitiatorStationGlobalStatisticsAddress()
{
	return ((uint16*)&TsManagerInitiatorStationGlobalStatistics);
}



/**********************************************************************************

TsManagerInitiator_Initialize 


Description:
------------
initialize internal databases and structures of the Ts manager initiator

Input:
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TsManagerInitiator_Initialize()
{
	uint32 stationIndex = 0;
	uint32 tidIndex = 0;

	/* Initialize Calendar wheel - timer will be set later */
	CalendarWheel_Initialize(&TsManagerInitiatorCalendarWheel, 
							 TS_MANAGER_CALENDAR_WHEEL_SIZE, 
							 TsManagerInitiatorCalendarWheelSlots, 
							 (uint32)TsManagerInitiatorStationTidDatabase, 
							 sizeof(TsManagerInitiatorStationTidParams_t));
	
	/* Initialize the internal structures - the databases are word alligned */
	memset32(TsManagerInitiatorStationTidDatabase, 0, CONVERT_BYTES_TO_WORDS(sizeof(TsManagerInitiatorStationTidParams_t)*HW_NUM_OF_STATIONS*NUM_OF_TID));
	memset(TsManagerInitiatorStationOpenedBaAgreements, 0, sizeof(TsManagerInitiatorStationOpenedBaAgreements));
	memset(TsManagerInitiatorVapDatabase, 0, sizeof(TsManagerInitiatorVapParams_t)*HW_NUM_OF_VAPS);
	memset(&TsManagerInitiatorGlobalParameters, 0, sizeof(TsManagerInitiatorGlobalParams_t));


	/* This initialization is done in case the global parameters message wont be sent from the driver */
	TsManagerInitiatorGlobalParameters.maxGlobalBaAgreements = TS_MANAGER_MAX_BA_AGREEMENT_INITIATOR_DEFAULT_VALUE;

	/* Initialize the initiator station/TID database */
    for(stationIndex = 0 ; stationIndex < HW_NUM_OF_STATIONS; stationIndex++)
    {
		for(tidIndex = 0; tidIndex < NUM_OF_TID; tidIndex++)
		{
			TsManagerInitiatorStationTidDatabase[stationIndex][tidIndex].commonPart.stationId = stationIndex;
			TsManagerInitiatorStationTidDatabase[stationIndex][tidIndex].commonPart.tid = tidIndex;
			TsManagerInitiatorStationTidDatabase[stationIndex][tidIndex].commonPart.entryType = TS_MANAGER_ENTRY_TYPE_INITIATOR;
		}
    }
	
	memset(&TsManagerInitiatorTidGlobalStatistics, 0, sizeof(TsManagerInitiatorTidGlobalStatistics_t));
	memset(&TsManagerInitiatorStationGlobalStatistics, 0, sizeof(TsManagerInitiatorStationGlobalStatistics_t));

	/*Initialize Ts Manager Initiator Lock Request Serializer indeces*/
	TsManagerInitiator_LockHead = TS_MANAGER_INITIATOR_INVALID_LOCK_IDX;
	TsManagerInitiator_LockTail = TS_MANAGER_INITIATOR_INVALID_LOCK_IDX;
}


