/***********************************************************************************
 File:			TwtManager.c
 Module:		TWT Manager
 Purpose: 		To manage TWT operations
 Description:   This file is the implementation of the TWT manager which is 
 				responsible of creating and canceling TWT agreements
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#ifdef ENET_INC_ARCH_WAVE600D2
#include "TwtRegs.h"
#endif
#include "TxSelector_Api.h"
#include "TxSelectorRegs.h"
#include "ResourceManager_API.h"
#include "StaDatabase_Api.h"
#include "TxPacketsClassifier_API.h"
#include "queue_utility.h"
#include "OSAL_Kmsg.h"
#include "TwtManager.h"
#include "TwtManager_API.h"
#include "HwQManager_API.h"
#include "Rd_Descriptors.h"
#include "TxPd_Descriptors.h"
#include "frame.h"
#include "Pac_Api.h"
#include "RegAccess_Api.h"
#include "DescriptorsDefinitions.h"
#include "HwMemoryMap.h"
#include "BSSmanager_API.h"
#include "mhi_umi.h"
#include "OSAL_UpperMacMessages.h"
#include "OSAL_Tasks.h"
#include "OSAL_Api.h"
#include "PacketDescriptor.h"
#include "OSAL_Interrupts.h"
#include "ConfigurationManager_api.h"
#include "logmacros.h"
#include "loggroups.h"
#include "StatisticsManager_api.h"
#include "Statistics_Descriptors.h"
#include "PreAggregator_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID                                           GLOBAL_GID_HE_TWT
#define LOG_LOCAL_FID                                           0

#define TWT_SETUP_ACTION_CODE                                   6
#define TWT_TEARDOWN_ACTION_CODE                                7
#define TWT_REQUEST                                             0
#define TWT_SUGGEST                                             1
#define TWT_DEMAND                                              2
#define TWT_ACCEPT                                              4
#define TWT_ALTERNATE                                           5
#define TWT_DICTATE                                             6
#define TWT_REJECT                                              7
#define TWT_NO_AGREEMENT                                        0
#define TWT_ONLY_AGREEMENT                                      1
#define NUM_OF_TWT_FLOWS                                        8
#define TWT_ON                                                  1
#define TWT_OFF                                                 0
#define INDIVIDUAL_TWT                                          0
#define BAND1_OFFSET_FROM_BAND0                                 BAND1_INTERNAL_SIZE
#define TWT_WAKE_INTERVAL_MANTISSA                              16 // in ms
#define TWT_WAKE_INTERVAL_EXP                                   6 // 16 * 2 ^ 6 = 1024 ms
#define TWT_CONST_INTERVAL                                      5000 
#define TWT_WAKE_DURATION                                       125 //125 * 256us = 32 ms
#define TWT_SP_ONE                                              0 
#define TWT_PAYLOAD_LENGTH                                      sizeof(TwtSetupFramePayload_t) - 3
#define TWT_SP_HIGH                                             1000 // 1000 * 32us = 32 ms
#define TWT_SP_LOW                                              31000 // 31000 * 32us = 992 ms
#define TWT_SHIFT_FLOW_ID                                       13 // 3 MSB - flow ID , 13 LSB - sta ID
#define TWT_MASK_STA_ID                                         0x1FFF // 3 MSB - flow ID , 13 LSB - sta ID
#define TWT_ELEMENT_ID                                          216
#define TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR                     5
#define TWT_CHANGE_UNITS_SHIFT_WAKE_DURATION                    8
#define COMMON_TWT_FRAME                                        2
#ifdef PF_TWT
#define CHANGE_SP_LENGTH_FOR_PF                                 3168 // may change according to needed - must be a multiple of 32
#define MAX_PSDU_FOR_TWT                                        2700 // may change according to needed
#endif

#if defined (TWT_SELECTOR_WORKAROUND) 
#define TWT_COUNTER_FIXED_OFFSET_IN_SELECTOR_RAM                TX_SELECTOR_BASE_ADDRESS + TX_SELECTOR_TWT_SP_GROUP_COUNTER_OFFSET_FROM_RAM_START
#define TWT_ZERO_VAL                                            0
#define TWT_BEFORE_ZERO                                         0xFFFFF
#define TX_SELECTOR_TWT_SP_GROUP_COUNTER_OFFSET_FROM_RAM_START  0x2090 
#define TWT_RAM_LINE                                            0x8 // 2 words
#endif

#ifdef ENET_INC_ARCH_WAVE600D2
/* Used for polling TWT registers */
#define TWT_POLLING_TIMEOUT                     (5)
#define TWT_LONG_POLLING_TIMEOUT                (55) /* according to VLSI analysis */
#define REG_TWT_SP_MASK                         (1u<<31)
#define REG_TWT_SP_SHIFT                        (31u)
#endif

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/
#define TWT_ASSIGN_STA_AND_FLOW_IDS_IN_CONTEXT_FIELD(STA,FLOW)  ((STA &  TWT_MASK_STA_ID) | (FLOW << TWT_SHIFT_FLOW_ID)) //we save in the context the station ID in the 13 LSB and the flow ID in the 3 MSB                     
#define TWT_EXTRACT_STA_ID_FROM_CONTEXT(CTX)                    (CTX  &  TWT_MASK_STA_ID)                         
#define TWT_EXTRACT_FLOW_ID_FROM_CONTEXT(CTX)                   (CTX  >> TWT_SHIFT_FLOW_ID)                    

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

typedef enum
{
    TWT_MANAGER_GLOBAL_STATE_DISABLE,
    TWT_MANAGER_GLOBAL_STATE_ENABLE,
    TWT_MANAGER_GLOBAL_STATE_IN_DISABLE_PROCESS,    
} TwtManagerGlobalState_e;

typedef enum
{
    TWT_MANAGER_FLOW_STATE_INACTIVE,
    TWT_MANAGER_FLOW_STATE_ACTIVE,
    TWT_MANAGER_FLOW_STATE_PROCESSING_TWT_REQ,
    TWT_MANAGER_FLOW_STATE_WAIT_FOR_RESOURCE_ALLOCATION,
    TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_PD,
    TWT_MANAGER_FLOW_STATE_WAIT_FOR_TX_CFM,
    TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_TX_CFM,
} TwtManagerFlowState_e;

typedef enum
{
    TWT_MANAGER_ACTION_TYPE_TEARDOWN = 0,
    TWT_MANAGER_ACTION_TYPE_SETUP    = 1,  
} TwtManagerActionType_e;

typedef struct _Agreement_t
{   
    TxPd_t*                     pd;
    TwtFramePayload_t           twtFramePayload;
    uint16                      pdRequestId;
    uint8                       spId;
    TwtManagerFlowState_e       state;
} Agreement_t;

typedef struct _TwtManagerStaDb_t
{
    Agreement_t                 flowIdDb[NUM_OF_TWT_FLOWS];    
    uint8                       numOfAgreementsExist :3; // numbe of agreements with the STA 
    uint8                       agreementToAnnul     :3; // flow Id of the agreement to annul
    uint8                       isStaHe              :1;
    uint8                       sendingAccept        :1; // accept response is being sent -  only for this phase
    uint8                       isStaMfp             :1;
    uint8                       isXFilterOpen        :1;
    uint8                       isStaConnected       :1;
    uint8                       reserved             :5;
} TwtManagerStaDb_t;

typedef struct _TwtManagerDatabase_t
{
    TwtManagerStaDb_t           twtStationDb[NUM_OF_STAS];
	uint32						totalAgreements;
    TwtManagerGlobalState_e     featureState; 
    uint8                       clientIdForVapManager;
} TwtManagerDatabase_t;

typedef struct _SpAvailAns_t
{     
    TwtFramePayload_t           resFramePayload;
    uint8                       spId; 
} SpAvailAns_t;

typedef struct _ServicePeriod_t
{     
    uint32                      twtSpLowLimit; 
    uint32                      twtSpStartTsf; 
    uint32                      twtSpHighLimit;       
    uint16                      numOfStaInServicePeriod;
    TwtManagerFlowType_e        twtType; 
    bool                        isSelectorUpdated;
} ServicePeriod_t;


typedef struct _SpAvailabilityManagerDataBase_t
{
    ServicePeriod_t             twtServicePeriodDb[MAX_NUM_OF_SP];                      
    uint32                      targetWakeTimeOfSpOne; // for degenerated sp availability only, delete next phase
} SpAvailabilityManagerDataBase_t;

    
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
#ifdef ENET_INC_ARCH_WAVE600D2
static void twtManagerConfigureTwtRegsNewSp(uint32 twtSpLowLimit, uint32 twtSpStartTsf, uint32 twtSpHighLimit, uint8 twtSpGroupIdx, uint8 bandId);
static void twtManagerConfigureTwtRegsUpdateSta(uint8 twtSpStaGroupIdx, uint16 twtSpStaIdx, bool twtSpStaValue);
static void twtManagerTypeUpdateEnable(void);
static void twtManagerSwHalt(void);
#endif
static void twtManagerConfigureTwt(K_MSG *psMsg);
static void twtManagerAddSta(K_MSG *psMsg);
static void twtManagerSetFilter(K_MSG *psMsgTwt);
static void twtManagerStaManagerSendConfirm(StaId staId);
static void twtManagerHandleRecievedTwtFrame(K_MSG *psMsg); 
static void twtManagerFrameParser(TwtFramePayload_t* twtFramePayload, StaId staId);
static void twtManagerHandler(TwtManagerActionType_e actionType, StaId staId, TwtFramePayload_t*  twtFramePayload, uint8 flowId);
static void twtManagerBuildAgreement(StaId staId, uint8 spId, uint8 flowId);
static void twtManagerCancelAgreement(StaId staId, uint8 spId, uint8 flowId);
#ifdef ENET_INC_ARCH_WAVE600B
static void twtManagerSetHeMuKeepUser(bool setKeepUser);
#endif
static void twtManagerSpAvailabilityManager(StaId staId, TwtFramePayload_t*  recievedFramePayload, SpAvailAns_t* spAvailAns);
static void twtManagerSelectorConfiguration(StaId staId, uint8 spId, uint8 flowId, TwtManagerActionType_e actionType);
static void twtManagerSelectorConfigurationOfNewSp(uint8 spId, StaId staId);
static void twtManagerSendTeardownFrame(StaId staId, uint8 spId, uint8 flowId);
static void twtManagerUpdateDataBaseAddAgreement(StaId staId, uint8 spId, uint8 flowId);
static void twtManagerUpdateDataBaseDelAgreement(StaId staId, uint8 spId, uint8 flowId);
static void twtManagerUpdateStatsDataBase(StaId staId, uint8 spId, uint8 flowId);
static void twtManagerFrameBuilder(StaId staId, uint8 flowId);
static void twtManagerResourceCfmReceived(K_MSG *psMsg);
static void twtManagerSendTwt(TxPd_t* pdPointer, StaId staId, uint8 flowId);
static void twtManagerTxCfmReceived(K_MSG *psMsg);
static void twtManagerSetState(StaId staId, uint8 flowId, TwtManagerFlowState_e state);
static void twtManagerRemoveSta(K_MSG *psMsgTwt);
static uint8 twtManagerFindFirstActiveAgreemnt(uint16 StaId);


void TwtManager_TaskEntry(K_MSG *twtManagerMessage);


/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const FunctionEntry_t afpTaskTable[TASK_TWT_MANAGER_END - TASK_TWT_MANAGER_START]=
{
    {twtManagerConfigureTwt,                    DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_ADD_VAP)},
    {twtManagerAddSta,                          DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_ADD_STA)},
	{twtManagerHandleRecievedTwtFrame,	       	DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_TWT_FRAME_RECIEVED)}, 
	{twtManagerResourceCfmReceived,             DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_PD_ALLOC_CFM)},
    {twtManagerTxCfmReceived,                   DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_TX_CFM)},  
    {twtManagerRemoveSta,                       DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_REMOVE_STA)},  
    {twtManagerSetFilter,                       DOUBLE_CHECK_MSG_TYPE(TWT_MANAGER_SET_FILTER)},
};

static TwtManagerDatabase_t twtDb; 
static SpAvailabilityManagerDataBase_t spAvailabilityDb;
TwtStatistics_t TwtStatistics;


/**********************************************************************************

TwtManager_TaskEntry 

Description:
------------
the entry point of the TWT manager task

Input: 
-----
twtManagerMessage - pointer to the message to handle	
			
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
void TwtManager_TaskEntry(K_MSG *twtManagerMessage)
{
	/* Use common task switching and Table */ 
	vTaskDispatcher(twtManagerMessage, afpTaskTable, TASK_TWT_MANAGER_START, TASK_TWT_MANAGER_END);
}


/*---------------------------------------------------------------------------------

/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

twtManagerConfigureTwt 

Description:
------------
configure the TWT feature on

Input: 
-----
K_MSG *psMsg - pointer to the message to handle	
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerConfigureTwt(K_MSG *psMsg)
{
    K_MSG*                     pVapMsg = (K_MSG *) (*((uint32 *)pK_MSG_DATA(psMsg)));
	UMI_SET_BSS*               pVapSetBss = (UMI_SET_BSS*) pK_MSG_DATA(pVapMsg);
    K_MSG*                     psNewMsg; 
    BSS_MANAGER_CONFIRM_EVENT* confirmEvent; 
	uint8                      vapIndex = pVapSetBss->vapId;
    bool                       twtMode = pVapSetBss->twtOpreationMode;

    if ((twtMode == TRUE) && (twtDb.featureState == TWT_MANAGER_GLOBAL_STATE_DISABLE))
    {
       twtDb.featureState = TWT_MANAGER_GLOBAL_STATE_ENABLE; 
       ILOG0_V("Twt manager: enable TWT feature"); 
    }

	/* Send message to BSS manager - confirm on Add vap event  */
	psNewMsg = OSAL_GET_MESSAGE(sizeof(BSS_MANAGER_CONFIRM_EVENT));
	confirmEvent = (BSS_MANAGER_CONFIRM_EVENT *)(psNewMsg->abData);
	confirmEvent->vapId = vapIndex;
	confirmEvent->eventIndex = VAP_MANAGER_SET_BSS;
	confirmEvent->clientId = twtDb.clientIdForVapManager; 
    OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psNewMsg, vapIndex); 

    
}

#if 0
static void twtManagerConfigureTwt(K_MSG *psMsg)
{
    UMI_TWT_CONFIG* pTwtConfigMsg;
    uint16          staId;
    uint8           flowId; 
    uint8           spId;

    pTwtConfigMsg = (UMI_TWT_CONFIG *)pK_MSG_DATA(psMsg);
        
    if (pTwtConfigMsg->enableTwt == FALSE) /* disable */
    {   
        twtDb.featureState = TWT_MANAGER_GLOBAL_STATE_IN_DISABLE_PROCESS; 
        ILOG0_V("Twt manager: disable TWT feature");
        /*cancel all active agreements*/
        for (staId = 0; staId < NUM_OF_STAS; staId++)
        {
            for (flowId = 0; flowId < NUM_OF_TWT_FLOWS; flowId++) 
            {
                 if (twtDb.twtStationDb[staId].flowIdDb[flowId].state == TWT_MANAGER_FLOW_STATE_ACTIVE)
                 {
                    spId = twtDb.twtStationDb[staId].flowIdDb[flowId].spId;
                    twtManagerSendTeardownFrame(staId, spId, flowId);
                 }               
            }
        }
        twtDb.featureState = TWT_MANAGER_GLOBAL_STATE_DISABLE; 
        ILOG0_V("Twt manager: TWT feature disabled");
    }
    else
    {
       twtDb.featureState = TWT_MANAGER_GLOBAL_STATE_ENABLE; 
       ILOG0_V("Twt manager: enable TWT feature"); 
    }
    /*send confirmation to host*/
    OSAL_SEND_MESSAGE(UM_MAN_TWT_CONFIG_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);

}
#endif

#ifdef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************
twtManagerConfigureTwtRegsNewSp

Description: configure regs when new SP created
------------
	
Input:
-----
uint32 twtSpLowLimit
uint32 twtSpStartTsf
uint32 twtSpHighLimit
uint8  twtSpGroupIdx
uint8  vapId

Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void twtManagerConfigureTwtRegsNewSp(uint32 twtSpLowLimit, uint32 twtSpStartTsf, uint32 twtSpHighLimit, uint8 twtSpGroupIdx, uint8 bandId)
{    
    RegTwtTxTwtSpGroupLowPhase_u lowPhaseReg;
    RegTwtTxTwtSpGroupStartTsf_u startTsfReg; 
    RegTwtTxTwtSpGroupUpdate_u   groupUpdateReg;
    bool                         done;

    done = FALSE;
    lowPhaseReg.val = 0;
    lowPhaseReg.bitFields.twtSpLowLimit = twtSpLowLimit;
    RegAccess_WritePerBand(REG_TWT_TX_TWT_SP_GROUP_LOW_PHASE ,lowPhaseReg.val, bandId);
    startTsfReg.val = 0;
    startTsfReg.bitFields.twtSpStartTsf = twtSpStartTsf;
    RegAccess_WritePerBand(REG_TWT_TX_TWT_SP_GROUP_START_TSF ,startTsfReg.val, bandId);
    groupUpdateReg.val = 0;
    groupUpdateReg.bitFields.twtSpHighLimit = twtSpHighLimit;
    groupUpdateReg.bitFields.twtSpGroupIdx = twtSpGroupIdx;  
    RegAccess_WritePerBand(REG_TWT_TX_TWT_SP_GROUP_UPDATE ,groupUpdateReg.val, bandId); // pulse write
    /*polling the "done" bit*/
    ILOG0_D("[twtManagerConfigureTwtRegsNewSp], twt registers in selector, new SP registers updated, waiting for selector. SP ID: %d", twtSpGroupIdx);
    RegAccess_WaitForFieldPerBand(REG_TWT_TX_TWT_SP_GROUP_UPDATE, REG_TWT_SP_SHIFT, REG_TWT_SP_MASK, TRUE, TWT_LONG_POLLING_TIMEOUT, &done, bandId);
    ILOG0_DDD("[twtManagerConfigureTwtRegsNewSp], twt in selector: high: %d, low: %d, tsf: %d",twtSpHighLimit, twtSpLowLimit, twtSpStartTsf);
    ASSERT(done == TRUE); /*polling TIMEOUT*/
    ILOG0_V("[twtManagerConfigureTwtRegsNewSp], twt registers in selector, selector is done");
}

/**********************************************************************************
twtManagerConfigureTwtRegsUpdateSta


Description: configure regs when adding or removing STA from SP
------------
	
Input:
-----
uint8  twtSpStaGroupIdx
uint16 twtSpStaIdx
bool   twtSpStaValue - add or remove STA fom SP profile
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void twtManagerConfigureTwtRegsUpdateSta(uint8 twtSpStaGroupIdx, uint16 twtSpStaIdx, bool twtSpStaValue)
{    
    RegTwtTxTwtSpStaUpdate_u staUpdateReg;
    bool                               done;
    uint8                              bandId; 
    
    done = FALSE;
    bandId = ConfigurationManager_GetBandForStation(twtSpStaIdx);
    staUpdateReg.val = 0;
    staUpdateReg.bitFields.twtSpStaGroupIdx = twtSpStaGroupIdx;
    staUpdateReg.bitFields.twtSpStaIdx = twtSpStaIdx;
    staUpdateReg.bitFields.twtSpStaValue = (uint8)twtSpStaValue;     
    RegAccess_WritePerBand(REG_TWT_TX_TWT_SP_STA_UPDATE,staUpdateReg.val, bandId);
    /*pollling the "done" bit*/
    ILOG0_DD("[twtManagerConfigureTwtRegsUpdateSta], twt registers in selector, staUpdateReg updated, waiting for selector. SID: %d, spId: %d", twtSpStaIdx, twtSpStaGroupIdx);
    RegAccess_WaitForFieldPerBand(REG_TWT_TX_TWT_SP_STA_UPDATE, REG_TWT_SP_SHIFT, REG_TWT_SP_MASK, TRUE, TWT_POLLING_TIMEOUT, &done, bandId);
    ASSERT(done == TRUE); /*polling TIMEOUT*/
    ILOG0_V("[twtManagerConfigureTwtRegsUpdateSta], twt registers in selector, selector is done");
}

/**********************************************************************************
twtManagerTypeUpdateEnable


Description: enable update of twt type for SP
------------
	
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void twtManagerTypeUpdateEnable(void)
{
    RegTwtTxTwtCfg_u twtTypeUpdateEnableReg;
    uint8 bandId;

    twtTypeUpdateEnableReg.val = 0;
    twtTypeUpdateEnableReg.bitFields.twtAvailUpdateEn = 0;  //Due to a bug in HW  
    for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
    {
        RegAccess_WritePerBand(REG_TWT_TX_TWT_CFG, twtTypeUpdateEnableReg.val, bandId);
    }
}

/**********************************************************************************
twtManagerSwHalt

Description: protection for the registers
------------
	
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
*********************************************************************************/
static void twtManagerSwHalt(void)
{
    RegTwtTxTwtSwHalt_u twtSwHaltReg;
    uint8 bandId;

    twtSwHaltReg.val = 0;
    twtSwHaltReg.bitFields.twtSwHalt = 0;    
    for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
	{
        RegAccess_WritePerBand(REG_TWT_TX_TWT_SW_HALT, twtSwHaltReg.val, bandId);
	} 
}
#endif
/**********************************************************************************

twtManagerAddSta 

Description:
------------
recives message about station that was added and checks if it supports HE MU

Input: 
-----
K_MSG *psMsg - pointer to the message to handle	
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerAddSta(K_MSG* psMsg)
{
    BssManagerStaManagerReq_t* staManagerReq;
	UMI_STA_ADD*               pAddSta;
	uint16                     staId;
    
    staManagerReq = (BssManagerStaManagerReq_t*)pK_MSG_DATA(psMsg);
    pAddSta = (UMI_STA_ADD*)pK_MSG_DATA(staManagerReq->psMsg);
    staId = pAddSta->u16SID;

    /*init station entry in DB*/    
    memset(&(twtDb.twtStationDb[staId]), 0, sizeof(TwtManagerStaDb_t));
    twtDb.twtStationDb[staId].isStaConnected = TRUE;
    /*Store HE flag in DB*/
    twtDb.twtStationDb[staId].isStaHe = MTLK_BFIELD_GET(pAddSta->u8FlagsExt, STA_ADD_FLAGS_EXT_IS_HE);
    twtDb.twtStationDb[staId].isStaMfp = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_MFP);
    twtDb.twtStationDb[staId].isXFilterOpen = MTLK_BFIELD_GET(pAddSta->u8Flags, STA_ADD_FLAGS_IS_8021X_FILTER_OPEN);
    ILOG0_D("Twt Manager: ADD STA, SID %d", staId); 
    /*Send cfm to STA manager */
	twtManagerStaManagerSendConfirm(staId);
}

/**********************************************************************************

twtManagerSetFilter 

Description:
------------
called when a set filter request is received

Input: 
-----
pointer to the message 	
		
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerSetFilter(K_MSG *psMsgTwt)
{
	BssManagerStaManagerReq_t* staManagerReq;
	K_MSG*                     pMsg;
	UMI_802_1X_FILTER*         pSetFilter;
	StaId                      staId;

    staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsgTwt);
    pMsg = staManagerReq->psMsg;
    pSetFilter = (UMI_802_1X_FILTER *)pK_MSG_DATA(pMsg);
    staId = pSetFilter->u16SID;
    
	twtDb.twtStationDb[staId].isXFilterOpen = TRUE;
    ILOG2_D("Twt Manager, Set Filter, SID %d", staId);

	twtManagerStaManagerSendConfirm(staId);
}

/**********************************************************************************

twtManagerStaManagerSendConfirm 

Description:
------------
Sends Confirmation message to STA manager

Input: 
-----
uint8  clientId
uint16 sid
		
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerStaManagerSendConfirm(StaId staId)
{
	K_MSG*                     pMsg;
	BssManagerStaManagerCfm_t* confirmMessage;
    uint8                      vapId;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_TWT_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = staId;
	/*Send confirmation message*/
	ILOG0_D("Twt Manager: Send Confirmation to STA manager, SID %d", staId);
    StaDb_GetVapId(staId, &vapId);
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, vapId);
}


/**********************************************************************************

twtManagerHandleRecievedTwtFrame 

Description:
------------
recives rx descriptor and calls to the Frame Parser

Input: 
-----
K_MSG *psMsg - pointer to the message to handle	
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerHandleRecievedTwtFrame(K_MSG *psMsg)
{       
    HwQueueManagerRequestParams_t      hwQueueManagerRequestParams;
    TwtManagerPacketReceivedMessage_t* twtManagerMessage;
    uint16                             staId;
    
	memset(&hwQueueManagerRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));

    twtManagerMessage = (TwtManagerPacketReceivedMessage_t*)pK_MSG_DATA(psMsg);
    staId = twtManagerMessage->stationId;
    if (twtDb.featureState == TWT_MANAGER_GLOBAL_STATE_ENABLE) //Check TWT feature operation mode
    {
        twtManagerFrameParser(twtManagerMessage->framePayload, staId); //Call TWT frame praser  
    }
    /* retrun the RD to the liberator which will put it in the proper free RD pool according to 
	   its RD source field */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_RX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_RX_LIBERATOR_INPUT_LIST;
	hwQueueManagerRequestParams.pHeadDesc = twtManagerMessage->rxDescriptor;
	HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);	    
}


/**********************************************************************************

twtManagerFrameParser

Description:
------------
recives TWT frame and checks if the frame is TWT setup frame or TWT teardown frame

Input: 
-----
TwtFrame_t* twtFrame
uint16      staId
			
Output:
-------
	
Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerFrameParser(TwtFramePayload_t* twtFramePayload , StaId staId)
{
    uint8 flowId;
    uint8 actionCode;
    bool  actionType;
    
    actionCode = twtFramePayload->actionCodeS1G;
    switch (actionCode)
    {
        case TWT_SETUP_ACTION_CODE:
            actionType = TWT_MANAGER_ACTION_TYPE_SETUP; // twt setup
            flowId = twtFramePayload->setup.twtFlowIdentifier;
            ILOG0_D("Twt Manager: twt request recieved, SID %d", staId);
            break;
            
        case TWT_TEARDOWN_ACTION_CODE:
            actionType = TWT_MANAGER_ACTION_TYPE_TEARDOWN; // twt teardown
            flowId = twtFramePayload->teardown.twtFlowIdentifier;
            ILOG0_D("Twt Manager: twt teardown recieved, SID %d", staId);
            break;
            
        default:
            ILOG0_DD("Twt Manager: twt unsupported frame recieved, SID %d, action code %d", staId, actionCode);
            return;
    }
    /*forward the information to the handler -  for setup&teardown frames only*/ 
    twtManagerHandler(actionType, staId, twtFramePayload, flowId);   
} 


/**********************************************************************************

twtManagerHandler

Description:
------------
in charge of the frame handling and negotiation

Input: 
-----
bool               actionType
uint16             staId
TwtFramePayload_t  twtFramePayload
uint8              flowId
		
Output:
-------
	
Returns:
--------
	void - 	
**********************************************************************************/
static void twtManagerHandler(TwtManagerActionType_e actionType, StaId staId, TwtFramePayload_t*  twtFramePayload, uint8 flowId)
{
    SpAvailAns_t spAnswer;
    uint8        spId;
        
    if (actionType == TWT_MANAGER_ACTION_TYPE_TEARDOWN) // teardown 
    {                    
        if (twtDb.twtStationDb[staId].flowIdDb[flowId].state == TWT_MANAGER_FLOW_STATE_ACTIVE)
        {
            spId = twtDb.twtStationDb[staId].flowIdDb[flowId].spId;            
            twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
            twtManagerCancelAgreement(staId,spId,flowId);
        }
    }                    
    else // setup
    {     
        if ((twtDb.twtStationDb[staId].isStaConnected == 1) &&
            (twtFramePayload->setup.twtRequest == 1) &&
       	    (twtDb.twtStationDb[staId].sendingAccept == 0) &&
       	    (twtDb.twtStationDb[staId].isStaHe == 1))  //TBD - consider OMI bit change from the station 
       	    /*Station connected, the frame is a request, we are not currently sending accept for the station and the station supports HE*/   
        {      	        
	        /*call SP availability*/
	        twtManagerSpAvailabilityManager(staId, twtFramePayload, &spAnswer);
            flowId = spAnswer.resFramePayload.setup.twtFlowIdentifier;
            twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_PROCESSING_TWT_REQ);
            /*When SP availability module is not reside in TWT, we should check that station wasn't removed*/
	        /*if accepted update the DB with SP ID*/
	        if (spAnswer.resFramePayload.setup.twtSetupCommand == TWT_ACCEPT)
	        {
                twtDb.twtStationDb[staId].flowIdDb[flowId].spId = spAnswer.spId;
                twtDb.twtStationDb[staId].sendingAccept = 1;
#ifdef PF_TWT
                if(spAvailabilityDb.twtServicePeriodDb[spAnswer.spId].isSelectorUpdated == FALSE)
                {           
                    twtManagerSelectorConfigurationOfNewSp(spAnswer.spId, staId);
                    spAvailabilityDb.twtServicePeriodDb[spAnswer.spId].isSelectorUpdated = TRUE;
                }
                StaDb_SetPsduTimeLimit(staId ,MAX_PSDU_FOR_TWT); 
#endif
	        }
            /*call frame builder with the SP availability's answer*/ 
            twtDb.twtStationDb[staId].flowIdDb[flowId].twtFramePayload = spAnswer.resFramePayload;
	        twtManagerFrameBuilder(staId, flowId);            
       	 }	   
    }           
}


/**********************************************************************************
twtManagerBuildAgreement

    
    
Description:
------------
builds aggreement by calling functions to configurate the TX Selector registers and bitmap, update DB, send message to HE GM and update statistics
    
Input: 
-----
 StaId staId
 uint8  spId
 uint8  flowId     
                 
Output:
-------      
    
Returns:
--------
    void - 
        
**********************************************************************************/
static void twtManagerBuildAgreement(StaId staId, uint8 spId, uint8 flowId)
{
#ifdef ENET_INC_ARCH_WAVE600B
    twtManagerSetHeMuKeepUser(FALSE);
#endif
	/*update DB*/
	twtManagerUpdateDataBaseAddAgreement(staId, spId, flowId);
    /*update selector's registers and bitmap*/
	twtManagerSelectorConfiguration(staId, spId, flowId, TWT_MANAGER_ACTION_TYPE_SETUP);
	//send message to HE Group Manager - TBD according to HE GM
	ILOG0_DDD("Twt manager: request accepted, TWT agreement created for SID: %d, flow ID: %d, sp ID: %d", staId, flowId, spId);
	//update the statistic data base.
	twtManagerUpdateStatsDataBase(staId, spId, flowId);
	
    StaDB_SetNoAggInPs(staId, FALSE);
}


/**********************************************************************************
twtManagerCancelAgreement
        
Description:
------------
cancels aggreement by calling functions to configurate the TX Selector registers and bitmap, update DB, send message to HE GM and update statistics
    
Input: 
-----
 StaId staId
 uint8  spId
 uint8  flowId     
                 
Output:
-------      
    
Returns:
--------
    void - 
        
**********************************************************************************/
static void twtManagerCancelAgreement(StaId staId, uint8 spId, uint8 flowId)
{
    twtManagerSelectorConfiguration(staId, spId, flowId, TWT_MANAGER_ACTION_TYPE_TEARDOWN); //configure selector
    //send message to group manager - TBD according to HE GM
    twtManagerUpdateDataBaseDelAgreement(staId, spId, flowId); // update DB
    ILOG0_DD("Twt manager: agreement has been teared down, SID: %d flow ID: %d", staId, flowId);

    if (twtDb.twtStationDb[staId].numOfAgreementsExist == TWT_NO_AGREEMENT)
    {
        StaDB_SetNoAggInPs(staId, TRUE);
    }
#ifdef ENET_INC_ARCH_WAVE600B
    twtManagerSetHeMuKeepUser(TRUE);
#endif // #ifdef ENET_INC_ARCH_WAVE600B
}


/**********************************************************************************
twtManagerSetHeMuKeepUser
        
Description:
------------
Enables/disables HE MU keep user
    
Input: 
-----
bool setKeepUser  
                 
Output:
-------      
    
Returns:
--------
    void - 
        
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600B
static void twtManagerSetHeMuKeepUser(bool setKeepUser)
{
	if (twtDb.totalAgreements == 0)
	{
		PreAggregator_SetHeMuKeepUser(setKeepUser);
	}
}
#endif
/**********************************************************************************	

twtManagerSpAvailabilityManager


Description:
------------
decides how to respond to the TWT request

Input: 
-----
uint16             staId
TwtFramePayload_t  twtFramePayload
SpAvailAns_t*      spAvailAns
		
Output:
-------
	

Returns:
--------
    void
**********************************************************************************/
static void twtManagerSpAvailabilityManager(StaId staId, TwtFramePayload_t*  recievedFramePayload, SpAvailAns_t* spAvailAns) 
{
    TwtFramePayload_t  twtFramePayload;
    TSF                targetWakeTime;    
    uint8              flowId = 0;
    uint8              spId;
    uint8              vapId;
#ifdef PF_TWT 
    uint64             wakeInterval;
    TSF                currentTimeStamp;
    uint32             spStartTsf;    
    uint32             nominalTwtWakeDuration;
#endif

    if (twtDb.twtStationDb[staId].numOfAgreementsExist == TWT_MAX_AGREEMENTS_ALLOWED) 
    {               
        flowId = twtDb.twtStationDb[staId].agreementToAnnul; //check DB for spId and flowId of the first agreement in order to annul it 
        spId = twtDb.twtStationDb[staId].flowIdDb[flowId].spId;
        twtManagerSendTeardownFrame(staId, spId, flowId); //send teardown frame about old agreement
    }    

    /* select flow ID */
    for (flowId = 1; flowId < NUM_OF_TWT_FLOWS; flowId++)
    {
        if (twtDb.twtStationDb[staId].flowIdDb[flowId].state == TWT_MANAGER_FLOW_STATE_INACTIVE)
        {
            break;
        }    
    }
    if (flowId == NUM_OF_TWT_FLOWS) // should not happen 
    {
        spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_REJECT;
        flowId = 0;
    }

    twtFramePayload = *recievedFramePayload;
    StaDb_GetVapId(staId, &vapId);
    SLOG0(0, 0, TwtFramePayload_t, &(twtFramePayload));
#ifndef PF_TWT
    spAvailAns->spId = vapId; // for the first mode
    
    /*set twt response frame:*/
    spAvailAns->resFramePayload.categoryCode = HE_TWT_CATEGORY;
    spAvailAns->resFramePayload.actionCodeS1G = TWT_SETUP_ACTION_CODE;
    spAvailAns->resFramePayload.setup.dialogToken = twtFramePayload.setup.dialogToken;
    spAvailAns->resFramePayload.setup.elementId = TWT_ELEMENT_ID;      
    spAvailAns->resFramePayload.setup.length = TWT_PAYLOAD_LENGTH; 
    spAvailAns->resFramePayload.setup.ndpPagingIndicator = TWT_OFF;
    spAvailAns->resFramePayload.setup.responderPmMode = TWT_OFF;
    spAvailAns->resFramePayload.setup.negotiationType = INDIVIDUAL_TWT;
    spAvailAns->resFramePayload.setup.reserved = 0;
    spAvailAns->resFramePayload.setup.twtRequest = TWT_OFF;
    spAvailAns->resFramePayload.setup.trigger = TWT_ON;
    spAvailAns->resFramePayload.setup.implicit = TWT_ON;   
    spAvailAns->resFramePayload.setup.flowType = TWT_MANAGER_FLOW_TYPE_UNANNOUNCED; //TBD - or set according to SP.
    spAvailAns->resFramePayload.setup.twtFlowIdentifier = flowId;
    spAvailAns->resFramePayload.setup.wakeIntervalExponent = TWT_WAKE_INTERVAL_EXP; // WAKE_INTEVAL = WAKE_INTERVAL_MANTISSA * (2 ^ WAKE_INTERVAL_EXP);
    spAvailAns->resFramePayload.setup.twtProtection = TWT_OFF;
    targetWakeTime = spAvailabilityDb.targetWakeTimeOfSpOne + ((spAvailAns->spId) * (TWT_SP_HIGH << TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR)); 
    spAvailAns->resFramePayload.setup.targetWakeTime = targetWakeTime;
    spAvailAns->resFramePayload.setup.nominalTwtWakeDuration = TWT_WAKE_DURATION;
    spAvailAns->resFramePayload.setup.twtWakeIntervalMantissa = TWT_WAKE_INTERVAL_MANTISSA;
    spAvailAns->resFramePayload.setup.twtChannel = TWT_OFF;

    
    switch(twtFramePayload.setup.twtSetupCommand)
    {
    case TWT_REQUEST:             
        if ((spAvailAns->resFramePayload.setup.length == twtFramePayload.setup.length) &&
        (spAvailAns->resFramePayload.setup.ndpPagingIndicator == twtFramePayload.setup.ndpPagingIndicator) &&
        (spAvailAns->resFramePayload.setup.responderPmMode == twtFramePayload.setup.responderPmMode) &&
        (spAvailAns->resFramePayload.setup.negotiationType == twtFramePayload.setup.negotiationType) &&
        (spAvailAns->resFramePayload.setup.implicit == twtFramePayload.setup.implicit) &&
        (spAvailAns->resFramePayload.setup.flowType == twtFramePayload.setup.flowType) &&
        (spAvailAns->resFramePayload.setup.twtProtection == twtFramePayload.setup.twtProtection)) //identical except TWT parameters- TWT, wake duration, wake interval (mantissa and exp), trigger, channel
        //length, ndpPagingIndicator, responderPmMode, broadcast, wakeTbttNegotiation, implicit, flowType, twtProtection 
        {
            spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_ACCEPT; 
        }
        else
        {
            spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_DICTATE;  
        }        
        break;
    
    case TWT_SUGGEST:
    case TWT_DEMAND: 
        if ((spAvailAns->resFramePayload.setup.length == twtFramePayload.setup.length) &&
        (spAvailAns->resFramePayload.setup.ndpPagingIndicator == twtFramePayload.setup.ndpPagingIndicator) &&
        (spAvailAns->resFramePayload.setup.responderPmMode == twtFramePayload.setup.responderPmMode) &&
        (spAvailAns->resFramePayload.setup.negotiationType == twtFramePayload.setup.negotiationType) &&
        (spAvailAns->resFramePayload.setup.trigger == twtFramePayload.setup.trigger) &&
        (spAvailAns->resFramePayload.setup.implicit == twtFramePayload.setup.implicit) &&
        (spAvailAns->resFramePayload.setup.flowType == twtFramePayload.setup.flowType) &&
        (spAvailAns->resFramePayload.setup.wakeIntervalExponent == twtFramePayload.setup.wakeIntervalExponent) &&
        (spAvailAns->resFramePayload.setup.twtProtection == twtFramePayload.setup.twtProtection) &&
        (spAvailAns->resFramePayload.setup.targetWakeTime == twtFramePayload.setup.targetWakeTime) &&
        (spAvailAns->resFramePayload.setup.nominalTwtWakeDuration == twtFramePayload.setup.nominalTwtWakeDuration) &&
        (spAvailAns->resFramePayload.setup.twtWakeIntervalMantissa == twtFramePayload.setup.twtWakeIntervalMantissa) &&        
        (spAvailAns->resFramePayload.setup.twtChannel == twtFramePayload.setup.twtChannel)) //all identical
        //length, ndpPagingIndicator, responderPmMode, broadcast, wakeTbttNegotiation, implicit, flowType, twtProtection, trigger, wakeIntervalExponent, nominalTwtWakeDuration, twtWakeIntervalMantissa, targetWakeTime, twtChannel 
        {
            spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_ACCEPT; 
        }
        else
        {
            spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_DICTATE;
        }        
        break;
        
    default:
        spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_DICTATE;
        break;
    }

#else // PF_TWT

    /* select SP ID */
    for (spId = 0; spId < MAX_NUM_OF_SP; spId++)
    {
        if (spAvailabilityDb.twtServicePeriodDb[spId].numOfStaInServicePeriod == 0)
        {
            spAvailAns->spId = spId;
            break;
        }
    }
    if (spId == MAX_NUM_OF_SP) // No available SP. This shouldn't happen - only 4 SP in PF6.
    {
        spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_REJECT;
        spId--; //so we won't exceed the array length. This shouldn't happen anyway.
    }
    /* fill frame payload according to requirements */
    spAvailAns->resFramePayload.categoryCode = HE_TWT_CATEGORY;
    spAvailAns->resFramePayload.actionCodeS1G = TWT_SETUP_ACTION_CODE;
    spAvailAns->resFramePayload.setup.dialogToken = twtFramePayload.setup.dialogToken;
    spAvailAns->resFramePayload.setup.elementId = TWT_ELEMENT_ID;      
    spAvailAns->resFramePayload.setup.length = TWT_PAYLOAD_LENGTH; 
    spAvailAns->resFramePayload.setup.ndpPagingIndicator = TWT_OFF;
    spAvailAns->resFramePayload.setup.responderPmMode = TWT_OFF;
    spAvailAns->resFramePayload.setup.negotiationType = INDIVIDUAL_TWT;
    spAvailAns->resFramePayload.setup.reserved = 0;
    spAvailAns->resFramePayload.setup.twtRequest = TWT_OFF;
    spAvailAns->resFramePayload.setup.trigger = TWT_ON;
    spAvailAns->resFramePayload.setup.implicit = TWT_ON; 
    spAvailAns->resFramePayload.setup.flowType = twtFramePayload.setup.flowType;
    spAvailAns->resFramePayload.setup.twtFlowIdentifier = flowId;
    spAvailAns->resFramePayload.setup.wakeIntervalExponent = twtFramePayload.setup.wakeIntervalExponent;
    spAvailAns->resFramePayload.setup.twtProtection = TWT_OFF;    
    spAvailAns->resFramePayload.setup.nominalTwtWakeDuration = twtFramePayload.setup.nominalTwtWakeDuration;
    spAvailAns->resFramePayload.setup.twtWakeIntervalMantissa = twtFramePayload.setup.twtWakeIntervalMantissa;
    spAvailAns->resFramePayload.setup.twtChannel = TWT_OFF;
    wakeInterval = spAvailAns->resFramePayload.setup.twtWakeIntervalMantissa << spAvailAns->resFramePayload.setup.wakeIntervalExponent;
    nominalTwtWakeDuration =  spAvailAns->resFramePayload.setup.nominalTwtWakeDuration << TWT_CHANGE_UNITS_SHIFT_WAKE_DURATION; // value is in units of 256us   
    if (twtFramePayload.setup.twtSetupCommand == TWT_DEMAND) // if demand take TWT from recievedFramePayload, but in SP DB add interval
    { 
        spAvailAns->resFramePayload.setup.targetWakeTime = twtFramePayload.setup.targetWakeTime; 
        spStartTsf = spAvailAns->resFramePayload.setup.targetWakeTime - (wakeInterval - nominalTwtWakeDuration);
        spAvailabilityDb.twtServicePeriodDb[spId].twtSpStartTsf = spStartTsf + wakeInterval - CHANGE_SP_LENGTH_FOR_PF;
    }
    else // if not demand set TWT according to TSF + const , note that TWT % (16 TU) should be 0 (because of WFA NAN requirement)
    {
        currentTimeStamp = Pac_TimGetTsf();
        targetWakeTime = currentTimeStamp + TWT_CONST_INTERVAL;
        if (targetWakeTime % (16 * TU) != 0)
        { 
            targetWakeTime = targetWakeTime + (16 * TU) - (targetWakeTime % (16 * TU));
        }
        spAvailAns->resFramePayload.setup.targetWakeTime = targetWakeTime;        
        spStartTsf = targetWakeTime - (wakeInterval - nominalTwtWakeDuration);
        spAvailabilityDb.twtServicePeriodDb[spId].twtSpStartTsf = spStartTsf - CHANGE_SP_LENGTH_FOR_PF; 
    }
    
    /* write params to SP DB */
    spAvailabilityDb.twtServicePeriodDb[spId].twtType = (TwtManagerFlowType_e)spAvailAns->resFramePayload.setup.flowType;        
    spAvailabilityDb.twtServicePeriodDb[spId].twtSpHighLimit = (nominalTwtWakeDuration - CHANGE_SP_LENGTH_FOR_PF) >> TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR; // twtSpHighLimit is in res of 32us
    spAvailabilityDb.twtServicePeriodDb[spId].twtSpLowLimit = (wakeInterval - nominalTwtWakeDuration + CHANGE_SP_LENGTH_FOR_PF) >> TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR; // twtSpLowLimit is in res of 32us
    
    spAvailabilityDb.twtServicePeriodDb[spId].numOfStaInServicePeriod++;
    spAvailAns->resFramePayload.setup.twtSetupCommand = TWT_ACCEPT; 
    SLOG0(0, 0, TwtFramePayload_t, &(spAvailAns->resFramePayload));
#endif
}


/**********************************************************************************
    
twtManagerSelectorConfiguration
    
    
Description:
------------
configurate the TX Selector registers and bitmap
    
Input: 
-----
 StaId staId
 uint8  spId
 uint8  flowId
 bool   actionType
 bool   flowType      
                 
Output:
-------      
    
Returns:
--------
    void - 
        
**********************************************************************************/
static void twtManagerSelectorConfiguration(StaId staId, uint8 spId, uint8 flowId, TwtManagerActionType_e actionType)
{
	UNUSED_PARAM(flowId);
    if (actionType == TWT_MANAGER_ACTION_TYPE_TEARDOWN) // Delete STA
    {
        if (twtDb.twtStationDb[staId].numOfAgreementsExist == TWT_ONLY_AGREEMENT)
        {
            TxSelector_UpdateBitmapRemTwtSta(staId); //update selector bitmap - avail bit
        }  
#ifdef ENET_INC_ARCH_WAVE600D2		
        twtManagerConfigureTwtRegsUpdateSta(spId, staId, FALSE);  //selector regs configuration  
#else
		TxSelector_ConfigureTwtRegsUpdateSta(spId, staId, FALSE);  //selector regs configuration  
#endif	
        spAvailabilityDb.twtServicePeriodDb[spId].numOfStaInServicePeriod--; //update SP DB
        if (spAvailabilityDb.twtServicePeriodDb[spId].numOfStaInServicePeriod == 0)
        {
            memset(&(spAvailabilityDb.twtServicePeriodDb[spId]), 0, sizeof(ServicePeriod_t));
        }
        ILOG0_V("twtManagerSelectorConfiguration: delete");
    }
    else // Add STA
    {   
#ifdef ENET_INC_ARCH_WAVE600D2   
        twtManagerConfigureTwtRegsUpdateSta(spId, staId, TRUE);  //selector regs configuration       
#else
		TxSelector_ConfigureTwtRegsUpdateSta(spId, staId, TRUE);  //selector regs configuration       
#endif
        TxSelector_UpdateBitmapAddTwtSta(staId, spAvailabilityDb.twtServicePeriodDb[spId].twtType); //update bitmap - avail bit & type bit  * 0-announced, 1-unannounced              
        ILOG0_V("twtManagerSelectorConfiguration: add");
    }    
}


/**********************************************************************************
    
twtManagerSelectorConfigurationOfNewSp
       
Description:
------------
configuration of the TX Selector registers when new service period is created
    
Input: 
-----
 uint8  spId
            
Output:
-------      
    
Returns:
--------
    void - 
        
**********************************************************************************/

static void twtManagerSelectorConfigurationOfNewSp(uint8 spId, StaId staId)
{
#ifdef PF_TWT
    uint32                  twtSpLowLimit;
    uint32                  twtSpStartTsf;
    uint32                  twtSpHighLimit;
    TwtManagerFlowType_e    twtType;
    uint8                   bandId;
#if defined (TWT_SELECTOR_WORKAROUND) 
    uint32                  counter;
    uint32                  counterRowAddress;
    TwtCountRamRow_t*       counterRow;
    TX_INTERRUPT_SAVE_AREA;

    OSAL_DISABLE_INTERRUPTS(&interrupt_save); /* according to JIRA WLANVLSIIP-2935. bug - when init TSF is smaller then actual TSF wrong value is set to regs. 
    FW workaround - add const value and disable interrupts*/  
#endif  //TWT_SELECTOR_WORKAROUND

     /* configure selector with SP params from DB */
    twtSpLowLimit = spAvailabilityDb.twtServicePeriodDb[spId].twtSpLowLimit;
    twtSpStartTsf = spAvailabilityDb.twtServicePeriodDb[spId].twtSpStartTsf;
    twtSpHighLimit = spAvailabilityDb.twtServicePeriodDb[spId].twtSpHighLimit;
    twtType = spAvailabilityDb.twtServicePeriodDb[spId].twtType;

    bandId = ConfigurationManager_GetBandForStation(staId);
#ifdef ENET_INC_ARCH_WAVE600D2
	twtManagerConfigureTwtRegsNewSp(twtSpLowLimit, twtSpStartTsf, twtSpHighLimit, spId, bandId); 	
#else	
    TxSelector_ConfigureTwtRegsNewSp(twtSpLowLimit, twtSpStartTsf, twtSpHighLimit, spId, twtType, bandId); 
#endif //ENET_INC_ARCH_WAVE600D2

#if defined (TWT_SELECTOR_WORKAROUND)        
    /* according to JIRA WLANVLSIIP-2942. bug - sometimes zero value will be written to the group counter. 
    FW workaround - check if counter value is 0 (Or 0xFFFFF). If so deactivate the SP and set it up again*/
    counter = 0;
    bandId = ConfigurationManager_GetBandForStation(staId);
    counterRowAddress = TWT_COUNTER_FIXED_OFFSET_IN_SELECTOR_RAM + (TWT_RAM_LINE * spId) + (BAND1_OFFSET_FROM_BAND0 * bandId);
    counterRow = (TwtCountRamRow_t*)counterRowAddress;
    while ((counter == TWT_ZERO_VAL) || (counter == TWT_BEFORE_ZERO))
    {
        counter = counterRow->counter;
        TxSelector_ConfigureTwtRegsDeactivateSp(spId, bandId);
        TxSelector_ConfigureTwtRegsNewSp(twtSpLowLimit, twtSpStartTsf, twtSpHighLimit, spId, twtType, bandId);
    } 
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
#endif  //TWT_SELECTOR_WORKAROUND
#endif //PF_TWT
}


/**********************************************************************************
    
twtManagerSendTeardownFrame
    
    
Description:
------------
handle sending teardown frame 
    
Input: 
-----
StaId staId
uint8  spId
uint8  flowId                
            
Output:
-------       
    
Returns:
--------  
 void       
**********************************************************************************/
static void twtManagerSendTeardownFrame(StaId staId, uint8 spId, uint8 flowId)
{
    TwtFramePayload_t teardownFramePayload;

	UNUSED_PARAM(spId);
    teardownFramePayload.categoryCode = HE_TWT_CATEGORY;
    teardownFramePayload.actionCodeS1G = TWT_TEARDOWN_ACTION_CODE;
    teardownFramePayload.teardown.twtFlowIdentifier = flowId;
    teardownFramePayload.teardown.reserved = 0;
    twtDb.twtStationDb[staId].flowIdDb[flowId].twtFramePayload = teardownFramePayload;
    twtManagerFrameBuilder(staId, flowId); // call frame builder        
}

/**********************************************************************************
    
twtManagerUpdateStatsDataBase

    
    
Description:
------------
update statistic data base.
    
Input: 
-----
StaId staId
uint8  spId
uint8  flowId             
            
Output:
-------       
    
Returns:
-------- 
 void       
**********************************************************************************/
static void twtManagerUpdateStatsDataBase(StaId staId, uint8 spId, uint8 flowId)
{
	TwtSetupFramePayload_t setupFrame;
	twt_params_t *twtStaParam;
	
	setupFrame = twtDb.twtStationDb[staId].flowIdDb[flowId].twtFramePayload.setup;
	twtStaParam = &(TwtStatistics.twtStaParams[staId]);
	
	TwtStatistics.numOfStaInSp[spId]++;
	twtStaParam->twtAgreement[flowId].agreementType = setupFrame.negotiationType;
	twtStaParam->twtAgreement[flowId].operation.announced = setupFrame.flowType;
	twtStaParam->twtAgreement[flowId].operation.implicit= setupFrame.implicit;
	twtStaParam->twtAgreement[flowId].operation.triggerEnabled = setupFrame.trigger;
	twtStaParam->twtAgreement[flowId].params.individual.wakeTimeHigh = (uint32)(setupFrame.targetWakeTime >> 32);
	twtStaParam->twtAgreement[flowId].params.individual.wakeTimeLow = (uint32)(setupFrame.targetWakeTime & 0xFFFFFFFF);
	twtStaParam->twtAgreement[flowId].params.individual.wakeInterval = setupFrame.twtWakeIntervalMantissa << setupFrame.wakeIntervalExponent;
	twtStaParam->twtAgreement[flowId].params.individual.minWakeDuration = setupFrame.nominalTwtWakeDuration << TWT_CHANGE_UNITS_SHIFT_WAKE_DURATION;  // value is in units of 256us   
	twtStaParam->twtAgreement[flowId].params.individual.channel = setupFrame.twtChannel;
	twtStaParam->twtAgreement[flowId].state = twtDb.twtStationDb[staId].flowIdDb[flowId].state;
	twtStaParam->numOfAgreementsForSta++;
}

/**********************************************************************************
    
twtManagerUpdateDataBaseAddAgreement
    
    
Description:
------------
update DB with the new agreement parameters
    
Input: 
-----
StaId staId
uint8  spId
uint8  flowId             
            
Output:
-------       
    
Returns:
-------- 
 void       
**********************************************************************************/
static void twtManagerUpdateDataBaseAddAgreement(StaId staId, uint8 spId, uint8 flowId)
{
	UNUSED_PARAM(spId);
    if (twtDb.twtStationDb[staId].numOfAgreementsExist == TWT_NO_AGREEMENT)
    {
        twtDb.twtStationDb[staId].agreementToAnnul = flowId;
    }
    twtDb.twtStationDb[staId].numOfAgreementsExist++;
	twtDb.totalAgreements++;
    twtDb.twtStationDb[staId].sendingAccept = 0; // only for current phase
}


/**********************************************************************************
    
twtManagerpdateDataBaseDelAgreement
    
    
Description:
------------
delete agreement from DB
    
Input: 
-----
StaId staId
uint8  spId
uint8  flowId               
           
Output:
-------        
    
Returns:
--------
    void - 
        
**********************************************************************************/
static void twtManagerUpdateDataBaseDelAgreement(StaId staId, uint8 spId, uint8 flowId)
{

    twtDb.twtStationDb[staId].numOfAgreementsExist--; 
	twtDb.totalAgreements--;

	TwtStatistics.numOfStaInSp[spId]--;
	TwtStatistics.twtStaParams[staId].twtAgreement[flowId].state = twtDb.twtStationDb[staId].flowIdDb[flowId].state;
	TwtStatistics.twtStaParams[staId].numOfAgreementsForSta--;
}


/**********************************************************************************

twtManagerFrameBuilder


Description:
------------
builds a frame according to the received parameters, allocates PD 

Input: 
-----
uint16             staId
TwtFramePayload_t  twtFramePayload
uint8              flowId
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
static void twtManagerFrameBuilder(StaId staId, uint8 flowId)
{
    RmPdRequestFillParameters_t pdRequestFillParameters;
    bool                        allocationStatus; 

    allocationStatus = FALSE;
    
    
    memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));   
    pdRequestFillParameters.returnTaskId = TASK_TWT_MANAGER;
    pdRequestFillParameters.returnMsgType = TWT_MANAGER_PD_ALLOC_CFM;
    pdRequestFillParameters.context = TWT_ASSIGN_STA_AND_FLOW_IDS_IN_CONTEXT_FIELD(staId ,flowId); 
    
    allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MANAGEMENT_FROM_FW, &pdRequestFillParameters); 
    if(allocationStatus)
    {     
        /*store PD*/
        twtDb.twtStationDb[staId].flowIdDb[flowId].pd = pdRequestFillParameters.packetDescriptor;
        /*Build and send TWT FRAME*/
        twtManagerSendTwt(pdRequestFillParameters.packetDescriptor, staId, flowId);
    }
    else
    {
        /* PD allocation falied */    
        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_WAIT_FOR_RESOURCE_ALLOCATION);
        /* Save the request ID */
		twtDb.twtStationDb[staId].flowIdDb[flowId].pdRequestId = pdRequestFillParameters.requestId;
    }
}


/**********************************************************************************
twtManagerResourceCfmReceived 
  
Description:
------------
get parameters and PD and call send TWT function

Input: 
-----
K_MSG *psMsg
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

static void twtManagerResourceCfmReceived(K_MSG *psMsg)
{
    RmPdFreeDescResponse_t*     pdAllocMsg;
    uint16                      staId;
    uint8                       flowId;
              
    pdAllocMsg = (RmPdFreeDescResponse_t *)pK_MSG_DATA(psMsg);
    staId = TWT_EXTRACT_STA_ID_FROM_CONTEXT(pdAllocMsg->context); 
    flowId = TWT_EXTRACT_FLOW_ID_FROM_CONTEXT(pdAllocMsg->context);
    ResourceManager_ReleaseRequest(twtDb.twtStationDb[staId].flowIdDb[flowId].pdRequestId, DESC_POOL_MANAGEMENT_FROM_FW); /*release request for pd*/
    ILOG0_DDD("Twt manager: PD cfm at state: %d, SID: %d flow ID: %d", twtDb.twtStationDb[staId].flowIdDb[flowId].state, staId, flowId);

    switch(twtDb.twtStationDb[staId].flowIdDb[flowId].state)
    {
        case TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_PD: /*check if station has been removed*/
            ResourceManager_ReleaseDescriptor(pdAllocMsg->packetDescriptor, DESC_POOL_MANAGEMENT_FROM_FW); /*release PD*/
            twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
            break;
            
        case TWT_MANAGER_FLOW_STATE_WAIT_FOR_RESOURCE_ALLOCATION:
            /*store PD*/
            twtDb.twtStationDb[staId].flowIdDb[flowId].pd = pdAllocMsg->packetDescriptor;
            /*Build and send TWT FRAME*/
            twtManagerSendTwt(pdAllocMsg->packetDescriptor, staId, flowId);
            break;
            
        default:
            ResourceManager_ReleaseDescriptor(pdAllocMsg->packetDescriptor, DESC_POOL_MANAGEMENT_FROM_FW); /*release PD*/           
            DEBUG_ASSERT(0);
            break;
    }    
}


/**********************************************************************************

twtManagerSendTwt
  
Description:
------------
fill PD parameters and put PD on mgmt queue

Input: 
-----
TxPd_t* pdPointer
uint16  staId
uint8   vapId
uint8   flowId
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

static void twtManagerSendTwt(TxPd_t* pdPointer, StaId staId, uint8 flowId)
{  
    TwtFramePayload_t* twtFramePayload;
    uint8              vapId;
    //uint8              mfp;
	//uint8              isFilterOpen;

    //mfp = twtDb.twtStationDb[staId].isStaMfp;
    //isFilterOpen = twtDb.twtStationDb[staId].isXFilterOpen;
    StaDb_GetVapId(staId, &vapId);
    twtFramePayload = (TwtFramePayload_t *)frame_getPayloadPointerForNewManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pdPointer->packetPointer, FALSE);
    *twtFramePayload = twtDb.twtStationDb[staId].flowIdDb[flowId].twtFramePayload; 
    if (twtFramePayload->actionCodeS1G == TWT_SETUP_ACTION_CODE)
    {
        pdPointer->dataLength = frame_sizeOfNewManagementFrameHeader(FALSE) + COMMON_TWT_FRAME + sizeof(TwtSetupFramePayload_t);
    }
    else
    {
        pdPointer->dataLength = frame_sizeOfNewManagementFrameHeader(FALSE) + COMMON_TWT_FRAME + sizeof(TwtTeardownFramePayload_t);
    }    
    pdPointer->txQTid = MANAGEMENT_TID;
    pdPointer->txQStaId = staId;
    pdPointer->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
    pdPointer->mgmtActionCode = HE_TWT_CATEGORY;
    pdPointer->mgmtActionValue = twtFramePayload->actionCodeS1G;
	/*if ((mfp == TRUE) && (isFilterOpen == TRUE))
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_ENC;
	}
	else
	{
		pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
	}*/
	pdPointer->pdType = PD_TYPE_MANAGEMENT_UNENC;
   	pdPointer->txQVapId = vapId;
    pdPointer->txQGroupId = HW_TX_Q_TYPE_STA_TID;
    pdPointer->mcUnicast = UNICAST;
    twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_WAIT_FOR_TX_CFM);
    /*put PD on queue*/
    ILOG0_DD("Twt manager: TWT transmission, pushing PD to mgmt queue, SID: %d, flow ID: %d", staId, flowId);
    TxPacketsClassifier_SendManagementPacketFromFw(pdPointer);   
}


/**********************************************************************************

twtManagerTxCfmReceived

Description:
------------  
when tx cfm recieved check status and according to it send again or not

Input: 
-----
K_MSG *psMsg
			
Output:
-------	

Returns:
--------
	void - 
**********************************************************************************/	

static void twtManagerTxCfmReceived(K_MSG *psMsg)
{
    TwtManagerPacketConfirmedMessage_t* cfmMsg;
    TxPd_t*                             pd;
    TwtFramePayload_t*                  twtFramePayload;
    uint16                              staId;
    uint8                               flowId;

    cfmMsg = (TwtManagerPacketConfirmedMessage_t*)pK_MSG_DATA(psMsg);
    pd = cfmMsg->pd;
    twtFramePayload = (TwtFramePayload_t*)frame_getPayloadPointerFromExistingManagementFrame((MANAGEMENT_BASIC_FRAME_HEADER *)pd->packetPointer);
    staId = pd->txQStaId;
    if (twtFramePayload->actionCodeS1G == TWT_SETUP_ACTION_CODE)
    {
        flowId = twtFramePayload->setup.twtFlowIdentifier;
    }
    else
    {
        flowId = twtFramePayload->teardown.twtFlowIdentifier;
    }
    ILOG0_DDD("Twt manager: tx cfm at state: %d, SID: %d flow ID: %d", twtDb.twtStationDb[staId].flowIdDb[flowId].state, staId, flowId);
    
    switch (twtDb.twtStationDb[staId].flowIdDb[flowId].state)
    {
        case TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_TX_CFM: /*check if station has been removed*/
            twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
            ResourceManager_ReleaseDescriptor(twtDb.twtStationDb[staId].flowIdDb[flowId].pd, DESC_POOL_MANAGEMENT_FROM_FW); /*release PD*/
            break;

        case TWT_MANAGER_FLOW_STATE_WAIT_FOR_TX_CFM:
            if ((pd->status == PD_STATUS_ACK_RECEIVED ) || (pd->status == PD_STATUS_BLOCK_ACK_RECEIVED))  /*check status - if ack recieved*/
            {
                ILOG0_DD("Twt manager: Tx confirmation, ACK recieved on TWT transmission, SID: %d flow ID: %d", staId, flowId);
                if (twtFramePayload->actionCodeS1G == TWT_SETUP_ACTION_CODE)
                {
                    if (twtFramePayload->setup.twtSetupCommand == TWT_ACCEPT)
                    {
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_ACTIVE); // accept
                        twtManagerBuildAgreement(staId, twtDb.twtStationDb[staId].flowIdDb[flowId].spId, flowId);
                    }
                    else
                    {
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE); // response is not ACCEPT
                    }
                }
                else
                {
                    twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE); // teardown
                    twtManagerCancelAgreement(staId, twtDb.twtStationDb[staId].flowIdDb[flowId].spId, flowId);
                }
                ResourceManager_ReleaseDescriptor(twtDb.twtStationDb[staId].flowIdDb[flowId].pd, DESC_POOL_MANAGEMENT_FROM_FW); /*release PD*/
            }
            else
            {
                /*send again*/
                ILOG0_DD("Twt manager:	Tx confirmation, NACK recieved on TWT transmission, sending again, SID: %d flow ID: %d", staId, flowId);
                twtManagerSendTwt(pd, staId, flowId);
            }
            break;
            
        case TWT_MANAGER_FLOW_STATE_INACTIVE:
            if (twtFramePayload->actionCodeS1G != TWT_TEARDOWN_ACTION_CODE) 
            {   /* if TWT_TEARDOWN_ACTION_CODE and TWT_MANAGER_FLOW_STATE_INACTIVE, station sent teardown too, don't assert */
                DEBUG_ASSERT(0);
            }
            ResourceManager_ReleaseDescriptor(twtDb.twtStationDb[staId].flowIdDb[flowId].pd, DESC_POOL_MANAGEMENT_FROM_FW); /*release PD*/
            break;
            
        default:
            DEBUG_ASSERT(0);
            break;
    }   
}


/**********************************************************************************

twtManagerSetState 

Description:
------------  
change the state of the agreement

Input: 
-----
StaId staId
uint8 flowId
TwtManagerFlowState_e state
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

static void twtManagerSetState(StaId staId, uint8 flowId, TwtManagerFlowState_e state)
{      
    /*Set the state in the correspond station database entry*/ 
    twtDb.twtStationDb[staId].flowIdDb[flowId].state = state;
    ILOG0_DDD("Twt manager: change state of SID: %d flow ID: %d to state: %d", staId, flowId, state);
}


/**********************************************************************************

twtManagerRemoveSta 
  
Description:
------------
when station is removed change state, configure selector and update DB

Input: 
-----
K_MSG *psMsgTwt	
		
Output:
-------	

Returns:
--------
	void 
	
**********************************************************************************/
static void twtManagerRemoveSta(K_MSG *psMsgTwt)
{                 
       BssManagerStaManagerReq_t* staManagerReq;
	   UMI_STOP_TRAFFIC*          pRemoveSta;
       uint16                     staId;
       uint8                      spId;
       uint8                      flowId;
       bool                       status;
       
       staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(psMsgTwt);
       pRemoveSta = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(staManagerReq->psMsg);       
       staId = pRemoveSta->u16SID;
       twtDb.twtStationDb[staId].isStaConnected = FALSE;
       ILOG0_D("Twt manager: remove station SID: %d", staId);
       if (twtDb.featureState == TWT_MANAGER_GLOBAL_STATE_ENABLE) //Check TWT feature operation mode
       {
            for (flowId = 0; flowId < NUM_OF_TWT_FLOWS; flowId++) /* for every flow */
            {
                
                switch(twtDb.twtStationDb[staId].flowIdDb[flowId].state)
                {
                    case TWT_MANAGER_FLOW_STATE_WAIT_FOR_RESOURCE_ALLOCATION:
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_PD);
                        status = ResourceManager_RemoveRequest(twtDb.twtStationDb[staId].flowIdDb[flowId].pdRequestId, DESC_POOL_MANAGEMENT_FROM_FW);
                        if (status == TRUE) // pending request was successfully canceled
                        {
                            twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
                        }
                        break;

                    case TWT_MANAGER_FLOW_STATE_WAIT_FOR_TX_CFM:
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_REMOVE_WHILE_WAITING_FOR_TX_CFM); 
                        break;

                    case TWT_MANAGER_FLOW_STATE_PROCESSING_TWT_REQ:
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
                        break;

                    case TWT_MANAGER_FLOW_STATE_ACTIVE:
                        spId = twtDb.twtStationDb[staId].flowIdDb[flowId].spId;
                        twtManagerSelectorConfiguration(staId, spId, flowId, FALSE); // false = delete
                        twtManagerUpdateDataBaseDelAgreement(staId, spId, flowId);
                        twtManagerSetState(staId, flowId, TWT_MANAGER_FLOW_STATE_INACTIVE);
                        break;
                        
                    case TWT_MANAGER_FLOW_STATE_INACTIVE:
                        break;

                    default:
                        ILOG0_D("Twt manager: remove station at unexpected state: %d", twtDb.twtStationDb[staId].flowIdDb[flowId].state);
                        DEBUG_ASSERT(0);
                        break;                    
                }
            }                 
       }   
       /*send cfm to STA manager*/
       twtManagerStaManagerSendConfirm(staId);
}

/**********************************************************************************
TwtManager_TwtGetStaParameters 
  

Input: pointer to twtParam struct, station Id.
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

void TwtManager_TwtGetStaParameters(twtStaParams_t *twtParams, uint16 StaId)
{
	TwtSetupFramePayload_t setupFrame;
	uint8 firstActiveAgreement;
	ASSERT(NULL != twtParams);
	DEBUG_ASSERT(StaId < NUM_OF_STAS);
	
	if((TWT_OFF == twtDb.twtStationDb[StaId].isStaConnected)  || (TWT_NO_AGREEMENT == twtDb.twtStationDb[StaId].numOfAgreementsExist))
	{
		twtParams->agreementType = TWT_AGREEMENT_TYPE_NONE;
		return;
	}
	else //station is connected and at least one active agreements exists
	{
		firstActiveAgreement = twtManagerFindFirstActiveAgreemnt(StaId);
		setupFrame = twtDb.twtStationDb[StaId].flowIdDb[firstActiveAgreement].twtFramePayload.setup;
		if(setupFrame.flowType == TWT_MANAGER_FLOW_TYPE_ANNOUNCED)
		{
			twtParams->operation.isAnnounced = TWT_ON;
		}
		else if(setupFrame.flowType == TWT_MANAGER_FLOW_TYPE_UNANNOUNCED)
		{
			twtParams->operation.isAnnounced = TWT_OFF;
		}
		
		twtParams->operation.isImplicit = setupFrame.implicit; //only implicit is supported
		twtParams->operation.isTriggerEnabled = setupFrame.trigger;
		
		if(setupFrame.negotiationType == INDIVIDUAL_TWT)
		{
			twtParams->agreementType = TWT_AGREEMENT_TYPE_INDIVIDUAL;
			twtParams->agreement.individual.wakeTime = setupFrame.targetWakeTime;
			twtParams->agreement.individual.wakeInterval = setupFrame.twtWakeIntervalMantissa << setupFrame.wakeIntervalExponent;
			twtParams->agreement.individual.minWakeDuration = setupFrame.nominalTwtWakeDuration << TWT_CHANGE_UNITS_SHIFT_WAKE_DURATION;  // value is in units of 256us
			twtParams->agreement.individual.channel = TWT_OFF;
		}
		else 
		{
			twtParams->agreementType = TWT_AGREEMENT_TYPE_BROADCAST;	//broadcast not supported
			twtParams->agreement.broadcast.listenInterval = (uint8)NULL;
			twtParams->agreement.broadcast.targetBeacon = (uint8)NULL;
		}
	}
}
/**********************************************************************************
twtManagerFindFirstActiveAgreemnt 
  

Input:  station Id.
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

uint8 twtManagerFindFirstActiveAgreemnt(uint16 StaId)
{
	uint8 flowId = 0;
	while(flowId < NUM_OF_TWT_FLOWS)
	{
		if(TWT_MANAGER_FLOW_STATE_ACTIVE == twtDb.twtStationDb[StaId].flowIdDb[flowId].state)
		{
			break;
		}
		else 
		{
			flowId++;
		}
	}
	return flowId;
}
/**********************************************************************************
TwtManager_TwtGetDefaultParameters 
  

Input: pointer to twtParam struct.
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

void TwtManager_TwtGetDefaultParameters(twtStaParams_t *twtParams)

{
	ASSERT(NULL != twtParams);
	twtParams->operation.isImplicit = TWT_ON; //only implicit is supported
	twtParams->operation.isTriggerEnabled = TWT_ON;
	twtParams->agreementType = TWT_AGREEMENT_TYPE_INDIVIDUAL;
}


/**********************************************************************************

SpAvailabilityManager_Initialize 
  

Input: 
-----	
		
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/

void SpAvailabilityManager_Initialize(void)
{// system TBD which SPs to initialize
    /* initial the SP in the selector registers*/
#ifndef PF_TWT
    uint32            spHighLimit;
    uint32            spLowLimit;
    uint32            currentTimeStamp;
    uint32            spStartTsf;
    uint8             spId;
    uint8  			  twtType = 0;
    uint8             bandId;
    uint8             vapId;
#if defined (TWT_SELECTOR_WORKAROUND) 
    uint32            counter;
    uint32            counterRowAddress;
    TwtCountRamRow_t* counterRow;
    TX_INTERRUPT_SAVE_AREA;
#endif
#endif

#ifdef ENET_INC_ARCH_WAVE600B
#ifdef ENET_INC_ARCH_WAVE600D2
	twtManagerSwHalt();
	twtManagerTypeUpdateEnable();	 
#else
    TxSelector_TwtSwHalt();
    TxSelector_TwtTypeUpdateEnable();    
#endif	
#endif

    /* init SP availability DB */
    memset(&spAvailabilityDb, 0, sizeof(SpAvailabilityManagerDataBase_t));

#ifndef PF_TWT
    spId = TWT_SP_ONE;
    spHighLimit = TWT_SP_HIGH; //  Nominal TWT Wake Duration. Values are in ~32uS resolution
    spLowLimit = TWT_SP_LOW; //  TWT Wake Interval - twtSpHighLimit. Values are in ~32uS resolution
    currentTimeStamp = GET_TSF_TIMER_LOW();
    spStartTsf = currentTimeStamp + TWT_CONST_INTERVAL; // start of the first low phase. according to JIRA 2935.        
    spAvailabilityDb.targetWakeTimeOfSpOne = spStartTsf + (spLowLimit << TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR); // targetwaketime = start of the first low phase + twtSpLowLimit (LowLimit is in res of 32us)
#if defined (TWT_SELECTOR_WORKAROUND) 
    OSAL_DISABLE_INTERRUPTS(&interrupt_save); /* according to JIRA WLANVLSIIP-2935. bug - when init TSF is smaller then actual TSF wrong value is set to regs. 
    FW workaround - add const value (we do anyway) and disable interrupts*/
#endif
    for (spId = TWT_SP_ONE; spId < MAX_NUM_OF_SP; spId++)
    {
/*#ifdef ENET_INC_ARCH_WAVE600B under comment dut to a bug in HW
        twtType = 0; //TBD according to system requirements
#endif*/
        spStartTsf += (spHighLimit << TWT_CHANGE_UNITS_SHIFT_FOR_SELECTOR); // there are 32 SPs, one after the other. (HighLimit is in res of 32us)
        vapId = spId; // TBD according to system requirements
        bandId = (uint8)ConfigurationManager_GetBandForVap(vapId);
        TxSelector_ConfigureTwtRegsNewSp(spLowLimit, spStartTsf, spHighLimit, spId, twtType, bandId);  //selector regs configuration   
#if defined (TWT_SELECTOR_WORKAROUND)        
        /* according to JIRA WLANVLSIIP-2942. bug - sometimes zero value will be written to the group counter. 
        FW workaround - check if counter value is 0 (Or 0xFFFFF). If so deactivate the SP and set it up again*/
        counter = 0;        
        counterRowAddress = TWT_COUNTER_FIXED_OFFSET_IN_SELECTOR_RAM + (TWT_RAM_LINE * spId) + (BAND1_OFFSET_FROM_BAND0 * bandId);
        counterRow = (TwtCountRamRow_t*)counterRowAddress;
        counter = counterRow->counter;
        while ((counter == TWT_ZERO_VAL) || (counter == TWT_BEFORE_ZERO))
        {
            TxSelector_ConfigureTwtRegsDeactivateSp(spId, bandId);
            TxSelector_ConfigureTwtRegsNewSp(spLowLimit, spStartTsf, spHighLimit, spId, twtType, bandId);
        } 
#endif
    }
#if defined (TWT_SELECTOR_WORKAROUND) 
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
#endif
#endif
}



/**********************************************************************************
TwtManager_PostInit 

  
Input: 
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
void TwtManager_PostInit(void)
{	
	/* set Id of twt manager registration to BSS manager*/
	twtDb.clientIdForVapManager = BSS_MANAGER_VAP_MANAGER_TWT_CLIENT;
}


/**********************************************************************************
TwtManager_Initialize 
  

Input: 
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
void TwtManager_Initialize(void)
{
#ifdef ENET_INC_ARCH_WAVE600D2
	uint32* staRamAddress = NULL;
	uint32* groupRamAddress = NULL;

	// Init RAM of band 0 (always needed)	
	staRamAddress = (void *)(B0_TWT_STATION_MEM_BASE_ADDR);	
	groupRamAddress = (void *)(B0_TWT_GROUP_MEM_BASE_ADDR);	
	memset32(staRamAddress, 0, CONVERT_BYTES_TO_WORDS(B0_TWT_STATION_MEM_SIZE)); // Band0
	memset32(groupRamAddress, 0, CONVERT_BYTES_TO_WORDS(B0_TWT_GROUP_MEM_SIZE)); // Band0

	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
	{
		// Init RAM of band 1 (if exists)
		staRamAddress = (void *)(B1_TWT_STATION_MEM_BASE_ADDR); 
		groupRamAddress = (void *)(B1_TWT_GROUP_MEM_BASE_ADDR); 
		memset32(staRamAddress, 0, CONVERT_BYTES_TO_WORDS(B1_TWT_STATION_MEM_SIZE)); // Band1
		memset32(groupRamAddress, 0, CONVERT_BYTES_TO_WORDS(B1_TWT_GROUP_MEM_SIZE)); // Band1
	}
#endif
	
    /* init twt DB */
    memset(&twtDb, 0, sizeof(TwtManagerDatabase_t));
    twtDb.featureState = TWT_MANAGER_GLOBAL_STATE_DISABLE;
}


