/***********************************************************************************
 File:			TxHanlderNtd.c
 Module:		Tx Handler 
 Purpose: 		To create Tx handler drivers which manage the transmission procedure
 Description:	This file is the implementation of the Tx handler NTD module which 
 				is responsible of the transmission decision.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "TxHandler_Api.h"
#include "TxHandlerNtd.h"
#include "RegAccess_Api.h"
#include "TxhNtdRegs.h"
#include "DeliaRecipes_Api.h"
#include "ErrorHandler_Api.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TX_HANDLER 
#define LOG_LOCAL_FID 2

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/



/**********************************************************************************

TxHandlerNtd_DisableAutoResponses 


Description:
------------
	disable auto reply for first line

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void TxHandlerNtd_DisableAutoResponses(uint8 line)
{
	RegTxhNtdAutoRespRxdLinesA_u rxdLines0to7Register;
	uint32 regMask;
	
	//max possible line # is 33
	ASSERT(line <= 33);

	//disable auto reply for the specified line - each register contains 8 lines (4 bits per line)
	//reg offset will be calculated by line / 8 (8 is the number of lines per reg) * 4 (4 is the offset in bytes between registers)
	
	rxdLines0to7Register.bitFields.autoRespRxdLinesA = 0x0;
	regMask = (0xF) << ((line % 8) * 4);
	RegAccess_WriteMasked(REG_TXH_NTD_AUTO_RESP_RXD_LINES_A + ((line / 8) * 4), regMask, rxdLines0to7Register.val);
}

/**********************************************************************************

TxHandlerNtd_DisableAllAutoResponses 


Description:
------------
			Disable auto reply for all RXD lines

Input:
-----
			None
		
Output:
-------
			None

Returns:
--------
			Void 
	
**********************************************************************************/
void TxHandlerNtd_DisableAllAutoResponses(void)
{
	RegTxhNtdAutoRespRxdLinesA_u rxdLines0to7Register;
	RegTxhNtdAutoRespRxdLinesB_u rxdLines8to15Register;
	RegTxhNtdAutoRespRxdLinesC_u rxdLines16to23Register;
	RegTxhNtdAutoRespRxdLinesD_u rxdLines24to31Register;
	RegTxhNtdAutoRespRxdLinesE_u rxdLines32to33Register;

    //init
  	rxdLines0to7Register.val = 0;
	rxdLines8to15Register.val = 0;
	rxdLines16to23Register.val = 0;
	rxdLines24to31Register.val = 0;
	rxdLines32to33Register.val = 0;

	//Disable AUTO response in all lines of RXD
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_A, rxdLines0to7Register.val);
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_B, rxdLines8to15Register.val);
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_C, rxdLines16to23Register.val);
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_D, rxdLines24to31Register.val); 
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_E, rxdLines32to33Register.val); 
}

/**********************************************************************************

TxHandlerNtd_EnableAutoResponsesInitMode 


Description:
------------
			Enable auto reply for all RXD lines as in Init Mode

Input:
-----
			None
		
Output:
-------
			None

Returns:
--------
			Void 
	
**********************************************************************************/
void TxHandlerNtd_EnableAutoResponsesInitMode(void)
{
	RegTxhNtdAutoRespRxdLinesA_u rxdLines0to7Register;
	RegTxhNtdAutoRespRxdLinesB_u rxdLines8to15Register;
	RegTxhNtdAutoRespRxdLinesC_u rxdLines16to23Register;
	RegTxhNtdAutoRespRxdLinesD_u rxdLines24to31Register;
	RegTxhNtdAutoRespRxdLinesE_u rxdLines32to33Register;

	//init	
	rxdLines0to7Register.val = 0;
	rxdLines8to15Register.val = 0;
	rxdLines16to23Register.val = 0;
	rxdLines24to31Register.val = 0;
	// TODO: not initalized in function: TxHandlerNtd_Initialize - should be?
	rxdLines32to33Register.val = 0;

	rxdLines0to7Register.bitFields.autoRespRxdLinesA = NTD_AUTO_RESPONSE_RXD_LINES_0_TO_7_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_A, rxdLines0to7Register.val);

	rxdLines8to15Register.bitFields.autoRespRxdLinesB =  NTD_AUTO_RESPONSE_RXD_LINES_8_TO_15_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_B, rxdLines8to15Register.val);

	rxdLines16to23Register.bitFields.autoRespRxdLinesC =  NTD_AUTO_RESPONSE_RXD_LINES_16_TO_23_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_C, rxdLines16to23Register.val);

	rxdLines24to31Register.bitFields.autoRespRxdLinesD =  NTD_AUTO_RESPONSE_RXD_LINES_24_TO_31_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_D, rxdLines24to31Register.val); 
		
	rxdLines32to33Register.bitFields.autoRespRxdLinesE =  NTD_AUTO_RESPONSE_RXD_LINES_32_TO_33_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_E, rxdLines32to33Register.val); 
}


/**********************************************************************************

TxHandlerNtd_Initialize 


Description:
------------
initialize internal structures and parameters of the Tx handler NTD 

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif
void TxHandlerNtd_Initialize(void)
{
 	RegTxhNtdAutoRespTcs0_u ackTcsRegister;
	RegTxhNtdAutoRespTcs1_u ctsTcsRegister;
	RegTxhNtdAutoRespTcs2_u baTcsRegister;
	RegTxhNtdAutoRespTcs3_u bfReportTcsRegister;
	RegTxhNtdAutoRespRxdLinesA_u rxdLines0to7Register;
	RegTxhNtdAutoRespRxdLinesB_u rxdLines8to15Register;
	RegTxhNtdAutoRespRxdLinesC_u rxdLines16to23Register;
	RegTxhNtdAutoRespRxdLinesD_u rxdLines24to31Register;
	RegTxhNtdAutoRespRxdLinesE_u rxdLines32to33Register;

	ackTcsRegister.val = 0;
	ctsTcsRegister.val = 0;
	baTcsRegister.val = 0;
	bfReportTcsRegister.val = 0;
	rxdLines0to7Register.val = 0;
	rxdLines8to15Register.val = 0;
	rxdLines16to23Register.val = 0;
	rxdLines24to31Register.val = 0;

	rxdLines0to7Register.bitFields.autoRespRxdLinesA = NTD_AUTO_RESPONSE_RXD_LINES_0_TO_7_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_A, rxdLines0to7Register.val);

	rxdLines8to15Register.bitFields.autoRespRxdLinesB =  NTD_AUTO_RESPONSE_RXD_LINES_8_TO_15_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_B, rxdLines8to15Register.val);

	rxdLines16to23Register.bitFields.autoRespRxdLinesC =  NTD_AUTO_RESPONSE_RXD_LINES_16_TO_23_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_C, rxdLines16to23Register.val);

	rxdLines24to31Register.bitFields.autoRespRxdLinesD =  NTD_AUTO_RESPONSE_RXD_LINES_24_TO_31_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_D, rxdLines24to31Register.val);	
		
	rxdLines32to33Register.bitFields.autoRespRxdLinesE =  NTD_AUTO_RESPONSE_RXD_LINES_32_TO_33_INIT_VALUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_RXD_LINES_E, rxdLines32to33Register.val); 

	ackTcsRegister.bitFields.autoResp0RecipePtr = (uint32)DeliaRecipes_GetAckRecipeAddress();
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_TCS0, ackTcsRegister.val);

	ctsTcsRegister.bitFields.autoResp1RecipePtr = (uint32)DeliaRecipes_GetCtsAutoResponseRecipeAddress();
	ctsTcsRegister.bitFields.autoResp1RxNavAbort = TRUE;
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_TCS1, ctsTcsRegister.val);
	
	baTcsRegister.bitFields.autoResp2RecipePtr = (uint32)DeliaRecipes_GetBaRecipeAddress();
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_TCS2, baTcsRegister.val); 
		
	bfReportTcsRegister.bitFields.autoResp3RecipePtr = (uint32)DeliaRecipes_GetBfReportAutoResponseRecipeAddress();
	RegAccess_Write(REG_TXH_NTD_AUTO_RESP_TCS3, bfReportTcsRegister.val); 
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))

#pragma ghs section text=default
#endif



