/***********************************************************************************
 File:			TxHanlderMap.c
 Module:		      Tx Handler 
 Purpose: 		To create a Tx handler drivers which manage the transmission procedure
 Description:		This file is the implementation of the Tx handler Map module 
				which is responsible of managing the transmission procedure.
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "TxHandler_Api.h"
#include "TxHandlerRegs.h"
#include "TxHandlerMap.h"
#include "RegAccess_Api.h"
#include "ShramTxHandlerTimingParams.h"
#include "stringLibApi.h"
#include "DeliaRecipes_Api.h"
#include "string.h"
#include "ieee80211.h"
#include "Utils_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TX_HANDLER 
#define LOG_LOCAL_FID 5

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

TxHandler_UpdateVapAcTimingParameters  


Description:
------------
configure specific Vap AC timing parameters

Input:
-----
	vapID - VAP id
	ac -  the priority that should be configured
	configurationTimingParams -  a pointer to the timing parameters structure	
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxHandler_UpdateVapAcTimingParameters(uint8 vapId, uint8 ac ,TxHandlerVapTimingParams_t *configurationTimingParams)
{
	HwAcTimingParams_t hwTimingParams;
	
	memset(&hwTimingParams,0,sizeof(HwAcTimingParams_t));
    
	hwTimingParams.aifsn =  configurationTimingParams->aifsn;
	hwTimingParams.minimumContentionWindow = configurationTimingParams->minimumContentionWindow;
	hwTimingParams.maximumContentionWindow = configurationTimingParams->maximumContentionWindow;
	hwTimingParams.currentContentionWindow = hwTimingParams.minimumContentionWindow;
	hwTimingParams.ifsn = hwTimingParams.aifsn;

	TxHandler_SetVapAcTimingParameters(vapId, ac, &hwTimingParams);
}

#ifdef ENET_INC_LMAC

/**********************************************************************************

TxHandler_SetVapAcTimingParameters  


Description:
------------


Input:
-----

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxHandler_SetVapAcTimingParameters(uint8 vapId, uint8 ac ,HwAcTimingParams_t *hwTimingParams_p)
{
    /* using memcpy instead of direct approach to memory in order to save write */
#if defined(ENET_INC_LMAC0) 	
	memcpy32(&HwVapsAcsTimingParameters[vapId][ac], hwTimingParams_p, CONVERT_BYTES_TO_WORDS(sizeof(HwAcTimingParams_t)));	
#elif defined(ENET_INC_LMAC1) 	
	memcpy32(&HwVapsAcsTimingParameters_B1[vapId][ac], hwTimingParams_p, CONVERT_BYTES_TO_WORDS(sizeof(HwAcTimingParams_t)));	
#endif
}

/**********************************************************************************

TxHandler_SetVapAcTimingParameters  


Description:
------------


Input:
-----

Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void TxHandler_GetVapAcTimingParameters(uint8 vapId, uint8 ac, HwAcTimingParams_t *hwTimingParams_p)
{
	/* access to TXH timing memory must be 32 bit aligned */
#if defined(ENET_INC_LMAC0) 	
	memcpy32(hwTimingParams_p, &HwVapsAcsTimingParameters[vapId][ac], CONVERT_BYTES_TO_WORDS(sizeof(HwAcTimingParams_t)));	
#elif defined(ENET_INC_LMAC1)
	memcpy32(hwTimingParams_p, &HwVapsAcsTimingParameters_B1[vapId][ac], CONVERT_BYTES_TO_WORDS(sizeof(HwAcTimingParams_t)));	
#endif
}
#endif //ENET_INC_LMAC

/**********************************************************************************

TxHandler_UpdateVapTimingParameters  


Description:
------------
configure all ACs timing parameters for a specific Vap

Input:
-----
	setWmmMessageParameters - a pointer to the message that contains the parameters	
	
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxHandler_UpdateVapTimingParameters(UMI_SET_WMM_PARAMETERS* setWmmMessageParameters)
{
	uint8 priorityIdx;	
	uint8 vapId = 0;
	HwAcTimingParams_t hwTimingParameters;
	
	memset(&hwTimingParameters, 0, sizeof(HwAcTimingParams_t));
	

	vapId = setWmmMessageParameters->vapId;
	
	for (priorityIdx = 0 ; priorityIdx <= HW_AC_VO ; priorityIdx++) 
	{   
		hwTimingParameters.aifsn =  setWmmMessageParameters->wmmParamsPerAc[priorityIdx].u8Aifsn;
		hwTimingParameters.minimumContentionWindow = VALUE_TO_EXPONENT(setWmmMessageParameters->wmmParamsPerAc[priorityIdx].u16CWmin);	
		hwTimingParameters.maximumContentionWindow = VALUE_TO_EXPONENT(setWmmMessageParameters->wmmParamsPerAc[priorityIdx].u16CWmax);
		hwTimingParameters.currentContentionWindow = hwTimingParameters.minimumContentionWindow;
		hwTimingParameters.ifsn = hwTimingParameters.aifsn;
		TxHandler_SetVapAcTimingParameters(vapId, priorityIdx, &hwTimingParameters);
	}

	/*Set all non-standard priorities*/
	for (; priorityIdx < HW_AC_NUM_OF_PRIORITIES ; priorityIdx++)
	{
		/*GPHP uses BO = 0*/
		if ((priorityIdx == HW_AC_GPHP) || (priorityIdx == HW_AC_BEACON))
		{
			hwTimingParameters.aifsn =	setWmmMessageParameters->wmmParamsPerAc[HW_AC_VO].u8Aifsn;
			hwTimingParameters.minimumContentionWindow = 0; 
			hwTimingParameters.maximumContentionWindow = 0;
		}
		else
		{
			/* Copy the parameters of BE to all other none standard priorities */
			hwTimingParameters.aifsn =	setWmmMessageParameters->wmmParamsPerAc[HW_AC_BE].u8Aifsn;
			hwTimingParameters.minimumContentionWindow = VALUE_TO_EXPONENT(setWmmMessageParameters->wmmParamsPerAc[HW_AC_BE].u16CWmin); 
			hwTimingParameters.maximumContentionWindow = VALUE_TO_EXPONENT(setWmmMessageParameters->wmmParamsPerAc[HW_AC_BE].u16CWmax);
		}
		hwTimingParameters.currentContentionWindow = hwTimingParameters.minimumContentionWindow;
		hwTimingParameters.ifsn = hwTimingParameters.aifsn;
		TxHandler_SetVapAcTimingParameters(vapId, priorityIdx, &hwTimingParameters);
	}	
}


/**********************************************************************************

TxHandler_DisableTxh  


Description:
------------
Disable the Txh map - will stop transmissions (use only for debug)


Input:
-----

Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
void TxHandler_DisableTxh(void)
{
	RegTxHandlerTxhMapControl_u txhMapControlRegister;
	
	RegAccess_Read(REG_TX_HANDLER_TXH_MAP_CONTROL, &txhMapControlRegister.val);
	txhMapControlRegister.bitFields.txhMapEnable = FALSE;
	RegAccess_Write(REG_TX_HANDLER_TXH_MAP_CONTROL, txhMapControlRegister.val);
}

/**********************************************************************************

TxHandler_Unfreeze  


Description:
------------
unfreeze the TX handler after transmission

Input:
-----
	unfreezeParams -  a pointer to the unfreeze parameters structure	
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void TxHandler_Unfreeze(TxHandlerUnfreezeParams_t *unfreezeParams)
{
	RegTxHandlerUnfreezeReportSet_u unfreezeParamsRegister;

	unfreezeParamsRegister.val = 0;

	unfreezeParamsRegister.bitFields.unfreezeCommand = TRUE;
	unfreezeParamsRegister.bitFields.ctsRxInfoUpdate = unfreezeParams->isCtsInformationValid;
	unfreezeParamsRegister.bitFields.txResultUpdate = unfreezeParams->isTxResultValid;
	unfreezeParamsRegister.bitFields.ctsRxInfo = unfreezeParams->ctsInformation;
	unfreezeParamsRegister.bitFields.txResult = unfreezeParams->transmissionInformation;
	unfreezeParamsRegister.bitFields.txFrameLength = unfreezeParams->isLongPacket;	

	RegAccess_Write(REG_TX_HANDLER_UNFREEZE_REPORT_SET, unfreezeParamsRegister.val);
}
 
/**********************************************************************************

TxHandlerMap_Initialize 


Description:
------------
initialize internal structures and parameters of the Tx handler Map whose values are known in initialization

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxHandlerMap_Initialize(void)
{
	RegTxHandlerTxhMapControl_u controlRegister;	
	RegTxHandlerRecipePtrAcId0To3_u recipePointerReg;
	RegTxHandlerTxhEpsilonMicroPrescalerNum_u epsilonHwCyclesPortionRegister;
	uint16 epsilonHwCyclesPortion = 0;

    controlRegister.val = 0;
	recipePointerReg.val = 0;
	epsilonHwCyclesPortionRegister.val = 0;

	// set all bits in the control register - halt the Tx handler in case of aggregation builder error
	controlRegister.bitFields.txhMapEnable = TRUE;
	controlRegister.bitFields.punishOnInternalCollisionEn = TRUE;
	controlRegister.bitFields.increaseOnIntColSsrcSlrcN = TRUE;
	controlRegister.bitFields.txGroupAddressedSetEcwminEn = TRUE;
	controlRegister.bitFields.retryCtrLimitZeroedCtrEn = TRUE;
	
	RegAccess_Write(REG_TX_HANDLER_TXH_MAP_CONTROL, controlRegister.val);

	/* Ptr to recipe when Regular AC backoff expired */
	recipePointerReg.bitFields.recipePtrAcId0To3 = DeliaRecipes_GetRecipe0Address();
	RegAccess_Write(REG_TX_HANDLER_RECIPE_PTR_AC_ID0TO3, recipePointerReg.val);	
	/* Ptr to recipe when Beacon backoff expired */
	RegAccess_Write(REG_TX_HANDLER_RECIPE_PTR_AC_BEACON,recipePointerReg.val);
	/* Ptr to recipe when GPHP\GPLP backoff expired */
	RegAccess_Write(REG_TX_HANDLER_RECIPE_PTR_AC_GPLP,recipePointerReg.val);
	RegAccess_Write(REG_TX_HANDLER_RECIPE_PTR_AC_GPHP,recipePointerReg.val);

    /* This function return frequency in MHz which is equal to number of cycles in micorsecond */
	epsilonHwCyclesPortion = System_ConvertNsToSystemClk(TX_HANDLER_MAP_EPSILON_VALUE_IN_NS); 
	epsilonHwCyclesPortionRegister.bitFields.txhEpsilonMicroPrescalerNum = (epsilonHwCyclesPortion) - 1;
	epsilonHwCyclesPortionRegister.bitFields.txhEpsilonMicroPrescalerNumFreqReduced = (epsilonHwCyclesPortion >> 1) - 1;
	RegAccess_Write(REG_TX_HANDLER_TXH_EPSILON_MICRO_PRESCALER_NUM, epsilonHwCyclesPortionRegister.val); 
}


