/***********************************************************************************
 File:			TxManager.c
 Module:		Tx manager
 Purpose: 		To handle Tx packets whose transmission proccess was ended
 Description:   This file is the implementation of the Tx manager which is responsible 
 				to handle Tx packets whose transmission proccess was ended
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "TxManager_Api.h"
#include "TxHandler_Api.h"
#include "TxManager.h"
#include "TxManager_StaManager.h"
#include "TxMulticastHandler_API.h"
#include "TsManager_API.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "Locker_Api.h"
#include "HwMemoryMap.h"
#include "BSSmanager_API.h"
#include "CsaManager_Api.h"
#include "CtsManager_Api.h"
#include "EventsManager_api.h"
#include "Dut_Api.h"
#include "TxPacketsClassifier_API.h"
#include "SmpsManager_Api.h"
#include "PsManager_Api.h"
#include "shramTxDesc.h"
#include "TxHandler_Api.h"
#include "StaDatabase_Api.h"
#ifndef TRAINING_WAVE600_Z0
#include "NdpaManager_Api.h"
#endif
#include "GidmManager_Api.h"
#include "TxSelector_Api.h"
#include "ShramStatistics.h"
#include "Statistics_Api.h"
#if defined(ENET_INC_ARCH_WAVE600)
#include "OtfAggregator_Api.h"
#include "TwtManager_API.h"
#else
#include "AggregationBuilder_Api.h"
#endif 
#include "Hdk_Api.h"
#include "HeGroupManager_API.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TX_MANAGER 
#define LOG_LOCAL_FID 0

#if defined(ENET_INC_ARCH_WAVE600)
#define TX_MANAGER_DEBUG
#endif

#define NUM_OF_VAP_DISABLE_QUEUE		4 
#define MAX_PD_LENGHT_SU_ETSI			1300
#define MAX_PD_LENGHT_VHT_MU_ETSI		950
#define MAX_PD_LENGHT_HE_MU_ETSI		850

#ifdef TX_DATA_THREAD_PATH
#define TX_MANAGER_STARVATION_TIMER_IN_MILLISECONDS	(100)
#endif

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

typedef struct 
{
	uint8 numOfDisableCallbacks; 
} txManagerVapDbElement;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
/* Messages handlers */
static void txManagerDoneListsNotEmpty(K_MSG* txManagerMessage);
#ifdef TX_DATA_THREAD_PATH
static void txManagerDataListsNotEmpty(K_MSG* txManagerMessage);
static void txManagerStarvationTimer(K_MSG* txManagerMessage);
#endif
static void txManagerDisableListsNotEmpty(K_MSG* txManagerMessage);
#ifndef ENET_INC_ARCH_WAVE600
static void txManagerFlushDisableList(K_MSG* txManagerMessage);
#endif
void txManagerStatisticsPerClientCount(TxPd_t *pd);


/* Lists handlers */
static void txManagerHostDataListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
static void txManagerDiscardedPacketsListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
static void txManagerPacketsFromFwListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
static void txManagerPacketsFromDriverListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
static void txManagerReliableMulticastListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
static void txManagerPsSettingsHwListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
#if defined (AGER_STATUS_WORKAROUND)
static void txManagerAgerLiberatorListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor);
#endif	 


/* FW packets handlers */
static void txManagerAddbaRequestHandler(TxPd_t *packetDescriptor);
static void txManagerCsaHandler(TxPd_t *packetDescriptor);
static void txManagerCtrlHandler(TxPd_t *packetDescriptor);
static void txManagerBarHandler(TxPd_t *packetDescriptor);
static void txManagerGeneralManagementHandler(TxPd_t *packetDescriptor);

#ifndef TRAINING_WAVE600_Z0
static void txManagerNdpaHandler(TxPd_t *packetDescriptor);
#endif 

/* Discarded packets handlers */
static TxPd_t* txManagerDiscardedDataPacketsHandler(TxPd_t *packetDescriptor);
static TxPd_t* txManagerGeneralDiscardedPacketHandler(TxPd_t *packetDescriptor);
static TxPd_t* txManagerDiscardedAddbaRequestHandler(TxPd_t *packetDescriptor);
static TxPd_t* txManagerDiscardedBarHandler(TxPd_t *packetDescriptor);
static TxPd_t* txManagerDiscardedCsaHandler(TxPd_t *packetDescriptor);


/* General functions */
static void txManagerAssert(TxPd_t *packetDescriptor);
static void txManagerListFatal(TxPd_t *headPd, TxPd_t *tailPd);
static void txManagerActionFrameHandler(TxPd_t *packetDescriptor);
static void txManagerDeauthFrameHandler(TxPd_t *packetDescriptor);


static void txManagerAddVap(K_MSG* txManagerMessage);
static void txManagerPostVapAddition(K_MSG* txManagerMessage);

static void txManagerEnableVap(K_MSG* txManagerMessage);
static void txManagerSetWmmParameters(K_MSG* txManagerMessage);
static void txManagerDisableVap(K_MSG* txManagerMessage); 
static void txManagerVapQueueDisabled(K_MSG *lockerMessage); 
static void txManagerRecoverFromAggregatorError(K_MSG* txManagerMessage);
static void TxManagerdutTransmitPacketReq(K_MSG* pMsg);
static void TxManagerCBdutTransmitPacketCfm(K_MSG* pMsg);


void TxManagerDropMsduOrAmsdu(bool aggregation, TxPd_t* headPd, StaId stationIndex, uint8 tid);
#ifdef ENET_INC_ARCH_WAVE600B
bool txManagerIsLengthGreaterThanEtsiLimitation(TxPd_t* headPd);
#endif

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static const FunctionEntry_t afpTaskTable[TASK_TX_MANAGER_END - TASK_TX_MANAGER_START]=
{
	/*Done List Handler*/
	{txManagerDoneListsNotEmpty,			        DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_DONE_LIST_NOT_EMPTY)},	
#ifdef TX_DATA_THREAD_PATH
	{txManagerDataListsNotEmpty,					DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_DATA_LIST_NOT_EMPTY)}, 
	{txManagerStarvationTimer,						DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STARVATION_TIMER)}, 
#endif		
	{txManagerDisableListsNotEmpty,			        DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_DISABLE_LIST_NOT_EMPTY)},		
#ifndef ENET_INC_ARCH_WAVE600
	{txManagerFlushDisableList,                     DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_VAP_GPHP_LOCK)},
#endif		
    /*STA Manager messages*/
	{txManagerStaOpen,								DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STA_OPEN)}, 
	{txManagerStaStopTraffic,						DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STA_STOP_TRAFFIC)}, 
	{txManagerStaPostStopTraffic,					DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STA_POST_STOP_TRAFFIC)}, 
	{txManagerStaClose,								DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STA_CLOSE)}, 
	{txManagerStaLockEv, 							DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_STA_LOCK_EV)}, 
	{txManagerPostVapAddition, 						DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_POST_VAP_ADDITION)},
	{txManagerEnableVap, 							DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_ENABLE_VAP)},
	{txManagerSetWmmParameters, 					DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_SET_WMM_PARAMETERS)},
	{txManagerDisableVap,							DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_DISABLE_VAP)},
	{txManagerVapQueueDisabled, 					DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_VAP_QUEUE_DISABLED)},
	{txManagerRecoverFromAggregatorError, 			DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_AGGREGATOR_ERROR_RECOVER)},

	/*CSA messages*/	
	{csaManagerVapOpen,								DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_VAP_OPEN)}, 
	{csaManagerVapRemove,							DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_VAP_CLOSE)}, 
	{csaManagerStaOpen,								DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_STA_OPEN)}, 
	{csaManagerStaRemove,                           DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_STA_CLOSE)}, 		
	{csaManagerTxStartReq,							DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_START_TX)}, 
	{csaManagerTxStopReq,							DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_STOP_TX)}, 
	{csaManagerTxCfm,								DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_TX_CFM)}, 
	{csaManagerPdAlloc,								DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_PD_ALLOC)}, 
	{csaManagerVapGphpLockCfm,						DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_VAP_GPHP_LOCK)}, 
	{csaManagerStaGphpLockCfm, 						DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_STA_GPHP_LOCK)}, 
#if defined CSA_REMOVE_RETRY	
	{csaManagerStaMngLockCfm,						DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_STA_MNG_LOCK)},
#endif
	{csaManagerStartTxUcDeauthTimerExpired,			DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_START_TX_UC_DEAUTH)},
	{csaManagerStartTxMcDeauthTimerExpired,			DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_START_TX_MC_DEAUTH)},
#if defined(ENET_INC_ARCH_WAVE600)
	{csaManagerStartTxUcDeauthTimerExpired_B1, 		DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_START_TX_UC_DEAUTH_B1)},
	{csaManagerStartTxMcDeauthTimerExpired_B1, 		DOUBLE_CHECK_MSG_TYPE(CSA_MANAGER_START_TX_MC_DEAUTH_B1)},
#endif

	/*CTS Messages*/
	{ctsManagerVapOpen, 							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_VAP_OPEN)}, 
	{ctsManagerVapRemove,							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_VAP_CLOSE)}, 
	{ctsManagerStartTx, 							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_START_TX)}, 
	{ctsManagerStopTx,								DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_STOP_TX)}, 
#if defined(ENET_INC_ARCH_WAVE600)
	{ctsManagerStopTx_B1,							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_STOP_TX_B1)}, 
#endif
	{ctsManagerTxCfm,								DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_TX_CFM)}, 
	{ctsManagerPdAlloc, 							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_PD_ALLOC)}, 
	{ctsManagerGphpLock,							DOUBLE_CHECK_MSG_TYPE(CTS_MANAGER_GPHP_LOCK)},
	{txManagerAddVap,								DOUBLE_CHECK_MSG_TYPE(TX_MANAGER_ADD_VAP)},
	/*Multicast Handler Messages*/
	{MulticastHandler_MulticastGroupActionReq,		DOUBLE_CHECK_MSG_TYPE(MULTICAST_HANDLER_GROUP_ACTION)}, 
	{MulticastHandler_StaStopTraffic,				DOUBLE_CHECK_MSG_TYPE(MULTICAST_HANDLER_STA_STOP_TRAFFIC)}, 
	/*SMPS Messages*/
	{smpsManagerVapOpen,							DOUBLE_CHECK_MSG_TYPE(SMPS_ADD_VAP)},
	{smpsManagerVapRemove,							DOUBLE_CHECK_MSG_TYPE(SMPS_REMOVE_VAP)},
	{smpsManagerStaOpen,							DOUBLE_CHECK_MSG_TYPE(SMPS_ADD_STA)},
	{smpsManagerStaRemove,							DOUBLE_CHECK_MSG_TYPE(SMPS_REMOVE_STA)},
	{smpsManagerTxCfm,								DOUBLE_CHECK_MSG_TYPE(SMPS_TX_CFM)},
	{smpsManagerPdAlloc,							DOUBLE_CHECK_MSG_TYPE(SMPS_PD_ALLOC_CFM)},
	{smpsManagerLockCfm,							DOUBLE_CHECK_MSG_TYPE(SMPS_LOCK_CFM)},
	{smpsManagerTxStartReq,							DOUBLE_CHECK_MSG_TYPE(SMPS_TX_START_REQ)},
	{smpsManager_GetChannelWidth,					DOUBLE_CHECK_MSG_TYPE(SMPS_CHANNEL_WIDTH)},		
	{TxManagerdutTransmitPacketReq,					DOUBLE_CHECK_MSG_TYPE(DUT_TRANSMIT_PACKET_REQ)},
	{TxManagerCBdutTransmitPacketCfm,				DOUBLE_CHECK_MSG_TYPE(DUT_ACTION_PACKET_CB_HNDLR_CFM)},
	
};

static const TxManagerListHandler TxManagerListsHandlers[HW_Q_MANAGER_DONE_LIST_LAST_DONE_LIST_HANDLED_BY_FW] =
{
	 txManagerHostDataListHandler,			/* HW_Q_MANAGER_DONE_LIST_HOST_DATA */
	 txManagerDiscardedPacketsListHandler,	/* HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS */	
	 txManagerPacketsFromFwListHandler,		/* HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_FW */
	 txManagerPacketsFromDriverListHandler,	/* HW_Q_MANAGER_DONE_LIST_MANAGEMENT_FROM_HOST */
	 txManagerReliableMulticastListHandler,	/* HW_Q_MANAGER_DONE_LIST_MULTICAST */
	 txManagerPsSettingsHwListHandler,		/* HW_Q_MANAGER_DONE_LIST_PS_SETTINGS_HW */
#if defined (AGER_STATUS_WORKAROUND)
	txManagerAgerLiberatorListHandler,	   	/* HW_Q_MANAGER_DONE_LIST_AGER_LIBERATOR */
#endif	 
	 txManagerListFatal,					/* HW_Q_MANAGER_DONE_LIST_ERROR */ 
};	


static const TxManagerPdHandler TxManagerFwListPdsHandlers[PD_TYPE_LAST] =
{
#if defined (ENET_INC_ARCH_WAVE600)
	txManagerAssert,	                /* PD_TYPE_DATA                                  */
	txManagerAssert,	                /* PD_TYPE_DATA_WPA_PT                           */
	txManagerAssert,	                /* PD_TYPE_NDP                                   */
	txManagerAddbaRequestHandler,	    /* PD_TYPE_ADDBA_REQ                             */
	txManagerBarHandler,	            /* PD_TYPE_BAR                                   */
	txManagerCsaHandler,	            /* PD_TYPE_CH_SWITCH_ANN                         */
	txManagerGeneralManagementHandler,	/* PD_TYPE_MANAGEMENT_UNENC_DO_NOT_FRAG          */
	txManagerGeneralManagementHandler,	/* PD_TYPE_MANAGEMENT_ENC_DO_NOT_FRAG            */
#ifndef TRAINING_WAVE600_Z0
	txManagerNdpaHandler,				/* PD_TYPE_MU_TRAINING							 */
#else
	txManagerAssert,	                /* PD_TYPE_MU_TRAINING                           */
#endif
	txManagerGeneralManagementHandler,	/* PD_TYPE_MANAGEMENT_UNENC                      */
	txManagerGeneralManagementHandler,	/* PD_TYPE_MANAGEMENT_MULTI_PD_UNENC_DO_NOT_FRAG */
	txManagerGeneralManagementHandler,	/* PD_TYPE_MANAGEMENT_ENC                        */
	txManagerCtrlHandler,	            /* PD_TYPE_CONTROL_UNENC                         */
	txManagerAssert,	                /* PD_TYPE_PROBE_RES                             */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_SIGNED_BIP                 */
	txManagerCsaHandler,                /* PD_TYPE_CH_SWITCH_ANN_ENC                     */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_MULTI_PD_ENC_DO_NOT_FRAG   */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_SIGNED_BIP_DO_NOT_FRAG     */
#ifdef WORKAROUND_FOR_HW_BUG_IN_ADDBA_REQ_ENCRYPTED
	txManagerAddbaRequestHandler,       /* PD_TYPE_ADDBA_REQ_ENC         */
#else
	txManagerAssert,					/* PD_TYPE_COLOR_SWITCH_ANNOUNCEMENT_ENC		 */
#endif
	txManagerAssert,	                /* PD_TYPE_COLOR_SWITCH_ANNOUNCEMENT_UNENC       */
	txManagerAssert,	                /* PD_TYPE_NDP_NO_RESPONSE                       */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_NO_RESPONSE_UNENC_NO_FRAG  */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_NO_RESPONSE_ENC_NO_FRAG    */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_NO_RESPONSE_UNENC_FRAG     */
	txManagerAssert,	                /* PD_TYPE_MANAGEMENT_NO_RESPONSE_ENC_FRAG       */
	txManagerAssert,	                /* PD_TYPE_RESERVED_25                           */
	txManagerAssert,	                /* PD_TYPE_RESERVED_26                           */
	txManagerAssert,	                /* PD_TYPE_RESERVED_27                           */
	txManagerAssert,	                /* PD_TYPE_RESERVED_28                           */
	txManagerAssert,	                /* PD_TYPE_NON_PD_TF                             */
	txManagerAssert,	                /* PD_TYPE_ONLY_ZLDS_MPDU                        */
	txManagerAssert,	                /* PD_TYPE_BEACON                                */

#else

	txManagerAssert,	                /* PD_TYPE_DATA */ 
	txManagerAssert,                    /* PD_TYPE_DATA_RESERVED */
	txManagerAssert,                    /* PD_TYPE_NDP */
	txManagerAddbaRequestHandler,       /* PD_TYPE_ADDBA_REQ */
	txManagerBarHandler,                /* PD_TYPE_BAR */
	txManagerCsaHandler,                /* PD_TYPE_CH_SWITCH_ANN */
	txManagerGeneralManagementHandler,  /* PD_TYPE_MANAGEMENT_UNENC_RESERVED */
	txManagerGeneralManagementHandler,  /* PD_TYPE_MANAGEMENT_ENC_RESERVED */
	txManagerNdpaHandler,               /* PD_TYPE_MU_TRAINING */
	txManagerGeneralManagementHandler,  /* PD_TYPE_MANAGEMENT_UNENC */
	txManagerAssert,                    /* PD_TYPE_MANAGEMENT_UNENC_FRAG */ /* This type is currently not in used */
	txManagerGeneralManagementHandler,  /* PD_TYPE_MANAGEMENT_ENC */
	txManagerCtrlHandler,               /* PD_TYPE_CONTROL_UNENC */
	txManagerAssert,                    /* PD_TYPE_PROBE_RES */
	txManagerAssert,                    /* PD_TYPE_MANAGEMENT_ENC_BIP */ /* Multicast management are not sent by FW */
	txManagerAssert,                    /* PD_TYPE_BEACON */

#endif //defined (ENET_INC_ARCH_WAVE600)
};

static const TxManagerDiscardedPdHandler TxManagerDiscardedListPdsHandlers[PD_TYPE_LAST] =
{
#if defined (ENET_INC_ARCH_WAVE600)

	txManagerDiscardedDataPacketsHandler,	/* PD_TYPE_DATA 								 */
	txManagerDiscardedDataPacketsHandler,	/* PD_TYPE_DATA_WPA_PT							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_NDP									 */
	txManagerDiscardedAddbaRequestHandler,	/* PD_TYPE_ADDBA_REQ							 */
	txManagerDiscardedBarHandler,			/* PD_TYPE_BAR									 */
	txManagerDiscardedCsaHandler,			/* PD_TYPE_CH_SWITCH_ANN						 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_UNENC_DO_NOT_FRAG 		 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_ENC_DO_NOT_FRAG			 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MU_TRAINING							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_UNENC 					 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_MULTI_PD_UNENC_DO_NOT_FRAG */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_ENC						 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_CONTROL_UNENC						 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_PROBE_RES							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_SIGNED_BIP				 */
	txManagerDiscardedCsaHandler,			/* PD_TYPE_CH_SWITCH_ANN_ENC					 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_MULTI_PD_ENC_DO_NOT_FRAG	 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_SIGNED_BIP_DO_NOT_FRAG	 */
#ifdef WORKAROUND_FOR_HW_BUG_IN_ADDBA_REQ_ENCRYPTED
	txManagerDiscardedAddbaRequestHandler,  /* PD_TYPE_ADDBA_REQ_ENC		 				*/
#else
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_COLOR_SWITCH_ANNOUNCEMENT_ENC		 */
#endif
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_COLOR_SWITCH_ANNOUNCEMENT_UNENC		 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_NDP_NO_RESPONSE						 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_NO_RESPONSE_UNENC_NO_FRAG  */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_NO_RESPONSE_ENC_NO_FRAG	 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_NO_RESPONSE_UNENC_FRAG	 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_MANAGEMENT_NO_RESPONSE_ENC_FRAG		 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_RESERVED_25							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_RESERVED_26							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_RESERVED_27							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_RESERVED_28							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_NON_PD_TF							 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_ONLY_ZLDS_MPDU						 */
	txManagerGeneralDiscardedPacketHandler, /* PD_TYPE_BEACON								 */

#else

	txManagerDiscardedDataPacketsHandler,	  /* PD_TYPE_DATA */ 
	txManagerDiscardedDataPacketsHandler,    /* PD_TYPE_DATA_RESERVED */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_NDP */
	txManagerDiscardedAddbaRequestHandler,   /* PD_TYPE_ADDBA_REQ */
	txManagerDiscardedBarHandler,            /* PD_TYPE_BAR */
	txManagerDiscardedCsaHandler,			  /* PD_TYPE_CH_SWITCH_ANN */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_UNENC_RESERVED */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_ENC_RESERVED */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_CONTROL_UNENC_RESERVED */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_UNENC */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_UNENC_FRAG */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_ENC */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_CONTROL_UNENC */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_PROBE_RES */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_MANAGEMENT_ENC_BIP */
	txManagerGeneralDiscardedPacketHandler,  /* PD_TYPE_BEACON */ 

#endif //defined (ENET_INC_ARCH_WAVE600)
};	



static const TxManagerPdHandler TxManagerFwManagementHandlers[MGMT_FRAME_SUBTYPE_LAST] = 
{  
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_ASSOCIATION_REQ */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_ASSOCIATION_RES */
	 txManagerAssert,                /* MGMT_FRAME_SUBTYPE_REASSOCIATION_REQ */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_REASSOCIATION_RES */
	 txManagerAssert,                /* MGMT_FRAME_SUBTYPE_PROBE_REQ */
	 txManagerAssert,                /* MGMT_FRAME_SUBTYPE_PROBE_RES */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_TIMING_ADVERTISEMENT */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_RESERVED1 */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_BEACON */
	 txManagerAssert,                /* MGMT_FRAME_SUBTYPE_ATIM */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_DISASSOCIATION */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_AUTHENTICATION */
	 txManagerDeauthFrameHandler,    /* MGMT_FRAME_SUBTYPE_DEAUTHENTICATION */
	 txManagerActionFrameHandler,	 /* MGMT_FRAME_SUBTYPE_ACTION */
	 txManagerActionFrameHandler,    /* MGMT_FRAME_SUBTYPE_ACTION_NO_ACK */
	 txManagerAssert,	             /* MGMT_FRAME_SUBTYPE_RESERVED2 */
};	


txManagerVapDbElement txManagerVapDb[HW_NUM_OF_VAPS]; 


const uint8 TxManagerActoLowTid[ACCESS_CATEGORY_NUM] =
{
    IEEE802_1D_BE_1,
    IEEE802_1D_BK_1,
    IEEE802_1D_VI_1, 
    IEEE802_1D_VO_1, 
};



#ifdef TX_MANAGER_DEBUG								
//TBD debug parameters should be removed
uint32 isr_TxManagerCount=0;
uint32 QhandlerCount[NUM_HW_Q_MANAGER_DONE_LIST]={0};
uint32 QDisabledCount[10]={0,0,0,0,0,0,0,0,0,0}; // HW_Q_MANAGER_NUM_OF_DISABLE_LISTS_MASTERS (10 in gen6, 7 in gen5)
uint32 ManagementFromHost =0;
////
#endif



/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

txManagerDoneListsNotEmpty 


Description:
------------
handles the event that one of the done lists became not empty 


Input: 
-----	
txManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
TxPd_t* AAA_DONE_LIST_HEAD;
TxPd_t* AAA_DONE_LIST_TAIL;


static void txManagerDoneListsNotEmpty(K_MSG* txManagerMessage)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint16 notEmptyDoneLists = 0;
	uint8 listIndex = 0;
	UNUSED_PARAM(txManagerMessage);
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	notEmptyDoneLists = HwQManager_GetNotEmptyTxDoneLists();

	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
#ifdef TX_MANAGER_DEBUG								
	isr_TxManagerCount++;
#endif	
	
	while(notEmptyDoneLists)
	{
	    if(HW_Q_MANAGER_LIST_BIT_IS_SET_MASK & notEmptyDoneLists)
	    {
			ASSERT(listIndex < HW_Q_MANAGER_DONE_LIST_LAST_DONE_LIST_HANDLED_BY_FW);
			HwQManager_ClearTxListsInt(listIndex);
	    	hwQueueManagerRequestParams.dplIndex = listIndex;
	    	HwQManager_FlushQ(&hwQueueManagerRequestParams);
#ifdef TX_MANAGER_DEBUG								
			QhandlerCount[listIndex]++;
#endif
			AAA_DONE_LIST_HEAD = (TxPd_t *)hwQueueManagerRequestParams.pHeadDesc;
			AAA_DONE_LIST_TAIL = (TxPd_t *)hwQueueManagerRequestParams.pTailDesc;

			ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
			ASSERT(hwQueueManagerRequestParams.pTailDesc != NULL_PD);
			ASSERT(((TxPd_t *)hwQueueManagerRequestParams.pTailDesc)->nextPdPointer == NEXT_PD_NULL);
			TxManagerListsHandlers[listIndex]((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc,(TxPd_t *)hwQueueManagerRequestParams.pTailDesc);

		}

		notEmptyDoneLists >>= HW_Q_MANAGER_SHIFT_TO_NEXT_LIST;
		listIndex++;
	}
#if defined (ENET_INC_ARCH_WAVE600)
	EventManager_TurnOnEvent(EVENT_ID_TX_DONE_LISTS);
#else
	EventManager_TurnOnEvent(EVENT_ID_DONE_PDS_PENDING_FOR_TX_MANAGER);
#endif //ENET_INC_ARCH_WAVE600

}


#ifdef TX_DATA_THREAD_PATH
static void txManagerDataListsNotEmpty(K_MSG* txManagerMessage)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint8 NotEmptyDoneList = 0;
	uint8 listIndex = HW_Q_MANAGER_READY_LIST_UNICAST_PD;

    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	NotEmptyDoneList = HwQManager_GetNotEmptyTxReadyHiPriLists();
	DEBUG_ASSERT(NotEmptyDoneList);

	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;

	while(NotEmptyDoneList)
	{
	    if(HW_Q_MANAGER_LIST_BIT_IS_SET_MASK & NotEmptyDoneList)
	    {
			HwQManager_ClearTxListsInt(listIndex);
	    	hwQueueManagerRequestParams.dplIndex = listIndex;
	    	HwQManager_FlushQ(&hwQueueManagerRequestParams);
			ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
			TxPacketsClassifier_SendDataPacket((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
	    }
		NotEmptyDoneList >>= HW_Q_MANAGER_SHIFT_TO_NEXT_LIST;
		listIndex++;
	}
	EventManager_TurnOnEvent(EVENT_ID_TX_LIST_READY_PDS_HIGH_PRI_NOT_EMPTY);
}

static void txManagerStarvationTimer(K_MSG* txManagerMessage)
{
	/*Let other tasks run to prevent starvation*/
	OSAL_CPU_SERVICE_COOPERATION();
    /* Set the timer to prevent other tasks starvation */
	OSAL_SET_TIMER_EXPLICIT(TX_MANAGER_STARVATION_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(TX_MANAGER_STARVATION_TIMER_IN_MILLISECONDS), TASK_TX_MANAGER);	
}
#endif


/**********************************************************************************

txManagerDisableListsNotEmpty 


Description:
------------
handles the event that one of the disable lists became not empty 


Input: 
-----	
txManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerDisableListsNotEmpty(K_MSG* txManagerMessage)
{
#ifdef ENET_INC_ARCH_WAVE600
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
#else
	RequesterLockParams_t lockRequesterParams;	
#endif
	uint16 notEmptyDisableLists = 0;
	uint32 tempBitmap = 0;
	uint8 listIndex = 0;
	UNUSED_PARAM(txManagerMessage);            

	notEmptyDisableLists = HwQManager_GetNotEmptyTxDisableLists();

	tempBitmap = notEmptyDisableLists;

	//("txManagerDisableListsNotEmpty : notEmptyDisableLists=%x",notEmptyDisableLists);
    
	while(tempBitmap)
	{
		listIndex = Utils_FindFirstSetAndClear(&tempBitmap);
		DEBUG_ASSERT(listIndex < 8);
#ifdef TX_MANAGER_DEBUG								
		QDisabledCount[listIndex]++;    
#endif
		HwQManager_ClearDisableListsInt(listIndex);

#ifdef ENET_INC_ARCH_WAVE600

	    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	    hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_DISABLED;
	    hwQueueManagerRequestParams.primaryAddr = listIndex; 
	    
	    HwQManager_FlushQ(&hwQueueManagerRequestParams);
		ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
		ASSERT(hwQueueManagerRequestParams.pTailDesc != NULL_PD);  
	            
		/* Using the same structure instead of having 2 which need to be initialized */
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;	
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
#else
        // W/A for HW bug which clears "Data in Q" bit of GPHP when disable Q is flushed
        /*Send Lock Request to GPHP (assuming the GPHP specific vap Q is according to the list Index)*/
		memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
		/*Request SW Locker to Wait for Lock Stop*/
        lockRequesterParams.callerContext = NULL;
		lockRequesterParams.returnMsg = TX_MANAGER_VAP_GPHP_LOCK;
		lockRequesterParams.returnTask = TASK_TX_MANAGER;
		Locker_LockSingleQueue(HW_TX_Q_TYPE_GPHP, listIndex, 0, &lockRequesterParams);
#endif
	}
#ifdef ENET_INC_ARCH_WAVE600
		EventManager_TurnOnEvent(EVENT_ID_TX_DISABLED_QUEUES_NOT_EMPTY);
#endif
	
}


#ifndef ENET_INC_ARCH_WAVE600
/**********************************************************************************

txManagerFlushDisableList 


Description:
------------
Continues handling the event that one of the disable lists became not empty 


Input: 
-----	
txManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerFlushDisableList(K_MSG* txManagerMessage)
{

    uint8 listIndex;
    HwQueueManagerRequestParams_t hwQueueManagerRequestParams_peek;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    TxSelectorActionParams_t txSelectorActionParams;
#ifdef ENET_INC_ARCH_WAVE600
	uint8 vapIndex;
	StaId stationIndex;
#endif


    LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(txManagerMessage);
    
    //ILOG0_V("Reached txManagerFlushDisableList");

   //Assuming the list Index is equivalent to the Vap Num
    listIndex = lockReg->stationOrVapNum;

    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
    hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_DISABLED;
    hwQueueManagerRequestParams.primaryAddr = listIndex; 
    
    HwQManager_FlushQ(&hwQueueManagerRequestParams);
	ASSERT(hwQueueManagerRequestParams.pHeadDesc != NULL_PD);
	ASSERT(hwQueueManagerRequestParams.pTailDesc != NULL_PD);  

    // Peek at the head PD of GPHP Q 
    memset(&hwQueueManagerRequestParams_peek,0, sizeof(HwQueueManagerRequestParams_t));
    hwQueueManagerRequestParams_peek.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
    hwQueueManagerRequestParams_peek.dplIndex = HW_TX_Q_TYPE_GPHP;
    hwQueueManagerRequestParams_peek.primaryAddr = listIndex;
    hwQueueManagerRequestParams_peek.secondaryAddr = 0;
    hwQueueManagerRequestParams_peek.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
    HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams_peek);
    if (hwQueueManagerRequestParams_peek.pHeadDesc != NULL_PD)
    {
 //      ILOG0_V("GPHP Q is not empty");
       txSelectorActionParams.queueType = HW_TX_Q_TYPE_GPHP;
       txSelectorActionParams.stationOrVapNum = listIndex;
       txSelectorActionParams.action = TX_SELECTOR_SET_DATA_IN_Q;
 
       TxSelector_SetDataInQ(&txSelectorActionParams);
    }

    // Unlock the GPHP queue
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_GPHP, listIndex, (0x1 << 0));
            
	/* Using the same structure instead of having 2 which need to be initialized */
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;

#ifdef ENET_INC_ARCH_WAVE600
	vapIndex = ((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->txQVapId;
	stationIndex = ((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->txQStaId;
	GeneralStatistics.dropReasonDsabled[ConfigurationManager_GetBandForVap(vapIndex)] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
	PerClientStatistics.dropCntReasonDsabled[stationIndex] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
	GeneralStatistics.dropReasonDsabled += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600
	
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);

    EventManager_TurnOnEvent(EVENT_ID_TX_DISABLED_QUEUES_NOT_EMPTY);
   
    //("txManagerDisableListsNotEmpty Un Masked Interrupt : notEmptyDisableLists=%x",notEmptyDisableLists);	
}
#endif

    
   
/***********************************************************************
* txManagerStatisticsPerClientCount
* 
* Description:
* ------------
*   Increment one of the of the following statistics' counter:
* 	PCT_STATISTICS_RETRY_COUNT
*	PCT_STATISTICS_SUCCESS_COUNT
*	PCT_STATISTICS_RETRY_EXHAUSTED_COUNT
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void txManagerStatisticsPerClientCount(TxPd_t *pd)
{
	uint8 pdCounter;

	
	if (pd->aggregationIndication)
	{
		/* AMSDU: accumulate PD Counter */			
		pdCounter = pd->pdCounter;
	}
	else
	{		
		pdCounter = 0x1;
	}		


	/* Check PD Status */
	if (pd->status!= PD_STATUS_ACK_RECEIVED)
	{
		/* PD Status is fail: collect Per client TX Exhausted */
		PerClientStatistics.exhaustedCount[pd->txQStaId] += pdCounter;	
	}
	else 
	{
		/* PD Status is success: check if PD Retransmitted */
		if (pd->retransmissionIndication== 1)
		{
			/* PD Retransmitted: collect Per Client TX Success */
			PerClientStatistics.successCount[pd->txQStaId] += pdCounter;

			
#if defined (ENET_INC_ARCH_WAVE600) & !defined (ENET_INC_ARCH_WAVE600B)			
			/* Check Retry Counter value (wave600B has a HW counter for retries) */
			if (pd->retryCount > 1)
			{				
				/* Retry Counter > 1: collect Per Client TX Retry Success */
				PerClientStatistics.retryCount[pd->txQStaId] += pdCounter;
			}
#endif			
			if (pd->retryCount > 0)
			{
				/* The total number of transmitted packets which were retransmissions for each client on the VAP */
				PerClientStatistics.packetRetransCount[pd->txQStaId] += pdCounter * pd->retryCount;
			}
			if (pd->retryCount > 0)
			{
				/* The number of packets that were successfully transmitted after one or more retransmissions */
				PerClientStatistics.oneOrMoreRetryCount[pd->txQStaId] += pdCounter;
			}
		}
	}	
}

/**********************************************************************************

txManagerHostDataListHandler 


Description:
------------
handles data packets (from host) that were transmitted (or discarded by ager before 
transmission)

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerHostDataListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
#ifdef TX_DATA_CB_UM_PATH
	TxPd_t* pNextPd = NULL;

	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
  
    hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
    hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_LOW_PR_READY_LIST_TEMP_HOST_Q;
    hwQueueManagerRequestParams.pHeadDesc = headPacketDescriptor;
    hwQueueManagerRequestParams.pTailDesc = tailPacketDescriptor;

    while (headPacketDescriptor != (TxPd_t*)NULL_PD)
    {   
        pNextPd = (TxPd_t*)GET_NEXT_PD(headPacketDescriptor);
		if (headPacketDescriptor->mcUnicast == UNICAST)
		{
			txManagerStatisticsPerClientCount(headPacketDescriptor);
		}
        headPacketDescriptor = pNextPd;    
    } 
	
    HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);   
#else
	/*Data PDs are routed directly to Host Interface Accelerator*/
	ASSERT(FALSE);
#endif
}

/**********************************************************************************

txManagerDiscardedPacketsListHandler 


Description:
------------
handles all packets that were discarded

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerDiscardedPacketsListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
	TxPd_t *currentPacketDescriptor = NULL;
	UNUSED_PARAM(tailPacketDescriptor);
	currentPacketDescriptor = headPacketDescriptor;
	while(currentPacketDescriptor != (TxPd_t *)NULL_PD)
	{
		currentPacketDescriptor = TxManagerDiscardedListPdsHandlers[currentPacketDescriptor->pdType](currentPacketDescriptor);
	}
}

/**********************************************************************************

txManagerPackestFromFwListHandler 


Description:
------------
handles packets from FW that were transmitted successfully (or discarded by Ager before 
transmission) 

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerPacketsFromFwListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
	TxPd_t *nextPacketDescriptor = NULL;
	TxPd_t *currentPd = NULL;
	UNUSED_PARAM(tailPacketDescriptor);

	currentPd = headPacketDescriptor;
	while (currentPd != (TxPd_t *)NULL_PD)
	{
		nextPacketDescriptor = (TxPd_t *)GET_NEXT_PD(currentPd);
		
#ifdef ENET_INC_ARCH_WAVE600
		if((!((currentPd->pdType == PD_TYPE_CONTROL_UNENC) && (currentPd->ctrlFrameSubtype == CTRL_FRAME_SUBTYPE_CTS_2_SELF))) && 
		   (!(((currentPd->pdType == PD_TYPE_MANAGEMENT_UNENC) || (currentPd->pdType == PD_TYPE_MANAGEMENT_ENC)) && (currentPd->mcUnicast == MULTICAST) && (currentPd->mgmtFrameSubtype == MGMT_FRAME_SUBTYPE_DEAUTHENTICATION))))
#else 
		if(!((currentPd->pdType == PD_TYPE_CONTROL_UNENC) && (currentPd->ctrlSubtype == PD_CONTROL_SUBTYPE_CTS)))
#endif
		{
			currentPd->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(currentPd->packetPointer);  
		}
		ILOG2_D("txManagerPackestFromFwListHandler packet descriptor %x",currentPd);

		ILOG0_DD("[CSA] txManagerPacketsFromFwListHandler :) pdType = 0x%x mgmtFrameSubtype = 0x%x ", currentPd->pdType, currentPd->mgmtFrameSubtype);

		TxManagerFwListPdsHandlers[currentPd->pdType](currentPd);
		currentPd = nextPacketDescriptor;
    }
}

/**********************************************************************************

txManagerPackestFromDriverListHandler 


Description:
------------
handles packets from driver that were transmitted successfully (or discarded by 
Ager before transmission) 

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerPacketsFromDriverListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
#ifdef TX_MANAGER_DEBUG								
	ManagementFromHost++;
#endif
    hostInterfaceRings_PutOnPdsList(headPacketDescriptor, tailPacketDescriptor,&driverPdsDoneList);
     
	EventManager_TriggerSwEvent(EVENT_ID_DESCRIPTORS_PENDING_FOR_HOST_IF);
}

/**********************************************************************************

txManagerPackestFromDriverListHandler 


Description:
------------
handles multicast packets that were transmitted(or discarded by Ager before transmission) 

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerReliableMulticastListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
	TxPd_t* pNextPd = NULL;
	UNUSED_PARAM(tailPacketDescriptor);

    while (headPacketDescriptor != (TxPd_t*)NULL_PD)
    {   
        pNextPd = (TxPd_t*)GET_NEXT_PD(headPacketDescriptor);
        MulticastHandler_ClonePacketDone(headPacketDescriptor);
		txManagerStatisticsPerClientCount(headPacketDescriptor);
        headPacketDescriptor = pNextPd;     
    }  		
}

/**********************************************************************************

txManagerPsSettingsHwListHandler 


Description:
------------
handles multicast packets that were transmitted(or discarded by Ager before transmission) 

Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerPsSettingsHwListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
	TxPd_t* pNextPd = NULL;
#if defined (ENET_INC_ARCH_WAVE600)
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	memset(&hwQueueManagerRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_TX_POOL_LIST_PS_SETTINGTS_NDP_PD;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
#endif
	UNUSED_PARAM(tailPacketDescriptor);

	while (headPacketDescriptor != (TxPd_t*)NULL_PD)
	{
#if defined	(WORKAROUND_FOR_HW_BUG_IN_TIM_IE)
		if (headPacketDescriptor->txQTid == IEEE802_1D_BE_1)
		{//the SW NDP use tid 0, while the HW use tid 7
			TxPacketsClassifier_SetNdpWasSentInd(headPacketDescriptor->txQStaId, FALSE);
		}
#endif
		pNextPd = (TxPd_t*)GET_NEXT_PD(headPacketDescriptor);
#if defined (ENET_INC_ARCH_WAVE600)
		// In wave600 PS Manager is in LM, so we just push the PD back in here.
		hwQueueManagerRequestParams.pHeadDesc = headPacketDescriptor;
		HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
#else
		PsManager_TransmitNdpCfm(headPacketDescriptor);
#endif
		headPacketDescriptor = pNextPd;     
	}  		
}
#if defined (AGER_STATUS_WORKAROUND)
/**********************************************************************************

txManagerAgerLiberatorListHandler 


Description:
------------
handles packets dropped by Ager before they were transmitted even once
Input: 
-----
headPacketDescriptor -  the head of the PDs list
tailPacketDescriptor -  the tail of the PDs list

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerAgerLiberatorListHandler(TxPd_t *headPacketDescriptor, TxPd_t *tailPacketDescriptor)
{
	TxPd_t* packetDescriptor = headPacketDescriptor;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	
	while (packetDescriptor != (TxPd_t*)NULL_PD)
	{
		packetDescriptor->status = PD_STATUS_DISCARDED_BY_AGER;
		/*Move to next PD - these PDs were discarded by Ager before transmission so they can't be part of an A-MSDU*/
		packetDescriptor = (TxPd_t*)GET_NEXT_PD(packetDescriptor);
	}
	/*Now push PDs to liberator*/	
	memset(&hwQueueManagerRequestParams, 0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.pHeadDesc = headPacketDescriptor;
	hwQueueManagerRequestParams.pTailDesc = tailPacketDescriptor;
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
}
#endif	 

/**********************************************************************************

txManagerAddbaRequestHandler 


Description:
------------
handles a PD of an ADDBA request (discarded or succeeded)

Input: 
-----
packetDescriptor - the pointer of the PD of the ADDBA request	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerAddbaRequestHandler(TxPd_t *packetDescriptor)
{
	K_MSG* tsManagerMsg = NULL; 
	TsManagerActionPacketConfirmedMessage_t *tsManagerMessageParameters = NULL;


	/*Update statistics*/
#ifdef ENET_INC_ARCH_WAVE600
	pBaaCounters->fwMngmntFramesConfirmed[ConfigurationManager_GetBandForVap(packetDescriptor->txQVapId)]++;
#else
	pBaaCounters->fwMngmntFramesConfirmed++;
#endif //ENET_INC_ARCH_WAVE600

	tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerActionPacketConfirmedMessage_t));
	tsManagerMessageParameters = ((TsManagerActionPacketConfirmedMessage_t *)tsManagerMsg->abData);
	tsManagerMessageParameters->packetDescriptor = (TxPd_t *)packetDescriptor;
	
	OSAL_SEND_MESSAGE(TS_MANAGER_ADDBA_REQUEST_CONFIRMED, TASK_TS_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

txManagerCtrlHandler 


Description:
------------
handles a PD of COntrol Unenc - should be CTS only 

Input: 
-----
packetDescriptor - the pointer of the PD of CTS to Self
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerCtrlHandler(TxPd_t *packetDescriptor)
{
	K_MSG* ctrlManagerMsg = NULL; 
	TxManagerCtrlPacketConfirmedMessage_t *ctrlManagerMessageParameters = NULL;

	ctrlManagerMsg = OSAL_GET_MESSAGE(sizeof(TxManagerCtrlPacketConfirmedMessage_t));
	ctrlManagerMessageParameters = ((TxManagerCtrlPacketConfirmedMessage_t *)ctrlManagerMsg->abData);
	ctrlManagerMessageParameters->pd = packetDescriptor;
	
	//check if this is CTS or Trigger frame according to PD subtype	
#ifdef ENET_INC_ARCH_WAVE600
	if(packetDescriptor->ctrlFrameSubtype == CTRL_FRAME_SUBTYPE_CTS_2_SELF)
#else
	if(packetDescriptor->ctrlSubtype == PD_CONTROL_SUBTYPE_CTS)
#endif
	{
		OSAL_SEND_MESSAGE(CTS_MANAGER_TX_CFM, TASK_TX_MANAGER, ctrlManagerMsg, packetDescriptor->txQVapId);
	}
	else
	{
		FATAL("txManagerCtrlHandler");
	}
}

/**********************************************************************************

txManagerCsaHandler 


Description:
------------
handles a PD of an ADDBA request (discarded or succeeded)

Input: 
-----
packetDescriptor - the pointer of the PD of the ADDBA request	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerCsaHandler(TxPd_t *packetDescriptor)
{
	K_MSG* csaManagerMsg = NULL; 
	CsaManagerPacketConfirmedMessage_t *csaManagerMessageParameters = NULL;

	/*Update statistics*/
#ifdef ENET_INC_ARCH_WAVE600
	pBaaCounters->fwMngmntFramesConfirmed[ConfigurationManager_GetBandForVap(packetDescriptor->txQVapId)]++;
#else
	pBaaCounters->fwMngmntFramesConfirmed++;
#endif //ENET_INC_ARCH_WAVE600

	csaManagerMsg = OSAL_GET_MESSAGE(sizeof(CsaManagerPacketConfirmedMessage_t));
	csaManagerMessageParameters = ((CsaManagerPacketConfirmedMessage_t *)csaManagerMsg->abData);
	csaManagerMessageParameters->pd = packetDescriptor;
	
	OSAL_SEND_MESSAGE(CSA_MANAGER_TX_CFM, TASK_TX_MANAGER, csaManagerMsg, VAP_ID_DO_NOT_CARE);
}


/**********************************************************************************

txManagerBarHandler 


Description:
------------
handles a PD of a BAR that was discarded by the Ager without being transmitted at all 
or because of retry limit

Input: 
-----
packetDescriptor - the pointer of the PD of the BAR	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerBarHandler(TxPd_t *packetDescriptor)
{
	K_MSG* tsManagerMsg = NULL; 
	TsManagerBarConfirmedMessage_t *tsManagerMessageParameters = NULL;

	tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerBarConfirmedMessage_t));
	tsManagerMessageParameters = ((TsManagerBarConfirmedMessage_t *)tsManagerMsg->abData);
	tsManagerMessageParameters->packetDescriptor = (TxPd_t *)packetDescriptor;
	
	OSAL_SEND_MESSAGE(TS_MANAGER_BAR_CONFIRMED, TASK_TS_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
}

/**********************************************************************************

txManagerGeneralManagementHandler 


Description:
------------
handles a PD of a general management packet 

Input: 
-----
packetDescriptor - the pointer of the PD of the management packet
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerGeneralManagementHandler(TxPd_t *packetDescriptor)
{
#ifdef ENET_INC_ARCH_WAVE600 
	TxManagerFwManagementHandlers[packetDescriptor->mgmtFrameSubtype](packetDescriptor); 
#else
	TxManagerFwManagementHandlers[packetDescriptor->mgmtSubtype](packetDescriptor); 
#endif
}



/**********************************************************************************

txManagerNdpaHandler 


Description:
------------
handles a PD of a NDPA message control packet 

Input: 
-----
packetDescriptor - the pointer of the PD of the ndpa packet
		
Output:
-------
	

Returns:
--------
	void - 

**********************************************************************************/

#ifndef TRAINING_WAVE600_Z0

static void txManagerNdpaHandler(TxPd_t *packetDescriptor)
{

	K_MSG* tsManagerMsg = NULL; 
	NdpaManagerPacketConfirmed_t *tsManagerMessageParameters = NULL;

	tsManagerMsg = OSAL_GET_MESSAGE(sizeof(NdpaManagerPacketConfirmed_t));
	tsManagerMessageParameters = ((NdpaManagerPacketConfirmed_t *)tsManagerMsg->abData);
	tsManagerMessageParameters->pd = packetDescriptor;
	
	OSAL_SEND_MESSAGE(NDPA_MANAGER_TX_CFM, TASK_GROUP_MANAGER, tsManagerMsg, VAP_ID_DO_NOT_CARE);
}
#endif

/**********************************************************************************

txManagerDiscardedDataPacketsHandler 


Description:
------------
handles discarded data packets 

Input: 
-----
packetDescriptor - the pointer of the first PD
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static TxPd_t* txManagerDiscardedDataPacketsHandler(TxPd_t *packetDescriptor)
{
	uint8 tid = TX_MANAGER_INVALID_TID_NUMBER;
	StaId stationIndex = INVALID_STA_INDEX;
    TxPd_t *headPacketDescriptor = NULL;
	TxPd_t *tailPacketDescriptor = NULL;
	K_MSG* tsManagerMsg = NULL; 
	TsManagerDataPacketDiscardedMessage_t  *tsManagerMessageParameters = NULL;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	uint32 count = 0;
	
    memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));

	headPacketDescriptor = packetDescriptor;	
	tailPacketDescriptor = packetDescriptor;
	
	while(((TxPd_t *)NULL_PD != packetDescriptor) && (PD_TYPE_DATA == packetDescriptor->pdType))
	{
		if((packetDescriptor->txQStaId != stationIndex) || (packetDescriptor->txQTid != tid))
		{
			stationIndex = packetDescriptor->txQStaId;
			tid = packetDescriptor->txQTid;
			
			tsManagerMsg = OSAL_GET_MESSAGE(sizeof(TsManagerDataPacketDiscardedMessage_t));
			tsManagerMessageParameters = ((TsManagerDataPacketDiscardedMessage_t *)tsManagerMsg->abData);
			tsManagerMessageParameters->stationId = stationIndex;
			tsManagerMessageParameters->tid = tid;
			OSAL_SEND_MESSAGE(TS_MANAGER_PACKET_DISCARDED, TASK_TS_MANAGER, tsManagerMsg, packetDescriptor->txQVapId);	
			
		}
		
		/* PD discarded: set status as NACK */
		packetDescriptor->status = PD_STATUS_NACK;
		
		if(packetDescriptor->aggregationIndication)
		{
			count += packetDescriptor->pdCounter;
			/* Take the whole A-MSDU */
			packetDescriptor = (TxPd_t *)CONVERT_OFFSET_TO_PD(packetDescriptor->aMsduTailPointer);
		}
		else
		{
			/* Single MSDU: increment by 1 */
			count += 1;
		}
	
		tailPacketDescriptor = packetDescriptor;
		packetDescriptor = (TxPd_t *)GET_NEXT_PD(packetDescriptor);

	}

	/* Release the PDs by their source */
    hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
    hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
	
#ifdef ENET_INC_ARCH_WAVE600
	GeneralStatistics.dropReasonDiscard[ConfigurationManager_GetBandForStation(stationIndex)] += count;
	PerClientStatistics.dropCntReasonDiscard[stationIndex] += count;

#else
	GeneralStatistics.dropReasonDiscard += count;
#endif //ENET_INC_ARCH_WAVE600
		
    hwQueueManagerRequestParams.pHeadDesc = headPacketDescriptor;
    hwQueueManagerRequestParams.pTailDesc = tailPacketDescriptor;
  
    HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 

	return (packetDescriptor);
}

/**********************************************************************************

txManagerGeneralDiscardedPacketHandler 


Description:
------------
handles a general discarded PD 

Input: 
-----
packetDescriptor - the pointer of the PD of the packet
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static TxPd_t* txManagerGeneralDiscardedPacketHandler(TxPd_t *packetDescriptor)
{
	TxPd_t *nextPacketDescriptor = NULL;

	nextPacketDescriptor = (TxPd_t*)GET_NEXT_PD(packetDescriptor);
	/*Set Status as NACK*/
	packetDescriptor->status = PD_STATUS_NACK;
	/* Cut the list since there are different sources on the discarded list */
	packetDescriptor->nextPdPointer = NEXT_PD_NULL;
	ASSERT(packetDescriptor->pdSource != HW_Q_MANAGER_DONE_LIST_DISCARDED_PACKETS); // PD source can't be "discarded". It means there is a bug. It will lead to infinite loop.

	ILOG0_DD("[CSA] txManagerGeneralDiscardedPacketHandler pdType = 0x%x mgmtFrameSubtype = 0x%x ", packetDescriptor->pdType, packetDescriptor->mgmtFrameSubtype);
	
	TxManagerListsHandlers[packetDescriptor->pdSource](packetDescriptor,packetDescriptor);

	return (nextPacketDescriptor);
}

/**********************************************************************************

txManagerDiscardedAddbaRequestHandler 


Description:
------------
handles a discarded ADDBA request

Input: 
-----
packetDescriptor - the pointer of the PD of the ADDBA request
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static TxPd_t* txManagerDiscardedAddbaRequestHandler(TxPd_t *packetDescriptor)
{
	TxPd_t *nextPacketDescriptor = NULL;

	nextPacketDescriptor = (TxPd_t*)GET_NEXT_PD(packetDescriptor);

	/* No point returning the PD to the lists handlers since the PD type and source is already known here */
	packetDescriptor->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(packetDescriptor->packetPointer);   
	txManagerAddbaRequestHandler(packetDescriptor);

	return (nextPacketDescriptor);
}

/**********************************************************************************

txManagerDiscardedBarHandler 


Description:
------------
handles a discarded BAR

Input: 
-----
packetDescriptor - the pointer of the PD of the BAR
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static TxPd_t* txManagerDiscardedBarHandler(TxPd_t *packetDescriptor)
{
	TxPd_t *nextPacketDescriptor = NULL;

	nextPacketDescriptor = (TxPd_t*)GET_NEXT_PD(packetDescriptor);

	/* No point returning the PD to the lists handlers since the PD type and source is already known here */
	packetDescriptor->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(packetDescriptor->packetPointer);    
	txManagerBarHandler(packetDescriptor);

	return (nextPacketDescriptor);
}

/**********************************************************************************

txManagerDiscardedAddbaRequestHandler 


Description:
------------
handles a discarded ADDBA request

Input: 
-----
packetDescriptor - the pointer of the PD of the ADDBA request
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static TxPd_t* txManagerDiscardedCsaHandler(TxPd_t *packetDescriptor)
{
	TxPd_t *nextPacketDescriptor = NULL;

	nextPacketDescriptor = (TxPd_t*)GET_NEXT_PD(packetDescriptor);

	/* No point returning the PD to the lists handlers since the PD type and source is already known here */
	packetDescriptor->packetPointer = CONVERT_DMA_SHRAM_ADDR_TO_WLAN_SHRAM_ADDR(packetDescriptor->packetPointer);    
	txManagerCsaHandler(packetDescriptor);

	return (nextPacketDescriptor);
}


    
/***********************************************************************
* txManagerDeauthFrameHandler
* 
* Description:
* ------------
* Handle MC DEAUTH frames sent during Channel Switch procedure
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
static void txManagerDeauthFrameHandler(TxPd_t *packetDescriptor)
{
	uint8 vapId = packetDescriptor->txQVapId;

	K_MSG *pTxmCFrameMsg = OSAL_GET_MESSAGE(sizeof(CsaManagerPacketConfirmedMessage_t));
	CsaManagerPacketConfirmedMessage_t* pCsaMngParams = ((CsaManagerPacketConfirmedMessage_t *)pTxmCFrameMsg->abData);

	pCsaMngParams->pd = packetDescriptor;

	ILOG0_V("[CSA] txManagerDeauthFrameHandler 1");

	OSAL_SEND_MESSAGE(CSA_MANAGER_TX_CFM, TASK_TX_MANAGER, pTxmCFrameMsg, vapId);
}


/**********************************************************************************

txManagerActionFrameHandler 


Description:
------------
handles management action frames that were transmitted(or discarded by Ager before 
transmission)  

Input: 
-----
packetDescriptor - the PD of the action frame	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerActionFrameHandler(TxPd_t *packetDescriptor)
{
	uint8 vapId = packetDescriptor->txQVapId;			


	/*Update statistics*/
#ifdef ENET_INC_ARCH_WAVE600
	pBaaCounters->fwMngmntFramesConfirmed[ConfigurationManager_GetBandForVap(vapId)]++;
#else
	pBaaCounters->fwMngmntFramesConfirmed++;
#endif //ENET_INC_ARCH_WAVE600
	
	switch (packetDescriptor->mgmtActionCode)
	{
		case PUBLIC_CATEGORY:
		{
			K_MSG *pTxActionPublicFrameCbMsg = OSAL_GET_MESSAGE(sizeof(dutMsgBtwnCores_t));
			dutMsgBtwnCores_t* pDutparamsBtwnCores = ((dutMsgBtwnCores_t *)pTxActionPublicFrameCbMsg->abData);
			
			pDutparamsBtwnCores->pD = (uint32*)packetDescriptor;
			
			OSAL_SEND_MESSAGE(DUT_ACTION_PACKET_CB_HNDLR_REQ, TASK_DUT,pTxActionPublicFrameCbMsg, vapId);
		}
			break;
			
		case BLOCK_ACK_CATEGORY:
		{
			K_MSG *tsManagerMessage	= OSAL_GET_MESSAGE( sizeof(TsManagerActionPacketConfirmedMessage_t));
			TsManagerActionPacketConfirmedMessage_t *tsManagerMessageParameters = 
				((TsManagerActionPacketConfirmedMessage_t *)tsManagerMessage->abData);
			
			tsManagerMessageParameters->packetDescriptor = (TxPd_t *)packetDescriptor;	
			
			OSAL_SEND_MESSAGE(TS_MANAGER_ACTION_PACKET_CONFIRMED, TASK_TS_MANAGER, tsManagerMessage, tsManagerMessageParameters->packetDescriptor->txQVapId);
		}
			break;

		case HT_CATEGORY:
		{
			K_MSG *smpsManagerMessage	= OSAL_GET_MESSAGE( sizeof(SmpsManagerPacketConfirmedMessage_t));
			SmpsManagerPacketConfirmedMessage_t *smpsManagerMessageParameters = 
				((SmpsManagerPacketConfirmedMessage_t *)smpsManagerMessage->abData);
			
			smpsManagerMessageParameters->pd = packetDescriptor;	
			
			OSAL_SEND_MESSAGE(SMPS_TX_CFM, TASK_TX_MANAGER, smpsManagerMessage, VAP_ID_DO_NOT_CARE);
		}
			break;
		case VHT_CATEGORY:
		{
			if (!(packetDescriptor->mgmtActionValue== VHT_CATEGORY_OPERATING_MODE_NOTIFICATION))
			{
				K_MSG*							gidmManagerMessage = OSAL_GET_MESSAGE(sizeof(GidmManagerPacketConfirmed_t));
				GidmManagerPacketConfirmed_t*	gidmManagerPacketconfirmed = (GidmManagerPacketConfirmed_t *)pK_MSG_DATA(gidmManagerMessage);
				
				gidmManagerPacketconfirmed->pd = packetDescriptor;	
				OSAL_SEND_MESSAGE(GIDM_MANAGER_TX_CFM, TASK_GROUP_MANAGER, gidmManagerMessage, vapId);
			}
			else	// in case it's an OMN message do the same as in HT_CATEGORY
			{
				K_MSG *smpsManagerMessage	= OSAL_GET_MESSAGE( sizeof(SmpsManagerPacketConfirmedMessage_t));
				SmpsManagerPacketConfirmedMessage_t *smpsManagerMessageParameters = ((SmpsManagerPacketConfirmedMessage_t *)smpsManagerMessage->abData);
				
				smpsManagerMessageParameters->pd = packetDescriptor;			
				OSAL_SEND_MESSAGE(SMPS_TX_CFM, TASK_TX_MANAGER, smpsManagerMessage, vapId);
			}				
		}
			break;
#ifdef ENET_INC_ARCH_WAVE600
        case HE_TWT_CATEGORY:
		{
			K_MSG* twtManagerMessage = OSAL_GET_MESSAGE(sizeof(TwtManagerPacketConfirmedMessage_t));
			TwtManagerPacketConfirmedMessage_t *twtManagerMessageParameters = (TwtManagerPacketConfirmedMessage_t *)pK_MSG_DATA(twtManagerMessage);
			
			twtManagerMessageParameters->pd = packetDescriptor;				
			OSAL_SEND_MESSAGE(TWT_MANAGER_TX_CFM, TASK_TWT_MANAGER, twtManagerMessage, twtManagerMessageParameters->pd->txQVapId);
		}
			break;
#endif //ENET_INC_ARCH_WAVE600
		default:
			DEBUG_ASSERT(0);
			break;
	}		
}


/**********************************************************************************

txManagerAssert 


Description:
------------
handles illegal PDs (or lists) that were received in the wrong list

Input: 
-----
packetDescriptor -  the address of the PD	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerAssert(TxPd_t *packetDescriptor)
{
	UNUSED_PARAM(packetDescriptor);	
	FATAL("txManagerAssert");
}

/**********************************************************************************

txManagerListFatal 


Description:
------------
handles illegal lists that were received in the wrong list

Input: 
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerListFatal(TxPd_t *headPd, TxPd_t *tailPd)
{
	UNUSED_PARAM(headPd);	
	UNUSED_PARAM(tailPd);	
	FATAL("txManagerListFatal");
}




/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

TxManager_TaskEntry 


Description:
------------
the entry point of the Tx manager task

Input: 
-----
txManagerMessage - pointer to the message to handle	

	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxManager_TaskEntry(K_MSG *txManagerMessage)
{
	/* Use common task switching and Table */
	vTaskDispatcher(txManagerMessage, afpTaskTable, TASK_TX_MANAGER_START, TASK_TX_MANAGER_END);
}

/**********************************************************************************

txManagerAddVap 


Description:
------------
Utility function that handles the ADD VAP event - enables GPLP queue


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerAddVap(K_MSG* txManagerMessage)
{
	uint8 vapId; 
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	UMI_SET_BSS* pSetBssMsg = (UMI_SET_BSS*) EXTRACT_VAP_MANAGER_MSG(txManagerMessage);
	vapId = pSetBssMsg->vapId;
	
	if (TRUE != ConfigurationManager_IsZwdfsVap(vapId))
	{
		Locker_EnableStaQueues(HW_TX_Q_TYPE_GPLP,vapId);
	}
	
	/*Send confirmation to VAP manager*/
	FILL_VAP_MANAGER_CONFIRM_MSG(txManagerMessage, vapId, VAP_MANAGER_POST_SET_BSS, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, txManagerMessage, vapId);
}

/**********************************************************************************

txManagerPostVapAddition 


Description:
------------
Handles the event of recieiving post VAP addition message

Input: 
-----	
txManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerPostVapAddition(K_MSG* txManagerMessage)
{
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	UMI_ADD_VAP *addVapMessageParameters = (UMI_ADD_VAP *) EXTRACT_VAP_MANAGER_MSG(txManagerMessage);

	if(addVapMessageParameters->operationMode == OPERATION_MODE_SNIFFER)
	{
#ifdef SNIFFER_DEBUG
		ILOG0_V("-------- NTD configured for Sniffer --------");
#endif /* #ifdef SNIFFER_DEBUG */
		
#ifndef ENET_INC_ARCH_WAVE600 
		/* disable autoreply for first RXD line (which is the only one enabled) */
		// In wave600 it is done in LM. PAC Manager registers to post add vap.
		TxHandlerNtd_DisableAutoResponses(0x0);
#endif
	}

	FILL_VAP_MANAGER_CONFIRM_MSG(txManagerMessage, addVapMessageParameters->vapId, VAP_MANAGER_POST_VAP_ADDITION, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, txManagerMessage, addVapMessageParameters->vapId);
}


/**********************************************************************************

TxManager_VapManagerEnableSta 


Description:
------------


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerEnableVap(K_MSG* txManagerMessage)
{
	uint8 vapId; 
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	UMI_SET_WMM_PARAMETERS* pPostVapActivationMsg = (UMI_SET_WMM_PARAMETERS*) EXTRACT_VAP_MANAGER_MSG(txManagerMessage);
	TxSelectorActionParams_t txSelectorActionParams;
	vapId = pPostVapActivationMsg->vapId; 
	
	if (TRUE != ConfigurationManager_IsZwdfsVap(vapId))
	{
		/*Power save initializing*/
		/*VAP active*/
		txSelectorActionParams.queueType = HW_TX_Q_TYPE_VAP_TID; 
		txSelectorActionParams.stationOrVapNum = vapId;
#ifdef ENET_INC_ARCH_WAVE600
		txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
		txSelectorActionParams.tidValueBitmap= TX_SELECTOR_PS_MODE_ACTIVE * TID_BITMAP_ALL_TIDS;	
		TxSelector_ClearPowerSavemode(&txSelectorActionParams);
		/*Clear PS request*/
		txSelectorActionParams.tidValueBitmap= TX_SELECTOR_RESET_POWER_SAVE_REQUEST * TID_BITMAP_ALL_TIDS;
		TxSelector_SetPowerSaveRequest(&txSelectorActionParams);
#else
		txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
		txSelectorActionParams.tidOrAc = 0;
		txSelectorActionParams.action = TX_SELECTOR_PS_MODE_ACTIVE;	
		TxSelector_ClearPowerSavemode(&txSelectorActionParams);	
		/*Clear PS request*/
		txSelectorActionParams.action = TX_SELECTOR_RESET_POWER_SAVE_REQUEST;
		TxSelector_SetPowerSaveRequest(&txSelectorActionParams);
#endif //ENET_INC_ARCH_WAVE600

		// Enable Global Queue for this VAP in Tx Selector
		Locker_EnableStaQueues(HW_TX_Q_TYPE_GLOBAL,vapId);
		Locker_EnableStaQueues(HW_TX_Q_TYPE_GPHP,vapId);
		// Enable MCAST Queue for this VAP in Tx Selector
		Locker_EnableStaQueues(HW_TX_Q_TYPE_VAP_TID,vapId);
		/*Send confirmation to VAP manager*/
	}
	FILL_VAP_MANAGER_CONFIRM_MSG(txManagerMessage, vapId, VAP_MANAGER_POST_VAP_ACTIVATION, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, txManagerMessage, vapId);
}


/**********************************************************************************

txManagerSetWmmParameters 


Description:
------------
handles the event that set WMM parameters message has been recieved from the driver 


Input: 
-----	
txManagerMessage - the message that contains the parameters
	
		
Output:
-------
	

Returns:
--------
	
	
**********************************************************************************/
static void txManagerSetWmmParameters(K_MSG* txManagerMessage)
{	
#ifndef ENET_INC_ARCH_WAVE600

	UMI_SET_WMM_PARAMETERS* setWmmMessageParameters = (UMI_SET_WMM_PARAMETERS*) EXTRACT_VAP_MANAGER_MSG(txManagerMessage);	

	// In wave600 it is done in LM. PAC Manager retisters with this event for this purpose
	TxHandler_UpdateVapTimingParameters(setWmmMessageParameters);


	FILL_VAP_MANAGER_CONFIRM_MSG(txManagerMessage, setWmmMessageParameters->vapId, VAP_MANAGER_SET_WMM_PARAMS, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
	OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, txManagerMessage, setWmmMessageParameters->vapId);
#else 
	UNUSED_PARAM(txManagerMessage);
	DEBUG_FATAL("txManagerSetWmmParameters"); //in wave600 this is done by pacManager 
#endif	
}


/**********************************************************************************

txManagerDisableVap 


Description:
------------
Utility function that enables a STA queues


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void txManagerDisableVap(K_MSG* txManagerMessage)
{
	uint8 vapId; 
	RequesterLockParams_t lockerLockParams;
	//KW_IGNORE ABV.GENERAL .. since abdata is never assigned, but used for typecast. 
	K_MSG* psMsg;
	UMI_STOP_VAP_TRAFFIC* stopTrafficPtr = (UMI_STOP_VAP_TRAFFIC*) EXTRACT_VAP_MANAGER_MSG(txManagerMessage);
	vapId = stopTrafficPtr->vapId; 

	if (TRUE != ConfigurationManager_IsZwdfsVap(vapId))
	{
		lockerLockParams.returnMsg = TX_MANAGER_VAP_QUEUE_DISABLED;
		lockerLockParams.returnTask = TASK_TX_MANAGER;
		
		//Disable Global Queue for this VAP in Tx Selector
		lockerLockParams.callerContext =(void *)HW_TX_Q_TYPE_GLOBAL; /*Use the caller context for the DPL index*/
		Locker_DisableStaQueues(HW_TX_Q_TYPE_GLOBAL ,vapId, &lockerLockParams);
		// Disable MCAST Queue for this VAP in Tx Selector
		lockerLockParams.callerContext = (void *) HW_TX_Q_TYPE_VAP_TID; /*Use the caller context for the DPL index*/
		Locker_DisableStaQueues(HW_TX_Q_TYPE_VAP_TID,vapId, &lockerLockParams);
		// Disable GPHP Queue for this VAP in Tx Selector
		lockerLockParams.callerContext = (void *) HW_TX_Q_TYPE_GPHP; /*Use the caller context for the DPL index*/
		Locker_DisableStaQueues(HW_TX_Q_TYPE_GPHP,vapId, &lockerLockParams);
		// Disable GPLP Queue for this VAP in Tx Selector
		lockerLockParams.callerContext = (void *) HW_TX_Q_TYPE_GPLP; /*Use the caller context for the DPL index*/
		Locker_DisableStaQueues(HW_TX_Q_TYPE_GPLP,vapId, &lockerLockParams);	
	}
	else
	{
		txManagerVapDb[vapId].numOfDisableCallbacks = 0; 
		psMsg = OSAL_GET_MESSAGE(sizeof(BSS_MANAGER_CONFIRM_EVENT)); 
		FILL_VAP_MANAGER_CONFIRM_MSG(psMsg, vapId, VAP_MANAGER_STOP_TRAFFIC, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
		OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
	}
}


/**********************************************************************************

txManagerVapQueueDisabled 


Description:
------------
Returned routine from locker , after locker request for vap queue disable acomplish


Input: 
-----	
K_MSG* lockerMessage

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static void txManagerVapQueueDisabled(K_MSG *lockerMessage)
{
	K_MSG* psMsg; 
	uint8 vapId;
	LockReqCb_t * lockReqCb; 
	uint8 dplIndex;
	uint8 tidIndex = 0;
	uint8 acIndex = 0;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	
	lockReqCb = ((LockReqCb_t *)lockerMessage->abData);
	vapId = lockReqCb->stationOrVapNum;
	dplIndex = (uint32)lockReqCb->callerContext;  /*Caller context was set to DPL index*/

	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = vapId;
	
	switch (dplIndex)
	{
		case HW_TX_Q_TYPE_GLOBAL:
			for (acIndex = 0 ; acIndex < ACCESS_CATEGORY_NUM ; acIndex++)
			{
				/*Flush the queue*/
				hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
				hwQueueManagerRequestParams.dplIndex = dplIndex;		
				hwQueueManagerRequestParams.secondaryAddr = TxManagerActoLowTid[acIndex];
				HwQManager_FlushQ(&hwQueueManagerRequestParams);
				/*Send to Liberator*/
				if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
				{
				  hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
				  hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
				  
				 /*head and tail already include the PDs*/
				 HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
				}
			}
			break;
		case HW_TX_Q_TYPE_VAP_TID:
			for (tidIndex = 0 ; tidIndex < NUM_OF_TID ; tidIndex++)
			{
				/*Flush the queue*/
				hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
				hwQueueManagerRequestParams.dplIndex = dplIndex;		
				hwQueueManagerRequestParams.secondaryAddr = tidIndex;
				HwQManager_FlushQ(&hwQueueManagerRequestParams);
				/*Send to Liberator*/
				if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
				{
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
				  	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
				  	/*head and tail already include the PDs*/
				 	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
				}
			}
#ifdef ENET_INC_ARCH_WAVE600
			/*Flush the managment queue*/
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
			hwQueueManagerRequestParams.dplIndex = dplIndex;							  
			hwQueueManagerRequestParams.secondaryAddr = MANAGEMENT_TID;
			HwQManager_FlushQ(&hwQueueManagerRequestParams);
			/*Send to Liberator*/
			if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
			{
			  hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			  hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
			 /*head and tail already include the PDs*/
			 HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
			}
#endif //ENET_INC_ARCH_WAVE600     
			break;
		case HW_TX_Q_TYPE_GPHP:
		case HW_TX_Q_TYPE_GPLP:			
			/*Flush the queue*/
			hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
			hwQueueManagerRequestParams.dplIndex = dplIndex;		
			hwQueueManagerRequestParams.secondaryAddr = 0;
			HwQManager_FlushQ(&hwQueueManagerRequestParams);
			/*Send to Liberator*/
			if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
			{
			  hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			  hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
			 /*head and tail already include the PDs*/
			 HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
			}
			
			break;			
		default:
			DEBUG_ASSERT(0);
			break;
	}

	txManagerVapDb[vapId].numOfDisableCallbacks++; 
	if(txManagerVapDb[vapId].numOfDisableCallbacks == NUM_OF_VAP_DISABLE_QUEUE)
	{
		txManagerVapDb[vapId].numOfDisableCallbacks = 0; 
		psMsg = OSAL_GET_MESSAGE(sizeof(BSS_MANAGER_CONFIRM_EVENT)); 
		FILL_VAP_MANAGER_CONFIRM_MSG(psMsg, vapId, VAP_MANAGER_STOP_TRAFFIC, BSS_MANAGER_VAP_MANAGER_TX_MANAGER_CLIENT);
		OSAL_SEND_MESSAGE(BSS_MANAGER_VAP_MANAGER_REGISTERED_MODULE_CONFIRM, TASK_BSS_MANAGER, psMsg, vapId);
	}
}

/***********************************************************************
* TxManagerDropMsduOrAmsdu
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void TxManagerDropMsduOrAmsdu(bool aggregation, TxPd_t* headPd, StaId stationIndex, uint8 tid)
{
	HwQueueManagerRequestParams_t		hwQueueManagerRequestParams;
	TxPd_t* 							tailPd = NULL;

	/* Get the tail of the AMSDU */
	if (aggregation == TRUE)
	{
		tailPd = (TxPd_t *)CONVERT_OFFSET_TO_PD(headPd->aMsduTailPointer);
		ASSERT(tailPd != (TxPd_t *)NULL_PD);
	}	

	// Extract the entire AMSDU from the queue
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = stationIndex;
	hwQueueManagerRequestParams.secondaryAddr = tid;
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.pHeadDesc = headPd;
	hwQueueManagerRequestParams.pTailDesc = tailPd;
	HwQManager_PopListFromHead(&hwQueueManagerRequestParams);

	// Send the PDs of the AMSDU to the liberator
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;

	GeneralStatistics.dropReasonAggError[ConfigurationManager_GetBandForStation(stationIndex)] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);

	if (aggregation == TRUE)
	{
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);
	}
	else
	{
		HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
	}	

	// Update the SN in StaDb
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;	
	HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);
	if ((NULL_PD == hwQueueManagerRequestParams.pHeadDesc) || !((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->retransmissionIndication)
	{
		// In case the queue is empty or the first PD wasn't transmitted yet - take the sequence number from the station database
		StaDB_UpdateLowSequenceNumberFromCurrent(stationIndex, tid);
	}
	else
	{
		// Take the sequence number from the PD
		StaDB_UpdateLowSequenceNumber(stationIndex, tid, ((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->sn);
	}
}
#ifdef ENET_INC_ARCH_WAVE600B
/***********************************************************************
* txManagerIsLengthGreaterThanEtsiLimitation
* 
* Description:
* ------------
* This function cheks if the PD length is greater than what can be transmitted under ETSI
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
bool txManagerIsLengthGreaterThanEtsiLimitation(TxPd_t* headPd)
{
	bool greaterThanLimit = FALSE;
	
	if (GroupManager_GetStaState(headPd->txQStaId) == GROUP_MANAGER_STATION_STATE_ACTIVE)
	{
		if (headPd->dataLength > MAX_PD_LENGHT_HE_MU_ETSI)
		{
			greaterThanLimit = TRUE;
		}
	}
	else if (heGroupManagerGetStationState(headPd->txQStaId) == HE_GROUP_MANAGER_STATION_IN_A_GROUP)
	{
		if (headPd->dataLength > MAX_PD_LENGHT_VHT_MU_ETSI)
		{
			greaterThanLimit = TRUE;
		}
	}
	else /* single user */
	{
		if (headPd->dataLength > MAX_PD_LENGHT_SU_ETSI)
		{
			greaterThanLimit = TRUE;
		}
	}
	
	return(greaterThanLimit);
}
#endif


/**********************************************************************************
txManagerRecoverFromAggregatorError 


Description:
------------
Continue the recovery from aggregation builder error, when retransmission AMSDU size is too big
for current chosen rate.

Input: 
-----	
K_MSG* txManagerMessage
		
Output:
-------

Returns:
--------
	void - 
	
**********************************************************************************/
#ifndef ENET_INC_ARCH_WAVE600
static void txManagerRecoverFromAggregatorError(K_MSG* txManagerMessage)
{
	HwQueueManagerRequestParams_t 	hwQueueManagerRequestParams;
	AggBuilderRecoverFromError_t*	pAggBuilderMsg = (AggBuilderRecoverFromError_t*)pK_MSG_DATA(txManagerMessage);
	TxPd_t* 						headPd = NULL;
	TxPd_t* 						tailPd = NULL;
	StaId							stationIndex;
	uint8							tid;
	TxHandlerUnfreezeParams_t 		txHandlerUnfreezeParams;
	uint32 usp;

	//Unfreeze the TxHandler so that HW can continue to work.
	//The problematic queues are still locked so they will not be chosen
	// this is done for wave500 here and for wave600 in LM since TX Handler was moved to LM in wave600
	memset(&txHandlerUnfreezeParams, 0, sizeof(TxHandlerUnfreezeParams_t));
	TxHandler_Unfreeze(&txHandlerUnfreezeParams);

	//iterate over all usp
	for (usp = 0; usp < 4; usp++)
	{
		if (pAggBuilderMsg->uspLock & (TRUE << usp))
		{
			stationIndex = pAggBuilderMsg->stationIndex[usp];
			tid = pAggBuilderMsg->tid[usp];

			if (usp == pAggBuilderMsg->primaryUsp)
			{
				headPd = pAggBuilderMsg->headPd;

				// If A-MSDU need to drop it
				if (headPd->aggregationIndication == TRUE)
				{
					// Get the tail of the AMSDU
					tailPd = (TxPd_t *)CONVERT_OFFSET_TO_PD(headPd->aMsduTailPointer);

					// Extract the entire AMSDU from the queue
					memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
					hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
					hwQueueManagerRequestParams.primaryAddr = stationIndex;
					hwQueueManagerRequestParams.secondaryAddr = tid;
					hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
					hwQueueManagerRequestParams.pHeadDesc = headPd;
					hwQueueManagerRequestParams.pTailDesc = tailPd;
					HwQManager_PopListFromHead(&hwQueueManagerRequestParams);

					// Send the PDs of the AMSDU to the liberator
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
					hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
					
					headPd->status = PD_STATUS_NACK;
#ifdef ENET_INC_ARCH_WAVE600
					GeneralStatistics.dropReasonAggError[ConfigurationManager_GetBandForVap(headPd->txQVapId)] += headPd->pdCounter;
					PerClientStatistics.dropCntReasonAggError[headPd->txQStaId)] += headPd->pdCounter;
#else
					GeneralStatistics.dropReasonAggError += headPd->pdCounter;
#endif //ENET_INC_ARCH_WAVE600

					HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);

					// Update the SN in StaDb
					hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
					hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;	
					HwQManager_PeekHeadPacket(&hwQueueManagerRequestParams);

					if ((NULL_PD == hwQueueManagerRequestParams.pHeadDesc) || !((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->retransmissionIndication)
					{
						// In case the queue is empty or the first PD wasn't transmitted yet - take the sequence number from the station database
						StaDB_UpdateLowSequenceNumberFromCurrent(stationIndex, tid);
					}
					else
					{
						// Take the sequence number from the PD
						StaDB_UpdateLowSequenceNumber(stationIndex, tid, ((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc)->sn);
					}
				}
			}

			// Unlock the queue
			Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, stationIndex, (0x1 << tid));
		}
	}

	if (pAggBuilderMsg->muGroupLock == TRUE)
	{
		DEBUG_ASSERT(pAggBuilderMsg->isMu == TRUE);

		Locker_UnLockGroup(pAggBuilderMsg->muGrpEntry);
	}
}    

/**********************************************************************************

isr_TxManager_TransmissionComplete 


Description:
------------
This function handles interrupt from the HW queue manager when one of the done list
became not empty

Input: 
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID  isr_TxManager_TransmissionComplete(void)
{
	/* Mask all not empty interrupts in HW event */
	EventManager_TurnOffEvent(EVENT_ID_DONE_PDS_PENDING_FOR_TX_MANAGER);

	OSAL_SEND_NO_DATA_MESSAGE(TX_MANAGER_DONE_LIST_NOT_EMPTY, TASK_TX_MANAGER, VAP_ID_DO_NOT_CARE);
}


#else //Wave600
static void txManagerRecoverFromAggregatorError(K_MSG* txManagerMessage)
{	
	OtfAggregatorRecoverFromError_t*	pOtfAggrMsg = (OtfAggregatorRecoverFromError_t*)pK_MSG_DATA(txManagerMessage);
	TxPd_t* 							headPd = NULL;	
	uint8								stationIndex;
	uint8								tid;

	headPd = pOtfAggrMsg->headPd;
	stationIndex = pOtfAggrMsg->stationIndex;
	tid = pOtfAggrMsg->tid;
	// If A-MSDU need to drop it
	if (headPd->aggregationIndication == TRUE)
	{
		TxManagerDropMsduOrAmsdu(headPd->aggregationIndication, headPd, stationIndex, tid);
	}
#ifdef ENET_INC_ARCH_WAVE600B
	else  
	{		
		/* If the PD is ETSI complient Assert. Else drop the PD (WLANRTSYS-14443) */
		if ((pOtfAggrMsg->etsi5GDuringPsduLimitError == TRUE) && (txManagerIsLengthGreaterThanEtsiLimitation(headPd) == TRUE))
		{							
			TxManagerDropMsduOrAmsdu(headPd->aggregationIndication, headPd, stationIndex, tid);
		}
		else
		{
			ASSERT(0);
		}		
	}
#else			
	else
	{
		// We expect that only AMSDUs will get this error. If we get this for single MSDU it might be wrong configuration in plan/phase.
		ASSERT(0);		
		
	}
#endif	
	// Unlock the queue
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, stationIndex, (0x1 << tid));
}
#endif

/**********************************************************************************

isr_TxManager_DisableListNotEmpty 


Description:
------------
This function handles interrupt from the HW queue manager when the disabled list 
is not empty

Input: 
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
ISR_VOID  isr_TxManager_DisableListNotEmpty(void)
{
	
	/* Mask all not empty interrupts in HW event */
	EventManager_TurnOffEvent(EVENT_ID_TX_DISABLED_QUEUES_NOT_EMPTY);
	
	OSAL_SEND_NO_DATA_MESSAGE(TX_MANAGER_DISABLE_LIST_NOT_EMPTY, TASK_TX_MANAGER, VAP_ID_DO_NOT_CARE);
}


/**********************************************************************************
TxManager_Init 


Description:
------------
TX Manager init function

Input: 
-----

		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void TxManager_Init()
{
	uint32 vapIndex; 
	
	for(vapIndex = 0; vapIndex < HW_NUM_OF_VAPS; vapIndex++)
	{
		txManagerVapDb[vapIndex].numOfDisableCallbacks = 0;
	}
	txManager_StaManagerInit();

#ifdef TX_DATA_THREAD_PATH
    /* Set the timer to prevent other tasks starvation */
	OSAL_SET_TIMER_EXPLICIT(TX_MANAGER_STARVATION_TIMER, OSAL_TIMERS_MS_TO_K_TICKS(TX_MANAGER_STARVATION_TIMER_IN_MILLISECONDS), TASK_TX_MANAGER);	
#endif
	
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif

static void TxManagerdutTransmitPacketReq(K_MSG* pMsg)
{
	uint8 *txPayload_p = PNULL;
	FM_PAYLOAD_MANAGEMENT_ACTION_PUBLIC_FRAME *manActionPublicRequestPayload_p = PNULL;
	RmPdRequestFillParameters_t pdRequestFillParameters;
	bool allocationStatus = FALSE;
	TxPd_t *pDesc = PNULL;
	TxPd_t *manPacketDesc_p;// = &(pDesc->manPacketDesc);
	dutTransmittPacketReqParams_t* pTransPacketParams = (dutTransmittPacketReqParams_t*)pK_MSG_DATA(pMsg);
	uint32 length = pTransPacketParams->length;
	uint8 vapId = pMsg->header.vapId;

	memset(&pdRequestFillParameters, 0, sizeof(RmPdRequestFillParameters_t));
	
#ifdef DUT_LOGS_ON
				ILOG0_D("TxManagerdutTransmitPacketReq,vapId to send to DUt = %d",vapId);
#endif
	
	allocationStatus = ResourceManager_GetDescriptorRequest(DESC_POOL_MANAGEMENT_FROM_FW, &pdRequestFillParameters);

	if(allocationStatus)
	{

		/*Store PD pointer*/
		pDesc = pdRequestFillParameters.packetDescriptor;

		manPacketDesc_p = pDesc;
		txPayload_p = (uint8 *)manPacketDesc_p->packetPointer;
		txPayload_p += frame_sizeOfNewManagementFrameHeader(FALSE);
		
		memset(txPayload_p, 0, MANAGEMENT_PACKET_PAYLOAD_SIZE);
		manActionPublicRequestPayload_p = (FM_PAYLOAD_MANAGEMENT_ACTION_PUBLIC_FRAME *)txPayload_p;
	
		// fill payload
		manActionPublicRequestPayload_p->u8CategoryCode = PUBLIC_CATEGORY;
		manActionPublicRequestPayload_p->u8ActionCode = PUBLIC_CATEGORY_VENDOR_SPECIFIC;
		manActionPublicRequestPayload_p->vendorSpecificPayload.au8OUI[0] = MTLK_OUI_0;
		manActionPublicRequestPayload_p->vendorSpecificPayload.au8OUI[1] = MTLK_OUI_1;
		manActionPublicRequestPayload_p->vendorSpecificPayload.au8OUI[2] = MTLK_OUI_2;
	
#ifdef DUT_LOGS_ON	
		ILOG0_V("TxManagerdutTransmitPacketReq: fill payload");
		SLOG0(0, 0, FM_PAYLOAD_MANAGEMENT_ACTION_PUBLIC_FRAME, manActionPublicRequestPayload_p);
#endif
		//length is bigger then 488 bytes. because for DUt is tx upTo 5000bytes and FW dosn't have this size of packets.		
		manPacketDesc_p->dataLength = length + frame_sizeOfNewManagementFrameHeader(FALSE);
		
		manPacketDesc_p->txQStaId= 0;
#ifdef ENET_INC_ARCH_WAVE600 
		manPacketDesc_p->txQTid = MANAGEMENT_TID;
		manPacketDesc_p->mgmtFrameSubtype = MGMT_FRAME_SUBTYPE_ACTION;
#else
		manPacketDesc_p->txQTid = IEEE802_1D_VO_2;
		manPacketDesc_p->mgmtSubtype = MGMT_FRAME_SUBTYPE_ACTION;
#endif
		manPacketDesc_p->mgmtActionCode = PUBLIC_CATEGORY;
		manPacketDesc_p->mgmtActionValue = PUBLIC_CATEGORY_VENDOR_SPECIFIC;
		manPacketDesc_p->pdType = PD_TYPE_MANAGEMENT_UNENC;
		manPacketDesc_p->txQVapId = vapId;
	
		manPacketDesc_p->txQGroupId = HW_TX_Q_TYPE_VAP_TID;
		manPacketDesc_p->mcUnicast = MULTICAST;
	
#ifdef DUT_LOGS_ON	
		ILOG0_V("TxManagerdutTransmitPacketReq: Send Packet to Classifier!!!!!");
		SLOG0(0, 0, TxPd_t, manPacketDesc_p);
#endif
	
		TxPacketsClassifier_SendManagementPacketFromFw(manPacketDesc_p);

	}
	else
	{
		/* PD allocation failed */	

		ILOG0_V("---- dutTransmitPacketReq, no free desc");
	}

	/*Send confirmation to DUT*/
	pTransPacketParams->allocationStatus = allocationStatus;
	OSAL_SEND_MESSAGE(pTransPacketParams->retMsg,pTransPacketParams->retTask,pMsg,vapId);
}

static void TxManagerCBdutTransmitPacketCfm(K_MSG* pMsg)
{
		TxPd_t *pDesc = PNULL;		
		dutMsgBtwnCores_t* pTransPacketCfmParam = (dutMsgBtwnCores_t*)pK_MSG_DATA(pMsg);

#ifdef DUT_LOGS_ON	
				ILOG0_V("TxManagerCBdutTransmitPacketCfm: Get CallBack packet from TxMngrDUT_ACTION_PACKET_CB_HNDLR_CFM!!!!!");
#endif		

		pDesc = (TxPd_t *)pTransPacketCfmParam->pD;
		if(pTransPacketCfmParam->toReleasePool == FALSE)
		{
		
#ifdef DUT_LOGS_ON	
			ILOG0_V("Send again the packet TxPacketsClassifier_SendManagementPacketFromFw");
#endif		
		
			TxPacketsClassifier_SendManagementPacketFromFw(pDesc);
		}
		else
		{
		
#ifdef DUT_LOGS_ON	
					ILOG0_V("RELEASE PACKET$$$$$ TxManagerCBdutTransmitPacketCfm");
#endif		
			ResourceManager_ReleaseDescriptor(pDesc, DESC_POOL_MANAGEMENT_FROM_FW);
		}

}



