/***********************************************************************************
 File:			TxManager_StaManager.c
 Module:		Tx manager STA Manager
 Purpose: 		handle STA Manager events
 Description:   
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
	//TBD to check which files can be removed
#include "BSSmanager_API.h"
#include "TxManager_Api.h"
#include "TxManager.h"
#include "TxMulticastHandler_API.h"
#include "TsManager_API.h"
#include "TxSelector_Api.h"
#include "HostInterface_API.h"
#include "Utils_Api.h"
#include "ResourceManager_API.h"
#include "StaDatabase_Api.h"
#include "Locker_Api.h"
#include "OSAL_Kmsg.h"
#include "ErrorHandler_Api.h"
#include "HwQManager_API.h"
#include "HwGlobalDefinitions.h"
#include "ShramPacketDescriptors.h"
#include "stringLibApi.h"
#include "um_interface.h"
#include "PacketDescriptor.h"
#include "HwMemoryMap.h"
#include "TxManager_StaManager.h"
#include "loggerAPI.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID GLOBAL_GID_TX_MANAGER 
#define LOG_LOCAL_FID 1


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/

	
/*TX Manager STA Manager states*/
typedef enum
{
	TX_MANAGER_STA_MANAGER_CLOSED,
	TX_MANAGER_STA_MANAGER_OPEN,
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
	TX_MANAGER_STA_MANAGER_STOP_TRAFFIC_WAIT_TID_7,
#endif	
	TX_MANAGER_STA_MANAGER_STOP_TRAFFIC_WAIT_DISABLE,
	TX_MANAGER_STA_MANAGER_WAIT_POST_STOP_TRAFFIC,
	TX_MANAGER_STA_MANAGER_WAIT_CLOSING,
	TX_MANAGER_STA_MANAGER_CLOSING_WAIT_DISABLE,
	TX_MANAGER_STA_MANAGER_NUM_STATES
} txManagerStaManager_state;

typedef struct TxManagerStaDb_s
{
	uint8     				  flags;
	StaId					  sid;
	uint8					  uapsd_AC_Bitmap; // only 4 bits are needed
	txManagerStaManager_state state:4;
	uint8					  vapId:4;
} TxManagerStaDb_t;


typedef void (*TxManager_StaManagerFunc)(TxManagerStaDb_t *staDb);

/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void TxManager_StaManagerOpenSta(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerStopTraffic(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerPostStopTraffic(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerClose(TxManagerStaDb_t *StaDb);
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)	
static void TxManager_StaManagerStopTrafficLockTid7(TxManagerStaDb_t *StaDb);
#endif
static void TxManager_StaManagerStopTrafficDisable(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerClosingDisable(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerFatal(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerEnableSta(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerFlushSta(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerSendConfirm(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerClearPowerSaveMode(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerClearPowerSaveRequest(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerSetPowerSaveType(TxManagerStaDb_t *StaDb);
static void TxManager_StaManagerChangeState(txManagerStaManager_state state, TxManagerStaDb_t *StaDb);

/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/
/*STA Open Event Handlers*/
static const TxManager_StaManagerFunc TxManager_StaManagerStaOpen[TX_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 TxManager_StaManagerOpenSta, 				  /* TX_MANAGER_STA_CLOSED */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_OPEN */
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		 
 	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_TID_7 */
#endif 	 
	 TxManager_StaManagerFatal,		  			  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_DISABLE */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_POST_STOP_TRAFFIC */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_CLOSING */
	 TxManager_StaManagerFatal  				  /* TX_MANAGER_STA_CLOSING_WAIT_DISABLE */
};

/*STA Stop Traffic Event Handlers*/
static const TxManager_StaManagerFunc TxManager_StaManagerStaStopTraffic[TX_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 TxManager_StaManagerFatal, 				  /* TX_MANAGER_STA_CLOSED */
	 TxManager_StaManagerStopTraffic,			  /* TX_MANAGER_STA_OPEN */
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		 
 	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_TID_7 */
#endif 	 
	 TxManager_StaManagerFatal,		  			  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_DISABLE */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_POST_STOP_TRAFFIC */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_CLOSING */
	 TxManager_StaManagerFatal  				  /* TX_MANAGER_STA_CLOSING_WAIT_DISABLE */
};

/*STA Post Stop Traffic Event Handlers*/
static const TxManager_StaManagerFunc TxManager_StaManagerStaPostStopTraffic[TX_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 TxManager_StaManagerFatal, 				  /* TX_MANAGER_STA_CLOSED */
	 TxManager_StaManagerFatal,			  		  /* TX_MANAGER_STA_OPEN */
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		 
 	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_TID_7 */
#endif 	 
	 TxManager_StaManagerFatal, 	  			  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_DISABLE */
	 TxManager_StaManagerPostStopTraffic,		  /* TX_MANAGER_STA_WAIT_POST_STOP_TRAFFIC */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_CLOSING */
	 TxManager_StaManagerFatal  				  /* TX_MANAGER_STA_CLOSING_WAIT_DISABLE */
};

/*STA Close Event Handlers*/
static const TxManager_StaManagerFunc TxManager_StaManagerStaClose[TX_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 TxManager_StaManagerFatal, 				  /* TX_MANAGER_STA_CLOSED */
	 TxManager_StaManagerFatal,			  		  /* TX_MANAGER_STA_OPEN */
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		 
 	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_TID_7 */
#endif 	 
	 TxManager_StaManagerFatal,		  			  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_DISABLE */
	 TxManager_StaManagerFatal,		  			  /* TX_MANAGER_STA_WAIT_POST_STOP_TRAFFIC */
	 TxManager_StaManagerClose,					  /* TX_MANAGER_STA_WAIT_CLOSING */
	 TxManager_StaManagerFatal  			      /* TX_MANAGER_STA_CLOSING_WAIT_DISABLE */
};

/*STA Locker Event Handlers*/
static const TxManager_StaManagerFunc TxManager_StaManagerStaLock[TX_MANAGER_STA_MANAGER_NUM_STATES] =
{
	 TxManager_StaManagerFatal, 				  /* TX_MANAGER_STA_CLOSED */
	 TxManager_StaManagerFatal,			  		  /* TX_MANAGER_STA_OPEN */
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		 
 	 TxManager_StaManagerStopTrafficLockTid7,	  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_TID_7 */
#endif 	 
	 TxManager_StaManagerStopTrafficDisable, 	  /* TX_MANAGER_STA_STOP_TRAFFIC_WAIT_DISABLE */
	 TxManager_StaManagerFatal,		  			  /* TX_MANAGER_STA_WAIT_POST_STOP_TRAFFIC */
	 TxManager_StaManagerFatal,					  /* TX_MANAGER_STA_WAIT_CLOSING */
	 TxManager_StaManagerClosingDisable 		  /* TX_MANAGER_STA_CLOSING_WAIT_DISABLE */
};

static TxManagerStaDb_t TxManager_StaManagerDb[HW_NUM_OF_STATIONS];
extern PerClientStatistics_t PerClientStatistics;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

TxManager_StaManagerSendConfirm 


Description:
------------
Send confirmation to STA Manager


Input: 
-----	
txManagerStaManager_ev - event
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerSendConfirm(TxManagerStaDb_t *StaDb)
{
	K_MSG *pMsg;
	BssManagerStaManagerCfm_t *confirmMessage;

	/*Allocate message*/
	pMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(pMsg);
	/*Set Client ID to the registered ID*/
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_TX_CLIENT;
	/*Set STA ID*/
	confirmMessage->sid = StaDb->sid;
	/*Send confirmation message*/
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, pMsg, StaDb->vapId);
}

/**********************************************************************************

TxManager_StaManagerEnableSta 


Description:
------------
Utility function that enables a STA queues


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerEnableSta(TxManagerStaDb_t *StaDb)
{
	/*Enable STA queues*/
	Locker_EnableStaQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid);
}

/**********************************************************************************

TxManager_StaManagerFlushSta 


Description:
------------
Utility function that flushes a STA queues


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerFlushSta(TxManagerStaDb_t *StaDb)
{
	uint8 i;
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;

	/*Flush STA*/
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	for (i = 0; i < NUM_TID; i++)
	{
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
		hwQueueManagerRequestParams.secondaryAddr = i;
		HwQManager_FlushQ(&hwQueueManagerRequestParams);
		/*Send to Liberator*/
		if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
		{
    		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
			hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		
#ifdef ENET_INC_ARCH_WAVE600
			GeneralStatistics.dropReasonDisconnect[ConfigurationManager_GetBandForStation(StaDb->sid)] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
			PerClientStatistics.dropCntReasonDisconnect[StaDb->sid] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
			GeneralStatistics.dropReasonDisconnect += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600
			
			/*head and tail already include the PDs*/
    		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
			/*We flushed the queue - make sure that Low SSN is set Current SSN - in case some packets gets into the
			queue after we enable the queue in the post stop traffic handling*/
			StaDB_UpdateLowSequenceNumberFromCurrent(StaDb->sid, i);
		}
	}
#ifdef ENET_INC_ARCH_WAVE600
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.secondaryAddr = MANAGEMENT_TID;
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	/*Send to Liberator*/
	if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
	{
		hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
		hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;
		/*head and tail already include the PDs*/
		HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
		/*We flushed the queue - make sure that Low SSN is set Current SSN - in case some packets gets into the
		queue after we enable the queue in the post stop traffic handling*/
		StaDB_UpdateLowSequenceNumberFromCurrentMngTid(StaDb->sid);
	}
#endif

}

/**********************************************************************************

TxManager_StaManagerClearPowerSaveMode 


Description:
------------
Utility function that Sets the STA to active


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerClearPowerSaveMode(TxManagerStaDb_t *StaDb)
{
	TxSelectorActionParams_t txSelectorActionParams;

	/*Set STA Power Save mode to active - this is needed to clear VAP PS mode if this was the only STA in PS -
	setting the STA to disassociate does not do it  ...*/
	txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID; 
	txSelectorActionParams.stationOrVapNum = StaDb->sid;
#ifdef ENET_INC_ARCH_WAVE600
	txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	txSelectorActionParams.tidValueBitmap= TX_SELECTOR_PS_MODE_ACTIVE * TID_BITMAP_ALL_TIDS;
#else
	txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
	txSelectorActionParams.tidOrAc = 0;
	txSelectorActionParams.action = TX_SELECTOR_PS_MODE_ACTIVE;
#endif //ENET_INC_ARCH_WAVE600

	TxSelector_ClearPowerSavemode(&txSelectorActionParams);	
}

/**********************************************************************************

TxManager_StaManagerClearPowerSaveRequest 


Description:
------------
Utility function that Clears the STA PS request


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerClearPowerSaveRequest(TxManagerStaDb_t *StaDb)
{
	TxSelectorActionParams_t txSelectorActionParams;

	/*Clear PS request*/
	txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID; 
	txSelectorActionParams.stationOrVapNum = StaDb->sid;
#ifdef ENET_INC_ARCH_WAVE600
	txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	txSelectorActionParams.tidValueBitmap = TX_SELECTOR_RESET_POWER_SAVE_REQUEST * TID_BITMAP_ALL_TIDS;
#else
	txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
	txSelectorActionParams.tidOrAc = 0;
	txSelectorActionParams.action = TX_SELECTOR_RESET_POWER_SAVE_REQUEST;
#endif //ENET_INC_ARCH_WAVE600

	TxSelector_SetPowerSaveRequest(&txSelectorActionParams);
}





/**********************************************************************************

TxManager_StaManagerSetPowerSaveType 


Description:
------------
Utility function that Sets the STA PS type


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerSetPowerSaveType(TxManagerStaDb_t *StaDb)
{
	uint8 acIndex;
	TxSelectorActionParams_t txSelectorActionParams;

	/*Set power save mode in Tx Selector*/
	txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID; 
	txSelectorActionParams.stationOrVapNum = StaDb->sid;
	if (MTLK_BFIELD_GET(StaDb->flags, STA_ADD_FLAGS_WME) == TRUE)
	{
		/*	Qos Info filed: 	|	   7		|		65				|	4		| 3   2   1   0  |
							|reserved	|	max SP length			| reserved	|BE  BK  VI VO| */		
		if (StaDb->uapsd_AC_Bitmap != 0)
		{
			/*Some TIDs are WMM - set one AC at a time*/
#ifndef ENET_INC_ARCH_WAVE600
			txSelectorActionParams.allTids = TX_SELECTOR_SINGLE_TID_UPDATE;
#endif //ENET_INC_ARCH_WAVE600
			for (acIndex = 0 ; acIndex < ACCESS_CATEGORY_NUM ; acIndex++)
			{
#ifdef ENET_INC_ARCH_WAVE600
				txSelectorActionParams.acValue = acIndex;
#else
				txSelectorActionParams.tidOrAc = acIndex;
#endif //ENET_INC_ARCH_WAVE600
				if (StaDb->uapsd_AC_Bitmap & (1 << ((ACCESS_CATEGORY_NUM -1) - acIndex)))
				{
					txSelectorActionParams.action = TX_SELECTOR_PS_TYPE_WMM;
				}
				else
				{
					txSelectorActionParams.action = TX_SELECTOR_PS_TYPE_LEGACY;
				}
				TxSelector_SetPowerSaveType(&txSelectorActionParams);
			}
		}
		else
		{
			/*WMM not supported - set all TIDs to Legacy*/
#ifdef ENET_INC_ARCH_WAVE600
			txSelectorActionParams.acValue= ACCESS_CATEGORY_INAVLID; //invalid value used to signal that this request is for all TIDs
#else
			txSelectorActionParams.allTids = TX_SELECTOR_ALL_TIDS_UPDATE;
			txSelectorActionParams.tidOrAc = 0;
#endif //ENET_INC_ARCH_WAVE600
			txSelectorActionParams.action = TX_SELECTOR_PS_TYPE_LEGACY;
			TxSelector_SetPowerSaveType(&txSelectorActionParams);
		}
	}
	else
	{
		/*if STA is non-Qos everything will go thru TID 0*/
#ifdef ENET_INC_ARCH_WAVE600
		txSelectorActionParams.acValue = ACCESS_CATEGORY_BE;
#else
		txSelectorActionParams.allTids = TX_SELECTOR_SINGLE_TID_UPDATE;
		txSelectorActionParams.tidOrAc = ACCESS_CATEGORY_BE;
#endif //ENET_INC_ARCH_WAVE600
		txSelectorActionParams.action = TX_SELECTOR_PS_TYPE_LEGACY;
		TxSelector_SetPowerSaveType(&txSelectorActionParams);
	}
}



/**********************************************************************************

TxManager_StaManagerChangeState 


Description:
------------
Change STA state


Input: 
-----	
txManagerStaManager_state - event
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerChangeState(txManagerStaManager_state state, TxManagerStaDb_t *StaDb)
{
	ILOG0_DDD("TX Manager STA Manager, Station %d changed from %d to %d", StaDb->sid, StaDb->state, state);
	StaDb->state = state;
}

/**********************************************************************************

TxManager_StaManagerOpenSta 


Description:
------------
handles STA Open event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerOpenSta(TxManagerStaDb_t *StaDb)
{
	/*Power save initializing*/
	/*STA active*/
	TxManager_StaManagerClearPowerSaveMode(StaDb);
	/*Clear PS request*/
	TxManager_StaManagerClearPowerSaveRequest(StaDb);
	/*Set PS type*/
	TxManager_StaManagerSetPowerSaveType(StaDb);
	/*Enable STA queues*/
	TxManager_StaManagerEnableSta(StaDb);
	/*Change STA State to Open*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_OPEN, StaDb);
	/*Send open STA Confirmation to STA Manager*/
	TxManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

TxManager_StaManagerStopTraffic 


Description:
------------
handles STA Stop Traffic event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerStopTraffic(TxManagerStaDb_t *StaDb)
{
	RequesterLockParams_t lockRequesterParams;

	memset(&lockRequesterParams, 0, sizeof(RequesterLockParams_t));
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
	/*Change STA State to Wait Lock TID7*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_STOP_TRAFFIC_WAIT_TID_7, StaDb);
	/*Request SW Locker to lock TID 7*/
	lockRequesterParams.callerContext = (void *)StaDb;
	lockRequesterParams.returnMsg = TX_MANAGER_STA_LOCK_EV;
	lockRequesterParams.returnTask = TASK_TX_MANAGER;
	Locker_LockSingleQueue(HW_TX_Q_TYPE_STA_TID, StaDb->sid, IEEE802_1D_VO_2, &lockRequesterParams);
#else
	/*Change State to Stop Traffic Wait Disable*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_STOP_TRAFFIC_WAIT_DISABLE, StaDb);
	/*Request SW Locker to disable STA*/
	lockRequesterParams.callerContext = (void *)StaDb;
	lockRequesterParams.returnMsg = TX_MANAGER_STA_LOCK_EV;
	lockRequesterParams.returnTask = TASK_TX_MANAGER;
	Locker_DisableStaQueues(HW_TX_Q_TYPE_STA_TID ,StaDb->sid, &lockRequesterParams);
#endif
}

/**********************************************************************************

TxManager_StaManagerPostStopTraffic 


Description:
------------
handles STA Post Stop Traffic event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerPostStopTraffic(TxManagerStaDb_t *StaDb)
{
	/*Enable STA*/
	TxManager_StaManagerEnableSta(StaDb);
	/*Change STA State to Wait Closing*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_WAIT_CLOSING, StaDb);
	/*Send Post Stop Traffic Confirmation to STA Manager*/
	TxManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

TxManager_StaManagerClose 


Description:
------------
handles STA Close event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerClose(TxManagerStaDb_t *StaDb)
{
	RequesterLockParams_t lockerLockParams;

	memset(&lockerLockParams, 0, sizeof(RequesterLockParams_t));
	/*Change STA State to Closing Wait Disable*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_CLOSING_WAIT_DISABLE, StaDb);
	/*Request SW Locker to disable STA*/
	lockerLockParams.callerContext = (void *)StaDb;
	lockerLockParams.returnMsg = TX_MANAGER_STA_LOCK_EV;
	lockerLockParams.returnTask = TASK_TX_MANAGER;
	Locker_DisableStaQueues(HW_TX_Q_TYPE_STA_TID ,StaDb->sid, &lockerLockParams);
}

/**********************************************************************************

TxManager_StaManagerStopTrafficLockTid7 


Description:
------------
handles Locker event for TID 7 during Stop Traffic 


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if defined (SENDER_LOCK_DATA_TID_ON_ADDBA_REQ)		
static void TxManager_StaManagerStopTrafficLockTid7(TxManagerStaDb_t *StaDb)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	RequesterLockParams_t lockerLockParams;

	memset(&lockerLockParams, 0, sizeof(RequesterLockParams_t));
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	/*Flush TID 7*/
	hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_STA_TID;
	hwQueueManagerRequestParams.primaryAddr = StaDb->sid;
	hwQueueManagerRequestParams.secondaryAddr = IEEE802_1D_VO_2;
	HwQManager_FlushQ(&hwQueueManagerRequestParams);
	/*Send to Liberator*/
	if (hwQueueManagerRequestParams.pHeadDesc != NULL_PD)
	{
    	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	    hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_DONE_LIST_LIBERATOR;

#ifdef ENET_INC_ARCH_WAVE600
		GeneralStatistics.dropReasonDisconnect[ConfigurationManager_GetBandForStation(StaDb->sid)] += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#else
		GeneralStatistics.dropReasonDisconnect += StatisticsSetPdStatusNackInList((TxPd_t *)hwQueueManagerRequestParams.pHeadDesc);
#endif //ENET_INC_ARCH_WAVE600
		
		/*head and tail already include the PDs*/
	    HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams); 
		/*We flushed the queue - make sure that Low SSN is set Current SSN - in case some packets gets into the
		queue after we unlock the queue*/
		StaDB_UpdateLowSequenceNumberFromCurrent(StaDb->sid, IEEE802_1D_VO_2);
	}
	/*Unlock TID 7*/
	Locker_UnLockPerTidQueues(HW_TX_Q_TYPE_STA_TID, StaDb->sid, (0x1 << IEEE802_1D_VO_2));

	/*Change State to Stop Traffic Wait Disable*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_STOP_TRAFFIC_WAIT_DISABLE, StaDb);
	/*Request SW Locker to disable STA*/
	lockerLockParams.callerContext = (void *)StaDb;
	lockerLockParams.returnMsg = TX_MANAGER_STA_LOCK_EV;
	lockerLockParams.returnTask = TASK_TX_MANAGER;
	Locker_DisableStaQueues(HW_TX_Q_TYPE_STA_TID ,StaDb->sid, &lockerLockParams);
}
#endif

/**********************************************************************************

TxManager_StaManagerStopTrafficDisable 


Description:
------------
handles Locker event for STA Disable during Stop Traffic 


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerStopTrafficDisable(TxManagerStaDb_t *StaDb)
{
	/*Flush STA queues*/
	TxManager_StaManagerFlushSta(StaDb);
	/*Change State to Wait Post Stop Traffic*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_WAIT_POST_STOP_TRAFFIC, StaDb);
	/*Send Stop Traffic Confirmation to STA Manager*/
	TxManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

TxManager_StaManagerClosingDisable 


Description:
------------
handles Locker event for STA Disable during Closing


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerClosingDisable(TxManagerStaDb_t *StaDb)
{
	/*Flush STA queues*/
	TxManager_StaManagerFlushSta(StaDb);
	/*Set STA Power Save mode to active - this is needed to clear VAP PS mode if this was the only STA in PS -
	setting the STA to disassociate does not do it  ...*/
	TxManager_StaManagerClearPowerSaveMode(StaDb);
	/*Change STA State to Closed*/
	TxManager_StaManagerChangeState(TX_MANAGER_STA_MANAGER_CLOSED, StaDb);
	/*Send Close Confirmation to STA Manager*/
	TxManager_StaManagerSendConfirm(StaDb);
}

/**********************************************************************************

TxManager_StaManagerFatal 


Description:
------------
handles unexpected event


Input: 
-----	
TxManagerStaDb_t - STA DB entry
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void TxManager_StaManagerFatal(TxManagerStaDb_t *StaDb)
{
	UNUSED_PARAM(StaDb);	
	FATAL("TX Manager STA Manager Fatal");
}

/*---------------------------------------------------------------------------------
/						global Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

txManagerStaOpen 


Description:
------------
STA ADD message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void txManagerStaOpen(K_MSG* txManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(txManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(psMsg);
	StaId sid = pAddSta->u16SID;
	TxManagerStaDb_t *staDb;
		
	/*From ADD STA message extract SID*/
	staDb = &TxManager_StaManagerDb[sid];
	/*Store VAP ID*/
	staDb->vapId = pAddSta->u8VapIndex;
	/*Store Flags*/
	staDb->flags = pAddSta->u8Flags;
	/*Store UAPSD queues*/
	staDb->uapsd_AC_Bitmap = pAddSta->u8UAPSD_Queues;
	/*Call STA open handler*/
	TxManager_StaManagerStaOpen[staDb->state](staDb);
}

/**********************************************************************************

txManagerStaStopTraffic 


Description:
------------
STA Stop traffic message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void txManagerStaStopTraffic(K_MSG* txManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(txManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(psMsg);
	StaId sid = pStopTraffic->u16SID;
	TxManagerStaDb_t *staDb;
		
	/*From Stop Traffic message extract SID*/
	staDb = &TxManager_StaManagerDb[sid];
	/*Call STA Stop Traffic handler*/
	TxManager_StaManagerStaStopTraffic[staDb->state](staDb);
}

/**********************************************************************************

txManagerStaPostStopTraffic 


Description:
------------
STA Post Stop traffic message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void txManagerStaPostStopTraffic(K_MSG* txManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(txManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC *pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(psMsg);
	StaId sid = pStopTraffic->u16SID;
	TxManagerStaDb_t *staDb;
		
	/*From Stop Traffic message extract SID*/
	staDb = &TxManager_StaManagerDb[sid];
	/*Call STA Post Stop Traffic handler*/
	TxManager_StaManagerStaPostStopTraffic[staDb->state](staDb);
}

/**********************************************************************************

txManagerStaClose 


Description:
------------
STA Close message handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void txManagerStaClose(K_MSG* txManagerMessage)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(txManagerMessage);
	K_MSG *psMsg = staManagerReq->psMsg;
	UMI_STA_REMOVE *pRemoveSta = (UMI_STA_REMOVE *)pK_MSG_DATA(psMsg);
	StaId sid = pRemoveSta->u16SID;
	TxManagerStaDb_t *staDb;
		
	/*From STA Close message extract SID*/
	staDb = &TxManager_StaManagerDb[sid];
	/*Call STA Close handler*/
	TxManager_StaManagerStaClose[staDb->state](staDb);
}

/**********************************************************************************

txManagerStaLockEv 


Description:
------------
TX Manager STA Manager Locker messages handler


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void txManagerStaLockEv(K_MSG* txManagerMessage)
{
	LockReqCb_t *lockReg = (LockReqCb_t *)pK_MSG_DATA(txManagerMessage);
	TxManagerStaDb_t *staDb = (TxManagerStaDb_t *)lockReg->callerContext;
		
	/*Locker message extract TX Manager STA Manager databse entry*/
	/*Call locker event handler*/
	TxManager_StaManagerStaLock[staDb->state](staDb);
}


/**********************************************************************************

txManager_StaManagerInit


Description:
------------
Initiailize TX MAnager STA Manager


Input: 
-----	
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void txManager_StaManagerInit()
{
	StaId i;

	/*Initialize DB*/
	for (i = 0; i < HW_NUM_OF_STATIONS; i++)
	{
		TxManager_StaManagerDb[i].sid = i;
		TxManager_StaManagerDb[i].state = TX_MANAGER_STA_MANAGER_CLOSED;
	}
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif




