/***********************************************************************************
 File:			TxMulticastHandler.c
 Module:		Tx
 Purpose:		
 Description:	
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "TxMulticastHandler_API.h"
#include "TxMulticastHandler.h"
#include "ResourceManager_API.h"
#include "stringLibApi.h"
#include "Utils_Api.h"
#include "ErrorHandler_Api.h"
#include "BSSmanager_API.h"
#include "HwQManager_API.h"
#include "StaDatabase_Api.h"
#include "TxPacketsClassifier_API.h"
#include "ShramPacketDescriptors.h"
#include "HwGlobalDefinitions.h"
#include "loggerAPI.h"
#include "StatisticsManager_api.h"
#include "ShramStatistics.h"
#ifdef ENET_INC_ARCH_WAVE600 
#include "HostInterface_API.h"
#endif
#include "ConfigurationManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX
#define LOG_LOCAL_FID 0


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

McastHandlerDb_t MulticastHandlerDb; // Database of Multicast Handler
MulticastStatistics_t MulticastHandlerStatistics;


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
uint32* MulticastHandler_GetStatistics_Address()
{
    return ((uint32*)(&MulticastHandlerStatistics));
}


/********************************************************************************
MulticastHandler_NewMulticastPacket

Description:
------------
	API for new Tx multicast packet

Input:
-----
	packet descriptor

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_NewMulticastPacket(TxPd_t* pMulticastPacket)
{
	MulticastMode_e 				mcastMode = NUM_OF_MULTICAST_MODE;
	uint8 							groupId = MAX_NUM_OF_GROUPS;
    HwQueueManagerRequestParams_t 	hwQueueManagerRequestParams;
	uint8							vapIndex;


	groupId = pMulticastPacket->txQStaId; // Group index is overlayed with STA index
	vapIndex = pMulticastPacket->txQVapId;
	mcastMode = MulticastHandlerDb.MulticastMode[vapIndex]; // check multicast mode

	if ((mcastMode == REGULAR_MULTICAST) || (groupId == GROUP_INDEX_BROADCAST))
	{     
        hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_DATA_DLM;
    	hwQueueManagerRequestParams.dplIndex = HW_TX_Q_TYPE_VAP_TID;
	    hwQueueManagerRequestParams.primaryAddr = vapIndex;
    	hwQueueManagerRequestParams.secondaryAddr = pMulticastPacket->txQTid;
	    hwQueueManagerRequestParams.pHeadDesc = pMulticastPacket;
#ifdef TX_DATA_THREAD_PATH
	    hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_ONE;
#else
		hwQueueManagerRequestParams.regIfNum = HW_Q_MANAGER_REG_IF_NUM_TWO;
#endif

        HwQManager_PushPacketToTail(&hwQueueManagerRequestParams);
	}
	else //RELIABLE_MULTICAST
	{
		MulticastHandler_SendReliableMulticast(vapIndex, groupId, pMulticastPacket);
	}
}


/********************************************************************************
MulticastHandler_ClonePacketDone

Description:
---------
	Handle clone PD from done list. Release the clone and check if origin PD can also be released.
	Clone PD is a unicast packet but it got to multicast done list due to the PD source.
	Regular multicast packets should not get here. They go through regular data path.
Input:
-----
	Clone PD

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_ClonePacketDone(TxPd_t* pPacketDesc)
{
	reliableMcastPacketDescriptor_t*	pClonePd = (reliableMcastPacketDescriptor_t*)pPacketDesc;
	TxPd_t* 				pOriginPd = (TxPd_t*)pClonePd->parentPd;

	DEBUG_ASSERT(pClonePd->pd.pdSource == HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST);
	DEBUG_ASSERT(pOriginPd->pdCounter > 0);

//	ILOG0_DDDD("MulticastHandler_ClonePacketDone: pdCounter = %d, , retryCount = %d, pOriginPd = %x, pClonePd = %x", pOriginPd->pdCounter, pClonePd->retryCount, pOriginPd, pClonePd);

	// Decrease the duplications counter
	pOriginPd->pdCounter--;

	// Check if all clone PDs are done.
	if (pOriginPd->pdCounter == 0) // if all the duplications already transmitted
	{
		// Release the origin PD (send back to host interface)
		MulticastHandler_ReleaseOriginPd(pOriginPd, HW_Q_MANAGER_REG_IF_NUM_ONE);

		// Statistics
		MulticastHandlerStatistics.allClonesFinishedTransmission++;
	}
#ifdef ENET_INC_ARCH_WAVE600 
	hostInterfaceDecrementPdByteCount(pPacketDesc->txQTid,pPacketDesc->txQStaId,pPacketDesc->dataLength); 				
#endif

	//Release the clone PD
	ResourceManager_ReleaseDescriptor((TxPd_t*)pPacketDesc,DESC_POOL_RELIABLE_MCAST);
}


/********************************************************************************
MulticastHandler_ReleaseOriginPd

Description:
---------
	Release the original multicast data PD (send to host interface)
Input:
-----
	Origin PD

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_ReleaseOriginPd(TxPd_t* pPacketDesc, RegInterfaceNum_e regIfNum)
{
	HwQueueManagerRequestParams_t hwQueueManagerRequestParams;
	memset(&hwQueueManagerRequestParams,0, sizeof(HwQueueManagerRequestParams_t));
	
	hwQueueManagerRequestParams.dlmNum = HW_Q_MANAGER_TX_LISTS_DLM;
	hwQueueManagerRequestParams.regIfNum = regIfNum;
	hwQueueManagerRequestParams.dplIndex = HW_Q_MANAGER_LOW_PR_READY_LIST_TEMP_HOST_Q;
	hwQueueManagerRequestParams.pHeadDesc = pPacketDesc;
	hwQueueManagerRequestParams.pTailDesc = pPacketDesc;
	HwQManager_PushPacketListToTail(&hwQueueManagerRequestParams);	 
}


/********************************************************************************
MulticastHandler_SendReliableMulticast

Description:
------------
	Handler for multicast reliable mode. Duplicate the PD and send the duplications as unicast data packets

Input:
-----
	PD pointer, Vap ID , group ID

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_SendReliableMulticast(uint8 vapIndex, uint8 groupIndex, TxPd_t* pMulticastPacket)
{
	uint32* 			pStaBitmap = NULL;
	uint16 				StaIndex;
	TxPd_t*	pNewClonePd = NULL;
	TxPd_t*	pClonePdsListHead = NULL;
	TxPd_t*	pCurrentClonePd = NULL;	
	uint8 				duplicateCounter = 0;
	TxPd_t*	tempNextPd = 0;

	//the groupIndex must be up to the max num of groups
	ASSERT(groupIndex < MAX_NUM_OF_GROUPS);

	// Get pointer to the STAs bit map of this group in this VAP
	pStaBitmap = MulticastHandlerDb.GroupMcastDb[groupIndex].VapMcastDb[vapIndex].StasInGroupBitmap;

	// find the first STA index (if doesn't exists, the value HW_NUM_OF_STATIONS is returned)
	StaIndex = Utils_CountTrailZeroBitmap(pStaBitmap, 0, STA_BITMAP_SIZE);

//	ILOG0_V("MulticastHandler_SendReliableMulticast");

	// Go over the entire STA bitmap. Try to take a Pd for every STA in the group.
	while (StaIndex < HW_NUM_OF_STATIONS)
	{
		// Get descriptor from the reliable multicast pool
		pNewClonePd = ResourceManager_GetDescriptor(DESC_POOL_RELIABLE_MCAST);

//		ILOG0_DD("pNewClonePd = %x, staIndex = %d", pNewClonePd, StaIndex);

		if (pNewClonePd != (TxPd_t*)NULL_PD)
		{
			// fill current STA index
			pNewClonePd->txQStaId = StaIndex;
			
			// Link all Clone PDs to one list
			if (pClonePdsListHead == NULL)
			{
				pClonePdsListHead = pNewClonePd;
				pCurrentClonePd = pClonePdsListHead;
			}
			else
			{
				pCurrentClonePd->nextPdPointer = SET_NEXT_PD(pNewClonePd);
				pCurrentClonePd = pNewClonePd;
			}

			// Count number of duplications
			duplicateCounter++;

			// Increment index to look for next STA in group
			StaIndex++;

			// check that we were not at the last STA index
			if (StaIndex < HW_NUM_OF_STATIONS)
			{
				// Get the next STA index in the group
				StaIndex = Utils_CountTrailZeroBitmap(pStaBitmap, StaIndex, STA_BITMAP_SIZE);
			}
		}
		else
		{
			// Not enough PDs. Release them all.
			pCurrentClonePd = pClonePdsListHead;
			while (duplicateCounter > 0)
			{
				tempNextPd = (TxPd_t*)GET_NEXT_PD(pCurrentClonePd);
				ResourceManager_ReleaseDescriptor(pCurrentClonePd, DESC_POOL_RELIABLE_MCAST);
				duplicateCounter--;
				pCurrentClonePd = tempNextPd;
			}

			// Release the origin PD (send back to host interface)
#ifdef TX_DATA_THREAD_PATH
			MulticastHandler_ReleaseOriginPd(pMulticastPacket, HW_Q_MANAGER_REG_IF_NUM_ONE);
#else
			MulticastHandler_ReleaseOriginPd(pMulticastPacket, HW_Q_MANAGER_REG_IF_NUM_TWO);
#endif

			// Statistics
			MulticastHandlerStatistics.notEnoughClonePds++;

			return;
		}
	}

	// Check if there are STAs in that group
	if (duplicateCounter > 0)
	{
		// Last Clone PD points to NULL
		pCurrentClonePd->nextPdPointer = NEXT_PD_NULL;
		
		// Duplicate the origin PD to all the clones
		pCurrentClonePd = pClonePdsListHead;
		while (pCurrentClonePd != (TxPd_t*)NULL_PD)
		{
			// Copy the origin PD to the clone PD (remember STA index, nextPd and PdSource)
			StaIndex = pCurrentClonePd->txQStaId;
			tempNextPd = (TxPd_t*)GET_NEXT_PD(pCurrentClonePd);
			MEMCPY(pCurrentClonePd, pMulticastPacket, sizeof(TxPd_t));
			pCurrentClonePd->txQStaId = StaIndex;
			pCurrentClonePd->nextPdPointer = SET_NEXT_PD(tempNextPd);
			pCurrentClonePd->pdSource = HW_Q_MANAGER_DONE_LIST_RELIABLE_MULTICAST;
				
			// Change the PD to unicast
			pCurrentClonePd->mcUnicast = UNICAST;
		
			// point from the clone to the origin PD
			((reliableMcastPacketDescriptor_t *)pCurrentClonePd)->parentPd = (uint32)pMulticastPacket;

			/* Collect statistics */
			
			PerClientStatistics.clonedCount[pCurrentClonePd->txQStaId]++;

#ifdef ENET_INC_ARCH_WAVE600
			hostInterfaceIncrementPdByteCount(pCurrentClonePd->txQTid,pCurrentClonePd->txQStaId,pCurrentClonePd->dataLength);
#endif
			// Get next PD
			pCurrentClonePd = (TxPd_t*)GET_NEXT_PD((TxPd_t*)pCurrentClonePd);
		}
		
		// set the number of the duplications in the origin PD (overlay the AMSDU PD counter field which is not needed for the clone)
		pMulticastPacket->pdCounter = duplicateCounter;
		
		// Send the duplicate PDs to the TX path
		TxPacketsClassifier_SendDataPacket((TxPd_t*)pClonePdsListHead);
	}
	else // There are no STAs in the group
	{
		// Release the origin PD (send back to host interface)
#ifdef TX_DATA_THREAD_PATH
		MulticastHandler_ReleaseOriginPd(pMulticastPacket, HW_Q_MANAGER_REG_IF_NUM_ONE);
#else
		MulticastHandler_ReleaseOriginPd(pMulticastPacket, HW_Q_MANAGER_REG_IF_NUM_TWO);
#endif		
		// Statistics
		MulticastHandlerStatistics.noStationsInGroup++;
	}
}


/********************************************************************************
MulticastHandler_AddStaToGroup

Description:
------------
	Add station to multicast group

Input:
-----
	Vap ID, Group index , STA index

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_AddStaToGroup(uint8 vapIndex, uint8 groupIndex, StaId staIndex)
{
	uint32* pStasInGroupBitmap = NULL;
	uint32* pGroupsOfStaBitmap = NULL;

	// Group index 0 is reserved for regular multicast (broadcast)
	DEBUG_ASSERT(groupIndex != 0);
	DEBUG_ASSERT(groupIndex < MAX_NUM_OF_GROUPS);
	DEBUG_ASSERT(vapIndex < HW_NUM_OF_VAPS);
	DEBUG_ASSERT(staIndex < HW_NUM_OF_STATIONS);
	
	// Get the bitmap of STAs of this group index for this VAP
	pStasInGroupBitmap = MulticastHandlerDb.GroupMcastDb[groupIndex].VapMcastDb[vapIndex].StasInGroupBitmap;

	// Set the bit representing the given STA
	Utils_SetBitInBitmap(pStasInGroupBitmap,staIndex);

	// Get the groups bitmap of this STA 
	pGroupsOfStaBitmap = MulticastHandlerDb.StaMcastDb[staIndex].GroupsOfStaBitmap;

	// Set the bit representing this group
	Utils_SetBitInBitmap(pGroupsOfStaBitmap,groupIndex);	
}


/********************************************************************************
MulticastHandler_RemoveStaFromGroup

Description:
------------
	Remove station from multicast group

Input:
-----
	Vap ID, Group index , STA index

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_RemoveStaFromGroup(uint8 vapIndex, uint8 groupIndex, StaId staIndex)
{
	uint32* pStasInGroupBitmap = NULL;
	uint32* pGroupsOfStaBitmap = NULL;

	// Group index 0 is reserved for regular multicast (broadcast)
	DEBUG_ASSERT(groupIndex != 0);
	DEBUG_ASSERT(groupIndex < MAX_NUM_OF_GROUPS);
	DEBUG_ASSERT(vapIndex < HW_NUM_OF_VAPS);
	DEBUG_ASSERT(staIndex < HW_NUM_OF_STATIONS);

	// Get the bitmap of STAs of this group index for this VAP
	pStasInGroupBitmap = MulticastHandlerDb.GroupMcastDb[groupIndex].VapMcastDb[vapIndex].StasInGroupBitmap;

	// Clear the bit representing the given STA
	Utils_ZeroBitInBitmap(pStasInGroupBitmap,staIndex);

	// Get the groups bitmap of this STA 
	pGroupsOfStaBitmap = MulticastHandlerDb.StaMcastDb[staIndex].GroupsOfStaBitmap;

	// Clear the bit representing this group
	Utils_ZeroBitInBitmap(pGroupsOfStaBitmap,groupIndex);	
}


/********************************************************************************
MulticastHandler_RemoveStaFromAllGroups

Description:
------------
	Remove station from all multicast groups

Input:
-----
	Vap ID, STA index

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_RemoveStaFromAllGroups(uint8 vapIndex, StaId staIndex)
{
	uint32* pGroupsOfStaBitmap = NULL;
	uint32 	groupIndex;

	DEBUG_ASSERT(staIndex < HW_NUM_OF_STATIONS);
	DEBUG_ASSERT(vapIndex < HW_NUM_OF_VAPS);	

	// Get the groups bitmap of this STA 
	pGroupsOfStaBitmap = MulticastHandlerDb.StaMcastDb[staIndex].GroupsOfStaBitmap;

	// Get the first group of this STA (will return ROUNDUP(MAX_NUM_OF_GROUPS,32) if there are no groups)
	groupIndex = Utils_CountTrailZeroBitmap(pGroupsOfStaBitmap, 0, GROUP_BITMAP_SIZE);
	
	while (groupIndex < MAX_NUM_OF_GROUPS)
	{
		// Remove the STA from the group. This also clears this group index from this STA bitmap in "pGroupsOfStaBitmap"
		MulticastHandler_RemoveStaFromGroup(vapIndex, groupIndex, staIndex);

		// Get the next group of this STA
		groupIndex = Utils_CountTrailZeroBitmap(pGroupsOfStaBitmap, 0, GROUP_BITMAP_SIZE);
	}

}


/********************************************************************************
MulticastHandler_StaStopTraffic

Description:
------------
	we got a request from driver to stop traffic for STA. We need to remove it from all multicast groups.

Input:
-----
	UMI_MULTICAST_ACTION pMulticastAction - pointer to driver struct with inputs

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_StaStopTraffic(K_MSG* multicastHandlerMessage)
{
	BssManagerStaManagerReq_t*	staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(multicastHandlerMessage);
	K_MSG*						psMsg = staManagerReq->psMsg;
	UMI_STOP_TRAFFIC*			pStopTraffic = (UMI_STOP_TRAFFIC *)pK_MSG_DATA(psMsg);
	uint16 						staIndex = pStopTraffic->u16SID;
	uint8 						vapIndex = psMsg->header.vapId;
	BssManagerStaManagerCfm_t*	confirmMessage = NULL;
	K_MSG*						psNewMsg = NULL;


	MulticastHandler_RemoveStaFromAllGroups(vapIndex, staIndex);

	// Allocate message for confirmation to STA Manager
	psNewMsg = OSAL_GET_MESSAGE(sizeof(BssManagerStaManagerCfm_t));
	confirmMessage = (BssManagerStaManagerCfm_t*) pK_MSG_DATA(psNewMsg);
	confirmMessage->clientId = BSS_MANAGER_STA_MANAGER_TXM_CLIENT;
	confirmMessage->sid = staIndex;

	//Send confirmation message
	OSAL_SEND_MESSAGE(BSS_MANAGER_STA_MANAGER_REG_CFM, TASK_BSS_MANAGER, psNewMsg, vapIndex);
}


/********************************************************************************
MulticastHandler_MulticastGroupActionReq

Description:
------------
	a request from driver to add STA to group / remove STA from group / remove from all groups

Input:
-----
	UMI_MULTICAST_ACTION pMulticastAction - pointer to driver struct with inputs

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_MulticastGroupActionReq(K_MSG* psMsg)
{
	UMI_MULTICAST_ACTION* 	pMulticastAction = (UMI_MULTICAST_ACTION*)pK_MSG_DATA(psMsg);	
	uint8 					vapIndex;
	StaId					staIndex = pMulticastAction->u16StaID;
	uint8					groupIndex = pMulticastAction->u8GroupID;
	uint8 					isBitSet = FALSE; // for wa
	uint32* 				pStasInGroupBitmap = NULL; // for wa

	// Get VAP index of this STA
	StaDb_GetVapId(staIndex, &vapIndex);

	//ILOG0_DDDD("ActionReq: u8Action = %d, vapIndex = %d, groupIndex = %d, staIndex = %d", pMulticastAction->u8Action, vapIndex, groupIndex, staIndex);

	switch (pMulticastAction->u8Action)
	{
		case ADD_STA_TO_MULTICAST_GROUP:
			/* workaround for Driver trying to add the same station twice: make sure the station is not set */
			// Get the bitmap of STAs of this group index for this VAP
			pStasInGroupBitmap = MulticastHandlerDb.GroupMcastDb[groupIndex].VapMcastDb[vapIndex].StasInGroupBitmap;			
			isBitSet = Utils_GetBitFromBitmap(pStasInGroupBitmap, staIndex);
			if (isBitSet == FALSE)
			{			
				// Add STA to multicast group
				MulticastHandler_AddStaToGroup(vapIndex, groupIndex, staIndex);
			}
			break;
		
		case REMOVE_STA_FROM_MULTICAST_GROUP:
			/* workaround for Driver trying to add the same station twice: make sure the station is not set */
			// Get the bitmap of STAs of this group index for this VAP
			pStasInGroupBitmap = MulticastHandlerDb.GroupMcastDb[groupIndex].VapMcastDb[vapIndex].StasInGroupBitmap;			
			isBitSet = Utils_GetBitFromBitmap(pStasInGroupBitmap, staIndex);
			if(isBitSet == TRUE)
			{			
				// Remove STA from multicast group
				MulticastHandler_RemoveStaFromGroup(vapIndex, groupIndex, staIndex);
			}
			break;			
		
		case REMOVE_STA_FROM_ALL_MULTICAST_GROUPS:
			// Remove STA from all multicast groups
			MulticastHandler_RemoveStaFromAllGroups(vapIndex, staIndex);
			break;

		default:
			ASSERT(0);
	}

	// Send confirmation to host
	OSAL_SEND_MESSAGE(UMI_MC_MAN_MULTICAST_ACTION_CFM, TASK_UM_IF_TASK, psMsg, psMsg->header.vapId);
}


/********************************************************************************
MulticastHandler_SetVapMulticastMode

Description:
------------
	Change the multicast mode of specific VAP

Input:
-----
	Vap ID, Multicast Mode

Output:
-------

Returns:
--------

********************************************************************************/
void MulticastHandler_SetVapMulticastMode(uint8 vapIndex, MulticastMode_e mcastMode)
{
	// We change the multicast mode on-th-fly and take a risk that some multicast frames might already be in FW/HW.
	MulticastHandlerDb.MulticastMode[vapIndex] = mcastMode;
}

/********************************************************************************
MulticastHandler_GetVapMulticastMode


Description:
------------
	return VAP's MC mode

Input:
-----


Output:
-------

Returns:
--------

********************************************************************************/
uint8 MulticastHandler_GetVapMulticastMode(uint8 vapIndex)
{
	return (uint8)MulticastHandlerDb.MulticastMode[vapIndex];
}

#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif


/********************************************************************************
MulticastHandler_Init

Description:
---------
	Init
Input:
-----
Output:
-------
Returns:
--------
********************************************************************************/


void MulticastHandler_Init(void)
{
	// Clear the Multicast Handler DB
	memset(&MulticastHandlerDb, 0, sizeof(McastHandlerDb_t));
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


