/***********************************************************************************
 File:			TxSelector_Api.h
 Module:		Tx Selector
 Purpose: 		Owner for all access to the Tx Selector
 Description:	This module is the API to the Tx Selector responsible for the 
				selection of the queue to transmit from.
************************************************************************************/
#ifndef TX_SELECTOR_API_H_
#define TX_SELECTOR_API_H_
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "OSAL_Kmsg.h"
#include "StaDb.h"
#include "Std.h"
#include "PacketDescriptor.h"
#include "ieee80211.h"
#include "InterruptManager_Api.h"
#ifdef ENET_INC_ARCH_WAVE600
#include "TwtManager.h"
#endif

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/



/*---------------------------------------------------------------------------------
/						Data Type Definition						
/----------------------------------------------------------------------------------*/

typedef enum TxSelectorAllStaUpdate
{
	TX_SELECTOR_SINGLE_STA_UPDATE	= 0,
	TX_SELECTOR_ALL_STAS_UPDATE 	= 1,
} TxSelectorAllStaUpdate_e;

typedef enum TxSelectorAllTidsUpdate
{
	TX_SELECTOR_SINGLE_TID_UPDATE	= 0,
	TX_SELECTOR_ALL_TIDS_UPDATE 	= 1,
} TxSelectorAllTidsUpdate_e;


#ifdef ENET_INC_ARCH_WAVE600

typedef struct TxSelectorActionParams
{
	uint16 stationOrVapNum;
	uint16 action; //values for per-queueType bitmapTypeSelection (single value, not per TID) need 16bits for OMI feature
	uint16 tidEnableBitmap; //Select the TID to update, one or more (or all)
	uint16 tidValueBitmap; //values for per-tid bitmapTypeSelection (0 or 1)
	TxQueueType_e queueType;
	uint8 acValue; //used for power save request which is requested in FW by AC and will be splitted into TIDs in the Tx Selector API function (ACCESS_CATEGORY_INAVLID == ALL TIDS)
	uint8 planIndex;
} TxSelectorActionParams_t;


typedef struct TxSelectorLockReqParam
{
	TxQueueType_e queueType;
	uint16 stationOrVapNum;
	uint8 tidOrAc;
	uint8 allTids;
    uint8 planIndex;
} TxSelectorLockReqParams_t;


typedef struct TxSelectorSelectionDetails
{
	uint8 selectionSource;
	uint8 tid;
	uint16 staNum;	
	uint8 vapNum;
	uint8 txqType;
	uint8 beamforming;
	uint8 accessCategory;
	uint8 emptySelection;
	uint8 validSelection;
} TxSelectorSelectionDetails_t;

typedef struct TxSelectorMuSelectionDetails
{
	uint8 tid;
	uint8 vapNum;
	uint16 staNum;
	uint8 txqId;
	uint8 beamforming;
	uint8 accessCategory;
	uint8 validSelection;
} TxSelectorMuSelectionDetails_t;

typedef struct TxSelectorMuActionParams
{
	uint16 groupIdx;
	uint8 userPositionIdx;
	uint8 action;
	TxSelectorAllStaUpdate_e allStas;
	uint8 allGroupValidUspBitmap : 4; // each bit is per USP station in group. when bit is set - station is valid in group
	uint8 reserved0				  :4;
} TxSelectorMuActionParams_t;

typedef struct TxSelectorAllGroupValidUspBitmap
{
	uint8 validUspBitmap : 4;
	uint8 reserved0		: 4;
} TxSelectorAllGroupValidUspBitmap_t;

typedef enum TxSelectorHeMu
{
	TX_SELECTOR_STA_HE_MU_DISABLE	    = 0,
	TX_SELECTOR_STA_HE_MU_ENABLE	    = 1,
} TxSelectorHeMu_e;

typedef enum TxSelectorLockUnlockPlan
{
	TX_SELECTOR_UNLOCK_PLAN			= 0,
	TX_SELECTOR_LOCK_PLAN           = 1,
} TxSelectorLockUnlockPlan_e;

typedef union
{
	uint16 val;
	struct
	{
		uint16 omiValid			:1;
		uint16 omiSupported		:1;
		uint16 tomiUlHeMuDis	:1;
		uint16 tomiDlHeMuDis	:1;
		uint16 romiHeErSuDis	:1;
		uint16 romiMaxNss		:3;
		uint16 omiBw			:2;
		uint16 tomiMaxNss		:3;
		uint16 reserved0        :3;
	} bitFields;
} TxSelectorOmiInfo_u;

typedef union
{
	uint8 val;
	struct
	{
 		uint8 suSelEn1Sec  :1;	// In case of VHT MU threshold fail - Configures if SU selection will happen
 		uint8 muThresh1Sec :1;	// Minimum number of secondaries to be selected to enable selection. Possible values 0-1
 		uint8 suSelEn2Sec  :1;	// In case of VHT MU threshold fail - Configures if SU selection will happen
 		uint8 muThresh2Sec :2;	// Minimum number of secondaries to be selected to enable selection. Possible values 0-2
		uint8 suSelEn3Sec  :1;	// In case of VHT MU threshold fail - Configures if SU selection will happen
 		uint8 muThresh3Sec :2;	// Minimum number of secondaries to be selected to enable selection. Possible values 0-3
	} bitFields;
} TxSelectorMuSelectCfg_u;

typedef enum TxSelectorVhtMuFallback // In case of VHT MU threshold fail - Configures if SU selection will happen
{
	TX_SELECTOR_DISABLE_VHT_MU_FALLBACK	= 0,
	TX_SELECTOR_ENABLE_VHT_MU_FALLBACK  = 1
} TxSelectorVhtMuFallback_e;

typedef enum TxSelectorVhtMuSec1Threshold
{
	TX_SELECTOR_VHT_MU_SEC1_THRESHOLD_0	= 0,
	TX_SELECTOR_VHT_MU_SEC1_THRESHOLD_1 = 1
} TxSelectorVhtMuSec1Threshold_e;

typedef enum TxSelectorVhtMuSec2Threshold
{
	TX_SELECTOR_VHT_MU_SEC2_THRESHOLD_0	= 0,
	TX_SELECTOR_VHT_MU_SEC2_THRESHOLD_1 = 1,
    TX_SELECTOR_VHT_MU_SEC2_THRESHOLD_2 = 2
} TxSelectorVhtMuSec2Threshold_e;

typedef enum TxSelectorVhtMuSec3Threshold
{
	TX_SELECTOR_VHT_MU_SEC3_THRESHOLD_0	= 0,
	TX_SELECTOR_VHT_MU_SEC3_THRESHOLD_1 = 1,
	TX_SELECTOR_VHT_MU_SEC3_THRESHOLD_2 = 2,
	TX_SELECTOR_VHT_MU_SEC3_THRESHOLD_3 = 3
} TxSelectorVhtMuSec3Threshold_e;

typedef enum TxSelectorDlWithUlEnable
{
	TX_SELECTOR_DL_WITH_UL_DISABLE	= 0,
	TX_SELECTOR_DL_WITH_UL_ENABLE   = 1
} TxSelectorDlWithUlEnable_e;

typedef enum TxSelectorMgmtBypassEnable
{
	TX_SELECTOR_MGMT_BYPASS_DISABLE	 = 0,
	TX_SELECTOR_MGMT_BYPASS_ENABLE   = 1
} TxSelectorMgmtBypassEnable_e;

typedef enum TxSelectorHeMuFallBack
{
	TX_SELECTOR_FALLBACK_TO_EMPTY	= 0,	//fallback to TXH as Empty Selection in case Plan is locked
	TX_SELECTOR_FALLBACK_TO_SU   	= 1		//fallback to SU Selection in case Plan is locked
} TxSelectorHeMuFallBack_e;

#else

typedef struct TxSelectorActionParams
{
	TxQueueType_e queueType;
	uint8 stationOrVapNum;
	uint8 tidOrAc;
	uint8 action;
	TxSelectorAllTidsUpdate_e allTids;
} TxSelectorActionParams_t;


typedef struct TxSelectorLockReqParam
{
	TxQueueType_e queueType;
	uint8 stationOrVapNum;
	uint8 tidOrAc;
	uint8 allTids;
} TxSelectorLockReqParams_t;


typedef struct TxSelectorSelectionDetails
{
	uint8 selectionSource;
	uint8 tid;
	uint8 staNum;	
	uint8 vapNum;
	uint8 txqType;
	uint8 beamforming;
	uint8 accessCategory;
	uint8 emptySelection;
	uint8 validSelection;
} TxSelectorSelectionDetails_t;


typedef struct TxSelectorMuSelectionDetails
{
	uint8 tid;
	uint8 vapNum;
	uint8 staNum;
	uint8 txqId;
	uint8 beamforming;
	uint8 accessCategory;
	uint8 validSelection;
} TxSelectorMuSelectionDetails_t;

typedef struct TxSelectorMuActionParams
{
	uint8 groupIdx;
	uint8 userPositionIdx;
	uint8 action;
	TxSelectorAllStaUpdate_e allStas;
} TxSelectorMuActionParams_t;


#endif //ENET_INC_ARCH_WAVE600


typedef enum TxSelectorInstructions
{
	TX_SELECTOR_INSTRUCTION_IDLE = 0,
	TX_SELECTOR_INSTRUCTION_ACTIVATE,
	TX_SELECTOR_INSTRUCTION_RECALCULATE,
	TX_SELECTOR_INSTRUCTION_CLEAR,
	TX_SELECTOR_NUM_OF_INSTRUCTION,
	TX_SELECTOR_MAX_NUM_OF_INSTRUCTION = 0x7,
} TxSelectorInstructions_e;

typedef enum TxSelectorEnableDisableQ
{
	TX_SELECTOR_ENABLE_QUEUE		= 0,
	TX_SELECTOR_DISABLE_QUEUE		= 1,
} TxSelectorEnableDisableQ_e;

typedef enum TxSelectorLockUnlockQ
{
	TX_SELECTOR_UNLOCK_Q			= 0,
	TX_SELECTOR_LOCK_Q				= 1,
} TxSelectorLockUnlockQ_e;

typedef enum TxSelectorPowerSaveType
{
	TX_SELECTOR_PS_TYPE_LEGACY		= 0,
	TX_SELECTOR_PS_TYPE_WMM 		= 1,
} TxSelectorPowerSaveType_e;

typedef enum TxSelectorPowerSaveMode
{
	TX_SELECTOR_PS_MODE_ACTIVE		= 0,
	TX_SELECTOR_PS_MODE_POWER_SAVE	= 1,
} TxSelectorPowerSaveMode_e;

typedef enum TxSelectorPowerSaveRequest
{
	TX_SELECTOR_RESET_POWER_SAVE_REQUEST	= 0,
	TX_SELECTOR_SET_POWER_SAVE_REQUEST		= 1,
} TxSelectorPowerSaveRequest_e;

typedef enum TxSelectorDataInQ
{
	TX_SELECTOR_RESET_DATA_IN_Q	    = 0,
	TX_SELECTOR_SET_DATA_IN_Q	    = 1,
} TxSelectorDataInQ_e;

typedef enum TxSelectorBeamformingType
{
	TX_SELECTOR_BF_TYPE_WITHOUT_DATA	= 0,
	TX_SELECTOR_BF_TYPE_WITH_DATA		= 1,
} TxSelectorBeamformingType_e;

typedef enum TxSelectorBeamformingRequest
{
	TX_SELECTOR_RESET_BEAMFORMING_REQUEST	= 0,
	TX_SELECTOR_SET_BEAMFORMING_REQUEST		= 1,
} TxSelectorBeamformingRequest_e;


typedef enum TxSelectorWrrPriority
{
	TX_SELECTOR_WRR_PRIORITY_0 = 0,
	TX_SELECTOR_WRR_PRIORITY_1,
	TX_SELECTOR_WRR_PRIORITY_2,
	TX_SELECTOR_WRR_PRIORITY_3,
	TX_SELECTOR_WRR_PRIORITY_4,
	TX_SELECTOR_WRR_PRIORITY_5,
	TX_SELECTOR_WRR_PRIORITY_6,
	TX_SELECTOR_WRR_PRIORITY_7,
	TX_SELECTOR_WRR_NUM_OF_PRIORITIES,
} TxSelectorWrrPriority_t;

typedef enum TxSelectorStrictPriority
{
	TX_SELECTOR_STRICT_PRIORITY_HIGH = 1,
	TX_SELECTOR_STRICT_PRIORITY_MEDIUM,
	TX_SELECTOR_STRICT_PRIORITY_LOW,
} TxSelectorStrictPriority_e;

typedef enum TxSelectorStrictPriorityAction
{
	TX_SELECTOR_MASK_QUEUES = 0,
	TX_SELECTOR_UNMASK_QUEUES = 1
} TxSelectorStrictPriorityAction_e;

typedef enum TxSelectorGroupSelectionOffset
{
	TX_SELECTOR_GROUP_SELECTION_OFFSET_VAP_ACTIVE_AND_STA = 0,
	TX_SELECTOR_GROUP_SELECTION_OFFSET_VAP_PS,
	TX_SELECTOR_GROUP_SELECTION_OFFSET_GLOBAL,
	TX_SELECTOR_GROUP_SELECTION_OFFSET_GPLP,
	TX_SELECTOR_GROUP_SELECTION_OFFSET_GPHP,
	TX_SELECTOR_GROUP_SELECTION_OFFSET_BEACON,
	TX_SELECTOR_NUM_OF_GROUP_SELECTION_OFFSET,
	TX_SELECTOR_MAX_NUM_OF_GROUP_SELECTION_OFFSET = 0xFF,
} TxSelectorGroupSelectionOffset_e;

typedef enum TxSelectorGroupSelectionEnable
{
	TX_SELECTOR_GROUP_SELECTION_ENABLED = 0,
	TX_SELECTOR_GROUP_SELECTION_DISABLED,
} TxSelectorGroupSelectionEnable_e;

typedef enum TxSelectorSource
{
	TX_SELECTOR_SOURCE_STA_TID_ACTIVE = 0,
	TX_SELECTOR_SOURCE_STA_TID_PS,
	TX_SELECTOR_SOURCE_VAP_TID_ACTIVE,
	TX_SELECTOR_SOURCE_VAP_TID_PS,
	TX_SELECTOR_SOURCE_GLOBAL,
	TX_SELECTOR_SOURCE_GPLP,
	TX_SELECTOR_SOURCE_GPHP,
	TX_SELECTOR_SOURCE_BEACON,
	TX_SELECTOR_NUM_OF_SOURCES,
	TX_SELECTOR_MAX_NUM_OF_SOURCES = 0xF,
} TxSelectorSource_e;

typedef enum TxSelectorSelectionValid
{
	TX_SELECTOR_SELECTION_NOT_VALID = 0,
	TX_SELECTOR_SELECTION_VALID,
} TxSelectorSelectionValid_e;

typedef enum TxSelectorSelectionExist
{
	TX_SELECTOR_SELECTION_DOES_NOT_EXIST = 0,
	TX_SELECTOR_SELECTION_EXIST,
} TxSelectorSelectionExist_e;

typedef enum TxSelectorAssociateStaVap
{
	TX_SELECTOR_DISASSOCIATE = 0,
	TX_SELECTOR_ASSOCIATE = 1
} TxSelectorAssociateStateSta2Vap_e;

typedef enum TxSelectorError
{
	TX_SELECTOR_FIFO_OVERFLOW = 0,
	TX_SELECTOR_ACTIVE_IN_FREEZE
} TxSelectorError_e;

typedef enum TxSelectorBitmapAccessType
{
	TX_SELECTOR_BIT_MAP_ACCESS_TYPE_READ = 0,
	TX_SELECTOR_BIT_MAP_ACCESS_TYPE_WRITE,
} TxSelectorBitmapAccessType_e;

typedef enum TxSelectorLockReqType
{
	TX_SELECTOR_LOCK_REQ_TYPE_TRY_DISABLE	= 0, 
	TX_SELECTOR_LOCK_REQ_TYPE_TRY_LOCK		= 1 ,
	TX_SELECTOR_LOCK_REQ_TYPE_FORCE_DISABLE	= 2 ,
	TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK 	= 3, 
}TxSelectorLockReqType_e;

typedef enum TxSelectorLockStatus
{
	TX_SELECTOR_LOCK_STATUS_IDLE = 0x0,
	TX_SELECTOR_LOCK_STATUS_PENDING = 0x1,
	TX_SELECTOR_LOCK_STATUS_LOCKED = 0x2,
	TX_SELECTOR_LOCK_STATUS_DISABLED = 0x4,
	TX_SELECTOR_LOCK_STATUS_REJECT = 0x8,
	TX_SELECTOR_LOCK_STATUS_ERROR = 0x10,
} TxSelectorLockStatus_e;


typedef enum TxSelectorMuStaPrimaryEnabled
{
	TX_SELECTOR_MU_STA_PRIMARY_DISABLED	= 0,
	TX_SELECTOR_MU_STA_PRIMARY_ENABLED 	= 1,
} TxSelectorMuStaPrimaryEnabled_e;

typedef enum TxSelectorGroupTxMode
{
	TX_SELECTOR_GROUP_DISABLED			= 0,
	TX_SELECTOR_DATA_ONLY,
	TX_SELECTOR_BF_ONLY,
	TX_SELECTOR_BF_WITH_DATA,
} TxSelectorGroupTxMode_e;


typedef union
{
	uint8 val;
	struct
	{
		uint8 groupIdx:7;
		uint8 isPrimaryInGroup:1;
	} bitFields;	
} TxSelectorPrimaryStaInfo_u;



typedef union
{
	uint8 val;
	struct
	{
		uint8 staId:7;
		uint8 isValid:1;
	} bitFields;
} TxSelectorGroupUserPositionStaInfo_u;

#ifndef ENET_INC_ARCH_WAVE600

typedef union
{
	uint32 val;
	struct
	{
		TxSelectorGroupUserPositionStaInfo_u staInfoUserPosition0;
		TxSelectorGroupUserPositionStaInfo_u staInfoUserPosition1;
		TxSelectorGroupUserPositionStaInfo_u staInfoUserPosition2;
		TxSelectorGroupUserPositionStaInfo_u staInfoUserPosition3;
	} bitFields;
} RegTxSelectorAllStaBitmapUpdate_u;

#else

typedef union
{
	uint32 val;
} RegTxSelectorAllStaBitmapUpdate_u;

#endif

typedef union
{
	uint8 val;
	struct
	{
		uint8 tid0:1;
		uint8 tid1:1;
		uint8 tid2:1;
		uint8 tid3:1;
		uint8 tid4:1;
		uint8 tid5:1;
		uint8 tid6:1;
		uint8 tid7:1;
		
	} bitFields;
} RegTxSelectorMuTidBitmap_u;


typedef enum TxSelectorLockUnlockGroup
{
	TX_SELECTOR_UNLOCK_GROUP		= 0,
	TX_SELECTOR_LOCK_GROUP			= 1,
} TxSelectorLockUnlockGroup_e;


typedef enum TxSelectorMuSecondarySelectionMethod
{
	TX_SELECTOR_MU_SECONDARY_TID_SAME_AS_PRIMARY	= 0,
	TX_SELECTOR_MU_ANY_SECONDARY_TID_ALLOWED		= 1,
} TxSelectorMuSecondarySelectionMethod_e;

typedef enum TxSelectorMuLockAfterSelection
{
	TX_SELECTOR_MU_DONT_LOCK_AFTER_SELECTION	= 0,
	TX_SELECTOR_MU_LOCK_AFTER_SELECTION			= 1,
} TxSelectorMuLockAfterSelection_e;

typedef enum TxSelectorEnableDisableMu
{
	TX_SELECTOR_MU_DISABLE	= 0,
	TX_SELECTOR_MU_ENABLE	= 1,
} TxSelectorEnableDisableMu_e;

typedef enum TxSelectorTwtType
{
	TX_SELECTOR_TWT_TYPE_UNANNOUNCED = 0,
	TX_SELECTOR_TWT_TYPE_ANNOUNCED	 = 1,
} TxSelectorTwtType_e;

typedef enum TxSelectorTwtAvail
{
	TX_SELECTOR_TWT_NOT_AVAILABLE   = 0,
	TX_SELECTOR_TWT_AVAILABLE	    = 1,
} TxSelectorTwtAvail_e;

typedef enum TxSelectorAControl
{
	TX_SELECTOR_A_CONTROL_DISABLE   = 0,
	TX_SELECTOR_A_CONTROL_ENABLE    = 1
} TxSelectorAControl_e;

/*---------------------------------------------------------------------------------
/						Public Function Declaration						
/----------------------------------------------------------------------------------*/

#ifdef ENET_INC_ARCH_WAVE600
#ifdef USE_AGER_EMULATOR
TxSelectorLockStatus_e TxSelector_TryLockRequest(uint16 staVapIndex , uint8 tidIndex, TxQueueType_e queueType);
#endif //USE_AGER_EMULATOR
void TxSelector_MapAllStasToGroup(uint8 groupIdx, RegTxSelectorAllStaBitmapUpdate_u allStaInfo, TxSelectorAllGroupValidUspBitmap_t allGroupValidUspBitmap);
void TxSelector_UpdateStaInGroup(uint8 groupIdx, uint8 userPosition, TxSelectorGroupUserPositionStaInfo_u staInfo, TxSelectorAllGroupValidUspBitmap_t allGroupValidUspBitmap);
#ifndef ENET_INC_ARCH_WAVE600B
void TxSelector_HeMuTidUpdate(uint16 stationNum, uint16 tidBitmap); 
#else // ENET_INC_ARCH_WAVE600B
void TxSelector_HeMuDlTidUpdate(uint16 stationNum, uint16 tidBitmap); 
void TxSelector_HeMuUlStationUpdate(uint16 stationNum, bool enable);
#ifndef ENET_INC_ARCH_WAVE600D2
void TxSelector_TwtTypeUpdateEnable(void);
void TxSelector_TwtSwHalt(void);
#endif
void TxSelector_SetOmiValid(uint16 stationNum, bool omiValid);
void TxSelector_SetOmiSupported(uint16 stationNum, bool omiSupported);
void TxSelector_SetOmiBits(uint16 stationNum, TxSelectorOmiInfo_u *txSelectorOmiInfo);
#endif //ENET_INC_ARCH_WAVE600B
void TxSelector_SetDLPlanIndex(uint16 stationNum, uint8 planIndex);
void TxSelector_SetULPlanIndex(uint16 stationNum, uint8 planIndex);
void TxSelector_LockPlanRequest(uint8 planIndex, uint16 vapId);
void TxSelector_UnlockPlan(uint8 planIndex, StaId staId);
void TxSelector_SelectModeConfig(TxSelectorDlWithUlEnable_e dlWithUlEnable, TxSelectorMgmtBypassEnable_e mgmtBypassEnable, AccessCategory_e mgmtAcMap, TxSelectorHeMuFallBack_e heMuFallBack);
#ifndef ENET_INC_ARCH_WAVE600D2
void TxSelector_ConfigureTwtRegsNewSp(uint32 twtSpLowLimit, uint32 twtSpStartTsf, uint32 twtSpHighLimit, uint8 twtSpGroupIdx, TwtManagerFlowType_e twtType, uint8 bandId);
void TxSelector_ConfigureTwtRegsUpdateSta(uint8 twtSpStaGroupIdx, uint16 twtSpStaIdx, bool twtSpStaValue);
void TxSelector_ConfigureTwtRegsDeactivateSp(uint8 twtSpDeactivateGroupIdx, uint8 bandId);
#endif
void TxSelector_UpdateBitmapAddTwtSta(StaId staId, TwtManagerFlowType_e twtType);
void TxSelector_UpdateBitmapRemTwtSta(StaId staId);


ISR_VOID isr_TxSelector_LockerRequestServiced_Band0(void);
ISR_VOID isr_TxSelector_LockerRequestServiced_Band1(void);


#if defined (ENET_INC_LMAC)
#ifdef WORKAROUND_FOR_HW_BUG_IN_PS_SETTING_IN_TWT
void TxSelector_ResetPsRequestForTwt(StaId staId);
#endif // WORKAROUND_FOR_HW_BUG_IN_PS_SETTING_IN_TWT
void TxSelector_GetLastMuSelection(TxSelectorMuSelectionDetails_t* selection, uint8 userPosition);
void TxSelector_ExecuteInstruction (TxSelectorInstructions_e instruction);
void TxSelector_SetBfSequenceRequest(TxSelectorActionParams_t *txSelectorActionParams);
void TxSelector_SetBfSequenceType(TxSelectorActionParams_t *txSelectorActionParams);
void TxSelector_SetVapWrrPriority(uint8 vapNum,uint8 wrrPsPriority, uint8 wrrActivePriority);
void TxSelector_SetStrictPriority(TxSelectorStrictPriority_e globalPriority, TxSelectorStrictPriority_e vapMpsPriority, TxSelectorStrictPriority_e activePriority);
void TxSelector_MaskGroupSelection(TxSelectorGroupSelectionOffset_e gruopOffset,TxSelectorGroupSelectionEnable_e selectionEnable);
void TxSelector_GetLastSelection(TxSelectorSelectionDetails_t* selection);
void TxSelector_GetLastSourceSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails,TxSelectorSource_e sourceNum);
void isr_TxSelector_Error(void);
void TxSelector_EnableDisableMuTx(TxSelectorEnableDisableMu_e muEnableDisable, uint8 maxGroupNum, TxSelectorMuSecondarySelectionMethod_e muSecondaryTid, TxSelectorMuLockAfterSelection_e muLockPolicy, TxSelectorMuSelectCfg_u *muSelectCfg);
void TxSelector_SetPrimaryPdCountThreshold(uint32 threshold);
void TxSelector_SetSecondaryPdCountThreshold(uint32 threshold);
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(void);
void TxSelector_Init(void);
#else
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(uint8 bandId);
void isr_TxSelector_LockerGroupRequestServiced(uint8 bandId);
void hwEventsTxSelectorLockerPlanRequestServiced(uint8 bandId);

#endif //ENET_INC_UMAC

#else // ENET_INC_ARCH_WAVE600
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(void);
void TxSelector_GetLastMuSelection(TxSelectorMuSelectionDetails_t* selection, uint8 userPosition);
TxSelectorLockStatus_e TxSelector_TryLockRequest(uint8 staVapIndex , uint8 tidIndex, uint8 qGroup);
void TxSelector_MapAllStasToGroup(uint8 groupIdx, RegTxSelectorAllStaBitmapUpdate_u allStaInfo);
void TxSelector_UpdateStaInGroup(uint8 groupIdx, uint8 userPosition, TxSelectorGroupUserPositionStaInfo_u staInfo); 
void TxSelector_ExecuteInstruction (TxSelectorInstructions_e instruction);
void TxSelector_SetBfSequenceRequest(TxSelectorActionParams_t *txSelectorActionParams);
void TxSelector_SetBfSequenceType(TxSelectorActionParams_t *txSelectorActionParams);
void TxSelector_SetVapWrrPriority(uint8 vapNum,uint8 wrrPsPriority, uint8 wrrActivePriority);
void TxSelector_SetStrictPriority(TxSelectorStrictPriority_e globalPriority, TxSelectorStrictPriority_e vapMpsPriority, TxSelectorStrictPriority_e activePriority);
void TxSelector_MaskGroupSelection(TxSelectorGroupSelectionOffset_e gruopOffset,TxSelectorGroupSelectionEnable_e selectionEnable);
void TxSelector_GetLastSelection(TxSelectorSelectionDetails_t* selection);
void TxSelector_GetLastSourceSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails,TxSelectorSource_e sourceNum);
void isr_TxSelector_Error(void);
void TxSelector_EnableDisableMuTx(TxSelectorEnableDisableMu_e muEnableDisable, uint8 maxGroupNum, TxSelectorMuSecondarySelectionMethod_e muSecondaryTid, TxSelectorMuLockAfterSelection_e muLockPolicy);
void TxSelector_SetPrimaryPdCountThreshold(uint16 threshold);
void TxSelector_SetSecondaryPdCountThreshold(uint16 threshold);
void TxSelector_Init(void);
ISR_VOID isr_TxSelector_LockerRequestServiced(void);
void isr_TxSelector_LockerGroupRequestServiced(void);
#endif //ENET_INC_ARCH_WAVE600


void TxSelector_SetStationToVapAssocState(StaId stationId, uint8 vapId);
void TxSelector_ClearStationFromVapAssocState(StaId stationId, uint8 vapId);
void TxSelector_ModifyPrimaryStaInfo(StaId staId, TxSelectorPrimaryStaInfo_u primaryStaInfo);
void TxSelector_ConfigureStaMuEnabledTids(StaId staId, RegTxSelectorMuTidBitmap_u tidBitmap);
bool TxSelector_IsDataInQueue(StaId stationOrVapIndex, uint8 tid, TxQueueType_e queueType);
void TxSelector_ResumeVapQueues(uint8 vapId); 
void TxSelector_EnableQueue(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_UnlockQueue(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_LockRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams);
void TxSelector_DisableRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams);
void TxSelector_SetPowerSaveType(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_SetPowerSavemode(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_ClearPowerSavemode(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_SetPowerSaveRequest(TxSelectorActionParams_t *txSelectorUpdateActionParams);
void TxSelector_LockGroupRequest(uint32 groupId);
void TxSelector_SetDataInQ(TxSelectorActionParams_t *txSelectorActionParams);
void TxSelector_UpdateGroupTxMode(uint8 groupIdx, TxSelectorGroupTxMode_e txMode);
void TxSelector_UnlockGroup(uint8 groupIdx);
#ifdef ENET_INC_ARCH_WAVE600B
void TxSelector_SetAControl(StaId staID, uint8 enable);
void TxSelector_MbssidConfig(uint8 transmitVapId, uint8 numOfVapsInGroup);
#endif //ENET_INC_ARCH_WAVE600B

#endif //TX_SELECTOR_API_H_


