/***********************************************************************************
 File:			TxSelector.c
 Module:		Tx Selector
 Purpose:		
 Description:	FW driver for TX Selector module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "TxSelector_Api.h"
#include "TxSelectorRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "BSSmanager_API.h"
#include "Locker_Api.h"
#include "OSAL_Tasks.h"
#include "ieee80211.h"
#include "Protocol_WmeDefinitions.h"
#include "ShramGroupDatabase.h"
#include "loggerAPI.h"
#include "TxSelectorRam.h"
#include "CpuLoad_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID 3

/* Locker  Configuration */
#define TX_SELECTOR_LOCKER_ENABLE							(1)
#define TX_SELECTOR_RESET_VALUE_ZERO						(0)
/* Used in TxSelector_SetStaWrr based on WRR Registers 0x8 - 0x10 */
#define TX_SELECTOR_WRR_VAP_PRIORITY_WIDTH 					(3)  /* there are 3 bits for every vap priority. Priority is in the range 0-7 */
#define TX_SELECTOR_WRR_VAP_PRIORITY_MASK				    (0x7)   /* The Vap mask of the register is 0x7 */
#define TX_SELECTOR_VAP_MSB_SHIFT							(3)
#define TX_SELECTOR_NUM_OF_VAPS_IN_WRR_REGISTER_MASK		(0x7) /* used to determine if priority is set to LOW or HIGH register  */	
#define TX_SELECTOR_NO_DATA_IN_Q							(0)
/* Vap Ac Matrix Definition Based On Vap Matrix Registers 0x68-0x74*/
#define TX_SELECTOR_VAP_AC_MATRIX_VAP_WIDTH					(8)	/* There are 7 bit for each vap + 1 reserved*/
#define TX_SELECTOR_VAP_AC_MATRIX_VAP_MASK					(0x7F)	
#define	TX_SELECTOR_VAP_AC_MATRIX_AC0_MASK					(0x1)
#define	TX_SELECTOR_VAP_AC_MATRIX_AC1_MASK    				(0x2)		
#define	TX_SELECTOR_VAP_AC_MATRIX_AC2_MASK					(0x4)		
#define	TX_SELECTOR_VAP_AC_MATRIX_AC3_MASK					(0x8)		
#define	TX_SELECTOR_VAP_AC_MATRIX_GPLP_MASK					(0x10)		
#define	TX_SELECTOR_VAP_AC_MATRIX_GPHP_MASK					(0x20)	
#define	TX_SELECTOR_VAP_AC_MATRIX_BEACON_MASK				(0x40)	


#define TX_SELECTOR_VAP_AC_MATRIX_REG_REG_OFFSET_SHIFT		(2)	/* There 4 Vaps in each register result a shift of 2 to determine the Register address offset */
#define TX_SELECTOR_VAP_AC_MATRIX_REG_BIT_OFFSET_MASK		(0x3) /* There 4 Vaps in each register result a mask of 2 to determine the bit offset in the register */	

#define TX_SELECTOR_VAP_ZERO_STA_BITMAP						(1)


/* MU PD Count Thresholds Definition */
#define TX_SELECTOR_PRIMARY_PD_COUNT_THRESHOLD				(4) //was 10
#define TX_SELECTOR_SECONDARY_PD_COUNT_THRESHOLD			(4) //was 10



#if defined (ENET_INC_UMAC)
#define REG_TX_SELECTOR_LOCKER_REQ 		(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER)
#define REG_TX_SELECTOR_LOCKER_STATUS	(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER_STATUS)
#else
#define REG_TX_SELECTOR_LOCKER_REQ 		(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER)
#define REG_TX_SELECTOR_LOCKER_STATUS	(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER_STATUS)

#endif


typedef enum TxSelectorBitTypeSelection
{
	TX_SELECTOR_DATA_IN_Q = 0,
	TX_SELECTOR_LOCK ,
	TX_SELECTOR_PS_REQUEST,
	TX_SELECTOR_POWER_SAVE_TYPE,
	TX_SELECTOR_POWER_SAVE_MODE,
	TX_SELECTOR_DISABLE,
	TX_SELECTOR_BF_SEQUENCE_REQUEST,
	TX_SELECTOR_BF_SEQUENCE_TYPE,
	TX_SELECTOR_NUM_OF_BIT_TYPES,
	TX_SELECTOR_MAX_NUM_OF_BIT_TYPES = 0xFF,
} TxSelectorBitTypeSelection_e;

typedef enum TxSelectorMuBitTypeSelection
{
	TX_SELECTOR_GROUP_USER_POSITION_STA = 0,
	TX_SELECTOR_GROUP_LOCK ,
	TX_SELECTOR_GROUP_TX_MODE,
	TX_SELECTOR_STA_GROUP_MAPPING,
	TX_SELECTOR_STA_MU_TID_ENABLE,
	TX_SELECTOR_NUM_OF_MU_BIT_TYPES,
	TX_SELECTOR_MAX_NUM_OF_MU_BIT_TYPES = 0xFF,
} TxSelectorMuBitTypeSelection_e;


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/
static void txSelectorResetStationTidRam(void);
static void	txSelectorResetVapTidRam(void);
static void	txSelectorResetGlobalRam(void);
static void	txSelectorResetGplpRam(void);
static void	txSelectorResetGphpRam(void);
static void	txSelectorResetBeaconRam(void);
static void txSelectorResetMuStationRam(uint32 numOfResetEntries);
static void txSelectorResetMuGroupRam(uint32 numOfResetEntries);
static void txSelectorUpdateBitmap(TxSelectorActionParams_t *updateBitmapParams,TxSelectorBitTypeSelection_e bitmapTypeSelection);
static void txSelectorUpdateMuBitmap(TxSelectorMuActionParams_t *updateMuBitmapParams,TxSelectorMuBitTypeSelection_e muBitmapTypeSelection);
static void txSelectorSetWrrPriority(uint8 vapNum, uint8 wrrPriority, uint32 baseAddress);
static void txSelectorFillSelectionParams(TxSelectorSelectionDetails_t* selectorSelection,uint32 selectionRegAddress);
static void txSelectorFillMuSelectionParams(TxSelectorMuSelectionDetails_t* selectorMuSelection,uint32 selectionRegAddress);
static void txSelectorWaitForMuBitmapFree(void);

/*---------------------------------------------------------------------------------
/						Static variables Declaration									
/----------------------------------------------------------------------------------*/
static TxSelectorRam_t *pTxSelectorRam;


/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

txSelectorUpdateBitmap 

Description:
------------
	Updates the information bits related to the queues of the chosen entry(Station, VAP, Global GPHP, GPLP, Beacon)
	The information bits reflects the queue state (Lock, Disabled, PS , PS request is pending, BF Sequence)

Input:
-----
	TxSelectorActionParams_t * bitmapUpdateParams - The entry parmetes(Q type,tid) and the required bit new value
	TxSelectorBitmapTypeSelection_e bitmapTypeSelection	- Which bits to change.
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
static void txSelectorUpdateBitmap(TxSelectorActionParams_t *updateBitmapParams,TxSelectorBitTypeSelection_e bitmapTypeSelection)
{
	RegTxSelectorTxSelBitmapUpdate_u bitMapUpadateReg;

	bitMapUpadateReg.val = 0;
	bitMapUpadateReg.bitFields.staSel = updateBitmapParams->stationOrVapNum;
	bitMapUpadateReg.bitFields.tidSel = updateBitmapParams->tidOrAc ;
	bitMapUpadateReg.bitFields.bitmapSel = updateBitmapParams->queueType;
	bitMapUpadateReg.bitFields.bitTypeSel = bitmapTypeSelection;
	bitMapUpadateReg.bitFields.bitValue = updateBitmapParams->action;
	bitMapUpadateReg.bitFields.allTids = updateBitmapParams->allTids;

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE, bitMapUpadateReg.val);
}


/**********************************************************************************

txSelectorUpdateMuBitmap 

Description:
------------
	Updates the information bits related to the MUGroup and STA entries
	The information bits reflects the Group state (Lock, Disabled STAs, STA USPs, Tx mode), and STA state (Primary enabled in Group, MU per TID Enabled/disabled).

	Need to verify that the Selector is free to recieve new bitmap configuration

Input:
-----
	TxSelectorMuActionParams_t *updateMuBitmapParams - The entry parmetes(Group index, USP, allSTAs update) and the required bit new value.
	TxSelectorMuBitTypeSelection_e muBitmapTypeSelection	- Which bits to change.
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

static void txSelectorUpdateMuBitmap(TxSelectorMuActionParams_t *updateMuBitmapParams, TxSelectorMuBitTypeSelection_e muBitmapTypeSelection)
{
	RegTxSelectorTxSelMuBitmapUpdate_u muBitmapUpdate;

	//wait for bitmap update free indication (usually it will be free for the first time we check. If there are consecutive updates than a delay may occur).
	txSelectorWaitForMuBitmapFree();

	
	muBitmapUpdate.val = 0;
	muBitmapUpdate.bitFields.memLineNum = updateMuBitmapParams->groupIdx;
	muBitmapUpdate.bitFields.userPositionIdx = updateMuBitmapParams->userPositionIdx;
	muBitmapUpdate.bitFields.muBitTypeSel = muBitmapTypeSelection;
	muBitmapUpdate.bitFields.muBitmapUpdateValue = updateMuBitmapParams->action;
	muBitmapUpdate.bitFields.allGrp = updateMuBitmapParams->allStas;
	
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, muBitmapUpdate.val);
}



/**********************************************************************************

txSelectorWaitForMuBitmapFree 

Description:
------------
	Wait until MU Bitmap is free for new configuration.

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

static void txSelectorWaitForMuBitmapFree(void)
{
	RegTxSelectorTxSelMuBitmapUpdate_u muBitmapUpdate;
	muBitmapUpdate.val = 0;

	while(muBitmapUpdate.bitFields.muBitmapUpdateFree == 0)
	{
    	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, &muBitmapUpdate.val);
	}

	/*When we get here - the bitmap is free for update*/
}


#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
static void txSelectorResetStationTidRam(void)
{
	TxSelectorStaTidRam_t staTidBitmap;
	uint8 i = 0;

	staTidBitmap.word0 = TX_SELECTOR_RESET_VALUE_ZERO;
	staTidBitmap.word1 = TX_SELECTOR_RESET_VALUE_ZERO;
	staTidBitmap.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_STATIONS_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->staTidBitmap[i].word0 = staTidBitmap.word0;
		pTxSelectorRam->staTidBitmap[i].word1 = staTidBitmap.word1;
	}
}

static void	txSelectorResetVapTidRam(void)
{
	TxSelectorVapTidRam_t vapTidBitmap;
	uint8 i = 0;

	vapTidBitmap.vapTidRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	vapTidBitmap.vapTidRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_VAP_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->vapTidBitmap[i].vapTidRamWord0 = vapTidBitmap.vapTidRamWord0;
	}
}

static void	txSelectorResetGlobalRam(void)
{
	TxSelectorGlobalRam_t globalBitmap;
	uint8 i = 0;

	globalBitmap.globalRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	globalBitmap.globalRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_GLOBAL_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->globalBitmap[i].globalRamWord0 = globalBitmap.globalRamWord0;
	}
}

static void	txSelectorResetGplpRam(void)
{
	TxSelectorGplpGphpRam_t gplpBitmap;
	uint8 i = 0;

	gplpBitmap.gplpGphpRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	gplpBitmap.gplpGphpRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_GPLP_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->gplpBitmap[i].gplpGphpRamWord0 = gplpBitmap.gplpGphpRamWord0;
	}
}

static void	txSelectorResetGphpRam(void)
{
	TxSelectorGplpGphpRam_t gphpBitmap;
	uint8 i = 0;

	gphpBitmap.gplpGphpRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	gphpBitmap.gplpGphpRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_GPHP_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->gphpBitmap[i].gplpGphpRamWord0 = gphpBitmap.gplpGphpRamWord0;
	}
}

static void	txSelectorResetBeaconRam(void)
{
	TxSelectorBeaconRam_t beaconBitmap;
	uint8 i = 0;

	beaconBitmap.beaconRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	beaconBitmap.beaconRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_BEACON_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->beaconBitmap[i].beaconRamWord0 = beaconBitmap.beaconRamWord0;
	}
}


/**********************************************************************************

txSelectorResetMuStaRam 

Description:
------------
	Reset selector MU STA Ram to zeros. This action disables MU Primary for all STA and disables all TIDs for MU Tx
Input:
-----
	uint32 numOfResetEntries - Number of entries in the ram.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorResetMuStationRam(uint32 numOfResetEntries)
{
	uint32 i = 0;
	TxSelectorPrimaryStaInfo_u primaryStaInfo;
	RegTxSelectorMuTidBitmap_u tidBitmap;

	primaryStaInfo.bitFields.groupIdx = 0x0;
	primaryStaInfo.bitFields.isPrimaryInGroup = FALSE;
	tidBitmap.val = 0x0;

	/* Set the registers values (Same for all entries)*/

	for(i = 0; i < numOfResetEntries; i++)
	{		
		TxSelector_ModifyPrimaryStaInfo(i, primaryStaInfo);
		TxSelector_ConfigureStaMuEnabledTids(i, tidBitmap);
	}
}



/**********************************************************************************

txSelectorResetMuGroupRam 

Description:
------------
	Reset selector MU Group Ram to zeros. This action disables all USPs in the group, disables Tx Mode, all groups are unlocked.
Input:
-----
	uint32 numOfResetEntries - Number of entries in the ram.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorResetMuGroupRam(uint32 numOfResetEntries)
{
	RegTxSelectorAllStaBitmapUpdate_u allStaInfo;
	uint32 i = 0;
	TxSelectorMuActionParams_t muBitmapParams;

	memset(&muBitmapParams, 0, sizeof(muBitmapParams));
	
	muBitmapParams.action = TX_SELECTOR_LOCK_GROUP;
	
	allStaInfo.val = 0;

	/* Set the registers values (Same for all entries)*/

	for(i = 0; i < numOfResetEntries; i++)
	{
		TxSelector_MapAllStasToGroup(i, allStaInfo);
		
		/* Lock all groups on initialization */
		muBitmapParams.groupIdx = i;
		txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_LOCK);
		
		TxSelector_UpdateGroupTxMode(i, TX_SELECTOR_GROUP_DISABLED);
	}
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif



/**********************************************************************************

txSelectorSetWrrPriority 

Description:
------------
	Configure the WRR priority for the selected entry 
	For each vap we can configure WRR between its Sta PS source & Sta active source,
	Each WRR type(PS,Active) is spread across 2 registers, in each register there 
	priorities for 8 vaps.
Input:
-----
	uint8 vapNum - Vap Number 
	uint8 wrrPriority - WRR Priority value
	uint32 baseAddress - The first register address according to the WRR type	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorSetWrrPriority(uint8 vapNum,uint8 wrrPriority,uint32 baseAddress)
{
	RegTxSelectorTxSelStaPsWrrLow_u wrrReg;
	
	uint32 wrrRegAddress = 0;
	uint32 wrrPrioritymask = 0;
	uint8 wrrPriorityBitOffset = 0;
	uint8 wrrPriorityRegisterOffset = 0;
	
	wrrReg.val = 0;
	/* the lower 3 bits determine the offset of the vap priority in the register */
	wrrPriorityBitOffset = (vapNum & TX_SELECTOR_NUM_OF_VAPS_IN_WRR_REGISTER_MASK)* TX_SELECTOR_WRR_VAP_PRIORITY_WIDTH;
	/* the 4 bit  (MSB) determines if we set a priority of a vap in the range 0-7 or in the range 8-15 
	   if vap num > 7 then it should be set in High register therefore offset of register is 1. else it is 0 */ 
	wrrPriorityRegisterOffset = (vapNum >> TX_SELECTOR_VAP_MSB_SHIFT) * HW_REG_SIZE; 
	/* Register Address for updating wrr priority */
	wrrRegAddress = baseAddress + wrrPriorityRegisterOffset;
	/**/
    RegAccess_Read(wrrRegAddress,&wrrReg.val);
	wrrPrioritymask = ~(TX_SELECTOR_WRR_VAP_PRIORITY_MASK << wrrPriorityBitOffset);
	/* set the WRR priority of the relevant vap */
	wrrReg.val  = ((wrrReg.val & wrrPrioritymask) | (wrrPriority << wrrPriorityBitOffset));
    RegAccess_Write( wrrRegAddress, wrrReg.val);
}

/**********************************************************************************

txSelectorFillSelectionparams 

Description:
------------
	Fill the selection parameters
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelection - Selection details
	uint32 selectionRegAddress	- The register address of the required source selection details
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorFillSelectionParams(TxSelectorSelectionDetails_t* selectorSelection,uint32 selectionRegAddress)
{
	RegTxSelectorTxSelSelection_u selectionReg;

	selectionReg.val = 0;

    RegAccess_Read(selectionRegAddress,&selectionReg.val);

	selectorSelection->selectionSource = selectionReg.bitFields.selectionSrcNum;
	selectorSelection->tid = selectionReg.bitFields.selectionTidNum;
	selectorSelection->staNum = selectionReg.bitFields.selectionStaNum;
	selectorSelection->vapNum = selectionReg.bitFields.selectionVapNum;
	selectorSelection->beamforming = selectionReg.bitFields.selectionBf;
	selectorSelection->accessCategory = selectionReg.bitFields.selectionAcNum;
	selectorSelection->validSelection = selectionReg.bitFields.selectionValid;
	selectorSelection->emptySelection= selectionReg.bitFields.selectionEmpty;
}


/**********************************************************************************

txSelectorFillMuSelectionParams 

Description:
------------
	Fill the selection parameters
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelection - Selection details
	uint32 selectionRegAddress	- The register address of the required source selection details
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorFillMuSelectionParams(TxSelectorMuSelectionDetails_t* selectorMuSelection, uint32 selectionRegAddress)
{
	RegTxSelectorTxSelMuUp0Result_u muSelectionReg;

	muSelectionReg.val = 0;

    RegAccess_Read(selectionRegAddress,&muSelectionReg.val);

	selectorMuSelection->tid = muSelectionReg.bitFields.muUp0Tid;
	selectorMuSelection->vapNum = muSelectionReg.bitFields.muUp0Vap;
	selectorMuSelection->staNum = muSelectionReg.bitFields.muUp0Sta;
	selectorMuSelection->beamforming = muSelectionReg.bitFields.muUp0BfReq;
	selectorMuSelection->accessCategory = muSelectionReg.bitFields.muUp0AcId;
	selectorMuSelection->txqId= muSelectionReg.bitFields.muUp0TxqId;
	selectorMuSelection->validSelection = muSelectionReg.bitFields.muUp0Valid;
}


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

TxSelector_ExecuteInstruction 

Description:
------------
	Execute an instruction of the Tx Selector.
Input:
-----
	TxSelectorInstructions_e instruction - The instruction to execute
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ExecuteInstruction (TxSelectorInstructions_e instruction)
{
	RegTxSelectorTxSelInstr_u instructionReg;

	ASSERT(instruction < TX_SELECTOR_NUM_OF_INSTRUCTION);

	instructionReg.val = 0;
	instructionReg.bitFields.instr = instruction;
	
    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_INSTR,instructionReg.val);
}

/**********************************************************************************

TxSelector_EnableQueue 

Description:
------------
	This procedure disables/enables  the queue specified by the parameters queue, stationVapNum, tidNum
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		- The queue type to disable (STA,VAP,VIP,GLOBAL,SUPER)
		- The VAP number if the queue is VAP or STA number if the queue is STA
		- The tid number of the queue to be disabled
		- TxSelectorEnableDisableQ_e disable - 0x0 - enable , 0x1 - disable
		- Indicates if we should perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_EnableQueue(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorActionParams->action = TX_SELECTOR_ENABLE_QUEUE;
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_DISABLE);
}

/**********************************************************************************

TxSelector_SetBfSequenceRequest 

Description:
------------
	This procedure sets the BF request bit, next time the related Q will be chosen a BF sequnce will be transmitted

Input:
-----
		- 
		- Indicates if we should perform action for all TIDs\ACs
		
		-- queue type  -  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		-- VAP number if the queue is VAP or STA number if the queue is STA
		-- tid\AC number of the queue to be disabled
		-- Action:
				0x0 - Reset BF Request   
				0x1 - Set BF Request
		-- allTids Indicates if we should perform action for all TIDs\ACs
		
**********************************************************************************/
void TxSelector_SetBfSequenceRequest(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_BF_SEQUENCE_REQUEST);
}

/**********************************************************************************

TxSelector_SetBfSequenceType 

Description:
------------
	This procedure sets the BF sequence type: 
		0 - without data	
		1 - with data: 		
		
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		-- queue type  -  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		-- VAP number if the queue is VAP or STA number if the queue is STA
		-- tid\AC number of the queue to be disabled
		-- Action:
				0x0 - BF Sequence with no data transmission following it
				0x1 - BF sequence is followed by data transmission  
		-- allTids Indicates if we should perform action for all TIDs\ACs

**********************************************************************************/
void TxSelector_SetBfSequenceType(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_BF_SEQUENCE_TYPE);
}

/**********************************************************************************

TxSelector_UnlockQueue 

Description:
------------
	This procedure locks/unlocks the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		- The queue type to lock (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		- the VAP number if the queue is VAP or STA number if the queue is STA
		- the tid\AC number of the queue to be disabled
		- 0x0 - unlock , 0x1 - lock
		- Indicates if we should perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_UnlockQueue(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorActionParams->action = TX_SELECTOR_UNLOCK_Q;
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_LOCK);
}


/**********************************************************************************

TxSelector_LockRequest 

Description:
------------
	Request for the HW Locker to execute lock request (Only force lock request is done by FW modules)
Input:
-----
	txSelectorLockRequestParams - Parameters of the lock request

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_LockRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams)
{
	RegTxSelectorTxSelSwMaster2Locker_u lockerReg;

	lockerReg.val = 0;

	lockerReg.bitFields.master2LockerStaSel = txSelectorLockRequestParams->stationOrVapNum;
	lockerReg.bitFields.master2LockerTidSel = txSelectorLockRequestParams->tidOrAc;
	lockerReg.bitFields.master2LockerBitmapSel = txSelectorLockRequestParams->queueType;
	lockerReg.bitFields.master2LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK;
	lockerReg.bitFields.master2LockerAllTids = txSelectorLockRequestParams->allTids;

	RegAccess_Write(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val);
	
}

/**********************************************************************************

TxSelector_DisableRequest 

Description:
------------
	
Input:
-----

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_DisableRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams)
{
	RegTxSelectorTxSelSwMaster2Locker_u lockerReg;

	lockerReg.val = 0;

	lockerReg.bitFields.master2LockerStaSel = txSelectorLockRequestParams->stationOrVapNum;
	lockerReg.bitFields.master2LockerTidSel = txSelectorLockRequestParams->tidOrAc;
	lockerReg.bitFields.master2LockerBitmapSel = txSelectorLockRequestParams->queueType;
	lockerReg.bitFields.master2LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_DISABLE;
	lockerReg.bitFields.master2LockerAllTids = txSelectorLockRequestParams->allTids;

	RegAccess_Write(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val);
}


/**********************************************************************************

TxSelector_LockGroupRequest 

Description:
------------
	Request for the HW Locker to execute lock request (Only force lock request is done by FW modules)
Input:
-----
	txSelectorLockRequestParams - Parameters of the lock request

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_LockGroupRequest(uint32 groupId)
{
	RegTxSelectorTxSelMuSwMaster2Locker_u lockerReg;

	lockerReg.val = 0;

	lockerReg.bitFields.master2MuLockerGrpSel = groupId;
	lockerReg.bitFields.master2MuLockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK;

	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_SW_MASTER2_LOCKER, lockerReg.val);	
}


/**********************************************************************************

TxSelector_SetPowerSaveType 

Description:
------------
	This procedure Set power save type of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save type  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid\AC number of the queue 
	- Power Save type 0x0 - Legacy , 0x1 - WMM
	- Should we perform action for all TIDs\ACs
		
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetPowerSaveType(TxSelectorActionParams_t *txSelectorActionParams)
{
	AccessCategoryTo8021d_t *acTo8021dConversionTable;
	uint8 accessCategory;

	accessCategory = txSelectorActionParams->tidOrAc;
	acTo8021dConversionTable = Protocol_GetAccessCategoryTo8021dConversionTable();
		
	txSelectorActionParams->tidOrAc = acTo8021dConversionTable->val8021d[accessCategory][FIRST_TID_PER_AC_INDEX];
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_POWER_SAVE_TYPE);

	txSelectorActionParams->tidOrAc = acTo8021dConversionTable->val8021d[accessCategory][SECOND_TID_PER_AC_INDEX];
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_POWER_SAVE_TYPE);
}
	
/**********************************************************************************

TxSelector_SetPowerSavemode 

Description:
------------
	This procedure Set power save mode of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid\AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetPowerSavemode(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_POWER_SAVE_MODE);
}


/**********************************************************************************

TxSelector_ClearPowerSavemode 

Description:
------------
	This procedure Clear power save mode of the queue specified by the parameters queue, stationVapNum, tidNum
	Clear and Set use the same API functions. The seperation is required for gen4.
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid\AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_ClearPowerSavemode(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_POWER_SAVE_MODE);
}



/**********************************************************************************

TxSelector_SetPowerSaveRequest

Description:
------------
	This procedure Set power save request of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid\AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetPowerSaveRequest(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams, TX_SELECTOR_PS_REQUEST);
}

/**********************************************************************************

TxSelector_SetDataInQ


Description:
------------
	This procedure Set/Reset Data in Q of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set Data In Q (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid\AC number of the queue 
	- TX_SELECTOR_SET_DATA_IN_Q 0x0 - Reset Data In Q , 0x1 - Set Data In Q
	- Should we perform action for all TIDs\ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetDataInQ(TxSelectorActionParams_t *txSelectorActionParams)
{
    txSelectorUpdateBitmap(txSelectorActionParams, TX_SELECTOR_DATA_IN_Q);
}

/**********************************************************************************
TxSelector_AddStaToVap 

Description:
------------
	Add or remove  Station. 
		Add		- Add station to VAPs associated stations bitmap and Enable the 8 STA TID queues.
		Remove	- Remove station from VAPs associated stations bitmap and disable the 8 STA TID queues.
Input:
-----
	StaId stationId - The station to be removed.
	uint8 vapId		- The station associated vap. 
	uint8 addOrRemove	- Remove or add station action.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetStationToVapAssocState(StaId stationId, uint8 vapId)
{	
	RegTxSelectorTxSelStaAssociate_u staAssociateReg;
	
	staAssociateReg.val = 0;

	/* Add station to associated stations to vap bitmap*/
	staAssociateReg.bitFields.staAssociateStation = stationId;
	staAssociateReg.bitFields.staAssociateVap = vapId;
	staAssociateReg.bitFields.staAssociateVapStaBitValue = TX_SELECTOR_ASSOCIATE;
    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val);

}
/**********************************************************************************
TxSelector_ClearStationFromVapAssocState 

Description:
------------
	Add or remove  Station. 
		Add		- Add station to VAPs associated stations bitmap and Enable the 8 STA TID queues.
		Remove	- Remove station from VAPs associated stations bitmap and disable the 8 STA TID queues.
Input:
-----
	StaId stationId - The station to be removed.
	uint8 vapId		- The station associated vap. 
	uint8 addOrRemove	- Remove or add station action.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ClearStationFromVapAssocState(StaId stationNum,uint8 vapId)
{	
	RegTxSelectorTxSelStaAssociate_u staAssociateReg;
	
	staAssociateReg.val = 0;

	/* Add station to associated stations to vap bitmap*/
	staAssociateReg.bitFields.staAssociateStation = stationNum;
	staAssociateReg.bitFields.staAssociateVap = vapId;
	staAssociateReg.bitFields.staAssociateVapStaBitValue = TX_SELECTOR_DISASSOCIATE;
    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val);

}

/**********************************************************************************

TxSelector_SetVapWrrPriority 

Description:
------------
	This configuration is used by the TX Selector in the 2 ports weighted round robin performed 
	for determining which source will be selected for input A of the strict priority selector.
	there are two registers for STA PS priority. TX_SELECTOR_STA_PS_WRR_LOW holds    
	priorities for stations associated with VAP 0-7, TX_SELECTOR_STA_PS_WRR_HIGH  
	holds priorities for stations associated with VAP 9-15. VAP priority can be 0-7
	and requires 3 bits.
Input:
-----
	uint8 vapNum - Vap Number 
	uint8 wrrPsPriority	- WRR Ps Priority value	
	uint8 wrrActivePriority - WRR Active Priority value	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetVapWrrPriority(uint8 vapNum,TxSelectorWrrPriority_t wrrPsPriority, TxSelectorWrrPriority_t wrrActivePriority)
{
	txSelectorSetWrrPriority(vapNum,wrrPsPriority,REG_TX_SELECTOR_TX_SEL_STA_PS_WRR_LOW);
	txSelectorSetWrrPriority(vapNum,wrrActivePriority,REG_TX_SELECTOR_TX_SEL_STA_ACTIVE_WRR_LOW);
}

/**********************************************************************************

TxSelector_SetStrictPriority 

Description:
------------
	Configures priority of between Global Q Vap MPS & Sta .
			1 - Highest Priority
			3 - Lowest Priority
	In order for priority to function properly all 3 priorities must be set with different values.
Input
-----
	TxSelectorStrictPriority_e globalPriority - Global source priority.
	TxSelectorStrictPriority_e vapMpsPriority - Vap MPS source priority.
	TxSelectorStrictPriority_e activePriority - Active source priority - Include Active vaps, Active Station & PS Stations.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetStrictPriority(TxSelectorStrictPriority_e globalPriority, TxSelectorStrictPriority_e vapMpsPriority, TxSelectorStrictPriority_e activePriority)
{
	RegTxSelectorTxSelStrictPrio_u strictPriorityReg;

	strictPriorityReg.val = 0;
    RegAccess_Read(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,&strictPriorityReg.val);
	strictPriorityReg.bitFields.strictPrioGlobalPrio = globalPriority;
	strictPriorityReg.bitFields.strictPrioVapMpsPrio = vapMpsPriority;
	strictPriorityReg.bitFields.strictPrioWrrPrio = activePriority;

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,strictPriorityReg.val);
}

/**********************************************************************************

TxSelector_MaskGroupSelection 

Description:
------------
	Bits [4:0] mask possible selections at the strict priority selection stage. 
	Bit value 0 - selection is masked, Bit value 1- selection is not masked. 
	Bit 0 - STA TID , Bit2 - VAP TID ,Bit2 - Global 
	Bit3 - GPHP		, Bit4 - GPLP	 ,Bit5 - Beacon
Input:
-----
	TxSelectorGroupSelectionOffset_e gruopOffset
	TxSelectorGroupSelectionEnable_e selectionEnable
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void TxSelector_MaskGroupSelection(TxSelectorGroupSelectionOffset_e gruopOffset,TxSelectorGroupSelectionEnable_e selectionEnable)
{
	RegTxSelectorTxSelStrictPrio_u groupSelectionReg;

	groupSelectionReg.val = 0;

    RegAccess_Read(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,&groupSelectionReg.val);

    groupSelectionReg.bitFields.strictPrioGroupSelEn =  selectionEnable << gruopOffset;

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,groupSelectionReg.val);
}

/**********************************************************************************

TxSelector_GetLockRequestsStatus 

Description:
------------
	Returns the status of the last Lock Request 

Input:
-----
	None

Returns:
--------
	TxSelectorLockStatus_e - Request Status
	
**********************************************************************************/
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(void)
{
	RegTxSelectorTxSelSwMaster0LockerStatus_u lockerStatusReg;
	TxSelectorLockStatus_e lockerStatus;
	
	lockerStatusReg.val = 0;

	RegAccess_Read(REG_TX_SELECTOR_LOCKER_STATUS,&lockerStatusReg.val);
	lockerStatus = (TxSelectorLockStatus_e)lockerStatusReg.bitFields.swMaster0ReqStatus;

	return lockerStatus;
}		

/**********************************************************************************

TxSelector_GetLastSelection 

Description:
------------
	Returns the details of the TX selector last selection
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelectionDetails - - Selection details
Output:
-------
	The last selection details of the selector
Returns:
--------
	void
**********************************************************************************/
void TxSelector_GetLastSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_SELECTION;

	txSelectorFillSelectionParams(selectorSelectionDetails,selectionRegAddress);
}


/**********************************************************************************

TxSelector_GetLastSourceSelection 

Description:
------------
	Returns the details of the specified source. The TX selector enters to the strict priority
	selector sources 0-5. This routine will return the details of the specified source.
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelectionDetails
	TxSelectorSource_e sourceNum - T
Output:
-------
	The selection Details according to the source
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_GetLastSourceSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails,TxSelectorSource_e sourceNum)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_SRC0_SELECTION + (sourceNum * (REG_TX_SELECTOR_TX_SEL_SRC1_SELECTION - REG_TX_SELECTOR_TX_SEL_SRC0_SELECTION));

	txSelectorFillSelectionParams(selectorSelectionDetails, selectionRegAddress);
}		



/**********************************************************************************

TxSelector_GetLastMuSelection 

Description:
------------
	Returns the details of the MU TX selector last selection of specified USP
Input:
-----
	TxSelectorMuSelectionDetails_t * selectorMuSelectionDetails - Selection details
	uint8 userPosition - user position to read
Output:
-------
	The last selection details of the MU selector
Returns:
--------
	void
**********************************************************************************/
void TxSelector_GetLastMuSelection(TxSelectorMuSelectionDetails_t* selectorMuSelectionDetails, uint8 userPosition)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_MU_UP0_RESULT + (userPosition * (REG_TX_SELECTOR_TX_SEL_MU_UP1_RESULT - REG_TX_SELECTOR_TX_SEL_MU_UP0_RESULT));
	
	txSelectorFillMuSelectionParams(selectorMuSelectionDetails, selectionRegAddress);

}

/**********************************************************************************

TxSelector_IsDataInQueue 


Description:
------------
Returns if there is a data in a specific queue


Input:
-----
stationOrVapIndex - the index of the station/VAP
tid - the TID
queueType - queue type (should be only station or VAP)



Output:
-------
	

Returns:
--------
Indication if data exists on the specific queue

	
**********************************************************************************/
uint8 TxSelector_IsDataInQueue(uint8 stationOrVapIndex, uint8 tid, TxQueueType_e queueType)
{
	FATAL("TxSelector_IsDataInQueue - back door not available in gen5b");
	return FALSE;

}



/**********************************************************************************

TxSelector_TryLockRequest 


Description:
------------
This function tries to lock a queue and if it is locked it returns immidiately.
This function should be used only by the Ager emulator.

Input:
-----
stationOrVapIndex - the index of the station/VAP
tid - the TID
queueType - queue type (should be only station or VAP)



Output:
-------
	

Returns:
--------
The lock request status

	
**********************************************************************************/
TxSelectorLockStatus_e TxSelector_TryLockRequest(uint8 stationOrVapIndex , uint8 tid, TxQueueType_e queueType)
{
	RegTxSelectorTxSelSwMaster1Locker_u lockParametersRegister;
	RegTxSelectorTxSelSwMaster1LockerStatus_u lockStatusRegister;

	lockParametersRegister.val = 0;
	lockStatusRegister.val = 0;

	lockParametersRegister.bitFields.master1LockerStaSel = stationOrVapIndex;
	lockParametersRegister.bitFields.master1LockerTidSel = tid;
	lockParametersRegister.bitFields.master1LockerBitmapSel = queueType;
	lockParametersRegister.bitFields.master1LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_TRY_LOCK;
	lockParametersRegister.bitFields.master1LockerAllTids = TX_SELECTOR_SINGLE_TID_UPDATE;

	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER, lockParametersRegister.val);

	/* Wait till result is ready */
	while(!lockStatusRegister.bitFields.master1LockerDoneInt)
	{
		RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS,&lockStatusRegister.val);
	}

	return ((TxSelectorLockStatus_e)lockStatusRegister.bitFields.swMaster1ReqStatus);
}

/**********************************************************************************

isr_TxSelector_Error 

Description:
------------
	Isr for error scenarios in the TX Selector

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
void isr_TxSelector_Error(void)
{
	FATAL("ISR TX Selector Error");
}

/**********************************************************************************

isr_TxSelector_LockerRequestServiced 

Description:
------------
	Isr when Locker has finished the requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
ISR_VOID isr_TxSelector_LockerRequestServiced(void)
{
	RegTxSelectorTxSelLockerIntClr_u lockerIntClearReg;
	RegTxSelectorTxSelSwMaster2LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
		
	lockerIntClearReg.val = 0;
	
	ACCUMULATE_CPU_IDLE_TIME();
	
	
#if defined (ENET_INC_UMAC)
	/* Read  lock request status */
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER_STATUS,&lockerStatus.val);
	lockerIntClearReg.bitFields.swMaster2LockerIntClr = 1;
#else
	/* Read  lock request status */
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER_STATUS,&lockerStatus.val);
	lockerIntClearReg.bitFields.swMaster3LockerIntClr = 1;
#endif
	/* Clear  locker interrutpt */
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOCKER_INT_CLR,lockerIntClearReg.val);

	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster2ReqStatus;
		
	Locker_LockRequestServiced(lockRequestStatus);

}


/**********************************************************************************

TxSelector_ModifyPrimaryStaInfo


Description:
------------
This function configures primary STA for specified group. The change will be reflected to STA MU DB in Selector's RAM

Input:
-----
staId - the sta id to map
primaryStaInfo :
	- groupIdx - group index to map STA to
	- isPrimaryStaInGroup - determines whether the STA will be primary in the mapped group (if not and this STA will be chosen by SU selector - it will not continue as MU Tx).

Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_ModifyPrimaryStaInfo(StaId staId, TxSelectorPrimaryStaInfo_u primaryStaInfo)
{

	TxSelectorMuActionParams_t muBitmapParams;

	muBitmapParams.allStas = TX_SELECTOR_SINGLE_STA_UPDATE;
	muBitmapParams.groupIdx = staId;
	muBitmapParams.action = primaryStaInfo.val;
	
	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_STA_GROUP_MAPPING);
}


/**********************************************************************************

TxSelector_MapAllStasToGroup


Description:
------------
This function maps 4 STAs to specified group in MU Group DB

Input:
-----
groupIdx - the sta id to map
allStaInfo : TxSelectorGroupUserPositionStaInfo_t per user position : Sta id and valid bit

Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_MapAllStasToGroup(uint8 groupIdx, RegTxSelectorAllStaBitmapUpdate_u allStaInfo)
{

	TxSelectorMuActionParams_t muBitmapParams;

	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE_ALL_STA_WDATA, allStaInfo.val);

	muBitmapParams.allStas = TX_SELECTOR_ALL_STAS_UPDATE;
	muBitmapParams.groupIdx = groupIdx;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_USER_POSITION_STA);

}


/**********************************************************************************

TxSelector_UpdateStaInGroup


Description:
------------
This function updates inputted STA to be in the requested user position in the group and sets it as primary if requested

Input:
-----
groupIdx - the  group index to add sta to
staInfo :
		- sta id - STA id to add/modify
		- is valid - is the STA id valid in the Group
		
staPrimaryEnabled - indicate whether need to update the STA also as primary in the Group


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_UpdateStaInGroup(uint8 groupIdx, uint8 userPosition, TxSelectorGroupUserPositionStaInfo_u staInfo)
{
	TxSelectorMuActionParams_t muBitmapParams;

	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.userPositionIdx = userPosition;
	muBitmapParams.action = staInfo.val;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_USER_POSITION_STA);
}


/**********************************************************************************

TxSelector_UpdateGroupTxMode


Description:
------------
This function updates Group Tx mode

Input:
-----
groupIdx - the  group index
tx mode - the Tx mode to update the group to


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_UpdateGroupTxMode(uint8 groupIdx, TxSelectorGroupTxMode_e txMode)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.action = txMode;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_TX_MODE);
}


/**********************************************************************************

TxSelector_ConfigureStaMuEnabledTids


Description:
------------
This function configures the enabled tids for MU Tx for specified STA

Input:
-----
staId - the  STA ID to update
tidBitmap - 8 bit per TID : 1- Enable, 0- Disable


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_ConfigureStaMuEnabledTids(StaId staId, RegTxSelectorMuTidBitmap_u tidBitmap)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.allStas = TX_SELECTOR_SINGLE_STA_UPDATE;
	muBitmapParams.userPositionIdx = 0;
	muBitmapParams.groupIdx = staId;
	muBitmapParams.action = tidBitmap.val;
	
	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_STA_MU_TID_ENABLE);
}


/**********************************************************************************

TxSelector_UnlockGroup


Description:
------------
This function unlocks group
Input:
-----
groupIdx - the  group index to unlock


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_UnlockGroup(uint8 groupIdx)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.action = TX_SELECTOR_UNLOCK_GROUP;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_LOCK);

}



/**********************************************************************************

TxSelector_EnableDisableMuTx



Description:
------------
This function enables/disables MU for the entire system, with the following fetures:

- determining group threshold
- determining secondary tid selection method
- determining lock policy after selection

Input:
-----
maxGroupNum - max number of groups - for selection of group above the threshold, the Tx will switch to SU
muSecondaryTid - secondary tid selection method
muLockPolicy - mu lock policy


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_EnableDisableMuTx(TxSelectorEnableDisableMu_e muEnableDisable, uint8 maxGroupNum, TxSelectorMuSecondarySelectionMethod_e muSecondaryTid, TxSelectorMuLockAfterSelection_e muLockPolicy)
{

	RegTxSelectorTxSelMuSelectCfg_u muSelectConfigReg;
		
	muSelectConfigReg.val = 0;
	muSelectConfigReg.bitFields.muEn = muEnableDisable;
	muSelectConfigReg.bitFields.muSecondarySel = muSecondaryTid;
	muSelectConfigReg.bitFields.maxGroupNum = maxGroupNum;
	muSelectConfigReg.bitFields.selectorMuLockEn = muLockPolicy;
	
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_SELECT_CFG, muSelectConfigReg.val);
}


/**********************************************************************************

TxSelector_SetPrimaryPdCountThreshold


Description:
------------
This function sets pd count primary threshold


Input:
-----
threshold - Primary STA queue should include at least this number of PDs in order for Tx to be MU


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_SetPrimaryPdCountThreshold(uint16 threshold)
{
	RegTxSelectorTxSelMuPrimaryThres_u primaryThreshold;	
	
	primaryThreshold.bitFields.pdPrimaryThres = threshold;	
	
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_PRIMARY_THRES, primaryThreshold.val);

}



/**********************************************************************************

TxSelector_SetSecondaryPdCountThreshold



Description:
------------
This function sets pd count secondary threshold


Input:
-----
threshold - Secondary STA queue should include at least this number of PDs in order for the STA to be included in MU Tx


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_SetSecondaryPdCountThreshold(uint16 threshold)
{
	RegTxSelectorTxSelMuSecondaryThres_u secondaryThreshold;	
		
	secondaryThreshold.bitFields.pdSecondaryThres = threshold;	
		
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_SECONDARY_THRES, secondaryThreshold.val);

}

/**********************************************************************************

isr_TxSelector_LockerGroupRequestServiced 

Description:
------------
	Isr when Locker has finished the requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
void isr_TxSelector_LockerGroupRequestServiced(void)
{
	RegTxSelectorTxSelMuSwMaster2LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
	RegTxSelectorTxSelMuLockerIntClr_u	lockClrInterruptReg;


	/* Read  lock request status */
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_MU_SW_MASTER2_LOCKER_STATUS,&lockerStatus.val);
	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster2MuReqStatus;

	/* Clear  lock interrupt status */
	lockClrInterruptReg.val = 0; 
	lockClrInterruptReg.bitFields.swMaster2MuLockerIntClr = 1; 
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_LOCKER_INT_CLR, lockClrInterruptReg.val);
	

	Locker_LockGroupRequestServiced(lockRequestStatus);

}

/**********************************************************************************

TxSelector_Init 

Description:
------------
	Init vapSta bitmaps - no stations area associated with vaps.
	Set Sta & VAp rams to default values (Q disabled and all other values are 0)
	GLOBAL,GPHP,GPLP,BEACON queues bitmap are all stored in FF, the sw reset initializes these FFF to default values
	Init MU sTA & Group bitmaps
	Enable SU & MU Lockers
	Set Primary and secondary PD thresholds for MU transmission
Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
#if defined (ENET_INC_LMAC)
#pragma ghs section text=".initialization" 
#endif
void TxSelector_Init(void)
{
	RegTxSelectorTxSelLockerEn_u lockerEnReg;
	RegTxSelectorTxSelMuLockerEn_u muLockerEnReg;
	RegTxSelectorTxSelStaAssociate_u selStaAssociateReg;

	lockerEnReg.val = 0;
	muLockerEnReg.val = 0;
	selStaAssociateReg.val = 0;

	/* Reset Selector using SW Reset Block*/

	
	/* Initialize the pointer of the Tx Selector Ram */
	pTxSelectorRam = (TxSelectorRam_t *)(TX_SELECTOR_BASE_ADDRESS + TX_SELECTOR_RAM_OFFSET_FROM_BASE_ADDRESS);

	/* Reset & Disable SU Bitmaps */
	txSelectorResetStationTidRam();
	txSelectorResetVapTidRam();
	txSelectorResetGlobalRam();
	txSelectorResetGplpRam();
	txSelectorResetGphpRam();
	txSelectorResetBeaconRam();


	/* Reset Vap\Sta Bitmaps  */
	selStaAssociateReg.bitFields.vapZero = TX_SELECTOR_VAP_ZERO_STA_BITMAP;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE,selStaAssociateReg.val);

	/* Reset MU Group Bitmap */	
	txSelectorResetMuGroupRam(TX_SELECTOR_NUM_OF_MU_GROUP_ENTRIES_IN_RAM);
	
	/* Reset MU STA Bitmap */
	txSelectorResetMuStationRam(TX_SELECTOR_NUM_OF_MU_STA_ENTRIES_IN_RAM);
	

	/* Enable SU Locker */
	lockerEnReg.bitFields.lockerEn = TX_SELECTOR_LOCKER_ENABLE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOCKER_EN,lockerEnReg.val);

	/* Enable MU Locker */
	muLockerEnReg.bitFields.muLockerEn = TX_SELECTOR_LOCKER_ENABLE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_LOCKER_EN, muLockerEnReg.val);

	//set PD thresholds
	TxSelector_SetPrimaryPdCountThreshold(TX_SELECTOR_PRIMARY_PD_COUNT_THRESHOLD);
	TxSelector_SetSecondaryPdCountThreshold(TX_SELECTOR_SECONDARY_PD_COUNT_THRESHOLD);

	//Enable MU Tx
	TxSelector_EnableDisableMuTx(TX_SELECTOR_MU_ENABLE, (TX_MU_GROUPS - 1), TX_SELECTOR_MU_ANY_SECONDARY_TID_ALLOWED, TX_SELECTOR_MU_LOCK_AFTER_SELECTION);
	
}
#if defined (ENET_INC_LMAC)
#pragma ghs section text=default
#endif

