/***********************************************************************************
 File:			TxSelector.c
 Module:		Tx Selector
 Purpose:		
 Description:	FW driver for TX Selector module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwGlobalDefinitions.h"
#include "TxSelector_Api.h"
#include "TxSelectorRegs.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "BSSmanager_API.h"
#include "Locker_Api.h"
#include "OSAL_Tasks.h"
#include "ieee80211.h"
#include "Protocol_WmeDefinitions.h"
#include "ShramGroupDatabase.h"
#include "loggerAPI.h"
#include "TxSelectorRam.h"
#include "CpuLoad_Api.h"
#include "ConfigurationManager_api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID 6

/* Locker  Configuration */
#define TX_SELECTOR_LOCKER_ENABLE							(1)
#define TX_SELECTOR_RESET_VALUE_ZERO						(0)
/* Used in TxSelector_SetStaWrr based on WRR Registers 0x8 - 0x10 */
#define TX_SELECTOR_WRR_VAP_PRIORITY_WIDTH 					(3)  /* there are 3 bits for every vap priority. Priority is in the range 0-7 */
#define TX_SELECTOR_WRR_VAP_PRIORITY_MASK				    (0x7)   /* The Vap mask of the register is 0x7 */
#define TX_SELECTOR_VAP_MSB_SHIFT							(3)
#define TX_SELECTOR_NUM_OF_VAPS_IN_WRR_REGISTER_MASK		(0x7) /* used to determine if priority is set to LOW0/1 or HIGH0/1 register  */	
#define TX_SELECTOR_NO_DATA_IN_Q							(0)
/* Vap Ac Matrix Definition Based On Vap Matrix Registers 0x68-0x74*/
#define TX_SELECTOR_VAP_AC_MATRIX_VAP_WIDTH					(8)	/* There are 7 bit for each vap + 1 reserved*/
#define TX_SELECTOR_VAP_AC_MATRIX_VAP_MASK					(0x7F)	
#define	TX_SELECTOR_VAP_AC_MATRIX_AC0_MASK					(0x1)
#define	TX_SELECTOR_VAP_AC_MATRIX_AC1_MASK    				(0x2)		
#define	TX_SELECTOR_VAP_AC_MATRIX_AC2_MASK					(0x4)		
#define	TX_SELECTOR_VAP_AC_MATRIX_AC3_MASK					(0x8)		
#define	TX_SELECTOR_VAP_AC_MATRIX_GPLP_MASK					(0x10)		
#define	TX_SELECTOR_VAP_AC_MATRIX_GPHP_MASK					(0x20)	
#define	TX_SELECTOR_VAP_AC_MATRIX_BEACON_MASK				(0x40)	


#define TX_SELECTOR_VAP_AC_MATRIX_REG_REG_OFFSET_SHIFT		(2)	/* There 4 Vaps in each register result a shift of 2 to determine the Register address offset */
#define TX_SELECTOR_VAP_AC_MATRIX_REG_BIT_OFFSET_MASK		(0x3) /* There 4 Vaps in each register result a mask of 2 to determine the bit offset in the register */	

#define TX_SELECTOR_VAP_ZERO_STA_BITMAP						(1)

#ifndef ENET_INC_ARCH_WAVE600D2
/* Used for polling TWT registers */
#define TX_SELECTOR_TWT_POLLING_TIMEOUT                     (5)
#define TX_SELECTOR_TWT_LONG_POLLING_TIMEOUT                (55) /* according to VLSI analysis */
#define REG_TX_SELECTOR_TWT_SP_MASK                         (1u<<31)
#define REG_TX_SELECTOR_TWT_SP_SHIFT                        (31u)
#endif

/* MU PD Count Thresholds Definition */
#define TX_SELECTOR_PRIMARY_PD_COUNT_THRESHOLD				(1) //was 10
#define TX_SELECTOR_SECONDARY_PD_COUNT_THRESHOLD			(1) //was 10

#if defined (ENET_INC_UMAC)
    #define REG_TX_SELECTOR_LOCKER_REQ 			(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER)
    #define REG_TX_SELECTOR_LOCKER_STATUS		(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS)
    #define REG_TX_SELECTOR_BITMAP_UPDATE_DATA	(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_DATA_MASTER1)
    #define REG_TX_SELECTOR_BITMAP_UPDATE		(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_MASTER1)
    #define REG_TX_SELECTOR_PLAN_LOCKER_REQ     (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER1_LOCKER)
    #define REG_TX_SELECTOR_PLAN_LOCKER_STATUS  (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER1_LOCKER_STATUS)
#else
    #if defined (ENET_INC_LMAC0)
        #define REG_TX_SELECTOR_LOCKER_REQ 			(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER)
        #define REG_TX_SELECTOR_LOCKER_STATUS		(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER_STATUS)
        #define REG_TX_SELECTOR_BITMAP_UPDATE_DATA	(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_DATA_MASTER2)
        #define REG_TX_SELECTOR_BITMAP_UPDATE		(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_MASTER2)
        #define REG_TX_SELECTOR_PLAN_LOCKER_REQ     (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER2_LOCKER)
        #define REG_TX_SELECTOR_PLAN_LOCKER_STATUS  (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER2_LOCKER_STATUS)
    #else //ENET_INC_LMAC1
        #define REG_TX_SELECTOR_LOCKER_REQ 			(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER)
        #define REG_TX_SELECTOR_LOCKER_STATUS		(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER_STATUS)
        #define REG_TX_SELECTOR_BITMAP_UPDATE_DATA	(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_DATA_MASTER3)
        #define REG_TX_SELECTOR_BITMAP_UPDATE		(REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE_MASTER3)
        #define REG_TX_SELECTOR_PLAN_LOCKER_REQ     (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER3_LOCKER)
        #define REG_TX_SELECTOR_PLAN_LOCKER_STATUS  (REG_TX_SELECTOR_TX_SEL_PLAN_SW_MASTER3_LOCKER_STATUS)
    #endif //ENET_INC_LMAC0
#endif //ENET_INC_UMAC

#ifndef ENET_INC_ARCH_WAVE600B
typedef enum TxSelectorBitTypeSelection
{
	TX_SELECTOR_DATA_IN_Q					=		0x0,
	TX_SELECTOR_LOCK						=		0x1,
	TX_SELECTOR_REQUEST 					=		0x2,
	TX_SELECTOR_PS_TYPE 					=		0x3,
	TX_SELECTOR_ACTIVE_PS_MODE				=		0x4,
	TX_SELECTOR_HE_MU_ENABLE				=		0x5,
	TX_SELECTOR_UL_DATA_IN_Q				=		0x6,
	TX_SELECTOR_DISABLE 					=		0x8,
	TX_SELECTOR_BF_SEQUENCE_REQUEST 		=		0x9,
	TX_SELECTOR_BF_SEQUENCE_TYPE			=		0xA,
	TX_SELECTOR_TWT_ANNOUNCED_TYPE			=		0xB, 
	TX_SELECTOR_TWT_SP						=		0xC,
	TX_SELECTOR_TWT_AVAILABILITY			=		0xD, 
	TX_SELECTOR_DL_PLAN_INDEX				=		0xE, 
	TX_SELECTOR_UL_PLAN_INDEX				=		0xF, 
	TX_SELECTOR_PLAN_LOCK					=		0x10,
	TX_SELECTOR_INSERT_A_CONTROL			=		0x11,
	TX_SELECTOR_MU_MIMO_GROUP_LOCK			=		0x12,
	TX_SELECTOR_MU_MIMO_TX_MODE_BF_REQUEST	=		0x13,
	TX_SELECTOR_OMI_VALID					=		0x14,
	TX_SELECTOR_OMI_SUPPORTED				=		0x15,	
	TX_SELECTOR_OMI_BITS					=		0x16,
	TX_SELECTOR_NUM_OF_BIT_TYPES,
	TX_SELECTOR_MAX_NUM_OF_BIT_TYPES 		= 		0xFF,
} TxSelectorBitTypeSelection_e;

#else // ENET_INC_ARCH_WAVE600B

typedef enum TxSelectorBitTypeSelection
{
	TX_SELECTOR_DATA_IN_Q					=		0x0,
	TX_SELECTOR_LOCK						=		0x1,
	TX_SELECTOR_REQUEST 					=		0x2,
	TX_SELECTOR_PS_TYPE 					=		0x3,
	TX_SELECTOR_ACTIVE_PS_MODE				=		0x4,
	TX_SELECTOR_DL_HE_MU_ENABLE				=		0x5,
	TX_SELECTOR_UL_DATA_IN_Q				=		0x6,
	TX_SELECTOR_DISABLE 					=		0x8,
	TX_SELECTOR_BF_SEQUENCE_REQUEST 		=		0x9,
	TX_SELECTOR_BF_SEQUENCE_TYPE			=		0xA,
	TX_SELECTOR_TWT_ANNOUNCED_TYPE			=		0xB, 
	TX_SELECTOR_TWT_SP						=		0xC,
	TX_SELECTOR_TWT_AVAILABILITY			=		0xD, 
	TX_SELECTOR_DL_PLAN_INDEX				=		0xE, 
	TX_SELECTOR_UL_PLAN_INDEX				=		0xF, 
	TX_SELECTOR_PLAN_LOCK					=		0x10,
	TX_SELECTOR_INSERT_A_CONTROL			=		0x11,
	TX_SELECTOR_MU_MIMO_GROUP_LOCK			=		0x12,
	TX_SELECTOR_MU_MIMO_TX_MODE_BF_REQUEST	=		0x13,
	TX_SELECTOR_OMI_VALID					=		0x14,
	TX_SELECTOR_OMI_SUPPORTED				=		0x15,	
	TX_SELECTOR_OMI_BITS					=		0x16,
	TX_SELECTOR_UL_HE_MU_ENABLE				=		0x17,
	TX_SELECTOR_NUM_OF_BIT_TYPES,
	TX_SELECTOR_MAX_NUM_OF_BIT_TYPES 		= 		0xFF,
} TxSelectorBitTypeSelection_e;

#endif //ENET_INC_ARCH_WAVE600B

typedef enum TxSelectorMuBitTypeSelection
{
	TX_SELECTOR_GROUP_USER_POSITION_STA = 0,
	TX_SELECTOR_GROUP_LOCK ,
	TX_SELECTOR_GROUP_TX_MODE,
	TX_SELECTOR_STA_GROUP_MAPPING,
	TX_SELECTOR_STA_MU_TID_ENABLE,
	TX_SELECTOR_NUM_OF_MU_BIT_TYPES,
	TX_SELECTOR_MAX_NUM_OF_MU_BIT_TYPES = 0xFF,
} TxSelectorMuBitTypeSelection_e;


/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

static void txSelectorUpdateBitmap(TxSelectorActionParams_t *updateBitmapParams,TxSelectorBitTypeSelection_e bitmapTypeSelection);
static void txSelectorUpdateMuBitmap(TxSelectorMuActionParams_t *updateMuBitmapParams,TxSelectorMuBitTypeSelection_e muBitmapTypeSelection);

#if defined (ENET_INC_LMAC)
static void txSelectorFillMuSelectionParams(TxSelectorMuSelectionDetails_t* selectorMuSelection,uint32 selectionRegAddress);
static void txSelectorSetWrrPriority(uint8 vapNum, uint8 wrrPriority, uint32 baseAddress);
static void txSelectorResetStationTidRam(void);
static void	txSelectorResetVapTidRam(void);
static void	txSelectorResetGlobalRam(void);
static void	txSelectorResetGplpRam(void);
static void	txSelectorResetGphpRam(void);
static void	txSelectorResetBeaconRam(void);
static void txSelectorResetMuStationRam(void);
static void txSelectorResetMuGroupRam(void);
static void	txSelectorResetPlanLockRam(void);
#ifndef ENET_INC_ARCH_WAVE600D2
static void	txSelectorResetTwtStaRam(void);
static void	txSelectorResetTwtCountRam(void);
#endif
static void txSelectorFillSelectionParams(TxSelectorSelectionDetails_t* selectorSelection,uint32 selectionRegAddress);
static void txSelectorWaitForMuBitmapFree(void);
#else
static void txSelectorWaitForMuBitmapFree(uint8 bandId);
#endif //(ENET_INC_UMAC)

/*---------------------------------------------------------------------------------
/						Static variables Declaration									
/----------------------------------------------------------------------------------*/
#if defined (ENET_INC_LMAC)
// only lmac0/1 can access it since there are RAMs per band
TxSelectorRam_t *pTxSelectorRam;
#endif

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
/**********************************************************************************

txSelectorUpdateBitmap 

Description:
------------
	Updates the information bits related to the queues of the chosen entry(Station, VAP, Global GPHP, GPLP, Beacon)
	The information bits reflects the queue state (Lock, Disabled, PS , PS request is pending, BF Sequence)

Input:
-----
	TxSelectorActionParams_t * bitmapUpdateParams - The entry parmetes(Q type,tid) and the required bit new value
	TxSelectorBitmapTypeSelection_e bitmapTypeSelection	- Which bits to change.
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
static void txSelectorUpdateBitmap(TxSelectorActionParams_t *updateBitmapParams,TxSelectorBitTypeSelection_e bitmapTypeSelection)
{
	//here - RegTxSelectorTxSelBitmapUpdateMaster2_u is used as prototype for the register.
	//the actual master is according to the cpu calling the routine
	RegTxSelectorTxSelBitmapUpdateMaster2_u bitMapUpdateReg;
	RegTxSelectorTxSelBitmapUpdateDataMaster2_u bitMapUpdateDataReg;	
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif //ENET_INC_UMAC

	if (updateBitmapParams->queueType == HW_TX_Q_TYPE_STA_TID)
	{
		ASSERT(updateBitmapParams->stationOrVapNum < HW_NUM_OF_STATIONS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForStation(updateBitmapParams->stationOrVapNum);
#endif //ENET_INC_UMAC
	}
	else
	{
		ASSERT(updateBitmapParams->stationOrVapNum < HW_NUM_OF_VAPS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForVap(updateBitmapParams->stationOrVapNum);
#endif 	//ENET_INC_UMAC
	}	

	/*REG_TX_SELECTOR_TX_SEL_BITMAP_UPDATE must be written last - this is the trigger for selector to start the command*/
	bitMapUpdateDataReg.val = 0;
	bitMapUpdateDataReg.bitFields.master2TidWriteEn = updateBitmapParams->tidEnableBitmap;	
	bitMapUpdateDataReg.bitFields.master2TidWriteValue = updateBitmapParams->tidValueBitmap;
	bitMapUpdateDataReg.bitFields.master2StaWriteData = updateBitmapParams->action;

	// There is a HW bug in selector of band0. For Global and GPLP it check the disable bit and DataInQ bit in the bitmap of band1.
#if defined (SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND)
#if defined (ENET_INC_UMAC)
	if ((ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND) &&
		((updateBitmapParams->queueType == HW_TX_Q_TYPE_GLOBAL) || (updateBitmapParams->queueType == HW_TX_Q_TYPE_GPLP)) &&
		(bandId == CONFIGURATION_MANAGER_BAND_0))
	{
		RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val, CONFIGURATION_MANAGER_BAND_0);
		RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val, CONFIGURATION_MANAGER_BAND_1);
	}
	else
	{
		RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val, bandId);
	}
#else
	RegAccess_Write(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val);
#endif

#else // SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND
#if defined (ENET_INC_UMAC)
		RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val, bandId);
#else
		RegAccess_Write(REG_TX_SELECTOR_BITMAP_UPDATE_DATA, bitMapUpdateDataReg.val);
#endif

#endif

	bitMapUpdateReg.val = 0;

    //For writing to plan bitmap we need both plan index & vap (for band selection), but we must write the plan index and not the 
    // STA or VAP number like all other bitmaps do
    if(TX_SELECTOR_PLAN_LOCK == bitmapTypeSelection)
    {
        bitMapUpdateReg.bitFields.master2StaSel = updateBitmapParams->planIndex;
    }
    else
    {
    	bitMapUpdateReg.bitFields.master2StaSel = updateBitmapParams->stationOrVapNum;
    }   
	bitMapUpdateReg.bitFields.master2BitmapSel = updateBitmapParams->queueType;
	bitMapUpdateReg.bitFields.master2BitTypeSel = bitmapTypeSelection;	

	// There is a HW bug in selector of band1. For Global and GPLP it check the disable bit and DataInQ bit in the bitmap of band0.
#if defined (SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND)
#if defined (ENET_INC_UMAC)
		if ((ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND) &&
			((updateBitmapParams->queueType == HW_TX_Q_TYPE_GLOBAL) || (updateBitmapParams->queueType == HW_TX_Q_TYPE_GPLP)) &&
			(bandId == 0))
		{
			RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val, CONFIGURATION_MANAGER_BAND_0);
			RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val, CONFIGURATION_MANAGER_BAND_1);
		}
		else
		{
			RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val, bandId);
		}
#else
		RegAccess_Write(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val);
#endif
	
#else // SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND
#if defined (ENET_INC_UMAC)
			RegAccess_WritePerBand(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val, bandId);
#else
			RegAccess_Write(REG_TX_SELECTOR_BITMAP_UPDATE, bitMapUpdateReg.val);
#endif
#endif

}


/**********************************************************************************

txSelectorUpdateMuBitmap 

Description:
------------
	Updates the information bits related to the MUGroup and STA entries
	The information bits reflects the Group state (Lock, Disabled STAs, STA USPs, Tx mode), and STA state (Primary enabled in Group, MU per TID Enabled/disabled).

	Need to verify that the Selector is free to recieve new bitmap configuration

Input:
-----
	TxSelectorMuActionParams_t *updateMuBitmapParams - The entry parmetes(Group index, USP, allSTAs update) and the required bit new value.
	TxSelectorMuBitTypeSelection_e muBitmapTypeSelection	- Which bits to change.
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/

static void txSelectorUpdateMuBitmap(TxSelectorMuActionParams_t *updateMuBitmapParams, TxSelectorMuBitTypeSelection_e muBitmapTypeSelection)
{
	RegTxSelectorTxSelMuBitmapUpdate_u muBitmapUpdate;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif //ENET_INC_UMAC

	if ((muBitmapTypeSelection == TX_SELECTOR_STA_GROUP_MAPPING) ||  (muBitmapTypeSelection == TX_SELECTOR_STA_MU_TID_ENABLE))
	{		
		ASSERT(updateMuBitmapParams->groupIdx < HW_NUM_OF_STATIONS);
#if defined (ENET_INC_UMAC)	
		// the groupIdx is actually StaId in the 2 cases checked above.
		bandId = ConfigurationManager_GetBandForStation(updateMuBitmapParams->groupIdx);
#endif //ENET_INC_UMAC
	}
	else
	{		
		ASSERT(updateMuBitmapParams->groupIdx < HW_NUM_OF_MU_GROUPS);
#if defined (ENET_INC_UMAC)		
		bandId = ConfigurationManager_GetBandForGroup(updateMuBitmapParams->groupIdx);
#endif //ENET_INC_UMAC
	}	

	//wait for bitmap update free indication (usually it will be free for the first time we check. If there are consecutive updates than a delay may occur).
#if defined (ENET_INC_UMAC)
	txSelectorWaitForMuBitmapFree(bandId);
#else
	txSelectorWaitForMuBitmapFree();
#endif //ENET_INC_UMAC


	muBitmapUpdate.val = 0;
	muBitmapUpdate.bitFields.memLineNum = updateMuBitmapParams->groupIdx;
	muBitmapUpdate.bitFields.userPositionIdx = updateMuBitmapParams->userPositionIdx;
	muBitmapUpdate.bitFields.muBitTypeSel = muBitmapTypeSelection;
	muBitmapUpdate.bitFields.muBitmapUpdateValue = updateMuBitmapParams->action;
	muBitmapUpdate.bitFields.allGrp = updateMuBitmapParams->allStas;
	muBitmapUpdate.bitFields.allGrpValidUp = updateMuBitmapParams->allGroupValidUspBitmap;

#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, muBitmapUpdate.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, muBitmapUpdate.val);
#endif
}



/**********************************************************************************

txSelectorWaitForMuBitmapFree 

Description:
------------
	Wait until MU Bitmap is free for new configuration.

Input:
-----
	None
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
#if defined (ENET_INC_UMAC)
static void txSelectorWaitForMuBitmapFree(uint8 bandId)
#else
static void txSelectorWaitForMuBitmapFree(void)
#endif //ENET_INC_UMAC
{
	volatile RegTxSelectorTxSelMuBitmapUpdate_u muBitmapUpdate;
	muBitmapUpdate.val = 0;


	while(muBitmapUpdate.bitFields.muBitmapUpdateFree == 0)
	{
#if defined (ENET_INC_UMAC)
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, &muBitmapUpdate.val, bandId);
#else
		RegAccess_Read(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE, &muBitmapUpdate.val);
#endif
	}

	/*When we get here - the bitmap is free for update*/
}


#if defined (ENET_INC_LMAC)

/**********************************************************************************

txSelectorFillMuSelectionParams 

Description:
------------
	Fill the selection parameters
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelection - Selection details
	uint32 selectionRegAddress	- The register address of the required source selection details
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorFillMuSelectionParams(TxSelectorMuSelectionDetails_t* selectorMuSelection, uint32 selectionRegAddress)
{
	RegTxSelectorTxSelMuUp0Result_u muSelectionReg;

	muSelectionReg.val = 0;

    RegAccess_Read(selectionRegAddress,&muSelectionReg.val);

	selectorMuSelection->tid = muSelectionReg.bitFields.muUp0Tid;
	selectorMuSelection->vapNum = muSelectionReg.bitFields.muUp0Vap;
	selectorMuSelection->staNum = muSelectionReg.bitFields.muUp0Sta;
	selectorMuSelection->beamforming = muSelectionReg.bitFields.muUp0BfReq;
	selectorMuSelection->accessCategory = muSelectionReg.bitFields.muUp0AcId;
	selectorMuSelection->txqId= muSelectionReg.bitFields.muUp0TxqId;
	selectorMuSelection->validSelection = muSelectionReg.bitFields.muUp0Valid;
}


/**********************************************************************************
txSelectorResetStationTidRam 

Description:
------------
	Reset STA TID RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void txSelectorResetStationTidRam(void)
{
	TxSelectorStaTidRam_t staTidBitmap;
	uint16 i = 0;

	staTidBitmap.word0 = TX_SELECTOR_RESET_VALUE_ZERO;
	staTidBitmap.word1 = TX_SELECTOR_RESET_VALUE_ZERO;
	staTidBitmap.word2 = TX_SELECTOR_RESET_VALUE_ZERO;
	staTidBitmap.disable = TRUE;	
	staTidBitmap.word3 = TX_SELECTOR_RESET_VALUE_ZERO;	

#if defined (ENET_INC_LMAC0)
	for(i=0; i<TX_SELECTOR_NUM_OF_VALID_STATIONS_ENTRIES_IN_RAM_B0; i++)
#elif defined (ENET_INC_LMAC1)
	for(i=0; i<TX_SELECTOR_NUM_OF_VALID_STATIONS_ENTRIES_IN_RAM_B1; i++) // Not all STA enrties are valid in band1
#endif //ENET_INC_LMAC0
	{
		pTxSelectorRam->staTidBitmap[i].word0 = staTidBitmap.word0;
		pTxSelectorRam->staTidBitmap[i].word1 = staTidBitmap.word1;
		pTxSelectorRam->staTidBitmap[i].word2 = staTidBitmap.word2;
		pTxSelectorRam->staTidBitmap[i].word3 = staTidBitmap.word3;
	}
}

/**********************************************************************************
txSelectorResetVapTidRam 

Description:
------------
	Reset VAP TID RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetVapTidRam(void)
{
	TxSelectorVapTidRam_t vapTidBitmap;
	uint8 i = 0;

	vapTidBitmap.vapTidRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	vapTidBitmap.vapTidRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_VAP_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->vapTidBitmap[i].vapTidRamWord0 = vapTidBitmap.vapTidRamWord0;
	}
}

/**********************************************************************************
txSelectorResetGlobalRam 

Description:
------------
	Reset Global RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetGlobalRam(void)
{
	TxSelectorGlobalRam_t globalBitmap;
	uint8 vapIndex = 0;

	globalBitmap.globalRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	globalBitmap.globalRamWord0.bitFields.disable = TRUE;


	for(vapIndex=0; vapIndex<TX_SELECTOR_NUM_OF_GLOBAL_ENTRIES_IN_RAM; vapIndex++)
	{
		pTxSelectorRam->globalBitmap[vapIndex].globalRamWord0 = globalBitmap.globalRamWord0;

#if defined (SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND)
#if defined (ENET_INC_LMAC1)
		// There is a HW bug in selector in which DATAinQ is always turned on in band1, and another bug that band0 look at band1 bit.
		// So we need to make sure that in DB we lock all the VAPs in band1 which belong to band0 
		uint8 firstVapOfBand0 = ConfigurationManager_GetNumOfVapsInMyBand();
		if ((vapIndex >= firstVapOfBand0) && (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND))
		{
			pTxSelectorRam->globalBitmap[vapIndex].globalRamWord0.bitFields.lockAc0 = TRUE;
			pTxSelectorRam->globalBitmap[vapIndex].globalRamWord0.bitFields.lockAc1 = TRUE;
			pTxSelectorRam->globalBitmap[vapIndex].globalRamWord0.bitFields.lockAc2 = TRUE;
			pTxSelectorRam->globalBitmap[vapIndex].globalRamWord0.bitFields.lockAc3 = TRUE;
		}
#endif
#endif
	}
}

/**********************************************************************************
txSelectorResetGplpRam 

Description:
------------
	Reset GPLP RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetGplpRam(void)
{
	TxSelectorGplpGphpRam_t gplpBitmap;
	uint8 vapIndex = 0;

	gplpBitmap.gplpGphpRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	gplpBitmap.gplpGphpRamWord0.bitFields.disable = TRUE;

	for(vapIndex=0; vapIndex<TX_SELECTOR_NUM_OF_GPLP_ENTRIES_IN_RAM; vapIndex++)
	{
		pTxSelectorRam->gplpBitmap[vapIndex].gplpGphpRamWord0 = gplpBitmap.gplpGphpRamWord0;

#if defined (SELECTOR_BUG_CDB_GPLP_GLOBAL_WORKAROUND)
#if defined (ENET_INC_LMAC1)
		// There is a HW bug in selector in which DATAinQ is always turned on in band1, and another bug that band0 look at band1 bit.
		// So we need to make sure that in DB we lock all the VAPs in band1 which belong to band0 
		uint8 firstVapOfBand0 = ConfigurationManager_GetNumOfVapsInMyBand();
		if ((vapIndex >= firstVapOfBand0) && (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND))
		{
			pTxSelectorRam->gplpBitmap[vapIndex].gplpGphpRamWord0.bitFields.lock = TRUE;
		}
#endif
#endif	
	}
}

/**********************************************************************************
txSelectorResetGphpRam 

Description:
------------
	Reset GPHP RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetGphpRam(void)
{
	TxSelectorGplpGphpRam_t gphpBitmap;
	uint8 i = 0;

	gphpBitmap.gplpGphpRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	gphpBitmap.gplpGphpRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_GPHP_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->gphpBitmap[i].gplpGphpRamWord0 = gphpBitmap.gplpGphpRamWord0;
	}
}

/**********************************************************************************
txSelectorResetBeaconRam 

Description:
------------
	Reset beacon RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetBeaconRam(void)
{
	TxSelectorBeaconRam_t beaconBitmap;
	uint8 i = 0;

	beaconBitmap.beaconRamWord0.val = TX_SELECTOR_RESET_VALUE_ZERO;
	
	beaconBitmap.beaconRamWord0.bitFields.disable = TRUE;

	for(i=0; i<TX_SELECTOR_NUM_OF_BEACON_ENTRIES_IN_RAM; i++)
	{
		pTxSelectorRam->beaconBitmap[i].beaconRamWord0 = beaconBitmap.beaconRamWord0;
	}
}

/**********************************************************************************
txSelectorResetPlanLockRam 

Description:
------------
	Reset Plan Lock RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetPlanLockRam(void)
{
	// Lock all the Plans
	
	pTxSelectorRam->planLockBitmap.planLockBitmapWord0 = ALL_PLANS_LOCKED;
	pTxSelectorRam->planLockBitmap.planLockBitmapWord1 = ALL_PLANS_LOCKED;
	pTxSelectorRam->planLockBitmap.planLockBitmapWord2 = ALL_PLANS_LOCKED;
	pTxSelectorRam->planLockBitmap.planLockBitmapWord3 = ALL_PLANS_LOCKED;
	
}

#ifndef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************
txSelectorResetTwtStaRam 

Description:
------------
	Reset TWT RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetTwtStaRam(void)
{ 
    uint32 regAddress;
    
#if defined (ENET_INC_LMAC0)
    regAddress = B0_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM + TX_SELECTOR_TWT_SP_GROUP_STA_BITMAP_OFFSET_FROM_RAM_START;   
#elif defined (ENET_INC_LMAC1)
    regAddress = B1_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM + TX_SELECTOR_TWT_SP_GROUP_STA_BITMAP_OFFSET_FROM_RAM_START;
#endif 

    memset32((void*)regAddress, 0, CONVERT_BYTES_TO_WORDS(TX_SELECTOR_TWT_SP_GROUP_COUNT_BITMAP_OFFSET_FROM_RAM_START - TX_SELECTOR_TWT_SP_GROUP_STA_BITMAP_OFFSET_FROM_RAM_START));
}

/**********************************************************************************

txSelectorResetTwtCountRam 

Description:
------------
	Reset TWT RAM
Input:
-----
	None
	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
static void	txSelectorResetTwtCountRam(void)
{
    uint32 regAddress;
    
#if defined (ENET_INC_LMAC0)
    regAddress = B0_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM + TX_SELECTOR_TWT_SP_GROUP_COUNT_BITMAP_OFFSET_FROM_RAM_START;
#elif defined (ENET_INC_LMAC1)
    regAddress = B1_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM + TX_SELECTOR_TWT_SP_GROUP_COUNT_BITMAP_OFFSET_FROM_RAM_START;
#endif 

    memset32((void*)regAddress, 0, CONVERT_BYTES_TO_WORDS(TX_SELECTOR_END_OF_BITMAP_OFFSET_FROM_RAM_START - TX_SELECTOR_TWT_SP_GROUP_COUNT_BITMAP_OFFSET_FROM_RAM_START));
}
#endif
/**********************************************************************************

txSelectorResetMuStaRam 

Description:
------------
	Reset selector MU STA Ram to zeros. This action disables MU Primary for all STA and disables all TIDs for MU Tx
Input:
-----
	uint32 numOfResetEntries - Number of entries in the ram.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorResetMuStationRam(void)
{
	uint32 i = 0;
	TxSelectorPrimaryStaInfo_u primaryStaInfo;
	RegTxSelectorMuTidBitmap_u tidBitmap;

	primaryStaInfo.bitFields.groupIdx = 0x0;
	primaryStaInfo.bitFields.isPrimaryInGroup = FALSE;
	tidBitmap.val = 0x0;

	/* Set the registers values (Same for all entries)*/
#if defined (ENET_INC_LMAC0)
	for(i = 0; i < TX_SELECTOR_NUM_OF_VALID_MU_STA_ENTRIES_IN_RAM_B0; i++)
#else
	for(i = 0; i < TX_SELECTOR_NUM_OF_VALID_MU_STA_ENTRIES_IN_RAM_B1; i++)
#endif //ENET_INC_LMAC0
	{		
		TxSelector_ModifyPrimaryStaInfo(i, primaryStaInfo);
		TxSelector_ConfigureStaMuEnabledTids(i, tidBitmap);
	}
}



/**********************************************************************************

txSelectorResetMuGroupRam 

Description:
------------
	Reset selector MU Group Ram to zeros. This action disables all USPs in the group, disables Tx Mode, all groups are unlocked.
Input:
-----
	uint32 numOfResetEntries - Number of entries in the ram.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorResetMuGroupRam(void)
{
	RegTxSelectorAllStaBitmapUpdate_u allStaInfo;
	uint32 i = 0;
	TxSelectorMuActionParams_t muBitmapParams;
	TxSelectorAllGroupValidUspBitmap_t bitmap;

	memset(&muBitmapParams, 0, sizeof(muBitmapParams));
	
	muBitmapParams.action = TX_SELECTOR_LOCK_GROUP;
	
	allStaInfo.val = 0;
	bitmap.validUspBitmap = 0;

	/* Set the registers values (Same for all entries)*/

	for(i = 0; i < TX_SELECTOR_NUM_OF_MU_GROUP_ENTRIES_IN_RAM; i++)
	{
		TxSelector_MapAllStasToGroup(i, allStaInfo, bitmap);
		
		/* Lock all groups on initialization */
		muBitmapParams.groupIdx = i;
		txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_LOCK);
		
		TxSelector_UpdateGroupTxMode(i, TX_SELECTOR_GROUP_DISABLED);
	}
}



/**********************************************************************************

txSelectorSetWrrPriority 

Description:
------------
	Configure the WRR priority for the selected entry 
	For each vap we can configure WRR between its Sta PS source & Sta active source,
	Each WRR type(PS,Active) is spread across 2 registers, in each register there 
	priorities for 8 vaps.
Input:
-----
	uint8 vapNum - Vap Number 
	uint8 wrrPriority - WRR Priority value
	uint32 baseAddress - The first register address according to the WRR type	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorSetWrrPriority(uint8 vapNum,uint8 wrrPriority,uint32 baseAddress)
{
	RegTxSelectorTxSelStaPsWrrLow0_u wrrReg;
	
	uint32 wrrRegAddress = 0;
	uint8 wrrPriorityBitOffset = 0;
	uint8 wrrPriorityRegisterOffset = 0;

	DEBUG_ASSERT(vapNum < HW_NUM_OF_VAPS);
	
	wrrReg.val = 0;
	/* the lower 3 bits determine the offset of the vap priority in the register */
	wrrPriorityBitOffset = (vapNum & TX_SELECTOR_NUM_OF_VAPS_IN_WRR_REGISTER_MASK)* TX_SELECTOR_WRR_VAP_PRIORITY_WIDTH;
	/*each registrer includes 8 vaps, so in order to determine if we set a priority of a vap in the range 0-7 or in the range 8-15, or 16-23 or 24-31 
	 we should devide by 8 (shift right by 3)*/ 
	wrrPriorityRegisterOffset = (vapNum >> TX_SELECTOR_VAP_MSB_SHIFT) * HW_REG_SIZE; 
	/* Register Address for updating wrr priority */
	wrrRegAddress = baseAddress + wrrPriorityRegisterOffset;

	/* set the WRR priority of the relevant vap */
	RegAccess_WriteMasked(wrrRegAddress,(TX_SELECTOR_WRR_VAP_PRIORITY_MASK << wrrPriorityBitOffset),(wrrPriority << wrrPriorityBitOffset));
}

/**********************************************************************************

txSelectorFillSelectionparams 

Description:
------------
	Fill the selection parameters
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelection - Selection details
	uint32 selectionRegAddress	- The register address of the required source selection details
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
static void txSelectorFillSelectionParams(TxSelectorSelectionDetails_t* selectorSelection,uint32 selectionRegAddress)
{
	RegTxSelectorTxSelSelection_u selectionReg;

	selectionReg.val = 0;

    RegAccess_Read(selectionRegAddress,&selectionReg.val);

	selectorSelection->selectionSource = selectionReg.bitFields.selectionSrcNum;
	selectorSelection->tid = selectionReg.bitFields.selectionTidNum;
	selectorSelection->staNum = selectionReg.bitFields.selectionStaNum;
	selectorSelection->vapNum = selectionReg.bitFields.selectionVapNum;
	selectorSelection->beamforming = selectionReg.bitFields.selectionBf;
	selectorSelection->accessCategory = selectionReg.bitFields.selectionAcNum;
	selectorSelection->validSelection = selectionReg.bitFields.selectionValid;
	selectorSelection->emptySelection= selectionReg.bitFields.selectionEmpty;
}

#endif

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/


/**********************************************************************************

TxSelector_EnableQueue 

Description:
------------
	This procedure disables/enables  the queue specified by the parameters queue, stationVapNum, tidNum
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		- The queue type to disable (STA,VAP,VIP,GLOBAL,SUPER)
		- The VAP number if the queue is VAP or STA number if the queue is STA
		- The tid number of the queue to be disabled
		- TxSelectorEnableDisableQ_e disable - 0x0 - enable , 0x1 - disable
		- Indicates if we should perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_EnableQueue(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorActionParams->action = TX_SELECTOR_ENABLE_QUEUE;
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_DISABLE);
}


/**********************************************************************************

TxSelector_UnlockQueue 

Description:
------------
	This procedure locks/unlocks the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		- The queue type to lock (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		- the VAP number if the queue is VAP or STA number if the queue is STA
		- the tid/AC number of the queue to be disabled
		- 0x0 - unlock , 0x1 - lock
		- Indicates if we should perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_UnlockQueue(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_LOCK);
}


/**********************************************************************************

TxSelector_LockRequest 

Description:
------------
	Request for the HW Locker to execute lock request (Only force lock request is done by FW modules)
Input:
-----
	txSelectorLockRequestParams - Parameters of the lock request

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_LockRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams)
{
	RegTxSelectorTxSelSwMaster2Locker_u lockerReg;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif //ENET_INC_UMAC

	lockerReg.val = 0;

	if(txSelectorLockRequestParams->queueType == HW_TX_Q_TYPE_STA_TID)
	{		
		ASSERT(txSelectorLockRequestParams->stationOrVapNum < HW_NUM_OF_STATIONS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForStation(txSelectorLockRequestParams->stationOrVapNum);
#endif //ENET_INC_UMAC
	}
	else
	{		
		ASSERT(txSelectorLockRequestParams->stationOrVapNum < HW_NUM_OF_VAPS);
 #if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForVap(txSelectorLockRequestParams->stationOrVapNum);
#endif //ENET_INC_UMAC
	}

	lockerReg.bitFields.master2LockerStaSel = txSelectorLockRequestParams->stationOrVapNum;
	lockerReg.bitFields.master2LockerTidSel = txSelectorLockRequestParams->tidOrAc;
	lockerReg.bitFields.master2LockerBitmapSel = txSelectorLockRequestParams->queueType;
	lockerReg.bitFields.master2LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK;
	lockerReg.bitFields.master2LockerAllTids = txSelectorLockRequestParams->allTids;

#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val);
#endif
}

/**********************************************************************************

TxSelector_DisableRequest 

Description:
------------
	
Input:
-----

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_DisableRequest(TxSelectorLockReqParams_t *txSelectorLockRequestParams)
{
	RegTxSelectorTxSelSwMaster2Locker_u lockerReg;

	lockerReg.val = 0;

#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif //ENET_INC_UMAC
	
	if(txSelectorLockRequestParams->queueType == HW_TX_Q_TYPE_STA_TID)
	{		
		ASSERT(txSelectorLockRequestParams->stationOrVapNum < HW_NUM_OF_STATIONS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForStation(txSelectorLockRequestParams->stationOrVapNum);
#endif //ENET_INC_UMAC
	}
	else
	{		
		ASSERT(txSelectorLockRequestParams->stationOrVapNum < HW_NUM_OF_VAPS);
 #if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForVap(txSelectorLockRequestParams->stationOrVapNum);
#endif //ENET_INC_UMAC
	}
	
	lockerReg.bitFields.master2LockerStaSel = txSelectorLockRequestParams->stationOrVapNum;
	lockerReg.bitFields.master2LockerTidSel = txSelectorLockRequestParams->tidOrAc;
	lockerReg.bitFields.master2LockerBitmapSel = txSelectorLockRequestParams->queueType;
	lockerReg.bitFields.master2LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_DISABLE;
	lockerReg.bitFields.master2LockerAllTids = txSelectorLockRequestParams->allTids;

#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_LOCKER_REQ, lockerReg.val);
#endif
}


/**********************************************************************************

TxSelector_LockPlanRequest 

Description:
------------
	Request for the HW Locker to execute plan lock request (Only force lock request is done by FW modules)
Input:
-----
	txSelectorLockRequestParams - Parameters of the lock request

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_LockPlanRequest(uint8 planIndex, uint16 vapId)
{
#if defined (ENET_INC_UMAC)

    RegTxSelectorTxSelPlanSwMaster1Locker_u planLockerReg;    
    uint8 bandId;

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_DD("[TxSelector_LockPlanRequest], planIndex: %d, vapId: %d", planIndex, vapId);
#endif

    bandId = ConfigurationManager_GetBandForVap(vapId);

    planLockerReg.val = 0;

    planLockerReg.bitFields.master1PlanLockerIdxSel     = planIndex;
    planLockerReg.bitFields.master1PlanLockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK;

    RegAccess_WritePerBand(REG_TX_SELECTOR_PLAN_LOCKER_REQ, planLockerReg.val, bandId);
#else
	UNUSED_PARAM(planIndex);
	UNUSED_PARAM(vapId);
	DEBUG_FATAL("TxSelector_LockPlanRequest should only be called from UM");
#endif //ENET_INC_UMAC
    
}

/**********************************************************************************

TxSelector_LockGroupRequest 

Description:
------------
	Request for the HW Locker to execute lock request (Only force lock request is done by FW modules)
Input:
-----
	txSelectorLockRequestParams - Parameters of the lock request

	
Output:
-------
	None
Returns:
--------
	void 	
**********************************************************************************/
void TxSelector_LockGroupRequest(uint32 groupId)
{
#if defined (ENET_INC_UMAC)

	RegTxSelectorTxSelMuSwMaster1Locker_u lockerReg;
	uint8 bandId;	
	lockerReg.val = 0;
	
	ASSERT(groupId < HW_NUM_OF_MU_GROUPS);
	bandId = ConfigurationManager_GetBandForGroup(groupId);

	lockerReg.val = 0;
	lockerReg.bitFields.master1MuLockerGrpSel = groupId;
	lockerReg.bitFields.master1MuLockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_FORCE_LOCK;

	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_MU_SW_MASTER1_LOCKER, lockerReg.val, bandId);
#else
	UNUSED_PARAM(groupId);
	DEBUG_FATAL("TxSelector_LockGroupRequest should only be called from UM");
#endif //ENET_INC_UMAC
}


/**********************************************************************************

TxSelector_SetPowerSaveType 

Description:
------------
	This procedure Set power save type of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save type  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid/AC number of the queue 
	- Power Save type 0x0 - Legacy , 0x1 - WMM
	- Should we perform action for all TIDs/ACs
		
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetPowerSaveType(TxSelectorActionParams_t *txSelectorActionParams)
{
	AccessCategoryTo8021d_t *acTo8021dConversionTable;
	uint8 accessCategory;

	accessCategory = txSelectorActionParams->acValue;

	if (accessCategory == ACCESS_CATEGORY_INAVLID) //means we need to set for all TIDs
	{
		txSelectorActionParams->tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	}
	else
	{
		ASSERT(accessCategory < ACCESS_CATEGORY_NUM);
		acTo8021dConversionTable = Protocol_GetAccessCategoryTo8021dConversionTable();			
		txSelectorActionParams->tidEnableBitmap = (0x1 << (acTo8021dConversionTable->val8021d[accessCategory][FIRST_TID_PER_AC_INDEX]) | \
												(0x1 << (acTo8021dConversionTable->val8021d[accessCategory][SECOND_TID_PER_AC_INDEX])));

		if (txSelectorActionParams->tidEnableBitmap & (0x1<<IEEE802_1D_VO_2))
		{//set the pwoer mode also in the MANAGEMENT_TID
			txSelectorActionParams->tidEnableBitmap |= 0x100;
		}
	}

	txSelectorActionParams->tidValueBitmap = (txSelectorActionParams->action)*txSelectorActionParams->tidEnableBitmap; //set all TID enabled bits with the chosen value
	
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_PS_TYPE);
}
	



/**********************************************************************************

TxSelector_ClearPowerSavemode 

Description:
------------
	This procedure Clear power save mode of the queue specified by the parameters queue, stationVapNum, tidNum
	Clear and Set use the same API functions. The seperation is required for gen4.
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid/AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_ClearPowerSavemode(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_ACTIVE_PS_MODE);
}



/**********************************************************************************

TxSelector_SetPowerSaveRequest

Description:
------------
	This procedure Set power save request of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid/AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetPowerSaveRequest(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams, TX_SELECTOR_REQUEST);
}


/**********************************************************************************

TxSelector_SetDataInQ


Description:
------------
	This procedure Set/Reset Data in Q of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set Data In Q (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid/AC number of the queue 
	- TX_SELECTOR_SET_DATA_IN_Q 0x0 - Reset Data In Q , 0x1 - Set Data In Q
	- Should we perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetDataInQ(TxSelectorActionParams_t *txSelectorActionParams)
{
    txSelectorUpdateBitmap(txSelectorActionParams, TX_SELECTOR_DATA_IN_Q);
}

#ifdef ENET_INC_ARCH_WAVE600B
void TxSelector_SetAControl(StaId staID, uint8 enable)
{
	TxSelectorActionParams_t txSelectorActionParams;

	txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
	txSelectorActionParams.stationOrVapNum = staID;
	txSelectorActionParams.action = enable;

	txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_INSERT_A_CONTROL);
}
#endif //ENET_INC_ARCH_WAVE600B

#ifndef ENET_INC_ARCH_WAVE600B
/**********************************************************************************

TxSelector_HeMuTidUpdate


Description:
------------
	This procedure Enable/disables HE MU bit for a specific STA 
Input:
-----
    stationNum - STA number to enable/disable HE MU
    tidBitmap - set HE MU for specific TID or more or all
    txSelectorHeMu - TX_SELECTOR_STA_HE_MU_DISABLE or TX_SELECTOR_STA_HE_MU_ENABLE
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_HeMuTidUpdate(uint16 stationNum, uint16 tidBitmap)
{
    TxSelectorActionParams_t txSelectorActionParams;

#if 0
	TxSelectorRam_t *pTxSelectorRam = NULL;
#if defined (ENET_INC_LMAC0)
	pTxSelectorRam = (TxSelectorRam_t *)(B0_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM); 
#elif defined (ENET_INC_LMAC1)
	pTxSelectorRam = (TxSelectorRam_t *)(B1_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM); 
#endif
	TxSelectorStaTidRam_t txSelectorStaTidRam;
#endif

    ASSERT(tidBitmap <= TID_BITMAP_ALL_TIDS);
    
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
	txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	txSelectorActionParams.tidValueBitmap =  tidBitmap;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = 0x0; // NA while writing a value per TID , only for station resolution update  

#if 0
	memcpy32(&txSelectorStaTidRam, &(pTxSelectorRam->staTidBitmap[stationNum].word0), sizeof(TxSelectorStaTidRam_t) >> 2);

	ILOG0_V("TxSelector_HeMuTidUpdate BEFORE");
	
	SLOG0(0, 0, TxSelectorStaTidRam_t, &txSelectorStaTidRam);
#endif

    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_HE_MU_ENABLE);

#if 0
	memcpy32(&txSelectorStaTidRam, &(pTxSelectorRam->staTidBitmap[stationNum].word0), sizeof(TxSelectorStaTidRam_t) >> 2);

	ILOG0_V("TxSelector_HeMuTidUpdate AFTER");
	
	SLOG0(0, 0, TxSelectorStaTidRam_t, &txSelectorStaTidRam);
#endif

}
#else // ENET_INC_ARCH_WAVE600B
/**********************************************************************************

TxSelector_HeMuDlTidUpdate


Description:
------------
	This procedure Enable/disables HE MU bit for a specific STA 
Input:
-----
    stationNum - STA number to enable/disable HE MU
    tidBitmap - set HE MU for specific TID or more or all
    txSelectorHeMu - TX_SELECTOR_STA_HE_MU_DISABLE or TX_SELECTOR_STA_HE_MU_ENABLE
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_HeMuDlTidUpdate(uint16 stationNum, uint16 tidBitmap)
{
    TxSelectorActionParams_t txSelectorActionParams;

    ASSERT(tidBitmap <= TID_BITMAP_ALL_TIDS);
    
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
	txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	txSelectorActionParams.tidValueBitmap =  tidBitmap;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = 0x0; // NA while writing a value per TID , only for station resolution update  

    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_DL_HE_MU_ENABLE);
}
/**********************************************************************************

TxSelector_HeMuUlStationUpdate


Description:
------------
	This procedure Enable/disables HE UL MU bit for a specific STA 
Input:
-----
    stationNum - STA number to enable/disable HE MU
    txSelectorHeMu - TX_SELECTOR_STA_HE_MU_DISABLE or TX_SELECTOR_STA_HE_MU_ENABLE
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_HeMuUlStationUpdate(uint16 stationNum, bool enable)
{
    TxSelectorActionParams_t txSelectorActionParams;
    
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = enable;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_UL_HE_MU_ENABLE);
}
#endif
/**********************************************************************************

TxSelector_SetDLPlanIndex


Description:
------------
	This procedure sets DL plan index for a specific STA 
Input:
-----
    stationNum - STA number to update the DL plan index
    planIndex - the plan index to write
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetDLPlanIndex(uint16 stationNum, uint8 planIndex)
{
    TxSelectorActionParams_t txSelectorActionParams;
    
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = planIndex;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_DL_PLAN_INDEX);
}

/**********************************************************************************

TxSelector_SetULPlanIndex


Description:
------------
	This procedure sets UL plan index for a specific STA 
Input:
-----
    stationNum - STA number to update the UL plan index
    planIndex - the plan index to write
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetULPlanIndex(uint16 stationNum, uint8 planIndex)
{
    TxSelectorActionParams_t txSelectorActionParams;
    
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = planIndex;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_UL_PLAN_INDEX);
}

/**********************************************************************************

TxSelector_UnlockPlan


Description:
------------
	This procedure unlocks a specific plan
Input:
-----
    planIndex - the plan index to unlock
    vapNum - VAP number is used to find the band
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_UnlockPlan(uint8 planIndex, StaId staId)
{
    TxSelectorActionParams_t txSelectorActionParams;

    txSelectorActionParams.stationOrVapNum = staId;
    txSelectorActionParams.planIndex = planIndex;
    txSelectorActionParams.action = TX_SELECTOR_UNLOCK_PLAN;
    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;

    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_PLAN_LOCK);
}
#ifdef ENET_INC_ARCH_WAVE600B
/**********************************************************************************

TxSelector_SetOmiValid


Description:
------------
	This procedure sets OMI valid bit
Input:
-----
    stationNum - STA number for setting its OMI bits in the bitmap
    omiValid - 1 - valid, 0 - invalid 
    
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetOmiValid(uint16 stationNum, bool omiValid)
{
    TxSelectorActionParams_t txSelectorActionParams;

    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = omiValid;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_OMI_VALID);
}
/**********************************************************************************

TxSelector_SetOmiSupported


Description:
------------
	This procedure sets OMI supported bit
Input:
-----
    stationNum - STA number for setting its OMI bits in the bitmap
    omiValid - 1 - valid, 0 - invalid 
    
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetOmiSupported(uint16 stationNum, bool omiSupported)
{
    TxSelectorActionParams_t txSelectorActionParams;

    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = omiSupported;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_OMI_SUPPORTED);
}
/**********************************************************************************

TxSelector_SetOmiBits


Description:
------------
	This procedure sets OMI bits (operation mode index) for a specific STA 
	this is part of the HE new features. OMI bits are received from the STA
	and copied to the bitmap. AP only sets the omiSupported bit on/off
Input:
-----
    stationNum - STA number for setting its OMI bits in the bitmap
    txSelectorOmiInfo - reflecting the OMI bits that should be written to the STA bitmap. 
    Set val=0 and fill the fields according to STA information and disable or enable
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetOmiBits(uint16 stationNum, TxSelectorOmiInfo_u *ptxSelectorOmiInfo)
{
    TxSelectorActionParams_t txSelectorActionParams;

    txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorActionParams.stationOrVapNum = stationNum;
    txSelectorActionParams.action = ptxSelectorOmiInfo->val;
    
    txSelectorUpdateBitmap(&txSelectorActionParams, TX_SELECTOR_OMI_BITS);
}
#endif


/**********************************************************************************
TxSelector_SetStationToVapAssocState 

Description:
------------
	Add		- Add station to VAPs associated stations bitmap and Enable the 8 STA TID queues.
Input:
-----
	uint16 stationId - The station to be removed.
	uint8 vapId		- The station associated vap. 
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetStationToVapAssocState(uint16 stationId, uint8 vapId)
{	
	RegTxSelectorTxSelStaAssociate_u staAssociateReg;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif	

	staAssociateReg.val = 0;

	ASSERT((stationId < HW_NUM_OF_STATIONS) && (vapId < HW_NUM_OF_VAPS));

#if defined (ENET_INC_UMAC)
	bandId = ConfigurationManager_GetBandForVap(vapId);
#endif //ENET_INC_UMAC

	/* Add station to associated stations to vap bitmap*/
	staAssociateReg.bitFields.staAssociateStation = stationId;
	staAssociateReg.bitFields.staAssociateVap = vapId;
	staAssociateReg.bitFields.staAssociateVapStaBitValue = TX_SELECTOR_ASSOCIATE;
#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val);
#endif
	
}


/**********************************************************************************
TxSelector_ClearStationFromVapAssocState

Description:
------------
	Remove	- Remove station from VAPs associated stations bitmap and disable the 8 STA TID queues.

Input:
-----
	uint16 stationId - The station to be removed.
	uint8 vapId		- The station associated vap. 
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ClearStationFromVapAssocState(uint16 stationNum,uint8 vapId)
{	
	RegTxSelectorTxSelStaAssociate_u staAssociateReg;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif	
	
	staAssociateReg.val = 0;

	ASSERT((stationNum < HW_NUM_OF_STATIONS) && (vapId < HW_NUM_OF_VAPS));

#if defined (ENET_INC_UMAC)
	bandId = ConfigurationManager_GetBandForVap(vapId);
#endif //ENET_INC_UMAC

	/* Add station to associated stations to vap bitmap*/
	staAssociateReg.bitFields.staAssociateStation = stationNum;
	staAssociateReg.bitFields.staAssociateVap = vapId;
	staAssociateReg.bitFields.staAssociateVapStaBitValue = TX_SELECTOR_DISASSOCIATE;
#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE, staAssociateReg.val);
#endif
	
}


#if defined (ENET_INC_UMAC)
#ifndef ENET_INC_ARCH_WAVE600D2
/**********************************************************************************

TxSelector_ConfigureTwtRegsNewSp

Description: configure regs when new SP created
------------
	
Input:
-----
uint32 twtSpLowLimit
uint32 twtSpStartTsf
uint32 twtSpHighLimit
uint8  twtSpGroupIdx
uint8  vapId

Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ConfigureTwtRegsNewSp(uint32 twtSpLowLimit, uint32 twtSpStartTsf, uint32 twtSpHighLimit, uint8 twtSpGroupIdx, TwtManagerFlowType_e twtType, uint8 bandId)
{    
    RegTxSelectorTxSelTwtSpGroupLowPhase_u lowPhaseReg;
    RegTxSelectorTxSelTwtSpGroupStartTsf_u startTsfReg; 
    RegTxSelectorTxSelTwtSpGroupUpdate_u   groupUpdateReg;
    bool                                   done;

	UNUSED_PARAM(twtType);
    done = FALSE;
    lowPhaseReg.val = 0;
    lowPhaseReg.bitFields.twtSpLowLimit = twtSpLowLimit;
/*#ifdef ENET_INC_ARCH_WAVE600B //under comment due to a bug in HW
    if (twtType == TWT_MANAGER_FLOW_TYPE_ANNOUNCED)
    {
	    lowPhaseReg.bitFields.twtAvailUpdateValue = TX_SELECTOR_TWT_TYPE_ANNOUNCED;
    }
    else
    {
        lowPhaseReg.bitFields.twtAvailUpdateValue = TX_SELECTOR_TWT_TYPE_UNANNOUNCED;
    }
#endif*/
    RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_LOW_PHASE ,lowPhaseReg.val, bandId);
    startTsfReg.val = 0;
    startTsfReg.bitFields.twtSpStartTsf = twtSpStartTsf;
    RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_START_TSF ,startTsfReg.val, bandId);
    groupUpdateReg.val = 0;
    groupUpdateReg.bitFields.twtSpHighLimit = twtSpHighLimit;
    groupUpdateReg.bitFields.twtSpGroupIdx = twtSpGroupIdx;  
    RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_UPDATE ,groupUpdateReg.val, bandId); // pulse write
    /*polling the "done" bit*/
    ILOG0_D("twt registers in selector, new SP registers updated, waiting for selector. SP ID: %d", twtSpGroupIdx);
    RegAccess_WaitForFieldPerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_UPDATE, REG_TX_SELECTOR_TWT_SP_SHIFT, REG_TX_SELECTOR_TWT_SP_MASK, TRUE, TX_SELECTOR_TWT_LONG_POLLING_TIMEOUT, &done, bandId);
    ILOG0_DDD("twt in selector: high: %d, low: %d, tsf: %d",twtSpHighLimit, twtSpLowLimit, twtSpStartTsf);
    ASSERT(done == TRUE); /*polling TIMEOUT*/
    ILOG0_V("twt registers in selector, selector is done");
}


/**********************************************************************************

TxSelector_ConfigureTwtRegsUpdateSta

Description: configure regs when adding or removing STA from SP
------------
	
Input:
-----
uint8  twtSpStaGroupIdx
uint16 twtSpStaIdx
bool   twtSpStaValue - add or remove STA fom SP profile
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ConfigureTwtRegsUpdateSta(uint8 twtSpStaGroupIdx, uint16 twtSpStaIdx, bool twtSpStaValue)
{    
    RegTxSelectorTxSelTwtSpStaUpdate_u staUpdateReg;
    bool                               done;
    uint8                              bandId; 
    
    done = FALSE;
    bandId = ConfigurationManager_GetBandForStation(twtSpStaIdx);
    staUpdateReg.val = 0;
    staUpdateReg.bitFields.twtSpStaGroupIdx = twtSpStaGroupIdx;
    staUpdateReg.bitFields.twtSpStaIdx = twtSpStaIdx;
    staUpdateReg.bitFields.twtSpStaValue = (uint8)twtSpStaValue;     
    RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_STA_UPDATE,staUpdateReg.val, bandId);
    /*pollling the "done" bit*/
    ILOG0_DD("twt registers in selector, staUpdateReg updated, waiting for selector. SID: %d, spId: %d", twtSpStaIdx, twtSpStaGroupIdx);
    RegAccess_WaitForFieldPerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_STA_UPDATE, REG_TX_SELECTOR_TWT_SP_SHIFT, REG_TX_SELECTOR_TWT_SP_MASK, TRUE, TX_SELECTOR_TWT_POLLING_TIMEOUT, &done, bandId);
    ASSERT(done == TRUE); /*polling TIMEOUT*/
    ILOG0_V("twt registers in selector, selector is done");
}

/**********************************************************************************

TxSelector_ConfigureTwtRegsDeactivateSp

Description: configure regs when deactivating a SP
------------
	
Input:
-----
uint8 twtSpDeactivateGroupIdx
uint8 vapId

Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ConfigureTwtRegsDeactivateSp(uint8 twtSpDeactivateGroupIdx, uint8 bandId)
{
    RegTxSelectorTxSelTwtSpGroupDeactivate_u deactivateReg;
    bool                                     done;
    
    done = FALSE;
    deactivateReg.val = 0;
    deactivateReg.bitFields.twtSpDeactivateGroupIdx = twtSpDeactivateGroupIdx;    
    RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_DEACTIVATE,deactivateReg.val, bandId);
    /*polling the "done" bit*/
    ILOG0_D("twt registers in selector, deactivateReg updated, waiting for selector. SP ID: %d", twtSpDeactivateGroupIdx);
    RegAccess_WaitForFieldPerBand(REG_TX_SELECTOR_TX_SEL_TWT_SP_GROUP_DEACTIVATE, REG_TX_SELECTOR_TWT_SP_SHIFT, REG_TX_SELECTOR_TWT_SP_MASK, TRUE, TX_SELECTOR_TWT_LONG_POLLING_TIMEOUT, &done, bandId);
    ASSERT(done == TRUE); /*polling TIMEOUT*/
    ILOG0_V("twt registers in selector, selector is done");
}

/**********************************************************************************
TxSelector_TwtTypeUpdateEnable

Description: enable update of twt type for SP
------------
	
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
#ifdef ENET_INC_ARCH_WAVE600B

void TxSelector_TwtTypeUpdateEnable()
{
    RegTxSelectorTxSelTwtCfg_u twtTypeUpdateEnableReg;
    uint8 bandId;

    twtTypeUpdateEnableReg.val = 0;
    twtTypeUpdateEnableReg.bitFields.twtAvailUpdateEn = 0;  //Due to a bug in HW  
    for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
    {
        RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_CFG, twtTypeUpdateEnableReg.val, bandId);
    }
}


/**********************************************************************************
TxSelector_TwtTSwHalt

Description: protection for the registers
------------
	
Input:
-----
	
Output:
-------
	None
Returns:
--------
	void 
*********************************************************************************/
void TxSelector_TwtSwHalt()
{
    RegTxSelectorTxSelTwtSwHalt_u twtSwHaltReg;
    uint8 bandId;

    twtSwHaltReg.val = 0;
    twtSwHaltReg.bitFields.twtSwHalt = 0;    
    for (bandId = CONFIGURATION_MANAGER_BAND_0 ; bandId < ConfigurationManager_GetNumOfActiveBands() ; bandId++)
	{
        RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_TWT_SW_HALT, twtSwHaltReg.val, bandId);
	} 
}

#endif //ENET_INC_ARCH_WAVE600B
#endif //ENET_INC_ARCH_WAVE600D2
/**********************************************************************************

TxSelector_UpdateBitmapAddTwtSta

Description: set the TWT Avail bit in the STA TID bitmap, and the TWT Type bit (announced = 0, unannounced = 1)
------------
	
Input:
-----
StaId staId
bool   twtType

	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_UpdateBitmapAddTwtSta(StaId staId, TwtManagerFlowType_e twtType)
{
    TxSelectorActionParams_t updateBitmapParams;
    
    updateBitmapParams.stationOrVapNum = staId;
    updateBitmapParams.queueType = HW_TX_Q_TYPE_STA_TID; 
//#ifndef ENET_INC_ARCH_WAVE600B under comment due to a bug in HW
    if (twtType == TWT_MANAGER_FLOW_TYPE_ANNOUNCED)
    {
	    updateBitmapParams.action = TX_SELECTOR_TWT_TYPE_ANNOUNCED;
    }
    else
    {
        updateBitmapParams.action = TX_SELECTOR_TWT_TYPE_UNANNOUNCED;
    }	
    txSelectorUpdateBitmap(&updateBitmapParams, TX_SELECTOR_TWT_ANNOUNCED_TYPE);
//#endif
    updateBitmapParams.action = TX_SELECTOR_TWT_AVAILABLE;
    txSelectorUpdateBitmap(&updateBitmapParams, TX_SELECTOR_TWT_AVAILABILITY); 
    ILOG0_V("TxSelector_UpdateBitmapAddTwtSta");
}


/**********************************************************************************

TxSelector_UpdateBitmapRemTwtSta

Description: reset the TWT Avail bit in the STA TID bitmap
------------
	
Input:
-----
StaId staId
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_UpdateBitmapRemTwtSta(StaId staId)
{ 
    TxSelectorActionParams_t updateBitmapParams;
    
    updateBitmapParams.stationOrVapNum = staId;
	updateBitmapParams.action = TX_SELECTOR_TWT_NOT_AVAILABLE; 
	updateBitmapParams.queueType = HW_TX_Q_TYPE_STA_TID;
    txSelectorUpdateBitmap(&updateBitmapParams, TX_SELECTOR_TWT_AVAILABILITY);
    ILOG0_V("TxSelector_UpdateBitmapRemTwtSta");
}



#endif //ENET_INC_UMAC


#if defined (ENET_INC_LMAC)
#ifdef WORKAROUND_FOR_HW_BUG_IN_PS_SETTING_IN_TWT
/**********************************************************************************
TxSelector_ResetPsRequestforTwt
 
Input: 
-----
			
Output:
-------	

Returns:
--------
	void - 
	
**********************************************************************************/
void TxSelector_ResetPsRequestForTwt(StaId staId)
{    
    TxSelectorActionParams_t    txSelectorActionParams;
    uint32                      staTidBitmapWord3;
    uint8                       twtAvail;
    uint8                       twtSp;

    staTidBitmapWord3   = pTxSelectorRam->staTidBitmap[staId].word3;
    twtAvail            = staTidBitmapWord3 & TX_SELECTOR_MASK_FOR_TWT_AVAIL;
    twtSp               = staTidBitmapWord3 & TX_SELECTOR_MASK_FOR_TWT_SP;
    
    if ((twtAvail == 1) && (twtSp == 0))
    {
        ILOG0_V("[TxSelector_ResetPsRequestForTwt], reset PsReq");
        /*Clear PS request*/    
        txSelectorActionParams.queueType = HW_TX_Q_TYPE_STA_TID; 
        txSelectorActionParams.stationOrVapNum = staId;
        txSelectorActionParams.tidEnableBitmap = TID_BITMAP_ALL_TIDS;
	    txSelectorActionParams.tidValueBitmap = TX_SELECTOR_RESET_POWER_SAVE_REQUEST * TID_BITMAP_ALL_TIDS;
        TxSelector_SetPowerSaveRequest(&txSelectorActionParams);       
    } 
}

#endif //WORKAROUND_FOR_HW_BUG_IN_PS_SETTING_IN_TWT
#endif //ENET_INC_LMAC


/**********************************************************************************

TxSelector_GetLockRequestsStatus 

Description:
------------
	Returns the status of the last Lock Request 

Input:
-----
	None

Returns:
--------
	TxSelectorLockStatus_e - Request Status
	
**********************************************************************************/
#if defined (ENET_INC_UMAC)
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(uint8 bandId)
#else
TxSelectorLockStatus_e TxSelector_GetLockRequestsStatus(void)
#endif //ENET_INC_UMAC
{
	RegTxSelectorTxSelSwMaster0LockerStatus_u lockerStatusReg;
	TxSelectorLockStatus_e lockerStatus;
	
	lockerStatusReg.val = 0;

#if defined (ENET_INC_UMAC)
	RegAccess_ReadPerBand(REG_TX_SELECTOR_LOCKER_STATUS,&lockerStatusReg.val, bandId);
#else
	RegAccess_Read(REG_TX_SELECTOR_LOCKER_STATUS,&lockerStatusReg.val);
#endif


	lockerStatus = (TxSelectorLockStatus_e)lockerStatusReg.bitFields.swMaster0ReqStatus;

	return lockerStatus;
}		



#ifdef USE_AGER_EMULATOR

/**********************************************************************************

TxSelector_TryLockRequest 


Description:
------------
This function tries to lock a queue and if it is locked it returns immidiately.
This function should be used only by the Ager emulator.

Input:
-----
stationOrVapIndex - the index of the station/VAP
tid - the TID
queueType - queue type (should be only station or VAP)



Output:
-------
	

Returns:
--------
The lock request status

	
**********************************************************************************/
TxSelectorLockStatus_e TxSelector_TryLockRequest(uint16 stationOrVapIndex , uint8 tid, TxQueueType_e queueType)
{
	RegTxSelectorTxSelSwMaster1Locker_u lockParametersRegister;
	RegTxSelectorTxSelSwMaster1LockerStatus_u lockStatusRegister;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif //ENET_INC_UMAC

	lockParametersRegister.val = 0;
	lockStatusRegister.val = 0;

	if(queueType == HW_TX_Q_TYPE_STA_TID)
	{
		ASSERT(stationOrVapIndex < HW_NUM_OF_STATIONS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForStation(stationOrVapIndex);
#endif //ENET_INC_UMAC
	}
	else
	{			
		ASSERT(stationOrVapIndex < HW_NUM_OF_VAPS);
#if defined (ENET_INC_UMAC)
		bandId = ConfigurationManager_GetBandForVap(stationOrVapIndex);
#endif //ENET_INC_UMAC
	}

	lockParametersRegister.bitFields.master1LockerStaSel = stationOrVapIndex;
	lockParametersRegister.bitFields.master1LockerTidSel = tid;
	lockParametersRegister.bitFields.master1LockerBitmapSel = queueType;
	lockParametersRegister.bitFields.master1LockerBitTypeSel = TX_SELECTOR_LOCK_REQ_TYPE_TRY_LOCK;
	lockParametersRegister.bitFields.master1LockerAllTids = TX_SELECTOR_SINGLE_TID_UPDATE;

#if defined (ENET_INC_UMAC)
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER, lockParametersRegister.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER, lockParametersRegister.val);
#endif

	/* Wait till result is ready */
	while(!lockStatusRegister.bitFields.master1LockerDoneInt)
	{
#if defined (ENET_INC_UMAC)
		RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS,&lockStatusRegister.val, bandId);
#else
		RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS,&lockStatusRegister.val);
#endif
	}

	return ((TxSelectorLockStatus_e)lockStatusRegister.bitFields.swMaster1ReqStatus);
}

#endif //USE_AGER_EMULATOR

#if defined (ENET_INC_LMAC)

/**********************************************************************************

isr_TxSelector_LockerRequestServiced_Band0 

Description:
------------
	Isr when Locker has finished the requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
ISR_VOID isr_TxSelector_LockerRequestServiced_Band0(void)
{
	RegTxSelectorTxSelLockerIntClr_u lockerIntClearReg;
	RegTxSelectorTxSelSwMaster2LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
		
	
	/* Read  lock request status */
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER2_LOCKER_STATUS,&lockerStatus.val);

	/* Clear  locker interrutpt */
	lockerIntClearReg.val = 0;
	lockerIntClearReg.bitFields.swMaster2LockerIntClr = TRUE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOCKER_INT_CLR,lockerIntClearReg.val);

	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster2ReqStatus;
		
	Locker_LockRequestServiced(lockRequestStatus);

}


/**********************************************************************************

isr_TxSelector_LockerRequestServiced_Band1 

Description:
------------
	Isr when Locker has finished the requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
ISR_VOID isr_TxSelector_LockerRequestServiced_Band1(void)
{
	RegTxSelectorTxSelLockerIntClr_u lockerIntClearReg;
	RegTxSelectorTxSelSwMaster3LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
		

	/* Read  lock request status */
	RegAccess_Read(REG_TX_SELECTOR_TX_SEL_SW_MASTER3_LOCKER_STATUS,&lockerStatus.val);

	/* Clear  locker interrutpt */
	lockerIntClearReg.val = 0;
	lockerIntClearReg.bitFields.swMaster3LockerIntClr = TRUE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOCKER_INT_CLR,lockerIntClearReg.val);

	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster3ReqStatus;
		
	Locker_LockRequestServiced(lockRequestStatus);

}

#else //ENET_INC_UMAC

/**********************************************************************************

isr_TxSelector_LockerRequestServiced_Band0 

Description:
------------
	Isr when Locker of band0 has finished the UM requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
ISR_VOID isr_TxSelector_LockerRequestServiced_Band0(void)
{
	RegTxSelectorTxSelLockerIntClr_u lockerIntClearReg;
	RegTxSelectorTxSelSwMaster1LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
		
	
	/* Read  lock request status */
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS,&lockerStatus.val, CONFIGURATION_MANAGER_BAND_0);

	/* Clear  locker interrutpt */
	lockerIntClearReg.val = 0;
	lockerIntClearReg.bitFields.swMaster1LockerIntClr = TRUE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOCKER_INT_CLR,lockerIntClearReg.val, CONFIGURATION_MANAGER_BAND_0);

	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster1ReqStatus;
		
	Locker_LockRequestServiced(lockRequestStatus);

}


/**********************************************************************************

isr_TxSelector_LockerRequestServiced_Band1 

Description:
------------
	Isr when Locker of band1 has finished the UM requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
ISR_VOID isr_TxSelector_LockerRequestServiced_Band1(void)
{
	RegTxSelectorTxSelLockerIntClr_u lockerIntClearReg;
	RegTxSelectorTxSelSwMaster1LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
		
	
	/* Read  lock request status */
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_SW_MASTER1_LOCKER_STATUS,&lockerStatus.val, CONFIGURATION_MANAGER_BAND_1);

	/* Clear  locker interrutpt */
	lockerIntClearReg.val = 0;
	lockerIntClearReg.bitFields.swMaster1LockerIntClr = TRUE;
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_LOCKER_INT_CLR,lockerIntClearReg.val, CONFIGURATION_MANAGER_BAND_1);

	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster1ReqStatus;
		
	Locker_LockRequestServiced(lockRequestStatus);

}


/**********************************************************************************

isr_TxSelector_LockerGroupRequestServiced 

Description:
------------
	Isr when Locker has finished the requested action

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
void isr_TxSelector_LockerGroupRequestServiced(uint8 bandId)
{
	RegTxSelectorTxSelMuSwMaster1LockerStatus_u lockerStatus;
	TxSelectorLockStatus_e lockRequestStatus;
	RegTxSelectorTxSelMuLockerIntClr_u	lockClrInterruptReg;

	/* Read  lock request status */
	RegAccess_ReadPerBand(REG_TX_SELECTOR_TX_SEL_MU_SW_MASTER1_LOCKER_STATUS,&lockerStatus.val, bandId);
	lockRequestStatus = (TxSelectorLockStatus_e)lockerStatus.bitFields.swMaster1MuReqStatus;

	/* Clear  lock interrupt status */
	lockClrInterruptReg.val = 0; 
	lockClrInterruptReg.bitFields.swMaster1MuLockerIntClr = 1; 
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_MU_LOCKER_INT_CLR, lockClrInterruptReg.val, bandId);
	
	Locker_LockGroupRequestServiced(lockRequestStatus);
}

/**********************************************************************************

isr_TxSelector_LockerPlanRequestServiced 

Description:
------------
	Isr when Locker has finished the plan lock

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
void hwEventsTxSelectorLockerPlanRequestServiced(uint8 bandId)
{
	RegTxSelectorTxSelPlanSwMaster1LockerStatus_u	lockerPlanStatus;
	TxSelectorLockStatus_e 							lockRequestStatus;
	RegTxSelectorTxSelPlanLockerIntClr_u			lockClrInterruptReg;

#ifdef DYNAMIC_GROUPING_DEBUG
    ILOG0_D("[hwEventsTxSelectorLockerPlanRequestServiced], bandId = %d", bandId);
#endif

	/* Read  lock request status */
	RegAccess_ReadPerBand(REG_TX_SELECTOR_PLAN_LOCKER_STATUS, &lockerPlanStatus.val, bandId);
	lockRequestStatus = (TxSelectorLockStatus_e)lockerPlanStatus.bitFields.swMaster1PlanReqStatus;

	/* Clear  lock interrupt status */
	lockClrInterruptReg.val = 0; 
	lockClrInterruptReg.bitFields.swMaster1PlanLockerIntClr = 1; 
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_PLAN_LOCKER_INT_CLR, lockClrInterruptReg.val, bandId);
	
	Locker_LockPlanRequestServiced(lockRequestStatus);
}


#endif 
/**********************************************************************************

TxSelector_ModifyPrimaryStaInfo


Description:
------------
This function configures primary STA for specified group. The change will be reflected to STA MU DB in Selector's RAM

Input:
-----
staId - the sta id to map
primaryStaInfo :
	- groupIdx - group index to map STA to
	- isPrimaryStaInGroup - determines whether the STA will be primary in the mapped group (if not and this STA will be chosen by SU selector - it will not continue as MU Tx).

Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_ModifyPrimaryStaInfo(StaId staId, TxSelectorPrimaryStaInfo_u primaryStaInfo)
{
	TxSelectorMuActionParams_t muBitmapParams;

	muBitmapParams.allStas = TX_SELECTOR_SINGLE_STA_UPDATE;
	muBitmapParams.groupIdx = staId;
	muBitmapParams.action = primaryStaInfo.val;
	
	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_STA_GROUP_MAPPING);
}


/**********************************************************************************

TxSelector_MapAllStasToGroup


Description:
------------
This function maps 4 STAs to specified group in MU Group DB

Input:
-----
groupIdx - the sta id to map
allStaInfo : TxSelectorGroupUserPositionStaInfo_t per user position : Sta id and valid bit

Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_MapAllStasToGroup(uint8 groupIdx, RegTxSelectorAllStaBitmapUpdate_u allStaInfo, TxSelectorAllGroupValidUspBitmap_t allGroupValidUspBitmap)
{

	TxSelectorMuActionParams_t muBitmapParams;
#if defined (ENET_INC_UMAC)
	uint8 bandId;
#endif 

#if defined (ENET_INC_UMAC)
	bandId = ConfigurationManager_GetBandForGroup(groupIdx);
	RegAccess_WritePerBand(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE_ALL_STA_WDATA, allStaInfo.val, bandId);
#else
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_BITMAP_UPDATE_ALL_STA_WDATA, allStaInfo.val);
#endif

	muBitmapParams.allStas = TX_SELECTOR_ALL_STAS_UPDATE;
	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.allGroupValidUspBitmap = allGroupValidUspBitmap.validUspBitmap;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_USER_POSITION_STA);

}


/**********************************************************************************

TxSelector_UpdateStaInGroup


Description:
------------
This function updates inputted STA to be in the requested user position in the group and sets it as primary if requested

Input:
-----
groupIdx - the  group index to add sta to
staInfo :
		- sta id - STA id to add/modify
		- is valid - is the STA id valid in the Group
		
staPrimaryEnabled - indicate whether need to update the STA also as primary in the Group


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_UpdateStaInGroup(uint8 groupIdx, uint8 userPosition, TxSelectorGroupUserPositionStaInfo_u staInfo, TxSelectorAllGroupValidUspBitmap_t allGroupValidUspBitmap)
{
	TxSelectorMuActionParams_t muBitmapParams;

	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.userPositionIdx = userPosition;
	muBitmapParams.action = staInfo.val;
	muBitmapParams.allGroupValidUspBitmap = allGroupValidUspBitmap.validUspBitmap;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_USER_POSITION_STA);
}


/**********************************************************************************

TxSelector_UpdateGroupTxMode


Description:
------------
This function updates Group Tx mode

Input:
-----
groupIdx - the  group index
tx mode - the Tx mode to update the group to


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_UpdateGroupTxMode(uint8 groupIdx, TxSelectorGroupTxMode_e txMode)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.action = txMode;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_TX_MODE);
}



/**********************************************************************************

TxSelector_UnlockGroup


Description:
------------
This function unlocks group
Input:
-----
groupIdx - the  group index to unlock


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_UnlockGroup(uint8 groupIdx)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.groupIdx = groupIdx;
	muBitmapParams.action = TX_SELECTOR_UNLOCK_GROUP;

	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_GROUP_LOCK);

}


/**********************************************************************************

TxSelector_ConfigureStaMuEnabledTids


Description:
------------
This function configures the enabled tids for MU Tx for specified STA

Input:
-----
staId - the  STA ID to update
tidBitmap - 8 bit per TID : 1- Enable, 0- Disable


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_ConfigureStaMuEnabledTids(StaId staId, RegTxSelectorMuTidBitmap_u tidBitmap)
{
	TxSelectorMuActionParams_t muBitmapParams;
	
	muBitmapParams.groupIdx = staId;
	muBitmapParams.action = tidBitmap.val;
	
	txSelectorUpdateMuBitmap(&muBitmapParams, TX_SELECTOR_STA_MU_TID_ENABLE);
}


#if defined (ENET_INC_LMAC)

/**********************************************************************************

TxSelector_GetLastMuSelection 

Description:
------------
	Returns the details of the MU TX selector last selection of specified USP
Input:
-----
	TxSelectorMuSelectionDetails_t * selectorMuSelectionDetails - Selection details
	uint8 userPosition - user position to read
Output:
-------
	The last selection details of the MU selector
Returns:
--------
	void
**********************************************************************************/
void TxSelector_GetLastMuSelection(TxSelectorMuSelectionDetails_t* selectorMuSelectionDetails, uint8 userPosition)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_MU_UP0_RESULT + (userPosition * (REG_TX_SELECTOR_TX_SEL_MU_UP1_RESULT - REG_TX_SELECTOR_TX_SEL_MU_UP0_RESULT));
	
	txSelectorFillMuSelectionParams(selectorMuSelectionDetails, selectionRegAddress);

}


/**********************************************************************************

TxSelector_SetBfSequenceRequest 

Description:
------------
	This procedure sets the BF request bit, next time the related Q will be chosen a BF sequnce will be transmitted

Input:
-----
		- 
		- Indicates if we should perform action for all TIDs/ACs
		
		-- queue type  -  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		-- VAP number if the queue is VAP or STA number if the queue is STA
		-- tid/AC number of the queue to be disabled
		-- Action:
				0x0 - Reset BF Request   
				0x1 - Set BF Request
		-- allTids Indicates if we should perform action for all TIDs/ACs
		
**********************************************************************************/
void TxSelector_SetBfSequenceRequest(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_BF_SEQUENCE_REQUEST);
}

/**********************************************************************************

TxSelector_SetBfSequenceType 

Description:
------------
	This procedure sets the BF sequence type: 
		0 - without data	
		1 - with data: 		
		
Input:
-----
	TxSelectorActionParams_t * txSelectorActionParams
		-- queue type  -  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
		-- VAP number if the queue is VAP or STA number if the queue is STA
		-- tid/AC number of the queue to be disabled
		-- Action:
				0x0 - BF Sequence with no data transmission following it
				0x1 - BF sequence is followed by data transmission  
		-- allTids Indicates if we should perform action for all TIDs/ACs

**********************************************************************************/
void TxSelector_SetBfSequenceType(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_BF_SEQUENCE_TYPE);
}




/**********************************************************************************

TxSelector_SetPowerSavemode 

Description:
------------
	This procedure Set power save mode of the queue specified by the parameters queue, stationVapNum, tidNum 
Input:
-----
	TxSelectorUpdateActionParams_t * txSelectorUpdateActionParams
	- The queue type to set power save mode  (STA,VAP,VIP,GLOBAL,GPLP,GPHP,BEACON)
	- The VAP number if the queue is VAP or STA number if the queue is STA
	- The tid/AC number of the queue 
	- Power Save Mode 0x0 - Active , 0x1 - Power Save
	- Should we perform action for all TIDs/ACs
Output:
-------
	None
Returns:
--------
	void	
**********************************************************************************/
void TxSelector_SetPowerSavemode(TxSelectorActionParams_t *txSelectorActionParams)
{
	txSelectorUpdateBitmap(txSelectorActionParams,TX_SELECTOR_ACTIVE_PS_MODE);
}



/**********************************************************************************

TxSelector_SetVapWrrPriority 

Description:
------------
	This configuration is used by the TX Selector in the 2 ports weighted round robin performed 
	for determining which source will be selected for input A of the strict priority selector.
	there are two registers for STA PS priority. TX_SELECTOR_STA_PS_WRR_LOW holds    
	priorities for stations associated with VAP 0-7, TX_SELECTOR_STA_PS_WRR_HIGH  
	holds priorities for stations associated with VAP 9-15. VAP priority can be 0-7
	and requires 3 bits.
Input:
-----
	uint8 vapNum - Vap Number 
	uint8 wrrPsPriority	- WRR Ps Priority value	
	uint8 wrrActivePriority - WRR Active Priority value	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetVapWrrPriority(uint8 vapNum,TxSelectorWrrPriority_t wrrPsPriority, TxSelectorWrrPriority_t wrrActivePriority)
{
	txSelectorSetWrrPriority(vapNum, wrrPsPriority, REG_TX_SELECTOR_TX_SEL_STA_PS_WRR_LOW0);
	txSelectorSetWrrPriority(vapNum, wrrActivePriority, REG_TX_SELECTOR_TX_SEL_STA_ACTIVE_WRR_LOW0);
}

/**********************************************************************************

TxSelector_SetStrictPriority 

Description:
------------
	Configures priority of between Global Q Vap MPS & Sta .
			1 - Highest Priority
			3 - Lowest Priority
	In order for priority to function properly all 3 priorities must be set with different values.
Input
-----
	TxSelectorStrictPriority_e globalPriority - Global source priority.
	TxSelectorStrictPriority_e vapMpsPriority - Vap MPS source priority.
	TxSelectorStrictPriority_e activePriority - Active source priority - Include Active vaps, Active Station & PS Stations.
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SetStrictPriority(TxSelectorStrictPriority_e globalPriority, TxSelectorStrictPriority_e vapMpsPriority, TxSelectorStrictPriority_e activePriority)
{
	RegTxSelectorTxSelStrictPrio_u strictPriorityReg;

	strictPriorityReg.val = 0;
    RegAccess_Read(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,&strictPriorityReg.val);
	strictPriorityReg.bitFields.strictPrioGlobalPrio = globalPriority;
	strictPriorityReg.bitFields.strictPrioVapMpsPrio = vapMpsPriority;
	strictPriorityReg.bitFields.strictPrioWrrPrio = activePriority;

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,strictPriorityReg.val);
}

/**********************************************************************************

TxSelector_MaskGroupSelection 

Description:
------------
	Bits [4:0] mask possible selections at the strict priority selection stage. 
	Bit value 0 - selection is masked, Bit value 1- selection is not masked. 
	Bit 0 - STA TID , Bit2 - VAP TID ,Bit2 - Global 
	Bit3 - GPHP		, Bit4 - GPLP	 ,Bit5 - Beacon
Input:
-----
	TxSelectorGroupSelectionOffset_e gruopOffset
	TxSelectorGroupSelectionEnable_e selectionEnable
Output:
-------
	None
Returns:
--------
	void
**********************************************************************************/
void TxSelector_MaskGroupSelection(TxSelectorGroupSelectionOffset_e gruopOffset,TxSelectorGroupSelectionEnable_e selectionEnable)
{
	RegTxSelectorTxSelStrictPrio_u groupSelectionReg;

	groupSelectionReg.val = 0;

    RegAccess_Read(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,&groupSelectionReg.val);

    groupSelectionReg.bitFields.strictPrioGroupSelEn =  selectionEnable << gruopOffset;

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STRICT_PRIO,groupSelectionReg.val);
}


/**********************************************************************************

TxSelector_GetLastSelection 

Description:
------------
	Returns the details of the TX selector last selection
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelectionDetails - - Selection details
Output:
-------
	The last selection details of the selector
Returns:
--------
	void
**********************************************************************************/
void TxSelector_GetLastSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_SELECTION;

	txSelectorFillSelectionParams(selectorSelectionDetails,selectionRegAddress);
}


/**********************************************************************************

TxSelector_GetLastSourceSelection 

Description:
------------
	Returns the details of the specified source. The TX selector enters to the strict priority
	selector sources 0-5. This routine will return the details of the specified source.
Input:
-----
	TxSelectorSelectionDetails_t * selectorSelectionDetails
	TxSelectorSource_e sourceNum - T
Output:
-------
	The selection Details according to the source
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_GetLastSourceSelection(TxSelectorSelectionDetails_t* selectorSelectionDetails,TxSelectorSource_e sourceNum)
{
	uint32 selectionRegAddress =  REG_TX_SELECTOR_TX_SEL_SRC0_SELECTION + (sourceNum * (REG_TX_SELECTOR_TX_SEL_SRC1_SELECTION - REG_TX_SELECTOR_TX_SEL_SRC0_SELECTION));

	txSelectorFillSelectionParams(selectorSelectionDetails, selectionRegAddress);
}		



/**********************************************************************************

TxSelector_IsDataInQueue 


Description:
------------
Returns if there is a data in a specific queue


Input:
-----
stationOrVapIndex - the index of the station/VAP
tid - the TID
queueType - queue type (should be only station or VAP)



Output:
-------
	

Returns:
--------
Indication if data exists on the specific queue

	
**********************************************************************************/
uint8 TxSelector_IsDataInQueue(uint16 stationOrVapIndex, uint8 tid, TxQueueType_e queueType)
{
	UNUSED_PARAM(stationOrVapIndex);	
	UNUSED_PARAM(tid);	
	UNUSED_PARAM(queueType);	
	FATAL("TxSelector_IsDataInQueue - back door not available in gen6");
	return FALSE;
}



/**********************************************************************************

isr_TxSelector_Error 

Description:
------------
	Isr for error scenarios in the TX Selector

Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void - 
**********************************************************************************/
void isr_TxSelector_Error(void)
{
	FATAL("ISR TX Selector Error");
}


/**********************************************************************************

TxSelector_EnableDisableMuTx



Description:
------------
This function enables/disables MU for the entire system, with the following fetures:

- determining group threshold
- determining secondary tid selection method
- determining lock policy after selection

Input:
-----
maxGroupNum - max number of groups - for selection of group above the threshold, the Tx will switch to SU
muSecondaryTid - secondary tid selection method
muLockPolicy - mu lock policy
pMuSelectCfg - config all group sizes threshold and SU fallback



Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_EnableDisableMuTx(TxSelectorEnableDisableMu_e muEnableDisable, uint8 maxGroupNum, TxSelectorMuSecondarySelectionMethod_e muSecondaryTid, TxSelectorMuLockAfterSelection_e muLockPolicy, TxSelectorMuSelectCfg_u *pMuSelectCfg)
{

	RegTxSelectorTxSelMuSelectCfg_u muSelectConfigReg;
		
	muSelectConfigReg.val = 0;
	muSelectConfigReg.bitFields.muEn = muEnableDisable;
	muSelectConfigReg.bitFields.muSecondarySel = muSecondaryTid;
	muSelectConfigReg.bitFields.maxGroupNum = maxGroupNum;
	muSelectConfigReg.bitFields.selectorMuLockEn = muLockPolicy;

    muSelectConfigReg.bitFields.suSelEn1Sec   = pMuSelectCfg->bitFields.suSelEn1Sec;
    muSelectConfigReg.bitFields.muThresh1Sec  = pMuSelectCfg->bitFields.muThresh1Sec;
    muSelectConfigReg.bitFields.suSelEn2Sec   = pMuSelectCfg->bitFields.suSelEn2Sec;
    muSelectConfigReg.bitFields.muThresh2Sec  = pMuSelectCfg->bitFields.muThresh2Sec;
    muSelectConfigReg.bitFields.suSelEn3Sec   = pMuSelectCfg->bitFields.suSelEn3Sec;
    muSelectConfigReg.bitFields.muThresh3Sec  = pMuSelectCfg->bitFields.muThresh3Sec;
	
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_SELECT_CFG, muSelectConfigReg.val);
}


/**********************************************************************************

TxSelector_SetPrimaryPdCountThreshold


Description:
------------
This function sets pd count primary threshold


Input:
-----
threshold - Primary STA queue should include at least this number of PDs in order for Tx to be MU


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/
void TxSelector_SetPrimaryPdCountThreshold(uint32 threshold)
{
	RegTxSelectorTxSelMuPrimaryThres_u primaryThreshold;	
	
	primaryThreshold.bitFields.pdPrimaryThres = threshold;	
	
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_PRIMARY_THRES, primaryThreshold.val);

}



/**********************************************************************************

TxSelector_SetSecondaryPdCountThreshold



Description:
------------
This function sets pd count secondary threshold


Input:
-----
threshold - Secondary STA queue should include at least this number of PDs in order for the STA to be included in MU Tx


Output:
-------
none	

Returns:
--------
void
	
**********************************************************************************/

void TxSelector_SetSecondaryPdCountThreshold(uint32 threshold)
{
	RegTxSelectorTxSelMuSecondaryThres_u secondaryThreshold;	
		
	secondaryThreshold.bitFields.pdSecondaryThres = threshold;	
		
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_SECONDARY_THRES, secondaryThreshold.val);

}



/**********************************************************************************

TxSelector_ExecuteInstruction 

Description:
------------
	Execute an instruction of the Tx Selector.
Input:
-----
	TxSelectorInstructions_e instruction - The instruction to execute
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_ExecuteInstruction (TxSelectorInstructions_e instruction)
{
	RegTxSelectorTxSelInstr_u instructionReg;

	ASSERT(instruction < TX_SELECTOR_NUM_OF_INSTRUCTION);

	instructionReg.val = 0;
	instructionReg.bitFields.instr = instruction;
	
    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_INSTR,instructionReg.val);
}

/**********************************************************************************

TxSelector_SelectModeConfig 

Description:
------------
	Config selection modes.
Input:
-----
	dlWithUlEnable - enable/disable DL with UL feature
	mgmtBypassEnable - enables/disable TID8 MGMT bypass
	mgmtAcMap - map TID8 to specific AC
	staWithLockPlanSelect - enable/disable selecting STA with locked plan
	
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_SelectModeConfig(TxSelectorDlWithUlEnable_e dlWithUlEnable, TxSelectorMgmtBypassEnable_e mgmtBypassEnable, AccessCategory_e mgmtAcMap, TxSelectorHeMuFallBack_e heMuFallBack)
{
    RegTxSelectorTxSelSelectModeCfg_u selectModeCfgReg;

	ASSERT(mgmtAcMap < ACCESS_CATEGORY_NUM);

    selectModeCfgReg.val = 0;

    selectModeCfgReg.bitFields.dlWithUlEn           = dlWithUlEnable;
    selectModeCfgReg.bitFields.mgmtBypassEn         = mgmtBypassEnable;  // Enables MGMT selection over other TIDs feature
    selectModeCfgReg.bitFields.mgmtAcNum            = mgmtAcMap;
    selectModeCfgReg.bitFields.heMuFallbackSu		= heMuFallBack;      // Enables selection fallback from HE MU to SU in case of plan lock.

    RegAccess_Write(REG_TX_SELECTOR_TX_SEL_SELECT_MODE_CFG ,selectModeCfgReg.val);

}
#ifdef ENET_INC_ARCH_WAVE600B
/**********************************************************************************

TxSelector_MbssidConfig 

Description:
------------
	Config MBSSID params.
Input:
-----
	uint8 transmitVapId, 
	uint8 numOfVapsInGroup
Output:
-------
	None
Returns:
--------
	void 
**********************************************************************************/
void TxSelector_MbssidConfig(uint8 transmitVapId, uint8 numOfVapsInGroup)
{
	RegTxSelectorTxSelTimMultiBssidCfg_u txSelTimMbssidCfg;

	txSelTimMbssidCfg.val = 0; 			
	txSelTimMbssidCfg.bitFields.timIeCompressed = 0;	
	txSelTimMbssidCfg.bitFields.primaryVapIs16 = transmitVapId >> 4;
	txSelTimMbssidCfg.bitFields.multiBssidGroupSize = numOfVapsInGroup;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_TIM_MULTI_BSSID_CFG, txSelTimMbssidCfg.val);
}
#endif
/**********************************************************************************

TxSelector_Init 

Description:
------------
	Init vapSta bitmaps - no stations area associated with vaps.
	Set Sta & VAp rams to default values (Q disabled and all other values are 0)
	GLOBAL,GPHP,GPLP,BEACON queues bitmap are all stored in FF, the sw reset initializes these FFF to default values
	Init MU sTA & Group bitmaps
	Enable SU & MU Lockers
	Set Primary and secondary PD thresholds for MU transmission
Input:
-----
	void
Output:
-------
	None
Returns:
--------
	void  
**********************************************************************************/
void TxSelector_Init(void)
{
	RegTxSelectorTxSelLockerEn_u lockerEnReg;
	RegTxSelectorTxSelMuLockerEn_u muLockerEnReg;
    RegTxSelectorTxSelPlanLockerEn_u planLockerEnReg;
	RegTxSelectorTxSelStaAssociate_u selStaAssociateReg;
    TxSelectorMuSelectCfg_u muSelectCfg;


	lockerEnReg.val = 0;
	muLockerEnReg.val = 0;
    planLockerEnReg.val = 0;
	selStaAssociateReg.val = 0;
	muSelectCfg.val = 0;

	/* Reset Selector using SW Reset Block*/

	
	/* Initialize the pointer of the Tx Selector Ram 
	Tx Selector RAM start from MU VHT RAM and is followed by all other RAMs, this is why the pointer points to MU RAM*/
#if defined (ENET_INC_LMAC0)
	pTxSelectorRam = (TxSelectorRam_t *)(B0_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM); 
#elif defined (ENET_INC_LMAC1)
	pTxSelectorRam = (TxSelectorRam_t *)(B1_TX_SELECTOR_TX_SEL_VHT_MU_GRP_RAM); 
#endif

	/* Reset & Disable SU Bitmaps */
	txSelectorResetStationTidRam();
	txSelectorResetVapTidRam();
	txSelectorResetGlobalRam();
	txSelectorResetGplpRam();
	txSelectorResetGphpRam();
	txSelectorResetBeaconRam();


	/* Reset Vap\Sta Bitmaps  */
	selStaAssociateReg.bitFields.vapZero = TX_SELECTOR_VAP_ZERO_STA_BITMAP;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_STA_ASSOCIATE,selStaAssociateReg.val);

	/* Reset MU Group Bitmap */	
	txSelectorResetMuGroupRam();
	
	/* Reset MU STA Bitmap */
	txSelectorResetMuStationRam();

	/* Lock all the Plans */
    txSelectorResetPlanLockRam();

    /* Reset TWT STA and count Bitmaps*/
#ifndef ENET_INC_ARCH_WAVE600D2
    txSelectorResetTwtStaRam();
    txSelectorResetTwtCountRam();
#endif
	/* Enable SU Locker */
	lockerEnReg.bitFields.lockerEn = TX_SELECTOR_LOCKER_ENABLE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_LOCKER_EN,lockerEnReg.val);

	/* Enable MU Locker */
	muLockerEnReg.bitFields.muLockerEn = TX_SELECTOR_LOCKER_ENABLE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_MU_LOCKER_EN, muLockerEnReg.val);

	/* Enable Plan Locker */
	planLockerEnReg.bitFields.planLockerEn = TX_SELECTOR_LOCKER_ENABLE;
	RegAccess_Write(REG_TX_SELECTOR_TX_SEL_PLAN_LOCKER_EN, planLockerEnReg.val);

	//set PD thresholds
	TxSelector_SetPrimaryPdCountThreshold(TX_SELECTOR_PRIMARY_PD_COUNT_THRESHOLD);
	TxSelector_SetSecondaryPdCountThreshold(TX_SELECTOR_SECONDARY_PD_COUNT_THRESHOLD);

	//Enable MU Tx
    muSelectCfg.bitFields.suSelEn1Sec  = TX_SELECTOR_DISABLE_VHT_MU_FALLBACK;
    muSelectCfg.bitFields.muThresh1Sec = TX_SELECTOR_VHT_MU_SEC1_THRESHOLD_1;
    muSelectCfg.bitFields.suSelEn2Sec  = TX_SELECTOR_DISABLE_VHT_MU_FALLBACK;
    muSelectCfg.bitFields.muThresh2Sec = TX_SELECTOR_VHT_MU_SEC2_THRESHOLD_2;
    muSelectCfg.bitFields.suSelEn3Sec  = TX_SELECTOR_DISABLE_VHT_MU_FALLBACK;
    muSelectCfg.bitFields.muThresh3Sec = TX_SELECTOR_VHT_MU_SEC3_THRESHOLD_3;

	TxSelector_EnableDisableMuTx(TX_SELECTOR_MU_ENABLE, (TX_MU_GROUPS - 1), TX_SELECTOR_MU_ANY_SECONDARY_TID_ALLOWED, TX_SELECTOR_MU_LOCK_AFTER_SELECTION, &muSelectCfg);
	
    //Config select mode
#if defined	(WORKAROUND_FOR_HW_BUG_IN_TIM_IE)
	TxSelector_SelectModeConfig(TX_SELECTOR_DL_WITH_UL_ENABLE, TX_SELECTOR_MGMT_BYPASS_ENABLE, ACCESS_CATEGORY_BE, TX_SELECTOR_FALLBACK_TO_SU);
#else
	TxSelector_SelectModeConfig(TX_SELECTOR_DL_WITH_UL_ENABLE, TX_SELECTOR_MGMT_BYPASS_DISABLE, ACCESS_CATEGORY_BE, TX_SELECTOR_FALLBACK_TO_SU);
#endif
}

#endif //! defined (ENET_INC_UMAC)


