/***********************************************************************************
 File:		TxSequencer.c
 Module:		Tx Sequencer
 Purpose:		
 Description:	FW driver for TX Selector module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "TxSequencer_Api.h"
#include "TxSequencerRegs.h"
#include "Pac_Api.h"
#include "ShramNewDeliaRecipes.h"
#include "RegAccess_Api.h"
#include "loggerAPI.h"
#include "stringLibApi.h"
#include "ShramLinkAdaptation.h"
#include "ErrorHandler_Api.h"
#include "Protocol_AirTimeAndDuration.h"
#include "DeliaRecipes_Api.h"
#include "PreAggregator_Api.h"

/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_TX_HW_ACCELERATORS
#define LOG_LOCAL_FID 7


#define TX_SEQUENCER_DISABLE_11B_SAMPLE_CCA				(1)
#define TX_SEQUENCER_ENABLE_OWNER_BIT_ERROR				(1)
#define TX_SEQUENCER_ENABLE_FIFO_FULL_ERROR				(1)
#define TX_SEQUENCER_ENABLE_RA_DECREMENT_LESS_THAN_ZERO_IRQ		(1)
#define TX_SEQUENCER_ENABLE_BAA_NEEDS_START_WHILE_BUSY_ERROR	(1)

#define TX_SEQUENCER_MINIMAL_SEQUENCE_TIME_TH			(60)	/* Minimal frame transmit time */
#define TX_SEQUENCER_MINIMAL_SU_TXOP_TIME_TH			(60)	/* Minimal time for another frame in TXOP for SU */
#define TX_SEQUENCER_MINIMAL_VHT_MU_TXOP_TIME_TH		(300)	/* Minimal time for another frame in TXOP for VHT MU */
#define TX_SEQUENCER_MINIMAL_HE_MU_TXOP_TIME_TH			(3000)	/* Minimal time for another frame in TXOP for HE MU */
#define TX_SEQUENCER_MINIMAL_BF_SEQUENCE_TIME_TH		(140)	/* Minimal time for additional BF request iteration */
#define TX_SEQUENCER_MINIMAL_ETSI_FILLER_TIME_TH		(250)	/* Minimal time for ETSI filler - usec*/
#define TX_SEQUENCER_ADDITION_ETSI_FILLER_TXOP_TIME_TH	(500)	/* Addition time for ETSI filler - usec*/
#define TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TIMEOUT		(0)		/* When to check CCA after timeout */
#define TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TX_IN_NS		(7500)	/* When CCA is available for sampling after Tx - 7.5us*/
//#define TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_RX_IN_NS	(3500)	/* When CCA is available for sampling after Rx - 3.5us*/
#define TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_RX_IN_NS		(3700)	/* When CCA is available for sampling after Rx - 3.7us - this was chaged from 3.5 due to Elad Ben Yosef & Yossi Amar request */
#define TX_SEQUENCER_MAC_PHY_TX_TXOP_SIGNAL_EN			(1)		/* TX_TXOP signal should be reflected to the PHY */

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/					Static Function Declaration									
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/					Static Functions Definitions									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

TxSequencerReport_t     *readTxSequencerReportPtr;




uint16 TxSequencer_GetFifoNumEntriesCount()
{
	RegTxSequencerSequencerFifoNumEntriesCount_u regTxSequencerSequencerFifoNumEntriesCount;

	RegAccess_Read(REG_TX_SEQUENCER_SEQUENCER_FIFO_NUM_ENTRIES_COUNT,(uint32*)&regTxSequencerSequencerFifoNumEntriesCount);	

	return regTxSequencerSequencerFifoNumEntriesCount.bitFields.sequencerFifoNumEntriesCount;

}


void TxSequencer_DecFifoNumEntriesCount(uint8 value)
{
	RegTxSequencerSequencerFifoRdEntriesNum_u regTxSequencerSequencerFifoRdEntriesNum;

	regTxSequencerSequencerFifoRdEntriesNum.val = 0;
	regTxSequencerSequencerFifoRdEntriesNum.bitFields.sequencerFifoRdEntriesNum = value;
	
    RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_FIFO_RD_ENTRIES_NUM,regTxSequencerSequencerFifoRdEntriesNum.val);
}

void TxSequencer_GetFifoEntry(TxSequencerReport_t* pCurrentSequencerReport)
{
    MEMCPY (pCurrentSequencerReport, readTxSequencerReportPtr, sizeof (TxSequencerReport_t));
}
	
void TxSequencer_incrementFifoEntryPtr(void)
{
#if defined(ENET_INC_LMAC0) 	
		if ((readTxSequencerReportPtr + 1) > (&SequencerFifoReport[0] + MAX_TX_SEQUENCER_FIFO_SIZE - 1))
		{
			readTxSequencerReportPtr = &SequencerFifoReport[0];
		}
#elif defined(ENET_INC_LMAC1)
		if ((readTxSequencerReportPtr + 1) > (&SequencerFifoReport_B1[0] + MAX_TX_SEQUENCER_FIFO_SIZE - 1))
		{
			readTxSequencerReportPtr = &SequencerFifoReport_B1[0];
		}
#endif
		else
		{
			readTxSequencerReportPtr++;
		}
}

uint32 TxSequencer_GetStationIndex(TxSequencerReport_t* CurrentSequencerReport)
{
	uint32 staIndex;// = CurrentSequencerReport->staNumberUsp0;

	switch (CurrentSequencerReport->primaryUsp)
	{
		//KW_FIX_FW_M
		case 0:
		default:
			staIndex = CurrentSequencerReport->staNumberUsp0;
			break;
			
		case 1:
			staIndex = CurrentSequencerReport->staNumberUsp1;
			break;
		
		case 2:
			staIndex = CurrentSequencerReport->staNumberUsp2;
			break;
		
		case 3:
			staIndex = CurrentSequencerReport->staNumberUsp3;
			break;			
	}

	return staIndex;
}

void TxSequencer_SetMinimalTimeToSendCfEnd(uint8 band)
{
	RegTxSequencerMinimalCfEndTimeThreshold_u minimalCfEndTimeThreshholdReg;

	minimalCfEndTimeThreshholdReg.val = 0;

	if(band == BAND_2_4_GHZ)
	{
		minimalCfEndTimeThreshholdReg.bitFields.minimalCfEndTimeThreshold = Protocol_GetRtsCfEnd11bAirTime((LM_PHY_11B_RATE_1 - LM_PHY_11B_FIRST_RATE) + 1);
	}
	else
	{
		minimalCfEndTimeThreshholdReg.bitFields.minimalCfEndTimeThreshold = Protocol_GetRtsCfEnd11agAirTime(LM_PHY_11A_RATE_6 - LM_PHY_11A_FIRST_RATE); 
	}


	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_CF_END_TIME_THRESHOLD,minimalCfEndTimeThreshholdReg.val);	
	
}

void TxSequencer_SetMinimalSuTxopTimeThreshold(uint16 txopTimeThreshold)
{
	RegTxSequencerMinimalSuTxopTimeThreshold_u minimalSuTxopTimeThresholdReg;

	minimalSuTxopTimeThresholdReg.val = 0;

	DEBUG_ASSERT(txopTimeThreshold <= PRE_AGGREGATOR_TXOP_AIR_TIME_DURATION_LIMIT);
	
	minimalSuTxopTimeThresholdReg.bitFields.minimalSuTxopTimeThreshold = txopTimeThreshold;
	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_SU_TXOP_TIME_THRESHOLD, minimalSuTxopTimeThresholdReg.val);	
}

void TxSequencer_SetMinimalVhtMuTxopTimeThreshold(uint16 txopTimeThreshold)
{
	RegTxSequencerMinimalVhtMuTxopTimeThreshold_u minimalVhtMuTxopTimeThresholdReg;

	minimalVhtMuTxopTimeThresholdReg.val = 0;

	ASSERT(txopTimeThreshold <= PRE_AGGREGATOR_TXOP_AIR_TIME_DURATION_LIMIT);
	
	minimalVhtMuTxopTimeThresholdReg.bitFields.minimalVhtMuTxopTimeThreshold = txopTimeThreshold;
	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_VHT_MU_TXOP_TIME_THRESHOLD, minimalVhtMuTxopTimeThresholdReg.val);
}

void TxSequencer_SetMinimalHeMuTxopTimeThreshold(uint16 txopTimeThreshold)
{
	RegTxSequencerMinimalHeMuTxopTimeThreshold_u minimalHeMuTxopTimeThresholdReg;

	minimalHeMuTxopTimeThresholdReg.val = 0;

	ASSERT(txopTimeThreshold <= PRE_AGGREGATOR_TXOP_AIR_TIME_DURATION_LIMIT);
	
	minimalHeMuTxopTimeThresholdReg.bitFields.minimalHeMuTxopTimeThreshold = txopTimeThreshold;
	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_HE_MU_TXOP_TIME_THRESHOLD, minimalHeMuTxopTimeThresholdReg.val);
}

#ifdef ENET_INC_ARCH_WAVE600B
void TxSequencer_SetAdditionEtsiFillerTxopTimeThreshold(uint16 additionEtsiFillerTimeTxopThreshold)
{
	RegTxSequencerAdditionEtsiFillerTxopTimeThreshold_u txSeqAdditionEtsiFillerTxopTimeThReg;

	txSeqAdditionEtsiFillerTxopTimeThReg.val = 0;
	txSeqAdditionEtsiFillerTxopTimeThReg.bitFields.additionEtsiFillerTxopTimeThreshold  = additionEtsiFillerTimeTxopThreshold;

	RegAccess_Write(REG_TX_SEQUENCER_ADDITION_ETSI_FILLER_TXOP_TIME_THRESHOLD, txSeqAdditionEtsiFillerTxopTimeThReg.val);
}

void TxSequencer_SetMinimalEtsiFillerTimeThreshold(uint16 etsiFillerTimeThreshold)
{
	RegTxSequencerMinimalEtsiFillerTimeThreshold_u txSeqMinimalEtsiFillerTimeThReg;

	txSeqMinimalEtsiFillerTimeThReg.val = 0;
	txSeqMinimalEtsiFillerTimeThReg.bitFields.minimalEtsiFillerTimeThreshold  = etsiFillerTimeThreshold;

	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_ETSI_FILLER_TIME_THRESHOLD, txSeqMinimalEtsiFillerTimeThReg.val);
}

void TxSequencer_SetEtsiMode(uint8 etsiModeEnable)
{
	RegTxSequencerEtsiMode_u txSeqEtsiModeReg;

	txSeqEtsiModeReg.val = 0;
	txSeqEtsiModeReg.bitFields.etsiMode = etsiModeEnable;

	RegAccess_Write(REG_TX_SEQUENCER_ETSI_MODE, txSeqEtsiModeReg.val);
}
#endif

void TxSequencer_Init(void)
{
	RegTxSequencerTxSeqDelRecipePointer_u txSeqDelRecipePtrReg;
	RegTxSequencerSequencerCcaControl_u sequencerCcaControlReg;
	RegTxSequencerMinimalSeqTimeThreshold_u minimalSeqTimeThresholdReg;
	RegTxSequencerMinimalBfSeqTimeThreshold_u minimalBfSeqTimeThresholdReg;
	RegTxSequencerTxNavTimerSampleSifsComp_u txNavTimerSampleSifsCompReg;
	RegTxSequencerCcaSampleTimeDelayTimeout_u ccaSampleTimeDelayTimeoutReg; 
	RegTxSequencerMacPhyTxTxopSignalEn_u macPhyTxTxopSignalEnReg;
	RegTxSequencerTxSequencerErrIrqEn_u errIrqEnReg;
	RegTxSequencerSequencerFifoBaseAddr_u regTxSequencerSequencerFifoBaseAddr;
	RegTxSequencerSequencerFifoDepthMinusOne_u regTxSequencerSequencerFifoDepthMinusOne;
	RegTxSequencerCcaSampleTimeDelayTx_u regTxSequencerCcaSampleTimeDelayTx;
	RegTxSequencerCcaSampleTimeDelayRx_u regTxSequencerCcaSampleTimeDelayRx;
	RegTxSequencerMaxNumBfSeqIterations_u sequencerBfMaxNumBfIter;


	txSeqDelRecipePtrReg.val =0;
	sequencerCcaControlReg.val =0;
	minimalSeqTimeThresholdReg.val = 0;
	minimalBfSeqTimeThresholdReg.val = 0;
	txNavTimerSampleSifsCompReg.val = 0;
	ccaSampleTimeDelayTimeoutReg.val = 0;
	macPhyTxTxopSignalEnReg.val = 0;
	errIrqEnReg.val = 0;
	regTxSequencerCcaSampleTimeDelayTx.val = 0;
	regTxSequencerCcaSampleTimeDelayRx.val = 0;

	/* Set Ptr to Recipe 0*/
	txSeqDelRecipePtrReg.bitFields.txSeqDelRecipePointer = (uint32)&(pDeliaRecipes->DeliaRecipe0[DELIA_RECIPE_0_FIRST_COMMAND]);
	RegAccess_Write(REG_TX_SEQUENCER_TX_SEQ_DEL_RECIPE_POINTER,txSeqDelRecipePtrReg.val);

	/*Set maximum of BF iteration for sequence in case of timeout*/
	sequencerBfMaxNumBfIter.val = TX_SEQUENCER_MAX_NUM_OF_BF_ITER;

	RegAccess_Write(REG_TX_SEQUENCER_MAX_NUM_BF_SEQ_ITERATIONS,sequencerBfMaxNumBfIter.val);

	/* Disable CCA sample in Phy Mode 11B*/
	sequencerCcaControlReg.bitFields.forceDisable11BSampleCca = TX_SEQUENCER_DISABLE_11B_SAMPLE_CCA;
	RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_CCA_CONTROL,sequencerCcaControlReg.val);	
	minimalSeqTimeThresholdReg.bitFields.minimalSeqTimeThreshold = TX_SEQUENCER_MINIMAL_SEQUENCE_TIME_TH;
	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_SEQ_TIME_THRESHOLD,minimalSeqTimeThresholdReg.val);	


	/*configure SU & MU (both VHT and HE) txop minimal time thresholds*/
	TxSequencer_SetMinimalSuTxopTimeThreshold(TX_SEQUENCER_MINIMAL_SU_TXOP_TIME_TH);
	TxSequencer_SetMinimalVhtMuTxopTimeThreshold(TX_SEQUENCER_MINIMAL_VHT_MU_TXOP_TIME_TH);
	TxSequencer_SetMinimalHeMuTxopTimeThreshold(TX_SEQUENCER_MINIMAL_HE_MU_TXOP_TIME_TH);
#ifdef ENET_INC_ARCH_WAVE600B
	TxSequencer_SetMinimalEtsiFillerTimeThreshold(TX_SEQUENCER_MINIMAL_ETSI_FILLER_TIME_TH);
	TxSequencer_SetAdditionEtsiFillerTxopTimeThreshold(TX_SEQUENCER_ADDITION_ETSI_FILLER_TXOP_TIME_TH);
#endif
	
	minimalBfSeqTimeThresholdReg.bitFields.minimalBfSeqTimeThreshold = TX_SEQUENCER_MINIMAL_BF_SEQUENCE_TIME_TH;
	RegAccess_Write(REG_TX_SEQUENCER_MINIMAL_BF_SEQ_TIME_THRESHOLD,minimalBfSeqTimeThresholdReg.val);	

	//configure minimal time to send CF-END (for 5G band by default, it will be changed in linkAdaptationSetChannelReq function later)
	TxSequencer_SetMinimalTimeToSendCfEnd(BAND_5_2_GHZ);

	/* Set Compensation*/
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostTxOfdmSifsComp = Pac_GetHwDelays(PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TX_OFDM);//13;
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostTx11BSifsComp = Pac_GetHwDelays(PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TX_11B);//5;
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostRxOfdmSifsComp = Pac_GetHwDelays(PAC_DELAY_TX_SEQ_SIFS_COMP_POST_RX_OFDM);//9;
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostRx11BSifsComp = Pac_GetHwDelays(PAC_DELAY_TX_SEQ_SIFS_COMP_POST_RX_11B);//3; 
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostTimeoutSifsComp = Pac_GetHwDelays(PAC_DELAY_TX_SEQ_SIFS_COMP_POST_TIMEOUT);//1;

#ifdef ENET_INC_ARCH_WAVE600B
	txNavTimerSampleSifsCompReg.bitFields.txNavTimerSamplePostEtsiFillerSifsComp = 2; // 2 us. Provided by Yosef Hai-Amar
#endif	
	RegAccess_Write(REG_TX_SEQUENCER_TX_NAV_TIMER_SAMPLE_SIFS_COMP,txNavTimerSampleSifsCompReg.val);
	
	/* Configure System Delays */
	/*REG_TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TX 0x34 */
	regTxSequencerCcaSampleTimeDelayTx.bitFields.ccaSampleTimeDelayTxOfdm = System_ConvertNsToSystemClk(TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TX_IN_NS); //unit in cycles = 7.5us * system time, means how much cycles after the energy in the air got down, the cca can be sampled in Tx
	RegAccess_Write(REG_TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TX,regTxSequencerCcaSampleTimeDelayTx.val);

	/*REG_TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_RX 0x38 */
	regTxSequencerCcaSampleTimeDelayRx.bitFields.ccaSampleTimeDelayRxOfdm = System_ConvertNsToSystemClk(TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_RX_IN_NS); //unit in cycles  = 1120/320*640 = 3.5us * system time- how much cycles after the energy in the air got down, the cca can be sampled in Rx
	RegAccess_Write(REG_TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_RX,regTxSequencerCcaSampleTimeDelayRx.val);

	ccaSampleTimeDelayTimeoutReg.bitFields.ccaSampleTimeDelayTimeout= TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TIMEOUT;
	RegAccess_Write(REG_TX_SEQUENCER_CCA_SAMPLE_TIME_DELAY_TIMEOUT,ccaSampleTimeDelayTimeoutReg.val);

	macPhyTxTxopSignalEnReg.bitFields.macPhyTxTxopSignalEn = TX_SEQUENCER_MAC_PHY_TX_TXOP_SIGNAL_EN;
	RegAccess_Write(REG_TX_SEQUENCER_MAC_PHY_TX_TXOP_SIGNAL_EN,macPhyTxTxopSignalEnReg.val);

		
	/* Enable Tx Sequencer Errors Interrupt*/
	errIrqEnReg.bitFields.compOwnerErrIrqEn = TX_SEQUENCER_ENABLE_OWNER_BIT_ERROR;
#ifdef ENET_INC_ARCH_WAVE600B
    errIrqEnReg.bitFields.sequencerFifoFullDropIrqEn = TX_SEQUENCER_ENABLE_FIFO_FULL_ERROR;
#else
	errIrqEnReg.bitFields.sequencerFifoFullIrqEn = TX_SEQUENCER_ENABLE_FIFO_FULL_ERROR;
#endif
	errIrqEnReg.bitFields.sequencerFifoDecrementLessThanZeroIrqEn = TX_SEQUENCER_ENABLE_RA_DECREMENT_LESS_THAN_ZERO_IRQ;
	errIrqEnReg.bitFields.baaNeedsStartWhileBusyIrqEn = TX_SEQUENCER_ENABLE_BAA_NEEDS_START_WHILE_BUSY_ERROR;
	RegAccess_Write(REG_TX_SEQUENCER_TX_SEQUENCER_ERR_IRQ_EN,errIrqEnReg.val);	

	// Initialize a pointer of the sequencer fifo report for this specific band:
#if defined(ENET_INC_LMAC0) 	
	readTxSequencerReportPtr = &SequencerFifoReport[0];
#elif defined(ENET_INC_LMAC1)
	readTxSequencerReportPtr = &SequencerFifoReport_B1[0];
#endif
	memset(readTxSequencerReportPtr, 0, (sizeof(TxSequencerReport_t) * MAX_TX_SEQUENCER_FIFO_SIZE));


	regTxSequencerSequencerFifoBaseAddr.bitFields.sequencerFifoBaseAddr = CONVERT_PHYSICAL_TO_SHRAM_OFFSET((uint32)readTxSequencerReportPtr);
	regTxSequencerSequencerFifoDepthMinusOne.bitFields.sequencerFifoDepthMinusOne = MAX_TX_SEQUENCER_FIFO_SIZE-1;
	RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_FIFO_BASE_ADDR,regTxSequencerSequencerFifoBaseAddr.val);
	RegAccess_Write(REG_TX_SEQUENCER_SEQUENCER_FIFO_DEPTH_MINUS_ONE,regTxSequencerSequencerFifoDepthMinusOne.val);

}

