/***********************************************************************************
 File:			UlPostRx.c
 Module:		UL Post Rx
 Purpose: 		To create a driver for processing of UL Post RX reports 
 Description:	This file is the implementation of the UL Post Rx driver 
				
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "UlPostRx.h"
#include "ShramUlPostRxFifos.h"
#include "RegAccess_Api.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "linkAdaptation_api.h"
#ifdef ENET_INC_ARCH_WAVE600D2	
#include "StaDatabase_Api.h"
#include "Utils_Api.h"
#endif //ENET_INC_ARCH_WAVE600D2	

#define LOG_LOCAL_GID GLOBAL_GID_UL_POST_RX
#define LOG_LOCAL_FID 0



/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/

#define UL_COMMON_FIFO_SIZE_MASK		(UL_POST_RX_COMMON_FIFO_SIZE - 1)
#define UL_USER_FIFO_SIZE_MASK			(UL_POST_RX_USER_FIFO_SIZE - 1)
#define UL_REPORT_MAX_TRANSACTION_NUM	0xFF
#define UL_CONTROL_ID_FIELD_MASK		0xF 
#define UL_CTRL_INFO_OMI_MASK			0xFFF
#define UL_CTRL_INFO_UPH_MASK			0xFF
#define UPH_FIELD_MASK					0x1F
#define UPH_FIELD_LENGTH				5

#define UL_CONTROL_ID_FIELD_LENGTH		4
#define UL_CONTROL_ID_UMRS_LENGTH		26
#define UL_CONTROL_ID_OM_LENGTH			12
#define UL_CONTROL_ID_HLA_LENGTH		16
#define UL_CONTROL_ID_BSR_LENGTH		26
#define UL_CONTROL_ID_UPH_LENGTH		8
#define UL_CONTROL_ID_BQR_LENGTH		10
#define UL_CONTROL_ID_CCI_LENGTH		8
#define UL_CONTROL_MAX_LENGTH_BITS		(UL_CONTROL_ID_FIELD_LENGTH+UL_CONTROL_ID_UMRS_LENGTH) //30



//OM control information subfield bits
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_RX_NSS_START_BIT								0x0
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_CHANNEL_WIDTH_START_BIT						0x3
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_UL_MULTI_USER_DISABLE_START_BIT				0x5
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_TX_NSTS_START_BIT							0x6
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_ER_SU_DISABLE_START_BIT						0x9
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_DL_MU_MIMO_RESOUND_RECOMMENDATION_START_BIT	0xA
#define UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_UL_MU_DATA_DISABLE_START_BIT					0xB

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/


	
/*---------------------------------------------------------------------------------
/						Data Type Definition					
/----------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------------
/						Static Variables									
/----------------------------------------------------------------------------------*/

static uint8 ulPostRxCommonReadIndex = 0;
static uint8 ulPostRxUserReadIndex = 0;

/*---------------------------------------------------------------------------------
/						Global Variables									
/----------------------------------------------------------------------------------*/

static CommonUlReport_t	*ulPostRxCommonFifoPtr;
static UserUlReport_t	*ulPostRxUserFifoPtr;
static CommonUlReport_t lastUlCommonReport;
static UlPostRxErrorCounters_t ulprErrorCounters;
static UlPostRxHeCtrlCounters_t ulprHeControlCounters;
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

static void ulPostRx_GetUserFifoEntry(UserUlReport_t* userFifoEntry);
static void ulPostRx_GetCommonFifoEntry(CommonUlReport_t* commonFifoEntry);
static bool ulPostRx_ProcessHeControl(uint32 heControl, uint16 aControlTypeBitmap, LaUlPostRxUphControl_t *uphControl, StaId staId, uint16 *omiControl);


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

ulPostRx_fifoNotEmptyEventHandler 


Description:
------------
Event handler that is invoked from PAC manager when ul user FIFO is not empty
Handler extracts ul user reports and matching ul common report and forwards them to relevant modules

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void UlPostRx_fifoNotEmptyEventHandler(void)
{

	RegUlPostRxUlprCommonFifoNumEntriesCount_u commonUlReportsNumber;
	RegUlPostRxUlprUserFifoNumEntriesCount_u userUlReportsNum;
	uint8 i;
	LaUlprReport_t ulPostRxReport;
	uint32 heControl = 0;
	bool isOmiReceived = FALSE;

	commonUlReportsNumber.val = 0;
	userUlReportsNum.val = 0;

	RegAccess_Read(REG_UL_POST_RX_ULPR_COMMON_FIFO_NUM_ENTRIES_COUNT, &commonUlReportsNumber.val);
	RegAccess_Read(REG_UL_POST_RX_ULPR_USER_FIFO_NUM_ENTRIES_COUNT, &userUlReportsNum.val);
	// no need to read interrupt status register , because only ulUserFifoNotEmpty interrupt is enabled
	for(i=0; i < userUlReportsNum.bitFields.ulprUserFifoNumEntriesCount; i++)
	{
		ulPostRx_GetUserFifoEntry(&ulPostRxReport.ulprUserReport);
		if(lastUlCommonReport.transactionNumber!=ulPostRxReport.ulprUserReport.transactionNumber)
		{
			// at leasy one common report must be in the FIFO when new user reports were sent
			DEBUG_ASSERT(commonUlReportsNumber.bitFields.ulprCommonFifoNumEntriesCount!=0);
			ulPostRx_GetCommonFifoEntry(&lastUlCommonReport);
			// at this point transactionNumber at both reports types must be matching
			DEBUG_ASSERT(lastUlCommonReport.transactionNumber == ulPostRxReport.ulprUserReport.transactionNumber);
			
		}
		// copy the common part to the report and forward it to LA
		memcpy32(&ulPostRxReport.ulprCommonReport, &lastUlCommonReport, (sizeof(CommonUlReport_t) >> 2));
		ulPostRxReport.uphControl.uphValid = 0;
		
		if(ulPostRxReport.ulprUserReport.aControlTypeBitmap!=0)
		{
			heControl = ulPostRxReport.ulprUserReport.aControl;
			isOmiReceived = ulPostRx_ProcessHeControl(heControl, ulPostRxReport.ulprUserReport.aControlTypeBitmap, &ulPostRxReport.uphControl, ulPostRxReport.ulprUserReport.staId, &ulPostRxReport.omiControl.val);
		}
		SLOG0(0, 0, LaUlprReport_t, &ulPostRxReport);
		linkAdaptationProcessUlprReport(&ulPostRxReport, isOmiReceived);		
	}	
}

/**********************************************************************************

isr_UlPostRxErrorHandler



Description:
------------
handling of error interrupts

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void isr_UlPostRxErrorHandler(void)
{
	RegUlPostRxUlprIntErrorStatus_u intErrorStatus;
	RegUlPostRxUlprIntErrorClear_u clearIntError;
	RegUlPostRxUlprUserFifoDebug_u userFifoDebug;
	RegUlPostRxUlprUserFifoClearStrb_u userFifoClearStrb;
	
	intErrorStatus.val = 0;
	clearIntError.val = 0;
	userFifoDebug.val = 0;
	userFifoClearStrb.val = 0;
	
	RegAccess_Read(REG_UL_POST_RX_ULPR_INT_ERROR_STATUS, &intErrorStatus.val);
#ifndef ENET_INC_ARCH_WAVE600B
	if(intErrorStatus.bitFields.ulprUserFifoDropStatus) 
	{
		RegAccess_Read(REG_UL_POST_RX_ULPR_USER_FIFO_DEBUG, &userFifoDebug.val);
		RegAccess_Read(REG_UL_POST_RX_ULPR_USER_FIFO_CLEAR_STRB, &userFifoClearStrb.val);

		// add the number of dropped user reports to FW internal counter
		ulprErrorCounters.ulprUserFifoDropCounter+= userFifoDebug.bitFields.ulprUserFifoFullDropCtr;

		// reset the counter to zero  - write the relevant clear bit
		userFifoClearStrb.bitFields.ulprUserFifoClearFullDropCtrStrb = 1;		
        RegAccess_Write(REG_UL_POST_RX_ULPR_USER_FIFO_CLEAR_STRB, userFifoClearStrb.val);

        //turn off this error interrupt
		clearIntError.bitFields.ulprUserFifoDropClr = 1;
		intErrorStatus.bitFields.ulprUserFifoDropStatus = 0;
	}
#else
    if(intErrorStatus.bitFields.ulprUserFirstFifoDropStatus)
    {
        RegAccess_Read(REG_UL_POST_RX_ULPR_USER_FIFO_DEBUG, &userFifoDebug.val);

		// add the number of dropped user reports to FW internal counter
		ulprErrorCounters.ulprUserFifoDropCounter+= userFifoDebug.bitFields.ulprUserFirstFifoFullDropCtr;
        
        //turn off this error interrupt
        clearIntError.bitFields.ulprUserFirstFifoDropClr = 1;
        intErrorStatus.bitFields.ulprUserFirstFifoDropStatus = 0;
        
    }
    if(intErrorStatus.bitFields.ulprUserNonFirstFifoDropStatus)
    {
        RegAccess_Read(REG_UL_POST_RX_ULPR_USER_FIFO_DEBUG, &userFifoDebug.val);

		// add the number of dropped user reports to FW internal counter
		ulprErrorCounters.ulprUserFifoDropCounter+= userFifoDebug.bitFields.ulprUserNonFirstFifoFullDropCtr;
        
        //turn off this error interrupt
        clearIntError.bitFields.ulprUserNonFirstFifoDropClr = 1;
        intErrorStatus.bitFields.ulprUserNonFirstFifoDropStatus = 0;            
    }	
#endif
	if(intErrorStatus.bitFields.ulprPulse1CollisionStatus) 
	{
		ulprErrorCounters.ulprPulse1CollisionCounter++;
		clearIntError.bitFields.ulprPulse1CollisionClr = 1;
		intErrorStatus.bitFields.ulprPulse1CollisionStatus = 0;
	}
	if(intErrorStatus.bitFields.ulprPulse2CollisionStatus) 
	{
		ulprErrorCounters.ulprPulse2CollisionCounter++;
		clearIntError.bitFields.ulprPulse2CollisionClr = 1;
		intErrorStatus.bitFields.ulprPulse2CollisionStatus = 0;
	}
	if(intErrorStatus.bitFields.ulprNonUniqueControlIdStatus) 
	{
		ulprErrorCounters.ulprNonUniqCtrlIdCounter++;
		clearIntError.bitFields.ulprNonUniqueControlIdClr = 1;
		
		intErrorStatus.bitFields.ulprNonUniqueControlIdStatus = 0;
	}
	RegAccess_Write(REG_UL_POST_RX_ULPR_INT_ERROR_CLEAR, clearIntError.val);
	
	if(intErrorStatus.bitFields.ulprCommonFifoDropStatus)
	{
		// Ul Post Rx interrupt error - common fifo drop
		ASSERT(0);
	}
#ifdef ENET_INC_ARCH_WAVE600B
	if(intErrorStatus.bitFields.ulprCommonFifoDecrementLessThanZeroStatus)
	{
		// Ul Post Rx interrupt error - common fifo decrement less than zero
		ASSERT(0);	
	}	
	if(intErrorStatus.bitFields.ulprUserFifoDecrementLessThanZeroStatus)
	{
		// Ul Post Rx interrupt error - user fifo decrement less than zero
		ASSERT(0);
	}	
#endif
}


/**********************************************************************************

ulPostRx_InitializeFifos


Description:
------------
initialization of pointers to ulCommonFifo and ulUserFifo

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
void UlPostRx_InitializeFifos(void)
{
	memset32(&lastUlCommonReport, 0, (sizeof(CommonUlReport_t) >> 2));
	lastUlCommonReport.transactionNumber = UL_REPORT_MAX_TRANSACTION_NUM;
	memset(&ulprErrorCounters, 0, sizeof(UlPostRxErrorCounters_t));
	memset(&ulprHeControlCounters, 0, sizeof(UlPostRxHeCtrlCounters_t));
	
#if defined(ENET_INC_LMAC0) 
	ulPostRxCommonFifoPtr = UlPostRxCommonReportFifo;
	ulPostRxUserFifoPtr = UlPostRxUserReportFifo;
#endif

#if defined(ENET_INC_LMAC1) 
	ulPostRxCommonFifoPtr = UlPostRxCommonReportFifo_B1;
	ulPostRxUserFifoPtr = UlPostRxUserReportFifo_B1;
#endif

	memset(ulPostRxCommonFifoPtr,	0, (sizeof(CommonUlReport_t)*UL_POST_RX_COMMON_FIFO_SIZE));
	memset(ulPostRxUserFifoPtr,		0, (sizeof(UserUlReport_t)*UL_POST_RX_USER_FIFO_SIZE));
}


/**********************************************************************************

ulPostRx_GetUserFifoEntry


Description:
------------
extract user report from UL FIFO

Input: userFifoEntry - pointer to destination address to which extracted report is copied
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void ulPostRx_GetUserFifoEntry(UserUlReport_t* userFifoEntry)
{
	RegUlPostRxUlprUserFifoRdEntriesNum_u numOfReadEntries;
	numOfReadEntries.val = 0;
	numOfReadEntries.bitFields.ulprUserFifoRdEntriesNum = 1;
	memcpy32(userFifoEntry, &ulPostRxUserFifoPtr[ulPostRxUserReadIndex], (sizeof(UserUlReport_t) >> 2));
	ulPostRxUserReadIndex++;
	ulPostRxUserReadIndex &= UL_USER_FIFO_SIZE_MASK;
	RegAccess_Write(REG_UL_POST_RX_ULPR_USER_FIFO_RD_ENTRIES_NUM, numOfReadEntries.val);
}

/**********************************************************************************

ulPostRx_GetCommonFifoEntry



Description:
------------
extract common report from UL FIFO

Input: commonFifoEntry - pointer to destination address to which extracted report is copied
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/
static void ulPostRx_GetCommonFifoEntry(CommonUlReport_t* commonFifoEntry)
{
	RegUlPostRxUlprCommonFifoRdEntriesNum_u numOfReadEntries;
	numOfReadEntries.val = 0;
	numOfReadEntries.bitFields.ulprCommonFifoRdEntriesNum = 1;
	memcpy32(commonFifoEntry, &ulPostRxCommonFifoPtr[ulPostRxCommonReadIndex], (sizeof(CommonUlReport_t) >> 2));
	ulPostRxCommonReadIndex++;
	ulPostRxCommonReadIndex &= UL_COMMON_FIFO_SIZE_MASK;
	RegAccess_Write(REG_UL_POST_RX_ULPR_COMMON_FIFO_RD_ENTRIES_NUM, numOfReadEntries.val);
}

/**********************************************************************************

ulPostRx_ProcessHeControl




Description:
------------
process A Control field of 30 bits from ulpr user report

Input: heControl - A Control field of 30 bits from ulpr user report
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

static bool ulPostRx_ProcessHeControl(uint32 heControl, uint16 aControlTypeBitmap, LaUlPostRxUphControl_t *uphControl, StaId staId, uint16 *omiControl)
{
	uint8 numOfShifts 			= 0;
	uint8 controlId 			= 0;
	uint8 controlInfo 			= 0;
	uint16 controlBitmap 		= 0;
	bool isOmiReceived  		= FALSE;

#ifndef ENET_INC_ARCH_WAVE600D2		
	UNUSED_PARAM(staId);
#endif
	ILOG0_DD("[ulPostRx_ProcessHeControl] heControl %d aControlTypeBitmap %d", heControl, aControlTypeBitmap);
	
	while((controlBitmap != aControlTypeBitmap ) && (numOfShifts < UL_CONTROL_MAX_LENGTH_BITS)) 
	{
		controlId 		 = ((heControl >> numOfShifts) & UL_CONTROL_ID_FIELD_MASK);
		numOfShifts 	+= UL_CONTROL_ID_FIELD_LENGTH;
		controlBitmap 	|= (1 << controlId);
		
		ILOG0_DDD("controlId %d numOfShifts %d controlBitmap %d", controlId, numOfShifts, controlBitmap);
		
		switch (controlId)
		{
			case UL_CONTROL_ID_OM:
			{
				*omiControl = ((heControl >> numOfShifts) & UL_CTRL_INFO_OMI_MASK);
				ILOG0_D("OMI is received. omi control: 0x%x", *omiControl);
				numOfShifts += UL_CONTROL_ID_OM_LENGTH;

				/*update HE ER SU configurations in STA DB:
				check if HE E SU disbale set or reset	*/
#ifdef ENET_INC_ARCH_WAVE600D2				
				StaDB_SetHeSuExtParams(staId, Utils_GetBitFromBitmap((uint32*)&omiControl, UL_CONTROL_OM_CONTROL_INFORMATION_SUBFIELD_ER_SU_DISABLE_START_BIT));
#endif //ENET_INC_ARCH_WAVE600D2

				ulprHeControlCounters.omiCounter++; 
				isOmiReceived = TRUE;
				break;
			}
			case UL_CONTROL_ID_UPH:
			{
				controlInfo = ((heControl >> numOfShifts) & UL_CTRL_INFO_UPH_MASK);
				uphControl->uphValid = 1;
				uphControl->uph = (controlInfo & UPH_FIELD_MASK);
				uphControl->minTransmitPowerFlag = ((controlInfo >> UPH_FIELD_LENGTH) & 0x1);
				ILOG0_D("[ulPostRx_ProcessHeControl] uph control bits %d", controlInfo);
				numOfShifts += UL_CONTROL_ID_UPH_LENGTH;
				ulprHeControlCounters.uphCounter++;
				break;
			}
			case UL_CONTROL_ID_UMRS:
				numOfShifts += UL_CONTROL_ID_UMRS_LENGTH;
				ulprHeControlCounters.umrsCounter++;
				break;
			case UL_CONTROL_ID_HLA:
				numOfShifts += UL_CONTROL_ID_HLA_LENGTH;
				ulprHeControlCounters.hlaCounter++;
				break;
			case UL_CONTROL_ID_BQR:
				numOfShifts += UL_CONTROL_ID_BQR_LENGTH;
				ulprHeControlCounters.bqrCounter++;
				break;
			case UL_CONTROL_ID_BSR:
				ILOG0_DD("[ulPostRx_ProcessHeControl] controlBitmap=%x, ulprHeControlCounters.bsrCounter=%d", controlBitmap, ulprHeControlCounters.bsrCounter);
				// TODO: should verify if also in this case we should extract controlInfo
				numOfShifts += UL_CONTROL_ID_BSR_LENGTH;
				ulprHeControlCounters.bsrCounter++;
				break;
			case UL_CONTROL_ID_CCI:
				numOfShifts += UL_CONTROL_ID_CCI_LENGTH;
				ulprHeControlCounters.cciCounter++;
				break;
			default:
				break;	
		}
	}
	return isOmiReceived;
}


/**********************************************************************************

ulPostRx_Initialize 


Description:
------------
initialize internal structures and registers of the UL Post Rx module 

Input:
-----
	
		
Output:
-------
	

Returns:
--------
	void - 
	
**********************************************************************************/

void UlPostRx_Initialize(void)
{
	RegUlPostRxUlprCommonFifoBaseAddr_u			commonFifoBaseAddrRegister;
	RegUlPostRxUlprCommonFifoDepthMinusOne_u	commonFifoDepthRegister;
	RegUlPostRxUlprUserFifoBaseAddr_u			userFifoBaseAddrRegister;
	RegUlPostRxUlprUserFifoDepthMinusOne_u		userFifoDepthRegister;
	RegUlPostRxUlprIntErrorEn_u					interruptErrorEnableRegister;
	RegUlPostRxUlprFifoNotEmptyEn_u				fifoInterruptEnableRegister;

	// OMERG  - hard-coded initialization for PF#6
	RegUlPostRxAControlFieldHwParsingEnable_u 	ulPostRxAControlFieldHwParsingEnable;
	RegUlPostRxAControlFieldSwReportEnable_u	ulPostRxAControlFieldSwReportEnable;
	RegUlPostRxVapHeVariantSupportBitmap_u		ulPostRxVapHeVariantSupportBitmap;
	RegUlPostRxStaHeVariantSupportBitmap310_u	ulPostRxStaHeVariantSupportBitmap310;
#ifdef ENET_INC_ARCH_WAVE600B
	RegUlPostRxUlMuDisableDataControl_u			ulPostRxUlMuDisableDataControl;

	ulPostRxUlMuDisableDataControl.val 			 = 0;
#endif
	ulPostRxAControlFieldHwParsingEnable.val	 = 0;
	ulPostRxAControlFieldSwReportEnable.val 	 = 0;
	ulPostRxVapHeVariantSupportBitmap.val 		 = 0;
	ulPostRxStaHeVariantSupportBitmap310.val 	 = 0;
	// End of hard-coded initialization for PF#6


	commonFifoBaseAddrRegister.val = 0;
	commonFifoDepthRegister.val = 0;
	userFifoBaseAddrRegister.val = 0;
	userFifoDepthRegister.val = 0;
	fifoInterruptEnableRegister.val = 0;
	interruptErrorEnableRegister.val = 0;

#if defined(ENET_INC_LMAC0) 
	commonFifoBaseAddrRegister.bitFields.ulprCommonFifoBaseAddr = 	CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&UlPostRxCommonReportFifo[0]);
	userFifoBaseAddrRegister.bitFields.ulprUserFifoBaseAddr = 		CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&UlPostRxUserReportFifo[0]);

#endif
#if defined(ENET_INC_LMAC1) 
	commonFifoBaseAddrRegister.bitFields.ulprCommonFifoBaseAddr = 	CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&UlPostRxCommonReportFifo_B1[0]);
	userFifoBaseAddrRegister.bitFields.ulprUserFifoBaseAddr = 		CONVERT_PHYSICAL_TO_SHRAM_OFFSET(&UlPostRxUserReportFifo_B1[0]);
#endif

	RegAccess_Write(REG_UL_POST_RX_ULPR_COMMON_FIFO_BASE_ADDR, commonFifoBaseAddrRegister.val);
	RegAccess_Write(REG_UL_POST_RX_ULPR_USER_FIFO_BASE_ADDR, userFifoBaseAddrRegister.val);

	commonFifoDepthRegister.bitFields.ulprCommonFifoDepthMinusOne = UL_POST_RX_COMMON_FIFO_SIZE - 1;
	userFifoDepthRegister.bitFields.ulprUserFifoDepthMinusOne = UL_POST_RX_USER_FIFO_SIZE - 1;
	RegAccess_Write(REG_UL_POST_RX_ULPR_COMMON_FIFO_DEPTH_MINUS_ONE, commonFifoDepthRegister.val);
	RegAccess_Write(REG_UL_POST_RX_ULPR_USER_FIFO_DEPTH_MINUS_ONE, userFifoDepthRegister.val);

	fifoInterruptEnableRegister.bitFields.ulprCommonFifoNotEmptyEnable = 0x0;
	fifoInterruptEnableRegister.bitFields.ulprUserFifoNotEmptyEnable = 0x1;
	RegAccess_Write(REG_UL_POST_RX_ULPR_FIFO_NOT_EMPTY_EN, fifoInterruptEnableRegister.val);

	interruptErrorEnableRegister.bitFields.ulprCommonFifoDropEn = 0x1;

// OMERG  - hard-coded initialization for PF#6 
	ulPostRxAControlFieldHwParsingEnable.bitFields.aControlFieldHwParsingEnable = 0x2; 
	ulPostRxAControlFieldSwReportEnable.bitFields.aControlFieldSwReportEnable = 0x12; 
	ulPostRxVapHeVariantSupportBitmap.bitFields.vapHeVariantSupportBitmap = 0xFFFFFFFF; 
	ulPostRxStaHeVariantSupportBitmap310.bitFields.staHeVariantSupportBitmap310 = 0xFFFFFFFF; 
#ifdef ENET_INC_ARCH_WAVE600B
#ifndef ENET_INC_ARCH_WAVE600D2
	ulPostRxUlMuDisableDataControl.bitFields.ulMuDisableDataFieldBitPosition = 0x2; 
#endif
	ulPostRxUlMuDisableDataControl.bitFields.ulMuDisableDataFieldEnable = 0x1; 

	RegAccess_Write(REG_UL_POST_RX_UL_MU_DISABLE_DATA_CONTROL, ulPostRxUlMuDisableDataControl.val);
#endif
	RegAccess_Write(REG_UL_POST_RX_A_CONTROL_FIELD_HW_PARSING_ENABLE, ulPostRxAControlFieldHwParsingEnable.val);
	RegAccess_Write(REG_UL_POST_RX_A_CONTROL_FIELD_SW_REPORT_ENABLE, ulPostRxAControlFieldSwReportEnable.val);
	RegAccess_Write(REG_UL_POST_RX_VAP_HE_VARIANT_SUPPORT_BITMAP, ulPostRxVapHeVariantSupportBitmap.val);
	RegAccess_Write(REG_UL_POST_RX_STA_HE_VARIANT_SUPPORT_BITMAP_31_0, ulPostRxStaHeVariantSupportBitmap310.val);
// End of hard-coded initialization for PF#6 -  only the first 32 stations would be set as HE variant supported. 

	
#ifndef ENET_INC_ARCH_WAVE600B
	interruptErrorEnableRegister.bitFields.ulprUserFifoDropEn = 0x1;
#else
    interruptErrorEnableRegister.bitFields.ulprUserFirstFifoDropEn = 0x1;
    interruptErrorEnableRegister.bitFields.ulprUserNonFirstFifoDropEn = 0x1;
	interruptErrorEnableRegister.bitFields.ulprCommonFifoDecrementLessThanZeroEn = 0x1;
	interruptErrorEnableRegister.bitFields.ulprUserFifoDecrementLessThanZeroEn	= 0x1;	
#endif
	interruptErrorEnableRegister.bitFields.ulprPulse1CollisionEn = 0x1;
	interruptErrorEnableRegister.bitFields.ulprPulse2CollisionEn = 0x1;
	interruptErrorEnableRegister.bitFields.ulprNonUniqueControlIdEn = 0x1;
	RegAccess_Write(REG_UL_POST_RX_ULPR_INT_ERROR_EN, interruptErrorEnableRegister.val);
}


