/***************************************************************
 File:		  VapDatabase.c
 Module:	  	  Vap Database
 Purpose: 	 
 Description:     This module is the implementation of the Vap Database module. Calls that access vap database parameters 
 			  should be made through the api function located in this file. Direct access (Not through the api) to the
 			  station database parameters is not permitted.
***************************************************************/

//---------------------------------------------------------------------------------
//						Includes						
//---------------------------------------------------------------------------------
#include "VapDatabase_Api.h"
#include "StaDatabase_Api.h"
#include "ShramVapDatabase.h"
#include "VapDb.h"
#include "stringLibApi.h"
#include "Pac_Api.h"
#if defined (ENET_INC_UMAC)
#include "BSSmanager_API.h"
#endif
#include "ErrorHandler_Api.h"
#include "Dut_Api.h"
#include "loggerAPI.h"
#include "PacketDescriptor.h"
#if defined(ENET_INC_ARCH_WAVE600)
#include "secManager_VapManager.h"
#else
#include "AggregationBuilder_Api.h"
#endif 
#if defined(ENET_INC_ARCH_WAVE600D2)	
#include "RxPp_Api.h"
#endif

//---------------------------------------------------------------------------------
//						Defines						
//---------------------------------------------------------------------------------
#define LOG_LOCAL_GID   GLOBAL_GID_VAP_DB
#define LOG_LOCAL_FID 0


//---------------------------------------------------------------------------------
//						Data Type Definition					
//---------------------------------------------------------------------------------

//---------------------------------------------------------------------------------
//						Globals					
//---------------------------------------------------------------------------------
VapDbObject_t	VapDatabaseObj;


//---------------------------------------------------------------------------------
//						Private Function Declaration 			
//---------------------------------------------------------------------------------
#if defined (ENET_INC_UMAC)
static void fillActivateVapParams( VapDbActivateVap_t* vapParams, UMI_ADD_VAP * pAddVapStructurePtr); 
#endif 

//---------------------------------------------------------------------------------
//						VAP Function implementation	 - move to VAP DB module			
//--------------------------------------------------------------------------------


/**********************************************************************************
GetPsduMaxLengthLimit 

Description:
------------
Caculate the PsduMaxLengthLimit.  It is calculated by multiplying the max AMPDU size (according to HT/VHT station support)
by the max number of MPDUs in A-MPDU

Input:
-----
bool vhtStation - indicates if the station is a VHT station or not.
		
Output:
-------

Returns:
--------
	void  
	
**********************************************************************************/
uint32 vapDbGetPsduMaxLengthLimit(bool vhtStation)
{
	uint32 length = 0;
	
	if (vhtStation)
	{
		length = MAX_VHT_MPDU_IN_A_MPDU_LENGTH_TX * MAX_MPDUS_IN_A_MPDU;
	}
	else
	{
		length = HT_MAX_A_MPDU_SIZE;
	}
	return length;
}

/**********************************************************************************
VapDB_AddStationCounter 

Description:
------------
A station was added to the VAP. Inrement the counter of num of stations connected to the VAP

Input:
-----
Vap Index
		
Output:
-------

Returns:
--------
	void  
	
**********************************************************************************/
void VapDb_AddStationCounter(uint8 vapIndex)
{
	VapDbSwInfo_t* pVapDbSwEntry = &(VapDatabaseObj.vapDbSwEntries[vapIndex]);

	pVapDbSwEntry->connectedStationsNum++;
}

/**********************************************************************************
VapDB_DecreaseStationCounter 

Description:
------------
A station was removed from the VAP. decrement the counter of num of stations connected to the VAP

Input:
-----
Vap Index
		
Output:
-------

Returns:
--------
	void  
	
**********************************************************************************/
void VapDb_DecreaseStationCounter(uint8 vapIndex)
{
	VapDbSwInfo_t* pVapDbSwEntry = &(VapDatabaseObj.vapDbSwEntries[vapIndex]);

	pVapDbSwEntry->connectedStationsNum--;
}

/**********************************************************************************
VapDb_SetFirstConnectedStaIndex 

Description:
------------
Set the sid of the first connected STA. This STA points to all the other STAs in the VAP.

Input:
-----
Vap Index, SID of first STA.
		
Output:
-------

Returns:
--------
	void  
	
**********************************************************************************/
void VapDb_SetFirstConnectedStaIndex(uint8 vapIndex, StaId staIndex)
{
	VapDbSwInfo_t* pVapDbSwEntry = &(VapDatabaseObj.vapDbSwEntries[vapIndex]);

	pVapDbSwEntry->firstConnectedStationIndex = staIndex;
}

/**********************************************************************************
VapDb_GetFirstConnectedStaIndex 

Description:
------------
Get the sid of the first connected STA in the VAP. This STA points to all the other STAs in the VAP.

Input:
-----
Vap Index, SID of first STA.
		
Output:
-------

Returns:
--------
	void  
	
**********************************************************************************/
StaId VapDb_GetFirstConnectedStaIndex(uint8 vapIndex)
{
	VapDbSwInfo_t* pVapDbSwEntry = &(VapDatabaseObj.vapDbSwEntries[vapIndex]);

	return pVapDbSwEntry->firstConnectedStationIndex;
}

/**********************************************************************************
VapDbCopyIeeeAddrToBssIdType 

Description:
------------
copy mac address from 6 byte array - IEEE_ADDR type to VapDbBssid_t type (two uint32)  

Input:
-----
IEEE_ADDR* pMacAddress

Output:
-------
VapDbBssid_t* pBssId

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void VapDbCopyIeeeAddrToBssIdType(IEEE_ADDR* pMacAddress, VapDbBssid_t* pBssId)
{
	uint32			  macAddr03 = 0;
	uint32			  macAddr45 = 0;

	macAddr03 = (pMacAddress->au8Addr[3] << 24) | 
				(pMacAddress->au8Addr[2] << 16) | 
		    	(pMacAddress->au8Addr[1] << 8)  |
		    	(pMacAddress->au8Addr[0]);
	macAddr45 = (pMacAddress->au8Addr[5] << 8) | (pMacAddress->au8Addr[4]);
	pBssId->macAddress03 = macAddr03;
	pBssId->macAddress45 = macAddr45;
}

/**********************************************************************************
VapDb_AddVap 

Description:
------------
Activate a vap. Initialize all the VAP parameters received in input parameter vapParams

Input:
-----
VapDbActivateVap_t vapParams - Vap parameters
uint8 vapId - Vap index

Output:
-------

Returns:
--------
	Status_e  
	
**********************************************************************************/
RetVal_e VapDb_AddVap(VapDbActivateVap_t* pVapParams, uint8 vapIndex)
{
	VapDbSwInfo_t* 		pVapDbSwEntry = NULL;
	VapDb_t* 			pVapDbHwEntry = NULL;
	
	ILOG2_D("VapDb_AddVap: vapIndex is %X", vapIndex);
	if	(vapIndex == INVALID_VAP) 
	{
		return RET_VAL_BAD_PARAM;
	}
	pVapDbSwEntry =  &(VapDatabaseObj.vapDbSwEntries[vapIndex]);
	pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
	pVapDbSwEntry->connectedStationsNum = 0;
	VapDbCopyIeeeAddrToBssIdType(&(pVapParams->bssId), (VapDbBssid_t*)&(pVapDbHwEntry->bssid.word0));
	vIEEE_ADDR_CopyAddr(&(pVapDbSwEntry->macAddress),&(pVapParams->bssId));	
	pVapDbSwEntry->state = VAP_STATE_ADDED;
	/* INITIALIZE HW DB */
	/* DW1 */
	pVapDbHwEntry->common.aMpduLimit = 0;
#if	defined (ENET_INC_ARCH_WAVE600)
	pVapDbHwEntry->common.heMultiTidAggSupport = NUM_OF_TIDS - 1; // Max number of TIDs allowed in RX aggregations in HE SU (minus one). For HE MU it is not configured here.
#endif
	// Req 2: 4.2.2.2 - save vap mode
	pVapDbHwEntry->common.vapMode = pVapParams->vapMode;
	pVapDbHwEntry->common.noAggInPs = 1;
	pVapDbHwEntry->common.psForceOneNss = 1;
	pVapDbHwEntry->common.maxSpLength = 0;
	pVapDbHwEntry->common.currentPacketDensity = PACKT_DENSITY_NO_RESTRICTION;  //  SAS : set to 1:No restriction [27:25] */
	/* Driver / HW set these ? */
	pVapDbHwEntry->common.rtsCtsTxMethod = RTS_CTS_TX_METHOD_NO_RTS_CTS;		  // SAS : set to 1:No RTS/CTS transmission      [29:28] 
	pVapDbHwEntry->common.ctsToSelfTxMethod = CTS_TO_SELF_TX_METHOD_NO_CTS2SELF;	  // default value - no CTS to self transmissions  [30:31]
	/* DW2 */
	/* PHY header length +	pStaDbHwEntry->maxMpduLengthLimit ? */
	pVapDbHwEntry->common.maxPsduLengthLimit = vapDbGetPsduMaxLengthLimit(FALSE); /* [19:0]  parameter determines if this is a VHT station */
#if	defined (ENET_INC_ARCH_WAVE600)
	pVapDbHwEntry->common.durationUpdateMode = DURATION_UPDATE_MODE_FULL_TXOP; //working with full txop mode, in VAP DB it is not possible to work in partial mode (duration must be 0)
	pVapDbHwEntry->common.pn03 = 0;
	/* DW20  */
	pVapDbHwEntry->common.pn45 = 0;
#else //wave500 
	/* DW19  */
	pVapDbHwEntry->common.pn01 = 0;
	/* DW20  */
	pVapDbHwEntry->common.pn25 = 0;
#endif  // ENET_INC_ARCH_WAVE600
	return RET_VAL_SUCCESS;
}

/**********************************************************************************
VapDb_GetVapMode

Description:
------------
	Get the VAP operation mode for the given VAP index
Input:
-----
uint8	Vap index

Output:
-------
	
Returns:
--------
	uint32 - Vap Operation Mode	  	
**********************************************************************************/
uint32 VapDb_GetVapMode(uint32 vapIndex)
{
    VapDb_t* pVapDbHwEntry = NULL;
	
    pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
  	return ((uint32) pVapDbHwEntry->common.vapMode);
}

/**********************************************************************************
VapDb_ResetVapEntry 

Description:
------------
Reset the VAP parameters in the Database. Set the VAP entry to FREE state

Input:
-----
uint8	Vap index

Output:
-------
	
Returns:
--------
	Status_e  	
**********************************************************************************/
RetVal_e VapDb_ResetVapEntry(uint8 vapIndex)
{
	VapDbSwInfo_t*		pVapDbSwEntry = NULL;
	VapDb_t*			pVapDbHwEntry = NULL;

	ILOG2_D("VapDb_ResetVapEntry: vapIndex is %X", vapIndex);
	pVapDbSwEntry =  &(VapDatabaseObj.vapDbSwEntries[vapIndex]);
	pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
	memset(pVapDbSwEntry, 0 ,sizeof(VapDbSwInfo_t));
	memset(pVapDbHwEntry, 0 ,sizeof(VapDb_t));
	pVapDbSwEntry->state = VAP_STATE_FREE;
	/* Encryption parameters are set by set security key during the 4W handshake - we set default values here */
	pVapDbHwEntry->common.dataEncryptionEnable = DISABLED; 	  //  [26]	 Data Encryption disabled
#ifdef ENET_INC_ARCH_WAVE600
pVapDbHwEntry->common._80211EncryptionEnable = DISABLED;//   [27]	Management Encryption disabled
#else
	pVapDbHwEntry->common.managementEncryptionEnable = DISABLED;//   [27]	Management Encryption disabled
#endif //ENET_INC_ARCH_WAVE600
	pVapDbHwEntry->common.dataKeyId = KEY_ID_0;				  //   [29:28]	
	pVapDbHwEntry->common._80211KeyId = KEY_ID_0;			  //   [31:30] 
	VapDb_SetFirstConnectedStaIndex(vapIndex, DB_ASYNC_SID);
	return RET_VAL_SUCCESS;
}


/**********************************************************************************
VapDB_GetManagementTcr2



Description:
------------
	returns TCR2 by reference  from vap Hw DB 
Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/


/**********************************************************************************
VapDB_SetDefaultKeyIndex 

Description:
------------
This function sets default key index in the VAP DB according to the parameters passed from the driver. 

Input:
-----
VapIndex, defaultKeyId

Output:
-------

Returns:
--------
RetVal_e [success/failure]

**********************************************************************************/
RetVal_e VapDB_SetDefaultKeyIndex(uint8 vapIndex, uint8 defaultKeyId)
{
	VapDb_t*	pVapDbHwEntry = NULL;
	StaId staId;
	uint8 dataEncryptionType; 

	pVapDbHwEntry = &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
	pVapDbHwEntry->common.dataKeyId = defaultKeyId;
	dataEncryptionType = pVapDbHwEntry->common.dataEncryptionType;
	if ((dataEncryptionType == UMI_RSN_CIPHER_SUITE_WEP40) || (dataEncryptionType == UMI_RSN_CIPHER_SUITE_WEP104))
	{
		// For WEP we need to set all the STA of this VAP to have the same default key index
		staId = VapDb_GetFirstConnectedStaIndex(vapIndex);
		while (staId != DB_ASYNC_SID)
		{
			StaDB_SetDefaultKeyIndex(staId, defaultKeyId, FALSE);
			staId = StaDb_GetNextSid(staId);
		}
	}
	return RET_VAL_SUCCESS;
}

/**********************************************************************************
VapDB_SetSecurityKey 

Description:
------------
This function sets the VAP DB security entry according to the parameters passed from the driver. 

Input:
-----
VapIndex
pSetKey - a pointer to the structure that contains all the parameters

Output:
-------

Returns:
--------
RetVal_e [success/failure]

**********************************************************************************/
RetVal_e VapDB_SetSecurityKey(uint8 vapIndex, bool rekeyDisableFlag, UMI_SET_KEY *pSetKey)
{
	VapDb_t* pVapDbHwEntry = NULL;
	uint16 vapKeysOffset = 0;
	uint8 keyId = 0;
	uint8 encryptionType;
	uint32 keyIndexValue = 0;
	uint8 index = 0;
#if defined(ENET_INC_ARCH_WAVE600D2)	
	bool staModeEnable = TRUE;
#endif
	StaId staId = 0;
	
	pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
    encryptionType = pSetKey->u16CipherSuite;

	keyId = pSetKey->u16KeyIndex;

#if defined(ENET_INC_ARCH_WAVE600)	
	//in wep remap is not used
	if((encryptionType != UMI_RSN_CIPHER_SUITE_WEP40) && (encryptionType != UMI_RSN_CIPHER_SUITE_WEP104))
	{
		pVapDbHwEntry->common.rxKeyId0Remapping = 0x0;
		pVapDbHwEntry->common.rxKeyId1Remapping = 0x1;
		pVapDbHwEntry->common.rxKeyId2Remapping = 0x0;
		pVapDbHwEntry->common.rxKeyId3Remapping = 0x1;

		keyId &= 0x1; //key IDs 0 & 2 mapped to 0, 1 & 3 mapped to 1
	}
#endif // ENET_INC_ARCH_WAVE600

	/* First copy the PN before enabling the encryption in order to avoid races */
	VapDB_SetPn(vapIndex, pSetKey->au8TxSeqNum);
#if defined(ENET_INC_ARCH_WAVE600D2)	
	RxPp_SetNewPn(vapIndex, pSetKey->au8RxSeqNum, keyId, rekeyDisableFlag, staModeEnable);
#else
	UNUSED_PARAM(rekeyDisableFlag);
#endif
	pVapDbHwEntry->common.dataEncryptionType = encryptionType;
	pVapDbHwEntry->common.dataEncryptionEnable = ENABLED;

	/*
			Keys DB
		------------------------
		|VAP 0 keys				|
		|VAP 1 keys				|
		|	...					|
		|VAP 15/31(gen6) keys	|
		|STA 0 keys				|
		|STA 1 keys				|
		|	...					|
		|STA 127/255(gen6) keys	|	
		------------------------
	*/
	/* 	Offset of the key value from the start of the security Key values DB is measured by: 
		OFFSET TO VAP 4 KEYS + OFFSET BY KEY ID */
	vapKeysOffset = vapIndex * SECURITY_KEYS_NUM;
	// Calculate the pointer itself to the correct place in KEYs DB
	keyIndexValue = vapKeysOffset + keyId;
#if !defined(ENET_INC_ARCH_WAVE600)
	/********************* GEN5 fix for TKIP MULTICAST ******************************/
	// We have a problem in our gen5 implementation - we expect key index to be 0/1 but hostapd sends 1/2.
	// Solution is: FW will store keys 1 in the place of key 0 and key 2 in the place of key 1.
	// TX Sender knows that and will act accordingly.
	if ((encryptionType == UMI_RSN_CIPHER_SUITE_TKIP)&&(VapDb_GetVapMode(vapIndex) == VAP_MODE_AP))
	{
		ASSERT(keyId > 0);
		keyId--; 			// Treat key 1/2 as key 0/1
		keyIndexValue--; 	// Store key 1/2 in the place of key 0/1
	}
	/********************* GEN5 fix for TKIP MULTICAST - end **************************/
#endif
	switch (keyId)
	{
		case KEY_ID_0:
			pVapDbHwEntry->common.keyIndex0 = keyIndexValue; 
			break;
		case KEY_ID_1:
			pVapDbHwEntry->common.keyIndex1 = keyIndexValue;
			break;
		case KEY_ID_2:
			pVapDbHwEntry->common.keyIndex2 = keyIndexValue;
			break;
		case KEY_ID_3:
			pVapDbHwEntry->common.keyIndex3 = keyIndexValue; 
			break;
		default:
			ASSERT(0);
	}
	switch(encryptionType)
	{
#ifdef ENET_INC_ARCH_WAVE600	
		case UMI_RSN_CIPHER_SUITE_GCMP128:
			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[vapIndex].keyValues[keyId].actualKey[index] = pSetKey->au8Tk1[GCMP_TK_SIZE - (index+1)];
			}
			break;
		case UMI_RSN_CIPHER_SUITE_GCMP256:
			/* In GCMP256 there are 2 valid keys */
			if(keyId == KEY_ID_0)
			{
				pVapDbHwEntry->common.keyIndex2 = keyIndexValue + 2;
			}
			else
			{
				pVapDbHwEntry->common.keyIndex3 = keyIndexValue + 2;
			}
			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[vapIndex].keyValues[keyId].actualKey[index] = pSetKey->au8Tk2[GCMP_TK_SIZE - (index+1)];
			}
			// Set the 2nd part of the key in the MIC entry
			for (index = 0 ; index < GCMP_TK_SIZE ; index++)
			{
				KeysDatabase[vapIndex].keyValues[keyId + 2].actualKey[index] = pSetKey->au8Tk1[GCMP_TK_SIZE - (index+1)];
			}
			break;
#endif // ENET_INC_ARCH_WAVE600			
		case UMI_RSN_CIPHER_SUITE_CCMP:
			for (index = 0 ; index < CCMP_TK_SIZE ; index++)
			{
				KeysDatabase[vapIndex].keyValues[keyId].actualKey[index] = pSetKey->au8Tk1[CCMP_TK_SIZE - (index+1)];
			}
			break;
		case UMI_RSN_CIPHER_SUITE_WEP40: 
			/* fall through */
		case UMI_RSN_CIPHER_SUITE_WEP104:
			staId = VapDb_GetFirstConnectedStaIndex(vapIndex);
			while (staId != DB_ASYNC_SID)
			{
				StaDB_SetWepSecurityKeyIndex(staId, keyId, keyIndexValue);
				staId = StaDb_GetNextSid(staId);
			}
			MEMCPY(&KeysDatabase[vapIndex].keyValues[keyId], pSetKey->au8Tk1, sizeof(Key_t));
			break;
		case UMI_RSN_CIPHER_SUITE_TKIP:
			MEMCPY(&KeysDatabase[vapIndex].keyValues[keyId], pSetKey->au8Tk1, sizeof(Key_t));
			/* for TKIP need to copy the MIC key as well */
            // MIC will be set in keyId + 2
			MEMCPY(&KeysDatabase[vapIndex].keyValues[(keyId + 2) & (SECURITY_KEYS_NUM-1)], pSetKey->au8Tk2, sizeof(Key_t));	


			/* In Tkip there are only key 0 and key 1 valid. key 2 and 3 are MIC keys */
			
			if(keyId == KEY_ID_0)
			{
				pVapDbHwEntry->common.keyIndex2 = keyIndexValue + 2;
			}
			else if (keyId == KEY_ID_1)
			{
				pVapDbHwEntry->common.keyIndex3 = keyIndexValue + 2;
			}
   			else if (keyId == KEY_ID_2)
			{
				pVapDbHwEntry->common.keyIndex0 = vapKeysOffset; // Key index 0
			}	
			break;
		default:
			ASSERT(0);	
	}

	return RET_VAL_SUCCESS;
}

/**********************************************************************************
VapDB_GetDefaultKeyIndex 

Description:
------------
Get the VAP default key index

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
uint32 VapDB_GetDefaultKeyIndex(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	return pVapDbHwEntries->common.dataKeyId;
}

/**********************************************************************************
VapDB_GetKeyIndexValue 

Description:
------------
Get the VAP key index value of a specific key index

Input:
-----
VapId - Vap ID
keyIndex

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
uint32 VapDB_GetKeyIndexValue(uint8 vapId, uint8 keyIndex)
{
	VapDb_t* pVapDbHwEntry = NULL;
	
	pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	switch (keyIndex)
	{
		case KEY_ID_0:
			return pVapDbHwEntry->common.keyIndex0; 
		case KEY_ID_1:
			return pVapDbHwEntry->common.keyIndex1;
		case KEY_ID_2:
			return pVapDbHwEntry->common.keyIndex2;
		case KEY_ID_3:
			return pVapDbHwEntry->common.keyIndex3; 
		default:
			ASSERT(0);
	}
	return SECURITY_INVALID_KEY_INDEX;	
}

/**********************************************************************************
VapDB_ClearSecurityKey 

Description:
------------
Clear security key. The function receives the key Index of the key to clear
and sets an Invalid index in the key entry of Key Indexes array in VAP DB.

Input:
-----
VapIndex
KeyId - The key ID of the key to clear

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
RetVal_e VapDB_ClearSecurityKey(uint8 vapIndex)
{
	VapDb_t* pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);

	pVapDbHwEntry->common.keyIndex0 = SECURITY_INVALID_KEY_INDEX; 	
	pVapDbHwEntry->common.keyIndex1 = SECURITY_INVALID_KEY_INDEX; 	
	pVapDbHwEntry->common.keyIndex2 = SECURITY_INVALID_KEY_INDEX; 	
	pVapDbHwEntry->common.keyIndex3 = SECURITY_INVALID_KEY_INDEX; 	
	return RET_VAL_SUCCESS;
}

/**********************************************************************************
VapDbInvalidateGroupKey 

Description:
------------
Invalidates a group key

Input:
-----
VapIndex
KeyId - The key ID of the key to invalidate

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void VapDb_InvalidateGroupKey(uint8 vapIndex, uint8 keyId)
{
	VapDb_t* pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);

	switch (keyId)
	{
		case KEY_ID_0:
			pVapDbHwEntry->common.keyIndex0 = SECURITY_INVALID_KEY_INDEX; 
			break;
		case KEY_ID_1:
			pVapDbHwEntry->common.keyIndex1 = SECURITY_INVALID_KEY_INDEX;
			break;
		case KEY_ID_2:
			pVapDbHwEntry->common.keyIndex2 = SECURITY_INVALID_KEY_INDEX;
			break;
		case KEY_ID_3:
			pVapDbHwEntry->common.keyIndex3 = SECURITY_INVALID_KEY_INDEX; 
			break;
		default:
			ASSERT(0);
	}
}

/**********************************************************************************
VapDbGroupKeyValid 

Description:
------------
This checks if a group kei index is still valid

Input:
-----
VapIndex
KeyId - The key ID of the key to check

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
bool VapDb_GroupKeyValid(uint8 vapIndex, uint8 keyId)
{
	VapDb_t* pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
	uint16 keyIndex = SECURITY_INVALID_KEY_INDEX;
	
	switch (keyId)
	{
		case KEY_ID_0:
			keyIndex = pVapDbHwEntry->common.keyIndex0; 
			break;
		case KEY_ID_1:
			keyIndex = pVapDbHwEntry->common.keyIndex1;
			break;
		case KEY_ID_2:
			keyIndex = pVapDbHwEntry->common.keyIndex2;
			break;
		case KEY_ID_3:
			keyIndex = pVapDbHwEntry->common.keyIndex3;
			break;
		default:
			ASSERT(0);
	}
	if (keyIndex == SECURITY_INVALID_KEY_INDEX)
	{
		return (FALSE);
	}
	return (TRUE);
}

/**********************************************************************************
VapDB_GetDataEncryptionType 

Description:
------------
	Get the VAP Encryption Type
Input:
-----
	VapId - Vap ID
Output:
-------
Returns:
--------
	None
**********************************************************************************/
uint32 VapDB_GetDataEncryptionType(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
    return pVapDbHwEntries->common.dataEncryptionType;
}

/**********************************************************************************
VapDB_SetDataEncryptionType 

Description:
------------
	Sets the VAP Encryption Type
Input:
-----
	VapId - Vap ID
	uint32 dataEncryptionType	
Output:
-------
Returns:
--------
	None
**********************************************************************************/
void VapDB_SetDataEncryptionType(uint8 vapId, uint32 dataEncryptionType)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
    pVapDbHwEntries->common.dataEncryptionType = dataEncryptionType;
}


/**********************************************************************************
VapDB_IsManagementEncryptionEnabled 

Description:
------------
return whether management encryption is enabled

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
management encryption status

**********************************************************************************/
uint8 VapDB_IsManagementEncryptionEnabled(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
#ifdef ENET_INC_ARCH_WAVE600
	return pVapDbHwEntries->common._80211EncryptionEnable;
#else
	return pVapDbHwEntries->common.managementEncryptionEnable;
#endif //ENET_INC_ARCH_WAVE600
}

/**********************************************************************************
VapDB_GetManagementEncryptionType 

Description:
------------
	Get the VAP Management Encryption Type
Input:
-----
	VapId - Vap ID
Output:
-------
Returns:
--------
	None
**********************************************************************************/
uint32 VapDB_GetManagementEncryptionType(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	
#ifdef ENET_INC_ARCH_WAVE600
	return pVapDbHwEntries->common._80211EncryptionType;
#else
    return pVapDbHwEntries->common.managementEncryptionType;
#endif //ENET_INC_ARCH_WAVE600
}

/**********************************************************************************
VapDB_SetManagementEncryptionType 

Description:
------------
Set the VAP Management Encryption Type

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void VapDB_SetManagementEncryptionType(uint8 vapId, uint32 encryptionType)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
#ifdef ENET_INC_ARCH_WAVE600
		pVapDbHwEntries->common._80211EncryptionType = encryptionType ;
#else
    pVapDbHwEntries->common.managementEncryptionType = encryptionType ;
#endif //ENET_INC_ARCH_WAVE600
	pVapDbHwEntries->common._80211KeyId			 = pVapDbHwEntries->common.dataKeyId;
}

/**********************************************************************************
VapDB_SetManagementEncryptionEnabled 

Description:
------------
Enable the VAP Management Encryption 

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
None

**********************************************************************************/
void VapDB_SetManagementEncryptionEnabled(uint8 vapId, uint8 managementEncryptionEnable)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
#ifdef ENET_INC_ARCH_WAVE600
	pVapDbHwEntries->common._80211EncryptionEnable = managementEncryptionEnable;
#else
	pVapDbHwEntries->common.managementEncryptionEnable = managementEncryptionEnable;
#endif //ENET_INC_ARCH_WAVE600
}



/**********************************************************************************
VapDB_DisableManagementEncryption 

Description:
------------
Disable the VAP Management Encryption

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
void VapDB_DisableManagementEncryption(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
#ifdef ENET_INC_ARCH_WAVE600
	pVapDbHwEntries->common._80211EncryptionEnable = FALSE;
#else
	pVapDbHwEntries->common.managementEncryptionEnable = FALSE;
#endif //ENET_INC_ARCH_WAVE600
}

/**********************************************************************************
VapDB_IsDataEncryptionEnabled 

Description:
------------
return whether data encryption is enabled

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
data encryption status

**********************************************************************************/
uint8 VapDB_IsDataEncryptionEnabled(uint8 vapId)
{
	VapDb_t* pVapDbHwEntries = NULL;
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	return pVapDbHwEntries->common.dataEncryptionEnable;
}

/**********************************************************************************
VapDB_SetDataEncryptionEnabled 

Description:
------------
return whether data encryption is enabled

Input:
-----
VapId - Vap ID
uint8 dataEncryptionEnable

Output:
-------

Returns:
--------
data encryption status

**********************************************************************************/
void VapDB_SetDataEncryptionEnabled(uint8 vapId, uint8 dataEncryptionEnable)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	pVapDbHwEntries->common.dataEncryptionEnable = dataEncryptionEnable;
}

/**********************************************************************************
VapDB_GetPn

Description:
------------
copy the PN from the VAP to the address given 

Input:
-----
VapId - Vap ID
pnAddress - the address to where to copy the PN


Output:
-------

Returns:
--------


**********************************************************************************/
void VapDB_GetPn(uint8 vapId, uint8 *pnAddress)
{
	VapDb_t* pVapDbHwEntries = NULL;
	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);
#ifdef ENET_INC_ARCH_WAVE600	
	pnAddress[0] = pVapDbHwEntries->common.pn03;  			//take the LS byte (pn0)
	pnAddress[1] = (pVapDbHwEntries->common.pn03) >>8; 		//take the 2th LS byte (pn1) 
	pnAddress[2] = (pVapDbHwEntries->common.pn03) >>16; 	//take the 2th MS byte (pn2)
	pnAddress[3] = (pVapDbHwEntries->common.pn03) >>24 ; 	//take the MS byte (pn 3) 

	pnAddress[4] = pVapDbHwEntries->common.pn45; 			//take the LS byte (pn4)
	pnAddress[5] = (pVapDbHwEntries->common.pn45) >>8 ; 	//take the 2th LS byte (pn 5) 
 
#else //wave500
	pnAddress[0] = pVapDbHwEntries->common.pn01 ;
	pnAddress[1] = pVapDbHwEntries->common.pn01 >> 8;
	pnAddress[2] = pVapDbHwEntries->common.pn25 ;
	pnAddress[3] = pVapDbHwEntries->common.pn25 >> 8;
	pnAddress[4] = pVapDbHwEntries->common.pn25 >> 16;
	pnAddress[5] = pVapDbHwEntries->common.pn25 >> 24;
#endif  //ENET_INC_ARCH_WAVE600	

}


/**********************************************************************************
VapDB_SetPn

Description:
------------
set the PN in the VAP

Input:
-----
VapId - Vap ID
pnAddress - the address of the PN to copy


Output:
-------

Returns:
--------


**********************************************************************************/
void VapDB_SetPn(uint8 vapId, uint8 *pnAddress)
{
	VapDb_t* pVapDbHwEntries = &(VapDatabaseObj.pVapDbHwEntries[vapId]);
	
#if	defined (ENET_INC_ARCH_WAVE600)
	uint16 pn03 = 0;
	uint32 pn45 = 0;

	pn03 = ((pnAddress[0]) | (pnAddress[1] << 8) | (pnAddress[2] << 16) | (pnAddress[3] << 24));
	pn45 = ((pnAddress[4]) | (pnAddress[5] << 8));
	pVapDbHwEntries->common.pn03 = pn03;
	pVapDbHwEntries->common.pn45 = pn45;
#else //wave500
	uint16 pn01 = 0;
	uint32 pn25 = 0;

	pn01 = ((pnAddress[0]) | (pnAddress[1] << 8));
	pn25 = ((pnAddress[2]) | (pnAddress[3] << 8) | (pnAddress[4] << 16) | (pnAddress[5] << 24));
	pVapDbHwEntries->common.pn01 = pn01;
	pVapDbHwEntries->common.pn25 = pn25;
#endif 
}


VapDbTid_t* VapDB_GetTidParams(uint8 vapIndex , uint8 tid)
{
	VapDb_t* pVapDbHwEntries = NULL;
	UNUSED_PARAM(tid);	
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);
#if	defined (ENET_INC_ARCH_WAVE600)
	return &pVapDbHwEntries->tid; 
#else //wave500
	return &pVapDbHwEntries->tid[0];  // for mcast / mang packets, there is only 1 relevant entry in tidParams
#endif 
}

uint8 VapDB_GetQosEnable(uint8 vapIndex)
{
	VapDb_t* pVapDbHwEntries = NULL;
	pVapDbHwEntries =  &(VapDatabaseObj.pVapDbHwEntries[vapIndex]);

	return pVapDbHwEntries->common.qosEnableCapability;
}

#ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED
/**********************************************************************************
VapDB_SetAid


Description:
------------
	Set VAP AID
Input:
-----
	VapId - Vap ID
	uint16 aid - 12 bits AID
Output:
-------
Returns:
--------
	None
**********************************************************************************/
void VapDB_SetAid(uint8 vapId, uint16 aid)
{
	VapDb_t* pVapDbHwEntry = NULL;
	pVapDbHwEntry =  &(VapDatabaseObj.pVapDbHwEntries[vapId]);

	pVapDbHwEntry->bssid.aid12 = aid;
}
#endif //#ifdef MU_MIMO_STATIC_GROUP_SEND_AID_TO_STATION_ENABLED

/**********************************************************************************
VapDB_Init

Description:
------------
Set Vap db object to point to the HW database in shared RAM and reset all the VAP DB entries

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif 


RetVal_e VapDB_Init(void)
{
#if defined (ENET_INC_UMAC)

	uint8 index;
	// initialization is temporarly located here for debug purpose. Should be called from vapSM_Init		
#endif
	
	ILOG2_V("VapDB_Init");
	VapDatabaseObj.pVapDbHwEntries = VapDbHwEntries;
#if defined (ENET_INC_UMAC)
	for(index = 0; index < HW_NUM_OF_VAPS; index++)
	{	
		VapDb_ResetVapEntry(index);
	}
#endif
	return RET_VAL_SUCCESS;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif


/**********************************************************************************
VapDB_GetVapMacAddress 

Description:
------------
Get the VAP MAC Address

Input:
-----
VapId - Vap ID

Output:
-------

Returns:
--------
Status code [success/failure]

**********************************************************************************/
IEEE_ADDR* VapDB_GetMacAddress(uint8 vapId)
{
	VapDbSwInfo_t *pVapDbSwEntry = NULL;
	
	pVapDbSwEntry =  &(VapDatabaseObj.vapDbSwEntries[vapId]);
    return &(pVapDbSwEntry->macAddress);
}

/***************************************************************************************************
 fillActivateVapParams

Description:  Send indication to host that beacon template was changed. 
	
****************************************************************************************************/
#if defined (ENET_INC_UMAC)
static void fillActivateVapParams( VapDbActivateVap_t* vapParams, UMI_ADD_VAP * pAddVapStructurePtr)
{
	vIEEE_ADDR_CopyAddr(&(vapParams->bssId), &(pAddVapStructurePtr->sBSSID)); 
	vapParams->bssIndex = pAddVapStructurePtr->vapId;
	// Req 2: 4.2.2.2 - save vap mode
	vapParams->vapMode = VAP_MODE_AP;	
	if (pAddVapStructurePtr->operationMode == OPERATION_MODE_VSTA)
	{
		vapParams->vapMode = VAP_MODE_STA;
	}
	return;
} 

void VapDb_AssignVap(UMI_ADD_VAP *vapInfoPtr)
{
	VapDbActivateVap_t vapParams;
	
	fillActivateVapParams(&vapParams, vapInfoPtr);
	VapDb_AddVap(&vapParams, vapInfoPtr->vapId);
	
}
#endif //#if defined (ENET_INC_UMAC)




