/***********************************************************************************
 File:			StatisticsHandler.c			
 Module:			Debug Enviroment 
 Purpose:			
 Description:		
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "HwCounters.h"
#include "HwMemoryMap.h"
#include "ErrorHandler_Api.h"
#include "stringLibApi.h"
#include "loggerAPI.h"
#include "StatisticsHandler.h"
#include "Statistics_Api.h"
#include "shramTxDesc.h"
#include "PhyDriver_API.h"
#include "Utils_Api.h"
#include "BSSmanager_API.h"
#include "StatisticsPhyHandler.h"
#include "PSD.h"
#include "Pac_Api.h"
#include "StatisticsManager_api.h"
#include "ShramStatistics.h"

/*---------------------------------------------------------------------------------
/						Defines 					
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_DEBUG_ENVIRONMENT
#define LOG_LOCAL_FID 6

#define RX_TH_MIN_VALUE		RX_THRESHOLD_DEFAULT_VALUE
#define RX_TH_MAX_VALUE 	127
#define RX_TH_OFFSET_DOWN	2
#define RX_TH_OFFSET_UP		7
#define RX_TH_PHY_REPORT_INIT	(RX_TH_MIN_VALUE)
#define MAX_NUM_OF_ACTIVE_RSSI 	(256)
#define RX_TH_INDEX_MASK		(0x00FF)
#define ACTIVE_RSSI_BITMAP_SIZE 				((MAX_NUM_OF_ACTIVE_RSSI + 7)>>3)
#define ACTIVE_RSSI_BITMAP_SIZE_IN_WORDS 		((ACTIVE_RSSI_BITMAP_SIZE +3)>>2)

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/

typedef struct RxThresholdDb
{
	uint32				activeRssiBitmap[ACTIVE_RSSI_BITMAP_SIZE_IN_WORDS];
	int8 				highRxTH;
	int8				minRssi;
	bool				ishighRxTH;
	uint8				activeRssiArry[MAX_NUM_OF_ACTIVE_RSSI];	
} RxThresholdDb_t;
/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Public Variables									
/----------------------------------------------------------------------------------*/

HostIfCounters_t *pHostIfCounters;
RxCounters_t *pRxCounters;
BaaCounters_t *pBaaCounters;
RxThresholdDb_t			rxThresholdDb = {0};


/*---------------------------------------------------------------------------------
/						Debug									
/----------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------------
/						Static Functions Definitions									
/----------------------------------------------------------------------------------*/
static void StatisticsHandler_WriteMinRssiToPhy(uint8 vapId);
static void StatisticsHandler_CheckAndUpdateRxTh(uint8 vapId);
static void StatisticsHandler_WriteRxThToPhy(int8 THval, uint8 vapId);
static uint8 StatisticsHandler_FindMinRssi(void);

/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/

/**********************************************************************************

StatisticsHandler_WriteRxThToPhy





Description:
------------
	Execute the required
	
Input:
-----
	Threshold val
	
**********************************************************************************/
static void StatisticsHandler_WriteRxThToPhy(int8 THval, uint8 vapId)
{
	//we will update the rx th in the phy only if:
	//	1. the new val is not the min val 
	//	2. the current val is the highRxTh
	if ((THval != RX_TH_MIN_VALUE) || (rxThresholdDb.ishighRxTH == TRUE))
	{
		K_MSG* hdkMsg = OSAL_GET_MESSAGE(sizeof(UMI_RX_TH));
		UMI_RX_TH* pRxThMsg = (UMI_RX_TH *)pK_MSG_DATA(hdkMsg);
		if (THval == RX_TH_MIN_VALUE)
		{//we are not using the highRxTh val
			rxThresholdDb.ishighRxTH = FALSE;
		}
		else
		{//we are using the highRxTh val
			rxThresholdDb.ishighRxTH = TRUE;
		}
		pRxThMsg->rxThValue = THval;
		OSAL_SEND_MESSAGE(HDK_SET_RX_TH_REQ, TASK_HDK, hdkMsg, vapId);	
		ILOG2_D("RX TH - WriteRxTh2Phy THval: %d", THval);
	}	
}


/**********************************************************************************

StatisticsHandler_SetRxThReq



Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg - highRxTH
	
**********************************************************************************/
void StatisticsHandler_SetRxThReq(K_MSG *statisticsManagerMsg)
{	
	UMI_RX_TH* rxThParam = (UMI_RX_TH*)pK_MSG_DATA(statisticsManagerMsg);

	if (rxThParam->getSetOperation == API_GET_OPERATION)
	{
		rxThParam->rxThValue = rxThresholdDb.highRxTH;
	}
	else
	{
		//update the high rx th 
		rxThresholdDb.highRxTH = rxThParam->rxThValue;
		ILOG2_D("RX TH - SetRxThReq THval: %d", rxThresholdDb.highRxTH);
			//if TH is higher than minimum RSSI of connected stations or no connected stations (minRSSI is on high value)
		if ((rxThresholdDb.minRssi < (rxThresholdDb.highRxTH + RX_TH_OFFSET_UP)) || (rxThresholdDb.minRssi == RX_TH_MAX_VALUE))
		{
		StatisticsHandler_WriteRxThToPhy(RX_TH_MIN_VALUE, statisticsManagerMsg->header.vapId);	
		}
		else
		{
		StatisticsHandler_WriteRxThToPhy(rxThresholdDb.highRxTH, statisticsManagerMsg->header.vapId);
		}
		Pac_RxcUpdateMinimalRssiThreshold(rxThresholdDb.highRxTH);
	}
	//send confirm to host
	OSAL_SEND_MESSAGE(UMI_MC_MAN_SET_RX_TH_CFM, TASK_UM_IF_TASK, statisticsManagerMsg, statisticsManagerMsg->header.vapId);
}

/**********************************************************************************

StatisticsHandler_WriteMinRssiToPhy


Description:
------------
	Execute the required
	
Input:
-----

**********************************************************************************/
static void StatisticsHandler_WriteMinRssiToPhy(uint8 vapId)
{
	K_MSG* hdkMsg = OSAL_GET_MESSAGE(sizeof(HdkMinRssi_t));
	HdkMinRssi_t* pMinRssiMsg = (HdkMinRssi_t *)pK_MSG_DATA(hdkMsg);
	uint8 regVal;
	
	if (rxThresholdDb.minRssi == RX_TH_MAX_VALUE)
	{//the min rssi is max val so there are no connacted stations, in this case we set the min rssi to 0x80, the init val
		regVal = (uint8)RX_TH_PHY_REPORT_INIT;
	}
	else
	{
		regVal = (uint8)rxThresholdDb.minRssi;
	}	
	pMinRssiMsg->minRssi = regVal;
	OSAL_SEND_MESSAGE(HDK_SET_MIN_RSSI_REQ, TASK_HDK, hdkMsg, vapId);	
	ILOG2_DD("RX TH - WriteMinRssi2Phy regVal: %d minRssi: %d", regVal,rxThresholdDb.minRssi);
}

/**********************************************************************************

StatisticsHandler_FindMinRssi

Description:
------------
	
	
Input:
-----

**********************************************************************************/
static uint8 StatisticsHandler_FindMinRssi(void)
{
	uint16 negativeMinRssi;
	uint16 minRssi;

	//find min positive rssi
	minRssi = Utils_CountTrailZeroBitmap(&rxThresholdDb.activeRssiBitmap[0],0,ACTIVE_RSSI_BITMAP_SIZE/2);
	//find the min negative rssi
	negativeMinRssi = Utils_CountTrailZeroBitmap(&rxThresholdDb.activeRssiBitmap[4],0,ACTIVE_RSSI_BITMAP_SIZE/2);
	if (negativeMinRssi < (MAX_NUM_OF_ACTIVE_RSSI/2))
	{//we found negative rssi so this will be min rssi
		minRssi = negativeMinRssi + (MAX_NUM_OF_ACTIVE_RSSI/2);
	}
	return minRssi;
}

/**********************************************************************************

StatisticsHandler_RxThRemoveSta



Description:
------------
	Execute the required
	
Input:
-----

**********************************************************************************/
void StatisticsHandler_RxThRemoveSta(StaId staId, uint8 vapId)
{	
	uint8 rssiVal = ((uint8)PhyStatistics_getRssi(staId)) & RX_TH_INDEX_MASK;

	ILOG2_DD("RX TH - RxThRemoveSta staId: %d , rssiVal %d",staId,rssiVal);
	ASSERT (rxThresholdDb.activeRssiArry[rssiVal] > 0);
	rxThresholdDb.activeRssiArry[rssiVal]--;
	if (rxThresholdDb.activeRssiArry[rssiVal] == 0)
	{
		Utils_ZeroBitInBitmap(rxThresholdDb.activeRssiBitmap,rssiVal);	
		if (Utils_CountTrailZeroBitmap(rxThresholdDb.activeRssiBitmap,0,ACTIVE_RSSI_BITMAP_SIZE) < MAX_NUM_OF_ACTIVE_RSSI)
		{//there are other connected stations
			//find the new min rssi val
			rxThresholdDb.minRssi = StatisticsHandler_FindMinRssi();
			//check if we currently using the highRxTh and the new min rssi is above the highRxTh
			if ((rxThresholdDb.ishighRxTH == FALSE) && (rxThresholdDb.minRssi > (rxThresholdDb.highRxTH + RX_TH_OFFSET_UP)))
			{
				//write the highRxTh to the phy
				StatisticsHandler_WriteRxThToPhy(rxThresholdDb.highRxTH, vapId);
			}
			//write the new min rssi val to the phy
			StatisticsHandler_WriteMinRssiToPhy(vapId);
		}
		else
		{//there are no connected stations
			//set the min rssi to the max val
			rxThresholdDb.minRssi = RX_TH_MAX_VALUE;
			if (rxThresholdDb.ishighRxTH == TRUE)
			{//the current th is the highRxTh
				//write the RX_TH_MIN_VALUE to the phy
				StatisticsHandler_WriteRxThToPhy(RX_TH_MIN_VALUE, vapId);
			}
			//write the new min rssi val to the phy
			StatisticsHandler_WriteMinRssiToPhy(vapId);
		}		
	}	
}

/**********************************************************************************

StatisticsHandler_RxThUpdateRssi



Description:
------------
	Execute the required
	
Input:
-----

**********************************************************************************/
void StatisticsHandler_RxThUpdateRssi(int8 oldRssiInput, int8 newRssiInput, uint8 vapId)
{
	bool BitmapChanged = FALSE;
	uint8 oldRssi = (uint8)oldRssiInput;
	uint8 newRssi = (uint8)newRssiInput;
	
	if (oldRssi != newRssi)
	{
		ILOG2_DD("RX TH - RxThUpdateRssi newRssi: %d oldRssi:%d",newRssi,oldRssi);
		ASSERT (rxThresholdDb.activeRssiArry[oldRssi] > 0);		
		rxThresholdDb.activeRssiArry[oldRssi]--;
		rxThresholdDb.activeRssiArry[newRssi]++;
		if (rxThresholdDb.activeRssiArry[oldRssi] == 0)
		{
			Utils_ZeroBitInBitmap(rxThresholdDb.activeRssiBitmap,(oldRssi & RX_TH_INDEX_MASK));	
			BitmapChanged = TRUE;
		}
		if (rxThresholdDb.activeRssiArry[newRssi] == 1)
		{
			Utils_SetBitInBitmap(rxThresholdDb.activeRssiBitmap,(newRssi & RX_TH_INDEX_MASK));	
			BitmapChanged = TRUE;
		}
		if ((BitmapChanged)&& ((oldRssiInput == rxThresholdDb.minRssi)||((newRssiInput < rxThresholdDb.minRssi))))
		{//the Bitmap changed and the old rssi was the minRssi or the newRssi is lower than the current minRssi
			if (Utils_CountTrailZeroBitmap(rxThresholdDb.activeRssiBitmap,0,ACTIVE_RSSI_BITMAP_SIZE) < MAX_NUM_OF_ACTIVE_RSSI)
			{
				//find the new min rssi val
				rxThresholdDb.minRssi = StatisticsHandler_FindMinRssi();
				//chack if we need to update the rx th
				StatisticsHandler_CheckAndUpdateRxTh(vapId);				
				//write the new min rssi val to the phy
				StatisticsHandler_WriteMinRssiToPhy(vapId);
			}
		}
	}
}

/**********************************************************************************

StatisticsHandler_CheckAndUpdateRxTh





Description:
------------
	Execute the required
	
Input:
-----
	
**********************************************************************************/
static void StatisticsHandler_CheckAndUpdateRxTh(uint8 vapId)
{
	//the current minRssi and update the rx threshold
	if (rxThresholdDb.minRssi <= (rxThresholdDb.highRxTH - RX_TH_OFFSET_DOWN))
	{
		StatisticsHandler_WriteRxThToPhy(RX_TH_MIN_VALUE, vapId);
	}
	else if ((rxThresholdDb.ishighRxTH == FALSE) && (rxThresholdDb.minRssi > (rxThresholdDb.highRxTH + RX_TH_OFFSET_UP)))
	{
		StatisticsHandler_WriteRxThToPhy(rxThresholdDb.highRxTH, vapId);
	}
}

/**********************************************************************************

StatisticsHandler_RxThAddSta


Description:
------------
	Execute the required
	
Input:
-----
	statisticsManagerMsg - Station attributes(Originated from Driver)
	
**********************************************************************************/
void StatisticsHandler_RxThAddSta(K_MSG *statisticsManagerMsg)
{
	BssManagerStaManagerReq_t *staManagerReq = (BssManagerStaManagerReq_t *)pK_MSG_DATA(statisticsManagerMsg);
	K_MSG *pMsg = staManagerReq->psMsg;
	UMI_STA_ADD *pAddSta = (UMI_STA_ADD *)pK_MSG_DATA(pMsg);
	StaId staId = pAddSta->u16SID;
	uint8 rssiVal = ((uint8)PhyStatistics_getRssi(staId)) & RX_TH_INDEX_MASK;
	bool BitmapChanged = FALSE;
	
	ILOG2_DD("RX TH - RxThAddSta staId: %d rssiVal:%d",staId, rssiVal);
	rxThresholdDb.activeRssiArry[rssiVal]++;
	if (rxThresholdDb.activeRssiArry[rssiVal] == 1)
	{
		Utils_SetBitInBitmap(rxThresholdDb.activeRssiBitmap,(rssiVal));	
		BitmapChanged = TRUE;
	}
	if (BitmapChanged == TRUE)
	{//the Bitmap changed and the old rssi was the minRssi or the newRssi is lower than the current minRssi
		//find the new min rssi val
		rxThresholdDb.minRssi = StatisticsHandler_FindMinRssi();
		//check if we need to update the rx th
		StatisticsHandler_CheckAndUpdateRxTh(statisticsManagerMsg->header.vapId);
		//write the new min rssi val to the phy
		StatisticsHandler_WriteMinRssiToPhy(statisticsManagerMsg->header.vapId);
	}
}

/**********************************************************************************

StatisticsHandler_returnRssiToDbmVal



Description:
------------
	
	
Input:
-----
	
	
**********************************************************************************/
uint8 StatisticsHandler_returnRssiToDbmVal(void)
{
	return GEN5_PHY_RSSI_TO_DBM; 
}

/**********************************************************************************

StatisticsHandler_getNoiseOffset




Description:
------------
	
	
Input:
-----
	
	
**********************************************************************************/
uint8 StatisticsHandler_getNoiseOffset(void)
{
	return GEN5_NOISE_OFFS;
}
/**********************************************************************************

StatisticsManager_InitRxThDB

Description:
------------

	
Input:
-----

**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif
void StatisticsManager_InitRxThDB(void)
{
	rxThresholdDb.minRssi = RX_TH_MAX_VALUE;	
	rxThresholdDb.highRxTH =  RX_TH_MIN_VALUE;	
}

/**********************************************************************************

StatisticsHandler_Init 

Description:
------------
	Set all Hw Counters to 0
		

**********************************************************************************/
void StatisticsHandler_Init(void)
{
	pHostIfCounters =  (HostIfCounters_t*)HOST_IF_COUNTERS_BASE_ADDR;
	pRxCounters =  (RxCounters_t*)RX_PP_COUNTERS_BASE_ADDR; 
	pBaaCounters = (BaaCounters_t *)BA_ANALYZER_COUNTER_BASE_ADDR;
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif

/**********************************************************************************

StatisticsHandler_UpdateBssLoadIeChannelUtilization 

Description:
------------
	updates channel load in bssload ie in shram
	
Input:
-----
	uint8 channelUtilization - current channel load read value

************************************************************************************/
void StatisticsHandler_UpdateBssLoadIeChannelUtilization(uint8 channelUtilization)
{
	uint8 vapId = 0;
	
	// todo: go over all vaps and update only active vaps
	for (vapId = 0; vapId < HW_NUM_OF_VAPS; vapId++)
	{
		bssLoadIePayload[vapId].bssLoad.channelUtilization = (channelUtilization * MAX_UINT8)/HUNDRED_PERCENT;
	}
}

/**********************************************************************************

StatisticsHandler_UpdateBssLoadIeAdmissionCapacity 

Description:
------------
	updates admission capacity in bssload ie in shram (per vap)
	
Input:
-----
	uint32 admissionCapacity

************************************************************************************/
void StatisticsHandler_UpdateBssLoadIeAdmissionCapacity(uint8 vapIndex, uint16 admissionCapacity)
{
	bssLoadIePayload[vapIndex].bssLoad.availableAdmissionCapacity = admissionCapacity;
}

/**********************************************************************************

StatisticsHandler_GetAdmissionCapacity 

Description:
------------
	returns admission capacity in bssload ie in shram (per vap)
	
Input:
-----
	vap ID
	
************************************************************************************/
uint16 StatisticsHandler_GetAdmissionCapacity(uint8 vapId)
{
	return bssLoadIePayload[vapId].bssLoad.availableAdmissionCapacity;
}

/**********************************************************************************

statisticsGetHostIfCountersAddress 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic
	
**********************************************************************************/
uint32* statisticsGetHostIfCountersAddress(void)
{
	return ((uint32*)pHostIfCounters);
}

/**********************************************************************************

statisticsGetRxCountersAddress 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic
	
**********************************************************************************/
uint32* statisticsGetRxCountersAddress(void)
{
	return ((uint32*)pRxCounters);
}

/**********************************************************************************

statisticsGetBaaxCountersAddress 


Description:
------------

Input: 
-----
		
Output:
-------
	
Returns:
--------
Address in memory of the statistic
	
**********************************************************************************/
uint32* statisticsGetBaaCountersAddress(void)
{
	return ((uint32*)pBaaCounters);
}


void StatisticsCalcLinkAdaptationOffsets(void)
{
	/*get the address of the VAP Frame error count*/	
	LinkAdaptationVapFrameErrorCountBaseAddress = (uint32)(&(pBaaCounters->txSenderError));

	/*get the address of the WLAN Frame error count*/	
	LinkAdaptationWlanFrameErrorCountBaseAddress =  (uint32)(&(pBaaCounters->txFrameErrorCount));
}

