/***********************************************************************************
 File:			StatisticsPhyHandler.c			
 Module:		Debug Enviroment
 Purpose: 		To handle PHY metrics counters - parallel to the HW 
 				counters in wave500
 Description:   This file is the implementation of the statistics handler which is 
 				responsible of handling the global statistics counters
************************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "StatisticsManager_api.h"
#include "HwGlobalDefinitions.h"
#include "EventsManager_api.h"
#include "StatisticsPhyHandler.h"
#include "StatisticsHandler.h"
#include "RxMpduStructure.h"
#include "Statistics_Api.h"
#include "AlphaFilter.h"
#include "OSAL_Interrupts.h"
#include "mhi_umi.h"
#include "OSAL_UpperMacMessages.h"
#include "DmaManager_Api.h"
#include "Pac_Api.h"
#include "RxHandler_PhyStatusApi.h"
#include "ShramPhyStatDb.h"
#include "MT_Math.h"
#include "CoC_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "RxDescriptor.h"
#include "stringLibApi.h"
#include "mhi_umi.h"
#include "Dut_Api.h"
#include "Utils_Api.h"
#include "HdkCdbManagerTask_api.h"
#include "ChannelSwitchManager_Api.h"
#include "StatisticsManager.h"
#include "ShramStatistics.h"


#define LOG_LOCAL_GID   GLOBAL_GID_DEBUG_ENVIRONMENT
#define LOG_LOCAL_FID 9

#define SINGLE_ANTENNA_MASK  						0xFF
#define SINGLE_ANTENNA_SHIFT 						8
#define RX_TH_MIN_VALUE_DB	 						(RX_THRESHOLD_DEFAULT_VALUE)
#define ALPHA_VALUE  								205  // 0.1 *(2^11)
#define CHANNEL_LOAD_TIMER_VALUE 					50 //50 mSec
#define PHY_STA_BITMAP_SIZE 						((HW_NUM_OF_STATIONS + 7)>>3)
#define PHY_STA_BITMAP_SIZE_IN_WORDS 				((PHY_STA_BITMAP_SIZE +3)>>2)
#define PHY_SHAPED_RSSI_KOEF                        1 // 1 = 1/2 a negative power of 2 or param/2 in case of 1
#define PHY_SHAPED_NOISE_KOEF                       1 // 1 = 1/2 a negative power of 2 or param/2 in case of 1
#define PHY_SHAPED_METRICS(A,B,C,KOEF)                (B == C ? A : PHY_SHAPED_FORMULA(A,B,KOEF))
#define PHY_METRIC_2_NOISE_INTEGER_VALUE_SHIFT      4
#define PHY_METRIC_2_NOISE_INTEGER_VALUE_MASK       0xF

typedef struct phyStatDb
{
	metricFifoEntry_t* pMetricFifo;
	uint32	activeStaBitmap[PHY_STA_BITMAP_SIZE_IN_WORDS];
	uint32	*pPhyMetricsFifoReadCounter;
	uint32	*pPhyMetricsFifoWriteCounter;
	uint32	fifoReadIndex;
	uint32	totalRxTime;
	uint32	totalChannelLoad;				// Total channel load between PHY status REQ
	uint32	totalChannelUtilization;		// Total channel utilization between PHY status REQ
	uint32	totalChannelLoadBssLoad;		// Total channel load between channel load timeouts - used to update BSS load IE
 	uint32	channelLoadMesStartTime;		// Start time of channel load measurements - used for channel load and channel utilization
 	uint32	channelLoadMesStartTimeBssLoad;	// Start time of BSS load measurements - used for BSS load IE
	K_MSG*	pMcMsg;
	uint8  	currentRxActiveAntBitMap[HDK_NUM_OF_BANDS]; 	//bitmap indicating current active antennas 
	uint8	numberOfStations;
	bool  	dmaInUse;
	uint8 	dmaClientId;
	bool  	timerActive;
}phyStatDb_t;


static phyStatDb_t phyStatisticDb;
/*Rate should be stored in MBPS * 10*/
const uint16 phyRate11bByRate[] =
{
	PHY_RATE_INVALID,		
	20,     /* 2 Mbits/s */
	55,    /* 5.5 Mbits/s */
	110,     /* 11 Mbits/s */
	10,     /* 1 Mbits/s */
	20,     /* 2 Mbits/s */
	55,    /* 5.5 Mbits/s */
	110     /* 11 Mbits/s */
};

/**********************************************************************************
PhyStatistics_MetricsFifoNotEmpty  

Description:
------------
	read from metrics fifo and avarege RF gain , noise and Rssi. 
Input:
-----
	msg from Rx handler	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_MetricsFifoNotEmpty(K_MSG* msg)
{
	TX_INTERRUPT_SAVE_AREA;
	metricFifoEntry_t* pFifoEntry = NULL;
	uint8 numOfAnt;
	int8 rssiNewValue;
	int8 rssiOldValue;
	uint8 noiseValue;
	uint8 rfGainValue;
	uint16 stationId = RX_MPDU_PHY_METRICS_INVALID_STA_INDEX;	/*has to be uint16 because invalid STA is 0x1FF*/
	uint8 oldValue;
	uint8 rdType;
	uint32 CurrRxTime =0;
	int8 oldMaxRssi;
	uint32 packetLength =  0;
	bool isAmpdu = FALSE; 
	uint8 rate11b;
    uint16 rate = 0;
    int16 noiseWithGain = 0;
        
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	// run till the fifo is empty
	while ((phyStatisticDb.dmaInUse == FALSE) && ((*(phyStatisticDb.pPhyMetricsFifoReadCounter) != (*(phyStatisticDb.pPhyMetricsFifoWriteCounter)))))
	{
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		pFifoEntry = (metricFifoEntry_t*)&(phyStatisticDb.pMetricFifo[phyStatisticDb.fifoReadIndex]);
#ifdef BEEROCK_DEBUG        
        SLOG0(0, 0, metricFifoEntry_t, pFifoEntry);
#endif
        rdType = pFifoEntry->rxPhyMetrics.reserved1;
#ifdef BEEROCK_DEBUG
		ILOG0_D("[PhyStatistics_MetricsFifoNotEmpty], rdType = %d", rdType);
#endif

		isAmpdu = pFifoEntry->rxPhyMetrics.aggregate;
		stationId = pFifoEntry->rxPhyMetrics.stationIndex; // get station id
		
		if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
		{
			ASSERT(stationId < HW_NUM_OF_STATIONS);
			if (Utils_GetBitFromBitmap(phyStatisticDb.activeStaBitmap,stationId)==0)
			{
				stationId = RX_MPDU_PHY_METRICS_INVALID_STA_INDEX;
			}
		}

        if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
        {
            if(isAmpdu == TRUE)
            {
                packetLength = pFifoEntry->rxPhyMetrics.geriscSpare2;   //MPDU when aggregate
            }
            else
            {
                packetLength = pFifoEntry->rxPhyMetrics.length;         //MPDU length when non-aggregate
            }
			
            if ((rdType == RD_TYPE_UNICAST_QOS_DATA) || (rdType == RD_TYPE_RSSI))
            {
                if (pFifoEntry->rxPhyMetrics.phyMode != PHY_MODE_11B)
                {
                    phyRxStatusDb.staPhyRxStatus[stationId].phyRate= pFifoEntry->rxPhyMetrics.phyRate; // save current rate
                }
                else
                {
                    rate11b = (pFifoEntry->rxPhyMetrics.rate & 0x7);
                    /*if rate is expected in MBPS and PHY rate is 11B we need to write it ourselves*/
                    phyRxStatusDb.staPhyRxStatus[stationId].phyRate =  phyRate11bByRate[rate11b];
                }
                rate = phyRxStatusDb.staPhyRxStatus[stationId].phyRate;
            }
            else
            {
                rate = pFifoEntry->rxPhyMetrics.phyRate;
            }
         /*
            To calculate the time [microsec] we shall use the rate provided metrics which is 10*Mega*Bits/sec
            rate [Mega*Bits/sec] = metricRate/10 [Mega*Bits/sec]
            We have the size recieved stored in metrics in Bytes
            length [Bytes]
            So the folrmula is: 
            time[microsec] = length[8bits]/rate[Mega*bits/sec]
            lengt[Bytes]/metricRate[Mega*bits/10sec]  = 10*lenght[8bits]/2^20*10^-6*metricRate[bits / microsec] 
            * we will reduce calculating to not overflow, -- 10^-6 =~ 2^-20 -- 
            8*10*lenght/metricRate [microsec] 
        */
            CurrRxTime = ((80 * packetLength) / rate);
#ifdef BEEROCK_DEBUG
            ILOG0_DDD("CurrRxTimes : %d, packetLength:%d, rate:%d",CurrRxTime,packetLength, rate);
#endif
            /* Increment the RX time in the per client rx time statistics */
            phyRxStatusDb.staPhyRxStatus[stationId].perClientRxtimeUsage += CurrRxTime;
            // Update total rx time for all valid stations
			phyStatisticDb.totalRxTime += CurrRxTime; // add current Rx packet air time to total Rx air time for channel load calculation
        }


		if (pFifoEntry->rxPhyMetrics.bssIndex != METRICS_INVALID_VAP)
		{
            // if RD_TYPE RSSI then increase statistics of packets and bytes received from Unconnected Station that is being monitored
            if ((stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX) && (rdType == RD_TYPE_RSSI))
            {
                //increase packet number
				pRxCounters->staCounts[stationId].rdCount++;
                //sum packet length
				pHostIfCounters->staCounts[stationId].rxOutStaNumOfBytes+=packetLength;
#ifdef BEEROCK_DEBUG
                SLOG0(0, 0, metricFifoEntry_t, pFifoEntry);
                ILOG0_DDD("[PhyStatistics_MetricsFifoNotEmpty], stationId = %d, isAmpdu = %d, packetLength = %d", stationId, isAmpdu, packetLength);
#endif
            }
			if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
			{
				/* Store maxRssi [dBm] in oldMaxRssi and initalize maxRssi to minimal [dB] value */							
				oldMaxRssi = phyRxStatusDb.staPhyRxStatus[stationId].maxRssi;
				phyRxStatusDb.staPhyRxStatus[stationId].maxRssi = RX_TH_MIN_VALUE_DB;

				/* Increment the RX time in the per client rx time statistics */
				phyRxStatusDb.staPhyRxStatus[stationId].perClientRxtimeUsage += CurrRxTime;

                // Udpate total rx time for all valid stations
                if (pFifoEntry->rxPhyMetrics.bssIndex != METRICS_INVALID_VAP)
                {
                    phyStatisticDb.totalRxTime += CurrRxTime; // add current Rx packet air time to total Rx air time for channel load calculation            
                }
			}	
			for (numOfAnt = 0; numOfAnt < NUM_OF_RX_ANT; numOfAnt++)
			{
				if (phyStatisticDb.currentRxActiveAntBitMap[CONFIGURATION_MANAGER_BAND_0] & (1<<numOfAnt))
				{
                    // calc noise  per ant
                    noiseValue = (pFifoEntry->rxPhyMetrics.noise & SINGLE_ANTENNA_MASK); // get single antenna noise value (will shift to next ant at end of loop)
					oldValue = phyRxStatusDb.devicePhyRxStatus.noise[numOfAnt];
					phyRxStatusDb.devicePhyRxStatus.noise[numOfAnt] = PHY_SHAPED_METRICS(noiseValue, oldValue, 0, PHY_SHAPED_NOISE_KOEF);
                    
                    // calc gain per ant
					rfGainValue = (pFifoEntry->rxPhyMetrics.RFGain & SINGLE_ANTENNA_MASK); // get single antenna noise value (will shift to next ant at end of loop)
					oldValue = phyRxStatusDb.devicePhyRxStatus.rf_gain[numOfAnt];
	 				phyRxStatusDb.devicePhyRxStatus.rf_gain[numOfAnt] = PHY_SHAPED_METRICS(rfGainValue, oldValue, 0, PHY_SHAPED_NOISE_KOEF);
					if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
					{
						rssiNewValue = (pFifoEntry->rxPhyMetrics.RSSI & SINGLE_ANTENNA_MASK);// get single antenna RSSI value
						rssiOldValue = phyRxStatusDb.staPhyRxStatus[stationId].rssi[numOfAnt];
						phyRxStatusDb.staPhyRxStatus[stationId].rssi[numOfAnt] = PHY_SHAPED_METRICS(rssiNewValue, rssiOldValue, RX_THRESHOLD_DEFAULT_VALUE, PHY_SHAPED_RSSI_KOEF);
						if (phyRxStatusDb.staPhyRxStatus[stationId].rssi[numOfAnt] > phyRxStatusDb.staPhyRxStatus[stationId].maxRssi)
						{
							phyRxStatusDb.staPhyRxStatus[stationId].maxRssi = phyRxStatusDb.staPhyRxStatus[stationId].rssi[numOfAnt];
						}					
                        // calculate noise including GAIN
                        noiseWithGain = REMOVE_PHY_RESOLUTION(CONVERT_PHY_LOG10_TO_LOG2(noiseValue)) - GEN5_NOISE_OFFS - rfGainValue;
                        phyRxStatusDb.staPhyRxStatus[stationId].noise[numOfAnt] = noiseWithGain;
					}

				}
				else
				{
					//antenna is not active - initalize all phy-statistics
					if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
					{
						phyRxStatusDb.staPhyRxStatus[stationId].rssi[numOfAnt] = (uint8)RX_TH_MIN_VALUE_DB;
					}
					phyRxStatusDb.devicePhyRxStatus.noise[numOfAnt] = 0;
					phyRxStatusDb.devicePhyRxStatus.rf_gain[numOfAnt] = 0;
				}
				pFifoEntry->rxPhyMetrics.RSSI = pFifoEntry->rxPhyMetrics.RSSI >> SINGLE_ANTENNA_SHIFT; // move to next antenna RSSI value
				pFifoEntry->rxPhyMetrics.noise = pFifoEntry->rxPhyMetrics.noise >> SINGLE_ANTENNA_SHIFT;// move to next antenna noise value
				pFifoEntry->rxPhyMetrics.RFGain = pFifoEntry->rxPhyMetrics.RFGain >> SINGLE_ANTENNA_SHIFT;// move to next antenna RF gain
			}
			if (stationId != RX_MPDU_PHY_METRICS_INVALID_STA_INDEX)
				{
					uint8 vapId;
					//convert maxRssi from dB back to dBm
					phyRxStatusDb.staPhyRxStatus[stationId].maxRssi = (int8)MAX((int16)(phyRxStatusDb.staPhyRxStatus[stationId].maxRssi  - StatisticsHandler_returnRssiToDbmVal()), (int16)RX_THRESHOLD_DEFAULT_VALUE);
					
					vapId = ConfigurationManager_GetFirstVapForBand((BandId_e)ConfigurationManager_GetBandForStation(stationId));
					StatisticsHandler_RxThUpdateRssi(oldMaxRssi, phyRxStatusDb.staPhyRxStatus[stationId].maxRssi, vapId);
				}
		}
		else
		{
			/*We need to identify valid MPDUs received from other BSS*/
			/*However, VAP ID is invalid for all Broadcast packets, including for Beacons which may be from our peer AP in Client mode or our Probe REQs from a connected STA*/
			if ((stationId == RX_MPDU_PHY_METRICS_INVALID_STA_INDEX))
			{
				/* This statistic is an activity indication, so we just set it to 1 */
				BaaCounters_t *baaCounterAddress_p = (BaaCounters_t*)statisticsGetBaaCountersAddress();
				baaCounterAddress_p->rxBasicReport = 1;
			}
			for (numOfAnt = 0; numOfAnt < NUM_OF_RX_ANT ; numOfAnt++)
			{
				//ILOG0_D("phyStatisticDb.currentRxActiveAntBitMap, %d", phyStatisticDb.currentRxActiveAntBitMap);
				if (phyStatisticDb.currentRxActiveAntBitMap[CONFIGURATION_MANAGER_BAND_0] & (1<<numOfAnt))
				{
					noiseValue = (pFifoEntry->rxPhyMetrics.noise& SINGLE_ANTENNA_MASK); // get single antenna noise value
					oldValue = phyRxStatusDb.devicePhyRxStatus.noise[numOfAnt];
					phyRxStatusDb.devicePhyRxStatus.noise[numOfAnt] = PHY_SHAPED_METRICS(noiseValue, oldValue, 0, PHY_SHAPED_NOISE_KOEF);
					rfGainValue = (pFifoEntry->rxPhyMetrics.RFGain& SINGLE_ANTENNA_MASK); // get single antenna noise value
					oldValue = phyRxStatusDb.devicePhyRxStatus.rf_gain[numOfAnt];
	 				phyRxStatusDb.devicePhyRxStatus.rf_gain[numOfAnt] = PHY_SHAPED_METRICS(rfGainValue, oldValue, 0, PHY_SHAPED_NOISE_KOEF);
				}
				//pFifoEntry->rxPhyMetrics.RSSI = pFifoEntry->rxPhyMetrics.RSSI >> SINGLE_ANTENNA_SHIFT; // move to next antenna RSSI value
				pFifoEntry->rxPhyMetrics.noise = pFifoEntry->rxPhyMetrics.noise >> SINGLE_ANTENNA_SHIFT;// move to next antenna noise value
				pFifoEntry->rxPhyMetrics.RFGain = pFifoEntry->rxPhyMetrics.RFGain >> SINGLE_ANTENNA_SHIFT;// move to next antenna RF gain
			}				
		}
		OSAL_DISABLE_INTERRUPTS(&interrupt_save);
		(*phyStatisticDb.pPhyMetricsFifoReadCounter)++;// increment read counter
		phyStatisticDb.fifoReadIndex = (phyStatisticDb.fifoReadIndex + 1)%SIZE_OF_METRICS_FIFO;// move to the next fifo entry
	}
   	EventManager_TurnOnEvent(EVENT_ID_METRICS_FIFO_NOT_EMPTY);
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

/**********************************************************************************
PhyStatistics_GetPhyStatusReq  

Description:
------------
	trigger DMA to copy phy stat Db from Shram to DDR
Input:
-----
	msg from Driver
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_GetPhyStatusReq(void)
{
	//In gen5 we call it here since it is in UM. In gen6 we send a MSG instead to the LM.
	channelSwitchManager_CalcChannelLoad();

	// Go to HDK to collect statistics
	OSAL_SEND_NO_DATA_MESSAGE(HDK_GET_PHY_STATISTICS, TASK_HDK, ConfigurationManager_GetFirstVapForBand0());
}



void PhyStatistics_HdkStatisticsCfm(K_MSG* msg)
{
	// Do DMA for all stats 
	statisticsManagerCopyStatisticsStart(); 		
}
    
/***********************************************************************
* PhyStatistics_DmaDone
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PhyStatistics_DmaDone(void)
{
	/* Clear activity indication */
	BaaCounters_t *baaCounterAddress_p = (BaaCounters_t*)statisticsGetBaaCountersAddress();
	baaCounterAddress_p->rxBasicReport = 1;
}

/**********************************************************************************
PhyStatistics_SetAntennaActiveBitmap

Description:
------------
	Set Active antenna mask on RX. Sent by CoC.
Input:
-----

Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_SetAntennaActiveBitmap(K_MSG* psMsg)
{
	cocUpdateAntMaskMsg_t* numOfActivateAnts = (cocUpdateAntMaskMsg_t*)pK_MSG_DATA(psMsg);

	phyStatisticDb.currentRxActiveAntBitMap[numOfActivateAnts->band] = numOfActivateAnts->currentRxAntsMask;
}


/**********************************************************************************
PhyStatistics_getRssi



Description:
------------
	 return the rssi of stationId in numOfAnt
Input:
-----
	stationId

Returns:
--------
	rssi
**********************************************************************************/
int8 PhyStatistics_getRssi(StaId stationId)
{
	return (phyRxStatusDb.staPhyRxStatus[stationId].maxRssi);
}

/**********************************************************************************
PhyStatistics_getTotalRssi



Description:
------------
	 return the sum of all STA RSSI
Input:
-----
	stationId

Returns:
--------
	rssi
**********************************************************************************/
int32 PhyStatistics_getTotalRssi()
{
	uint32	activeStaBitmap[PHY_STA_BITMAP_SIZE_IN_WORDS];
	uint8	staId;
	uint8	count;
	int32 	totalRssi = 0;

	/*Copy STA bitmap*/
	memcpy32(activeStaBitmap, phyStatisticDb.activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS);
	for (count = 0; count < phyStatisticDb.numberOfStations; count++)
	{		
		/* Find a STA */
		staId = (uint8)Utils_FindLastSetAndClear(activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS * NUM_OF_BITS_IN_UINT32);
		/* Increment total RSSI */
		totalRssi += phyRxStatusDb.staPhyRxStatus[staId].maxRssi;
	}
	return (totalRssi);
}
/***********************************************************************
* PhyStatistics_resetStaPhyRxStatus
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PhyStatistics_resetStaPhyRxStatus()
{
	uint32	activeStaBitmap[PHY_STA_BITMAP_SIZE_IN_WORDS];
	StaId	staId;
	int8	maxRssi;
	uint8	count;
	uint8 	vapId;
	
	memcpy32(activeStaBitmap, phyStatisticDb.activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS);
	for (count = 0; count < phyStatisticDb.numberOfStations; count++)
	{		
		/* Find a STA */
		staId = (uint8)Utils_FindLastSetAndClear(activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS * NUM_OF_BITS_IN_UINT32);
		/* Store STA's MAX RSSI */
		maxRssi = phyRxStatusDb.staPhyRxStatus[staId].maxRssi;
		/* Clear STA's status */
		memset32(&phyRxStatusDb.staPhyRxStatus[staId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
		/* Initialize antenna values (rssi and maxRssi) */
		memset(&phyRxStatusDb.staPhyRxStatus[staId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
		phyRxStatusDb.staPhyRxStatus[staId].maxRssi = RX_THRESHOLD_DEFAULT_VALUE;

		vapId = ConfigurationManager_GetFirstVapForBand((BandId_e)ConfigurationManager_GetBandForStation(staId));
		StatisticsHandler_RxThUpdateRssi(maxRssi ,RX_THRESHOLD_DEFAULT_VALUE, vapId);
	}
}

/**********************************************************************************
PhyStatistics_addStaPhyRxStatus




Description:
------------
	 return the rssi of stationId in numOfAnt
Input:
-----
	stationId , numOfAnt

Returns:
--------
	rssi
**********************************************************************************/
void PhyStatistics_addStaPhyRxStatus(StaId stationId, uint8 maxRssi)
{
	uint8 indexAnt;
	
	memset32(&phyRxStatusDb.staPhyRxStatus[stationId],0,CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
	// convert maxRssi from dB to dBm
	phyRxStatusDb.staPhyRxStatus[stationId].maxRssi = (int8)MAX((int16)(maxRssi- StatisticsHandler_returnRssiToDbmVal()), (int16)RX_THRESHOLD_DEFAULT_VALUE);
	// initalize all rssi values to minimum
	for (indexAnt = 0; indexAnt<PHY_STATISTICS_MAX_RX_ANT ; indexAnt++)
	{
		phyRxStatusDb.staPhyRxStatus[stationId].rssi[indexAnt] = (uint8)RX_THRESHOLD_DEFAULT_VALUE; 
	}
}

/**********************************************************************************
PhyStatistics_removeStaPhyRxStatus





Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
**********************************************************************************/
void PhyStatistics_removeStaPhyRxStatus(StaId stationId)
{
	/* Clear STA's status */
	memset32(&phyRxStatusDb.staPhyRxStatus[stationId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
	/* Initialize antenna values (rssi and maxRssi) */
	memset(&phyRxStatusDb.staPhyRxStatus[stationId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
	phyRxStatusDb.staPhyRxStatus[stationId].maxRssi = RX_THRESHOLD_DEFAULT_VALUE;
}

/**********************************************************************************
PhyStatistics_addSta




Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
**********************************************************************************/
void PhyStatistics_addSta(StaId stationId)
{
	// Set the bit representing the given STA
	Utils_SetBitInBitmap(phyStatisticDb.activeStaBitmap,stationId);
	/* Increment num of stations */
	phyStatisticDb.numberOfStations += 1;
}

/**********************************************************************************
PhyStatistics_removeSta




Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
*********************************************************************************/
void PhyStatistics_removeSta(StaId stationId)
{
	// Clear the bit representing the given STA
	Utils_ZeroBitInBitmap(phyStatisticDb.activeStaBitmap,stationId);
	/* Decrement num of stations */
	phyStatisticDb.numberOfStations -= 1;
}

/**********************************************************************************
PhyStatistics_Init


Description:
------------
	 Initialization of phy stats
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
#if defined (ENET_INC_UMAC)
#pragma ghs section text=".initialization" 
#endif
void PhyStatistics_Init()
{
	// reset DBs
	memset(&phyRxStatusDb , 0 , sizeof(wholePhyRxStatusDb_t)); 
	memset(&phyStatisticDb , 0 , sizeof(phyStatDb_t));
	// set alpha value
	memset32(&phyMetricFifo , 0  , SIZE_OF_METRICS_FIFO*PHY_STATUS_NUMBER_OF_WORDS + 2);
	phyStatisticDb.pPhyMetricsFifoReadCounter = (uint32*)(&phyMetricFifo[0]);
	phyStatisticDb.pPhyMetricsFifoWriteCounter = (uint32*)(&phyMetricFifo[1]);
	phyStatisticDb.pMetricFifo = (metricFifoEntry_t*)(&phyMetricFifo[2]); // set metrics fifo pointer
	*(phyStatisticDb.pPhyMetricsFifoReadCounter) = 0;
}
#if defined (ENET_INC_UMAC)
#pragma ghs section text=default
#endif


