/***********************************************************************************
 File:			StatisticsPhyHandler.c			
 Module:		Debug Enviroment
 Purpose: 		To handle PHY metrics counters - parallel to the HW 
 				counters in wave500
 Description:   This file is the implementation of the statistics handler which is 
 				responsible of handling the global statistics counters
************************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "StatisticsManager_api.h"
#include "HwGlobalDefinitions.h"
#include "EventsManager_api.h"
#include "StatisticsPhyHandler.h"
#include "StatisticsHandler.h"
#include "RxMpduStructure.h"
#include "Statistics_Api.h"
#include "AlphaFilter.h"
#include "OSAL_Interrupts.h"
#include "mhi_umi.h"
#include "OSAL_UpperMacMessages.h"
#include "DmaManager_Api.h"
#include "Pac_Api.h"
#include "ShramPhyStatDb.h"
#include "MT_Math.h"
#include "CoC_Api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "RxDescriptor.h"
#include "stringLibApi.h"
#include "mhi_umi.h"
#include "Dut_Api.h"
#include "Utils_Api.h"
#include "HdkCdbManagerTask_api.h"
#include "PacExtrapolator_api.h"
#include "StatisticsManager.h"
#include "ShramStatistics.h"
#include "mhi_umi.h"


#define LOG_LOCAL_GID   GLOBAL_GID_DEBUG_ENVIRONMENT
#define LOG_LOCAL_FID 11

#define RX_TH_MIN_VALUE_DB	 						(RX_THRESHOLD_DEFAULT_VALUE)
#define CHANNEL_LOAD_TIMER_VALUE 					50 //50 mSec

#define PHY_STA_BITMAP_SIZE 						((HW_NUM_OF_STATIONS + 7)>>3)
#define PHY_STA_BITMAP_SIZE_IN_WORDS 				((PHY_STA_BITMAP_SIZE +3)>>2)


typedef struct phyStatDb_s
{
	uint8  	channelLoadCalculatedCount;
	uint8	rxAbortCounterCount;
}phyStatDb_t;

typedef struct phyBandDb_s
{
	uint32	activeStaBitmap[PHY_STA_BITMAP_SIZE_IN_WORDS];
	uint8	numberOfStations;
}phyBandDd_t;


static phyStatDb_t phyStatisticDb;
/*Rate should be stored in MBPS * 10*/
static phyBandDd_t phyBandDb[NUM_OF_CONFIGURATION_MANAGER_BANDS];


/**********************************************************************************
PhyStatistics_MetricsFifoNotEmpty  

Description:
------------
	read from metrics fifo and avarege Rf gain , noise and Rssi. 
Input:
-----
	msg from Rx handler	
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_MetricsFifoNotEmpty(K_MSG* msg)
{
	UNUSED_PARAM(msg);
//TBD - remove this func
}

/**********************************************************************************
PhyStatistics_GetPhyStatusReq  

Description:
------------
	trigger DMA to copy phy stat Db from Shram to DDR
Input:
-----
	msg from Driver
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_GetPhyStatusReq(void)
{
	// Go to HDK to collect statistics
	OSAL_SEND_NO_DATA_MESSAGE(HDK_GET_PHY_STATISTICS, TASK_HDK, ConfigurationManager_GetFirstVapForBand0());

	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
	{
		OSAL_SEND_NO_DATA_MESSAGE(HDK_GET_PHY_STATISTICS, TASK_HDK, ConfigurationManager_GetFirstVapForBand1());
	}
}


/***********************************************************************
* PhyStatistics_ChannelLoadCalculated
* 
* Description:
* ------------
* 
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PhyStatistics_ChannelLoadCalculated(K_MSG* msg)
{	
	UNUSED_PARAM(msg);	
	phyStatisticDb.channelLoadCalculatedCount++;
	
	if (phyStatisticDb.channelLoadCalculatedCount == ConfigurationManager_GetNumOfActiveBands())
	{
		phyStatisticDb.channelLoadCalculatedCount = 0;

		ILOG0_DDDD("***PhyStatistics_ChannelLoadCalculated***channel_load Band0:%d, totalChannelUtilization Band0:%d, channel_load Band1:%d, totalChannelUtilization Band1:%d", 
						phyRxStatusDb.devicePhyRxStatus[0].channel_load, 
						phyRxStatusDb.devicePhyRxStatus[0].totalChannelUtilization, 
						phyRxStatusDb.devicePhyRxStatus[1].channel_load, 
						phyRxStatusDb.devicePhyRxStatus[1].totalChannelUtilization);

		// Both LMAC answered: do DMA for all stats 
		statisticsManagerCopyStatisticsStart(); 		
	}
}

void PhyStatistics_HdkStatisticsCfm(K_MSG* msg)
{
	uint8 numActiveBands = ConfigurationManager_GetNumOfActiveBands();
	UNUSED_PARAM(msg);

	phyStatisticDb.rxAbortCounterCount++;

	// Wait for both all active bands to reply
	if (phyStatisticDb.rxAbortCounterCount == numActiveBands)
	{
		phyStatisticDb.rxAbortCounterCount = 0;

		// Now go get the channel load from all active bands
		pacExtrapolator_DisableModule();
		
		OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CALCULATE_CHANNEL_LOAD, TASK_CHANNEL_SWITCH_MANAGER , ConfigurationManager_GetFirstVapForBand0());
		
		if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_DUAL_BAND)
		{
			OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_CALCULATE_CHANNEL_LOAD, TASK_CHANNEL_SWITCH_MANAGER , ConfigurationManager_GetFirstVapForBand1());
		}
	}
}

/**********************************************************************************
PhyStatistics_ChannelSwitchNotify  

Description:
------------
	notification that channel switch is in progress
Input:
-----
	msg from CSM
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_ChannelSwitchNotify(K_MSG* msg)
{
	uint8 	bandId = ConfigurationManager_GetBandForVap(msg->header.vapId);
	uint32	activeStaBitmap[PHY_STA_BITMAP_SIZE_IN_WORDS];
	StaId	staId;
	uint8	count;

	pacExtrapolator_DisableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_DisableMinRssiModule();
#endif

    ILOG1_V("PhyStatistics_ChannelSwitchNotify");
	memset(&phyRxStatusDb.devicePhyRxStatus[bandId] , 0 , sizeof(devicePhyRxStatusDb_t));
	memcpy32(activeStaBitmap, phyBandDb[bandId].activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS);
	for (count = 0; count < phyBandDb[bandId].numberOfStations; count++)
	{		
		/* Find a STA */
		staId = (uint8)Utils_FindLastSetAndClear(activeStaBitmap, PHY_STA_BITMAP_SIZE_IN_WORDS * NUM_OF_BITS_IN_UINT32);
        
		/* Clear STA's status */
		memset32(&phyRxStatusDb.staPhyRxStatus[staId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
        
		// initalize all rssi values to minimum
		memset(&phyRxStatusDb.staPhyRxStatus[staId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
		memset(&phyRxStatusDb.staPhyRxStatus[staId].rssiAckAvarage.rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT);    
        
		/*Set Max RSSI to Minimum*/
		pacExtrapolatorRssi.maxRssi[staId] = (int16)RX_THRESHOLD_DEFAULT_VALUE;				
		/* Set min rssi to maximum*/
		pacExtrapolatorRssi.minRssi[staId] = (int16)MAX_INT8;		
	}
	pacExtrapolator_EnableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_EnableMinRssiModule();
	pacExtrapolator_TriggerMinRssiModule();
#endif	
	OSAL_SEND_NO_DATA_MESSAGE(CHANNEL_SWITCH_MANAGER_SET_NOTIFY_CFM, TASK_CHANNEL_SWITCH_MANAGER, msg->header.vapId);
}

    
/***********************************************************************
* PhyStatistics_DmaDone
* 
* Description:
* ------------
* This function enables phy stat DB updating (it was halted when we got a request from Driver to copy stats from SHRAM to DDR)
* 
* Input:
* ------
* None
* 
* Output:
* -------
* None
* 
* Returns:
* --------
* None
* 
************************************************************************/
void PhyStatistics_DmaDone(void)
{
 	pacExtrapolator_EnableModule();
}
/**********************************************************************************
Statistics_SetAntennaActiveBitmap
Description:
------------
	Set Active antenna mask on RX. Sent by CoC.
Input:
-----

Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_SetAntennaActiveBitmap(K_MSG* psMsg)
{
	UNUSED_PARAM(psMsg);
//TBD - remove this func
}

/**********************************************************************************
PhyStatistics_removeStaPhyRxStatus





Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
**********************************************************************************/
void PhyStatistics_removeStaPhyRxStatus(StaId stationId)
{
	/* Clear STA's status */
	memset32(&phyRxStatusDb.staPhyRxStatus[stationId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
	/* Initialize antenna values (rssi and maxRssi) */
	memset(&phyRxStatusDb.staPhyRxStatus[stationId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
}
/**********************************************************************************
PhyStatistics_addSta




Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
**********************************************************************************/
void PhyStatistics_addSta(StaId stationId,uint8 maxRssi)
{
	BandId_e bandId = ConfigurationManager_GetBandForStation(stationId);
	
	// Set the bit representing the given STA
	Utils_SetBitInBitmap(phyBandDb[bandId].activeStaBitmap,stationId);
	/* Increment num of stations */
	phyBandDb[bandId].numberOfStations += 1;
	/*	Add STA to PAC extrapolator*/
	pacExtrapolator_DisableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_DisableMinRssiModule();
#endif
	/* Clear STA's status */
	memset32(&phyRxStatusDb.staPhyRxStatus[stationId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
	// initalize all rssi values to minimum
	memset(&phyRxStatusDb.staPhyRxStatus[stationId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
	memset(&phyRxStatusDb.staPhyRxStatus[stationId].rssiAckAvarage.rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT);     
    
	pacExtrapolator_addSta(stationId,maxRssi);
	pacExtrapolator_EnableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_EnableMinRssiModule();	
	pacExtrapolator_TriggerMinRssiModule();
#endif	
}

/**********************************************************************************
PhyStatistics_removeSta




Description:
------------
	 
Input:
-----
	stationId

Returns:
--------
	
**********************************************************************************/
void PhyStatistics_removeSta(StaId stationId)
{
	BandId_e bandId = ConfigurationManager_GetBandForStation(stationId);
	
	// Clear the bit representing the given STA
	Utils_ZeroBitInBitmap(phyBandDb[bandId].activeStaBitmap,stationId);
	/* Decrement num of stations */
	phyBandDb[bandId].numberOfStations -= 1;
	/*	Remove STA from PAC extrapolator*/
	pacExtrapolator_DisableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_DisableMinRssiModule();
#endif
	/* Clear STA's status */
	memset32(&phyRxStatusDb.staPhyRxStatus[stationId], 0, CONVERT_BYTES_TO_WORDS(sizeof(stationPhyRxStatusDb_t)));
	// initalize all rssi values to minimum
	memset(&phyRxStatusDb.staPhyRxStatus[stationId].rssi[0], RX_THRESHOLD_DEFAULT_VALUE, PHY_STATISTICS_MAX_RX_ANT); 
	pacExtrapolator_removeSta(stationId);
	pacExtrapolator_EnableModule();
#ifndef PAC_EXTRAPOLATOR_DISABLE_MIN_RSSI	
	pacExtrapolator_EnableMinRssiModule();
	pacExtrapolator_TriggerMinRssiModule();
#endif	
}



/**********************************************************************************
PhyStatistics_Init


Description:
------------
	 Initialization of phy stats
Input:
-----
	None
Output:
-------
	None
Returns:
--------
	None
**********************************************************************************/
void PhyStatistics_Init()
{
	// reset DBs
	memset(&phyRxStatusDb , 0 , sizeof(wholePhyRxStatusDb_t)); 
	memset(&phyStatisticDb , 0 , sizeof(phyStatDb_t));
	memset(&phyBandDb[0] , 0 , sizeof(phyBandDd_t)*NUM_OF_CONFIGURATION_MANAGER_BANDS);
	pacExtrapolator_Init();

}


