/***********************************************************************************
 File:			DmaDriver.c
 Module:		DmaDriver
 Purpose:		
 Description:	FW driver forDmaDriver module
************************************************************************************/
/*---------------------------------------------------------------------------------
/						Includes						
/----------------------------------------------------------------------------------*/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "DmaManager_Api.h"
#include "HwQManager_API.h"
#include "RegAccess_Api.h"
#include "OSAL_Api.h"
#include "DmacWrapperRegs.h"
#include "DmacWrapper.h"
#include "WlanArmDma1Regs.h"
#include "WlanArmDma0Regs.h"
#include "ShramDmacWrapper.h"
#include "stringLibApi.h"
#include "HwMemoryMap.h"
#include "ConfigurationManager_api.h"
#include "loggerAPI.h"
#include "ErrorHandler_Api.h"
#include "WlanArmDma1Regs.h"


/*---------------------------------------------------------------------------------
/						Defines						
/----------------------------------------------------------------------------------*/
#define LOG_LOCAL_GID   GLOBAL_GID_DMA_MANAGER
#define LOG_LOCAL_FID 1

/*---------------------------------------------------------------------------------
/						Macros						
/----------------------------------------------------------------------------------*/




#define DMAC_USE_LEGACY_MODE_ENABLE		(0x1)
#define DMAC_USE_LEGACY_MODE_DISSABLE	(0x0)
#define DMAC_FW_2_INTERRUPT 			(0x4) //FW 2  - upper mac
#define DMAC_MANIP_BUFFER_TEST 			(0x1F)// the ManipListsShramBuffer must be aligend to 32 since the DMA cache line is 32 bytes




/*---------------------------------------------------------------------------------
/					Data Type Definition												
/----------------------------------------------------------------------------------*/
typedef void (*DmaGeneralFunc)(dmaDesc_t* pDmaDesc);


typedef union
{
	uint32	val;
	struct 
	{
		uint32	interruptEnable:32;                                    /*  */ 
	} bitFields; 
} RegDmaIntEnableReg_u; 

typedef struct dmaStat
{	
	RegDmacWrapperDma1Stat0IndReg_u dma1Stat0;
	RegDmacWrapperDma1Stat1IndReg_u dma1Stat1;
	RegWlanArmDma1Cpc3_u			armDma1Cpc3;
	RegWlanArmDma1Ccr3_u			armDma1Ccr3;
	RegWlanArmDma1Sar3_u			armDma1Sar3;
	RegWlanArmDma1Dar3_u			armDma1Dar3;

}dmaStat_t;


/*---------------------------------------------------------------------------------
/						Static Function Declaration									
/----------------------------------------------------------------------------------*/

void dmaDriverHostToShramHandler(dmaDesc_t* pDmaDesc);
void dmaDriverShramToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverShramToShramHandler(dmaDesc_t* pDmaDesc);
void dmaDriverHostIfQosToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverHostIfToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverRxCoorCountersToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverRxcountersToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverBaacountersToHostHandler(dmaDesc_t* pDmaDesc);
void dmaDriverSkipHandler(dmaDesc_t* pDmaDesc);
void dmaDriverInvalidHandler(dmaDesc_t* pDmaDesc);


/*---------------------------------------------------------------------------------
/						Static variables Declaration									
/----------------------------------------------------------------------------------*/

DmaGeneralFunc dmaHwHandler[NUM_OF_DMA_DIRECTIONS] = {
	dmaDriverHostToShramHandler , 
	dmaDriverShramToHostHandler ,
	dmaDriverShramToShramHandler,
#ifdef ENET_INC_ARCH_WAVE600D2
	dmaDriverHostIfQosToHostHandler,  // the location of the functions pointers should be in the same place as in the enum "dmaDirection_e"
#endif
	dmaDriverHostIfToHostHandler,
#ifdef ENET_INC_ARCH_WAVE600D2
	dmaDriverRxCoorCountersToHostHandler,// the location of the functions pointers should be in the same place as in the enum "dmaDirection_e"
#endif
	dmaDriverRxcountersToHostHandler,
	dmaDriverBaacountersToHostHandler,
	dmaDriverSkipHandler,
	dmaDriverInvalidHandler
	};


dmaStat_t  dmaStat = {0};


uint32 Dmac1WrapperManipulationLists[DMAC_WRAPPER_MANIPULATION_LISTS_DWORD_SIZE] =
{
	// ch0  

	0x0e210820, // SAR@8,DAR@14
	0x26401d80, // AUTO_LD_ST@22,LOOPS@37
	0x2f012c03, // AUTO_LD_ST@46,AUTO_LD_ST@54
	0x00000000, // AUTO_LD_ST@62,DAR_JUNK@70
	0x00000000, // AUTO_STORE_JUNK@78, COND_CLOSE@84
	0x00000000, // COND_END@87
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,


	// ch1
	0x0e210820, // SAR@8,DAR@14
	0x29801640, // AUTO_LD_ST@22,LOOPS@41
	0x3e403240, // AUTO_LD_ST@50,AUTO_LD_ST@62
	0x56014a40,// AUTO_LD_ST@74,COND_END@86
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	// ch2
	0x0e210820, // SAR@8,DAR@14
	0x29801640, // AUTO_LD_ST@22,LOOPS@41
	0x3e403240, // AUTO_LD_ST@50,AUTO_LD_ST@62
	0x52034a40, // AUTO_LD_ST@74,COND_CLOSE@82
	0x00005601, // COND_END@86
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,


	// ch3
	0x0e210820, // SAR@8,DAR@14
	0x29801640, // AUTO_LD_ST@22,LOOPS@41
	0x3e403240, // AUTO_LD_ST@50,AUTO_LD_ST@62
	0x52034a40, // AUTO_LD_ST@74,COND_CLOSE@82
	0x00005601, // COND_END@86
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
};



uint32 Dmac0WrapperManipulationLists[DMAC_WRAPPER_MANIPULATION_LISTS_DWORD_SIZE] =
{
	// ch0  

	0x0e210820, // SAR@8,DAR@14
	0x26401d80, // AUTO_LD_ST@22,LOOPS@37
	0x2f012c03, // AUTO_LD_ST@46,AUTO_LD_ST@54
	0x00000000, // AUTO_LD_ST@62,DAR_JUNK@70
	0x00000000, // AUTO_STORE_JUNK@78, COND_CLOSE@84
	0x00000000, // COND_END@87
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,


	// ch1
	0x0e210820, // SAR@8,DAR@14
	0x29801640, // AUTO_LD_ST@22,LOOPS@41
	0x3e403240, // AUTO_LD_ST@50,AUTO_LD_ST@62
	0x56014a40,// AUTO_LD_ST@74,COND_END@86
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,


};


uint32 Dmac1WrapperChannelPrograms[DMAC_WRAPPER_CHANNEL_PROGRAMS_DWORD_SIZE] =
{
	//ch#0 code at offset 0x100

	0x00340031, // 0)wfp 0; 2)sev 0
	0x00bc0236, // 4)wfe 0 invalidate cache; 6)movsar
	0x00000000,
	0x000002bc, // 12)movdar
	0x18180000, // 18)nop; 19)nop
	0x42f901bc, // 20)movccr
	0x181800be, // 26)nop; 27)nop
	0x18180020, // 28)lp; 30)nop; 31)nop 
	0x04381818, // 32)nop; 33)nop; 34)lpend
	0x000001bc, // 36)movccr
	0x18180000, // 42)nop; 43)nop
	0x18403418, // 44)nop; 45)sev 8; 47)nop
	0x00340031, // start of ch#0 2nd program
	0x00bc0236, 
	0x00000000,
	0x000002bc,
	0x18180000,
	0x42f901bc, 
	0x181800be, 
	0x18180020,	
	0x04381818, 
	0x000001bc, 
	0x18180000, 
	0x18403418,
	0x1818602c, 
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,

	// ch#1 code at offset 0x200
    0x08340831,
    0x00bc0a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18180835,
    0x18181818,
    0x18181818,
    0x08340831,
    0x00bc0a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18180835,
    0x1818b82c,
    0x00000000,
	
	// ch#2 code at offset 0x300
    0x10341031,
    0x00bc1236,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18185034,
    0x18181818,
    0x18181818,
    0x10341031,
    0x00bc1236,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18185034,
    0x1818b82c,
    0x00000000,
	
	// ch#3 code at offset 0x400
    0x18341831,
    0x00bc1a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18185834,
    0x18181818,
    0x18181818,
    0x18341831,
    0x00bc1a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x42f901bc,
    0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18185834,
    0x1818b82c,
    0x00000000,
};


uint32 Dmac0WrapperChannelPrograms[DMAC_WRAPPER_CHANNEL_PROGRAMS_DWORD_SIZE] =
{
	//ch#0 code at offset 0x100

	0x00340031, // 0)wfp 0; 2)sev 0
	0x00bc0236, // 4)wfe 0 invalidate cache; 6)movsar
	0x00000000,
	0x000002bc, // 12)movdar
	0x18180000, // 18)nop; 19)nop
	0x42f901bc, // 20)movccr
	0x181800be, // 26)nop; 27)nop
	0x18180020, // 28)lp; 30)nop; 31)nop 
	0x04381818, // 32)nop; 33)nop; 34)lpend
	0x000001bc, // 36)movccr
	0x18180000, // 42)nop; 43)nop
	0x18403418, // 44)nop; 45)sev 8; 47)nop
	0x00340031, // start of ch#0 2nd program
	0x00bc0236, 
	0x00000000,
	0x000002bc,
	0x18180000,
	0x42f901bc, 
	0x181800be, 
	0x18180020,	
	0x04381818, 
	0x000001bc, 
	0x18180000, 
	0x18403418,
	0x1818602c, 
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,
	0x00000000,


	// ch#1 code at offset 0x200
    0x08340831,
    0x00bc0a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
	0x42f901bc, 
	0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18180835,
    0x18181818,
    0x18181818,
    0x08340831,
    0x00bc0a36,
    0x00000000,
    0x000002bc,
    0x18180000,
    0x000001bc,
    0x18180000,
    0x18181813,
	0x42f901bc, 
	0x181800be,
    0x18180020,
    0x04381813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x000001bc,
    0x18180000,
    0x18181813,
    0x18180835,
    0x1818b82c,
    0x00000000,

};


/*---------------------------------------------------------------------------------
/						Public Functions Definitions									
/----------------------------------------------------------------------------------*/



void dmaDriverHostToShramHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;
	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);


 
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;


	pDmaDesc->destination = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pDmaDesc->destination);
	pDmaDesc->source = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->source);

	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}

void dmaDriverShramToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;

	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
 
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;

	pDmaDesc->source = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pDmaDesc->source);	
	pDmaDesc->destination =CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = (uint32)pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}


void dmaDriverShramToShramHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;

	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;


	pDmaDesc->source = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR(pDmaDesc->destination);	

		
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;
	
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}

#ifdef ENET_INC_ARCH_WAVE600D2
void dmaDriverHostIfQosToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;
	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;
	
	pDmaDesc->source = CONVERT_WLAN_HOST_IF_QOS_ADDR_TO_DMA_HOST_IF_QOS_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);	
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}

#endif


void dmaDriverHostIfToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;
	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;
	
	pDmaDesc->source = CONVERT_WLAN_HOSTIF_ADDR_TO_DMA_HOST_IF_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);	
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	
}

#ifdef ENET_INC_ARCH_WAVE600D2
void dmaDriverRxCoorCountersToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;

	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;

	pDmaDesc->source = CONVERT_WLAN_RX_COOR_COUNTER_ADDR_TO_DMA_RX_COOR_COUNTER_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);	
	
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	

}
#endif


void dmaDriverRxcountersToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;

	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;

	pDmaDesc->source = CONVERT_WLAN_RXCOUNTER_ADDR_TO_DMA_RXCOUNTER_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);	
	
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}

void dmaDriverBaacountersToHostHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;

	RegDmacWrapperDma1Ch3Fifo2PushSrcAddr_u sourceAddressRegister;
	RegDmacWrapperDma1Ch3Fifo2PushDestAddr_u destinationAddressRegister; 
	RegDmacWrapperDma1Ch3Fifo2PushCReg_u triggerRegister;


 
 	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	
	sourceAddressRegister.val = 0;
	destinationAddressRegister.val = 0;
    triggerRegister.val = 0;
						
	pDmaDesc->source = CONVERT_WLAN_BAACOUNTER_ADDR_TO_DMA_BAACOUNTER_ADDR(pDmaDesc->source);	
	pDmaDesc->destination = CONVERT_HOST_DDR_MEMORY_ADDR_TO_DMA_DDR_MEMORY_ADDR(pDmaDesc->destination);	
	
	sourceAddressRegister.bitFields.dma1Ch3Fifo2PushSrcAddr = pDmaDesc->source;
	destinationAddressRegister.bitFields.dma1Ch3Fifo2PushDestAddr = (uint32)pDmaDesc->destination;

   	triggerRegister.bitFields.dma1Ch3Fifo2PushLength = pDmaDesc->length ;
	triggerRegister.bitFields.dma1Ch3Fifo2PushLast = TRUE;

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_SRC_ADDR, sourceAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_DEST_ADDR, destinationAddressRegister.val);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_FIFO2_PUSH_C_REG, triggerRegister.val);

	OSAL_ENABLE_INTERRUPTS(interrupt_save);	


}



void dmaDriverSkipHandler(dmaDesc_t* pDmaDesc)
{
	TX_INTERRUPT_SAVE_AREA;
	UNUSED_PARAM(pDmaDesc);	
    OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	dmaManagerDmaHwDone();
    OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

void dmaDriverInvalidHandler(dmaDesc_t* pDmaDesc)
{
	UNUSED_PARAM(pDmaDesc);	
	ASSERT(0);
}

void DmaDriver_ActivateHW(dmaDesc_t* pDmaDesc)
{
	dmaHwHandler[pDmaDesc->direction](pDmaDesc);
}


ISR_VOID isr_DmaDone(void)
{
	RegDmacWrapperAgent7IntClearReg_u clearInterruptRegister;
	RegDmacWrapperAgent7PendIntReg_u  statusInterruptRegister;
	
	clearInterruptRegister.val = 0;
	statusInterruptRegister.val = 0;
	
	RegAccess_Read(REG_DMAC_WRAPPER_AGENT_7_PEND_INT_REG,&statusInterruptRegister.val);
	DEBUG_ASSERT(statusInterruptRegister.bitFields.agent7FwLastJobIdPendInt == DMAC_FW_2_INTERRUPT);//FW2 (upper mac) interrupt arriverd
	
	clearInterruptRegister.bitFields.agent7FwLastJobIdIntClear = DMAC_FW_2_INTERRUPT;
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_7_INT_CLEAR_REG, clearInterruptRegister.val);

	dmaManagerDmaHwDone();

}

#if  DMA_DEBUG
void DmaDriver_UpdateFwDmaStatus(void)
{
	RegAccess_Read(REG_DMAC_WRAPPER_DMA1_STAT0_IND_REG,&dmaStat.dma1Stat0.val);
	RegAccess_Read(REG_DMAC_WRAPPER_DMA1_STAT1_IND_REG,&dmaStat.dma1Stat1.val);
	RegAccess_Read(REG_WLAN_ARM_DMA1_CPC3,&dmaStat.armDma1Cpc3.val);
	RegAccess_Read(REG_WLAN_ARM_DMA1_CCR3,&dmaStat.armDma1Ccr3.val);
	RegAccess_Read(REG_WLAN_ARM_DMA1_SAR3,&dmaStat.armDma1Sar3.val);
	RegAccess_Read(REG_WLAN_ARM_DMA1_DAR3,&dmaStat.armDma1Dar3.val);
	
}

void DmaDriver_ResetFwDmaStatus(void)
{
	dmaStat.dma1Stat0.val= 0;
	dmaStat.dma1Stat1.val = 0;
	dmaStat.armDma1Cpc3.val = 0;
	dmaStat.armDma1Ccr3.val = 0;
	dmaStat.armDma1Sar3.val = 0;
	dmaStat.armDma1Dar3.val = 0;
}

#endif

void Dma_HwInit(void)
{

	RegDmacWrapperDma0BaReg_u		dmac0BaseAddress;
	RegDmacWrapperDma1BaReg_u		dmac1BaseAddress;
	RegDmacWrapperDma0ConfigReg_u	dmac0Config;
	RegDmacWrapperDma1ConfigReg_u	dmac1Config;
	RegDmacWrapperDma0Misc0Reg_u	dmac0Misc0;
	RegDmacWrapperDma0Misc1Reg_u	dmac0Misc1;
	RegDmacWrapperDma1MiscReg_u		dmac1Misc;
	RegDmacWrapperDma0JunkBaseAddrReg_u dma0JunkBaseAddr;
	RegDmacWrapperDma1JunkBaseAddrReg_u dma1JunkBaseAddr;
	RegDmacWrapperAgent0IntMaskReg_u dmacAgent0IntMask;
	RegDmacWrapperAgent1IntMaskReg_u dmacAgent1IntMask;
	RegDmacWrapperAgent2IntMaskReg_u dmacAgent2IntMask;
	RegDmacWrapperAgent3IntMaskReg_u dmacAgent3IntMask;
	RegDmacWrapperAgent4IntMaskReg_u dmacAgent4IntMask;
	RegDmacWrapperAgent5IntMaskReg_u dmacAgent5IntMask;
	RegDmacWrapperAgent6IntMaskReg_u dmacAgent6IntMask;
	RegDmacWrapperAgent7IntMaskReg_u dmacAgent7IntMask;
	RegDmacWrapperTx0SenderAlignerEn_u	tx0SenderAlignerEn;
	RegDmacWrapperTx1SenderAlignerEn_u	tx1SenderAlignerEn;	
	RegDmacWrapperDma0Ch0ConfigReg_u 	dma0Ch0ConfigReg;
	RegDmacWrapperDma0Ch1ConfigReg_u 	dma0Ch1ConfigReg;
	RegDmacWrapperDma1Ch0ConfigReg_u 	dma1Ch0ConfigReg;
	RegDmacWrapperDma1Ch1ConfigReg_u 	dma1Ch1ConfigReg;
	RegDmacWrapperDma1Ch2ConfigReg_u 	dma1Ch2ConfigReg;
	RegDmacWrapperDma1Ch3ConfigReg_u 	dma1Ch3ConfigReg;
	RegDmaIntEnableReg_u				regDmaIntEnable;

	CardConfigurationPciMode_e cardConfiguationPciMode = ConfigurationManager_GetConfigurationPciMode();


	dmac0BaseAddress.val		= 0;
	dmac1BaseAddress.val		= 0;
	dmac0Config.val 			= 0;
	dmac1Config.val 			= 0;
	dmac0Misc0.val				= 0;
	dmac0Misc1.val				= 0;
	dmac1Misc.val				= 0;
	dma0JunkBaseAddr.val		= 0;
	dma1JunkBaseAddr.val		= 0;
	dmacAgent0IntMask.val		= 0;
	dmacAgent1IntMask.val 		= 0;
	dmacAgent2IntMask.val		= 0;
	dmacAgent3IntMask.val		= 0;
	dmacAgent4IntMask.val		= 0;
	dmacAgent5IntMask.val 		= 0;
	dmacAgent6IntMask.val		= 0;
	dmacAgent7IntMask.val		= 0;
	tx0SenderAlignerEn.val		= 0;
	tx1SenderAlignerEn.val		= 0;
	dma0Ch0ConfigReg.val		= 0;
	dma0Ch1ConfigReg.val		= 0;
	dma1Ch0ConfigReg.val		= 0;
	dma1Ch1ConfigReg.val		= 0;
	dma1Ch2ConfigReg.val		= 0;
	dma1Ch3ConfigReg.val		= 0;
	regDmaIntEnable.val			= 0;

	tx0SenderAlignerEn.bitFields.tx0SenderAlignerEn = DMAC_WRAPPER_SENDER_ALIGNER_ENABLE;
	RegAccess_Write(REG_DMAC_WRAPPER_TX0_SENDER_ALIGNER_EN, tx0SenderAlignerEn.val);

	tx1SenderAlignerEn.bitFields.tx1SenderAlignerEn = DMAC_WRAPPER_SENDER_ALIGNER_ENABLE;
	RegAccess_Write(REG_DMAC_WRAPPER_TX1_SENDER_ALIGNER_EN, tx1SenderAlignerEn.val);


	//0x900	RAM BA DMA0
	dmac0BaseAddress.val =  System_GetXbarAddress() + SHARED_RAM_OFFSET_FROM_BAR0;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_BA_REG, dmac0BaseAddress.val);
	//0x800	RAM BA DMA1
	dmac1BaseAddress.val =  System_GetXbarAddress() + SHARED_RAM_OFFSET_FROM_BAR0;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_BA_REG, dmac1BaseAddress.val);
	//0x904	Config DMA0
	dmac0Config.bitFields.dma0ChManipBa = CONVERT_BYTES_TO_DOUBLE_WORDS((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_0].ManipListsShramBuffer - (uint32)SHARED_RAM_BASE_ADDR);
	dmac0Config.bitFields.dma0ChCodeAlloc = DMAC_WRAPPER_CHANNEL_CODING_ALLOCATED_SPACE;

	if (cardConfiguationPciMode == CONFIGURATION_PCI_MODE_SINGLE_PCI)
	{
		dmac0Config.bitFields.dma0AddrTogConf = DMAC_WRAPPER_CONFIG_AXI_ADDR_TOGGLE_VAL_FIXED_ZERO;
	}
	else // if (cardConfiguationPciMode == CONFIGURATION_PCI_MODE_DUAL_PCI)
	{
		dmac0Config.bitFields.dma0AddrTogConf = DMAC_WRAPPER_CONFIG_AXI_ADDR_TOGGLE_VAL_TOGGLE;
	}
	
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_CONFIG_REG, dmac0Config.val);
	//0x804	Config DMA1
	dmac1Config.bitFields.dma1ChManipBa = CONVERT_BYTES_TO_DOUBLE_WORDS((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_1].ManipListsShramBuffer - (uint32)SHARED_RAM_BASE_ADDR);
	// channel manipulation image start address calculation must not be greater than 16bit (i.e. < 0x10000) (calculation example below is for dma1 ch3):
	// Manip_ba_b4_shift = dma1_ch_manip_ba + (8 * 4) + (dma1_ch_code_alloc * 3)  + dma1_ch3_prg_len
	// Manip_ba = Manip_ba_b4_shift * 8
	ASSERT(dmac1Config.bitFields.dma1ChManipBa <= DMAC_WRAPPER_DMA1_CHANNEL_MANIP_BA_MAX_ADDRESS);

	dmac1Config.bitFields.dma1ChCodeAlloc = DMAC_WRAPPER_CHANNEL_CODING_ALLOCATED_SPACE;

	if (cardConfiguationPciMode == CONFIGURATION_PCI_MODE_SINGLE_PCI)
	{
		dmac1Config.bitFields.dma1AddrTogConf = DMAC_WRAPPER_CONFIG_AXI_ADDR_TOGGLE_VAL_FIXED_ZERO;
	}
	else // if (cardConfiguationPciMode == CONFIGURATION_PCI_MODE_DUAL_PCI)
	{
		dmac1Config.bitFields.dma1AddrTogConf = DMAC_WRAPPER_CONFIG_AXI_ADDR_TOGGLE_VAL_TOGGLE;
	}

	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CONFIG_REG, dmac1Config.val);
	//0x910	Misc0 DMA0	0x0080_080d
	dmac0Misc0.bitFields.dma0MonHdrOffset0 			= DMAC_WRAPPER_HEADER_MONITOR_OFFSET;  
	dmac0Misc0.bitFields.dma0MonHdrEn0 				= DMAC_WRAPPER_HEADER_MONITOR_ENABLE;
	dmac0Misc0.bitFields.dma0Tx0BufferFullMargin 	= DMAC_WRAPPER_TX_BUFFER_FULL_MARGIN;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_MISC0_REG, dmac0Misc0.val);
	//0x914	Misc1 DMA0	0x0080_080d
	dmac0Misc1.bitFields.dma0MonHdrOffset1			= DMAC_WRAPPER_HEADER_MONITOR_OFFSET;  
	dmac0Misc1.bitFields.dma0MonHdrEn1				= DMAC_WRAPPER_HEADER_MONITOR_ENABLE;
	dmac0Misc1.bitFields.dma0Tx1BufferFullMargin	= DMAC_WRAPPER_TX_BUFFER_FULL_MARGIN;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_MISC1_REG, dmac0Misc1.val);
	//0x810	Misc DMA1 0x0000_2000
	dmac1Misc.bitFields.dma1LastJobIdChSel = DMAC_WRAPPER_LAST_JOB_ID_INT_CHANNEL_SELECTION_2;
	// TBD CDB - how should we configure this in 3x1? is it like dual band or single band?
	if (ConfigurationManager_GetBandConfigurationMode() == CONFIGURATION_MODE_SINGLE_BAND)
	{
		dmac1Misc.bitFields.cdbMode = FALSE; 	// cdb mode - 0 - 3x1/4x0 (36 users to band0 no/one user for band1)
	}
	else
	{
		dmac1Misc.bitFields.cdbMode = TRUE; 	// cdb mode - 1 - 2x2 (18 users for band0 18 users band1)
	}
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_MISC_REG, dmac1Misc.val);
	//0x920	Junk BA DMA0
	dma0JunkBaseAddr.bitFields.dma0JunkBaseAddrConf = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_0].JunkYard);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_JUNK_BASE_ADDR_REG, dma0JunkBaseAddr.val);
	//0x818	Junk BA DMA1
	dma1JunkBaseAddr.bitFields.dma1JunkBaseAddrConf = CONVERT_WLAN_SHRAM_ADDR_TO_DMA_SHRAM_ADDR((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_1].JunkYard);
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_JUNK_BASE_ADDR_REG, dma1JunkBaseAddr.val);
	//0xa00	Int Enable Agent #0 (Tx0)	0x100_0000
	dmacAgent0IntMask.val	= 0x01000000; //Bits [24]      : enable "tx0 last job is done" interrupt
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_0_INT_MASK_REG, dmacAgent0IntMask.val);
	//0xa10	Int Enable Agent #1 (Tx1)	0x200_0000
	dmacAgent1IntMask.val 	= 0x02000000; //Bits [25]      : enable "tx1 last job is done" interrupt
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_1_INT_MASK_REG, dmacAgent1IntMask.val);
	//0xa20	Int Enable Agent #2 (Rx0)	0x1000_0000
	dmacAgent2IntMask.val	= 0x10000000; //Bits [28]      : enable "rx0 last job is done" interrupt
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_2_INT_MASK_REG, dmacAgent2IntMask.val);
	//0xa30	Int Enable Agent #3 (Rx1)	0x2000_0000
	dmacAgent3IntMask.val	= 0x20000000; //Bits [29]      : enable "rx1 last job is done" interrupt
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_3_INT_MASK_REG, dmacAgent3IntMask.val);
	//0xa40	Int Enable Agent #4 (Host)	0xff000
	dmacAgent4IntMask.val	= 0x000ff000; //Bits [19:12] : enable of "host last job id" interrupts [7:0]
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_4_INT_MASK_REG, dmacAgent4IntMask.val);
	//0xa50	Int Enable Agent #5 (CPU0)	0x10_0000
	dmacAgent5IntMask.val 	= 0x00100000; //Bits [20] : enable of "fw0 last job id" interrupts [2:0]
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_5_INT_MASK_REG, dmacAgent5IntMask.val);
	//0xa60	Int Enable Agent #6 (CPU1)	0x20_0000
	dmacAgent6IntMask.val	= 0x00200000; //Bits [21] : enable of "fw1 last job id" interrupts [2:0]
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_6_INT_MASK_REG, dmacAgent6IntMask.val);
	//0xa70	Int Enable Agent #7 (CPU2)	0x40_0000
	dmacAgent7IntMask.val	= 0x00400000; //Bits [22] : enable of "fw2 last job id" interrupts [2:0]
	RegAccess_Write(REG_DMAC_WRAPPER_AGENT_7_INT_MASK_REG, dmacAgent7IntMask.val);
	//	0xd00 DMA0	Channel #0 (Tx0)	config	
	dma0Ch0ConfigReg.bitFields.dma0Ch0CacheInvldEn		= DMAC_WRAPPER_CHANNEL_CONFIG_CACHE_INVALID_ENABLE;
	dma0Ch0ConfigReg.bitFields.dma0Ch0TxbufFcEnable		= DMAC_WRAPPER_CHANNEL_CONFIG_CYCLIC_BUFF_FLOW_CONTROL_ENABLE;
	dma0Ch0ConfigReg.bitFields.dma0Ch0PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma0Ch0ConfigReg.bitFields.dma0Ch0LegacyModeEn		= DMAC_USE_LEGACY_MODE_DISSABLE;
	dma0Ch0ConfigReg.bitFields.dma0Ch0BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_CH0_CONFIG_REG, dma0Ch0ConfigReg.val);
	//	0xd20 DMA0	Channel #1 (Tx1)	config	
	dma0Ch1ConfigReg.bitFields.dma0Ch1CacheInvldEn		= DMAC_WRAPPER_CHANNEL_CONFIG_CACHE_INVALID_ENABLE;
	dma0Ch1ConfigReg.bitFields.dma0Ch1TxbufFcEnable		= DMAC_WRAPPER_CHANNEL_CONFIG_CYCLIC_BUFF_FLOW_CONTROL_ENABLE;
	dma0Ch1ConfigReg.bitFields.dma0Ch1PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma0Ch1ConfigReg.bitFields.dma0Ch1LegacyModeEn		= DMAC_USE_LEGACY_MODE_DISSABLE;
	dma0Ch1ConfigReg.bitFields.dma0Ch1BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA0_CH1_CONFIG_REG, dma0Ch1ConfigReg.val);
	//0xc00	DMA1 Channel #0 (Rx0) 	config	
	dma1Ch0ConfigReg.bitFields.dma1Ch0CacheInvldEn		= DMAC_WRAPPER_CHANNEL_CONFIG_CACHE_INVALID_ENABLE;
	dma1Ch0ConfigReg.bitFields.dma1Ch0RxbufFcEnable		= DMAC_WRAPPER_CHANNEL_CONFIG_CYCLIC_BUFF_FLOW_CONTROL_ENABLE;
	dma1Ch0ConfigReg.bitFields.dma1Ch0PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma1Ch0ConfigReg.bitFields.dma1Ch0LegacyModeEn		= DMAC_USE_LEGACY_MODE_DISSABLE;
	dma1Ch0ConfigReg.bitFields.dma1Ch0BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;	
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH0_CONFIG_REG, dma1Ch0ConfigReg.val);
	//0xc20	DMA1 Channel #1 (RX1) config	
	dma1Ch1ConfigReg.bitFields.dma1Ch1CacheInvldEn		= DMAC_WRAPPER_CHANNEL_CONFIG_CACHE_INVALID_ENABLE;
	dma1Ch1ConfigReg.bitFields.dma1Ch1RxbufFcEnable		= DMAC_WRAPPER_CHANNEL_CONFIG_CYCLIC_BUFF_FLOW_CONTROL_ENABLE;
	dma1Ch1ConfigReg.bitFields.dma1Ch1PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma1Ch1ConfigReg.bitFields.dma1Ch1LegacyModeEn		= DMAC_USE_LEGACY_MODE_DISSABLE;
	dma1Ch1ConfigReg.bitFields.dma1Ch1BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;	
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH1_CONFIG_REG, dma1Ch1ConfigReg.val);
	//0xc40	DMA1 Channel #2 (Host)	config	
	dma1Ch2ConfigReg.bitFields.dma1Ch2PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma1Ch2ConfigReg.bitFields.dma1Ch2LegacyModeEn		= DMAC_USE_LEGACY_MODE_ENABLE;
	dma1Ch2ConfigReg.bitFields.dma1Ch2BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;	
	dma1Ch2ConfigReg.bitFields.dma1Ch2WmbMode 			= 1;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH2_CONFIG_REG, dma1Ch2ConfigReg.val);
	//0xc60	DMA1 Channel #3 (FW)	config	
	dma1Ch3ConfigReg.bitFields.dma1Ch3PrgLen			= DMAC_WRAPPER_CHANNEL_CONFIG_SIZEOF_PROGRAM;
	dma1Ch3ConfigReg.bitFields.dma1Ch3LegacyModeEn		= DMAC_USE_LEGACY_MODE_ENABLE;
	dma1Ch3ConfigReg.bitFields.dma1Ch3BusAndTrnsfSize	= DMAC_WRAPPER_CHANNEL_CONFIG_BUS_AND_TRANSFER_SIZE_16_BYTES;	
	dma1Ch3ConfigReg.bitFields.dma1Ch3WmbMode 			= 1;
	RegAccess_Write(REG_DMAC_WRAPPER_DMA1_CH3_CONFIG_REG, dma1Ch3ConfigReg.val);

	/* Enable DMA0 Interrupts */
	regDmaIntEnable.bitFields.interruptEnable = DMA_ENABLE_INTERRUPTS_ALL_CHANNELS;	
	RegAccess_Write(REG_WLAN_ARM_DMA0_INT_EN, regDmaIntEnable.val);

	/* Enable DMA1 Interrupts */
	regDmaIntEnable.bitFields.interruptEnable = DMA_ENABLE_INTERRUPTS_ALL_CHANNELS;	
	RegAccess_Write(REG_WLAN_ARM_DMA1_INT_EN, regDmaIntEnable.val);

	if(((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_0].ManipListsShramBuffer & DMAC_MANIP_BUFFER_TEST) != 0)
	{
		FATAL("the manipulation base address must to be aligned to dma cache address");
	}
	if(((uint32)&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_1].ManipListsShramBuffer & DMAC_MANIP_BUFFER_TEST) != 0)
	{
		FATAL("the manipulation base address must to be aligned to dma cache address");
	}
	
	//Manipulation lists and Microcode programs for DMA0
	memcpy32(&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_0].ManipListsShramBuffer, &Dmac0WrapperManipulationLists, (DMAC_WRAPPER_MANIPULATION_LISTS_DWORD_SIZE));
	memcpy32(&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_0].ChannelProgramsShramBuffer, &Dmac0WrapperChannelPrograms, (DMAC_WRAPPER_CHANNEL_PROGRAMS_DWORD_SIZE));	

	//Manipulation lists and Microcode programs for DMA1
	memcpy32(&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_1].ManipListsShramBuffer, &Dmac1WrapperManipulationLists, (DMAC_WRAPPER_MANIPULATION_LISTS_DWORD_SIZE));
	memcpy32(&DmaWrapperDb[CONFIGURATION_MANAGER_BAND_1].ChannelProgramsShramBuffer, &Dmac1WrapperChannelPrograms, (DMAC_WRAPPER_CHANNEL_PROGRAMS_DWORD_SIZE));

}



