/*******************************************************************************
*
*   Source File: logger.c
*
*	AUTHOR: Ariel Groenteman
*
*   Description:
*
*   Copyright:
*
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***						Include Files									***/
/******************************************************************************/
#include "stringLibApi.h"
#include "System_GlobalDefinitions.h"
#include "System_Configuration.h"
#include "ErrorHandler_Api.h"
#include "Pac_Api.h"
#include "MT_NSSMemoryPool.h"
#include "logmacro_mixins.h"
#include "shram_logger_buffers.h"
#include "logdefs.h"
#include "loggerDef.h"

#include "loggerAPI.h"
#include "logger.h"
#include "RegAccess_Api.h"
#include "HwLoggerAPI.h"
#include "mhi_umi.h"
#include "RxMpduStructure.h"
#include "Utils_Api.h"
#include "ShramHim.h"
#include "LoggerRegs.h"

#if defined (ENET_INC_UMAC)
#include "mhi_umi.h"
#include "HostInterface_API.h"
#endif //  ENET_INC_UMAC

/******************************************************************************/
/***						  Constants										***/
/******************************************************************************/

#define LOG_LOCAL_GID GLOBAL_GID_LOGGER
#define LOG_LOCAL_FID 0



#define LM_INSTANCE 0
#define UM_INSTANCE 1

#define LOGGER_VERSION 0
#define LOGGER_FW_OID  1

#define LOG_BASE_2_OF_32 5

#ifdef ENET_INC_ARCH_WAVE600

#if defined (ENET_INC_UMAC)
#define LOGGER_DATA_IN_PTR					REG_LOGGER_UMAC_DATA_IN
#define LOGGER_DATA_DONE_REG				REG_LOGGER_UMAC_MESAGE_DONE
#elif  defined (ENET_INC_LMAC0)
#define LOGGER_DATA_IN_PTR					REG_LOGGER_LMAC1_DATA_IN
#define LOGGER_DATA_DONE_REG				REG_LOGGER_LMAC1_MESAGE_DONE
#else //defined (ENET_INC_LMAC1)
#define LOGGER_DATA_IN_PTR					REG_LOGGER_LMAC2_DATA_IN
#define LOGGER_DATA_DONE_REG				REG_LOGGER_LMAC2_MESAGE_DONE
#endif

#else

#if defined (ENET_INC_UMAC)
#define LOGGER_DATA_IN_PTR					REG_LOGGER_UMAC_DATA_IN
#define LOGGER_DATA_DONE_REG				REG_LOGGER_UMAC_MESAGE_DONE
#else
#define LOGGER_DATA_IN_PTR					REG_LOGGER_LMAC_DATA_IN
#define LOGGER_DATA_DONE_REG				REG_LOGGER_LMAC_MESAGE_DONE
#endif

#endif //ENET_INC_ARCH_WAVE600


/******************************************************************************/
/***					Internal Data Structures							***/
/******************************************************************************/

/******************************************************************************/
/***						Public Variables								***/
/******************************************************************************/

/******************************************************************************/
/***						Static Variables								***/
/******************************************************************************/

static LogAgentLocalParam_t LoggerLocalParam;
static mtlk_log_buf_entry_t loggerLocalBuffer[LOGGER_SIZE_OF_LOCAL_BUFFER];
static HwLoggerHeader_t swLoggerMessageHeaderTemplate; 

/******************************************************************************/
/***					Static Function Declaration							***/
/******************************************************************************/
static bool IsGidFiltered(uint8 gid);
static bool IsSeverityFiltered(int32 level);

void ReleaseBuffer(uint32 bufferIdToRelease);
static void ProcessMissedLog(void);


/******************************************************************************/
/***							Debug Section								***/
/******************************************************************************/



/******************************************************************************/
/***						Public Functions Definitions					***/
/******************************************************************************/

/******************************************************************************
 **
 ** Name:		Logger_InitLoggerLocalParam
 **
 ** Purpose:	Initializes the local pointer to the logAgent local param
 **
 ** Input:		//LogAgentLocalParam_t **LogAgentLocalParamPtr
 **
 ** Output:		None
 **
 ** Returns:	STATUS_SUCCESS - Upon successful initialization
 **				STATUS_FAIL    - When initialization failed
 **
 ******************************************************************************/
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=".initialization" 
#endif

void Logger_InitLoggerLocalParam(void)
{
    loggerActiveMode = LOGGER_STATE_ACTIVE_ONLINE;  // assume at init we start in online mode
    /* set default values to groupFilters and severityLevel */
    memset (LoggerLocalParam.groupFilters.logAgentLoggerGroupsBitMap,
        MAX_UINT32,
        LOGGER_NUM_OF_GROUPS_BIT_MAP * sizeof(uint32));

    LoggerLocalParam.state = LOGGER_STATE_READY;
	
	LoggerLocalParam.isOn = TRUE;
}
#if (defined (ENET_INC_UMAC) && !defined (ENET_INC_ARCH_WAVE600))
#pragma ghs section text=default
#endif



#ifndef DISABLE_LOGGER_DEBUG
static void InitBufferDebugInfo(void)
{
    memset(LoggerLocalParam.loggerBufferDebugInfo[LoggerLocalParam.currentBufferId], 0, LOGGER_DEBUG_INFO_SIZE * sizeof(LoggerBufferDebugInfo_t));
    LoggerLocalParam.loggerBufferDebugInfoCounter[LoggerLocalParam.currentBufferId] = 0;
}
#endif

/******************************************************************************
 **
 ** Name:		LogAgent_FlushBuffer
 **
 ** Purpose:	Gives the user the option to ask the logAgent to flush it's buffer
 **
 ** Input:		None
 **
 ** Output:		None
 **
 ** Returns:	STATUS_SUCCESS
 **
 ******************************************************************************/
//void Logger_FlushBuffer(void)
//{
//    switch (LoggerLocalParam.state)
//    {
//        case LOGGER_STATE_ACTIVE:
//            Logger_StopBufferTimer();
//
//            Logger_SendBufferToHost(
//                LoggerLocalParam.currentBuffer.buffer_p,
//                LoggerLocalParam.loggerBufferInfo[LoggerLocalParam.currentBufferId].length,
//                KERNEL_NUM);
//
//            break;
//
//        default:
//            break;
//    }
//}





/******************************************************************************
 **
 ** Name:		Logger_ReleaseBuffer
 **
 ** Purpose:	Releases the buffer pointed to by *buffer.
 **
 ** Input:		buffer - pointer to the buffer being released
 **
 ** Output:		None
 **
 ** Returns:	STATUS_SUCCESS - Upon successful release.
 **             STATUS_FAIL    - If buffer doesn't belong to neither UM logAgent
 **                              nor LM logAgent
 **
 ******************************************************************************/
void LocateAndReleaseBuffer(uint32 bufferIdToRelease)
{
 	TX_INTERRUPT_SAVE_AREA;

  
	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
    ReleaseBuffer(bufferIdToRelease);
	
	OSAL_ENABLE_INTERRUPTS(interrupt_save);
	
    ProcessMissedLog();
}

/******************************************************************************
 ** Name:       ReleaseBuffer
 ** Purpose:    Releases the buffer pointed to by *buffer_p.
 ** Input:      uint32 bufferIdToRelease, char *buffer_p
 ** Returns:    none
 ******************************************************************************/
void ReleaseBuffer(uint32 bufferIdToRelease)
{
	uint8 *buffer_p;
	buffer_p = LoggerLocalParam.loggerBufferInfo[bufferIdToRelease].buffer_p;
    ASSERT ((bufferIdToRelease < LOGGER_NUM_OF_BUFFERS) && (PNULL != buffer_p));

    if (PNULL != buffer_p)
    {
		LoggerLocalParam.loggerBufferInfo[bufferIdToRelease].bufferInUse = FALSE;
    }
}


/******************************************************************************
 **
 ** Name:		Logger_CheckFilters
 **
 ** Purpose:	Checks if gid and level are filtered or not
 **
 ** Input:		level - The severity level of the log to be checked
 **             gid   - The GID of the log to be checked
 **             flag  - Will be set to TRUE if returning STATUS_SUCCESS, and set
 **                     to FALSE otherwise
 **
 ** Output:		None
 **
 ** Returns:	STATUS_SUCCESS - If gid and level are not filtered
 **             STATUS_FAILED  - Otherwise
 **
 ******************************************************************************/
Status_e Logger_CheckFilters(int32 level, int32 gid, bool *flag)
{
    Status_e retVal = STATUS_FAIL;
    bool isGidFiltered = FALSE;
    bool isSeverityFiltered = FALSE;

    isGidFiltered = IsGidFiltered(gid);
    isSeverityFiltered = IsSeverityFiltered(level);

    if (!isGidFiltered && !isSeverityFiltered)
    {
        retVal = STATUS_SUCCESS;
        *flag   = TRUE;
    }
    else
    {
        retVal = STATUS_FAIL;
        *flag   = FALSE;
    }

    return retVal;
}


bool IsLoggerEnabled(void)
{
    bool isLoggerEnabled = FALSE;

    if (LoggerLocalParam.isOn &&
        ((LoggerLocalParam.state == LOGGER_STATE_READY) ||
        (LoggerLocalParam.state == LOGGER_STATE_ACTIVE) ||
        (LoggerLocalParam.state == LOGGER_STATE_CYCLIC_MODE)))
    {
        isLoggerEnabled = TRUE;
    }

    return (isLoggerEnabled);
}
/******************************************************/
/***						Logprep Additions Start	***/
/******************************************************/
/******************************************************************************/
/***						Function Definitions From log.h					***/
/***						All function descriptions are available in log.h***/
/******************************************************************************/

int mtlk_log_get_flags(int level, int oid, uint8 gid)
{
    int32 retVal = 0;
    bool flag = FALSE;
	UNUSED_PARAM(oid);
	
    if (IsLoggerEnabled())
    {
        Logger_CheckFilters(level, gid, &flag);

        if (!flag)
        {
            retVal = 0;
        }
        else
        {
            retVal = LOG_TARGET_REMOTE;
        }
    }

    return retVal;
}



uint32 mtlk_log_get_timestamp(void)
{
    volatile uint32 currTime = GET_TSF_TIMER_LOW();
    return currTime;
}

mtlk_log_buf_entry_t *logReserveBuffer(uint32 pkt_size)
{
	if (pkt_size > LOGGER_SIZE_OF_LOCAL_BUFFER)
	{
		ASSERT(0);
	}
	
	return &loggerLocalBuffer[0];
}


void logFillLogEvent(uint32 oid, uint8 gid, uint32 fid, uint32 lid, uint32 len, OUT uint8 **buff_p)
{
	UNUSED_PARAM(len);	
	UNUSED_PARAM(buff_p);	
	swLoggerMessageHeaderTemplate.fid = fid; 
	swLoggerMessageHeaderTemplate.gid = gid;
	swLoggerMessageHeaderTemplate.lid = lid;
	swLoggerMessageHeaderTemplate.exp = 0; 
	swLoggerMessageHeaderTemplate.oid = oid;
}

void logReleaseBuffer(mtlk_log_buf_entry_t *p, uint32 byteLen)
{
	uint32 wordIndex;
	uint32 wordLength;
	uint32* buf_p = (uint32*)p;

	TX_INTERRUPT_SAVE_AREA;
	OSAL_DISABLE_INTERRUPTS(&interrupt_save); // disable interrupts so we won't get any logs when trying to send the last one of these stream 
	swLoggerMessageHeaderTemplate.length = (byteLen + sizeof(HwLoggerHeader_t));

	RegAccess_Write(LOGGER_DATA_IN_PTR, swLoggerMessageHeaderTemplate.word0); 
	RegAccess_Write(LOGGER_DATA_IN_PTR, swLoggerMessageHeaderTemplate.word1); 

	wordLength = CEILING(byteLen, sizeof(uint32));	// convert to length in words units

	for(wordIndex = 0; wordIndex < wordLength; wordIndex++)
	{
		RegAccess_Write(LOGGER_DATA_IN_PTR,*buf_p);
		buf_p ++; /* point to the next parameter*/ 
	}
	
	RegAccess_Write(LOGGER_DATA_DONE_REG, 0x1);


	OSAL_ENABLE_INTERRUPTS(interrupt_save);
}

#ifndef ENET_INC_ARCH_WAVE600
void logFillMsgHdrTemplate(UMI_ADD_STREAM_REQ_t* addStreamStructurePtr)
{
	swLoggerMessageHeaderTemplate.be0 	= addStreamStructurePtr->be0;
	swLoggerMessageHeaderTemplate.be1 	= addStreamStructurePtr->be1; 
	swLoggerMessageHeaderTemplate.dest 	= addStreamStructurePtr->streamId; 
	swLoggerMessageHeaderTemplate.logVer = addStreamStructurePtr->logVersion; 
	swLoggerMessageHeaderTemplate.wlanIf = addStreamStructurePtr->wlanIf; 
	swLoggerMessageHeaderTemplate.pri	= LOGGER_DEFUALT_PRIORITY; 
}
#else
void logFillMsgHdrTemplate(HwLoggerHeader_t* hwLoggerHeaderPtr)
{
	swLoggerMessageHeaderTemplate.be0	= hwLoggerHeaderPtr->be0;
	swLoggerMessageHeaderTemplate.be1	= hwLoggerHeaderPtr->be1; 
	swLoggerMessageHeaderTemplate.dest	= hwLoggerHeaderPtr->dest; 
	swLoggerMessageHeaderTemplate.logVer = hwLoggerHeaderPtr->logVer; 
	swLoggerMessageHeaderTemplate.wlanIf = hwLoggerHeaderPtr->wlanIf; 
	swLoggerMessageHeaderTemplate.band	= hwLoggerHeaderPtr->band;  
	swLoggerMessageHeaderTemplate.pri	= LOGGER_DEFUALT_PRIORITY; 
}
#endif


/******************************************************/
/***						Logprep Additions End	***/
/******************************************************/

/******************************************************************************/
/***						Private Functions Definitions					***/
/******************************************************************************/


/******************************************************************************
 **
 ** Name:		IsGidFiltered
 **
 ** Purpose:	Checks if GID of this logData is filtered or not
 **
 ** Input:		gid - The GID to check if filtered
 **
 ** Output:		None
 **
 ** Returns:	TRUE  - If GID is filtered
 **				FALSE - If GID is not filtered
 **
 ******************************************************************************/
static bool IsGidFiltered(uint8 gid)
{
    bool retVal = FALSE;

    uint32 mod = (gid & (NUM_OF_BITS_IN_UINT32 - 1));
    uint32 index = (gid >> LOG_BASE_2_OF_32);

    ASSERT(index < LOGGER_NUM_OF_GROUPS_BIT_MAP);

    retVal = ((((0x1) << mod) & ((uint32)LoggerLocalParam.groupFilters.logAgentLoggerGroupsBitMap[index])) == 0x00);

    return retVal;
}

/******************************************************************************
 **
 ** Name:		IsSeverityFiltered
 **
 ** Purpose:	Checks if severityLevel of this logData is filtered or not
 **
 ** Input:		level - The severityLevel to check if filtered
 **
 ** Output:		None
 **
 ** Returns:	TRUE  - If severityLevel is filtered
 **				FALSE - If severityLevel is not filtered
 **
 ******************************************************************************/
static bool IsSeverityFiltered(int32 level)
{
	UNUSED_PARAM(level);	
	return FALSE;
}


static void ProcessMissedLog(void)
{
    uint32 missedLogsDueToLackOfBuffers = LoggerLocalParam.missedLogsDueToLackOfBuffers;
    uint32 missedLogsDueToLackOfBD = LoggerLocalParam.missedLogsDueToLackOfBD;

    if (missedLogsDueToLackOfBuffers + missedLogsDueToLackOfBD > 0)
    {
        LoggerLocalParam.missedLogsDueToLackOfBuffers = 0;
        LoggerLocalParam.missedLogsDueToLackOfBD = 0;

        ILOG0_DD("** missed logs: lack of buffers: %d, lack of BD: %d",
            missedLogsDueToLackOfBuffers,
            missedLogsDueToLackOfBD);
    }
}

void Logger_IncMissedLogsDueToLackOfBD()
{
    LoggerLocalParam.missedLogsDueToLackOfBD++;
}







