/*******************************************************************************
*    
*   Source File: loggerAPI.c
*   
*   AUTHOR: Ariel Groenteman
*
*   Description: 
*       
*   Copyright: 
*       
*   Revision History:
*
*******************************************************************************/

/******************************************************************************/
/***                        Include Files                                   ***/
/******************************************************************************/
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "ErrorHandler_Api.h"
#include "System_Information.h"
#include "stringLibApi.h"
#include "OSAL_Api.h"
#include "Dut_Api.h"
#include "OSAL_UpperMacMessages.h"
#include "Utils_Api.h"
#include "queue_utility.h"
#include "logger.h"

#include "shram_man_msgs.h"
/*Needed for the task table, compiled only in the UM*/
#if defined (ENET_INC_UMAC)
#include "init_ifmsg.h"
#include "um_interface.h"
#endif
#include "HostInterface_API.h"
#include "EventsManager_api.h"
#include "DmaManager_Api.h"
#include "ErrorHandler_Api.h"
#include "HwLoggerAPI.h"
#include "loggerAPI.h"

/******************************************************************************/
/***                          Constants                                     ***/
/******************************************************************************/

#define LOG_LOCAL_GID GLOBAL_GID_LOGGER /* specific module GID */
#define LOG_LOCAL_FID 2 /* specific FID within this GID */


/******************************************************************************/
/***                    External Data Structures                            ***/
/******************************************************************************/

/******************************************************************************/
/***                        Public Variables                                ***/
/******************************************************************************/

/****************************************************************************/
/* Functions Prototypes                                                     */
/****************************************************************************/

#if defined (ENET_INC_UMAC)
static void LoggerApi_SendUmBufferToHost(K_MSG *kMsg_p);
static void LoggerApi_HandleDbgTraceInd(K_MSG *kMsg_p);
#ifndef ENET_INC_ARCH_WAVE600
static uint16 LoggerApi_loggerStationIndex = SID_FOR_LOGGER_RX_DATA;
#else 
static uint16 LoggerApi_loggerStationIndex = INVALID_LOGGER_SID; //Valid value will be configured by driver on set data path command
#endif
#endif


/******************************************************************************/
/***                        Static Variables                                ***/
/******************************************************************************/

#if defined (ENET_INC_UMAC)
/* Table of events and their corresponding functions in TASK_LOGGER */
static const FunctionEntry_t afpTaskTable[TASK_LOGGER_END-TASK_LOGGER_START] =
{
    {LoggerApi_HandleGeneralCfm, 		DOUBLE_CHECK_MSG_TYPE(KMSG_LOGGER_LM_LOGGER_GENERAL_CFM)},
    {LoggerApi_SendLmBufferToHost, 		DOUBLE_CHECK_MSG_TYPE(KMSG_LOGGER_LM_LOGGER_BUFFER_IND)},
    {LoggerApi_SendUmBufferToHost,		DOUBLE_CHECK_MSG_TYPE(KMSG_LOGGER_UM_SEND_BUFFER)},
    {LoggerApi_HandleDbgTraceInd, 		DOUBLE_CHECK_MSG_TYPE(KMSG_LOGGER_DBG_TRACE_IND)},


};
#endif




/******************************************************************************/
/***                            Debug Section                               ***/
/******************************************************************************/



/******************************************************************************/
/***                        Public Functions Definitions                    ***/
/******************************************************************************/
/******************************************************************************
 ** Name:       Logger_Task
 ** Purpose:    global, entry point for Logger module task
 ** Input:      K_MSG* psMsg
 ** Output:     None
 ******************************************************************************/
#if defined (ENET_INC_UMAC)
void LOGGER_Task(K_MSG *psMsg)
{
    /* Use common task switching and Table */
    vTaskDispatcher(psMsg, afpTaskTable, TASK_LOGGER_START, TASK_LOGGER_END);
}
#endif
/******************************************************************************
 **
 ** Function Name:  LoggerApi_LoggerInit
 **
 ** DESCRIPTION:    Initializes the logAgent to a LOGGER_STATE_READY state
 **
 ** PARAMETERS:     None
 **
 ** RETURNS:        STATUS_SUCCESS - Upon successful initialization
 **                 STATUS_FAIL    - When initialization failed
 **
 ** REMARKS:        Note that this function is called once in initialization
 **                 process, thus it is surrounded by:
 **                 #pragma ghs section text=".DDR_TEXT"
 **
 ** RESTRICTIONS:   None
 **          
 ******************************************************************************/
#if !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization" 
#endif
void LoggerApi_LoggerInit(void)
{
    Logger_InitLoggerLocalParam();

}
#if !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default
#endif


/******************************************************************************
 **
 ** Function Name:  LoggerApi_SetFilter
 **
 ** DESCRIPTION:    Lets the user adjust the GID groupFilters during runtime
 **
 ** PARAMETERS:     psMsg - pointer to message from the host
 **
 ** RETURNS:        STATUS_SUCCESS
 **
 ** REMARKS:        psMsg->abData is expected to be of type UmiLoggerMsgSetFilter_t
 **                 this struct is expected to contain:
 **                 1. The ID of the CPU this message is addressed to, under
 **                    targetCPU (see UmiCpuId_e for optional values)
 **                 2. The new filter mask under gidFilterMask
 **
 ** RESTRICTIONS:   psMsg->abData is expected to be of type UmiLoggerMsgSetFilter_t
 **          
 ******************************************************************************/
#if defined (ENET_INC_UMAC)
void LoggerApi_SetFilter(UmiLoggerMsgSetFilter_t *filterMsg)
{
	UNUSED_PARAM(filterMsg);	
}

#ifdef DEBUG_UM_INTERFACE
void LoggerApi_SetSeverity(UmiLoggerMsgSetSeverity_t *severityMsg)
{
	if (severityMsg->getSetOperation == API_GET_OPERATION)
	{
		severityMsg->newLevel = 0;
	}
}
#endif

/******************************************************************************
 **
 ** Function Name:  LoggerApi_SetLogAgentMode
 **
 ** DESCRIPTION:    Lets the user change the mode of the local logAgent at runtime
 **
 ** PARAMETERS:     psMsg - pointer to message from the host
 **
 ** RETURNS:        STATUS_SUCCESS
 **
 ** REMARKS:        psMsg->abData is expected to be of type UmiLoggerMsgSetMode_t
 **                 this struct is expected to contain:
 **                 1. The ID of the CPU this message is addressed to, under
 **                    targetCPU (see UmiCpuId_e for optional values)
 **                 2. The new mode under modeReq. Available modes can be found
 **                    in the enum LogAgentState_e.
 **
 ** RESTRICTIONS:   psMsg->abData is expected to be of type UmiLoggerMsgSetMode_t
 **          
 ******************************************************************************/
void LoggerApi_SetLogAgentMode(K_MSG *kMsg_p)
{
    UmiLoggerMsgSetMode_t *newModeReq_p = PNULL;

	newModeReq_p = (UmiLoggerMsgSetMode_t *)pK_MSG_DATA(kMsg_p);
#ifdef ENET_INC_ARCH_WAVE600
	SLOG0(0, 0, UmiLoggerMsgSetMode_t, newModeReq_p);
#endif

	LoggerApi_SetLoggerMode(kMsg_p, newModeReq_p->modeReq, newModeReq_p->targetCPU);
}

void LoggerApi_SetLoggerMode(K_MSG *kMsg_p, uint32 loggerNewState, uint32 targetCPU)
{
	UNUSED_PARAM(kMsg_p);	
	UNUSED_PARAM(loggerNewState);	
	UNUSED_PARAM(targetCPU);	
}

/******************************************************************************
 **
 ** Function Name:  LoggerApi_FlushBuffer
 **
 ** DESCRIPTION:    Gives the user the option to ask the logAgent to flush it's buffer
 **
 ** PARAMETERS:     psMsg - pointer to message from the host
 **
 ** RETURNS:        STATUS_SUCCESS
 **
 ** REMARKS:        psMsg->abData is expected to be of type UmiLoggerMsgFlushBuffer_t
 **                 this struct is expected to contain:
 **                 1. The ID of the CPU this message is addressed to, under
 **                    targetCPU (see UmiCpuId_e for optional values)
 **
 ** RESTRICTIONS:   psMsg->abData is expected to be of type UmiLoggerMsgFlushBuffer_t
 **          
 ******************************************************************************/
void LoggerApi_FlushBuffer(UmiLoggerMsgFlushBuffer_t *flushMsg_p)
{
	TX_INTERRUPT_SAVE_AREA;
    bool flush = FALSE;
    
//	UmiLoggerMsgFlushBuffer_t *flushMsg_p = (UmiLoggerMsgFlushBuffer_t *)pK_MSG_DATA(kMsg_p);
    UmiCpuId_e targetCpuId = (UmiCpuId_e)(flushMsg_p->targetCPU);

	SLOG0(0, 0, UmiLoggerMsgFlushBuffer_t, flushMsg_p);

    if ((UMI_CPU_ID_ALL == targetCpuId) || 
        (UMI_CPU_ID_UM == targetCpuId))
    {
        flush = TRUE;
    }

    if (flush)
    {
        OSAL_DISABLE_INTERRUPTS(&interrupt_save);
        OSAL_ENABLE_INTERRUPTS(interrupt_save);
    }
}


/******************************************************************************
 ** Function Name:  LoggerApi_SendUmBufferToHost
 ** DESCRIPTION:    sends the buffer to host
 ** PARAMETERS:     K_MSG *kMsg_p
 ** RETURNS:        none
 ******************************************************************************/
static void LoggerApi_SendUmBufferToHost(K_MSG *kMsg_p)
{
	UNUSED_PARAM(kMsg_p);	
}

#endif



#if defined(ENET_INC_UMAC)


void LoggerApi_SendLmBufferToHost(K_MSG *psMsg)
{
	UNUSED_PARAM(psMsg);	
	ASSERT(0);
}

#endif

/******************************************************************************/
/***                        Private Functions Definitions                   ***/
/******************************************************************************/
void LoggerApi_LogKernelEvent(LogKernelEvents_e kernelEvent, K_MSG *kMsg_p)
{
	switch (kernelEvent)
	{
		case KERNEL_EVENT_RECV_KNL_MSG:
		{
			uint32 fromTo;
			uint32 from_to_kMsgIntId;
			UNUSED_PARAM(from_to_kMsgIntId);

			// 16 LSB: fromTask -> toTask
			fromTo = kMsg_p->header.sFrom.taskID << 8;
			fromTo |= (0x000000ff & kMsg_p->header.sTo.taskID);

			from_to_kMsgIntId = (fromTo << 16) | (0x0000ffff & kMsg_p->header.tKMsgType);
			
#if defined (ENET_INC_UMAC)
//		    if ((KCC_TIMER_TICK != kMsg_p->header.tKMsgType) 
//			     && (KCC_TIMER_EXPIRE != kMsg_p->header.tKMsgType)
//		         && (KMSG_LOGGER_UM_SEND_BUFFER != kMsg_p->header.tKMsgType)
//		         && (UMI_UM_GET_STATISTICS_CFM != kMsg_p->header.tKMsgType)
// 			     && (BSS_MANAGE_LM_GET_STATS_CFM != kMsg_p->header.tKMsgType)
//			     && (BSS_MANAGE_UM_GET_STATISTICS_REQ != kMsg_p->header.tKMsgType)
//			     && (LMI_LM_GET_STATS_REQ != kMsg_p->header.tKMsgType))
#endif 
		//        (TASK_KNL_NULL != kMsg_p->sFrom.taskID)
		        
		    { 
//		        ILOG0_D("KERNEL_EVENT_RECV_KNL_MSG fromTask_toTask_kMsgId: %08x", from_to_kMsgIntId);
		    }
		}
			break;
			
		case KERNEL_EVENT_RECV_FROM_HOST_MSG:
		if ((UM_DBG_MAC_WATCHDOG_REQ != kMsg_p->header.tKMsgType)				
			&& (UM_MAN_QUERY_BCL_VALUE != kMsg_p->header.tKMsgType)
			&& (UM_MAN_GET_STATISTICS_REQ != kMsg_p->header.tKMsgType)
				
				)
			{
				ILOG0_D("KERNEL_EVENT_RECV_FROM_HOST_MSG kMsgType: %08x", kMsg_p->header.tKMsgType);
			}
			break;
			
		case KERNEL_EVENT_SEND_TO_HOST_MSG:
			if (
				 (MC_DBG_MAC_WATCHDOG_CFM != kMsg_p->header.tKMsgType)
				&& (MC_MAN_QUERY_BCL_CFM != kMsg_p->header.tKMsgType)
				&& (MC_MAN_GET_STATISTICS_CFM  != kMsg_p->header.tKMsgType)				
				)
			{
				ILOG0_D("KERNEL_EVENT_SEND_TO_HOST_MSG kMsgType: %08x", kMsg_p->header.tKMsgType);
			}
			break;
			
		default:
			break;
	}
}


/******************************************************************************
 ** Function Name:  SerialTrace
 ** DESCRIPTION:    SerialTrace, sends a char array to the Host with max size of MAX_DBG_TRACE_DATA
 **                     Host is responsible to print out on the serial terminal
 ** PARAMETERS:     uint8 *data_p, char array
 **					uint32 val1, uint32 val2, uint32 val3 - 3 parameters that the driver will print in hex format
 ** RETURNS:        None
 ******************************************************************************/

#ifdef SERIAL_TRACE_ENABLE
#ifdef ENET_INC_UMAC // note : serial trace is only supported in UMAC
void SerialTrace(const char *data_p, uint32 val1, uint32 val2, uint32 val3)
{
    UmiDbgTraceInd_t umiDbgTraceInd;
    uint32 length = strlen(data_p) + 1;

    if (length > MAX_DBG_TRACE_DATA)
    {
        length = MAX_DBG_TRACE_DATA;
    }
    
    umiDbgTraceInd.length = length;
    MEMCPY(&(umiDbgTraceInd.au8Data), data_p, length);

	// Those 3 parameters will be printed to CLI by driver in this format: 0x%08X
	umiDbgTraceInd.val1 = val1;
	umiDbgTraceInd.val2 = val2;
	umiDbgTraceInd.val3 = val3;

    vIF_SendMsg(&sUmiIndFreeQueue, UMI_MC_DBG_TRACE_IND, TASK_UM_IF_TASK, &umiDbgTraceInd, sizeof(UmiDbgTraceInd_t), 0);
	
}

void LoggerApiTestSerialTrace(uint32 val1, uint32 val2, uint32 val3)
{
    SERIAL_TRACE("hello world!", val1, val2, val3);
}
#endif //SERIAL_TRACE_ENABLE
#endif //ENET_INC_UMAC


#if defined (ENET_INC_UMAC)
static void LoggerApi_HandleDbgTraceInd(K_MSG *kMsg_p)
{
    OSAL_QUEUE_MSG(&sUmiIndFreeQueue, kMsg_p);
}


uint16 LoggerApi_GetLoggerStationIndex(void)
{
	return LoggerApi_loggerStationIndex;
}

void LoggerApi_SetLoggerStationIndex(uint16 loggerStationIndex)
{
#ifndef ENET_INC_ARCH_WAVE600
	DEBUG_ASSERT(loggerStationIndex == SID_FOR_LOGGER_RX_DATA);
#else
	LoggerApi_loggerStationIndex = loggerStationIndex;
#endif //ENET_INC_ARCH_WAVE600
}



/******************************************************************************
 ** Function Name:  LoggerApi_HandleDbgTraceRes
 ** DESCRIPTION:    handle trace res from the host and free the message ind
 ** PARAMETERS:     K_MSG *kMsg_p
 ** RETURNS:        None
 ******************************************************************************/
void LoggerApi_HandleDbgTraceRes(K_MSG *kMsg_p)
{
    OSAL_QUEUE_MSG(&sUmiIndFreeQueue, kMsg_p);
}


/******************************************************************************
 ** Name:		LoggerAPIGetPoolStart
 ** Purpose:	returns the address to the buffer address pool
 ** Input:		void* inMemAllocated_p, uint32 inNBuffers
 ** Returns:	address to the start of the buffer pool
 ******************************************************************************/
uint32 LoggerAPIGetPoolStart(void* inMemAllocated_p, uint32 inNBuffers)
{
    uint32 addr = 0;
//	if (logger_isLoggerOn())
	{
		addr = ((uint32)inMemAllocated_p + (sizeof(MemPoolHdr_T ) * inNBuffers));
	}
    return (addr);
}

void LoggerApi_HandleGeneralCfm(K_MSG *psMsg)
{
	UNUSED_PARAM(psMsg);	
	ASSERT(0);
}
#endif //  (ENET_INC_UMAC)



#if defined (ENET_INC_LMAC)

void LoggerApi_LmProcessReqFromUM(K_MSG *kMsg_p)
{	
	UNUSED_PARAM(kMsg_p);	
	ASSERT(0);
}

#endif // ENET_INC_LMAC


/* TLOG will be set on by user if relevant */
#ifdef TLOG_FLAG
void TLOG0(K_MSG *kMsg_p, bool isFromDriver)
{

	uint16 msg_size;
	
	/* msg_size value is the original payload of the message unless this value is greater than the maximum capacity of log buffer */
	msg_size = LOGGER_SIZE_OF_LOCAL_BUFFER - (sizeof(mtlk_log_event_data_t) + LOGPKT_SIGNAL_STRUCT_SIZE);
	/* we take the minimum value between the original payload size and the maximum capacity of log buffer and use it in the macro */
	msg_size = MIN(msg_size, kMsg_p->header.payloadSize);
	/* send TLOG */
	TLOG_Sender0(kMsg_p, isFromDriver, msg_size);
}
#endif // TLOG FLAG


