
#include "System_Configuration.h"
#include "System_GlobalDefinitions.h"
#include "System_MainApi.h"
#include "MT_NSSMemoryPool.h"
#include "linkAdaptation_api.h"
#include "lmtimerUtility.h"
#include "OSAL_Interrupts.h"
#include "Pac_Api.h"
#include "ErrorHandler_Api.h"
#include "System_Timers.h"
#include "EmeraldEnvRegs.h"
#include "RegAccess_Api.h"
#include "lm.h"
#include "MT_Math.h"
#include "HdkCdbManagerTask_api.h"
#include "System_MainApi.h"
/******************************************/
/*        DO NOT REMOVE THIS LINE!        */
/******************************************/
#include "loggerAPI.h"

#define LOG_LOCAL_GID   GLOBAL_GID_LOWER_MAC
#define LOG_LOCAL_FID 5

/*Global timers + one timer per-STA (BF) + one for Beacon Blocking + one fo online calibration timer*/
#define TIMER_UTILITY_POOL_SIZE      (NUMBER_OF_LINK_ADAPTATION_TIMERS + HW_NUM_OF_STATIONS + 1)
#define LM_TIMER_UTILITY_TSF_MASK 0x7FFFFFFF /*Use only 31 bits out of 32 in order to to module comparition*/
#define LM_TIMER_UTILITY_RANGE (1u<<31)

/************************************************************/
/***************          Timer Utilty Private  functions      ***************/
/************************************************************/
static void TimerUtiltyScheduleInterrupt(int32 timeToGo);
static  void TimerUtiltyRemoveInterrupt(void);


/*******************************************************/
/*******************          Globals            *******************/
/******************************************************/
typedef struct timer_Event
{
    uint32          expirationTsfValue;
    uint32          eventParam; 
    void            (*eventHandler)(uint32 eventParam);
    void*           eventNext;   
	uint16			eventId;
}timer_Event_t;

#ifdef CPU_MIPS
#pragma alignvar (4)
#endif

#if !defined (ENET_INC_ARCH_WAVE600)
uint8 *TimerUtilityObjectPool;
#else
uint8 TimerUtilityObjectPool[ MT_POOL_SIZE(sizeof(timer_Event_t), TIMER_UTILITY_POOL_SIZE) ];
#ifdef CPU_ARC
#pragma Align_to(4,TimerUtilityObjectPool)
#endif

#endif



MemPool_T TimerUtilityPool;
timer_Event_t* pTimerUtiltyHead;

/*************************************************************/
/******************        private   Functions          ********************/
/*************************************************************/ 
/***************************************************
 ** Name:		TimerUtiltyInterruptHandler
 ** Purpose:	       handle timer interrupt, execute event 
 **                        handler and configure timer for the next event
 ** Input:		None
 ** Returns:	       None
 ***************************************************/
void TimerUtiltyInterruptHandler()
{
    timer_Event_t* piterator = pTimerUtiltyHead;
    uint32 currentTsfValue = GET_TSF_TIMER_LOW();
    timer_Event_t* ptemp;

	currentTsfValue &= LM_TIMER_UTILITY_TSF_MASK;
	/*It is possible that timer expired while list was being modified - therefore, it is possible that current TSF is below the lowest expiration TSF*/
	if (pTimerUtiltyHead == NULL)
	{
		return; 
	}
	else if (cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue, currentTsfValue, FALSE, (uint32)LM_TIMER_UTILITY_RANGE)> 0)
	{
    	TimerUtiltyScheduleInterrupt(cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue,currentTsfValue, FALSE,(uint32)LM_TIMER_UTILITY_RANGE));
		return; 
	}
    while((piterator != NULL) && ((cyclicMinusOperator(piterator->expirationTsfValue, currentTsfValue, FALSE, (uint32)LM_TIMER_UTILITY_RANGE)) <= 0))
    {
        piterator = piterator->eventNext;
    }
    while(pTimerUtiltyHead != piterator)
    {
        (*(pTimerUtiltyHead->eventHandler))(pTimerUtiltyHead->eventParam);
        ptemp = pTimerUtiltyHead;
        pTimerUtiltyHead = pTimerUtiltyHead->eventNext;
        FreeBuf(&TimerUtilityPool,ptemp);
    }
    if(pTimerUtiltyHead != NULL)
    {
    	TimerUtiltyScheduleInterrupt(cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue,currentTsfValue, FALSE,(uint32)LM_TIMER_UTILITY_RANGE));
    }
}

/***************************************************
 ** Name:		TimerUtiltyScheduleInterrupt
 ** Purpose:	       Configure interrupt from timer3
 ** Input:		timeToGo- 
 ** Returns:	       None
 ***************************************************/    
static void TimerUtiltyScheduleInterrupt(int32 timeToGo)
{
    MT_SetModeTimer(MT_TIMER_3,MT_TIMER_DISABLE, MT_TIMER_ONE_SHOT);
    MT_SetTimeTimer(MT_TIMER_3,timeToGo);
    MT_SetModeTimer(MT_TIMER_3, MT_TIMER_ENABLE, MT_TIMER_ONE_SHOT);
}

/***************************************************
 ** Name:		TimerUtiltyRemoveInterrupt
 ** Purpose:	       Remove interrupt from timer3
 ** Input:		None 
 ** Returns:	       None
 ***************************************************/  
static  void TimerUtiltyRemoveInterrupt()
{
    MT_SetModeTimer(MT_TIMER_3,MT_TIMER_DISABLE, MT_TIMER_ONE_SHOT);
}

/*************************************************************/
/*******************        API   Functions          **********************/
/*************************************************************/
/***************************************************
 ** Name:		TimerUtiltyAddEvent
 ** Purpose:	       add new timer event
 ** Input:		expirationTime
 **                        eventHandler - function to handle event
 **                        moduleEventIdentification - pointer for module usage  
 ** Returns:	       event id - unique for every event 
 ***************************************************/
uint16 TimerUtiltyAddEvent(uint32 expirationTime, void (*eventHandler)(uint32 eventParam), uint32 eventParam)
{
    timer_Event_t*   newEvent;
    timer_Event_t*   piterator  = NULL;
    timer_Event_t**  ppiterator = NULL;
    uint32           currentTsfValue;
    TX_INTERRUPT_SAVE_AREA;

	OSAL_DISABLE_INTERRUPTS(&interrupt_save);
	currentTsfValue = GET_TSF_TIMER_LOW();	
    newEvent = AllocBuf(&TimerUtilityPool);
    newEvent->eventHandler = eventHandler;
    newEvent->expirationTsfValue = (expirationTime + currentTsfValue) & LM_TIMER_UTILITY_TSF_MASK;
    newEvent->eventParam = eventParam;
	piterator  = pTimerUtiltyHead;
	//List is empty  OR Replace head 
    if(( pTimerUtiltyHead == NULL) || ((cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue,newEvent->expirationTsfValue, FALSE, (uint32)LM_TIMER_UTILITY_RANGE)) >= 0))
    {
        newEvent->eventNext = pTimerUtiltyHead;
        pTimerUtiltyHead = newEvent;
        //Overriding interrupt
        TimerUtiltyScheduleInterrupt(expirationTime);
    }
    else
    {
		ppiterator = (timer_Event_t**)&(piterator->eventNext);
		piterator = piterator->eventNext;
        while((piterator != NULL) && ((cyclicMinusOperator(piterator->expirationTsfValue,newEvent->expirationTsfValue, FALSE,(uint32)LM_TIMER_UTILITY_RANGE)) <= 0))
        {
            ppiterator = (timer_Event_t**)&(piterator->eventNext);
            piterator = piterator->eventNext;
        }
        *ppiterator = newEvent;
        newEvent->eventNext = piterator;       
        //timer_utilty_Schedule_interrupt(expiration_time);
    }
	OSAL_ENABLE_INTERRUPTS(interrupt_save);  
	return newEvent->eventId;
}

/***************************************************
 ** Name:		TimerUtiltyRemoveEvent
 ** Purpose:	       remove timer event
 ** Input:		pointer to event id   
 ** Returns:	       none
 ***************************************************/
void TimerUtiltyRemoveEvent(uint16 *eventId)
{
	timer_Event_t* piterator; /*Can't initialize here as head may change before we disable interrupts*/
	timer_Event_t* ptemp;
	uint32 currentTsfValue = 0;
	TX_INTERRUPT_SAVE_AREA;

	/*
	We can enter this function without having any event to remove - 
	in case of remove station - when station removing, we call this function in order to remove the current event, but actually the timer can be already expired and event can be alreadly removed
	in this case we do nothing 
	*/
	OSAL_DISABLE_INTERRUPTS(&interrupt_save); /*Disable interrupts here*/
	/*This functiopn may be called before we know the timer has expired. In the expiration ISR we set the event ID to MAX_UINT8/MAX_UINT16*/
	if (*eventId == MAX_UINT16)
	{
		/*Timer already expired, just return*/
		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		return;    
	}
	/*Timer has not expired yet, list can not be empty*/
	ASSERT(pTimerUtiltyHead != NULL);
	//Replace head
	if(pTimerUtiltyHead->eventId == *eventId)
	{
		currentTsfValue =  GET_TSF_TIMER_LOW();
		currentTsfValue &= LM_TIMER_UTILITY_TSF_MASK;
		ptemp = pTimerUtiltyHead;
		pTimerUtiltyHead = pTimerUtiltyHead->eventNext;
		FreeBuf(&TimerUtilityPool,ptemp);
		
		if(pTimerUtiltyHead != NULL)
		{	//Overriding interrupt, 
			if (cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue,currentTsfValue, FALSE,(uint32)LM_TIMER_UTILITY_RANGE) > 0)
			{
				TimerUtiltyScheduleInterrupt(cyclicMinusOperator(pTimerUtiltyHead->expirationTsfValue,currentTsfValue, FALSE,(uint32)LM_TIMER_UTILITY_RANGE));
			}
			else
			{
				TimerUtiltyScheduleInterrupt(0);
			}
		}
		else
		{	//remove interrupt
			TimerUtiltyRemoveInterrupt();
		}
		/*Clear Timer ID*/
		*eventId = MAX_UINT16;

		OSAL_ENABLE_INTERRUPTS(interrupt_save);
		return;    
	}
	/*Set iterator to head - we already know we are not the head event*/
	piterator  = pTimerUtiltyHead;
	while(piterator->eventNext != NULL) 
	{
		if(((timer_Event_t*)piterator->eventNext)->eventId == *eventId)
		{
			ptemp = piterator->eventNext;
			piterator->eventNext = ((timer_Event_t*)piterator->eventNext)->eventNext;
			FreeBuf(&TimerUtilityPool,ptemp);
			/*Clear Timer ID*/
			*eventId = MAX_UINT16;
			OSAL_ENABLE_INTERRUPTS(interrupt_save);
			return;
		}
		piterator = piterator->eventNext;
	}
   	/*We got here without finding the event*/
   	ASSERT(FALSE);
}

/***************************************************
 ** Name:		TimerUtiltyInit
 ** Purpose:	       
 ** Input:		none  
 ** Returns:	       none
 ***************************************************/
#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=".initialization_start" 
#endif

void TimerUtiltyInit()
{  
	timer_Event_t*	 newEvent;
	uint16 objectId;

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
	System_MainAllocInitializationMemory((uint8 **)&TimerUtilityObjectPool, MT_POOL_SIZE(sizeof(timer_Event_t), TIMER_UTILITY_POOL_SIZE));
#endif


	CreatePool(&TimerUtilityPool, TimerUtilityObjectPool,MT_POOL_SIZE(sizeof(timer_Event_t), TIMER_UTILITY_POOL_SIZE),
        sizeof(timer_Event_t), TIMER_UTILITY_POOL_SIZE);
    pTimerUtiltyHead = NULL;
	// Allocated buffer IDs by going over all buffers and assigning hard coded values
	for(objectId = 0 ; objectId < TIMER_UTILITY_POOL_SIZE ; objectId++)
	{
		newEvent = GetBufferAddressById(&TimerUtilityPool, objectId);
		newEvent->eventId = objectId;
	}
	/*Enable timer3 interrupt*/
	MT_enableDisableGpTimerIrq(MT_TIMER_3, 1);
}

#if defined (ENET_INC_LMAC) && !defined (ENET_INC_ARCH_WAVE600)
#pragma ghs section text=default 
#endif
